/************************************************************************
* FILE:        hc_tclThermalSensor.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: base class of all thermal sensors
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 05.02.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "hc_tclThermalSensor.h"
#include "hc_tclThermalSensorControl.h"
#include "hc_tclAbstractSensorControl.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclThermalSensor.cpp.trc.h"
#endif

// names of thermal sensor IDs
const char* m_astrThermalSensorID[enThermalSensorID_MAX] = {
		"UNDEF",
		"V850",
		"Panel",
		"CDDRIVE",
		"DVDDRIVE",
		"GPS",
		"GYRO",
		"EMMC",
		"AMP",
		"IMX",
		"HEATSINK",
		"BLUERAY",
		"XMModule",
		"TVModule",
		"TVModuleSlave",
		"Panel2"
};

const char* strGetThermalSensorID(tenThermalSensorID enSensorID)
{
	if (enThermalSensorID_MAX > enSensorID)
		return m_astrThermalSensorID[enSensorID];
	return m_astrThermalSensorID[enThermalSensorID_UNDEF];
};

// abbreviation of thermal sensor
const char* m_astrThermalSensorAbbreviation[enThermalSensorID_MAX] = {
		"UNDEF",
		"V850_",
		"PANEL",
		"CD___",
		"DVD__",
		"GPS__",
		"GYRO_",
		"EMMC_",
		"AMP__",
		"IMX__",
		"HSINK",
		"BR___",
		"XMMOD",
		"TVMOD",
		"TVMSL",
		"PAN2_"
};

const char* strGetThermalSensorAbbreviation(tenThermalSensorID enSensorID)
{
	if (enThermalSensorID_MAX > enSensorID)
		return m_astrThermalSensorAbbreviation[enSensorID];
	return m_astrThermalSensorAbbreviation[enThermalSensorID_UNDEF];
};

// names of thermal sensor states
const char* m_astrThermalState[enThermalState_MAX] = {
		"UNAVAILABLE",
		"CONFIGFAILURE",
		"INIT",
		"TEMPUNAVAIL",
		"CRITICALLOW",
		"LOW",
		"COLD",
		"NORMAL",
		"WARM",
		"HIGH",
		"EMERGENCYHIGH"
};

const char* strGetThermalState(tenThermalState enState)
{
	if (enThermalState_MAX > enState)
		return m_astrThermalState[enState];
	return m_astrThermalState[enThermalState_UNAVAILABLE];
};

// names of temperature qualifier
const char* m_astrTemperatureQualifier[enTemperatureQualifier_MAX] = {
		"INVALID",
		"ESTIMATED",
		"CALCULATED",
		"REAL",
		"UNSUPPORTED"
};

const char* strGetTemperatureQualifier(tenTemperatureQualifier enQualifier)
{
	if (enTemperatureQualifier_MAX > enQualifier)
		return m_astrTemperatureQualifier[enQualifier];
	return m_astrTemperatureQualifier[enTemperatureQualifier_INVALID];
}
/*******************************************************************************
*
* FUNCTION:    hc_tclThermalSensor()
*
* DESCRIPTION: constructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclThermalSensor::hc_tclThermalSensor()
: hc_tclThermalSensor_SM()
, hc_tclThermalSensor_Config()
, hc_tclThermalSensor_Statistics()
, hc_tclThermalSensor_EF()
, hc_tclThermalSensor_ITC()
, hc_tclThermalSensor_FAN()
{
   ETG_TRACE_USR4(("hc_tclThermalSensor() entered."));

   (tVoid)memset((tVoid*)&m_oIfThermalInfo, OSAL_NULL, sizeof(m_oIfThermalInfo));
   m_enSensorID = enThermalSensorID_UNDEF;
   m_u32AbstractSensorNotifies = 0;
}

/*******************************************************************************
*
* FUNCTION:    hc_tclThermalSensor()
*
* DESCRIPTION: constructor
*
* PARAMETER:   tenThermalSensorID enSensorID		own thermal sensor ID
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclThermalSensor::hc_tclThermalSensor(tenThermalSensorID enSensorID)
: hc_tclThermalSensor_SM()
, hc_tclThermalSensor_Config()
, hc_tclThermalSensor_Statistics()
, hc_tclThermalSensor_EF()
, hc_tclThermalSensor_ITC()
, hc_tclThermalSensor_FAN()
{
   ETG_TRACE_USR4(("hc_tclThermalSensor(id: %d) entered.", enSensorID ));

   (tVoid)memset((tVoid*)&m_oIfThermalInfo, OSAL_NULL, sizeof(m_oIfThermalInfo));
   m_enSensorID = enSensorID;
   m_u32AbstractSensorNotifies = 0;
   if (FALSE == hc_tclThermalSensorControl::bNotify(this, enSensorID))
   {
	   ETG_TRACE_FATAL(("hc_tclThermalSensor: notification of thermal sensor %d failed", enSensorID ));
   }
}

/*******************************************************************************
*
* FUNCTION:    ~hc_tclThermalSensor()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclThermalSensor::~hc_tclThermalSensor()
{
   ETG_TRACE_USR4(("~hc_tclThermalSensor() entered."));
   vDenotifyAbstractSensor();
   hc_tclThermalSensorControl::vDenotify(m_enSensorID);
}

/*******************************************************************************
*
* FUNCTION: 	vGetReferences(tVoid)
*
* DESCRIPTION: 	Function to get all reference needed by this class.
* 				A reference should always be the Interface class of the object
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensor::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));
   // load configuration
   vLoadConfig(enGetSensorID());
   // first state machine update
   tBool _bTempChanged = FALSE;
   tBool _bStateChanged = FALSE;
   vThermalSensor_SM(m_oIfThermalInfo.u.tThermalInfo, _bTempChanged, _bStateChanged);
   // update EF
   tBool _bEfChanged = FALSE;
   vUpdate_EmergencyFunctionState(_bEfChanged);
   // update ITCs
   tBool _bItcChanged = FALSE;
   vUpdate_ITCState(_bItcChanged);
}

/*******************************************************************************
*
* FUNCTION: 	vHandleMessage_ThermalInfoStandard()
*
* DESCRIPTION: 	handler for internal standard thermal information messages
*
* PARAMETER:   	hc_tclBaseIf::TMsg* pMsg: reference of received messagde
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensor::vHandleMessage_ThermalInfoStandard(hc_tclBaseIf::TMsg &Msg,
		tBool &bTemperatureChanged, tBool &bSensorStateChanged)
{
   ETG_TRACE_USR4(("vHandleMessage_ThermalInfoStandard() entered %u -> data: %d.", ETG_ENUM(HC_INT_MSG , (tU32)Msg.eCmd), Msg.u.u32Data));
   if(hc_tclBaseIf::eNewThermalInfo_STANDARD != Msg.eCmd)
	   return;
   else if (FALSE == bIsIfClienthandlerValid(Msg))
   {
	   ETG_TRACE_USR1(("eNewThermalInfo_STANDARD: invalid thermal client interface %50s (current %50s)",
			   Msg.u.tThermalInfo.strClassName, m_oIfThermalInfo.u.tThermalInfo.strClassName));
	   return;
   }
   else if (TRUE == bIsIfAvailableChanged(Msg) || TRUE == bIsIfStateChanged(Msg) || TRUE == bIsIfTempChanged(Msg))
   {
	   (tVoid)memcpy((tVoid*)&m_oIfThermalInfo, (const tVoid*)&Msg, sizeof(m_oIfThermalInfo));
	   vThermalSensor_SM(m_oIfThermalInfo.u.tThermalInfo, bTemperatureChanged, bSensorStateChanged);
   }
   if ((TRUE == bTemperatureChanged) || (TRUE == bSensorStateChanged))
   {
	   // update sensor statistics
	   vUpdate_Statistics();
	   // trace information
	   //vTraceInfo();
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleMessage_ThermalInfoAUTOSAR()
*
* DESCRIPTION: 	handler for internal AUTOSAR thermal information messages
*
* PARAMETER:   	hc_tclBaseIf::TMsg* pMsg: reference of received messagde
* 				tBool &bTemperatureChanged			temperature changed
* 				tBool &bSensorStateChanged			sensor state changed
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensor::vHandleMessage_ThermalInfoAUTOSAR(hc_tclBaseIf::TMsg &Msg,
		tBool &bTemperatureChanged, tBool &bSensorStateChanged)
{
   ETG_TRACE_USR4(("vHandleMessage_ThermalInfoAUTOSAR() entered %u -> data: %d.", ETG_ENUM(HC_INT_MSG , (tU32)Msg.eCmd), Msg.u.u32Data));
   if(hc_tclBaseIf::eNewThermalInfo_AUTOSAR != Msg.eCmd)
	   return;
   ::thermal_management_inc_fi::EModuleState enAUTOSAR = Msg.u.tThermalInfo.uSensorState.enAUTOSAR;
   Msg.eCmd = hc_tclBaseIf::eNewThermalInfo_STANDARD;
   switch (enAUTOSAR)
   {
   case ::thermal_management_inc_fi::ePOOR:
	   if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_C2N) <= Msg.u.tThermalInfo.tTemperature)
		   Msg.u.tThermalInfo.uSensorState.enHC = enThermalState_EMERGENCYHIGH;
	   else if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_W2N) >= Msg.u.tThermalInfo.tTemperature)
		   Msg.u.tThermalInfo.uSensorState.enHC = enThermalState_CRITICALLOW;
	   break;
   case ::thermal_management_inc_fi::eBAD:
	   if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_C2N) <= Msg.u.tThermalInfo.tTemperature)
		   Msg.u.tThermalInfo.uSensorState.enHC = enThermalState_HIGH;
	   else if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_W2N) >= Msg.u.tThermalInfo.tTemperature)
		   Msg.u.tThermalInfo.uSensorState.enHC = enThermalState_LOW;
	   break;
   case ::thermal_management_inc_fi::eGOOD:
	   Msg.u.tThermalInfo.uSensorState.enHC = enThermalState_NORMAL;
	   break;
   default:
	   Msg.u.tThermalInfo.uSensorState.enHC = enThermalState_INIT;
	   if (Msg.u.tThermalInfo.bSensorStateReceived)
		   Msg.u.tThermalInfo.uSensorState.enHC = enThermalState_TEMPUNAVAIL;
	   Msg.u.tThermalInfo.bSensorStateSupported = FALSE;
	   break;
   }
   vHandleMessage_ThermalInfoStandard(Msg, bTemperatureChanged, bSensorStateChanged);
}

/*******************************************************************************
*
* FUNCTION: 	bNotifyAbstractSensor()
*
* DESCRIPTION: 	notification / assignment abstract sensors
*
* PARAMETER:   	tenAbstractSensorID enAbstractSensor
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tBool hc_tclThermalSensor::bNotifyAbstractSensor(tenAbstractSensorID enAbstractSensor)
{
   ETG_TRACE_USR4(("bNotifyAbstractSensor(sensor %d) entered.", enAbstractSensor));
   if (TRUE == hc_tclAbstractSensorControl::bNotify(this, enAbstractSensor))
   {
	   m_u32AbstractSensorNotifies = m_u32AbstractSensorNotifies | (1 << (tU32)enAbstractSensor);
	   return TRUE;
   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION: 	vDenotifyAbstractSensor()
*
* DESCRIPTION: 	denotification / deassignment for all abstract sensors
*
* PARAMETER:   	none
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclThermalSensor::vDenotifyAbstractSensor(tVoid)
{
   ETG_TRACE_USR4(("vDenotifyAbstractSensor() entered."));
   for (tU8 _u8Index = 0; (tU8)enAbstractSensorID_MAX > _u8Index; ++_u8Index)
   {
	   if ((1 << (tU32)_u8Index) == (m_u32AbstractSensorNotifies & (1 << (tU32)_u8Index)))
		   hc_tclAbstractSensorControl::vDenotify((tenAbstractSensorID)_u8Index);
   }
   m_u32AbstractSensorNotifies = 0;
}

/*******************************************************************************
*
* FUNCTION: 	vTraceInfo()
*
* DESCRIPTION: 	trace status information
*
* PARAMETER:   	none
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclThermalSensor::vTraceInfo()
{
   ETG_TRACE_USR4(("vTraceInfo() entered."));
   tChar _strIdentStart[30] = "";
   sprintf(_strIdentStart, "HC.APPL.TSEN.%5s.START", strGetThermalSensorAbbreviation(enGetSensorID()));
   tChar _strIdentAttr[30] = "";
   sprintf(_strIdentAttr, "HC.APPL.TSEN.%5s.ATTR_", strGetThermalSensorAbbreviation(enGetSensorID()));
   tChar _strIdentEnd[30] = "";
   sprintf(_strIdentEnd, "HC.APPL.TSEN.%5s.END__", strGetThermalSensorAbbreviation(enGetSensorID()));
   // trace starting of trace block
   ETG_TRACE_USR2(("%026s: ", _strIdentStart ));
   vTraceInfo_Sensor(_strIdentAttr);
   vTraceInfo_SM(_strIdentAttr);
   vTraceInfo_SensorIF(_strIdentAttr);
   vTraceInfo_Statistics(_strIdentAttr);
   vTraceInfo_EF(_strIdentAttr);
   vTraceInfo_ITC(_strIdentAttr);
   vTraceInfo_FAN(_strIdentAttr);
   vTraceInfo_Config(_strIdentAttr);
   // trace ending of trace block
   ETG_TRACE_USR2(("%026s: ", _strIdentEnd ));
}

/*******************************************************************************
*
* FUNCTION: 	vTraceInfo_Sensor()
*
* DESCRIPTION: 	trace sensor information
*
* PARAMETER:   	const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclThermalSensor::vTraceInfo_Sensor(const char* strIdentifier)
{
   ETG_TRACE_USR4(("vTraceInfo_Sensor() entered." ));

   ETG_TRACE_USR2(("%26s: %30s=%25s ", strIdentifier, "                NAME",
		   strGetThermalSensorID(enGetSensorID()) ));
   ETG_TRACE_USR2(("%26s: %30s=%d ", strIdentifier, "                  ID",
		   enGetSensorID() ));
}

/*******************************************************************************
*
* FUNCTION: 	vTraceInfo_SensorIF()
*
* DESCRIPTION: 	trace sensor interface information
*
* PARAMETER:   	const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclThermalSensor::vTraceInfo_SensorIF(const char* strIdentifier)
{
   ETG_TRACE_USR4(("vTraceInfo_SensorIF() entered." ));

   ETG_TRACE_USR2(("%26s: %30s=%50s ", strIdentifier, "             IF_NAME",
		   m_oIfThermalInfo.u.tThermalInfo.strClassName ));
   ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier, "            IF_AVAIL",
		   ETG_ENUM(HC_Bool, m_oIfThermalInfo.u.tThermalInfo.bIfAvailable) ));
   ETG_TRACE_USR2(("%26s: %30s=%d ", strIdentifier, "           IF_SEN_ID",
		   m_oIfThermalInfo.u.tThermalInfo.enSensorID ));
   ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier, "    IF_SEN_TEMP_SUPP",
		   ETG_ENUM(HC_Bool, m_oIfThermalInfo.u.tThermalInfo.bTemperatureSupported) ));
   ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier, "    IF_SEN_TEMP_RECV",
		   ETG_ENUM(HC_Bool, m_oIfThermalInfo.u.tThermalInfo.bTemperatureReceived) ));
   ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier, "     IF_SEN_TEMP_EST",
		   ETG_ENUM(HC_Bool, m_oIfThermalInfo.u.tThermalInfo.bTemperatureEstimated) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC]", strIdentifier, "     IF_SEN_TEMP_VAL",
		   m_oIfThermalInfo.u.tThermalInfo.tTemperature ));
   ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier, "   IF_SEN_STATE_SUPP",
		   ETG_ENUM(HC_Bool, m_oIfThermalInfo.u.tThermalInfo.bSensorStateSupported) ));
   ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier, "   IF_SEN_STATE_RECV",
		   ETG_ENUM(HC_Bool, m_oIfThermalInfo.u.tThermalInfo.bSensorStateReceived) ));
   ETG_TRACE_USR2(("%26s: %30s=%d ", strIdentifier, "    IF_SEN_STATE_VAL",
		   m_oIfThermalInfo.u.tThermalInfo.uSensorState.enHC ));
}

/*******************************************************************************
*
* FUNCTION: 	vGetLastIfMessage()
*
* DESCRIPTION: 	copy the internal message to parameter
*
* PARAMETER:   	hc_tclBaseIf::TMsg &Msg
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclThermalSensor::vGetLastIfMessage(hc_tclBaseIf::TMsg &oMsg) const
{
	(tVoid)memcpy((tVoid*)&oMsg, (const tVoid*)&m_oIfThermalInfo, sizeof(oMsg));
}
