/*******************************************************************************
* FILE:        hc_tclFAN.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*-------------------------------------------------------------------------------
*
* DESCRIPTION: class to control fan
*
*-------------------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 28.07.2015| CM-AI/EPB2 Bernard | init
*
*******************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/
#include "hc_AppMain.h"
#include "hc_tclFAN.h"
#include "hc_tclFANControl.h"
#include "hc_tclFAN_Power.h"
#include "hc_tclFAN_Speed.h"
#include "hc_tclFAN_Current.h"

/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclFAN.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/

// mapping list of FAN control modes depending of FAN control mode events
const tenFANCtrlMode m_enConfigControlMode[enFANCtrlModeEvent_MAX] = {
		enFANCtrlMode_AUTO,          // enFANCtrlModeEvent_MANUAL_END
		enFANCtrlMode_INIT,          // enFANCtrlModeEvent_INIT_START
		enFANCtrlMode_AUTO,          // enFANCtrlModeEvent_AUTO_START
		enFANCtrlMode_AUTO_SELFTEST, // enFANCtrlModeEvent_AUTO_SELFTEST_START
		enFANCtrlMode_AUTO_MEASURE,  // enFANCtrlModeEvent_AUTO_MEASURE_START
		enFANCtrlMode_MANUAL,        // enFANCtrlModeEvent_MANUAL_START
		enFANCtrlMode_STOPPED        // enFANCtrlModeEvent_STOPPED_START
};

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    hc_tclFAN()
*
* DESCRIPTION: constructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclFAN::hc_tclFAN()
: hc_tclFAN_Statistics()
{
   ETG_TRACE_USR4(("hc_tclFAN() entered."));

   m_poFANControl = OSAL_NULL;
   m_poPower = OSAL_NULL;
   m_poSpeed = OSAL_NULL;
   m_poCurrent = OSAL_NULL;
   m_enFANID = enFANID_SYSTEM;
   (tVoid)memset((tVoid*)&m_sControlModeInfo, OSAL_NULL, sizeof(m_sControlModeInfo));
   m_sControlModeInfo.enEvent = enFANCtrlModeEvent_DOENTCARE;
   m_sControlModeInfo.enReqEvent = enFANCtrlModeEvent_DOENTCARE;
   (tVoid)memset((tVoid*)m_asControlModeStateInfo, OSAL_NULL, sizeof(m_asControlModeStateInfo));
   m_u16LastMeasuredCurrent = 0;
   (tVoid)memset((tVoid*)m_au8SensorFanSpeed, OSAL_NULL, sizeof(m_au8SensorFanSpeed));
}

/*******************************************************************************
*
* FUNCTION:    hc_tclFAN()
*
* DESCRIPTION: constructor
*
* PARAMETER:   hc_tclFANControl* poFANControl       reference of fan control
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclFAN::hc_tclFAN(hc_tclFANControl* poFANControl)
{
   ETG_TRACE_USR4(("hc_tclFAN() entered."));

   m_poFANControl = poFANControl;
   m_poPower = OSAL_NULL;
   m_poSpeed = OSAL_NULL;
   m_poCurrent = OSAL_NULL;
   m_enFANID = enFANID_SYSTEM;
   (tVoid)memset((tVoid*)&m_sControlModeInfo, OSAL_NULL, sizeof(m_sControlModeInfo));
   m_sControlModeInfo.enEvent = enFANCtrlModeEvent_DOENTCARE;
   m_sControlModeInfo.enReqEvent = enFANCtrlModeEvent_DOENTCARE;
   (tVoid)memset((tVoid*)m_asControlModeStateInfo, OSAL_NULL, sizeof(m_asControlModeStateInfo));
   m_u16LastMeasuredCurrent = 0;
   (tVoid)memset((tVoid*)m_au8SensorFanSpeed, OSAL_NULL, sizeof(m_au8SensorFanSpeed));
}

/*******************************************************************************
*
* FUNCTION:    ~hc_tclFAN()
*
* DESCRIPTION: destructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclFAN::~hc_tclFAN()
{
   ETG_TRACE_USR4(("~hc_tclFAN() entered."));

   m_poFANControl = OSAL_NULL;

   if (OSAL_NULL != m_poPower)
	   OSAL_DELETE m_poPower;
   m_poPower = OSAL_NULL;
   if (OSAL_NULL != m_poSpeed)
	   OSAL_DELETE m_poSpeed;
   m_poSpeed = OSAL_NULL;
   if (OSAL_NULL != m_poCurrent)
	   OSAL_DELETE m_poCurrent;
   m_poCurrent = OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION:    vGetReferences()
*
* DESCRIPTION: Function to get all reference needed by this class.
* 			   A reference should always be the Interface class of the object
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));
   m_poPower = OSAL_NEW hc_tclFAN_Power(this);
   if (OSAL_NULL != m_poPower)
	   m_poPower->vGetReferences();
   m_poSpeed = OSAL_NEW hc_tclFAN_Speed(this);
   if (OSAL_NULL != m_poSpeed)
	   m_poSpeed->vGetReferences();
   m_poCurrent = OSAL_NEW hc_tclFAN_Current(this);
   if (OSAL_NULL != m_poCurrent)
	   m_poCurrent->vGetReferences();
}

/*******************************************************************************
*
* FUNCTION:    vStartCommunication()
*
* DESCRIPTION: Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));
   // reset own state informations
   (tVoid)memset((tVoid*)&m_sControlModeInfo, OSAL_NULL, sizeof(m_sControlModeInfo));
   m_sControlModeInfo.enEvent = enFANCtrlModeEvent_DOENTCARE;
   m_sControlModeInfo.enReqEvent = enFANCtrlModeEvent_DOENTCARE;
   (tVoid)memset((tVoid*)m_asControlModeStateInfo, OSAL_NULL, sizeof(m_asControlModeStateInfo));
   // trigger start
   if (OSAL_NULL != poGetPower())
	   poGetPower()->vStartCommunication();
   if (OSAL_NULL != poGetSpeed())
	   poGetSpeed()->vStartCommunication();
   if (OSAL_NULL != poGetCurrent())
	   poGetCurrent()->vStartCommunication();
}

/*******************************************************************************
*
* FUNCTION:    vStopCommunication()
*
* DESCRIPTION: Function to stops all dynamic objects e.g. threads, ...
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN::vStopCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStopCommunication() entered."));
   // trigger stop
   if (OSAL_NULL != poGetPower())
	   poGetPower()->vStopCommunication();
   if (OSAL_NULL != poGetSpeed())
	   poGetSpeed()->vStopCommunication();
   if (OSAL_NULL != poGetCurrent())
	   poGetCurrent()->vStopCommunication();
}

/*******************************************************************************
*
* FUNCTION:    vMonitorFANSpeed()
*
* DESCRIPTION: supervision of FAN speed
*              !!! cyclic call !!!
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN::vMonitorFAN(tVoid)
{
   ETG_TRACE_USR4(("vMonitorFAN() entered."));

   // supervision of FAN power state, FAN speed and FAN current
   if (enFANCtrlMode_STOPPED != enGetControlMode())
   {
	   if (OSAL_NULL != poGetPower())
		   poGetPower()->vMonitorFANPwrState();
	   if (OSAL_NULL != poGetSpeed())
		   poGetSpeed()->vMonitorFANSpeed();
	   if (OSAL_NULL != poGetCurrent())
		   poGetCurrent()->vMonitorFANCurrent();
   }
   // update FAN control mode
   vSM_ControlMode();
}

/*******************************************************************************
*
* FUNCTION:    vSM_ControlMode()
*
* DESCRIPTION: state machine of FAN control mode
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN::vSM_ControlMode(tVoid)
{
   ETG_TRACE_USR4(("vSM_ControlMode() entered."));
   // load current control mode depending of current control mode event
   tenFANCtrlMode _enMode = enGetControlMode();
   // load current control mode event or pending requested control mode event
   tenFANCtrlModeEvent _enEvent = enGetControlModeEvent();
   if (enFANCtrlModeEvent_DOENTCARE == enGetControlModeEvent())
   {
	   (tVoid)enSetControlModeEvent(enGetReqFanControlEvent());
	   (tVoid)enSetReqFanControlEvent(enFANCtrlModeEvent_DOENTCARE);
   }
   // load current time index and and decrement
   tU32 _u32TimeIndex = u32GetControlModeTimeIndex();
   (tVoid)u32SetControlModeTimeIndex(u32GetNextTimeIndex(_u32TimeIndex));
   // loop
   tU32 _u32LoopCount = 0;
   do
   {
	   _enMode = enGetControlMode();
	   _enEvent = enGetControlModeEvent();
	   if (enFANCtrlMode_MAX > _enMode && enFANCtrlMode_INIT <= _enMode)
	   {
		   (tVoid)enSetControlMode(enGetControlMode(enGetControlModeEvent()));
		   (tVoid)enSetControlModeEvent(enFANCtrlModeEvent_DOENTCARE);
	   }
	   else
	   {
		   (tVoid)enSetControlMode(enFANCtrlMode_INIT);
	   }
	   ++_u32LoopCount;
	   ETG_TRACE_USR3(("vSM_ControlMode (loop %d): Mode(%u -> %u), Event(%u -> %u)",
			   _u32LoopCount, ETG_ENUM(HC_FANMODE, _enMode), ETG_ENUM(HC_FANMODE, enGetControlMode()),
			   ETG_ENUM(HC_FANMODEEVENT, _enEvent), ETG_ENUM(HC_FANMODEEVENT, enGetControlModeEvent()) ));
   } while (bSM_ControlMode_Looping(_u32LoopCount, _enMode));
   // handle active control mode
   vHandle_ControlMode();
}

/*******************************************************************************
*
* FUNCTION:    vSM_ControlMode_INIT()
*
* DESCRIPTION: state machine of FAN control mode INIT
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN::vSM_ControlMode_INIT(tVoid)
{
   ETG_TRACE_USR4(("vSM_ControlMode_INIT() entered."));
   // current control mode
   tenFANCtrlMode _enMode = enFANCtrlMode_INIT;
   // current control mode event
   tenFANCtrlModeEvent _enEvent = enGetControlModeEvent();
   tenFANCtrlModeState _enState = enGetControlModeState(_enMode);
   // time index
   tU32 _u32TimeIndex = u32GetControlModeStateTimeIndex(_enMode);
   (tVoid)u32SetControlModeStateTimeIndex(_enMode, u32GetNextTimeIndex(_u32TimeIndex));
   tU32 _u32LoopCount = 0;
   do
   {
	   // next state
	   _enState = enGetControlModeState(_enMode);
	   // state change
	   switch (_enState)
	   {
	   case enFANCtrlModeState_INIT:
		   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
		   {
			   // TODO: (tVoid)enSetControlModeEvent(enFANCtrlModeEvent_AUTO_SELFTEST_START);
			   (tVoid)enSetControlModeEvent(enFANCtrlModeEvent_AUTO_START);
	   		   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_FINISHED);
		   }
		   break;
	   case enFANCtrlModeState_FINISHED:
		   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
		   {
			   (tVoid)enSetControlModeEvent(enFANCtrlModeEvent_AUTO_START);
			   (tVoid)u32SetControlModeStateTimeIndex(_enMode, 1);
		   }
		   break;
	   default:
	   	   {
	   		   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_INIT);
	   	   }
		   break;
	   }
	   ++_u32LoopCount;
	   ETG_TRACE_USR3(("vSM_ControlMode_INIT (loop %d): Mode(%u -> %u), Event(%u -> %u)",
			   _u32LoopCount, ETG_ENUM(HC_FANMODE, _enMode), ETG_ENUM(HC_FANMODE, enGetControlMode()),
			   ETG_ENUM(HC_FANMODEEVENT, _enEvent), ETG_ENUM(HC_FANMODEEVENT, enGetControlModeEvent()) ));
	   ETG_TRACE_USR3(("vSM_ControlMode_INIT (loop %d): State(%u -> %u), TimeIndex(%d -> %d)",
			   _u32LoopCount, ETG_ENUM(HC_FANMODESTATE, _enState), ETG_ENUM(HC_FANMODESTATE, enGetControlModeState(_enMode)),
			   _u32TimeIndex, u32GetControlModeStateTimeIndex(_enMode) ));
   } while (bSM_ControlModeState_Looping(_u32LoopCount, _enMode, _enState));
}

/*******************************************************************************
*
* FUNCTION:    vSM_ControlMode_AUTO()
*
* DESCRIPTION: state machine of FAN control mode AUTO
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN::vSM_ControlMode_AUTO(tVoid)
{
   ETG_TRACE_USR4(("vSM_ControlMode_AUTO() entered."));
   // current control mode
   tenFANCtrlMode _enMode = enFANCtrlMode_AUTO;
   // current control mode event
   tenFANCtrlModeEvent _enEvent = enGetControlModeEvent();
   tenFANCtrlModeState _enState = enGetControlModeState(_enMode);
   // time index
   tU32 _u32TimeIndex = u32GetControlModeStateTimeIndex(_enMode);
   (tVoid)u32SetControlModeStateTimeIndex(_enMode, u32GetNextTimeIndex(_u32TimeIndex));
   tU32 _u32LoopCount = 0;
   // loop
   do
   {
	   // next state
	   _enState = enGetControlModeState(_enMode);
	   // state change
	   switch (_enState)
	   {
	   case enFANCtrlModeState_INIT:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   if (0 < u8GetSpeed())
				   {
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_ON);
				   }
				   else
				   {
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_OFF);
				   }
			   }
	   	   }
		   break;
	   case enFANCtrlModeState_AUTO_OFF:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   if (OSAL_C_U8_MAX == u8GetReqFanSpeed(_enMode))
				   {
					   (tVoid)u8SetReqFanSpeed(_enMode, u8GetSpeed(enThermalSensorID_MAX));
				   }
				   if (OSAL_C_U8_MAX != u8GetReqFanSpeed(_enMode))
				   {
					   if (HC_FANCONTROL_FANSPEED_MIN >= u8GetReqFanSpeed(_enMode))
					   {
						   if (u8GetSpeed() != u8GetSpeed(enThermalSensorID_MAX))
							   (tVoid)u8SetSpeed(u8GetSpeed(enThermalSensorID_MAX));
						   // ignore request ... too slow
						   (tVoid)u8SetReqFanSpeed(_enMode, OSAL_C_U8_MAX);
						   (tVoid)u8SetReqFanSpeedStart(_enMode, OSAL_C_U8_MAX);
					   }
					   else
					   {
						   // start fan
						   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_STARTUP);
					   }
				   }
			   }
	   	   }
		   break;
	   case enFANCtrlModeState_AUTO_STARTUP:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   if (OSAL_C_U8_MAX != u8GetReqFanSpeed(_enMode))
				   {
					   if (FALSE == bIsSpeedAvailable() || FALSE == bIsCurrentAvailable())
					   {
						   // TODO: setting error flags
						   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_FAILED);
					   }
					   else if (HC_FANCONTROL_FANSPEED_STARTUP != u8SetSpeedStart(HC_FANCONTROL_FANSPEED_STARTUP))
					   {
						   // TODO: setting error flags
						   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_FAILED);
					   }
					   else if (HC_FANCONTROL_FANSPEED_STARTUP != u8SetSpeed(HC_FANCONTROL_FANSPEED_STARTUP))
					   {
						   // TODO: setting error flags
						   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_FAILED);
					   }
					   else
					   {
						   (tVoid)u32SetControlModeStateTimeIndex(_enMode, HC_FANCONTROL_DELAYTIMECYCLE_STARTUP);
						   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_STARTUP_DELAY);
					   }
				   }
			   }
	   	   }
	   	   break;
	   case enFANCtrlModeState_AUTO_STARTUP_DELAY:
		   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   if (FALSE == bIsSpeedAvailable() || FALSE == bIsCurrentAvailable())
				   {
					   // TODO: setting error flags
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_FAILED);
				   }
				   else if (0 == u32GetControlModeStateTimeIndex(_enMode))
				   {
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_ON);
				   }
			   }
		   }
		   break;
	   case enFANCtrlModeState_AUTO_ON:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   if (FALSE == bIsSpeedAvailable() || FALSE == bIsCurrentAvailable())
				   {
					   // TODO: setting error flags
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_FAILED);
				   }
				   else
				   {
					   // set requested fan speed
					   if (OSAL_C_U8_MAX != u8GetReqFanSpeed(_enMode))
					   {
						   if (u8GetReqFanSpeed(_enMode) != u8GetSpeed() && u8GetReqFanSpeed(_enMode) != u8SetSpeed(u8GetReqFanSpeed(_enMode)))
						   {
							   // TODO: setting error flags
							   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_FAILED);
						   }
						   (tVoid)u8SetReqFanSpeed(_enMode, OSAL_C_U8_MAX);
						   (tVoid)u8SetReqFanSpeedStart(_enMode, OSAL_C_U8_MAX);
					   }
					   // handle minimum fan speed
					   if (0 == u8GetSpeed())
					   {
						   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_OFF);
					   }
					   else if (HC_FANCONTROL_FANSPEED_MIN >= u8GetSpeed())
					   {
						   if (HC_FANCONTROL_FANSPEED_OFF != u8SetSpeedStart(HC_FANCONTROL_FANSPEED_OFF))
						   {
							   // TODO: setting error flags
							   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_FAILED);
						   }
						   else if (HC_FANCONTROL_FANSPEED_OFF != u8SetSpeed(HC_FANCONTROL_FANSPEED_OFF))
						   {
							   // TODO: setting error flags
							   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_FAILED);
						   }
					   }
				   }
			   }
	   	   }
		   break;
	   case enFANCtrlModeState_AUTO_FAILED:
	   	   {
			   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_UNDEF);
			   (tVoid)enSetControlModeEvent(enFANCtrlModeEvent_AUTO_SELFTEST_START);
	   	   }
		   break;
	   case enFANCtrlModeState_FINISHED:
	   	   {
	   		   // do nothing
	   	   }
		   break;
	   default:
		   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_INIT);
		   break;
	   }
	   ++_u32LoopCount;
	   ETG_TRACE_USR3(("vSM_ControlMode_AUTO (loop %d): Mode(%u -> %u), Event(%u -> %u)",
			   _u32LoopCount, ETG_ENUM(HC_FANMODE, _enMode), ETG_ENUM(HC_FANMODE, enGetControlMode()),
			   ETG_ENUM(HC_FANMODEEVENT, _enEvent), ETG_ENUM(HC_FANMODEEVENT, enGetControlModeEvent()) ));
	   ETG_TRACE_USR3(("vSM_ControlMode_AUTO (loop %d): State(%u -> %u), TimeIndex(%d -> %d)",
			   _u32LoopCount, ETG_ENUM(HC_FANMODESTATE, _enState), ETG_ENUM(HC_FANMODESTATE, enGetControlModeState(_enMode)),
			   _u32TimeIndex, u32GetControlModeStateTimeIndex(_enMode) ));
   } while (bSM_ControlModeState_Looping(_u32LoopCount, _enMode, _enState));
}

/*******************************************************************************
*
* FUNCTION:    vSM_ControlMode_AUTO_SELFTEST()
*
* DESCRIPTION: state machine of FAN control mode AUTO_SELFTEST
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN::vSM_ControlMode_AUTO_SELFTEST(tVoid)
{
   ETG_TRACE_USR4(("vSM_ControlMode_AUTO_SELFTEST() entered."));
   // current control mode
   tenFANCtrlMode _enMode = enFANCtrlMode_AUTO_SELFTEST;
   // current control mode event
   tenFANCtrlModeEvent _enEvent = enGetControlModeEvent();
   tenFANCtrlModeState _enState = enGetControlModeState(_enMode);
   // time index
   tU32 _u32TimeIndex = u32GetControlModeStateTimeIndex(_enMode);
   (tVoid)u32SetControlModeStateTimeIndex(_enMode, u32GetNextTimeIndex(_u32TimeIndex));
   tU32 _u32LoopCount = 0;
   do
   {
	   // next state
	   _enState = enGetControlModeState(_enMode);
	   // state change
	   switch (_enState)
	   {
	   case enFANCtrlModeState_INIT:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   (tVoid)u32SetControlModeStateTimeIndex(_enMode, HC_FANCONTROL_DELAYTIMECYCLE_WAIT4AVAIL);
				   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_WAIT4AVAIL);
			   }
	   	   }
		   break;

	   case enFANCtrlModeState_AUTO_SELFTEST_WAIT4AVAIL:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   if (FALSE == bIsSpeedAvailable() || FALSE == bIsCurrentAvailable())
				   {
					   if (0 == u32GetControlModeStateTimeIndex(_enMode))
					   {
						   ETG_TRACE_USR1(("vSM_ControlMode_AUTO_SELFTEST(): timeout availability ... speed_if = %u, current_if = %u, power_if = %u",
								bIsSpeedAvailable(), bIsCurrentAvailable(), bIsPowerAvailable() ));
						   // TODO: setting error flags
						   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_FAILED);
					   }
				   }
				   else
				   {
					   // required interfaces available
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_OFF);
				   }
			   }
	   	   }
		   break;
	   case enFANCtrlModeState_AUTO_SELFTEST_OFF:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   tU8 _u8ReqFanSpeed = HC_FANCONTROL_FANSPEED_OFF;
				   // set required start fan speed
				   if (_u8ReqFanSpeed != u8SetSpeedStart(_u8ReqFanSpeed))
				   {
					   ETG_TRACE_USR1(("vSM_ControlMode_AUTO_SELFTEST(): setting of required start fan speed failed ... speed_start = %u",
							_u8ReqFanSpeed ));
					   // TODO: setting error flags
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_FAILED);
				   }
				   // set required fan speed to 0%
				   else if (_u8ReqFanSpeed != u8SetSpeed(_u8ReqFanSpeed))
				   {
					   ETG_TRACE_USR1(("vSM_ControlMode_AUTO_SELFTEST(): setting of required fan speed failed ... speed_start = %u",
							_u8ReqFanSpeed ));
					   // TODO: setting error flags
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_FAILED);
				   }
				   else
				   {
					   (tVoid)u32SetControlModeStateTimeIndex(_enMode, HC_FANCONTROL_DELAYTIMECYCLE_OFF);
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_WAIT4OFF);
				   }
			   }
	   	   }
	   	   break;
	   case enFANCtrlModeState_AUTO_SELFTEST_WAIT4OFF:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   if (HC_FANCONTROL_CURRENT_MIN < u16GetCurrent())
				   {
					   if (0 == u32GetControlModeStateTimeIndex(_enMode))
					   {
						   ETG_TRACE_USR1(("vSM_ControlMode_AUTO_SELFTEST(): timeout speed OFF ... current = %u mA",
								u16GetCurrent() ));
						   // TODO: setting error flags
						   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_FAILED);
					   }
				   }
				   else
				   {
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_STARTUP);
				   }
			   }
	   	   }
		   break;
	   case enFANCtrlModeState_AUTO_SELFTEST_STARTUP:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   tU8 _u8ReqFanSpeed = HC_FANCONTROL_FANSPEED_STARTUP;
				   // set required start fan speed
				   if (_u8ReqFanSpeed != u8SetSpeedStart(_u8ReqFanSpeed))
				   {
					   ETG_TRACE_USR1(("vSM_ControlMode_AUTO_SELFTEST(): setting of required start fan speed failed ... speed_start = %u",
							_u8ReqFanSpeed ));
					   // TODO: setting error flags
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_FAILED);
				   }
				   // set required fan speed to 0%
				   else if (_u8ReqFanSpeed != u8SetSpeed(_u8ReqFanSpeed))
				   {
					   ETG_TRACE_USR1(("vSM_ControlMode_AUTO_SELFTEST(): setting of required fan speed failed ... speed_start = %u",
							_u8ReqFanSpeed ));
					   // TODO: setting error flags
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_FAILED);
				   }
				   else if (_u8ReqFanSpeed == u8GetSpeed())
				   {
					   (tVoid)u32SetControlModeStateTimeIndex(_enMode, HC_FANCONTROL_DELAYTIMECYCLE_ON);
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_WAIT4STARTUP);
				   }
			   }
	   	   }
	   	   break;
	   case enFANCtrlModeState_AUTO_SELFTEST_WAIT4STARTUP:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   if (HC_FANCONTROL_CURRENT_UNPLUGGED >= u16GetCurrentAve())
				   {
					   if (0 == u32GetControlModeStateTimeIndex(_enMode))
					   {
						   ETG_TRACE_USR1(("vSM_ControlMode_AUTO_SELFTEST(): timeout speed %u percent ... current = %u mA",
								   HC_FANCONTROL_FANSPEED_STARTUP, u16GetCurrent() ));
						   // TODO: setting error flags
						   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_FAILED);
					   }
				   }
				   else if (0 == u32GetControlModeStateTimeIndex(_enMode))
				   {
					   m_u16LastMeasuredCurrent = u16GetCurrentAve();
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_MAX);
				   }
			   }
	   	   }
		   break;
	   case enFANCtrlModeState_AUTO_SELFTEST_MAX:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   tU8 _u8ReqFanSpeed = HC_FANCONTROL_FANSPEED_MAX;
				   // set required fan speed to 0%
				   if (_u8ReqFanSpeed != u8SetSpeed(_u8ReqFanSpeed))
				   {
					   ETG_TRACE_USR1(("vSM_ControlMode_AUTO_SELFTEST(): setting of required fan speed failed ... speed_start = %u",
							_u8ReqFanSpeed ));
					   // TODO: setting error flags
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_FAILED);
				   }
				   else if (_u8ReqFanSpeed == u8GetSpeed())
				   {
					   (tVoid)u32SetControlModeStateTimeIndex(_enMode, HC_FANCONTROL_DELAYTIMECYCLE_ON);
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_WAIT4MAX);
				   }
			   }
	   	   }
	   	   break;
	   case enFANCtrlModeState_AUTO_SELFTEST_WAIT4MAX:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   if (HC_FANCONTROL_CURRENT_UNPLUGGED >= u16GetCurrentAve())
				   {
					   if (0 == u32GetControlModeStateTimeIndex(_enMode))
					   {
						   ETG_TRACE_USR1(("vSM_ControlMode_AUTO_SELFTEST(): timeout speed 100 percent ... current = %u mA",
								u16GetCurrent() ));
						   // TODO: setting error flags
						   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_FAILED);
					   }
				   }
				   else if (m_u16LastMeasuredCurrent >= u16GetCurrentAve())
				   {
					   if (0 == u32GetControlModeStateTimeIndex(_enMode))
					   {
						   ETG_TRACE_USR1(("vSM_ControlMode_AUTO_SELFTEST(): timeout speed 100 percent ... current = %u mA, current_startup = %u mA ",
								u16GetCurrent(), m_u16LastMeasuredCurrent ));
						   // TODO: setting error flags
						   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_FAILED);
					   }
				   }
				   else if (0 == u32GetControlModeStateTimeIndex(_enMode))
				   {
					   m_u16LastMeasuredCurrent = u16GetCurrentAve();
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_MIN);
				   }
			   }
	   	   }
		   break;
	   case enFANCtrlModeState_AUTO_SELFTEST_MIN:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   tU8 _u8ReqFanSpeed = HC_FANCONTROL_FANSPEED_MIN;
				   // set required fan speed to minimum
				   if (_u8ReqFanSpeed != u8SetSpeed(_u8ReqFanSpeed))
				   {
					   ETG_TRACE_USR1(("vSM_ControlMode_AUTO_SELFTEST(): setting of required fan speed failed ... speed_start = %u ",
							_u8ReqFanSpeed ));
					   // TODO: setting error flags
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_FAILED);
				   }
				   else if (_u8ReqFanSpeed == u8GetSpeed())
				   {
					   (tVoid)u32SetControlModeStateTimeIndex(_enMode, HC_FANCONTROL_DELAYTIMECYCLE_ON);
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_WAIT4MIN);
				   }
			   }
	   	   }
	   	   break;
	   case enFANCtrlModeState_AUTO_SELFTEST_WAIT4MIN:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   if (HC_FANCONTROL_CURRENT_UNPLUGGED >= u16GetCurrentAve())
				   {
					   if (0 == u32GetControlModeStateTimeIndex(_enMode))
					   {
						   ETG_TRACE_USR1(("vSM_ControlMode_AUTO_SELFTEST(): timeout minimum speed ... current = %u mA ",
								u16GetCurrent() ));
						   // TODO: setting error flags
						   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_FAILED);
					   }
				   }
				   else if (m_u16LastMeasuredCurrent <= u16GetCurrentAve())
				   {
					   if (0 == u32GetControlModeStateTimeIndex(_enMode))
					   {
						   ETG_TRACE_USR1(("vSM_ControlMode_AUTO_SELFTEST(): timeout minimum speed ... current = %u mA, current_max = %u mA ",
								u16GetCurrent(), m_u16LastMeasuredCurrent ));
						   // TODO: setting error flags
						   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_FAILED);
					   }
				   }
				   else if (0 == u32GetControlModeStateTimeIndex(_enMode))
				   {
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_AUTO_SELFTEST_SUCCESS);
				   }
			   }
	   	   }
		   break;
	   case enFANCtrlModeState_AUTO_SELFTEST_SUCCESS:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_INIT);
				   (tVoid)enSetControlModeEvent(enFANCtrlModeEvent_AUTO_START);
			   }
	   	   }
		   break;
	   case enFANCtrlModeState_AUTO_SELFTEST_FAILED:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   // TODO: set ITCs
				   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_FINISHED);
				   (tVoid)enSetControlModeEvent(enFANCtrlModeEvent_STOPPED_START);
			   }
	   	   }
		   break;
	   case enFANCtrlModeState_FINISHED:
	   	   {
	   		   // do nothing
	   	   }
		   break;
	   default:
	   	   {
	   		   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_INIT);
	   	   }
		   break;
	   }
	   ++_u32LoopCount;
	   ETG_TRACE_USR3(("vSM_ControlMode_AUTO_SELFTEST (loop %d): Mode(%u -> %u), Event(%u -> %u)",
			   _u32LoopCount, ETG_ENUM(HC_FANMODE, _enMode), ETG_ENUM(HC_FANMODE, enGetControlMode()),
			   ETG_ENUM(HC_FANMODEEVENT, _enEvent), ETG_ENUM(HC_FANMODEEVENT, enGetControlModeEvent()) ));
	   ETG_TRACE_USR3(("vSM_ControlMode_AUTO_SELFTEST (loop %d): State(%u -> %u), TimeIndex(%d -> %d)",
			   _u32LoopCount, ETG_ENUM(HC_FANMODESTATE, _enState), ETG_ENUM(HC_FANMODESTATE, enGetControlModeState(_enMode)),
			   _u32TimeIndex, u32GetControlModeStateTimeIndex(_enMode) ));
   } while (bSM_ControlModeState_Looping(_u32LoopCount, _enMode, _enState));
}

/*******************************************************************************
*
* FUNCTION:    vSM_ControlMode_AUTO_MEASURE()
*
* DESCRIPTION: state machine of FAN control mode AUTO_MEASURE
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN::vSM_ControlMode_AUTO_MEASURE(tVoid)
{
   ETG_TRACE_USR4(("vSM_ControlMode_AUTO_MEASURE() entered."));
   // current control mode
   tenFANCtrlMode _enMode = enFANCtrlMode_AUTO_MEASURE;
   // current control mode event
   tenFANCtrlModeEvent _enEvent = enGetControlModeEvent();
   tenFANCtrlModeState _enState = enGetControlModeState(_enMode);
   // time index
   tU32 _u32TimeIndex = u32GetControlModeStateTimeIndex(_enMode);
   (tVoid)u32SetControlModeStateTimeIndex(_enMode, u32GetNextTimeIndex(_u32TimeIndex));
   tU32 _u32LoopCount = 0;
   do
   {
	   // next state
	   _enState = enGetControlModeState(_enMode);
	   // state change
	   switch (_enState)
	   {
	   case enFANCtrlModeState_INIT:
	   	   {
	   		   // TODO: implement me
	   	   }
		   break;
	   case enFANCtrlModeState_FINISHED:
	   	   {
	   		   // do nothing
	   	   }
		   break;
	   default:
	   	   {
	   		   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_INIT);
	   	   }
		   break;
	   }
	   ++_u32LoopCount;
	   ETG_TRACE_USR3(("vSM_ControlMode_AUTO_MEASURE (loop %d): Mode(%u -> %u), Event(%u -> %u)",
			   _u32LoopCount, ETG_ENUM(HC_FANMODE, _enMode), ETG_ENUM(HC_FANMODE, enGetControlMode()),
			   ETG_ENUM(HC_FANMODEEVENT, _enEvent), ETG_ENUM(HC_FANMODEEVENT, enGetControlModeEvent()) ));
	   ETG_TRACE_USR3(("vSM_ControlMode_AUTO_MEASURE (loop %d): State(%u -> %u), TimeIndex(%d -> %d)",
			   _u32LoopCount, ETG_ENUM(HC_FANMODESTATE, _enState), ETG_ENUM(HC_FANMODESTATE, enGetControlModeState(_enMode)),
			   _u32TimeIndex, u32GetControlModeStateTimeIndex(_enMode) ));
   } while (bSM_ControlModeState_Looping(_u32LoopCount, _enMode, _enState));
}

/*******************************************************************************
*
* FUNCTION:    vSM_ControlMode_MANUAL()
*
* DESCRIPTION: state machine of FAN control mode MANUAL
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN::vSM_ControlMode_MANUAL(tVoid)
{
   ETG_TRACE_USR4(("vSM_ControlMode_MANUAL() entered."));
   // current control mode
   tenFANCtrlMode _enMode = enFANCtrlMode_MANUAL;
   // current control mode event
   tenFANCtrlModeEvent _enEvent = enGetControlModeEvent();
   tenFANCtrlModeState _enState = enGetControlModeState(_enMode);
   // time index
   tU32 _u32TimeIndex = u32GetControlModeStateTimeIndex(_enMode);
   (tVoid)u32SetControlModeStateTimeIndex(_enMode, u32GetNextTimeIndex(_u32TimeIndex));
   tU32 _u32LoopCount = 0;
   do
   {
	   // next state
	   _enState = enGetControlModeState(_enMode);
	   // state change
	   switch (_enState)
	   {
	   case enFANCtrlModeState_INIT:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   if (0 < u8GetSpeed())
				   {
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_MANUAL_ON);
				   }
				   else
				   {
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_MANUAL_OFF);
				   }
			   }
	   	   }
		   break;
	   case enFANCtrlModeState_MANUAL_OFF:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   tU8 _u8Speed = OSAL_C_U8_MAX;
				   tU8 _u8SpeedStart = OSAL_C_U8_MAX;
				   // extract requested states
				   if (OSAL_C_U8_MAX != u8GetReqFanSpeed(_enMode))
				   {
					   _u8Speed = u8GetReqFanSpeed(_enMode);
				   }
				   if (OSAL_C_U8_MAX != u8GetReqFanSpeedStart(_enMode))
				   {
					   _u8SpeedStart = u8GetReqFanSpeed(_enMode);
				   }
				   if (OSAL_C_U8_MAX == _u8Speed)
				   {
					   if (enFANPowerState_ON == enGetReqFanPowerState(_enMode))
					   {
						   _u8Speed = HC_FANCONTROL_FANSPEED_MAX;
					   }
					   else if (enFANPowerState_OFF == enGetReqFanPowerState(_enMode))
					   {
						   _u8Speed = 0;
					   }
				   }
				   if (OSAL_C_U8_MAX != _u8SpeedStart)
				   {
					   (tVoid)u8SetSpeedStart(_u8SpeedStart);
				   }
				   if (OSAL_C_U8_MAX != _u8Speed)
				   {
					   (tVoid)u8SetSpeedStart(_u8SpeedStart);
					   (tVoid)u8SetSpeed(_u8Speed);
				   }
				   // delete request
				   (tVoid)enSetReqFanPowerState(_enMode,enFANPowerState_UNDEF);
				   (tVoid)u8SetReqFanSpeed(_enMode, OSAL_C_U8_MAX);
				   (tVoid)u8SetReqFanSpeedStart(_enMode, OSAL_C_U8_MAX);
				   // change to state ON
				   if (0 < u8GetSpeed())
				   {
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_MANUAL_ON);
				   }
			   }
	   	   }
		   break;
	   case enFANCtrlModeState_MANUAL_ON:
	   	   {
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   tU8 _u8Speed = OSAL_C_U8_MAX;
				   tU8 _u8SpeedStart = OSAL_C_U8_MAX;
				   // extract requested states
				   if (OSAL_C_U8_MAX != u8GetReqFanSpeed(_enMode))
				   {
					   _u8Speed = u8GetReqFanSpeed(_enMode);
				   }
				   if (OSAL_C_U8_MAX != u8GetReqFanSpeedStart(_enMode))
				   {
					   _u8SpeedStart = u8GetReqFanSpeed(_enMode);
				   }
				   if (OSAL_C_U8_MAX == _u8Speed)
				   {
					   if (enFANPowerState_ON == enGetReqFanPowerState(_enMode))
					   {
						   _u8Speed = HC_FANCONTROL_FANSPEED_MAX;
					   }
					   else if (enFANPowerState_OFF == enGetReqFanPowerState(_enMode))
					   {
						   _u8Speed = 0;
					   }
				   }
				   if (OSAL_C_U8_MAX != _u8SpeedStart)
				   {
					   (tVoid)u8SetSpeedStart(_u8SpeedStart);
				   }
				   if (OSAL_C_U8_MAX != _u8Speed)
				   {
					   (tVoid)u8SetSpeedStart(_u8SpeedStart);
					   (tVoid)u8SetSpeed(_u8Speed);
				   }
				   // delete request
				   (tVoid)enSetReqFanPowerState(_enMode,enFANPowerState_UNDEF);
				   (tVoid)u8SetReqFanSpeed(_enMode, OSAL_C_U8_MAX);
				   (tVoid)u8SetReqFanSpeedStart(_enMode, OSAL_C_U8_MAX);
				   // change to state OFF
				   if (0 == u8GetSpeed())
				   {
					   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_MANUAL_OFF);
				   }
			   }
	   	   }
		   break;
	   case enFANCtrlModeState_FINISHED:
	   	   {
	   		   // do nothing
	   	   }
		   break;
	   default:
		   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_INIT);
		   break;
	   }
	   ++_u32LoopCount;
	   ETG_TRACE_USR3(("vSM_ControlMode_MANUAL (loop %d): Mode(%u -> %u), Event(%u -> %u)",
			   _u32LoopCount, ETG_ENUM(HC_FANMODE, _enMode), ETG_ENUM(HC_FANMODE, enGetControlMode()),
			   ETG_ENUM(HC_FANMODEEVENT, _enEvent), ETG_ENUM(HC_FANMODEEVENT, enGetControlModeEvent()) ));
	   ETG_TRACE_USR3(("vSM_ControlMode_MANUAL (loop %d): State(%u -> %u), TimeIndex(%d -> %d)",
			   _u32LoopCount, ETG_ENUM(HC_FANMODESTATE, _enState), ETG_ENUM(HC_FANMODESTATE, enGetControlModeState(_enMode)),
			   _u32TimeIndex, u32GetControlModeStateTimeIndex(_enMode) ));
   } while (bSM_ControlModeState_Looping(_u32LoopCount, _enMode, _enState));
}

/*******************************************************************************
*
* FUNCTION:    vSM_ControlMode_STOPPED()
*
* DESCRIPTION: state machine of FAN control mode STOPPED
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN::vSM_ControlMode_STOPPED(tVoid)
{
   ETG_TRACE_USR4(("vSM_ControlMode_STOPPED() entered."));
   // current control mode
   tenFANCtrlMode _enMode = enFANCtrlMode_STOPPED;
   // current control mode event
   tenFANCtrlModeEvent _enEvent = enGetControlModeEvent();
   tenFANCtrlModeState _enState = enGetControlModeState(_enMode);
   // time index
   tU32 _u32TimeIndex = u32GetControlModeStateTimeIndex(_enMode);
   (tVoid)u32SetControlModeStateTimeIndex(_enMode, u32GetNextTimeIndex(_u32TimeIndex));
   tU32 _u32LoopCount = 0;
   do
   {
	   // next state
	   _enState = enGetControlModeState(_enMode);
	   // state change
	   switch (_enState)
	   {
	   case enFANCtrlModeState_INIT:
	   	   {
	   		   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_STOPPED_OFF);
	   	   }
		   break;
	   case enFANCtrlModeState_STOPPED_OFF:
	   	   {
			   tU8 _u8ReqFanSpeed = HC_FANCONTROL_FANSPEED_OFF;
			   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
			   {
				   // set required start fan speed
				   if (_u8ReqFanSpeed != u8SetSpeedStart(_u8ReqFanSpeed))
				   {
					   ETG_TRACE_USR1(("vSM_ControlMode_STOPPED(): setting of required start fan speed failed ... speed_start = %u",
							_u8ReqFanSpeed ));
				   }
				   // set required fan speed to 0%
				   else if (_u8ReqFanSpeed != u8SetSpeed(_u8ReqFanSpeed))
				   {
					   ETG_TRACE_USR1(("vSM_ControlMode_STOPPED(): setting of required fan speed failed ... speed_start = %u",
							_u8ReqFanSpeed ));
				   }
				   (tVoid)u32SetControlModeStateTimeIndex(_enMode, HC_FANCONTROL_DELAYTIMECYCLE_OFF);
		   		   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_STOPPED_WAIT4OFF);
			   }
	   	   }
		   break;
	   case enFANCtrlModeState_STOPPED_WAIT4OFF:
	   	   {
	   		   if (0 == u32GetControlModeStateTimeIndex(_enMode))
	   			   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_FINISHED);
	   	   }
		   break;
	   case enFANCtrlModeState_FINISHED:
	   	   {
			   if (0 < u8GetSpeed())
			   {
				   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_STOPPED_OFF);
			   }
	   	   }
		   break;
	   default:
	   	   {
	   		   (tVoid)enSetControlModeState(_enMode, enFANCtrlModeState_INIT);
	   	   }
		   break;
	   }
	   ++_u32LoopCount;
	   ETG_TRACE_USR3(("vSM_ControlMode_STOPPED (loop %d): Mode(%u -> %u), Event(%u -> %u)",
			   _u32LoopCount, ETG_ENUM(HC_FANMODE, _enMode), ETG_ENUM(HC_FANMODE, enGetControlMode()),
			   ETG_ENUM(HC_FANMODEEVENT, _enEvent), ETG_ENUM(HC_FANMODEEVENT, enGetControlModeEvent()) ));
	   ETG_TRACE_USR3(("vSM_ControlMode_STOPPED (loop %d): State(%u -> %u), TimeIndex(%d -> %d)",
			   _u32LoopCount, ETG_ENUM(HC_FANMODESTATE, _enState), ETG_ENUM(HC_FANMODESTATE, enGetControlModeState(_enMode)),
			   _u32TimeIndex, u32GetControlModeStateTimeIndex(_enMode) ));
   } while (bSM_ControlModeState_Looping(_u32LoopCount, _enMode, _enState));
}


/*******************************************************************************
*
* FUNCTION:    vHandle_ControlMode()
*
* DESCRIPTION: handles current control mode
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN::vHandle_ControlMode(tVoid)
{
   ETG_TRACE_USR4(("vHandle_ControlMode() entered."));
   switch (enGetControlMode())
   {
   case enFANCtrlMode_INIT:
   	   {
   		   vSM_ControlMode_INIT();
   	   }
	   break;
   case enFANCtrlMode_AUTO:
   	   {
   		   vSM_ControlMode_AUTO();
   	   }
	   break;
   case enFANCtrlMode_AUTO_SELFTEST:
	   {
		   vSM_ControlMode_AUTO_SELFTEST();
	   }
	   break;
   case enFANCtrlMode_AUTO_MEASURE:
	   {
		   vSM_ControlMode_AUTO_MEASURE();
	   }
	   break;
   case enFANCtrlMode_MANUAL:
	   {
		   vSM_ControlMode_MANUAL();
	   }
	   break;
   case enFANCtrlMode_STOPPED:
	   {
		   vSM_ControlMode_STOPPED();
	   }
	   break;
   default:
	   break;
   }
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo_FAN()
*
* DESCRIPTION: trace attributes of FAN
*
* PARAMETER:   const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN::vTraceInfo_FAN(const char* strIdentifier) const
{
   ETG_TRACE_USR4(("vTraceInfo_FAN() entered."));

	ETG_TRACE_USR2(("%26s: <<< FAN >>> ", strIdentifier ));
	// common fan state information
	ETG_TRACE_USR2(("%26s: %30s=%u ",       strIdentifier, "        FAN_CUR_MODE",
			ETG_ENUM(HC_FANMODE, enGetControlMode()) ));
	ETG_TRACE_USR2(("%26s: %30s=%u ",       strIdentifier, "  FAN_CUR_MODE_EVENT",
			ETG_ENUM(HC_FANMODEEVENT, enGetControlModeEvent()) ));
	ETG_TRACE_USR2(("%26s: %30s=%u ",       strIdentifier, "   FAN_CUR_MODESTATE",
			ETG_ENUM(HC_FANMODESTATE, enGetControlModeState()) ));
	ETG_TRACE_USR2(("%26s: %30s=%d [time cycles] ",strIdentifier, "FAN_CUR_MODESTATE_TM",
			u32GetControlModeStateTimeIndex() ));
	ETG_TRACE_USR2(("%26s: %30s=%u ",       strIdentifier, "  FAN_REQ_MODE_EVENT",
			ETG_ENUM(HC_FANMODEEVENT, enGetReqFanControlEvent()) ));
	ETG_TRACE_USR2(("%26s: %30s=%u ",       strIdentifier, "  FAN_REQ_POWERSTATE",
			ETG_ENUM(HC_FANPWRSTATE, enGetReqFanPowerState()) ));
	ETG_TRACE_USR2(("%26s: %30s=%d [perc] ",strIdentifier, "       FAN_REQ_SPEED",
			u8GetReqFanSpeed() ));
	ETG_TRACE_USR2(("%26s: %30s=%d [perc] ",strIdentifier, " FAN_REQ_SPEED_START",
			u8GetReqFanSpeedStart() ));
	// power state
	ETG_TRACE_USR2(("%26s: %30s=0x%08x ",   strIdentifier, "FAN_PWR_STATE_OBJREF",
			poGetPower() ));
	if (OSAL_NULL != poGetPower())
	{
		poGetPower()->vTraceInfo_FANPwrState(strIdentifier);
	}
	// speed
	ETG_TRACE_USR2(("%26s: %30s=0x%08x ",   strIdentifier, "    FAN_SPEED_OBJREF",
			poGetSpeed() ));
	if (OSAL_NULL != poGetSpeed())
	{
		poGetSpeed()->vTraceInfo_FANSpeed(strIdentifier);
	}
	for (tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_au8SensorFanSpeed) > _u16Idx; ++_u16Idx)
	{
		tChar _strAttrName[60] = "";
		(tVoid)memset((tVoid*)_strAttrName, OSAL_NULL, sizeof(_strAttrName));
		(tVoid)sprintf(_strAttrName, "     FAN_SPEED_TS_%02u", _u16Idx);
		ETG_TRACE_USR2(("%26s: %30s=%u [perc] {sensor: %u} ",strIdentifier, _strAttrName,
				u8GetSpeed((tenThermalSensorID)_u16Idx),
				ETG_ENUM(HC_SensorID, _u16Idx) ));
	}
	// current
	ETG_TRACE_USR2(("%26s: %30s=0x%08x ",   strIdentifier, "  FAN_CURRENT_OBJREF",
			poGetCurrent() ));
	if (OSAL_NULL != poGetCurrent())
	{
		poGetCurrent()->vTraceInfo_FANCurrent(strIdentifier);
	}
	// statistics
	vTraceInfo_FAN_Statistics(strIdentifier);
}

/*******************************************************************************
*
* FUNCTION:    enGetControlMode()
*
* DESCRIPTION: returns the current FAN control mode
*
* PARAMETER:   None
*
* RETURNVALUE: tenFANCtrlMode
*
*******************************************************************************/
tenFANCtrlMode hc_tclFAN::enGetControlMode(tVoid) const
{
   ETG_TRACE_USR4(("enGetControlMode() entered."));

   return m_sControlModeInfo.enMode;
}

/*******************************************************************************
*
* FUNCTION:    enGetControlMode()
*
* DESCRIPTION: returns the FAN control mode depending of specified FAN control
*              mode event
*
*              following base function depending of parameter enEvent:
*              - enFANCtrlModeEvent_MAX			returns assigned mode of current
*                                               event
*              - enFANCtrlModeEvent_DOENTCARE	returns current mode
*
* PARAMETER:   tenFANCtrlModeEvent enEvent
*
* RETURNVALUE: tenFANCtrlMode
*
*******************************************************************************/
tenFANCtrlMode hc_tclFAN::enGetControlMode(tenFANCtrlModeEvent enEvent) const
{
   ETG_TRACE_USR4(("enGetControlMode(event: %d) entered.", enEvent ));

   tenFANCtrlModeEvent _enEvent = enEvent;
   if (enFANCtrlModeEvent_DOENTCARE == _enEvent)
   {
	   _enEvent = enGetControlModeEvent();
   }
   if (enFANCtrlModeEvent_MAX > _enEvent && enFANCtrlModeEvent_MANUAL_END <= _enEvent)
   {
	   return m_enConfigControlMode[_enEvent];
   }
   return enGetControlMode();
}

/*******************************************************************************
*
* FUNCTION:    enSetControlMode()
*
* DESCRIPTION: sets the FAN control mode and returns the new one
*
* PARAMETER:   tenFANCtrlMode enMode
*
* RETURNVALUE: tenFANCtrlMode
*
*******************************************************************************/
tenFANCtrlMode hc_tclFAN::enSetControlMode(tenFANCtrlMode enMode)
{
   ETG_TRACE_USR4(("enSetControlMode() entered."));

   if (enFANCtrlMode_MAX > enMode && enFANCtrlMode_INIT <= enMode)
   {
	   m_sControlModeInfo.enMode = enMode;
   }
   return enGetControlMode();
}

/*******************************************************************************
*
* FUNCTION:    enGetControlModeEvent()
*
* DESCRIPTION: returns the current FAN control mode event
*
* PARAMETER:   None
*
* RETURNVALUE: tenFANCtrlModeEvent
*
*******************************************************************************/
tenFANCtrlModeEvent hc_tclFAN::enGetControlModeEvent(tVoid) const
{
   ETG_TRACE_USR4(("enGetControlModeEvent() entered."));

   return m_sControlModeInfo.enEvent;
}

/*******************************************************************************
*
* FUNCTION:    enSetControlModeEvent()
*
* DESCRIPTION: sets FAN control mode event and returns new one
*
* PARAMETER:   None
*
* RETURNVALUE: tenFANCtrlModeEvent
*
*******************************************************************************/
tenFANCtrlModeEvent hc_tclFAN::enSetControlModeEvent(tenFANCtrlModeEvent enEvent)
{
   ETG_TRACE_USR4(("enSetControlModeEvent() entered."));

   if (enFANCtrlModeEvent_MAX > enEvent && enFANCtrlModeEvent_MANUAL_END <= enEvent)
   {
	   m_sControlModeInfo.enEvent = enEvent;
   }
   else if (enFANCtrlModeEvent_DOENTCARE == enEvent)
   {
	   m_sControlModeInfo.enEvent = enEvent;
   }
   return enGetControlModeEvent();
}

/*******************************************************************************
*
* FUNCTION:    u32GetControlModeTimeIndex()
*
* DESCRIPTION: returns the current FAN control mode time index [second]
*
* PARAMETER:   None
*
* RETURNVALUE: tenFANCtrlModeEvent
*
*******************************************************************************/
tU32 hc_tclFAN::u32GetControlModeTimeIndex(tVoid) const
{
   ETG_TRACE_USR4(("u32GetControlModeTimeIndex() entered."));

   return m_sControlModeInfo.u32TimeIndex;
}

/*******************************************************************************
*
* FUNCTION:    u32SetControlModeTimeIndex()
*
* DESCRIPTION: sets FAN control mode time index [second] and returns the new one
*
* PARAMETER:   None
*
* RETURNVALUE: tenFANCtrlModeEvent
*
*******************************************************************************/
tU32 hc_tclFAN::u32SetControlModeTimeIndex(tU32 u32TimeIndex)
{
   ETG_TRACE_USR4(("u32SetControlModeTimeIndex() entered."));

   m_sControlModeInfo.u32TimeIndex = u32TimeIndex;
   return u32GetControlModeTimeIndex();
}

/*******************************************************************************
*
* FUNCTION:    enGetControlModeState()
*
* DESCRIPTION: returns the state of specified FAN control mode
*
* PARAMETER:   tenFANCtrlMode enMode
*
* RETURNVALUE: tenFANCtrlModeState
*
*******************************************************************************/
tenFANCtrlModeState hc_tclFAN::enGetControlModeState(tenFANCtrlMode enMode) const
{
   ETG_TRACE_USR4(("enGetControlModeState() entered."));

   if(enFANCtrlMode_MAX > enMode && enFANCtrlMode_INIT <= enMode)
   {
	   return m_asControlModeStateInfo[enMode].enState;
   }
   return enFANCtrlModeState_UNDEF;
}

/*******************************************************************************
*
* FUNCTION:    enSetControlModeState()
*
* DESCRIPTION: sets the state of specified FAN control mode and returns the new one
*
* PARAMETER:   tenFANCtrlMode enMode
* 			   tenFANCtrlModeState enState
*
* RETURNVALUE: tenFANCtrlModeState
*
*******************************************************************************/
tenFANCtrlModeState hc_tclFAN::enSetControlModeState(tenFANCtrlMode enMode, tenFANCtrlModeState enState)
{
   ETG_TRACE_USR4(("enGetControlModeState() entered."));

   if(enFANCtrlMode_MAX > enMode && enFANCtrlMode_INIT <= enMode)
   {
	   if (enFANCtrlModeState_MAX > enState && enFANCtrlModeState_INIT <= enState)
	   {
		   m_asControlModeStateInfo[enMode].enState = enState;
	   }
   }
   return enGetControlModeState(enMode);
}

/*******************************************************************************
*
* FUNCTION:    u32GetControlModeStateTimeIndex()
*
* DESCRIPTION: returns the time index of specified FAN control mode
*
* PARAMETER:   tenFANCtrlMode enMode
*
* RETURNVALUE: tU32
*
*******************************************************************************/
tU32 hc_tclFAN::u32GetControlModeStateTimeIndex(tenFANCtrlMode enMode) const
{
   ETG_TRACE_USR4(("u32GetControlModeStateTimeIndex() entered."));

   if(enFANCtrlMode_MAX > enMode && enFANCtrlMode_INIT <= enMode)
   {
	   return m_asControlModeStateInfo[enMode].u32TimeIndex;
   }
   return 0;
}

/*******************************************************************************
*
* FUNCTION:    u32SetControlModeStateTimeIndex()
*
* DESCRIPTION: sets the time index of specified FAN control mode and returns the new one
*
* PARAMETER:   tenFANCtrlMode enMode
*              tU32 u32TimeIndex
*
* RETURNVALUE: tenFANCtrlModeEvent
*
*******************************************************************************/
tU32 hc_tclFAN::u32SetControlModeStateTimeIndex(tenFANCtrlMode enMode, tU32 u32TimeIndex)
{
   ETG_TRACE_USR4(("u32SetControlModeStateTimeIndex() entered."));

   if(enFANCtrlMode_MAX > enMode && enFANCtrlMode_INIT <= enMode)
   {
	   m_asControlModeStateInfo[enMode].u32TimeIndex = u32TimeIndex;
   }
   return u32GetControlModeStateTimeIndex(enMode);
}

/*******************************************************************************
*
* FUNCTION:    enGetPowerState()
*
* DESCRIPTION: returns the current FAN power state
*
* PARAMETER:   None
*
* RETURNVALUE: tenFANPowerState
*
*******************************************************************************/
tenFANPowerState hc_tclFAN::enGetPowerState(tVoid) const
{
   ETG_TRACE_USR4(("enGetPowerState() entered."));

   if (OSAL_NULL != poGetPower())
   {
	   return poGetPower()->enGetPwrStateAck();
   }
   return enFANPowerState_UNDEF;
}

/*******************************************************************************
*
* FUNCTION:    bIsPowerAvailable()
*
* DESCRIPTION: returns availability of FAN power state control
*
* PARAMETER:   None
*
* RETURNVALUE: tBool
*
*******************************************************************************/
tBool hc_tclFAN::bIsPowerAvailable(tVoid) const
{
   ETG_TRACE_USR4(("enGetPowerState() entered."));

   if (OSAL_NULL != poGetPower())
   {
	   return poGetPower()->bFanPowerAvail();
   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    u8GetSpeed()
*
* DESCRIPTION: returns the acknowledged fan speed
*
* PARAMETER:   None
*
* RETURNVALUE: tU8 [percent]
*
*******************************************************************************/
tU8 hc_tclFAN::u8GetSpeed(tVoid) const
{
   ETG_TRACE_USR4(("u8GetSpeed() entered."));

   if (OSAL_NULL != poGetSpeed())
   {
	   return poGetSpeed()->u8GetFANSpeedAck();
   }
   return 0;
}

/*******************************************************************************
*
* FUNCTION:    u8GetSpeed()
*
* DESCRIPTION: returns the fan speed of specified sensor
*
* PARAMETER:   tenThermalSensorID enSensorID
*
* RETURNVALUE: tU8 [percent]
*
*******************************************************************************/
tU8 hc_tclFAN::u8GetSpeed(tenThermalSensorID enSensorID) const
{
   ETG_TRACE_USR3(("u8GetSpeed(sensor: %u) entered.",
		   ETG_ENUM(HC_SensorID, enSensorID) ));

   if (enThermalSensorID_MAX == enSensorID)
   {
	   tU8 _u8Speed = 0;
	   for (tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_au8SensorFanSpeed) > _u16Idx; ++_u16Idx)
	   {
		   if ((100 >= m_au8SensorFanSpeed[_u16Idx]) && (_u8Speed < m_au8SensorFanSpeed[_u16Idx]))
			   _u8Speed = m_au8SensorFanSpeed[_u16Idx];
	   }
	   return _u8Speed;
   }
   else if (HC_C_U16_ARRAYELEMENTS(m_au8SensorFanSpeed) > (tU16)enSensorID)
   {
	   return m_au8SensorFanSpeed[enSensorID];
   }
   return 0;
}

/*******************************************************************************
*
* FUNCTION:    u8SetSpeed()
*
* DESCRIPTION: set new fan speed returns the fan speed was set
*
* PARAMETER:   None
*
* RETURNVALUE: tU8                         fan speed to set
*
*******************************************************************************/
tU8 hc_tclFAN::u8SetSpeed(tU8 u8Speed)
{
   ETG_TRACE_USR4(("u8SetSpeed(speed: %u) entered.", u8Speed));

   if (OSAL_NULL != poGetSpeed() && HC_FANCONTROL_FANSPEED_MAX >= u8Speed)
   {
	   return poGetSpeed()->u8SetFANSpeedReq(u8Speed);
   }
   return 0;
}

/*******************************************************************************
*
* FUNCTION:    u8SetSpeedStart()
*
* DESCRIPTION: set start fan speed returns the start fan speed was set
*
* PARAMETER:   None
*
* RETURNVALUE: tU8                         fan speed to set
*
*******************************************************************************/
tU8 hc_tclFAN::u8SetSpeedStart(tU8 u8Speed)
{
   ETG_TRACE_USR4(("u8SetSpeedStart(speed: %u) entered.", u8Speed));

   if (OSAL_NULL != poGetSpeed())
   {
	   return poGetSpeed()->u8SetFANSpeedReqStart(u8Speed);
   }
   return 0;
}

/*******************************************************************************
*
* FUNCTION:    bIsSpeedAvailable()
*
* DESCRIPTION: returns availability of FAN speed control
*
* PARAMETER:   None
*
* RETURNVALUE: tU8
*
*******************************************************************************/
tBool hc_tclFAN::bIsSpeedAvailable(tVoid) const
{
   ETG_TRACE_USR4(("bIsSpeedAvailable() entered."));

   if (OSAL_NULL != poGetSpeed())
   {
	   return poGetSpeed()->bGetFanSpeedAvail();
   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    u16GetCurrent()
*
* DESCRIPTION: returns the measured FAN current
*
* PARAMETER:   None
*
* RETURNVALUE: tU16
*
*******************************************************************************/
tU16 hc_tclFAN::u16GetCurrent(tVoid) const
{
   ETG_TRACE_USR4(("u16GetCurrent() entered."));

   if (OSAL_NULL != poGetCurrent())
   {
	   return poGetCurrent()->u16GetFANCurrent();
   }
   return 0;
}

/*******************************************************************************
*
* FUNCTION:    u16GetCurrentAve()
*
* DESCRIPTION: returns the measured average FAN current
*
* PARAMETER:   None
*
* RETURNVALUE: tU16
*
*******************************************************************************/
tU16 hc_tclFAN::u16GetCurrentAve(tVoid) const
{
   ETG_TRACE_USR4(("u16GetCurrentAve() entered."));

   if (OSAL_NULL != poGetCurrent())
   {
	   return poGetCurrent()->u16GetFANCurrentAve();
   }
   return 0;
}

/*******************************************************************************
*
* FUNCTION:    bIsSpeedAvailable()
*
* DESCRIPTION: returns availability of FAN current measurement
*
* PARAMETER:   None
*
* RETURNVALUE: tU8
*
*******************************************************************************/
tBool hc_tclFAN::bIsCurrentAvailable(tVoid) const
{
   ETG_TRACE_USR4(("bIsCurrentAvailable() entered."));

   if (OSAL_NULL != poGetCurrent())
   {
	   return poGetCurrent()->bGetFanCurrentAvail();
   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bSetReqFanControl()
*
* DESCRIPTION: set a new fan control request
*
* PARAMETER:   tenFANCtrlModeEvent enEvent
*              tenFANPowerState enPwrState
*              tU8 u8Speed
*              tU8 u8SpeedStart			     (default: 255 - dontcare)
*
* RETURNVALUE: 	tBool    {TRUE = successful | else = failed}
*
*******************************************************************************/
tBool hc_tclFAN::bSetReqFanControl(tenFANCtrlModeEvent enEvent, tenFANPowerState enPwrState, tU8 u8Speed, tU8 u8SpeedStart)
{
   ETG_TRACE_USR4(("bSetReqFanControl() entered."));

   tenFANCtrlMode _enMode = enGetControlMode(enSetReqFanControlEvent(enEvent));
   (tVoid)enSetReqFanPowerState(_enMode, enPwrState);
   (tVoid)u8SetReqFanSpeed(_enMode,u8Speed);
   (tVoid)u8SetReqFanSpeedStart(_enMode,u8SpeedStart);

   return TRUE;
}

/*******************************************************************************
*
* FUNCTION:    bSetReqFanControl()
*
* DESCRIPTION: set a new fan control request of specified control mode
*
* PARAMETER:   tenFANCtrlMode enMode
*              tenFANPowerState enPwrState
*              tU8 u8Speed
*              tU8 u8SpeedStart			     (default: 255 - dontcare)
*
* RETURNVALUE: 	tBool    {TRUE = successful | else = failed}
*
*******************************************************************************/
tBool hc_tclFAN::bSetReqFanControl(tenFANCtrlMode enMode, tenFANPowerState enPwrState, tU8 u8Speed, tU8 u8SpeedStart)
{
   ETG_TRACE_USR3(("bSetReqFanControl(mode: %d, power: %d, speed: %d, startspeed: %d) entered.",
		   enMode, enPwrState, u8Speed, u8SpeedStart ));

   return bSetReqFanControl(enThermalSensorID_UNDEF, enMode, enPwrState, u8Speed, u8SpeedStart);
}

/*******************************************************************************
*
* FUNCTION:    bSetReqFanControl()
*
* DESCRIPTION: set a new fan control request of specified control mode and
*              specified thermal sensor
*
* PARAMETER:   tenThermalSensorID enSensorID
*              tenFANCtrlMode enMode
*              tenFANPowerState enPwrState
*              tU8 u8Speed
*              tU8 u8SpeedStart			     (default: 255 - dontcare)
*
* RETURNVALUE: 	tBool    {TRUE = successful | else = failed}
*
*******************************************************************************/
tBool hc_tclFAN::bSetReqFanControl(tenThermalSensorID enSensorID,
		tenFANCtrlMode enMode, tenFANPowerState enPwrState, tU8 u8Speed, tU8 u8SpeedStart)
{
   ETG_TRACE_USR3(("bSetReqFanControl(mode: %d, power: %d, speed: %d, startspeed: %d) entered.",
		   enMode, enPwrState, u8Speed, u8SpeedStart ));

   tU8 _u8NewSpeed = u8Speed;
   if (HC_C_U16_ARRAYELEMENTS(m_au8SensorFanSpeed) > (tU16)enSensorID
       && enThermalSensorID_UNDEF < enSensorID && 100 >= u8Speed && enFANCtrlMode_AUTO == enMode)
   {
	   if (m_au8SensorFanSpeed[enSensorID] != u8Speed)
	   {
		   m_au8SensorFanSpeed[enSensorID] = u8Speed;
		   _u8NewSpeed = u8GetSpeed(enThermalSensorID_MAX);
		   if((OSAL_NULL != m_poSpeed) && (u8Speed != m_poSpeed->u8SetFANSpeedReq(enSensorID, u8Speed)))
		   {
			   ETG_TRACE_ERR(("bSetReqFanControl: speed %d percent of sensor %u could not set!",
					   u8Speed, ETG_ENUM(HC_SensorID, enSensorID) ));
		   }
	   }
   }
   (tVoid)enSetReqFanPowerState(enMode, enPwrState);
   (tVoid)u8SetReqFanSpeed(enMode,_u8NewSpeed);
   (tVoid)u8SetReqFanSpeedStart(enMode,u8SpeedStart);

   return TRUE;
}

/*******************************************************************************
*
* FUNCTION:    enGetReqFanPowerState()
*
* DESCRIPTION: returns the requested fan power state of specified fan control mode
*
* PARAMETER:   tenFANCtrlMode enMode
*
* RETURNVALUE: tenFANPowerState
*
*******************************************************************************/
tenFANPowerState hc_tclFAN::enGetReqFanPowerState(tenFANCtrlMode enMode) const
{
   ETG_TRACE_USR4(("enGetReqFanPowerState() entered."));

   if (enFANCtrlMode_MAX > enMode && enFANCtrlMode_INIT <= enMode)
   {
	   return m_asControlModeStateInfo[enMode].enReqPowerState;
   }
   return enFANPowerState_UNDEF;
}

/*******************************************************************************
*
* FUNCTION:    enSetReqFanPowerState()
*
* DESCRIPTION: sets the requested fan power state and returns new one of
*              specified fan control mode
*
* PARAMETER:   tenFANCtrlMode enMode
*              tenFANPowerState enPwrState
*
* RETURNVALUE: tenFANPowerState
*
*******************************************************************************/
tenFANPowerState hc_tclFAN::enSetReqFanPowerState(tenFANCtrlMode enMode, tenFANPowerState enPwrState)
{
   ETG_TRACE_USR3(("enSetReqFanPowerState(mode: %d, power: %d) entered.", enMode, enPwrState ));

   if (enFANCtrlMode_MAX > enMode && enFANCtrlMode_INIT <= enMode)
   {
	   if (enFANPowerState_OFF == enPwrState || enFANPowerState_ON == enPwrState)
	   {
		   m_asControlModeStateInfo[enMode].enReqPowerState = enPwrState;
	   }
   }
   return enGetReqFanPowerState(enMode);
}

/*******************************************************************************
*
* FUNCTION:    u8GetReqFanSpeed()
*
* DESCRIPTION: returns the requested fan speed of specified fan control mode
*
* PARAMETER:   tenFANCtrlMode enMode
*
* RETURNVALUE: tU8
*
*******************************************************************************/
tU8 hc_tclFAN::u8GetReqFanSpeed(tenFANCtrlMode enMode) const
{
   ETG_TRACE_USR4(("u8GetReqFanSpeed() entered."));

   if (enFANCtrlMode_MAX > enMode && enFANCtrlMode_INIT <= enMode)
   {
	   return m_asControlModeStateInfo[enMode].u8ReqFanSpeed;
   }
   return OSAL_C_U8_MAX;
}

/*******************************************************************************
*
* FUNCTION:    u8SetReqFanSpeed()
*
* DESCRIPTION: sets the requested fan speed and returns the new one of
*              specified fan control mode
*
* PARAMETER:   tenFANCtrlMode enMode
*
* RETURNVALUE: tU8
*
*******************************************************************************/
tU8 hc_tclFAN::u8SetReqFanSpeed(tenFANCtrlMode enMode, tU8 u8Speed)
{
   ETG_TRACE_USR3(("u8SetReqFanSpeed(mode: %d, speed: %d) entered.", enMode, u8Speed ));

   if (enFANCtrlMode_MAX > enMode && enFANCtrlMode_INIT <= enMode)
   {
	   if (100 >= u8Speed)
	   {
		   m_asControlModeStateInfo[enMode].u8ReqFanSpeed = u8Speed;
	   }
	   else if (OSAL_C_U8_MAX == u8Speed)
	   {
		   m_asControlModeStateInfo[enMode].u8ReqFanSpeed = u8Speed;
	   }
	   else
	   {
		   m_asControlModeStateInfo[enMode].u8ReqFanSpeed = 100;
	   }
   }
   return u8GetReqFanSpeed(enMode);
}

/*******************************************************************************
*
* FUNCTION:    u8GetReqFanSpeedStart()
*
* DESCRIPTION: returns the requested fan speed for start of specified fan control mode
*
* PARAMETER:   tenFANCtrlMode enMode
*
* RETURNVALUE: tU8
*
*******************************************************************************/
tU8 hc_tclFAN::u8GetReqFanSpeedStart(tenFANCtrlMode enMode) const
{
   ETG_TRACE_USR4(("bGetReqFanControl() entered."));

   if (enFANCtrlMode_MAX > enMode && enFANCtrlMode_INIT <= enMode)
   {
	   return m_asControlModeStateInfo[enMode].u8ReqFanSpeedStart;
   }
   return OSAL_C_U8_MAX;
}

/*******************************************************************************
*
* FUNCTION:    u8GetReqFanSpeedStart()
*
* DESCRIPTION: sets the the requested fan speed for start and returns the new one
*              of specified fan control mode
*
* PARAMETER:   tenFANCtrlMode enMode
*              tU8 u8Speed
*
* RETURNVALUE: tU8
*
*******************************************************************************/
tU8 hc_tclFAN::u8SetReqFanSpeedStart(tenFANCtrlMode enMode, tU8 u8Speed)
{
   ETG_TRACE_USR3(("u8SetReqFanSpeedStart(mode: %d, speed: %d) entered.", enMode, u8Speed ));

   if (enFANCtrlMode_MAX > enMode && enFANCtrlMode_INIT <= enMode)
   {
	   if (100 >= u8Speed)
	   {
		   m_asControlModeStateInfo[enMode].u8ReqFanSpeedStart = u8Speed;
	   }
	   else if (OSAL_C_U8_MAX == u8Speed)
	   {
		   m_asControlModeStateInfo[enMode].u8ReqFanSpeedStart = u8Speed;
	   }
	   else
	   {
		   m_asControlModeStateInfo[enMode].u8ReqFanSpeedStart = 100;
	   }
   }
   return u8GetReqFanSpeedStart(enMode);
}
