/************************************************************************
* FILE:        hc_ClientiMXtemp.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: class for client handler iMX temperature
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 06.03.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#ifndef OSAL_S_IMPORT_INTERFACE_GENERIC
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#endif //#ifndef OSAL_S_IMPORT_INTERFACE_GENERIC

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#ifndef HC_USE_OSAL
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#else
#endif

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#include "hc_tclSimuMode.h"
#include "hc_ClientiMXtemp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_ClientiMXtemp.cpp.trc.h"
#endif

// datapool access
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_fc_heatctrl_if.h"

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*************************************************************************
*
* FUNCTION:    hc_ClientiMXtemp()
*
* DESCRIPTION: constructor
*
* PARAMETER:   const hc_tclAppMain* poMainAppl: main - object of this application
*
* RETURNVALUE: none
*
*************************************************************************/
hc_ClientiMXtemp::hc_ClientiMXtemp(const hc_tclAppMain* poMainAppl)
: I_hc_ClientiMXtemp(poMainAppl)
{
   ETG_TRACE_USR4(("hc_ClientiMXtemp() entered."));

   _bAvailable = FALSE;
   // simulation mode
   m_poSimuMode = OSAL_NULL;
   // polling device properties
   dp_tclHeatCtrlDPIMXAdapter_Period _oCycleTime;
   m_u8PollingCycleTime = _oCycleTime.tGetData();
   if (0 == _oCycleTime.tGetData())
	   m_u8PollingCycleTime = HC_CLIENTIMXTEMP_CYCLETIME;
   dp_tclHeatCtrlDPIMXAdapter_Device _oDevice;
   if ((0 == _oDevice.u8GetData(m_strDevice, sizeof(m_strDevice)))
	   || (0 == OSAL_u32StringLength(m_strDevice)))
   {
	   (tVoid)OSAL_szStringNCopy(m_strDevice,HC_CLIENTIMXTEMP_DEVICE, sizeof(m_strDevice)-1);
   }
   // thermal information messages for all supported sensors
   (tVoid)memset((tVoid*)m_aoThermalInfo, OSAL_NULL, sizeof(m_aoThermalInfo));
   for (tU32 _u32Index = 0; (tU8)enClientiMXtemp_SensorID_MAX > _u32Index; ++_u32Index)
   {
	   m_aoThermalInfo[_u32Index].eCmd = hc_tclBaseIf::eNewThermalInfo_STANDARD;
	   (tVoid)OSAL_szStringNCopy(m_aoThermalInfo[_u32Index].strClassName,
			   "I_hc_tclThermalSensorControl", sizeof(m_aoThermalInfo[_u32Index].strClassName)-1);
	   m_aoThermalInfo[_u32Index].u.tThermalInfo.bSensorStateSupported = FALSE;
	   m_aoThermalInfo[_u32Index].u.tThermalInfo.bTemperatureSupported = TRUE;
	   (tVoid)OSAL_szStringNCopy(m_aoThermalInfo[_u32Index].u.tThermalInfo.strClassName,
			   HC_CLIENTIMXTEMP_IFNAME, sizeof(m_aoThermalInfo[_u32Index].u.tThermalInfo.strClassName)-1);
	   switch(_u32Index)
	   {
	   case enClientiMXtemp_SensorID_IMX:
		   m_aoThermalInfo[_u32Index].u.tThermalInfo.enSensorID = enThermalSensorID_IMX;
		   break;
	   default:
		   {
			   ETG_TRACE_USR4(("hc_ClientiMXtemp() unhandled sensor id %d", _u32Index));
		   }
		   break;
	   }
   }
   // trigger messages
   (tVoid)memset((tVoid*)&m_oTriggerIMXTemp, OSAL_NULL, sizeof(m_oTriggerIMXTemp));
   (tVoid)OSAL_szStringNCopy(m_oTriggerIMXTemp.strClassName, HC_CLIENTIMXTEMP_IFNAME, sizeof(m_oTriggerIMXTemp.strClassName)-1);
   m_oTriggerIMXTemp.eCmd = hc_tclBaseIf::eTriggerIMXTemp;
   // timer creation
   m_hTimerIMXTemp = OSAL_NULL;
   if (OSAL_OK != OSAL_s32TimerCreate( (OSAL_tpfCallback)cb_vTimerIMXTemp, (tVoid*)this, &m_hTimerIMXTemp))
   {
	   ETG_TRACE_FATAL(("hc_ClientiMXtemp: couldn't create timer to poll iMX temperature."));
   }
   else
   {
	   (tVoid)OSAL_s32TimerSetTime(m_hTimerIMXTemp, 0, 0);
   }
}


/*************************************************************************
*
* FUNCTION:    ~hc_ClientiMXtemp()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*************************************************************************/
hc_ClientiMXtemp::~hc_ClientiMXtemp()
{
   ETG_TRACE_USR4(("~hc_ClientiMXtemp() entered."));
   // delete timer
   if (OSAL_NULL != m_hTimerIMXTemp)
   {
	   (tVoid)OSAL_s32TimerSetTime(m_hTimerIMXTemp, 0, 0);
	   (tVoid)OSAL_s32TimerDelete(m_hTimerIMXTemp);
   }
   m_poSimuMode = OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION: 	vHandleMessage()
*
* DESCRIPTION: 	handler for internal messages
*
* PARAMETER:   	hc_tclBaseIf::TMsg* pMsg: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_ClientiMXtemp::vHandleMessage(hc_tclBaseIf::TMsg* pMsg)
{
   ETG_TRACE_USR4(("vHandleMessage() entered %u -> data: %d.", ETG_ENUM(HC_INT_MSG , (tU32)pMsg->eCmd), pMsg->u.u32Data));
   HC_NULL_POINTER_CHECK(pMsg);
   switch (pMsg->eCmd)
   {
	case hc_tclBaseIf::eUpdateSimuMode:
		if (FALSE == pMsg->u.UpdateSimuMode.bState)
  		{
  		   for (tU32 _u32Index = 0; (tU8)enClientiMXtemp_SensorID_MAX > _u32Index; ++_u32Index)
  		   {
  		   	if (TRUE == m_aoThermalInfo[_u32Index].u.tThermalInfo.bSensorStateReceived
  		        || TRUE == m_aoThermalInfo[_u32Index].u.tThermalInfo.bTemperatureReceived)
  		   		hc_tclAppMain::theServer()->vPostInternalMessage(&m_aoThermalInfo[_u32Index]);
  		   }
  		}
	   	break;
   case hc_tclBaseIf::eTriggerIMXTemp:
	   sendIMXTEMP_TemperatureGet();
	   break;
   default:
	   break;
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceMessage()
*
* DESCRIPTION: 	handler for trace command messages
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_ClientiMXtemp::vHandleTraceMessage(const tUChar* puchData)
{
	   ETG_TRACE_USR4(("vHandleTraceMessage() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);
}

/*******************************************************************************
*
* FUNCTION: 	vGetReferences(tVoid)
*
* DESCRIPTION: 	Function to get all reference needed by this class.
* 				A reference should always be the Interface class of the object
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_ClientiMXtemp::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));
   // reference of simulation mode interface
   m_poSimuMode = dynamic_cast<I_hc_tclSimuMode*>(_cpoMain->getHandler("I_hc_tclSimuMode"));
}

/*******************************************************************************
*
* FUNCTION: 	tVoid vStartCommunication()
*
* DESCRIPTION: 	Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_ClientiMXtemp::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));
   HC_NULL_POINTER_CHECK(hc_tclAppMain::theServer());
   onAvailable();
   sendIMXTEMP_TemperatureGet();
   if (TRUE == bIsServiceAvailable())
   {
	   (tVoid)OSAL_s32TimerSetTime(m_hTimerIMXTemp, u32GetPollingCycleTime(), 0);
   }
   else
   {
	   ETG_TRACE_FATAL(("vStartCommunication: client handler unavailable"));
   }
}

/*******************************************************************************
*
* FUNCTION: 	tVoid vTraceInfo()
*
* DESCRIPTION: 	Function to trace
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_ClientiMXtemp::vTraceInfo()
{
   ETG_TRACE_USR4(("vTraceInfo() entered."));
}

/*******************************************************************************
*
* FUNCTION: 	u32GetPollingCycleTime()
*
* DESCRIPTION: 	returns the polling cycle time in milliseconds
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tU32 hc_ClientiMXtemp::u32GetPollingCycleTime() const
{
   ETG_TRACE_USR4(("u32GetPollingCycleTime() entered."));

   return ((tU32)m_u8PollingCycleTime * 1000);
}

/*******************************************************************************
*
* FUNCTION: 	tVoid onAvailable()
*
* DESCRIPTION: 	callback for available service
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_ClientiMXtemp::onAvailable()
{
   ETG_TRACE_USR4(("onAvailable() entered."));

   _bAvailable = TRUE;

   for (tU32 _u32Index = 0; (tU8)enClientiMXtemp_SensorID_MAX > _u32Index; ++_u32Index)
   {
   	m_aoThermalInfo[_u32Index].u.tThermalInfo.bIfAvailable = _bAvailable;
   	if (TRUE == m_aoThermalInfo[_u32Index].u.tThermalInfo.bSensorStateReceived
           || TRUE == m_aoThermalInfo[_u32Index].u.tThermalInfo.bTemperatureReceived)
   		if ((OSAL_NULL != m_poSimuMode) && (FALSE == m_poSimuMode->bGetSimuMode()))
   			hc_tclAppMain::theServer()->vPostInternalMessage(&m_aoThermalInfo[_u32Index]);
   }
}

/*******************************************************************************
*
* FUNCTION: 	tVoid onUnavailable()
*
* DESCRIPTION: 	callback for unavailable service
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_ClientiMXtemp::onUnavailable()
{
   ETG_TRACE_USR4(("onUnavailable() entered."));

   _bAvailable = FALSE;

   for (tU32 _u32Index = 0; (tU8)enClientiMXtemp_SensorID_MAX > _u32Index; ++_u32Index)
   {
   	m_aoThermalInfo[_u32Index].u.tThermalInfo.bIfAvailable = _bAvailable;
   	if (TRUE == m_aoThermalInfo[_u32Index].u.tThermalInfo.bSensorStateReceived
           || TRUE == m_aoThermalInfo[_u32Index].u.tThermalInfo.bTemperatureReceived)
   		if ((OSAL_NULL != m_poSimuMode) && (FALSE == m_poSimuMode->bGetSimuMode()))
   			hc_tclAppMain::theServer()->vPostInternalMessage(&m_aoThermalInfo[_u32Index]);
   }
}

/*******************************************************************************
*
* FUNCTION: 	onGET_TEMP()
*
* DESCRIPTION: 	callback function for received temperature
*
* PARAMETER: 	tenClientiMXtemp_SensorID SensorID
*               tU32 Temperature
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_ClientiMXtemp::onGET_TEMP(tS32 Temperature)
{
   ETG_TRACE_USR4(("onGET_TEMP(Temperature: %d) entered.", Temperature ));

	m_aoThermalInfo[enClientiMXtemp_SensorID_IMX].u.tThermalInfo.bTemperatureReceived = TRUE;
	m_aoThermalInfo[enClientiMXtemp_SensorID_IMX].u.tThermalInfo.tTemperature = (tS16Temperature)(Temperature / 100);
	if ((OSAL_NULL != m_poSimuMode) && (FALSE == m_poSimuMode->bGetSimuMode()))
		hc_tclAppMain::theServer()->vPostInternalMessage(&m_aoThermalInfo[enClientiMXtemp_SensorID_IMX]);
}

/*******************************************************************************
*
* FUNCTION: 	sendIMXTEMP_TemperatureGet()
*
* DESCRIPTION: 	send GET request for iMX temperature
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_ClientiMXtemp::sendIMXTEMP_TemperatureGet()
{
   ETG_TRACE_USR4(("sendIMXTEMP_TemperatureGet() entered." ));

   if (FALSE == bIsServiceAvailable())
	   return;
#ifndef HC_USE_OSAL
   char _szBuffer[2048];
   ssize_t ret_in;
   tS32 _s32TempInMC = 0;
   int _iFd = open(m_strDevice, O_RDONLY);
   if (-1 != _iFd)
   {
	   (tVoid)memset((tVoid*)_szBuffer,OSAL_NULL,sizeof(_szBuffer));
	   ret_in = read(_iFd, _szBuffer, sizeof(_szBuffer));
	   close(_iFd);
	   if (0 < ret_in)
	   {
		   _s32TempInMC = (int)atoi(_szBuffer);
		   onGET_TEMP(_s32TempInMC);
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("sendIMXTEMP_TemperatureGet: read failed (%d)", ret_in ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("sendIMXTEMP_TemperatureGet: device %s couldn't open", m_strDevice ));
	   onUnavailable();
   }
#else
   OSAL_tIODescriptor _hDev = OSAL_IOOpen(m_strDevice,OSAL_EN_READONLY);
   if (OSAL_ERROR != _hDev)
   {
	   tS8 _as8Buffer[32];
	   tS32 _s32Length = OSAL_s32IORead(_hDev, _as8Buffer, sizeof(_as8Buffer));
	   (tVoid)OSAL_s32IOClose(_hDev);
	   if (0 < _s32Length)
	   {
		   onGET_TEMP((tU32)OSAL_s32AsciiToS32(_as8Buffer));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("sendIMXTEMP_TemperatureGet: read failed (%d)", _s32Length ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("sendIMXTEMP_TemperatureGet: device %s couldn't open", m_strDevice ));
	   onUnavailable();
   }
#endif
}

/*******************************************************************************
*
* FUNCTION: 	cb_vTimerIMXTemp()
*
* DESCRIPTION: 	timer callback to poll iMX temperature
*
* PARAMETER: 	tVoid* pArg
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
OSAL_tpfCallback hc_ClientiMXtemp::cb_vTimerIMXTemp(tVoid* pArg)
{
   ETG_TRACE_USR4(("cb_vTimerIMXTemp(pArg 0x%08x) entered.", pArg ));

   hc_ClientiMXtemp* _poClientIMXTemp = (hc_ClientiMXtemp*)pArg;

   if (OSAL_NULL != _poClientIMXTemp && OSAL_NULL != hc_tclAppMain::theServer())
   {
	   hc_tclAppMain::theServer()->vPostInternalMessage(&_poClientIMXTemp->m_oTriggerIMXTemp);
	   (tVoid)OSAL_s32TimerSetTime(_poClientIMXTemp->m_hTimerIMXTemp, _poClientIMXTemp->u32GetPollingCycleTime(), 0);
   }

   return OSAL_NULL;
}
