/************************************************************************
* FILE:        hc_tclClientTVMODULEtemp.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: class implementation of client handler TV module temperature
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 16.08.2017| CM-AI/EPB2 Bernard | init
*
*************************************************************************/
// generic OSAL interfaces
#ifndef OSAL_S_IMPORT_INTERFACE_GENERIC
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#endif //#ifndef OSAL_S_IMPORT_INTERFACE_GENERIC
// generic CCA interfaces
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library
// generic ETG trace interface
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
// own ETG traces
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclClientTVMODULEtemp.cpp.trc.h"
#endif
// generic datapool interface
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_fc_heatctrl_if.h"
// own interfaces
#include "hc_tclSimuMode.h"
#include "hc_tclClientTVMODULEtemp.h"
// assignment of own sensor IDs to internal sensor IDs
const tenThermalSensorID m_aenInternalSensorID[enTVModuleSensorID_MAX] = {
		enThermalSensorID_TVModule,
		enThermalSensorID_TVModuleSlave
};

/*************************************************************************
*
* FUNCTION:    hc_tclClientTVMODULEtemp()
*
* DESCRIPTION: constructor
*
* PARAMETER:   const hc_tclAppMain* poMainAppl: main - object of this application
*
* RETURNVALUE: none
*
*************************************************************************/
hc_tclClientTVMODULEtemp::hc_tclClientTVMODULEtemp(const hc_tclAppMain* poMainAppl)
: I_hc_tclClientTVMODULEtemp(poMainAppl)
{
   ETG_TRACE_USR4(("hc_tclClientTVMODULEtemp(poMainAppl: 0x%08x) entered.", poMainAppl ));
   m_poSimuMode = OSAL_NULL;
   // activation states
   m_bEnable = FALSE;
   m_bAavailable = FALSE;
   // TV module I2C information
   (tVoid)memset((tVoid*)m_strDevice, OSAL_NULL, sizeof(m_strDevice));
   m_u8SlaveAddress = 0;
   m_u8RegisterAddress = 0;
   m_u16PollInterval = HC_CLIENTTVMODULETEMP_CYCLETIME_SECONDS;
   // assignment of internal sensor IDs to own sensor IDs and init thermal information messages for all own supported sensors
   (tVoid)memset((tVoid*)m_aenOwnSensorID, OSAL_NULL, sizeof(m_aenOwnSensorID));
   (tVoid)memset((tVoid*)m_aoThermalInfo, OSAL_NULL, sizeof(m_aoThermalInfo));
   for(tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_aenInternalSensorID) > _u16Idx; ++_u16Idx)
   {
	   if ((HC_C_U16_ARRAYELEMENTS(m_aenOwnSensorID) > m_aenInternalSensorID[_u16Idx])
		   && (enThermalSensorID_UNDEF != m_aenInternalSensorID[_u16Idx]))
	   {
		   // assignment of internal sensor IDs to own sensor IDs
		   m_aenOwnSensorID[m_aenInternalSensorID[_u16Idx]] = (tenTVModuleSensorID)_u16Idx;
		   // init thermal information messages of own supported sensor
		   m_aoThermalInfo[_u16Idx].eCmd = hc_tclBaseIf::eNewThermalInfo_STANDARD;
		   (tVoid)OSAL_szStringNCopy(m_aoThermalInfo[_u16Idx].strClassName,
				   "I_hc_tclThermalSensorControl", sizeof(m_aoThermalInfo[_u16Idx].strClassName)-1);
		   m_aoThermalInfo[_u16Idx].u.tThermalInfo.bSensorStateSupported = FALSE;
		   m_aoThermalInfo[_u16Idx].u.tThermalInfo.bTemperatureSupported = TRUE;
		   (tVoid)OSAL_szStringNCopy(m_aoThermalInfo[_u16Idx].u.tThermalInfo.strClassName,
				   HC_CLIENTTVMODULETEMP_IFNAME, sizeof(m_aoThermalInfo[_u16Idx].u.tThermalInfo.strClassName)-1);
		   m_aoThermalInfo[_u16Idx].u.tThermalInfo.enSensorID = m_aenInternalSensorID[_u16Idx];
	   }
   }
   // timer handling
   (tVoid)memset((tVoid*)&m_oTriggerReadTemp, OSAL_NULL, sizeof(m_oTriggerReadTemp));
   (tVoid)OSAL_szStringNCopy(m_oTriggerReadTemp.strClassName, HC_CLIENTTVMODULETEMP_IFNAME, sizeof(m_oTriggerReadTemp.strClassName)-1);
   m_oTriggerReadTemp.eCmd = hc_tclBaseIf::eTriggerTVModuleTemp;
   m_hTimerReadTemp = OSAL_NULL;
   if (OSAL_OK != OSAL_s32TimerCreate( (OSAL_tpfCallback)cb_vTimerReadTemp, (tVoid*)this, &m_hTimerReadTemp))
   {
	   ETG_TRACE_FATAL(("hc_tclClientTVMODULEtemp: couldn't create timer to poll read temperature."));
	   m_hTimerReadTemp = OSAL_NULL;
   }
   else
   {
	   (tVoid)OSAL_s32TimerSetTime(m_hTimerReadTemp, 0, 0);
   }
}

/*************************************************************************
*
* FUNCTION:    ~hc_tclClientTVMODULEtemp()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*************************************************************************/
hc_tclClientTVMODULEtemp::~hc_tclClientTVMODULEtemp()
{
   ETG_TRACE_USR4(("~hc_tclClientTVMODULEtemp() entered."));
   if (OSAL_NULL != m_hTimerReadTemp)
   {
	   (tVoid)OSAL_s32TimerSetTime(m_hTimerReadTemp, 0, 0);
	   (tVoid)OSAL_s32TimerDelete(m_hTimerReadTemp);
   }
   m_bAavailable = FALSE;
   m_bEnable = FALSE;
   m_poSimuMode = OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION: 	vHandleMessage()
*
* DESCRIPTION: 	handler for internal messages
*
* PARAMETER:   	hc_tclBaseIf::TMsg* pMsg: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclClientTVMODULEtemp::vHandleMessage(hc_tclBaseIf::TMsg* pMsg)
{
   ETG_TRACE_USR4(("vHandleMessage() entered %u -> data: %d.", ETG_ENUM(HC_INT_MSG , (tU32)pMsg->eCmd), pMsg->u.u32Data));
   HC_NULL_POINTER_CHECK(pMsg);
   switch (pMsg->eCmd)
   {
   case hc_tclBaseIf::eUpdateSimuMode:
	   vHandleMessage_UpdateSimuMode(*pMsg);
	   if (FALSE == pMsg->u.UpdateSimuMode.bState)
	   {
		   for (tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_aoThermalInfo) > _u16Idx; ++_u16Idx)
		   {
			   if (TRUE == m_aoThermalInfo[_u16Idx].u.tThermalInfo.bTemperatureReceived)
				   onGET_Temp((tenTVModuleSensorID)_u16Idx, m_aoThermalInfo[_u16Idx].u.tThermalInfo.tTemperature);
		   }
	   }
	   break;
   case hc_tclBaseIf::eTriggerTVModuleTemp:
   	   {
   		   sendTemperatureGet();
   		   if (OSAL_NULL != m_hTimerReadTemp && TRUE == bIsEnable())
   		   {
   			   (tVoid)OSAL_s32TimerSetTime(m_hTimerReadTemp, u32GetPollingCycleTime(), 0);
   		   }
   	   }
	   break;
   case hc_tclBaseIf::eTriggerLoadSettings:
	   {
		   vHandleMessage_LoadSettings(*pMsg);
	   }
	   break;
   default:
	   break;
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceMessage()
*
* DESCRIPTION: 	handler for trace command messages
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclClientTVMODULEtemp::vHandleTraceMessage(const tUChar* puchData)
{
	ETG_TRACE_USR4(("vHandleTraceMessage() entered (data: 0x%08x).", puchData ));
	HC_NULL_POINTER_CHECK(puchData);

    tU32 u32MsgCode = (puchData[1]<<8) | puchData[2];
    ETG_TRACE_USR1(("hc_tclAppMain::vHandleTraceMessage(): process command: %d.", u32MsgCode));

    switch (u32MsgCode) {

    case FC_HEATCTRL_HC_GETTVMODULECLIENTHANDLERSTATE:
        {

        }
        break;

    default:
		{

		}
		break;
    }
}

/*******************************************************************************
*
* FUNCTION: 	vGetReferences(tVoid)
*
* DESCRIPTION: 	Function to get all reference needed by this class.
* 				A reference should always be the Interface class of the object
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclClientTVMODULEtemp::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));
   // reference of simulation mode interface
   m_poSimuMode = dynamic_cast<I_hc_tclSimuMode*>(_cpoMain->getHandler("I_hc_tclSimuMode"));
}

/*******************************************************************************
*
* FUNCTION: 	tVoid vStartCommunication()
*
* DESCRIPTION: 	Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclClientTVMODULEtemp::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));
   vHandleMessage_LoadSettings(m_oTriggerReadTemp);
}

/*******************************************************************************
*
* FUNCTION: 	tVoid vTraceInfo()
*
* DESCRIPTION: 	Function to trace
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclClientTVMODULEtemp::vTraceInfo()
{
   ETG_TRACE_USR4(("vTraceInfo() entered."));
}

/*******************************************************************************
*
* FUNCTION: 	bIsServiceAvailable()
*
* DESCRIPTION: 	returns the availability of this service
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	tBool ... {TRUE - available | else unavailable}
*
*******************************************************************************/
tBool hc_tclClientTVMODULEtemp::bIsServiceAvailable() const
{
   ETG_TRACE_USR4(("bIsServiceAvailable() entered."));

   return FALSE;
}

/*******************************************************************************
*
* FUNCTION: 	u32GetPollingCycleTime()
*
* DESCRIPTION: 	returns the polling cycle time in milliseconds
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tU32 hc_tclClientTVMODULEtemp::u32GetPollingCycleTime() const
{
   ETG_TRACE_USR4(("u32GetPollingCycleTime() entered."));

   return (((tU32)m_u16PollInterval) * 1000);
}

/*******************************************************************************
*
* FUNCTION: 	onGET_Temp()
*
* DESCRIPTION: 	callback function for received temperature
*
* PARAMETER: 	tenTVModuleSensorID enSensor
*               tS16 s16Temperature [dC]
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclClientTVMODULEtemp::onGET_Temp(tenTVModuleSensorID enSensor, tS16 s16Temperature)
{
   ETG_TRACE_USR4(("onGET_Temp(Sensor: %u, Temperature: %d dC) entered.",
		   ETG_CENUM(tenTVModuleSensorID, enSensor), s16Temperature ));

   if (HC_C_U16_ARRAYELEMENTS(m_aoThermalInfo) > enSensor)
   {
		m_aoThermalInfo[enSensor].u.tThermalInfo.bIfAvailable = TRUE;
		m_aoThermalInfo[enSensor].u.tThermalInfo.bTemperatureReceived = TRUE;
		m_aoThermalInfo[enSensor].u.tThermalInfo.tTemperature = s16Temperature;
		if ((OSAL_NULL != m_poSimuMode) && (FALSE == m_poSimuMode->bGetSimuMode())
			&& (enThermalSensorID_UNDEF != m_aoThermalInfo[enSensor].u.tThermalInfo.enSensorID)
			&& (enThermalSensorID_MAX > m_aoThermalInfo[enSensor].u.tThermalInfo.enSensorID))
			hc_tclAppMain::theServer()->vPostInternalMessage(&m_aoThermalInfo[enSensor]);
   }
}

/*******************************************************************************
*
* FUNCTION: 	sendTemperatureGet()
*
* DESCRIPTION: 	send temperature GET request
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclClientTVMODULEtemp::sendTemperatureGet()
{
   ETG_TRACE_USR4(("sendTemperatureGet() entered." ));

   // reading new TV module temperature from I2C device
   tU16 _u16RegisterData = 0;
   tBool _bSuccess = bReadI2cRegister( strGetDevice(), u8GetSlaveAddress(), u8GetRegisterAddress(), _u16RegisterData);
   tS16 _s16Temperature = s16Convert2Temperature(_u16RegisterData);
   onGET_Temp(enTVModuleSensorID_TVModule, _s16Temperature);

   ETG_TRACE_ERR(("sendTemperatureGet: reading of reg_addr=0x%02x from %60s with slave address 0x%02x finished %d with reg_data=0x%04x",
		   u8GetRegisterAddress(), strGetDevice(), u8GetSlaveAddress(), _bSuccess, _u16RegisterData ));
}

/*******************************************************************************
*
* FUNCTION: 	enConvertSensorID()
*
* DESCRIPTION: 	converts the own sensor id to internal sensor id
*
* PARAMETER: 	tenTVModuleSensorID enSensor ... own sensor id
*
* RETURNVALUE: 	tenThermalSensorID
*
*******************************************************************************/
tenThermalSensorID hc_tclClientTVMODULEtemp::enConvertSensorID(tenTVModuleSensorID enSensor) const
{
   ETG_TRACE_USR4(("enConvertSensorID(Sensor: %u) entered.", ETG_CENUM(tenTVModuleSensorID, enSensor) ));
   if (HC_C_U16_ARRAYELEMENTS(m_aenInternalSensorID) > (tU16)enSensor)
	   return m_aenInternalSensorID[enSensor];
   return enThermalSensorID_UNDEF;
}

/*******************************************************************************
*
* FUNCTION: 	enConvertSensorID()
*
* DESCRIPTION: 	converts the internal sensor id to own sensor id
*
* PARAMETER: 	tenThermalSensorID enSensor ... internal sensor id
*
* RETURNVALUE: 	tenTVModuleSensorID
*
*******************************************************************************/
tenTVModuleSensorID hc_tclClientTVMODULEtemp::enConvertSensorID(tenThermalSensorID enSensor) const
{
   ETG_TRACE_USR4(("enConvertSensorID(Sensor: %u) entered.", ETG_CENUM(tenThermalSensorID, enSensor) ));
   if (HC_C_U16_ARRAYELEMENTS(m_aenOwnSensorID) > (tU16)enSensor)
	   return m_aenOwnSensorID[enSensor];
   return enTVModuleSensorID_SUPPORTED;
}

/*******************************************************************************
*
* FUNCTION: 	vHandleMessage_UpdateSimuMode()
*
* DESCRIPTION: 	handle trigger to update simulation mode
*
* PARAMETER: 	message
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclClientTVMODULEtemp::vHandleMessage_UpdateSimuMode(hc_tclBaseIf::TMsg &Msg)
{
   ETG_TRACE_USR4(("vHandleMessage_UpdateSimuMode(pMsg: 0x%08x) entered.", &Msg ));
}

/*******************************************************************************
*
* FUNCTION: 	vHandleMessage_LoadSettings()
*
* DESCRIPTION: 	handle trigger to load settings
*
* PARAMETER: 	message
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclClientTVMODULEtemp::vHandleMessage_LoadSettings(hc_tclBaseIf::TMsg &Msg)
{
   ETG_TRACE_USR4(("vHandleMessage_LoadSettings(pMsg: 0x%08x) entered.", &Msg ));
   // enable polling of TV module
   tU8 _u8Enable = OSAL_C_U8_MAX;
   dp_tclKdsCMVariantCoding _oVariantCoding;
   tU8 _u8Status = _oVariantCoding.u8GetDTV(_u8Enable);
   if (DP_U8_ELEM_STATUS_VALID == _u8Status)
   {
	   _u8Enable = (((0x01 == _u8Enable) || (0x02 == _u8Enable)) ? (TRUE == _u8Enable) : OSAL_C_U8_MAX);
   }
   m_bEnable = (0x01 == _u8Enable);
   ETG_TRACE_FATAL(("vHandleMessage_LoadSettings: TV module existing = %d (KDS: 0x%02x, return: 0x%02x)", bIsEnable(), _u8Enable, _u8Status ));
   // load TV module i2c information
   memset(m_strDevice, OSAL_NULL, sizeof(m_strDevice));
   dp_tclHeatCtrlDPTVModule_I2CAdapter_Device _oDevice;
   if ((0 == _oDevice.u8GetData(m_strDevice, sizeof(m_strDevice)-1)) || (0 == strlen(m_strDevice)))
   {
	   OSAL_szStringNCopy(m_strDevice, HC_CLIENTTVMODULETEMP_DEFAULT_DEVICE, sizeof(m_strDevice)-1);
   }
   dp_tclHeatCtrlDPTVModule_I2CAdapter_DeviceAddress _oSlaveAddress;
   m_u8SlaveAddress = (tU8)_oSlaveAddress.tGetData();
   dp_tclHeatCtrlDPTVModule_I2CAdapter_DataAddress _oRegisterAddress;
   m_u8RegisterAddress = _oRegisterAddress.tGetData();
   dp_tclHeatCtrlDPTVModule_I2CAdapter_Period _oPollInterval;
   m_u16PollInterval = _oPollInterval.tGetData();
   // start monitor
   if ((TRUE == m_bEnable) && (OSAL_NULL != hc_tclAppMain::theServer()))
   {
	   hc_tclAppMain::theServer()->vPostInternalMessage(&m_oTriggerReadTemp);
   }
}

/*******************************************************************************
*
* FUNCTION: 	bFindInLookup
*
* DESCRIPTION: 	find the start and end index in the lookup table catAdc2Temp
*               for specified ADC value
*
* PARAMETER: 	tS16 s16ADCvalue
*               tU16& u16IdxStart
*               tU16& u16IdxEnd
*
* RETURNVALUE: 	tBool    {True - successfull found  | else failed}
*
*******************************************************************************/
struct tClientTVMODULEAdc2Temp
{
	tS16 s16ADC;   // signed ADC value
	tU16 u16ADC;   // unsigned ADC value
	tS16 s16Temp;  // temperature [°C]
};
const tClientTVMODULEAdc2Temp catAdc2Temp[] = {
		{ 22560, 0x5820, -400},
		{ 22464, 0x57C0, -390},
		{ 22368, 0x5760, -380},
		{ 22256, 0x56F0, -370},
		{ 22160, 0x5690, -360},
		{ 22048, 0x5620, -350},
		{ 21920, 0x55A0, -340},
		{ 21808, 0x5530, -330},
		{ 21680, 0x54B0, -320},
		{ 21552, 0x5430, -310},
		{ 21408, 0x53A0, -300},
		{ 21264, 0x5310, -290},
		{ 21120, 0x5280, -280},
		{ 20976, 0x51F0, -270},
		{ 20816, 0x5150, -260},
		{ 20656, 0x50B0, -250},
		{ 20480, 0x5000, -240},
		{ 20304, 0x4F50, -230},
		{ 20112, 0x4E90, -220},
		{ 19920, 0x4DD0, -210},
		{ 19728, 0x4D10, -200},
		{ 19536, 0x4C50, -190},
		{ 19328, 0x4B80, -180},
		{ 19104, 0x4AA0, -170},
		{ 18880, 0x49C0, -160},
		{ 18656, 0x48E0, -150},
		{ 18416, 0x47F0, -140},
		{ 18176, 0x4700, -130},
		{ 17920, 0x4600, -120},
		{ 17664, 0x4500, -110},
		{ 17392, 0x43F0, -100},
		{ 17120, 0x42E0, -90},
		{ 16848, 0x41D0, -80},
		{ 16560, 0x40B0, -70},
		{ 16272, 0x3F90, -60},
		{ 15952, 0x3E50, -50},
		{ 15648, 0x3D20, -40},
		{ 15344, 0x3BF0, -30},
		{ 15024, 0x3AB0, -20},
		{ 14688, 0x3960, -10},
		{ 14352, 0x3810,  0},
		{ 14016, 0x36C0, 10},
		{ 13664, 0x3560, 20},
		{ 13296, 0x33F0, 30},
		{ 12944, 0x3290, 40},
		{ 12576, 0x3120, 50},
		{ 12192, 0x2FA0, 60},
		{ 11824, 0x2E30, 70},
		{ 11440, 0x2CB0, 80},
		{ 11040, 0x2B20, 90},
		{ 10656, 0x29A0, 100},
		{ 10256, 0x2810, 110},
		{  9840, 0x2670, 120},
		{  9424, 0x24D0, 130},
		{  9008, 0x2330, 140},
		{  8592, 0x2190, 150},
		{  8160, 0x1FE0, 160},
		{  7744, 0x1E40, 170},
		{  7312, 0x1C90, 180},
		{  6864, 0x1AD0, 190},
		{  6432, 0x1920, 200},
		{  6000, 0x1770, 210},
		{  5568, 0x15C0, 220},
		{  5120, 0x1400, 230},
		{  4672, 0x1240, 240},
		{  4224, 0x1080, 250},
		{  3776, 0x0EC0, 260},
		{  3328, 0x0D00, 270},
		{  2880, 0x0B40, 280},
		{  2416, 0x0970, 290},
		{  1968, 0x07B0, 300},
		{  1520, 0x05F0, 310},
		{  1072, 0x0430, 320},
		{   624, 0x0270, 330},
		{   176, 0x00B0, 340},
		{  -256, 0xFF00, 350},
		{  -704, 0xFD40, 360},
		{ -1152, 0xFB80, 370},
		{ -1584, 0xF9D0, 380},
		{ -2016, 0xF820, 390},
		{ -2464, 0xF660, 400},
		{ -2880, 0xF4C0, 410},
		{ -3312, 0xF310, 420},
		{ -3728, 0xF170, 430},
		{ -4160, 0xEFC0, 440},
		{ -4576, 0xEE20, 450},
		{ -4992, 0xEC80, 460},
		{ -5392, 0xEAF0, 470},
		{ -5792, 0xE960, 480},
		{ -6192, 0xE7D0, 490},
		{ -6592, 0xE640, 500},
		{ -6992, 0xE4B0, 510},
		{ -7376, 0xE330, 520},
		{ -7744, 0xE1C0, 530},
		{ -8128, 0xE040, 540},
		{ -8512, 0xDEC0, 550},
		{ -8864, 0xDD60, 560},
		{ -9232, 0xDBF0, 570},
		{ -9584, 0xDA90, 580},
		{ -9936, 0xD930, 590},
		{-10272, 0xD7E0, 600},
		{-10608, 0xD690, 610},
		{-10928, 0xD550, 620},
		{-11248, 0xD410, 630},
		{-11568, 0xD2D0, 640},
		{-11888, 0xD190, 650},
		{-12176, 0xD070, 660},
		{-12480, 0xCF40, 670},
		{-12768, 0xCE20, 680},
		{-13072, 0xCCF0, 690},
		{-13344, 0xCBE0, 700},
		{-13616, 0xCAD0, 710},
		{-13888, 0xC9C0, 720},
		{-14160, 0xC8B0, 730},
		{-14432, 0xC7A0, 740},
		{-14688, 0xC6A0, 750},
		{-14928, 0xC5B0, 760},
		{-15184, 0xC4B0, 770},
		{-15408, 0xC3D0, 780},
		{-15648, 0xC2E0, 790},
		{-15872, 0xC200, 800},
		{-16112, 0xC110, 810},
		{-16336, 0xC030, 820},
		{-16544, 0xBF60, 830},
		{-16752, 0xBE90, 840},
		{-16960, 0xBDC0, 850},
		{-17152, 0xBD00, 860},
		{-17360, 0xBC30, 870},
		{-17552, 0xBB70, 880},
		{-17728, 0xBAC0, 890},
		{-17920, 0xBA00, 900},
		{-18112, 0xB940, 910},
		{-18288, 0xB890, 920},
		{-18448, 0xB7F0, 930},
		{-18624, 0xB740, 940},
		{-18784, 0xB6A0, 950},
		{-18944, 0xB600, 960},
		{-19104, 0xB560, 970},
		{-19248, 0xB4D0, 980},
		{-19408, 0xB430, 990},
		{-19600, 0xB370, 1000},
};
tBool hc_tclClientTVMODULEtemp::bFindInLookup(tS16 s16ADCvalue, tU16& u16IdxStart, tU16& u16IdxEnd) const
{
   ETG_TRACE_USR2(("bFindInLookup(ADCvalue: %d (0x%04x), IdxStart: %d, IdxEnd: %d) entered.", s16ADCvalue, s16ADCvalue, u16IdxStart, u16IdxEnd ));
   tBool _bReturn = FALSE;
   if ((u16IdxStart <  u16IdxEnd) && (HC_C_U16_ARRAYELEMENTS(catAdc2Temp)> u16IdxEnd))
   {
	   tU16 _u16MidEnd = (u16IdxEnd+u16IdxStart) / 2;
	   if (u16IdxStart == _u16MidEnd)
	   {
		   _bReturn = TRUE;
	   }
	   else if (catAdc2Temp[_u16MidEnd].s16ADC >= s16ADCvalue)
	   {
		   u16IdxStart = _u16MidEnd;
		   _bReturn = bFindInLookup(s16ADCvalue, u16IdxStart, u16IdxEnd);
	   }
	   else
	   {
		   u16IdxEnd = _u16MidEnd;
		   _bReturn = bFindInLookup(s16ADCvalue, u16IdxStart, u16IdxEnd);
	   }
   }
   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION: 	s16Convert2Temperature()
*
* DESCRIPTION: 	convert ADC value to temperature [d°C]
*
* PARAMETER: 	const tU16 u16ADCvalue
*
* RETURNVALUE: 	tS16
*
*******************************************************************************/
tS16 hc_tclClientTVMODULEtemp::s16Convert2Temperature(tU16 u16ADCvalue) const
{
   ETG_TRACE_USR4(("s16Convert2Temperature(ADCvalue: 0x%04x) entered.", u16ADCvalue ));
   tS16 _s16Return = OSAL_C_S16_MIN;
   tU16 _u16IdxStart = 0;
   tU16 _u16IdxEnd = HC_C_U16_ARRAYELEMENTS(catAdc2Temp)-1;
   tS16 _s16ADCvalue = (tS16)u16ADCvalue;
   tBool _bFound = bFindInLookup(_s16ADCvalue, _u16IdxStart, _u16IdxEnd);

   ETG_TRACE_USR2(("s16Convert2Temperature: bFindInLookup for ADCvalue 0x%04x (%d) return with %d and IdxStart %d and IdxEnd %d.", u16ADCvalue, _s16ADCvalue, _bFound, _u16IdxStart, _u16IdxEnd ));

   if (TRUE == _bFound)
   {
	   tS16 _s16ADCDeltaAll = catAdc2Temp[_u16IdxStart].s16ADC - catAdc2Temp[_u16IdxEnd].s16ADC;
	   tS16 _s16ADCvalueDelta = catAdc2Temp[_u16IdxStart].s16ADC - _s16ADCvalue;
	   tS16 _s16TempDelta = ((catAdc2Temp[_u16IdxEnd].s16Temp)-(catAdc2Temp[_u16IdxStart].s16Temp)) * (_s16ADCvalueDelta/_s16ADCDeltaAll);
	   _s16Return = catAdc2Temp[_u16IdxStart].s16Temp + _s16TempDelta;
   }
   ETG_TRACE_USR2(("s16Convert2Temperature: returns for ADCvalue 0x%04x the Temperature %ddC", u16ADCvalue, _s16Return ));
   return _s16Return;
}
/*******************************************************************************
*
* FUNCTION: 	cb_vTimerReadTemp()
*
* DESCRIPTION: 	timer callback to read temperature
*
* PARAMETER: 	tVoid* pArg
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
OSAL_tpfCallback hc_tclClientTVMODULEtemp::cb_vTimerReadTemp(tVoid* pArg)
{
   ETG_TRACE_USR4(("cb_vTimerIMXTemp(pArg 0x%08x) entered.", pArg ));

   hc_tclClientTVMODULEtemp* _poClientTemp = (hc_tclClientTVMODULEtemp*)pArg;

   if (OSAL_NULL != _poClientTemp && OSAL_NULL != hc_tclAppMain::theServer())
   {
	   hc_tclAppMain::theServer()->vPostInternalMessage(&_poClientTemp->m_oTriggerReadTemp);
   }

   return OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION: 	bReadI2cRegister()
*
* DESCRIPTION: 	read I2C register
*
* PARAMETER: 	const char* strDev				full device path like '/dec/i2c-1'
*               const tU8 u8SlaveAddress        slave address like 0x49
*               const tU8 u8RegisterAddress     register address like 0x00
* 				tU16& u16RegisterData
*
* RETURNVALUE: 	tBool		{TRUE - successful, else - failed}
*
*******************************************************************************/
tBool hc_tclClientTVMODULEtemp::bReadI2cRegister(const char* strDev, const tU8 u8SlaveAddress, const tU8 u8RegisterAddress, tU16& u16RegisterData)
{
	tBool _bResult = FALSE;
	const char *_strDev = ((OSAL_NULL != strDev) && (0 < strlen(strDev)))? strDev : HC_CLIENTTVMODULETEMP_DEFAULT_DEVICE;
	tS32 _fd = -1;

	// open I2C bus to read the register
	_fd = open(_strDev, O_RDWR);
	if(0 > _fd)
	{
		ETG_TRACE_ERR(("bReadI2cRegister: I2C bus %s open failed!", _strDev ));
	}
	else
	{
		// set device address of the slave
		if( ioctl(_fd, I2C_SLAVE, u8SlaveAddress) < 0)
		{
			ETG_TRACE_ERR(("bReadI2cRegister: setting slave address %u failed!", u8SlaveAddress ));
		}
		else
		{
	         // prepare message sequence
	         tU8 _aReadBuffer[2] = {0};
	         tU8 _aWriteBuffer[1] = {0};
	         struct i2c_msg _messages[1];
	         memset(_messages, OSAL_NULL, sizeof(_messages));
	         struct i2c_rdwr_ioctl_data _data;


	         /*
	         _messages[0].addr = _u8SlaveAddress;
	         _messages[0].flags = 0x0; //write mode
	         _messages[0].len = 1;
	         _messages[0].buf = _aWriteBuffer;
	         _messages[0].buf[0] = u8RegisterAddress;
	         */
	         _messages[0].addr = u8SlaveAddress;
	         _messages[0].flags = I2C_M_RD; // read mode
	         _messages[0].len = sizeof(_aReadBuffer);
	         _messages[0].buf = _aReadBuffer;

	         _data.msgs = _messages;
	         _data.nmsgs = 1;

	         // read data / register
	         if(ioctl(_fd, I2C_RDWR, &_data) < 0)
	         {
	        	 ETG_TRACE_ERR(("_bReadRegister: reading of data failed!"));
	         }
	         else
	         {
	        	 ETG_TRACE_USR2(("bReadI2cRegister: aReadBuffer = 0x%02x 0x%02x", _aReadBuffer[0], _aReadBuffer[1]));
	        	 // set return data
	        	 u16RegisterData = ((tU16)_aReadBuffer[0] << 8) + ((tU16)_aReadBuffer[1]);
	        	 _bResult = TRUE;
	         }
		}
		// close I2C bus
		close(_fd);
	}

	return _bResult;
}

/*******************************************************************************
*
* FUNCTION: 	bWriteI2cRegister()
*
* DESCRIPTION: 	read I2C register
*
* PARAMETER: 	const char* strDev				full device path like '/dec/i2c-1'
*               const tU8 u8SlaveAddress        slave address like 0x49
*               const tU8 u8RegisterAddress     register address like 0x00
* 				tU8 u8RegisterData
*
* RETURNVALUE: 	tBool		{TRUE - successful, else - failed}
*
*******************************************************************************/
tBool hc_tclClientTVMODULEtemp::bWriteI2cRegister(const char* strDev, const tU8 u8SlaveAddress, const tU8 u8RegisterAddress, tU8 u8RegisterData) const
{
	tBool _bResult = FALSE;
	const char *_strDev = ((OSAL_NULL != strDev) && (0 < strlen(strDev)))? strDev : HC_CLIENTTVMODULETEMP_DEFAULT_DEVICE;
	int _fd = -1;

	// open I2C bus to read the register
	_fd = open(_strDev, O_RDWR);
	if(_fd < 0)
	{
		ETG_TRACE_ERR(("bWriteI2cRegister: I2C bus %s open failed!", _strDev ));
	}
	else
	{
	  // set device address of the slave
	  if( ioctl(_fd, I2C_SLAVE, u8SlaveAddress) < 0)
	  {
		 ETG_TRACE_ERR(("bWriteI2cRegister: setting slave address %u failed!", u8SlaveAddress ));
	  }
	  else
	  {
		 tU8 _aWriteBuffer[2] = {0};
		 _aWriteBuffer[0] = u8RegisterAddress; //Address of Register
		 _aWriteBuffer[1] = u8RegisterData; //Data
		 if (write(_fd, _aWriteBuffer, sizeof(_aWriteBuffer)) != sizeof(_aWriteBuffer))
		 {
			ETG_TRACE_ERR(("bWriteI2cRegister: Error in writing Register %u", u8RegisterAddress ));
		 }
		 else
		 {
			_bResult = TRUE;
		 }
	  }
	  close(_fd);
	}
	return (_bResult);
}

/*******************************************************************************
*
* FUNCTION: 	vHandleMessage_TrcCmd_GetState()
*
* DESCRIPTION: 	handle trace command HC_GetTVModuleClientHandlerState
*
* PARAMETER: 	const tUChar* puchData
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclClientTVMODULEtemp::vHandleMessage_TrcCmd_GetState(const tUChar* puchData)
{

}
