/**
 * @file   : cfgmgr_UpdateManager.cpp
 * @author : nce5kor
 * @date   : Aug 24, 2017
 * @brief  : Central class for Data/ localdata / registration / setSystemToDelivery request-response
 * @copyright (c) 2017 Robert Bosch Car Multimedia Gmbh
 * @addgroup : cfgmgr
 * @{
 */

#ifndef cfgmgr_UpdateManager_H_
#include "cfgmgr_UpdateManager.h"
#endif

#ifndef CFGMGR_ASF_APPCLIENT_H_
#include "../asf/cfgmgr_ASFAppClient.h"
#endif

using namespace asf;

cfgmgr_UpdateManager* cfgmgr_UpdateManager::m_pUpdateManager = NULL;

/***********************************************************************************************/
/* 	         				cfgmgr_UpdateManager  Constructor      						       */
/***********************************************************************************************/
cfgmgr_UpdateManager::cfgmgr_UpdateManager():m_pListener(NULL),m_pInterface(NULL)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::cfgmgr_UpdateManager constructor");

	m_pDatapoolHandler = new cfgmgr_DatapoolHandler();
}

/***********************************************************************************************/
/* 	         				cfgmgr_UpdateManager  Destructor      						       */
/***********************************************************************************************/
cfgmgr_UpdateManager::~cfgmgr_UpdateManager()
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::~cfgmgr_UpdateManager destructor");

	if(NULL != m_pUpdateManager)
	{
		delete m_pUpdateManager;
		m_pUpdateManager = NULL;
	}

	m_pListener = NULL;
	m_pInterface = NULL;
	
	if(NULL != m_pDatapoolHandler)
	{
		delete m_pDatapoolHandler;
		m_pDatapoolHandler = NULL;
	}
}

/***********************************************************************************************/
/* 	   			          Method to get the instance of update Manager      				   */
/***********************************************************************************************/
cfgmgr_UpdateManager* cfgmgr_UpdateManager::getInstanceofUpdateManager()
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::getInstanceofUpdateManager");

	if (NULL == m_pUpdateManager)
	{
		m_pUpdateManager = new cfgmgr_UpdateManager();
	}

	return m_pUpdateManager;
}

/***********************************************************************************************/
/* 	   			          Method to register the Client Handler        				           */
/***********************************************************************************************/
tVoid cfgmgr_UpdateManager::registerClientHandler(cfgmgr_INAClientHandler* pHandler)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::registerClientHandler");
	m_pInterface = pHandler;
}

/***********************************************************************************************/
/* 	   			          Method to set the Listener 			       				           */
/***********************************************************************************************/
tVoid cfgmgr_UpdateManager::setListener(cfgmgr_IResponseListener *pListener)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::setListener");

	if(NULL == m_pListener)
	{
		m_pListener = pListener;
	}
}

/***********************************************************************************************/
/* 	   			          Method to unset the Listener 			       				           */
/***********************************************************************************************/
tVoid cfgmgr_UpdateManager::unsetListener()
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::unsetListener");
	if(NULL != m_pListener)
	{
		m_pListener = NULL;
	}
}

/***********************************************************************************************/
/* 	   			          Handling of Registrations of Private Item Requests from Apps 		   */
/***********************************************************************************************/
Reg_Status_T cfgmgr_UpdateManager::registerPrivateConfigItem(ClientData_T& clientData)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::registerPrivateConfigItem");

	Reg_Status_T status = REG_STATUS_FAILED;
	::boost::shared_ptr<ConfigDataProxy > proxy;

	cfgmgr_ClientDatabase *pCMApp = cfgmgr_ClientDatabase::createProxyDatabaseObject();
	if(NULL != pCMApp)
	{
		status = pCMApp->searchForConfigItemName(clientData.itemName, clientData.dbusName, PRIVATE_LIST);

		if(status != REG_STATUS_ALREADY_REGISTERED)
		{
			if(NULL != m_pInterface)
			{
				proxy = m_pInterface->createProxy(clientData.dbusName);
				pCMApp->storePrivateAppInfo(clientData.itemName, clientData.dbusName,proxy, clientData.formatId);
				status = REG_STATUS_SUCCESS;
			}
			else
			{
				log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::registerPrivateConfigItem -- m_pInterface Pointer is NULL");
			}
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::registerPrivateConfigItem -- pCMApp Pointer is NULL");
	}

	return status;
}

/***********************************************************************************************/
/* 	   			          Handling of Unregistrations of Private Item Requests from Apps 	   */
/***********************************************************************************************/
Reg_Status_T cfgmgr_UpdateManager::unregisterPrivateConfigItem(tStr itemName, tStr dbusName)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::unregisterPrivateConfigItem with itemName - ", itemName.c_str(), ",dbusName", dbusName.c_str());

	Reg_Status_T status = REG_STATUS_FAILED;

	cfgmgr_ClientDatabase *pCMApp = cfgmgr_ClientDatabase::createProxyDatabaseObject();

	if(NULL != pCMApp)
	{
		status = pCMApp->deletePrivateAppInfo(itemName, dbusName);
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::unregisterPrivateConfigItem -- pCMApp Pointer is NULL");
	}

	return status;
}

/***********************************************************************************************/
/* 	   			          Handling of Registrations of Public Item Requests from Apps 		   */
/***********************************************************************************************/
Reg_Status_T cfgmgr_UpdateManager::registerPublicConfigItem(ClientData_T& clientData)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::registerPublicConfigItem");

	Reg_Status_T status = REG_STATUS_FAILED;
	::boost::shared_ptr<ConfigDataProxy > proxy;

	cfgmgr_ClientDatabase *pCMApp = cfgmgr_ClientDatabase::createProxyDatabaseObject();
	if(NULL != pCMApp)
	{
		status = pCMApp->searchForConfigItemName(clientData.itemName, clientData.dbusName, PUBLIC_LIST);

		if((status != REG_STATUS_ALREADY_REGISTERED) && (status != REG_STATUS_UNKNOWN_ITEM))
		{
			if(NULL != m_pInterface)
			{
				proxy = m_pInterface->createProxy(clientData.dbusName);
				pCMApp->storePublicAppInfo(clientData.itemName, clientData.dbusName,proxy, clientData.formatId);
				status = REG_STATUS_SUCCESS;
			}
			else
			{
				log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::registerPublicConfigItem -- m_pInterface Pointer is NULL");
			}
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::registerPublicConfigItem -- pCMApp Pointer is NULL");
	}

	return status;
}

/***********************************************************************************************/
/* 	   			          Handling of Unregistrations of Public Item Requests from Apps 	   */
/***********************************************************************************************/
Reg_Status_T cfgmgr_UpdateManager::unregisterPublicConfigItem(tStr itemName, tStr dbusName)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::unregisterPublicConfigItem with itemName - ", itemName.c_str(), ", dbusName - ", dbusName.c_str());

	Reg_Status_T status = REG_STATUS_FAILED;

	cfgmgr_ClientDatabase *pCMApp = cfgmgr_ClientDatabase::createProxyDatabaseObject();

	if(NULL != pCMApp)
	{
		status = pCMApp->searchForConfigItemName(itemName, dbusName, PUBLIC_LIST);

		if(status != REG_STATUS_UNKNOWN_ITEM)
		{
			status = pCMApp->deletePublicAppInfo(itemName, dbusName);
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::unregisterPublicConfigItem -- pCMApp Pointer is NULL");
	}
	return status;
}

/***********************************************************************************************************/
/* 	   		Method called from Data request Handler to send the Default Private/public Data to Apps	 	   */
/***********************************************************************************************************/
tVoid cfgmgr_UpdateManager::SendData(map<tStr,tStr> configData, DescmoState_T configItem, tbool isCfgMgrRegistered, tStr actionType )
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::SendData with configItemName - ", configItem.profileName.c_str(), ", isCfgMgrRegistered - " , isCfgMgrRegistered);

	ConfigItem_T configDataList;
    addDataToList(configData, configDataList);

    if(!isCfgMgrRegistered)
    {
    	if("SET" == actionType)
    	{
    	    mConfigItem = configDataList;
    		SetDefaultPrivateData(configDataList,configItem);
    	}
    	else if("GET" == actionType)
    	{
    		GetDefaultPrivateData(configDataList,configItem);
    	}
    }
    else //PUBLIC DATA
    {
	    if("SET" == actionType)
    	{
	        mConfigItem = configDataList;
    		SetDefaultPublicData(configDataList,configItem);
    	}
    	else if("GET" == actionType)
    	{
    		GetDefaultPublicData(configDataList,configItem);
    	}
    }

}

/***********************************************************************************************************/
/* 	   		Method called from Data request Handler to send the XML Private/public Data to Apps	 	       */
/***********************************************************************************************************/
tVoid cfgmgr_UpdateManager::SendDataXml(DescmoState_T configItem)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::SendDataXml with profileName - ", configItem.profileName.c_str(), ", xmlData - " , configItem.xmlData.c_str(), ", xmlSignature - ", configItem.xmlDataSignature.c_str());

	::boost::shared_ptr<ConfigDataProxy> proxy;

	cfgmgr_ClientDatabase *pCMApp = cfgmgr_ClientDatabase::createProxyDatabaseObject();

	if(NULL != pCMApp)
	{
		pCMApp->getProxy(configItem.profileName, proxy,PRIVATE_LIST);
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::SendDataXml -- pCMApp Pointer is NULL");
	}

	if(NULL != m_pInterface)
	{
		m_pInterface->SendXMLData(configItem, proxy);
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::SendDataXml -- m_pInterface Pointer is NULL");
	}

	//TODO - AS of now XML data is only considred as Private. Public Data is coming as Default always.
}

/***********************************************************************************************************/
/* 	   		Method called from App Client Class on request of set Private Item               	 	       */
/***********************************************************************************************************/
tVoid cfgmgr_UpdateManager::onSetPrivateItem(Cfgmgr_ItemStatus_T result, DescmoState_T configItem, tStr actionType)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::onSetPrivateItem");
	log(&CMIT,DLT_LOG_INFO,"RESPONSE : ",printResult(result));
	log(&CMIT,DLT_LOG_INFO,"------------------------------------");

	if(NULL != m_pListener)
	{
		m_pListener->onDefaultDataResponse(mConfigItem, result, configItem, actionType);
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::onSetPrivateItem -- m_pListener Pointer is NULL");
	}
}

/***********************************************************************************************************/
/* 	   		Method called from App Client Class on request of get Private Item               	 	       */
/***********************************************************************************************************/
tVoid cfgmgr_UpdateManager::onGetPrivateItem(Cfgmgr_ItemStatus_T result, ConfigItem_T data, DescmoState_T configItem, tStr actionType)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::onGetPrivateItem");
	log(&CMIT,DLT_LOG_INFO,"RESPONSE : ",printResult(result));
	
	for(tUInt16 i = 0; i < data.size();i++)
	{
		log(&CMIT,DLT_LOG_INFO,"Key   : ",data[i].key);
		log(&CMIT,DLT_LOG_INFO,"Value : ",data[i].value);
	}
	log(&CMIT,DLT_LOG_INFO,"------------------------------------");

	if(NULL != m_pListener)
	{
		m_pListener->onDefaultDataResponse(data, result, configItem, actionType);
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::onGetPrivateItem -- m_pListener Pointer is NULL");
	}
}

/***********************************************************************************************************/
/* 	   		Method called from App Client Class on request of XML Data set/get	               	 	       */
/***********************************************************************************************************/
tVoid cfgmgr_UpdateManager::onExchangeDESCMO(Cfgmgr_ItemStatus_T result, tStr xmlData, DescmoState_T configItem)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::onExchangeDESCMO ");
	log(&CMIT,DLT_LOG_INFO,"RESPONSE : ",printResult(result));
	log(&CMIT,DLT_LOG_INFO,"------------------------------------");
	
	if(NULL != m_pListener)
	{
		m_pListener->onXmlDataResponse(xmlData, result, configItem);
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::onExchangeDESCMO -- m_pListener Pointer is NULL");
	}
}

/***********************************************************************************************************/
/* 	   								Method to construct the Vector from Map						 	       */
/***********************************************************************************************************/
tVoid cfgmgr_UpdateManager::addDataToList(map<tStr,tStr> data, ConfigItem_T& itemList)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::addDataToList");

	ConfigData_T item;

	map<tStr,tStr>::iterator it;

	for(it = data.begin(); it != data.end(); ++it)
	{
		item.key = it->first;
		item.value = it->second;
		itemList.push_back(item);
	}
}

/***********************************************************************************************************/
/* 	   								Method to Check whether Config Item is Subscribed			 	       */
/***********************************************************************************************************/
tbool cfgmgr_UpdateManager::checkForConfigItemSubscribed(tStr profileName,FormatIdentifier &formatId,tbool &isCfgMgrRegistered)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::checkForConfigItemSubscribed with profileName - ", profileName.c_str());

	tbool status = false;
	formatId = DEFAULT_FORMAT;

	cfgmgr_ClientDatabase *pCMApp = cfgmgr_ClientDatabase::createProxyDatabaseObject();

	if(NULL != pCMApp)
	{
		//First search through the private list to find the format flag
		status = pCMApp->getFormatIdType(profileName, isCfgMgrRegistered, formatId);

		if(status)
		{
			//if found in private list then the config item is private
			log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_UpdateManager::checkForConfigItemSubscribed, found in PRIVATE list");
		}
		else
		{
			log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_UpdateManager::checkForConfigItemSubscribed, NOT found in Private list");
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::checkForConfigItemSubscribed -- pCMApp pointer is NULL");
	}

	return status;
}

/***********************************************************************************************************/
/* 	   								Method to SendPublic Data to Native/Managed Apps 				 	   */
/***********************************************************************************************************/
Cfgmgr_ItemStatus_T cfgmgr_UpdateManager::SendDataPublic(map<tStr,tStr> configData, tStr configItemName)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::SendDataPublic");

	Cfgmgr_ItemStatus_T itemStatus = STATUS_FAILED;
	ConfigItem_T configDataList;

	addDataToList(configData, configDataList);
	itemStatus = setDpData(configDataList,configItemName);

	if(itemStatus == STATUS_SUCCESS)
	{
		updateClientPublicItem(CFGMGR_UPDATE_ON_SET,configItemName,"");
	}

	return itemStatus;
}

/***********************************************************************************************************/
/* 	   								Method to Get Public Data from Native/Managed Apps     		 	       */
/***********************************************************************************************************/

Cfgmgr_ItemStatus_T cfgmgr_UpdateManager::getPublicData(tStr configItemName, ConfigItem_T& itemList )
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::getPublicData");

	Cfgmgr_ItemStatus_T itemStatus = STATUS_FAILED;
	itemStatus = getDpData(itemList,configItemName);

	return itemStatus;
}

/***********************************************************************************************************/
/* 	   					Method to SetDefault Private Data to Native/Managed Apps 		 	   			   */
/***********************************************************************************************************/
tVoid cfgmgr_UpdateManager::SetDefaultPrivateData(ConfigItem_T configDataList,DescmoState_T configItem)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::SetDefaultPrivateData");

	log(&CMIT,DLT_LOG_INFO,"REQUEST ITEM : ",configItem.profileName.c_str());
	for(tUInt16 i = 0; i < configDataList.size();i++)
	{
		log(&CMIT,DLT_LOG_INFO, "Key   : ", configDataList[i].key);
		log(&CMIT,DLT_LOG_INFO, "Value : ", configDataList[i].value);
	}
	log(&CMIT,DLT_LOG_INFO,"------------------------------------");

	::boost::shared_ptr<ConfigDataProxy> proxy;

	cfgmgr_ClientDatabase *pCMApp = cfgmgr_ClientDatabase::createProxyDatabaseObject();

	if(NULL != pCMApp)
	{
		tbool status = pCMApp->getProxy(configItem.profileName, proxy,PRIVATE_LIST);

		if(status)
		{
			log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::SetDefaultPrivateData - SET PRIVATE DATA");

			if(NULL != m_pInterface)
			{
				m_pInterface->updatePrivateData(configItem, configDataList, proxy);
			}
		}
		else
		{
			log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::SetDefaultPrivateData - CONFIGITEM NOT REGISTERED");
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::SetDefaultPrivateData pCMApp is NULL");
	}	
}

/***********************************************************************************************************/
/* 	   					Method to GetDefault Private Data to Native/Managed Apps 		 	   			   */
/***********************************************************************************************************/
tVoid cfgmgr_UpdateManager::GetDefaultPrivateData(ConfigItem_T configDataList, DescmoState_T configItem)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::GetDefaultPrivateData");

	log(&CMIT,DLT_LOG_INFO,"REQUEST ITEM : ",configItem.profileName.c_str());
	log(&CMIT,DLT_LOG_INFO,"------------------------------------");

	::boost::shared_ptr<ConfigDataProxy> proxy;

	cfgmgr_ClientDatabase *pCMApp = cfgmgr_ClientDatabase::createProxyDatabaseObject();

	if(NULL != pCMApp)
	{
		tbool status = pCMApp->getProxy(configItem.profileName, proxy,PRIVATE_LIST);

		if(status)
		{
			log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::GetDefaultPrivateData - GET PRIVATE DATA");

			if(NULL != m_pInterface)
			{
				m_pInterface->getPrivateData(configItem,proxy);
			}
		}
		else
		{
			log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::GetDefaultPrivateData - CONFIGITEM NOT REGISTERED");
		}

	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::GetDefaultPrivateData pCMApp is NULL");
	}
}

/***********************************************************************************************************/
/* 	   					Method to SetDefault Public Data to Native/Managed Apps 		 	   			   */
/***********************************************************************************************************/
tVoid cfgmgr_UpdateManager::SetDefaultPublicData(ConfigItem_T configDataList,DescmoState_T configItem)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::SetDefaultPublicData");

	tStr pKey;
	Cfgmgr_ItemStatus_T itemStatus = STATUS_FAILED;
	DatapoolType dp_type = DP_UNKNOWN;
	tBool isActivated = false;
	ConfigItem_T orgConfigDataList;

	/*CSA value check is needed to ensure that CSA is not set to 0 after activating once  - AIVI-66911*/
	cfgmgr_ClientDatabase *pCMApp = cfgmgr_ClientDatabase::createProxyDatabaseObject();

	if(NULL != pCMApp)
	{
		dp_type = pCMApp->getDpType(configItem.profileName,pKey);
	}

	if(dp_type != DP_UNKNOWN)
	{
		itemStatus = m_pDatapoolHandler->getDatapool(configItem.profileName,orgConfigDataList,dp_type);
	}
	else
	{
		log(&CMIT,DLT_LOG_ERROR,"getDatapool for CSA status failed.");
		itemStatus = STATUS_UNKNOWN_ITEM;
	}

	if(itemStatus == STATUS_SUCCESS)
	{
		if((dp_type == DP_CSA) && (orgConfigDataList[0].value == "1"))
		{
			isActivated = true;
			if(configDataList[0].value == "1")
			{
				log(&CFM0,DLT_LOG_INFO,"CSA.Connectivity_and_services_status is Already Activated.CSA can be Activated Only Once.");
			}
			else if(configDataList[0].value == "0")
			{
				log(&CFM0,DLT_LOG_INFO,"CSA.Connectivity_and_services_status is Already Activated.Can not be Reset to 0");
			}
		}
	}

	if(!isActivated)
	{
		itemStatus = setDpData(configDataList,configItem.profileName);

		if(NULL != m_pListener)
		{
			m_pListener->onDefaultDataResponse(configDataList, itemStatus, configItem, "SET");
		}

		if(itemStatus == STATUS_SUCCESS)
		{
			updateClientPublicItem(CFGMGR_UPDATE_ON_SET,configItem.profileName,"");
		}
	}
	else
	{
		if(NULL != m_pListener)
		{
			m_pListener->onDefaultDataResponse(configDataList, STATUS_SUCCESS, configItem, "SET");
		}
	}
}

/***********************************************************************************************************/
/* 	   					Method to GetDefault Public Data to Native/Managed Apps 		 	   			   */
/***********************************************************************************************************/
tVoid cfgmgr_UpdateManager::GetDefaultPublicData(ConfigItem_T configDataList,DescmoState_T configItem)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::GetDefaultPublicData");

	Cfgmgr_ItemStatus_T itemStatus = STATUS_FAILED;

	itemStatus = getDpData(configDataList,configItem.profileName);
	if(NULL != m_pListener)
	{
		m_pListener->onDefaultDataResponse(configDataList, itemStatus, configItem, "GET");
	}
}

/**********************************************************************************************************/

Cfgmgr_ItemStatus_T cfgmgr_UpdateManager::setDpData(ConfigItem_T& configDataList, tStr configItemName)
{
	tStr pKey;
	Cfgmgr_ItemStatus_T itemStatus = STATUS_FAILED;

	cfgmgr_ClientDatabase *pCMApp = cfgmgr_ClientDatabase::createProxyDatabaseObject();

	if((configDataList.size() == CFGMGR_ONE) && (NULL != pCMApp))
	{
		ConfigData_T data = configDataList.at(CFGMGR_ZERO);
		DatapoolType dp_type = pCMApp->getDpType(configItemName,pKey);

		log(&CMIT,DLT_LOG_INFO,"REQUEST ITEM : ",configItemName);
		log(&CMIT,DLT_LOG_INFO,"Key   : ", data.key);
		log(&CMIT,DLT_LOG_INFO,"Value : ", data.value);
		log(&CMIT,DLT_LOG_INFO,"------------------------------------");

		if(dp_type != DP_UNKNOWN)
		{
			if(pKey == data.key)
			{
				itemStatus = m_pDatapoolHandler->setDatapool(data.key, data.value,dp_type);
			}
			else
			{
				log(&CMIT,DLT_LOG_ERROR,"KEY NOT VALID");
				itemStatus = STATUS_UNKNOWN_ITEM;
			}
		}
		else
		{
			log(&CMIT,DLT_LOG_ERROR,"ITEM NOT PRESENT");
			itemStatus = STATUS_UNKNOWN_ITEM;
		}
	}
	else
	{
		log(&CMIT,DLT_LOG_ERROR,"DATALISTSIZE INVALID, SIZE = ",configDataList.size());
	}

	log(&CMIT,DLT_LOG_INFO,"RESPONSE : ",printResult(itemStatus));
	log(&CMIT,DLT_LOG_INFO,"------------------------------------");

	return itemStatus;
}

/************************************************************************************************************/

Cfgmgr_ItemStatus_T cfgmgr_UpdateManager::getDpData(ConfigItem_T& datalist, tStr configItemName)
{
	tStr pKey;
	Cfgmgr_ItemStatus_T itemStatus = STATUS_FAILED;

	cfgmgr_ClientDatabase *pCMApp = cfgmgr_ClientDatabase::createProxyDatabaseObject();
	if(NULL != pCMApp)
	{
		DatapoolType dp_type = pCMApp->getDpType(configItemName,pKey);

		log(&CMIT,DLT_LOG_INFO,"REQUEST ITEM : ", configItemName.c_str());
		log(&CMIT,DLT_LOG_INFO,"------------------------------------");

		if(dp_type != DP_UNKNOWN)
		{
			itemStatus = m_pDatapoolHandler->getDatapool(configItemName,datalist,dp_type);
		}
		else
		{
			log(&CMIT,DLT_LOG_ERROR,"ITEM NOT PRESENT");
			itemStatus = STATUS_UNKNOWN_ITEM;
		}
	}

	log(&CMIT,DLT_LOG_INFO,"RESPONSE : ",printResult(itemStatus));
	if(itemStatus == STATUS_SUCCESS)
	{
		for(tUInt16 i = 0; i < datalist.size();i++)
		{
			log(&CMIT,DLT_LOG_INFO,"Key   : ",datalist[i].key);
			log(&CMIT,DLT_LOG_INFO,"Value : ",datalist[i].value);
		}
	}
	log(&CMIT,DLT_LOG_INFO,"------------------------------------");

	return itemStatus;
}

/**********************************************************************************************************/

tVoid cfgmgr_UpdateManager::sendUpdate(ConfigItem_T& configDataList, tStr profileName)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::sendUpdate");

	vector< ::boost::shared_ptr<ConfigDataProxy> > proxylist;
	cfgmgr_ClientDatabase *pCMApp = cfgmgr_ClientDatabase::createProxyDatabaseObject();

	if((NULL != m_pInterface) && (NULL != pCMApp))
	{
		if(pCMApp->getProxyforRegisteredPublicItem(profileName, proxylist))
		{
			for(tUInt16 count = CFGMGR_ZERO; count < proxylist.size(); count++)
			{
				m_pInterface->updatePublicData(profileName, configDataList, proxylist.at(count));
			}
		}
		else
		{
			log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::sendUpdate - Clients registered for '",profileName,"' is ZERO");
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::sendUpdate - NULL check not satisfied");
	}
}

/**********************************************************************************************************/

tStr cfgmgr_UpdateManager::printResult(Cfgmgr_ItemStatus_T result)
{
	tStr resultStr;
	switch(result)
	{
		case STATUS_SUCCESS: resultStr = "STATUS_SUCCESS" ; break;
		case STATUS_FAILED: resultStr = "STATUS_FAILED" ; break;
		case STATUS_UNKNOWN_ITEM: resultStr = "STATUS_UNKNOWN_ITEM" ; break;
		case STATUS_NO_PERMISSION: resultStr = "STATUS_NO_PERMISSION" ; break;
		case STATUS_INAVLID_VALUE: resultStr = "STATUS_INAVLID_VALUE" ; break;
		case STATUS_INVALID_SIGNATURE: resultStr = "STATUS_INVALID_SIGNATURE" ; break;
	}
    return resultStr;

}

/**********************************************************************************************************/

tVoid cfgmgr_UpdateManager::updateClientPublicItem(UpdateType updateType, tStr reg_ItemName, tStr dbusName)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_UpdateManager::updateClientPublicItem");

	tStr profileName;
	DatapoolType dpType;
	vector<ConfigData_T> publicItem;
	publicItem.clear();
	map<tStr,DatapoolType> publicItemList;

	cfgmgr_ClientDatabase *pCMApp = cfgmgr_ClientDatabase::createProxyDatabaseObject();
	if(NULL != pCMApp)
	{
		pCMApp->getListofAllPublicItemNames(publicItemList);

		map<tStr,DatapoolType>::iterator it;
		for(it = publicItemList.begin(); it != publicItemList.end(); ++it)
		{
			profileName = it->first;
			dpType = it->second;

			if(updateType == CFGMGR_UPDATE_ON_REG)
			{
				::boost::shared_ptr<ConfigDataProxy> proxy;

				if(reg_ItemName == profileName)
				{
					pCMApp->getProxyRegItem(profileName, dbusName,proxy);
					if(NULL != proxy)
					{
						m_pDatapoolHandler->getDatapool(profileName,publicItem,dpType);
						m_pInterface->updatePublicData(profileName, publicItem, proxy);
					}
					else
					{
						log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::updateClientPublicItem, clients not available");
					}
					break;
				}
			}
			else if(updateType == CFGMGR_UPDATE_ON_SET)
			{
				if(reg_ItemName == profileName)
				{
					m_pDatapoolHandler->getDatapool(profileName,publicItem,dpType);
					sendUpdate(publicItem,profileName);
					break;
				}
			}
			else if (updateType == CFGMGR_UPDATE_ON_SSTDS)
			{
				m_pDatapoolHandler->getDatapool(profileName,publicItem,dpType);
				sendUpdate(publicItem,profileName);
			}
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_UpdateManager::updateClientPublicItem, pCMApp is NULL");
	}
	return;
}
