/**
 * @file   : cfgmgr_DataRequestHandler.cpp
 * @author : mrr8kor
 * @date   : Aug 18, 2017
 * @brief  : To handle Data from Ota client to update handler
 * @copyright (c) 2017 Robert Bosch Car Multimedia Gmbh
 * @addgroup : cfgmgr
 * @{
 */

#ifndef cfgmgr_DataRequestHandler_H_
#include "cfgmgr_DataRequestHandler.h"
#endif

cfgmgr_DataRequestHandler* cfgmgr_DataRequestHandler::pRequestHandler = NULL;

/********************************************************************************/
/* 			Function to Get the instance of Update Manager			            */
/********************************************************************************/
cfgmgr_UpdateManager* cfgmgr_DataRequestHandler::getInstanceOfUpdateManager()
{
	return cfgmgr_UpdateManager::getInstanceofUpdateManager();
}

/********************************************************************************/
/* 						cfgmgr_DataRequestHandler Constructor			        */
/********************************************************************************/
cfgmgr_DataRequestHandler::cfgmgr_DataRequestHandler(): /*mpDataPtr(NULL),*/m_pOtaListener(NULL),m_pInterface(NULL)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_DataRequestHandler::cfgmgr_DataRequestHandler Constructor");

#ifdef __OTA_COMPONENT_ENABLE__
	mpDataPtr(NULL)
#endif
	m_VIN.clear();
	m_pInterface = getInstanceOfUpdateManager();
}

/********************************************************************************/
/* 						cfgmgr_DataRequestHandler Destructor			        */
/********************************************************************************/
cfgmgr_DataRequestHandler::~cfgmgr_DataRequestHandler()
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_DataRequestHandler::cfgmgr_DataRequestHandler Destructor");

	m_pOtaListener = NULL;

	if(NULL != pRequestHandler)
	{
		delete pRequestHandler;
		pRequestHandler = NULL;
	}

	if(NULL != m_pInterface)
	{
		m_pInterface = NULL;
	}

#ifdef __OTA_COMPONENT_ENABLE__
	if(NULL != mpDataPtr)
	{
		delete mpDataPtr;
		mpDataPtr = NULL;
	}
#endif

	m_pOtaListener = NULL;
}

/********************************************************************************/
/* 			Function to Get the instance of Data Request Handler                */
/********************************************************************************/
cfgmgr_DataRequestHandler* cfgmgr_DataRequestHandler::getInstanceofDataRequestHandler()
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_DataRequestHandler::getInstanceofDataRequestHandler");

	if (NULL == pRequestHandler)
	{
		pRequestHandler = new cfgmgr_DataRequestHandler();
	}

	return pRequestHandler;
}

/********************************************************************************/
/* 						Function to register client handler				        */
/********************************************************************************/
tVoid cfgmgr_DataRequestHandler::registerClientHandler(cfgmgr_IOtaClientListener* pListener)
{
	m_pOtaListener = pListener;
}

/***********************************************************************************************/
/* perform checks on the data received from Ota client							*/
/***********************************************************************************************/
tVoid cfgmgr_DataRequestHandler::processRequest(DescmoState_T configItem)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_DataRequestHandler::processRequest");

	FormatIdentifier formatId = DEFAULT_FORMAT;
	tbool isCfgMgrRegistered = false;

	stCustomSetData customData;
	map<tStr,tStr> configData;
	configData.clear();
	cfgmgr_descmoResult descmoResult;

	// Check for ECU ID -- 1st level of verification
	if(CFGMGR_ECU_ID ==  configItem.ecuId)
	{
		if(NULL != m_pInterface)
		{
			if( m_pInterface->checkForConfigItemSubscribed(configItem.profileName, formatId, isCfgMgrRegistered) )
			{
				log(&CFM0,DLT_LOG_INFO,"cfgmgr_DataRequestHandler::processRequest, configItem is registered.");

				if(DEFAULT_FORMAT == formatId)
				{
					log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_DataRequestHandler::processRequest, DEFAULT format. Hence process info");
					tStr VINData = m_VIN;

					log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_DataRequestHandler::processRequest, Read from Target VIN --> DESCMO - ", VINData);
					log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_DataRequestHandler::processRequest, XMLDATA              --> DESCMO - ", configItem.xmlData);
					log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_DataRequestHandler::processRequest, SIGNATURE            --> DESCMO - ", configItem.xmlDataSignature);

#ifdef __OTA_COMPONENT_ENABLE__
					mpDataPtr = new cfgmgr_DESCMOHandler(configItem.xmlData, configItem.xmlDataSignature,VINData);

					if(NULL != mpDataPtr)
					{
						descmoResult = mpDataPtr->validateData();
						log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_DataRequestHandler::DESCMO RESULT - ",descmoResult);

						// Use the DESCMO Library to perform the VIN and Signature Check
						if(SUCCESS == descmoResult)
						{
							descmoResult = mpDataPtr->decodeXMLData(configData,customData);

							log(&CFM0,DLT_LOG_DEBUG,"#### Decoded Custom Data:");
							log(&CFM0,DLT_LOG_DEBUG,"customData.Name = ", customData.Name.c_str());
							log(&CFM0,DLT_LOG_DEBUG,"customData.ActionType = ", customData.ActionType.c_str());
							log(&CFM0,DLT_LOG_DEBUG,"customData.MethodName = ", customData.MethodName.c_str());

							map<tStr,tStr>::iterator it;
							for(it = configData.begin(); it != configData.end(); ++it)
							{
								log(&CFM0,DLT_LOG_DEBUG,it->first.c_str(), it->second.c_str());
							}

							if(SUCCESS == descmoResult)
							{
								m_pInterface->setListener(this);

								log(&CFM0,DLT_LOG_DEBUG,"#### Decoded Key-Value Pairs:");
								map<tStr,tStr>::iterator it;
								for(it = configData.begin(); it != configData.end(); ++it)
								{
									log(&CFM0,DLT_LOG_DEBUG,it->first.c_str(), it->second.c_str());
								}

								m_pInterface->SendData(configData, configItem, isCfgMgrRegistered, customData.ActionType);
							}
							else
							{
								log(&CFM0,DLT_LOG_ERROR,"cfgmgr_DataRequestHandler::processRequest -- Decoding xml Data failed");
								sendResponseToOTAComponent(configItem.ecuId, configItem.correlatorId, descmoResult, configItem.xmlData);
							}
						}
						else
						{
							log(&CFM0,DLT_LOG_ERROR,"cfgmgr_DataRequestHandler::processRequest -- VIN and Signature Validation failed");
							sendResponseToOTAComponent(configItem.ecuId, configItem.correlatorId, descmoResult, configItem.xmlData);
						}
						delete mpDataPtr;
						mpDataPtr = NULL;
					}
					else
					{
						log(&CFM0,DLT_LOG_ERROR,"cfgmgr_DataRequestHandler::processRequest -- mpDataPtr is NULL");
						sendResponseToOTAComponent(configItem.ecuId, configItem.correlatorId,FAILED,configItem.xmlData);
					}
#endif
				}
				else if(XML_FORMAT == formatId)
				{
					log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_DataRequestHandler::processRequest, XML format");

					m_pInterface->setListener(this);

					m_pInterface->SendDataXml(configItem);
				}
			}
			else
			{
				log(&CFM0,DLT_LOG_ERROR,"cfgmgr_DataRequestHandler::processRequest -- Config Item not present in the database");
				sendResponseToOTAComponent(configItem.ecuId, configItem.correlatorId,NOT_SUPPORTED,configItem.xmlData);
			}
		}
		else
		{
			log(&CFM0,DLT_LOG_ERROR,"cfgmgr_DataRequestHandler::processRequest -- m_pInterface pointer is NULL");
			sendResponseToOTAComponent(configItem.ecuId, configItem.correlatorId,FAILED,configItem.xmlData);
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"ECU_ID does not match – IGNORED");
	}
}

/********************************************************************************/
/* 						Function to send the response to OTA class              */
/********************************************************************************/
tVoid cfgmgr_DataRequestHandler::sendResponseToOTAComponent(const tUInt32 ECUId,const tUInt32 CorrelatorId,const tUInt32 ErrorCode,const tStr xmlData)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_DataRequestHandler::sendResponseToOTAComponent with ECUId - ", ECUId, ",CorrelatorId - ", CorrelatorId, ",ErrorCode - ", ErrorCode, ",XML Data - ", xmlData.c_str());

	if(this->getInstanceOfUpdateManager() != NULL)
	{
		getInstanceOfUpdateManager()->unsetListener();
	}

	if(NULL != m_pOtaListener)
	{
		m_pOtaListener->sendResponseToComponent(ECUId,CorrelatorId,ErrorCode,xmlData);
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_DataRequestHandler::sendResponseToOTAComponent -- m_pOtaListener pointer is NULL");
	}
}

/***********************************************************************************************/
/* Reading VIN using ConfigItem, without specifying KDS address, specifically				*/
/***********************************************************************************************/
tVoid cfgmgr_DataRequestHandler::ReadVIN()
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_DataRequestHandler::ReadVIN");

	tStr VINData;
	tUInt8 u8BufferVin[CFGMGR_VIN_DATA_LENGTH];
	tUInt32 vinResponse = DP_s32GetConfigItem("Vin", "", u8BufferVin, CFGMGR_VIN_DATA_LENGTH);

	log(&CFM0,DLT_LOG_DEBUG,"VIN RESPONSE - ",vinResponse);
	if ( DP_S32_NO_ERR == vinResponse)
	{
		for(tUInt8 count = CFGMGR_ZERO; count < CFGMGR_VIN_DATA_LENGTH ; count++)
		{
			VINData.append(CFGMGR_ONE,u8BufferVin[count]);
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_DataRequestHandler::ReadVIN ERROR");
	}

	m_VIN = VINData;
}

/************************************************************************************************/
/* 	  Function called from update manager on receiving the Default response from Applications   */
/************************************************************************************************/
void cfgmgr_DataRequestHandler::onDefaultDataResponse(vector<ConfigData_T> configData, Cfgmgr_ItemStatus_T status, DescmoState_T& configItem, tStr actionType)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_DataRequestHandler::onDefaultDataResponse with status - ", status);

	Cfgmgr_ItemStatus_T finalStatus = status;

	tStr xmlData;
	if(STATUS_SUCCESS == status)
	{
		//If Action type is GET then Descmo lib function to be used to convert the received key-value pair into xml format to send it to FOTA
		//Else send the xml data as it is received from FOTA
		if("GET" == actionType)
		{
			map<tStr,tStr> finalDataMap;

			vector<ConfigData_T>::iterator it;
			for(it = configData.begin(); it != configData.end(); it++)
			{
				finalDataMap.insert(pair<tStr,tStr>(it->key,it->value));
			}

			if(0 < finalDataMap.size())
			{
				tStr VINData = m_VIN;

#ifdef __OTA_COMPONENT_ENABLE__
				mpDataPtr = new cfgmgr_DESCMOHandler(configItem.xmlData, configItem.xmlDataSignature,VINData);

				if(NULL != mpDataPtr)
				{
					cfgmgr_descmoResult descmoResult = mpDataPtr->validateData();
					if(descmoResult == SUCCESS)
					{
						if(mpDataPtr->prepareXMLData (finalDataMap, xmlData) == SUCCESS)
						{
							log(&CFM0,DLT_LOG_DEBUG,"Parsed XML Data  - ", xmlData.c_str());
						}
						else
						{
							log(&CFM0,DLT_LOG_ERROR,"cfgmgr_DataRequestHandler::onDefaultDataResponse -- prepareXMLData FAILED");
							finalStatus = STATUS_FAILED;
						}
					}
					else
					{
						log(&CFM0,DLT_LOG_ERROR,"cfgmgr_DataRequestHandler::onDefaultDataResponse -- DESCMO Validation failed during prepare XML");
						finalStatus = STATUS_FAILED;
					}
					delete mpDataPtr;
					mpDataPtr = NULL;
				}
				else
				{
					log(&CFM0,DLT_LOG_ERROR,"cfgmgr_DataRequestHandler::onDefaultDataResponse -- mpDataPtr pointer is NULL");
					finalStatus = STATUS_FAILED;
				}
#endif
			}
			else
			{
				log(&CFM0,DLT_LOG_ERROR,"cfgmgr_DataRequestHandler::onDefaultDataResponse -- Proper Key Value Pairs are not received");

				finalStatus = STATUS_FAILED;
			}
		}
		else
		{
			xmlData = configItem.xmlData;
		}
	}
	else
	{
		xmlData = configItem.xmlData;
	}

#if 0
	cfgmgr_descmoResult descmoResultStatus;
	descmoResultStatus = clientResponseStatusMapping(finalStatus);
	this->sendResponseToOTAComponent(configItem.ecuId, configItem.correlatorId, (uint32)descmoResultStatus, xmlData);
#endif

	this->sendResponseToOTAComponent(configItem.ecuId, configItem.correlatorId, (uint32)finalStatus, xmlData);
}

/********************************************************************************************/
/* 	  Function called from update manager on receiving the XML response from Applications   */
/********************************************************************************************/
tVoid cfgmgr_DataRequestHandler::onXmlDataResponse(tStr xmlData, Cfgmgr_ItemStatus_T status, DescmoState_T& configItem)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_DataRequestHandler::onXmlDataResponse with xmlData - ", xmlData.c_str(), ",status - ", status);
	
	tStr retXML = xmlData;
#if 0
	cfgmgr_descmoResult descmoResultStatus = clientResponseStatusMapping(status);
#endif

	if(STATUS_SUCCESS != status)
	{
		retXML = configItem.xmlData;
	}
	this->sendResponseToOTAComponent(configItem.ecuId, configItem.correlatorId, (tUInt32)status, retXML);
}

#if 0
cfgmgr_descmoResult cfgmgr_DataRequestHandler::clientResponseStatusMapping(Cfgmgr_ItemStatus_T status)
{
	cfgmgr_descmoResult mappedStatus = SUCCESS;

	switch(status)
	{
	case STATUS_SUCCESS: mappedStatus = SUCCESS;
	break;
	case STATUS_FAILED: mappedStatus = FAILED;
	break;
	case STATUS_UNKNOWN_ITEM: mappedStatus = NOT_SUPPORTED;
	break;
	case STATUS_INAVLID_VALUE: mappedStatus = INVALID;
	break;
	default:
	break;
	}

	return mappedStatus;
}
#endif
