/**
 * @file   : cfgmgr_ASFOtaClient.cpp
 * @author : mrr8kor
 * @date   : Aug 24, 2017
 * @brief  : ASF Ota Client implementation for cfgmgr
 * @copyright (c) 2017 Robert Bosch Car Multimedia Gmbh
 * @addgroup : cfgmgr
 * @{
 */
#ifndef CFGMGR_ASF_OTA_CLIENT_H_
#include "cfgmgr_ASFOtaClient.h"
#endif

namespace asf{

using namespace fcota_descmo_ecu_srv::FcOtaDescmoEcuSrv;

/********************************************************************************/
/* 			Method to get the instance of Data Request  Handler class           */
/********************************************************************************/
cfgmgr_DataRequestHandler* cfgmgr_ASFOtaClient::getInstanceofDataRequestHandler()
{
	return cfgmgr_DataRequestHandler::getInstanceofDataRequestHandler();
}

/********************************************************************************/
/* 						cfgmgr_ASFOtaClient constructor  			            */
/********************************************************************************/
cfgmgr_ASFOtaClient::cfgmgr_ASFOtaClient() :BaseComponent(),
      m_poOtaProxy(::fcota_descmo_ecu_srv::FcOtaDescmoEcuSrv::FcOtaDescmoEcuSrvProxy::createProxy("fcOtaDescmoEcuSrvPort",*this)),
      m_pRequestHandler(NULL)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFOtaClient::cfgmgr_ASFOtaClient Constructor");

	m_pRequestHandler = getInstanceofDataRequestHandler();

	if(NULL != m_pRequestHandler)
	{
		m_pRequestHandler->registerClientHandler(this);
	}

	if(NULL != m_pRequestHandler)
	{
		m_pRequestHandler->ReadVIN();
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFOtaClient::cfgmgr_ASFOtaClient -- Data Request Handler is NULL");
	}
}

/********************************************************************************/
/* 						cfgmgr_ASFOtaClient Destructor  			            */
/********************************************************************************/
cfgmgr_ASFOtaClient::~cfgmgr_ASFOtaClient()
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFOtaClient::cfgmgr_ASFOtaClient Destructor");

	m_pRequestHandler = NULL;
}

/***********************************************************************************************/
/* 						Function called once Ota App is available for Communication      	   */
/***********************************************************************************************/
tVoid cfgmgr_ASFOtaClient::onAvailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const ServiceStateChange &stateChange)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFOtaClient::onAvailable");

	// Register the Application to OTA Interface using the Register method
	trRegisterEcuInfo regInfo;
	regInfo.setEcuId(CFGMGR_ECU_ID);
	regInfo.setName(CFGMGR_ECU_NAME);

	if(NULL != m_poOtaProxy)
	{
		log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ASFOtaClient::onAvailable, sendRegisterEcuRequest called");

		m_poOtaProxy->sendRegisterEcuRequest(*this,regInfo);
	}
	else
	{
		log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFOtaClient::onAvailable -- OTA Proxy is NULL");
	}
}

/***********************************************************************************************/
/* 						Function called once Ota App is unavailable for Communication      	   */
/***********************************************************************************************/
tVoid cfgmgr_ASFOtaClient::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const ServiceStateChange &stateChange)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFOtaClient::onUnavailable");

	if(NULL != m_poOtaProxy)
	{
		log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ASFOtaClient::onUnavailable, sendDescmoStateDeregisterAll called");
		m_poOtaProxy->sendDescmoStateDeregisterAll();
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFOtaClient::onUnavailable -- OTA Proxy is NULL");
	}
}

/***********************************************************************************************/
/* 						Function called on update of Descmo State  -- Error case     	       */
/***********************************************************************************************/
tVoid cfgmgr_ASFOtaClient::onDescmoStateError(const ::boost::shared_ptr< FcOtaDescmoEcuSrvProxy >& proxy, const ::boost::shared_ptr< DescmoStateError >& error)
{
	log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFOtaClient::onDescmoStateError called");
}

/***********************************************************************************************/
/* 						Function called on update of Descmo State  -- Success case     	       */
/***********************************************************************************************/
tVoid cfgmgr_ASFOtaClient::onDescmoStateUpdate(const ::boost::shared_ptr< FcOtaDescmoEcuSrvProxy >& proxy, const ::boost::shared_ptr< DescmoStateUpdate >& update)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFOtaClient::onDescmoStateUpdate");

	if(update)
	{
		trDescmoState DescmoData = update->getDescmoState();

		m_descmoData.ecuId = DescmoData.getEcuId();
		m_descmoData.correlatorId = DescmoData.getCorrelatorId();
		m_descmoData.xmlData = DescmoData.getXmlData();
		m_descmoData.xmlDataSignature = DescmoData.getXmlDataSignature();
		m_descmoData.profileName = DescmoData.getProfileName();

		log(&CFM0,DLT_LOG_INFO,"DescmoData: ECU ID = ", m_descmoData.ecuId);
		log(&CFM0,DLT_LOG_INFO,"DescmoData: CORRELATOR ID = ", m_descmoData.correlatorId);
		log(&CFM0,DLT_LOG_INFO,"DescmoData: PROFILE NAME = ", m_descmoData.profileName.c_str());

		if(NULL != m_pRequestHandler)
		{
			m_pRequestHandler->processRequest(m_descmoData);
		}
		else
		{
			log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFOtaClient::onDescmoStateUpdate -- m_pRequestHandler pointer is INVALID");
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFOtaClient::onDescmoStateUpdate -- Update pointer is INVALID");
	}
}
/***********************************************************************************************/
/* 						Function called on request of Register ECu  	-- Error case          */
/***********************************************************************************************/
tVoid cfgmgr_ASFOtaClient::onRegisterEcuError(const ::boost::shared_ptr< FcOtaDescmoEcuSrvProxy >& proxy, const ::boost::shared_ptr< RegisterEcuError >& error)
{
	log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFOtaClient::onRegisterEcuError called");
}

/***********************************************************************************************/
/* 						Function called on request of Register ECu  	-- Success case        */
/***********************************************************************************************/
tVoid cfgmgr_ASFOtaClient::onRegisterEcuResponse(const ::boost::shared_ptr< FcOtaDescmoEcuSrvProxy >& proxy, const ::boost::shared_ptr< RegisterEcuResponse >& response)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFOtaClient::onRegisterEcuResponse called");

	// Check for the status of Register ECU request. If Register is successful then register for the property else try register again
	if(NULL != response )
	{
		trRegisterEcuResponse data = response->getResponse();

		tenRegisterEcuResponse regResponse = data.getResult();

		if(regResponse != tenRegisterEcuResponse__ERROR_INVALID_ECU)
		{
			if(NULL != m_poOtaProxy)
			{
				log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ASFOtaClient::onRegisterEcuResponse, sendDescmoStateRegister called");
				m_poOtaProxy->sendDescmoStateRegister(*this);
			}
		}
		else
		{
			log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFOtaClient::regResponse -- ", regResponse);

			// Perform the Registration Again
			trRegisterEcuInfo regInfo;
			regInfo.setEcuId(CFGMGR_ECU_ID);
			regInfo.setName(CFGMGR_ECU_NAME);

			if(NULL != m_poOtaProxy)
			{
				log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ASFOtaClient::onRegisterEcuResponse, sendRegisterEcuRequest called");
				m_poOtaProxy->sendRegisterEcuRequest(*this,regInfo);
			}
			else
			{
				log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFOtaClient::onRegisterEcuResponse -- OTA Proxy is NULL");
			}
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"RegisterEcuResponse Pointer is NULL");
	}
}

/***********************************************************************************************/
/* 						Function called on request of Set Descmo Result ECu -- Error case      */
/***********************************************************************************************/
tVoid cfgmgr_ASFOtaClient::onSetDescmoResultError(const ::boost::shared_ptr< FcOtaDescmoEcuSrvProxy >& proxy, const ::boost::shared_ptr< SetDescmoResultError >& error)
{
	log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFOtaClient::onSetDescmoResultError");
}

/***********************************************************************************************/
/* 						Function called on request of Set Descmo Result ECu -- Success case    */
/***********************************************************************************************/
tVoid cfgmgr_ASFOtaClient::onSetDescmoResultResponse(const ::boost::shared_ptr< FcOtaDescmoEcuSrvProxy >& proxy, const ::boost::shared_ptr< SetDescmoResultResponse >& response)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFOtaClient::onSetDescmoResultResponse");
}

/***********************************************************************************************/
/* Send the result back to OTA component														*/
/***********************************************************************************************/
tVoid cfgmgr_ASFOtaClient::sendResponseToComponent(const tUInt32 ECUId,const tUInt32 CorrelatorId,const tUInt32 ErrorCode,const tStr xmlData)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFOtaClient::sendResponseToComponent with ECUId - ", ECUId, ",CorrelatorId - ", CorrelatorId, ",ErrorCode - ", ErrorCode, ",xmlData - ", xmlData.c_str());

	trSetDescmoResultInfo result;
	result.setEcuId(ECUId);
	result.setCorrelatorId(CorrelatorId);
	result.setXmlData(xmlData);
	result.setErrorCode(ErrorCode);

	if(m_descmoData.correlatorId == CorrelatorId)
	{
		if(NULL != m_poOtaProxy)
		{
			log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFOtaClient::sendResponseToComponent, sendSetDescmoResultRequest called");
			m_poOtaProxy->sendSetDescmoResultRequest(*this, result);
			log(&CFM0,DLT_LOG_DEBUG,"RESPONSE SENT TO OTA");
		}
		else
		{
			log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFOtaClient::sendResponseToComponent -- OTA Proxy is NULL");
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFOtaClient::sendResponseToComponent -- Correlator ID Mismatch");
	}
}

}
