/**
 * @file   : cfgmgr_ASFAppServer.cpp
 * @author : bma9kor
 * @date   : Aug 24, 2017
 * @brief  : The services listed in cfgmgr_ConfigManagement.fidl file are implemented in these files.
 *			 It allows other applications (clients) to un-/register for properties and set/get of public config items, handled by config manager.
 * @copyright (c) 2017 Robert Bosch Car Multimedia Gmbh
 * @addgroup : cfgmgr
 * @{
 */
#ifndef CFGMGR_ASF_APPSERVER_H_
#include "cfgmgr_ASFAppServer.h"
#endif

namespace asf{

using namespace de::bosch::cm::ConfigManager::ConfigManagement;

/********************************************************************************/
/* Method to get the Instance of Registration Req Handler Class      		    */
/********************************************************************************/
cfgmgr_RegistrationRequestHandler* cfgmgr_ASFAppServer::getInstanceofRegistrationRequestHandler()
{
	return cfgmgr_RegistrationRequestHandler::getInstanceofRegistrationRequestHandler();
}

/********************************************************************************/
/* 							cfgmgr_ASFAppServer constructor		      		    */
/********************************************************************************/
cfgmgr_ASFAppServer::cfgmgr_ASFAppServer():ConfigManagementStub("configPort"),m_pRequestHandler(NULL),mp_LocalDataPtr(NULL)
{
	m_pRequestHandler = getInstanceofRegistrationRequestHandler();
	mp_LocalDataPtr = cfgmgr_LocalDataHandler::getInstanceofLocalDataHandler();

	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppServer object is created successfully");
}

/********************************************************************************/
/* 							cfgmgr_ASFAppServer Destructor		      		    */
/********************************************************************************/
cfgmgr_ASFAppServer::~cfgmgr_ASFAppServer()
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppServer::~cfgmgr_ASFAppServer Destructor");

	if(NULL != m_pRequestHandler)
	{
		m_pRequestHandler = NULL;
	}
	if(NULL != mp_LocalDataPtr)
	{
		mp_LocalDataPtr = NULL;
	}
}

/********************************************************************************/
/* Method to register an application in CfgMgr for private config items		 */
/********************************************************************************/
tVoid cfgmgr_ASFAppServer::onRegisterPrivateItemRequest (const ::boost::shared_ptr< RegisterPrivateItemRequest >& request)
{
	log(&CFM0,DLT_LOG_INFO,"Received cfgmgr_ASFAppServer::onRegisterPrivateItemRequest, configItemName= ", request->getPrivateItemName().c_str());

	Reg_Status_T registrationStatus;
	ClientData_T clientRegData;
	clientRegData.itemName = request->getPrivateItemName();
	clientRegData.dbusName = request->getClientDBusName();
	clientRegData.formatId = (request->getDataFormat() == EnItemFormats__XML) ? XML_FORMAT : DEFAULT_FORMAT;
	clientRegData.itemType = PRIVATE_LIST;

	if(NULL != m_pRequestHandler)
	{
		if((trim(clientRegData.itemName).empty()) || (trim(clientRegData.dbusName).empty()))
		{
			registrationStatus = REG_STATUS_UNKNOWN_ITEM;
		}
		else
		{
			registrationStatus = m_pRequestHandler->registerConfigItem(clientRegData);
		}

		log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ASFAppServer::onRegisterPrivateItemRequest , Reg Status = ", registrationStatus);

		switch(registrationStatus)
		{
		case REG_STATUS_SUCCESS:
		{
			sendRegisterPrivateItemResponse(EnRegisterStatus__SUCCESS);
			break;
		}
		case REG_STATUS_ALREADY_REGISTERED:
		{
			sendRegisterPrivateItemResponse(EnRegisterStatus__ALREADY_REGISTERED);
			break;
		}
		case REG_STATUS_UNKNOWN_ITEM:
		{
			sendRegisterPrivateItemResponse(EnRegisterStatus__UNKNOWN_ITEM);
			break;
		}
		default:
		{
			sendRegisterPrivateItemResponse(EnRegisterStatus__FAILED);
			break;
		}
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppServer::onRegisterPrivateItemRequest -- m_pRequestHandler pointer is NULL");
	}
}

/*********************************************************************************************************************/
/* Method for Applications to UnregisterPrivateData from CfgMgr to no longer receive updates of Custom Settings Group*/
/*********************************************************************************************************************/
tVoid cfgmgr_ASFAppServer::onUnregisterPrivateItemRequest (const ::boost::shared_ptr< UnregisterPrivateItemRequest >& request)
{
	log(&CFM0,DLT_LOG_INFO,"Received cfgmgr_ASFAppServer::onUnregisterPrivateItemRequest, configItemName= ", request->getPrivateItemName().c_str());

	Reg_Status_T registrationStatus;
	ClientData_T clientRegData;
	clientRegData.itemName = request->getPrivateItemName();
	clientRegData.dbusName = request->getClientDBusName();
	clientRegData.formatId = DEFAULT_FORMAT;
	clientRegData.itemType = PRIVATE_LIST;

	if(NULL != m_pRequestHandler)
	{
		if((trim(clientRegData.itemName).empty()) || (trim(clientRegData.dbusName).empty()))
		{
			registrationStatus = REG_STATUS_UNKNOWN_ITEM;
		}
		else
		{
			registrationStatus = m_pRequestHandler->unregisterConfigItem(clientRegData);
		}

		log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ASFAppServer::onUnregisterPrivateItemRequest , Reg Status = ", registrationStatus);

		switch(registrationStatus)
		{
		case REG_STATUS_SUCCESS:
			sendUnregisterPrivateItemResponse(EnRegisterStatus__SUCCESS);
			break;
		case REG_STATUS_ALREADY_UNREGISTERED:
			sendUnregisterPrivateItemResponse(EnRegisterStatus__ALREADY_UNREGISTERED);
			break;
		case REG_STATUS_UNKNOWN_ITEM:
			sendUnregisterPrivateItemResponse(EnRegisterStatus__UNKNOWN_ITEM);
			break;
		default:
			sendUnregisterPrivateItemResponse(EnRegisterStatus__FAILED);
			break;
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppServer::onUnregisterPrivateItemRequest -- m_pRequestHandler pointer is NULL");
	}

}

/**********************************************************************************************************/
/* Method for Applications to RegisterPublicData in CfgMgr to receive updates of public configuration data*/
/**********************************************************************************************************/
tVoid cfgmgr_ASFAppServer::onRegisterPublicItemRequest (const ::boost::shared_ptr< RegisterPublicItemRequest >& request)
{
	log(&CFM0,DLT_LOG_INFO,"Received cfgmgr_ASFAppServer::onRegisterPublicItemRequest, configItemName= ", request->getPublicItemName().c_str());

	Reg_Status_T registrationStatus;
	ClientData_T clientRegData;
	clientRegData.itemName = request->getPublicItemName();
	clientRegData.dbusName = request->getClientDBusName();
	clientRegData.formatId = (request->getDataFormat() == EnItemFormats__XML) ? XML_FORMAT : DEFAULT_FORMAT;
	clientRegData.itemType = PUBLIC_LIST;

	if(NULL != m_pRequestHandler)
	{
		if((trim(clientRegData.itemName).empty()) || (trim(clientRegData.dbusName).empty()))
		{
			registrationStatus = REG_STATUS_UNKNOWN_ITEM;
		}
		else
		{
			registrationStatus = m_pRequestHandler->registerConfigItem(clientRegData);
		}

		log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ASFAppServer::onRegisterPublicItemRequest , Reg Status = ", registrationStatus);

		switch(registrationStatus)
		{
		case REG_STATUS_SUCCESS:
		{
			sendRegisterPublicItemResponse(EnRegisterStatus__SUCCESS);
			(cfgmgr_UpdateManager::getInstanceofUpdateManager())->updateClientPublicItem(CFGMGR_UPDATE_ON_REG,clientRegData.itemName,clientRegData.dbusName);
			break;
		}
		case REG_STATUS_ALREADY_REGISTERED:
			sendRegisterPublicItemResponse(EnRegisterStatus__ALREADY_REGISTERED);
			break;
		case REG_STATUS_UNKNOWN_ITEM:
			sendRegisterPublicItemResponse(EnRegisterStatus__UNKNOWN_ITEM);
			break;
		default:
			sendRegisterPublicItemResponse(EnRegisterStatus__FAILED);
			break;
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppServer::onRegisterPublicItemRequest -- m_pRequestHandler pointer is NULL");
	}
}

/************************************************************************************************************************/
/* Method for Applications to UnregisterPublicData from CfgMgr to no longer receive updates of Public configuration data*/
/************************************************************************************************************************/
tVoid cfgmgr_ASFAppServer::onUnregisterPublicItemRequest (const ::boost::shared_ptr< UnregisterPublicItemRequest >& request)
{
	log(&CFM0,DLT_LOG_INFO,"Received cfgmgr_ASFAppServer::onUnregisterPublicItemRequest, configItemName= ", request->getPublicItemName().c_str());

	Reg_Status_T registrationStatus;
	ClientData_T clientRegData;
	clientRegData.itemName = request->getPublicItemName();
	clientRegData.dbusName = request->getClientDBusName();
	clientRegData.formatId = DEFAULT_FORMAT;
	clientRegData.itemType = PUBLIC_LIST;

	if(NULL != m_pRequestHandler)
	{
		if((trim(clientRegData.itemName).empty()) || (trim(clientRegData.dbusName).empty()))
		{
			registrationStatus = REG_STATUS_UNKNOWN_ITEM;
		}
		else
		{
			registrationStatus = m_pRequestHandler->unregisterConfigItem(clientRegData);
		}

		log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ASFAppServer::onUnregisterPublicItemRequest , Reg Status = ",registrationStatus);

		switch(registrationStatus)
		{
		case REG_STATUS_SUCCESS:
			sendUnregisterPublicItemResponse(EnRegisterStatus__SUCCESS);
			break;
		case REG_STATUS_ALREADY_UNREGISTERED:
			sendUnregisterPublicItemResponse(EnRegisterStatus__ALREADY_UNREGISTERED);
			break;
		case REG_STATUS_UNKNOWN_ITEM:
			sendUnregisterPublicItemResponse(EnRegisterStatus__UNKNOWN_ITEM);
			break;
		default:
			sendUnregisterPublicItemResponse(EnRegisterStatus__FAILED);
			break;
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppServer::onRegisterPublicItemRequest -- m_pRequestHandler pointer is NULL");
	}
}

/***********************************************************************************************/
/* Method for HMI to update the local public items in config manager.						*/
/* Currently there are no local item														*/
/***********************************************************************************************/
tVoid cfgmgr_ASFAppServer::onSetPublicItemRequest (const ::boost::shared_ptr< SetPublicItemRequest >& request)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppServer::onSetPublicItemRequest ,configItemName = ",request->getPublicItemName().c_str());

	tStr configItemName = request->getPublicItemName();
	::std::vector<ConfigData> publicItem = request->getPublicItem();

	ConfigData_T tempData;
	vector<ConfigData_T> localDataItem;
	localDataItem.clear();

	Cfgmgr_ItemStatus_T status = STATUS_FAILED;
	EnItemStatus retStatus = EnItemStatus__FAILED;

	for(tUInt32 count = CFGMGR_ZERO; count < publicItem.size(); count++)
	{
		tempData.key = publicItem[count].getKey();
		tempData.value = publicItem[count].getValue();

		localDataItem.push_back(tempData);
	}

	if(NULL != mp_LocalDataPtr)
	{
		vector<ConfigData_T>::iterator it;
		map<tStr,tStr> dataMap;

		for(it = localDataItem.begin(); it != localDataItem.end(); it++)
		{
			dataMap.insert(pair<tStr,tStr>(it->key,it->value));
		}
		log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ASFAppServer::onSetPublicItemRequest , inserted values to Map ");

		status = mp_LocalDataPtr->SendDataPublic(dataMap,configItemName);

		log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ASFAppServer::onSetPublicItemRequest ,SendPublicData Status = ",status);

		switch(status)
		{
			case STATUS_FAILED : 	retStatus = EnItemStatus__FAILED;
									break;

			case STATUS_SUCCESS : 	retStatus = EnItemStatus__SUCCESS;
									break;

			case STATUS_UNKNOWN_ITEM : 	retStatus = EnItemStatus__UNKNOWN_ITEM;
										break;

			case STATUS_NO_PERMISSION : retStatus = EnItemStatus__NO_PERMISSION;
									   	break;
			default:
				break;
		}
		sendSetPublicItemResponse(retStatus);
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppServer::onSetPublicItemRequest, mp_LocalDataPtr is NULL ");
	}
}

/********************************************************************************************************************/
/* Method for Applications to get Public configuration data from ConfigManager									    */
/* The generic configuration data that applications are interested in. Can either be publicItemName or localItemName*/
/* Currently there are no local item																				*/
/********************************************************************************************************************/
tVoid cfgmgr_ASFAppServer::onGetPublicItemRequest (const ::boost::shared_ptr< GetPublicItemRequest >& request)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppServer::onGetPublicItemRequest, configItemName= ", request->getPublicItemName().c_str());

	std::string configItemName = request->getPublicItemName();
	ConfigItem_T configData;
	configData.clear();

	Cfgmgr_ItemStatus_T status = STATUS_FAILED;
	::std::vector<ConfigData> publicItem;
	publicItem.clear();

	EnItemStatus retStatus = EnItemStatus__FAILED;

	if(NULL != mp_LocalDataPtr)
	{
		status = mp_LocalDataPtr->getPublicData(configItemName,configData);

		switch(status)
		{
			case STATUS_FAILED 			: 	retStatus = EnItemStatus__FAILED;
							   				break;

			case STATUS_SUCCESS 		: 	retStatus = EnItemStatus__SUCCESS;
											break;

			case STATUS_UNKNOWN_ITEM 	: 	retStatus = EnItemStatus__UNKNOWN_ITEM;
											break;

			case STATUS_NO_PERMISSION 	: 	retStatus = EnItemStatus__NO_PERMISSION;
											break;
			default:
				break;
		}

		if(STATUS_SUCCESS == status)
		{
			ConfigData tempData;

			for(tUInt32 count = CFGMGR_ZERO; count < configData.size(); count++)
			{
				tempData.setKey(configData.at(count).key);
				tempData.setValue(configData.at(count).value);

				publicItem.push_back(tempData);
			}
		}
		sendGetPublicItemResponse(publicItem, retStatus);
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppServer::onGetPublicItemRequest, mp_LocalDataPtr is NULL");
	}
}

}
