/*
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2020
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 *
 * FILE                : AVRoutingParser.cpp
 * COMPONENT Name      : AVRoutingParser
 * @details         :
 * AUTHOR              : Sunil Sagar
 * Date                : 27.04.2020
 * Revision History    : 1.0
 * Date 27.04.2020     : Initial version
 * Date 28.05.2020     : Reviewed version
 */

#include "AVRoutingParser.h"
#include "plugin_trace.h"
#include <iostream>
#include <bits/stdc++.h>
using namespace std;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AVROUTING_PARSER
#include "trcGenProj/Header/AVRoutingParser.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

#ifdef TEST_XML
	tCString  XML_PATH = "./AVRoutingAdapter.xml";
#else
	tCString  XML_PATH = "/usr/lib/avdecc/plugins/AVRoutingAdapter.xml";
#endif

#define ATTR_MATCH 0
#define ATTR_NAME "name"
#define ATTR_TYPE "type"
#define ATTR_MODE "mode"
#define ATTR_CLASS "class"
#define ATTR_CAMTXT "camTextID"
#define ATTR_EID "eId"
#define ATTR_IP "IP"
#define ATTR_MAC "mac"
#define ATTR_VAL "value"
#define ATTR_SINKNAM "sinkNam"
#define ATTR_LINK "link"
#define ENT_SET_TAG "tSetting"
#define ROOT_TAG "avdecc_routing_adapter"
#define ENT_CONF_TAG "avdecc_entity_cfg"
#define AVD_GROUP_CONF "avdecc_group_cfg"
#define ENT_TAG  "tENTITY"
#define SINK_TAG "tSINK"
#define ENT_NAME_CMG "CMG"
#define ENT_NAME_CMP "CMP"
#define ENT_NAME_CMA "CMA"
#define HEXADECIMAL_BASE 16
#define MANDATORY_ATTR_COUNT 3
#define MIN_DEV_COUNT 1
#define ATTR_CHANNEL "channel"
#define SRC_TAG "tSOURCE"
#define ATTR_SRCNAM "srcNam"
#define ATTR_POS "pos"
#define ATTR_SOURCE1 "source_1" 
#define ATTR_SOURCE2 "source_2"
#define ATTR_VIEW "view"

tclAvRoutingParser* tclAvRoutingParser::m_pInstance = nullptr;

/*! ***********************************************************************
 *FUNCTION: 		tclAvRoutingParser
 *@details: 		Constr
 *REQUIREMENT ID:	NA
 *@param:		NA
 *@return: 		NA
 *@details
 *HISTORY:
 *05.28.2020		1.0		Sunil Sagar		Reviewed
 ************************************************************************/
tclAvRoutingParser::tclAvRoutingParser():m_bParseState(false)
{
	ETG_TRACE_USR4(("tclAvRoutingParser: Constructor"));
        m_bPosTag = false;
        m_entityCabinChannelMap.clear();	
}

/*! ***********************************************************************
 *FUNCTION: 		tclAvRoutingParser
 *@details: 		Destr
 *REQUIREMENT ID:	NA
 *@param:		NA
 *@return: 		NA
 *@details
 *HISTORY:
 *05.28.2020		1.0		Sunil Sagar		Reviewed
 ************************************************************************/
tclAvRoutingParser::~tclAvRoutingParser()
{
	ETG_TRACE_USR4(("tclAvRoutingParser: Destructor"));
}

/*! ***********************************************************************
 *FUNCTION: 		pGetInstance
 *@details: 		Getting self instance
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		NA
 *@return: 		tclAvRoutingParser ptr
 *@details
 *HISTORY:
 *05.28.2020		1.0		Sunil Sagar		Reviewed
 ************************************************************************/
tclAvRoutingParser* tclAvRoutingParser::pGetInstance()
{
	vInitPlatformEtg();	
	if( nullptr == m_pInstance ) {
		m_pInstance = new tclAvRoutingParser();
	}
	ETG_TRACE_USR4(("tclAvRoutingParser: Parser Instance address = %p",m_pInstance));
	return m_pInstance;
}

/*! ***********************************************************************
 *FUNCTION: 		vExtractLinkInfo
 *@details: 		ExtractLinkInfo
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		TiXmlElement ptr
 *@return: 		tVoid
 *@details
 *HISTORY:
 *06.17.2020		0.1		Sunil Sagar		Initial Version
 ************************************************************************/
tVoid tclAvRoutingParser::vExtractLinkInfo(TiXmlElement* pElement)
{
	if( pElement ) {	
		for( ; pElement; pElement=pElement->NextSiblingElement())
		{
			tCString  pTagName = pElement->Value();
			if ( ATTR_MATCH == strcmp(SINK_TAG, pTagName)){
				std::string linkName, sinkName;
				TiXmlAttribute* pXmlAttribute = pElement->FirstAttribute();
				for ( ; pXmlAttribute;pXmlAttribute = pXmlAttribute->Next()){
					 tCString  pAttribName = pXmlAttribute->Name();
					 tCString  pAttribValue = pXmlAttribute->Value();
					 if ( ATTR_MATCH == strcmp(ATTR_NAME, pAttribName)){
						linkName = pAttribValue;	
					 }
					 else if (ATTR_MATCH == strcmp(ATTR_SINKNAM, pAttribName)){
						sinkName = pAttribValue;
					 }
					 else {
						//Attribute not required
					 } 
				}
				m_entitySinkLinkCabin.insert(std::pair<std::string,std::string>(linkName,sinkName));
			}
		}
	}
}

/*! ***********************************************************************
 *FUNCTION: 		mapGetSinkCabinLinkList
 *@details: 		mapping of entity SinkCabinLink List
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		NA
 *@return: 		tSinkLinkCabinMap
 *@details
 *HISTORY:
 *08.02.2020		0.1		Sunil Sagar		Initial Version
 ************************************************************************/
tSinkLinkCabinMap tclAvRoutingParser::mapGetSinkCabinLinkList()
{
	return m_entitySinkLinkCabin;
}
/*! ***********************************************************************
 *FUNCTION: 		getCabinChannelMap
 *@details: 		mapping of Cabin Channel List
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		NA
 *@return: 		tSinkLinkCabinMap
 *@details
 *HISTORY:
 *08.02.2020		0.1		Sathiya	Initial Version
 ************************************************************************/
tCabinChannelMap tclAvRoutingParser::getCabinChannelMap()
{
	return m_entityCabinChannelMap;
}
/*! ***********************************************************************
 *FUNCTION: 		vValidateDeviceEntry
 *@details: 		Validating Parsed Data
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		NA
 *@return: 		tVoid
 *@details
 *HISTORY:
 *06.17.2020		0.1		Sunil Sagar		Initial Version
 ************************************************************************/
tVoid tclAvRoutingParser::vValidateDeviceEntry()
{
	tU8 u8DeviceCountCMP = 0;
	tU8 u8DeviceCountCMA = 0;	
	for(tU8 u8Index= 0 ; u8Index < m_entityDetail.size();u8Index++ ){
		if( ATTR_MATCH == strcmp(ENT_NAME_CMP,m_entityDetail[u8Index].m_sname.c_str())){	
			u8DeviceCountCMP++;
		}
		else if(ATTR_MATCH == strcmp(ENT_NAME_CMA,m_entityDetail[u8Index].m_sname.c_str())){	
			u8DeviceCountCMA++;
		}
	}
	if(u8DeviceCountCMP >= MIN_DEV_COUNT && u8DeviceCountCMA >= MIN_DEV_COUNT ) {
		ETG_TRACE_USR4(("tclAvRoutingParser: CMP CMA Available"));
	}
	else {
		ETG_TRACE_ERRMEM(("vValidateDeviceEntry():DeviceCountCMP= %d DeviceCountCMA= %d",u8DeviceCountCMP,u8DeviceCountCMA));
		std::string sErrorDesc = "Device Unavailable ";
		m_parsingErrorObj.vSetErrorCode(tclAvRoutingParsingError::AVROUTING_ERROR_DEVICE_UNAVAILABLE);
		m_parsingErrorObj.vSetErrorDescription(sErrorDesc);
	}
}

/*! ***********************************************************************
 *FUNCTION: 		vValidateDeviceEntry
 *@details: 		Validating Parsed Data
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		tEntity
 *@return: 		tVoid
 *@details
 *HISTORY:
 *06.17.2020		0.1		Sunil Sagar		Initial Version
 ************************************************************************/
tVoid tclAvRoutingParser::vValidateDeviceEntry(const tEntity &entityData)
{
	for(tU8 u8Index= 0 ; u8Index < m_entityDetail.size();u8Index++ ){
		if( entityData.m_u64eId == m_entityDetail[u8Index].m_u64eId){
			ETG_TRACE_ERRMEM(("tclAvRoutingParser::vValidateDeviceEntry() - Duplicate Entity Id %lx", entityData.m_u64eId));	
			ETG_TRACE_FATAL(("tclAvRoutingParser::vValidateDeviceEntry() - Duplicate Entity Id %lx", entityData.m_u64eId));
			std::string sErrorDesc = "Duplicate Entity ID ";
			m_parsingErrorObj.vSetErrorCode(tclAvRoutingParsingError::AVROUTING_ERROR_DUPLICATE_ENTITYID);
			m_parsingErrorObj.vSetErrorDescription(sErrorDesc);
		}
		if( ATTR_MATCH == strcmp(entityData.m_sIP.c_str(),m_entityDetail[u8Index].m_sIP.c_str())){
			ETG_TRACE_ERRMEM(("tclAvRoutingParser::vValidateDeviceEntry() - Duplicate IP %s", entityData.m_sIP.c_str()));	
			ETG_TRACE_FATAL(("tclAvRoutingParser::vValidateDeviceEntry() - Duplicate IP %s", entityData.m_sIP.c_str()));
			std::string sErrorDesc = "Duplicate Entity IP ";
			m_parsingErrorObj.vSetErrorCode(tclAvRoutingParsingError::AVROUTING_ERROR_DUPLICATE_ENTITYIP);
			m_parsingErrorObj.vSetErrorDescription(sErrorDesc);
		}
	}
}

/*! ***********************************************************************
 *FUNCTION: 		bParseXml
 *@details: 		Parsing XML
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		NA
 *@return: 		tBool
 *@details
 *HISTORY:
 *05.28.2020		1.0		Sunil Sagar		Reviewed
 ************************************************************************/
tBool tclAvRoutingParser::bParseXml()
{
	if ( m_bParseState ) {
		ETG_TRACE_USR4(("tclAvRoutingParser: Xml already parsed"));
		return true;
	}
	TiXmlDocument xmlDoc(XML_PATH);
	if( xmlDoc.LoadFile()){
		TiXmlHandle xmlHandler(&xmlDoc);
		TiXmlElement* pElement = xmlHandler.FirstChild(ROOT_TAG).FirstChild(ENT_CONF_TAG).FirstChild(ENT_TAG).ToElement();
		if( pElement ) {
			tCString  pTagName = pElement->Value();
			ETG_TRACE_USR4(("tclAvRoutingParser: Tag Name is %s",pTagName));	
			tBool bParseState =false;
			for( ; pElement; pElement=pElement->NextSiblingElement())
			{
				bParseState = bParseEntity(pElement);
			}
			m_bParseState = bParseState;
			vValidateDeviceEntry();
			pElement = xmlHandler.FirstChild(ROOT_TAG).FirstChild(AVD_GROUP_CONF).FirstChild(SINK_TAG).ToElement();
			vExtractLinkInfo(pElement);
		}	
	}
	else {
		//Failed to load file
		ETG_TRACE_ERRMEM(("tclAvRoutingParser::bParseXml() - avroutingadapter.xml load Failed - Error: %s", xmlDoc.ErrorDesc()));
		m_parsingErrorObj.vSetErrorCode(xmlDoc.ErrorId());
		m_parsingErrorObj.vSetErrorDescription(xmlDoc.ErrorDesc());
		m_bParseState = false;
		m_bParseState = false;
	}
	return m_bParseState;
}

/*! ***********************************************************************
 *FUNCTION: 		bParseEntity
 *@details: 		Parsing Entity Tag
 *REQUIREMENT ID:	NA(Utility Function)
 *@param TiXmlElement ptr
 *@return: 		tBool
 *@details
 *HISTORY:
 *05.28.2020		1.0		Sunil Sagar		Reviewed
 ************************************************************************/
tBool tclAvRoutingParser::bParseEntity(TiXmlElement* pElement)
{
	ETG_TRACE_USR4(("tclAvRoutingParser: bParseEntity entered "));
   	if ( !pElement ) {
		ETG_TRACE_ERR(("tclAvRoutingParser:: XML EntityParse Failed "));
		return false;
	}

	TiXmlAttribute* pXmlAttribute = pElement->FirstAttribute();
	tEntity entityDetail;
	tU8 u8AttributeCount= 0;
	for ( ; pXmlAttribute;pXmlAttribute = pXmlAttribute->Next()){
		 tCString  pAttribName = pXmlAttribute->Name();
		 tCString  pAttribValue = pXmlAttribute->Value();
		 ETG_TRACE_USR4(("tclAvRoutingParser: pAttribName %s",pAttribName));
		 ETG_TRACE_USR4(("tclAvRoutingParser: pAttribValue %s",pAttribValue));;
		 if ( ATTR_MATCH == strcmp(ATTR_NAME, pAttribName)){
			entityDetail.m_sname = pAttribValue;
			u8AttributeCount++;
		 }
		 else if (ATTR_MATCH == strcmp(ATTR_EID, pAttribName)){
			char* pEndString = nullptr;
			entityDetail.m_u64eId = std::strtoull(pAttribValue,&pEndString,HEXADECIMAL_BASE);
			u8AttributeCount++;
		 }
		 else if (ATTR_MATCH == strcmp(ATTR_IP, pAttribName)){
			entityDetail.m_sIP = pAttribValue;
			u8AttributeCount++;
		 }
		 else if (ATTR_MATCH == strcmp(ATTR_MAC, pAttribName)){
			entityDetail.m_sMac = pAttribValue;
		 }
		 else {
			//Attribute not required
			ETG_TRACE_USR4(("tclAvRoutingParser: Entity Attribute not required"));
		 } 
	}
	if ( ATTR_MATCH == strcmp(entityDetail.m_sname.c_str(),ENT_NAME_CMG)) {
 		vParseSetting(pElement,entityDetail);
 		vSetEntityLinkInfo(pElement,entityDetail);
	}
	if ( ATTR_MATCH == strcmp(entityDetail.m_sname.c_str(),ENT_NAME_CMA)) {
 		vParseChannelnfo(pElement,entityDetail);
 	}
	if ( ATTR_MATCH == strcmp(entityDetail.m_sname.c_str(),ENT_NAME_CMP)) {
 		vParseCameraInfo(pElement);
	}
	if( MANDATORY_ATTR_COUNT == u8AttributeCount ){
		vValidateDeviceEntry(entityDetail);		
		m_entityDetail.push_back(entityDetail);
	}
	else {
		ETG_TRACE_FATAL(("tclAvRoutingParser:Entity not added attribute missing"));
	}

	return true;
}

/*! ***********************************************************************
 *FUNCTION: 		vSetEntityLinkInfo
 *@details: 		Parsing Link info
 *REQUIREMENT ID:	NA(Utility Function)
 *@param TiXmlElement ptr
 *@param tEntity
 *@return: 		tVoid
 *@details
 *HISTORY:
 *08.02.2020		0.1		Sunil Sagar		Initial Version
 ************************************************************************/
tVoid tclAvRoutingParser::vSetEntityLinkInfo(TiXmlElement* pElement,tEntity &entityDetail)
{
	if( pElement ) {	
		for( pElement = pElement->FirstChildElement(); pElement; pElement=pElement->NextSiblingElement()) {
			tCString  pTagName = pElement->Value();
			if ( ATTR_MATCH == strcmp(SINK_TAG, pTagName)){
				TiXmlAttribute* pXmlAttribute = pElement->FirstAttribute();
				for ( ; pXmlAttribute;pXmlAttribute = pXmlAttribute->Next()){
					 tCString  pAttribName = pXmlAttribute->Name();
					 tCString  pAttribValue = pXmlAttribute->Value();
					 if ( ATTR_MATCH == strcmp(ATTR_LINK, pAttribName)){
						entityDetail.m_sinkLinkName = pAttribValue;	
					 }
					 else {
						//Attribute not required
					 } 
				}
			}
		}
	}

}

/*! ***********************************************************************
 *FUNCTION: 		vParseSetting
 *@details: 		Parsing Setting Tag
 *REQUIREMENT ID:	NA(Utility Function)
 *@param TiXmlElement ptr
 *@param tEntity
 *@return: 		tVoid
 *@details
 *HISTORY:
 *05.28.2020		1.0		Sunil Sagar		Reviewed
 ************************************************************************/
tVoid tclAvRoutingParser::vParseSetting(TiXmlElement* pElement,tEntity &entityDetail)
{
	if(!pElement) {
		ETG_TRACE_ERR(("tclAvRoutingParser:: XML bParseSetting Element null "));
		return;
	}
	for( pElement = pElement->FirstChildElement(); pElement; pElement=pElement->NextSiblingElement()){
		tCString  pTagName = pElement->Value();
		ETG_TRACE_USR4(("tclAvRoutingParser: Tag Name is %s",pTagName));
		if( ATTR_MATCH == strcmp(ENT_SET_TAG, pTagName) ) {
			TiXmlAttribute* pXmlAttribute = pElement->FirstAttribute();
			std::string settingName, settingVal;
			for ( ; pXmlAttribute;pXmlAttribute = pXmlAttribute->Next()){
				 tCString  pAttribName = pXmlAttribute->Name();
				 tCString  pAttribValue = pXmlAttribute->Value();
				 ETG_TRACE_USR4(("tclAvRoutingParser: pAttribName %s",pAttribName));
				 ETG_TRACE_USR4(("tclAvRoutingParser: pAttribValue %s",pAttribValue));
				 if ( ATTR_MATCH == strcmp(ATTR_NAME, pAttribName)){
					settingName = pAttribValue;				
				 }
				 else if (ATTR_MATCH == strcmp(ATTR_VAL, pAttribName)){
					settingVal = pAttribValue;
				 }
				  else {
					//Attribute not required
					ETG_TRACE_USR4(("tclAvRoutingParser: Setting Attribute not required"));
				 } 
			}
			entityDetail.m_entitySetting.insert(std::pair<std::string,std::string>(settingName,settingVal));
		}
	}
}

/*! ***********************************************************************
 *FUNCTION: 		vParseChannelnfo
 *@details: 		Parsing link and channel Tag
 *REQUIREMENT ID:	NA(Utility Function)
 *@param TiXmlElement ptr
 *@param tEntity
 *@return: 		tVoid
 *@details
 *HISTORY:
 *05.28.2020		1.0		Sathiya Intial Version
 ************************************************************************/

tVoid tclAvRoutingParser::vParseChannelnfo(TiXmlElement* pElement,tEntity &entityDetail)
{
        ETG_TRACE_USR4(("tclAvRoutingParser: vParseChannelnfo Entered"));
	if(!pElement) {
		ETG_TRACE_ERR(("tclAvRoutingParser:: XML vParseChannelnfo Element null "));
		return;
	}
	for( pElement = pElement->FirstChildElement(); pElement; pElement=pElement->NextSiblingElement()){
		tCString  pTagName = pElement->Value();
		ETG_TRACE_USR4(("tclAvRoutingParser: Tag Name is %s",pTagName));
		if( ATTR_MATCH == strcmp(SINK_TAG, pTagName) ) {
			TiXmlAttribute* pXmlAttribute = pElement->FirstAttribute();
			std::string linkVal, ChannelVal;
			for ( ; pXmlAttribute;pXmlAttribute = pXmlAttribute->Next()){
				 tCString  pAttribName = pXmlAttribute->Name();
				 tCString  pAttribValue = pXmlAttribute->Value();
				 ETG_TRACE_USR4(("vParseChannelInfo: pAttribName %s",pAttribName));
				 ETG_TRACE_USR4(("vParseChannelInfo: pAttribValue %s",pAttribValue));
				 if ( ATTR_MATCH == strcmp(ATTR_LINK, pAttribName)){
					linkVal = pAttribValue;				
				 }
				 else if (ATTR_MATCH == strcmp(ATTR_CHANNEL, pAttribName)){
					ChannelVal = pAttribValue;
					m_entityCabinChannelMap.push_back({linkVal,ChannelVal});
				 }
				  else {
					//Attribute not required
					ETG_TRACE_USR4(("tclAvRoutingParser: vParseChannelnfo Attribute not required"));
				 } 
			}
			//m_entityCabinChannelMap.insert(std::pair<std::string,std::string>(linkVal,ChannelVal));
		}
	}
	std::vector<std::pair<std::string, std::string> >::iterator itr;
	for(itr = m_entityCabinChannelMap.begin();itr != m_entityCabinChannelMap.end();itr++)
	{
		ETG_TRACE_USR4(("tclAvRoutingParser: vParseChannelnfo first value is %s",(itr->first).c_str()));
		ETG_TRACE_USR4(("tclAvRoutingParser: vParseChannelnfo second value is %s",(itr->second).c_str()));
	}

}

tVoid tclAvRoutingParser::vParseCameraInfo(TiXmlElement* pElement)
{
	if(!pElement) {
		ETG_TRACE_ERR(("tclAvRoutingParser:: XML vParseCameraInfo Element null "));
		return;
	}
	tCamPortInfo stCaminfo = {0,"",0,"","","","","","","","",0};
	tCamData stCamData = {0,"",0};
	tCamPortSplitInfo stSplitCamInfo = {0,"","","","","",0};
	tSplitCamData stSplitCamData = {0,"",0};
	for( pElement = pElement->FirstChildElement(); pElement; pElement=pElement->NextSiblingElement()){
		char* pEndString = nullptr;
		TiXmlAttribute* pXmlAttribute = pElement->Parent()->ToElement()->FirstAttribute();
		for ( ; pXmlAttribute;pXmlAttribute = pXmlAttribute->Next()){
			const char*  pAttribName = pXmlAttribute->Name();
			const char*  pAttribValue = pXmlAttribute->Value();
			if (ATTR_MATCH == strcmp(ATTR_POS,pAttribName)){
				m_bPosTag = true;
			}
		}
		pXmlAttribute = pElement->Parent()->ToElement()->FirstAttribute();
		for ( ; pXmlAttribute;pXmlAttribute = pXmlAttribute->Next()){
			const char*  pAttribName = pXmlAttribute->Name();
			const char*  pAttribValue = pXmlAttribute->Value();
			stCaminfo.pos = 1;
			stSplitCamInfo.pos = 1;
			if (ATTR_MATCH == strcmp(ATTR_POS,pAttribName)){
				stCaminfo.pos = std::strtoull(pAttribValue, &pEndString, HEXADECIMAL_BASE);
				stSplitCamInfo.pos = std::strtoull(pAttribValue, &pEndString, HEXADECIMAL_BASE);    //split position
			}
			else if (ATTR_MATCH == strcmp(ATTR_EID,pAttribName)){
				stCaminfo.u64EntityID = std::strtoull(pAttribValue, &pEndString, HEXADECIMAL_BASE);
			}
			else if (ATTR_MATCH == strcmp(ATTR_IP,pAttribName)){
				stCaminfo.sIP = pAttribValue;
			}
			else{
				//attribute not required
			}
		}
		stCamData.pos = stCaminfo.pos;
		stSplitCamData.pos = stCaminfo.pos;
		const char*  pTagName = pElement->Value();
		if( ATTR_MATCH == strcmp(SRC_TAG, pTagName) ) {
			TiXmlAttribute* pXmlAttribute = pElement->FirstAttribute();
			tS16 cameraVal = 0;
			for ( ; pXmlAttribute;pXmlAttribute = pXmlAttribute->Next()){
				 const char*  pAttribName = pXmlAttribute->Name();
				 const char*  pAttribValue = pXmlAttribute->Value();

				 if(ATTR_MATCH == strcmp(ATTR_NAME,pAttribName)){
				 	stCaminfo.name = pAttribValue;
					stSplitCamInfo.name = pAttribValue;   //split name
					stSplitCamData.srcNam = pAttribValue;  //split name
				 }
				 else if ( ATTR_MATCH == strcmp(ATTR_SRCNAM, pAttribName)){
					stCaminfo.srcNam = pAttribValue;
					stCamData.srcNam = pAttribValue;
					stSplitCamInfo.srcNam = pAttribValue;   //split srcname
				 }
				 else if(ATTR_MATCH == strcmp(ATTR_TYPE, pAttribName)){
				 	stCaminfo.type = pAttribValue;
				 }
				 else if(ATTR_MATCH == strcmp(ATTR_MODE, pAttribName)){
				 	stCaminfo.mode = pAttribValue;
				 }
				 else if(ATTR_MATCH == strcmp(ATTR_CLASS, pAttribName)){
				 	stCaminfo.camclass = pAttribValue;
				 }
				 else if(ATTR_MATCH == strcmp(ATTR_VIEW, pAttribName)){       //View
				 	stSplitCamInfo.view = pAttribValue;
					stCaminfo.view = pAttribValue;
				 }
				 else if(ATTR_MATCH == strcmp(ATTR_SOURCE1, pAttribName)){		//source_1
				 	stSplitCamInfo.source_1 = pAttribValue;
					stCaminfo.source_1 = pAttribValue;
				 }
				 else if(ATTR_MATCH == strcmp(ATTR_SOURCE2, pAttribName)){     //source_2
				 	stSplitCamInfo.source_2 = pAttribValue;
					stCaminfo.source_2 = pAttribValue;
				 }
				 else if (ATTR_MATCH == strcmp(ATTR_CAMTXT, pAttribName)){
					stringstream ssCameraVal(pAttribValue);
					ssCameraVal>>cameraVal;
					stCaminfo.camtextid = cameraVal;
					stCamData.camtextid = cameraVal;
					stSplitCamInfo.camtextid = cameraVal;   //split camtextid
					stSplitCamData.camtextid = cameraVal;  //split camtextid
				 }
				  else {
					//Attribute not required
					ETG_TRACE_USR4(("tclAvRoutingParser: vParseCameraInfo Attribute not required"));
				 } 
			}
			m_caminfo.push_back(stCaminfo);
			m_camData.push_back(stCamData);
			m_camSplitViewInfo.push_back(stSplitCamInfo);
			m_camSplitData.push_back(stSplitCamData);
		}
	}
	
}

/*! ***********************************************************************
 *FUNCTION: 		vGetEntityInfo
 *@details: 		IP and MAC info provider
 *REQUIREMENT ID:	NA(Utility Function)
 *@param tPhylInfo
 *@param devname
 *@param entityVector
 *@return: 		tVoid
 *@details
 *HISTORY:
 *05.28.2020		1.0		Sunil Sagar		Reviewed
 ************************************************************************/
tVoid tclAvRoutingParser::vGetEntityInfo(tPhylInfo phyInfo,tCString pDevName, vector<std::string>&entityData)
{
	for(tU8 u8Index= 0 ; u8Index < m_entityDetail.size();u8Index++ ){
		if( ATTR_MATCH == strcmp(pDevName,m_entityDetail[u8Index].m_sname.c_str())){
			ETG_TRACE_USR4(("tclAvRoutingParser: vGetEntityInfo DevName is matched"));
			if( phyInfo == tPhylInfo::ENTITY_IP){
				entityData.push_back(m_entityDetail[u8Index].m_sIP);
			}
			else if( phyInfo == tPhylInfo::ENTITY_MAC ){
				entityData.push_back(m_entityDetail[u8Index].m_sMac);
			}	
		}
	}
}

/*! ***********************************************************************
 *FUNCTION: 		vectorGetEntityID
 *@details: 		Entity Id provider
 *REQUIREMENT ID:	NA(Utility Function)
 *@param devname
 *@return: 		vector of entity id
 *@details
 *HISTORY:
 *05.28.2020		1.0		Sunil Sagar		Reviewed
 ************************************************************************/
tEntityIdVector tclAvRoutingParser::vectorGetEntityID(tCString pDevName )
{
	tEntityIdVector entityIdData;
	//vector will be filled by Parser
	for(tU8 u8Index= 0 ; u8Index < m_entityDetail.size();u8Index++ ){
		if( ATTR_MATCH == strcmp(pDevName,m_entityDetail[u8Index].m_sname.c_str())){
			ETG_TRACE_USR4(("tclAvRoutingParser: EntityID pushed "));
			entityIdData.push_back(m_entityDetail[u8Index].m_u64eId);
		}
	}
	return entityIdData;
}
/*! ***********************************************************************
 *FUNCTION: 		vectorGetEntityIP
 *@details: 		IP provider
 *REQUIREMENT ID:	NA(Utility Function)
 *@param devname
 *@return: 		vector of IP address
 *@details
 *HISTORY:
 *05.28.2020		1.0		Sunil Sagar		Reviewed
 ************************************************************************/
tEntityIPVector tclAvRoutingParser::vectorGetEntityIP(tCString pDevName )
{
	tEntityIPVector entityIpData;
	//vector will be filled by Parser
	vGetEntityInfo(tPhylInfo::ENTITY_IP,pDevName,entityIpData);
	return entityIpData;
	
}

/*! ***********************************************************************
 *FUNCTION: 		vectorGetEntityMac
 *@details: 		MAC provider
 *REQUIREMENT ID:	NA(Utility Function)
 *@param devname
 *@return: 		vector of Mac address
 *@details
 *HISTORY:
 *05.28.2020		1.0		Sunil Sagar		Reviewed
 ************************************************************************/
tEntityMacVector tclAvRoutingParser::vectorGetEntityMac(tCString pDevName )
{
	tEntityMacVector entityMacData;
	//vector will be filled by Parser
	vGetEntityInfo(tPhylInfo::ENTITY_MAC,pDevName,entityMacData);
	return entityMacData;
	
}

/*! ***********************************************************************
 *FUNCTION: 		bIsEntityIdExist
 *@details: 		Check existence of entity id
 *REQUIREMENT ID:	NA(Utility Function)
 *@param entity id
 *@return: 		tBool
 *@details
 *HISTORY:
 *05.28.2020		1.0		Sunil Sagar		Reviewed
 ************************************************************************/
tBool tclAvRoutingParser::bIsEntityIdExist(const tU64  u64EntityId )
{
	tBool bdeviceState = false;
	for( tU8 u8Index= 0 ; u8Index < m_entityDetail.size();u8Index++ ){
		if( u64EntityId == m_entityDetail[u8Index].m_u64eId) {	
			ETG_TRACE_USR4(("tclAvRoutingParser: EntityID = %x found ",u64EntityId));
			bdeviceState = true;
		}
	}
	ETG_TRACE_USR4(("tclAvRoutingParser: EntityId existence state = %d ",bdeviceState));
	return bdeviceState;
}

/*! ***********************************************************************
 *FUNCTION: 		bIsIpExist
 *@details: 		Check existence of IP address
 *REQUIREMENT ID:	NA(Utility Function)
 *@param IP address
 *@return: 		tBool
 *@details
 *HISTORY:
 *05.28.2020		1.0		Sunil Sagar		Reviewed
 ************************************************************************/
tBool tclAvRoutingParser::bIsIpExist(tCString  pEntityIp)
{
	tBool bdeviceState = false;
	for(tU8 u8Index= 0 ; u8Index < m_entityDetail.size();u8Index++ ){
		if( ATTR_MATCH == strcmp(pEntityIp,m_entityDetail[u8Index].m_sIP.c_str())){	
			ETG_TRACE_USR4(("tclAvRoutingParser: IP exist "));
			bdeviceState = true;
		}
	}
	ETG_TRACE_USR4(("tclAvRoutingParser: IP existence state = %d ",bdeviceState));	
	return bdeviceState;
}

/*! ***********************************************************************
 *FUNCTION: 		mapGetMediaGlassSetting
 *@details : 		Setting of Media Glass Can be found
 *REQUIREMENT ID:	NA(Utility Function)
 *@param entity id
 *@return: 		tBool
 *@details 
 *HISTORY:
 *05.22.2020		1.0		Sunil Sagar		Reviewed
 ************************************************************************/
tSettingNameValueMap tclAvRoutingParser::mapGetMediaGlassSetting(const tU64 u64EntityId)
{
	tSettingNameValueMap mediaGlassSetting;
	for( tU8 u8Index= 0 ; u8Index < m_entityDetail.size();u8Index++ ){
		if( u64EntityId == m_entityDetail[u8Index].m_u64eId){	
			ETG_TRACE_USR4(("tclAvRoutingParser: Media Glass Setting found "));
			mediaGlassSetting = m_entityDetail[u8Index].m_entitySetting;
		}
	}
	return mediaGlassSetting;
}

/*! ***********************************************************************
 *FUNCTION: 		mapGetCompleteDeviceList
 *@details: 		Complete device list in xml
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		NA
 *@return: 		Mapping of entity name and IP
 *@details
 *HISTORY:
 *05.22.2020		1.0		Sunil Sagar		Reviewed
 ************************************************************************/
tEntityNameIPMap tclAvRoutingParser::mapGetCompleteDeviceList()
{
	ETG_TRACE_USR4(("tclAvRoutingParser: sGetCompleteDeviceList "));	
	tEntityNameIPMap deviceList;	
	for( tU8 u8Index= 0 ; u8Index < m_entityDetail.size();u8Index++ ){
		deviceList.insert(std::pair<std::string,std::string>( m_entityDetail[u8Index].m_sname,m_entityDetail[u8Index].m_sIP));	
	}
	return deviceList;
}

/*! ***********************************************************************
 *FUNCTION: 		u8GetErrorCode
 *@details: 		GetErrorCode
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		NA
 *@return: 		Mapping of entity name and IP
 *@details
 *HISTORY:
 *08.03.2020		0.1		Sunil Sagar		Initial Version
 ************************************************************************/
tU8 tclAvRoutingParser::u8GetErrorCode()
{
	return m_parsingErrorObj.u8GetErrorCode();
}

/*! ***********************************************************************
 *FUNCTION: 		sGetErrorDescription
 *@details: 		GetErrorDescription
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		NA
 *@return: 		Mapping of entity name and IP
 *@details
 *HISTORY:
 *08.03.2020		0.1		Sunil Sagar		Initial Version
 ************************************************************************/
std::string tclAvRoutingParser::sGetErrorDescription()
{
	return m_parsingErrorObj.sGetErrorDescription();
}

/*! ***********************************************************************
 *FUNCTION: 		sGetEntitySinkLinkName
 *@details: 		Get Entity Sink Link Name for Cabin Info
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		tU64
 *@return: 		Link Name
 *@details
 *HISTORY:
 *08.03.2020		0.1		Sunil Sagar		Initial Version
 ************************************************************************/
std::string tclAvRoutingParser::sGetEntitySinkLinkName(const tU64 u64EntityId)
{
	std::string sLinkName;
	for( tU8 u8Index= 0 ; u8Index < m_entityDetail.size();u8Index++ ){
		if( u64EntityId == m_entityDetail[u8Index].m_u64eId){	
			ETG_TRACE_USR4(("tclAvRoutingParser: Entity found %lx",u64EntityId));
			sLinkName = m_entityDetail[u8Index].m_sinkLinkName;
		}
	}
	return sLinkName;
}

/*! ***********************************************************************
 *FUNCTION: 		sGetEntityIP
 *@details: 		Get Entity IP with given entity Id
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		tU64
 *@return: 		IP address
 *@details
 *HISTORY:
 *08.03.2020		0.1		Sunil Sagar		Initial Version
 ************************************************************************/
std::string tclAvRoutingParser::sGetEntityIP(const tU64 u64EntityId)
{
	std::string sEntityIP;
	for( tU8 u8Index= 0 ; u8Index < m_entityDetail.size();u8Index++ ){
		if( u64EntityId == m_entityDetail[u8Index].m_u64eId){
			ETG_TRACE_USR4(("tclAvRoutingParser: Entity found %lx",u64EntityId));
			sEntityIP = m_entityDetail[u8Index].m_sIP;
		}
	}
	return sEntityIP;
}

/*! ***********************************************************************
 *FUNCTION: 		vPrintDeviceList
 *@details: 		Complete device list print
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		NA
 *@return: 		Mapping of entity name and IP
 *@details
 *HISTORY:
 *05.22.2020		1.0		Sunil Sagar		Reviewed
 ************************************************************************/
tVoid tclAvRoutingParser::vPrintDeviceList()
{
	ETG_TRACE_USR4(("tclAvRoutingParser: vPrintDeviceList "));
	for( tU8 u8Index= 0 ; u8Index < m_entityDetail.size();u8Index++ ){
		ETG_TRACE_USR4(("Entity Name :-  %s",m_entityDetail[u8Index].m_sname));
		ETG_TRACE_USR4(("Entity Id :-    %lx",m_entityDetail[u8Index].m_u64eId));
		ETG_TRACE_USR4(("Entity IP :-    %s",m_entityDetail[u8Index].m_sIP));
		ETG_TRACE_USR4(("Entity MAC :-   %s",m_entityDetail[u8Index].m_sMac));
		ETG_TRACE_USR4(("Entity LINK :-  %s",m_entityDetail[u8Index].m_sinkLinkName));
		if ( ATTR_MATCH == strcmp(m_entityDetail[u8Index].m_sname.c_str(),ENT_NAME_CMG)) {
			std::map<std::string,std::string>::iterator it;
			for (it = m_entityDetail[u8Index].m_entitySetting.begin();
					it != m_entityDetail[u8Index].m_entitySetting.end(); it++) {
					ETG_TRACE_USR4(("Media Glass Setting Name :-  %s  ",(it->first).c_str()));
					ETG_TRACE_USR4(("Media Glass Setting Value :- %s  ",(it->second).c_str()));
			}
		}
	}
}

/*! ***********************************************************************
 *FUNCTION: 		sGetEntityName
 *@details: 		Get Entity Name with given entity Id
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		tU64
 *@return: 		EntityName
 *@details
 *HISTORY:
 *08.12.2020		0.1		Sunil Sagar		Initial Version
 ************************************************************************/
std::string tclAvRoutingParser::sGetEntityName(const tU64 u64EntityId)
{
	std::string sEntityName;
	for( tU8 u8Index= 0 ; u8Index < m_entityDetail.size();u8Index++ ){
		if( u64EntityId == m_entityDetail[u8Index].m_u64eId){
			ETG_TRACE_USR4(("tclAvRoutingParser: Entity found %lx",u64EntityId));
			sEntityName = m_entityDetail[u8Index].m_sname;
		}
	}
	return sEntityName;
}

tVecCamData tclAvRoutingParser::vecGetCamSrc()
{
	return m_camData;
}

uint16_t tclAvRoutingParser::u16GetCamPos(tU64 u64EntityId)
{
	uint16_t position = 0;
	for_each(m_caminfo.begin(), m_caminfo.end(), [u64EntityId, &position](tCamPortInfo myData){
		if(myData.u64EntityID == u64EntityId)
			position = myData.pos;
	});
	ETG_TRACE_USR4(("tclAvRoutingParser: u16GetCamPos position at %lx",position));
	return position;
}

uint16_t tclAvRoutingParser::u16GetCamPos(std::string strIP)
{
	ETG_TRACE_USR4(("tclAvRoutingParser: u16GetCamPos string at %s",strIP.c_str()));
	uint16_t position = 0;
	for_each(m_caminfo.begin(), m_caminfo.end(), [strIP, &position](tCamPortInfo myData){
		if(strcmp(myData.sIP.c_str(), strIP.c_str()) == ATTR_MATCH)
			position = myData.pos;
	ETG_TRACE_USR4(("tclAvRoutingParser: comparison pass"));	
	});
	ETG_TRACE_USR4(("tclAvRoutingParser: u16GetCamPos IP position at %lx",position));
	return position;
}

tBool tclAvRoutingParser::bIsPosTagAvailable(){
	return m_bPosTag;
}

vecCamPortSplitInfo tclAvRoutingParser::vecGetSplitCamInfo()
{
	return m_camSplitViewInfo;
}
std::vector<std::string> tclAvRoutingParser::getSourceNamesForSplitView(std::string strSrcName, int position)
{
	ETG_TRACE_USR4(("tclAvRoutingParser: getSourceNamesForSplitView string req %s",strSrcName.c_str()));
	std::vector<std::string> vecSrcNamesforSplitView;
	vecSrcNamesforSplitView.clear();
	for(int i =0; i < m_camSplitViewInfo.size(); i++)
	{
		if((strcmp(m_camSplitViewInfo.at(i).name.c_str(), strSrcName.c_str()) == ATTR_MATCH) && (m_camSplitViewInfo.at(i).pos == position))
		{
			vecSrcNamesforSplitView.push_back(m_camSplitViewInfo.at(i).source_1);
			vecSrcNamesforSplitView.push_back(m_camSplitViewInfo.at(i).source_2);
			break;
		}
	}	
	ETG_TRACE_USR4(("tclAvRoutingParser: getSourceNamesForSplitView vec Size %d",vecSrcNamesforSplitView.size()));
	return vecSrcNamesforSplitView;
}
tVecSplitCamData tclAvRoutingParser::vecGetSplitCamSrc()
{
	return m_camSplitData;
}
tVecCamData tclAvRoutingParser::vecGetIPCamSrcData()
{
	tCamData stIPCamData;
	m_IPCameraData.clear();
	for(tU8 u8Index= 0 ; u8Index < m_camData.size();u8Index++ )
	{
		if((0 == strcmp(m_camData[u8Index].srcNam.c_str() , "CMP1_IPCAM")) 
		|| (0 == strcmp(m_camData[u8Index].srcNam.c_str() , "CMP2_IPCAM")) 
		|| (0 == strcmp(m_camData[u8Index].srcNam.c_str() , "CMP3_IPCAM")))
		{
			stIPCamData.pos = m_camData[u8Index].pos;
			stIPCamData.srcNam = m_camData[u8Index].srcNam.c_str();
			stIPCamData.camtextid = m_camData[u8Index].camtextid;
			
			m_IPCameraData.push_back(stIPCamData);
			//Sort the list based on position and send to hmi on request.
			std::sort(m_IPCameraData.begin() , m_IPCameraData.end(), sortIPcameraListOnPosition);
		}	
	
	ETG_TRACE_USR4(("clCameraPlugin:vecGetIPCamSrcData m_IPCameraData size is %d", m_IPCameraData.size()));
	}
	return m_IPCameraData;
}

bool tclAvRoutingParser::sortIPcameraListOnPosition(tCamData& a, tCamData& b)
{
    return a.pos < b.pos;
}
