/* ***************************************************************************************
* FILE:          clPluginQueueThread.cpp
* SW-COMPONENT:  avdecc_appl_plugins
* DESCRIPTION:  clPluginQueueThread.cpp is part of avdecc_appl_plugins library
* COPYRIGHT:  (c) 2020-21 Robert Bosch Car Multimedia GmbH
* HISTORY: Initial Version - 29/05/2020
* AUTHOR:  Supriya Seshadri
* REVISION: 
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
/*****************************************************************
| includes
|----------------------------------------------------------------*/
#include <clPluginQueueThread.h>
#include "clCommunicationProtocol.h"
#include "plugin_trace.h"
#include <iostream>
using namespace std;
#include <map>
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_PLUGIN_MAIN
#define ETG_I_FILE_PREFIX AVDECCPLUGINS::PLUGINQUEUETHREAD::
#include "trcGenProj/Header/clPluginQueueThread.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


#define MAX_PLUGIN_QUEUE_SIZE 100

clPluginDataProvider * clPluginQueueThread::m_pPluginMain = NULL;
/************************************************************************
*FUNCTION: 		clPluginQueueThread()
*DESCRIPTION   :Constructor of the class.
*PARAMETER:		None
*RETURNVALUE:  	None
*HISTORY:
*revision 0.1	ess7kor	29/05/2020
*************************************************************************/
clPluginQueueThread::clPluginQueueThread(clPluginDataProvider *pPluginMain) {
ETG_TRACE_USR4(("clPluginQueueThread: clPluginQueueThread() "));
	m_pPluginMain = pPluginMain;
	m_thPluginQueue = new vector<std::shared_ptr<NotificationData>>(0);
	//pthread_mutexattr_t mattr;
	//pthread_mutexattr_init(&mattr);
	//pthread_mutex_init(&m_thMutexLock,&mattr);
	_pCtrlMap = clControlMapping::getInstance();
	pthread_cond_init(&m_thWaitCond,NULL);
	m_threadID = 0;
}


/************************************************************************
*FUNCTION: 		~clPluginQueueThread()
*DESCRIPTION   :Destructor of the class.
*PARAMETER:		None
*RETURNVALUE:  	None
*HISTORY:
*revision 0.1	ess7kor	29/05/2020
*************************************************************************/
clPluginQueueThread::~clPluginQueueThread(){
	delete m_thPluginQueue;
	_pCtrlMap = NULL;
	ETG_TRACE_USR4(("clPluginQueueThread: ~clPluginQueueThread() "));
}

/************************************************************************
*FUNCTION: 		bCreatePluginQueuethread()
*DESCRIPTION   :Function that creates the End Station Thread.
*PARAMETER:		None
*RETURNVALUE: 	True if Thread creation is successful, false otherwise
*HISTORY:
*revision 0.1	ess7kor	29/05/2020
*************************************************************************/
bool clPluginQueueThread::bCreatePluginQueuethread()
{
	ETG_TRACE_USR4(("clPluginQueueThread: bCreatePluginQueuethread() "));
	
	int status = pthread_create(&m_threadID, NULL, (THREADFUNCPTR)&clPluginQueueThread::thThreadStart,(VOIDPTR)this);
	if( status != 0 )
	{
		ETG_TRACE_FATAL(("Thread creation failed!!"));
		return false;	
	}
	ETG_TRACE_USR4(("clPluginQueueThread: bCreatePluginQueuethread() created"));
	return true;

}
/************************************************************************
*FUNCTION: 		thThreadStart()
*DESCRIPTION   :Thread Main function that Waits for the thread event
*				and calls the extraction of end stations.
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	ess7kor	29/05/2020
*************************************************************************/
void * clPluginQueueThread::thThreadStart(void * args)
{
	ETG_TRACE_USR4(("thThreadStart"));
	cout << "thThreadStart\n" ;
	while(true)
	{
		//Wait for the message through signal
		pthread_mutex_lock(&m_thMutexLock);
		if (0 != pthread_cond_wait(&m_thWaitCond, &m_thMutexLock)){
			break;
		}
		pthread_mutex_unlock(&m_thMutexLock);
		if(!processQueue()){
		ETG_TRACE_FATAL(("bExtractPluginQueueInfo returned false"));
		} 
	}
		ETG_TRACE_USR4(("in clPluginQueueThread()"));
}

/************************************************************************
*FUNCTION: 		setPluginQueueEvent()
*DESCRIPTION:	Sets the event for thread Execution
*PARAMETER:		NotificationData contains entity_id, cmd_type, desc_index
*RETURNVALUE: 	True if enqueue successful, if failed then false
*HISTORY:
*revision 0.1	ess7kor	29/05/2020
*************************************************************************/
bool clPluginQueueThread::setPluginQueueEvent(NotificationData &tData)
{
	ETG_TRACE_USR4(("setPluginQueueEvent"));
	pthread_mutex_lock(&m_thMutexLock);
	std::shared_ptr<NotificationData> sPa(new NotificationData);
	sPa->entity_id = tData.entity_id;
	sPa->cmd_type = tData.cmd_type;
	sPa->desc_index = tData.desc_index;
	bool status = true;
	std::string strEntityName = bFindEntityName(*(sPa.get()));
    pthread_mutex_unlock(&m_thMutexLock);
    m_mapResp = _pCtrlMap->getMappingContent();
	PairEntityDescIndex mapkey = std::pair<std::string, int>(strEntityName, sPa.get()->desc_index);

	
	if (m_mapResp.count(mapkey) > 0)
	{
		for_each(m_mapResp.begin(), m_mapResp.end(), 
		[&status, sPa, strEntityName, this]
		(std::pair<PairEntityDescIndex, PairObjFuncPtr> mapvalue)
		{ 
		PairEntityDescIndex itr = mapvalue.first;
		if (itr.first == strEntityName.c_str() && itr.second == sPa.get()->desc_index) {
			ETG_TRACE_USR4(("clPluginQueueThread::setPluginQueueEvent, entName, %s", strEntityName.c_str()));
			ETG_TRACE_USR4(("clPluginQueueThread::setPluginQueueEvent, controlblock, %d", sPa.get()->desc_index));
			if (!(this->addPluginDataToQueue(sPa)))
			{
				ETG_TRACE_FATAL(("setPluginQueueEvent Failed"));

				status = false;
			} 
			ETG_TRACE_USR4(("setPluginQueueEvent:Element found\n")); 
		}
		});

	}
	pthread_cond_signal(&m_thWaitCond);
	return status;
}


/************************************************************************
*FUNCTION: 		processQueue()
*DESCRIPTION   :Extracts the End stations 
				and stores the details in the Plugin Main Vector.
*PARAMETER:		None
*RETURNVALUE: 	Bool
*HISTORY:
*revision 0.1	ess7kor	29/05/2020
*************************************************************************/
bool clPluginQueueThread::processQueue()
{
	ETG_TRACE_USR4(("clPluginQueueThread::processQueue"));
	std::shared_ptr<NotificationData> myData;
	//int u32EntityIndex = 0;
	std::string entName;
	while( !m_thPluginQueue->empty() )
	{

		pthread_mutex_lock(&m_thMutexLock);
		myData = m_thPluginQueue->at(0);
    	  m_thPluginQueue->erase(m_thPluginQueue->begin());
          entName = bFindEntityName(*(myData.get()));
		pthread_mutex_unlock(&m_thMutexLock);
		ETG_TRACE_USR4(("getEndStationDeails entity_id is %d", myData.get()->entity_id));
		ETG_TRACE_USR4(("getEndStationDetails cmd_type is %d", myData.get()->cmd_type));
		ETG_TRACE_USR4(("getEndStationDetails desc_index is %d", myData.get()->desc_index));

		ETG_TRACE_USR4(("clPluginQueueThread::processQueue, entName, %s", entName.c_str()));
		if(_pCtrlMap != NULL){
			vector<PairObjFuncPtr> vectRespCallback;
			vectRespCallback = _pCtrlMap->getRespCallBack(entName.c_str(), static_cast<int>(myData.get()->desc_index));
            uint16_t cmd_type = myData.get()->cmd_type;
            uint64_t entity_id = myData.get()->entity_id;
			if (vectRespCallback.size() > 0) {
                for_each(vectRespCallback.begin(), vectRespCallback.end(), [cmd_type,entity_id](PairObjFuncPtr n) { (n.first->*n.second)(cmd_type,entity_id); });
			}
			else {
				ETG_TRACE_USR4(("clPluginQueueThread::processQueue, No CallBacks Registered"));
			}
		
		}
		
	} 
	return true;
}

/************************************************************************
*FUNCTION: 		addPluginDataToQueue()
*DESCRIPTION   :Add to queue. If the plugin size exceeds PLUGIN_QUEUE_SIZE
				return false without enqueuing, else enque and return true
*PARAMETER:		NotificationData
*RETURNVALUE: 	bool
*HISTORY:
*revision 0.1	HHD1KOR	04/06/2020
*************************************************************************/
bool clPluginQueueThread::addPluginDataToQueue(std::shared_ptr<NotificationData> pluginData)
{
	ETG_TRACE_USR4(("clPluginQueueThread::addPluginDataToQueue"));
	if( m_thPluginQueue->size() >= MAX_PLUGIN_QUEUE_SIZE )
	{
		ETG_TRACE_ERRMEM(("clPluginQueueThread::addPluginDataToQueue,size() >= PLUGIN_QUEUE_SIZE"));
		return false;
	}
	pthread_mutex_lock(&m_thMutexLock);
	m_thPluginQueue->push_back(pluginData);
	pthread_mutex_unlock(&m_thMutexLock);
	ETG_TRACE_USR4(("clPluginQueueThread::addPluginDataToQueue, %d",m_thPluginQueue->size()));

	if( m_thPluginQueue->size() >= MAX_PLUGIN_QUEUE_SIZE)
	{
		ETG_TRACE_FATAL(("clPluginQueueThread:: queue full"));
	}
	return true;
}
/************************************************************************
*FUNCTION: 		addPluginDataToQueue()
*DESCRIPTION   :This function returns the name of the entity for the 
*				NotificationData passed by comparing entity ids of 
*				NotificationData and vecEndStation
*PARAMETER:		NotificationData
*RETURNVALUE: 	string EntityName
*HISTORY:
*revision 0.1	HHD1KOR	04/06/2020
*************************************************************************/
std::string clPluginQueueThread::bFindEntityName(NotificationData myData)
{
	ETG_TRACE_USR4(("clPluginQueueThread::bFindEntityName entered!"));
	if(m_pPluginMain != NULL){
		//pthread_mutex_lock(&m_thMutexLock);
		vecEndStation = m_pPluginMain->vecGetEndStationVector();
		//pthread_mutex_unlock(&m_thMutexLock);
	}
	
	std::string entityName = "";
	
	for_each(vecEndStation.begin(), vecEndStation.end(), [myData, &entityName](stEntityDesc ESdata){
		ETG_TRACE_USR4(("getEndStationDetails u64EntityID is %d", ESdata.u64EntityID));
		ETG_TRACE_USR4(("getEndStationDetails u32EntityIndex is %d", ESdata.u32EntityIndex));
		ETG_TRACE_USR4(("getEndStationDetails strEndStationName is %s", ESdata.strEndStationName.c_str()));
		ETG_TRACE_USR4(("getEndStationDetails strEntityName is %s",ESdata.strEntityName.c_str()));
		ETG_TRACE_USR4(("getEndStationDetails end_station is %d",ESdata.end_station));
		ETG_TRACE_USR4(("getEndStationDetails entity is %d", ESdata.entity));
		ETG_TRACE_USR4(("getEndStationDetails configuration is %d", ESdata.configuration));
		
		if(ESdata.u64EntityID == myData.entity_id )
		{
			ETG_TRACE_USR4(("clPluginQueueThread::bFindEntityName entName is %s", ESdata.strEntityName.c_str()));
			entityName = ESdata.strEntityName.c_str();
		}
	});
	return entityName;
}
