/* ***************************************************************************************
* FILE:          clPluginDataProvider.cpp
* SW-COMPONENT:  avdecc_appl_plugins
* DESCRIPTION:   clPluginDataProvider.cpp is part of avdecc_appl_plugins library,
*
* COPYRIGHT:  (c) 2020-21 Robert Bosch Car Multimedia GmbH
* HISTORY: Initial Version   18/05/2020
* AUTHOR:Supriya Seshadri
* REVISION:
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "clPluginDataProvider.h"
#include "end_station.h"
#include "entity_descriptor.h"
#include "configuration_descriptor.h"
#include "audio_unit_descriptor.h"
#include "stream_input_descriptor.h"
#include "stream_output_descriptor.h"
#include "control_descriptor.h"
#include "PluginDataType.h"
#include "PluginConstants.h"
#include "clEndStationThread.h"
#include "clPluginQueueThread.h"
#include "plugin_trace.h"
#include "TTFisCmdHandler/TTFisCmdHandler.h"
#include <iostream>
#include "clCameraPlugin.h"
#include "clExternalMediaPlayerPlugin.h"
#include "clSettingsPlugin.h"
#include "clRouterPlugin.h"
#include "clControlMapping.h"
#include "clMasterControlPlugin.h"
using namespace std;
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_PLUGIN_MAIN
#define ETG_I_FILE_PREFIX AVDECCPLUGINS::PLUGINDATAPROVIDER::
#include "trcGenProj/Header/clPluginDataProvider.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

#ifndef PLUGIN_NAME
#define PLUGIN_NAME avdecc_plugin_server_so
#endif

#define CLIENT_PLUGIN_ENTRY(libname) AC_CLIENT_PLUGIN_ENTRY(libname)
#define CLIENT_PLUGIN_EXIT(libname) AC_CLIENT_PLUGIN_EXIT(libname)

clPluginDataProvider* clPluginDataProvider::poSelf = clPluginDataProvider::getInstance();
vector<stEntityDesc> clPluginDataProvider::m_vecEndStations = {};
clEndStationThread * clPluginDataProvider::m_pEndStationThread = NULL;
clPluginQueueThread * clPluginDataProvider::m_pPluginQueueThread = NULL;
clCommunicationProtocol * clPluginDataProvider::m_pCommControl = NULL;
bool clPluginDataProvider::entityDisconnected = false;
vector<uint64_t> clPluginDataProvider::vecReadComplete = {};


/************************************************************************
*NAME        : CLIENT_PLUGIN_ENTRY
*DESCRIPTION : Design section 8.4.1 - PluginBase
************************************************************************/
extern "C" void* CLIENT_PLUGIN_ENTRY(PLUGIN_NAME)()
{
    ETG_TRACE_USR4(("clPluginDataProvider: CLIENT_PLUGIN_ENTRY()"));
    try
    {
       ETG_TRACE_USR4(("plugin instance : %p", clPluginDataProvider::poSelf));
    }
    catch(...)
    {
        ETG_TRACE_ERR(("Exception... %p", clPluginDataProvider::poSelf));
    }

    return clPluginDataProvider::poSelf;
}


/************************************************************************
*NAME        : CLIENT_PLUGIN_EXIT
*DESCRIPTION : Design section 8.4.1 - PluginBase
************************************************************************/
extern "C" void CLIENT_PLUGIN_EXIT(PLUGIN_NAME)(void *inst)
{
    ETG_TRACE_USR4(("clPluginDataProvider: CLIENT_PLUGIN_EXIT()"));

    if(NULL != inst)
    {
       delete reinterpret_cast<clPluginDataProvider*>(inst);
    }
}

/************************************************************************
*NAME        : clPluginDataProvider
*DESCRIPTION : Constructor
************************************************************************/
clPluginDataProvider::clPluginDataProvider()
{
   ETG_TRACE_USR4(("clPluginDataProvider::clPluginDataProvider()"));

    vInitPlatformEtg();   // Start Trace
    clientIf = NULL;
	
	pthread_mutexattr_init(&mattr);
	pthread_mutex_init(&m_thMutexLock,&mattr);

	m_vecEndStations.clear();
	m_PluginHelpers.clear();
	vecReadComplete.clear();

	//Communucation Protocol Object creation logic
	m_pCommControl = new clCommunicationProtocol();

	//End Station Obj and thread creation
	m_pEndStationThread = new clEndStationThread(this);
	if (NULL != m_pEndStationThread) {
		if (!m_pEndStationThread->bCreateEndStationthread()) {
			ETG_TRACE_FATAL(("clPluginDataProvider: bCreateEndStationthread was not created"));
		}
	}

	//Start the PluginQueueThread
	m_pPluginQueueThread = new clPluginQueueThread(this);
	if(NULL != m_pPluginQueueThread) {
		if(!m_pPluginQueueThread->bCreatePluginQueuethread()) {
			ETG_TRACE_FATAL(("clPluginDataProvider: bCreatePluginQueuethread was not created"));
		}
	}
	vCreatePluginHelpers();
	
	TTFisCmdHandler::initialize();
}




/************************************************************************
*NAME        : ~clPluginDataProvider
*DESCRIPTION : Destructor
************************************************************************/
clPluginDataProvider::~clPluginDataProvider()
{
	ETG_TRACE_USR4(("clPluginDataProvider::~clPluginDataProvider()"));

	clientIf = NULL;
	if(NULL != m_pEndStationThread) { delete m_pEndStationThread; }
	if(NULL != m_pPluginQueueThread) { delete m_pPluginQueueThread; }
	if(NULL != m_pCommControl) { delete m_pCommControl; }
	//Destory the instances of the Plugin Helpers
	for_each(m_PluginHelpers.begin(), m_PluginHelpers.end(), [](clPluginBase *ptr)
	{
		delete ptr;
	});	
	
}

/************************************************************************
*NAME        : getInstance
*DESCRIPTION : Design section 8.4.1 - PluginBase
************************************************************************/
clPluginDataProvider* clPluginDataProvider::getInstance()
{
    ETG_TRACE_USR4(("clPluginDataProvider::getInstance()"));
     
	
    if( nullptr == poSelf)
    {
        poSelf = new clPluginDataProvider;
        poSelf->registerPlugin();
    }
    return poSelf;
}

/*
 * get end station descriptor matching the given entityId
 */
avdecc_lib::end_station* clPluginDataProvider::getEndStationDesc(uint64_t entity_id)
{
    ACClientInterface *AccclientIf = m_pCommControl->getACClientInterface();

    uint32_t end_station_count = AccclientIf->get_end_station_count();
    for (uint32_t i = 0; i < end_station_count; i++)
    {
        avdecc_lib::end_station *end_station = AccclientIf->get_end_station_by_index(i);
        if (!end_station)
        {
            continue;
        }

        if (entity_id == end_station->entity_id())
        {
            return end_station;
        }
    }
    return NULL;
}

/************************************************************************
*NAME        : notification_callback
*DESCRIPTION : Design section 8.4.1 - PluginBase
************************************************************************/
void clPluginDataProvider::notification_callback(void * user_obj, int32_t notification_type, uint64_t entity_id, uint16_t cmd_type,
                                                uint16_t desc_type, uint16_t desc_index, uint32_t cmd_status,
                                                ACNotificationId * notification_id)
{

    ETG_TRACE_USR4(("clPluginDataProvider: notification_callback type is %d command type is %d", notification_type, cmd_type));
    ETG_TRACE_USR4(("clPluginDataProvider: desc_index is %d, entity_id is %llx",  desc_index, entity_id));
    bool benumerationstatus = 0;
    if (notification_type == avdecc_lib::END_STATION_CONNECTED)
    {
        avdecc_lib::end_station *end_station = getEndStationDesc(entity_id);
        if (end_station)
        {
            benumerationstatus = end_station->enumeration_status();
            ETG_TRACE_USR4(("clPluginDataProvider: enumeration status is %d", benumerationstatus));
        }
    }

    if ((notification_type == avdecc_lib::END_STATION_READ_COMPLETED)||((notification_type == avdecc_lib::END_STATION_CONNECTED)&& (benumerationstatus ==1))){
		//inform the plugin base about updation of end station availability.	
		//notify end station read complete from the end station thread.
		ETG_TRACE_FATAL(("clPluginDataProvider: END_STATION_READ_COMPLETED %llx", entity_id));
		clPluginDataProvider::entityDisconnected = false;
		
		
		//clPluginDataProvider::getInstance()->clearEndStationDetails();
		pthread_mutex_lock(&m_thMutexLock);
		clPluginDataProvider::vecReadComplete.push_back(entity_id);
		pthread_mutex_unlock(&m_thMutexLock);

		if (NULL != m_pEndStationThread) {
			m_pEndStationThread->setEndStationEvent();
		}
	}
	
	if(notification_type == avdecc_lib::END_STATION_DISCONNECTED){
		ETG_TRACE_FATAL(("clPluginDataProvider: END_STATION_DISCONNECTED %llx", entity_id));
		clPluginDataProvider::entityDisconnected = true;
        m_pEndStationThread->EntityDisConnected(entity_id);
		pthread_mutex_lock(&m_thMutexLock);
		for(auto itr = m_vecEndStations.begin(); itr != m_vecEndStations.end(); ++itr){
			if(itr->u64EntityID == entity_id){
				m_vecEndStations.erase(itr);
				ETG_TRACE_USR4(("clPluginDataProvider: END_STATION_DISCONNECTED Configuration Disconnect: %d", itr->configuration));
				break;
		}
		}
		for(auto itr = vecReadComplete.begin(); itr != vecReadComplete.end(); ++itr){
			if(*itr == entity_id){
				vecReadComplete.erase(itr);
				ETG_TRACE_USR4(("clPluginDataProvider: END_STATION_DISCONNECTED Removed from vecReadComplete"));
				break;
		}
		}
                pthread_mutex_unlock(&m_thMutexLock);
		for(auto EndStation: m_vecEndStations)
		{
			ETG_TRACE_USR4(("clPluginDataProvider: END_STATION_DISCONNECTED m_vecEndStations Entity id:%d", EndStation.u64EntityID));
		}
		ETG_TRACE_FATAL(("clPluginDataProvider: END_STATION_DISCONNECTED vector size:%d", m_vecEndStations.size()));
		if (NULL != m_pEndStationThread) {
			m_pEndStationThread->setEndStationEvent();
		}
	}

    if ((notification_type == avdecc_lib::UNSOLICITED_RESPONSE_RECEIVED || notification_type == avdecc_lib::RESPONSE_RECEIVED) && (cmd_type != AEM_CMD_REGISTER_UNSOLICITED_NOTIFICATION))
    {
		ETG_TRACE_FATAL(("clPluginDataProvider: UNSOLICITED_RESPONSE_RECEIVED"));
		//Push the response to the plugin queue and set an plugin event    
		if(NULL != m_pPluginQueueThread) {
			NotificationData tData;
			tData.desc_index = desc_index;
			tData.entity_id = entity_id;
			tData.cmd_type = cmd_type;
			m_pPluginQueueThread->setPluginQueueEvent(tData);
		}
	}
}


/************************************************************************
*NAME        : acmp_notification_callback
*DESCRIPTION : Design section 8.4.1 - PluginBase
************************************************************************/
void clPluginDataProvider::acmp_notification_callback(void* user_obj, int32_t notification_type, uint16_t cmd_type,
                                                     uint64_t talker_entity_id, uint16_t talker_unique_id,
                                                     uint64_t listener_entity_id, uint16_t listener_unique_id,
                                                     uint32_t cmd_status, ACNotificationId* notification_id)
{
    ETG_TRACE_USR4(("clPluginDataProvider::acmp_notification_callback()"));
}


/************************************************************************
*NAME        : registerPlugin
*DESCRIPTION : Design section 8.4.1 - PluginBase
************************************************************************/
void clPluginDataProvider::registerPlugin()
{
    ETG_TRACE_USR4(("clPluginDataProvider::registerPlugin()"));

    clientIf = AC_CLIENT_REGISTER(this);
    if(NULL != clientIf)
    {
        registerGenCB();
        registerACMPCB();
        ETG_TRACE_USR4(("clPluginDataProvider::registerPlugin(), Registration Successful"));
			ETG_TRACE_FATAL(("clientIf is %d", clientIf));

		if (NULL != m_pCommControl) {
			ETG_TRACE_USR4(("clPluginDataProvider::registerPlugin(), Setting ACClientInterface"));
			m_pCommControl->setACClientInterface(clientIf);
		} 
 

	}
	
}


/************************************************************************
*NAME        : registerGenCB
*DESCRIPTION : Design section 8.4.1 - PluginBase
************************************************************************/
int clPluginDataProvider::registerGenCB()
{
    ETG_TRACE_USR4(("clPluginDataProvider::registerGenCB()"));

    int status = 0;
    if(NULL != clientIf)
    {
       status = clientIf->registerGenCB(notification_callback);
    }
    return status;
}


/************************************************************************
*NAME        : registerACMPCB
*DESCRIPTION : Design section 8.4.1 - PluginBase
************************************************************************/
int clPluginDataProvider::registerACMPCB()
{
    ETG_TRACE_USR4(("clPluginDataProvider::registerACMPCB()"));

    int status = 0;
    if(NULL != clientIf)
    {
       status = clientIf->registerACMPCB(acmp_notification_callback);
    }
    return status;
}
/************************************************************************
*FUNCTION: 		getEndStationDetails()
*DESCRIPTION: 	This function returns the ENd station details matching the Entity Name
*				recieved in the parameter. this function is mostly used by the plugin helpers
*				to get the endstation details before sending some control commands
*PARAMETER:		std::string strEntityName is the Name of the Entity for which ENdstation details is requested
				stEntityDesc &tEntityDetails is the End station details
*RETURNVALUE: 	bool status that indicates if the endstation details were found (1) or not (0)
*HISTORY:
*revision 0.1	ess7kor	08/05/2020
*************************************************************************/
bool clPluginDataProvider::getEndStationDetails(std::string strEntityName, stEntityDesc &stEntityDetails)
{
	ETG_TRACE_FATAL(("clPluginDataProvider: getEndStationDetails EntityName is %s", strEntityName.c_str()));

	//To get the end station details, the recieved entity name will be matched against
	//the entityname found in each entity detail structure from the vector
	//and if found the entire entity structure is copied and status is sent as TRUE
	bool bfound = false;
	ETG_TRACE_FATAL(("clPluginDataProvider: getEndStationDetailsStruct: vector Count: %d ", m_vecEndStations.size()));
	pthread_mutex_lock(&m_thMutexLock);
	std::for_each(m_vecEndStations.begin(), m_vecEndStations.end(),
		[strEntityName, &stEntityDetails, &bfound](const stEntityDesc& s) {
		ETG_TRACE_USR4(("clPluginDataProvider: getEndStationDetailsStruct:Inside strEntityName is %s",s.strEntityName.c_str()));	
		if (strcmp(s.strEntityName.c_str(),strEntityName.c_str()) == 0) {
			bfound = true;
			ETG_TRACE_FATAL(("clPluginDataProvider:getEndStationDetailsStruct: Element Found "));
			stEntityDetails = s;
			
		}
		
	});
	pthread_mutex_unlock(&m_thMutexLock);
	if(bfound)
	{
	ETG_TRACE_USR4(("getEndStationDetailsStruct u64EntityID is %d", stEntityDetails.u64EntityID));
	ETG_TRACE_USR4(("getEndStationDetailsStruct u32EntityIndex is %d", stEntityDetails.u32EntityIndex));
	ETG_TRACE_USR4(("getEndStationDetailsStruct strEndStationName is %s", stEntityDetails.strEndStationName.c_str()));
	ETG_TRACE_USR4(("getEndStationDetailsStruct strEntityName is %s",stEntityDetails.strEntityName.c_str()));
	ETG_TRACE_USR4(("getEndStationDetailsStruct end_station is %d",stEntityDetails.end_station));
	ETG_TRACE_USR4(("getEndStationDetailsStruct entity is %d", stEntityDetails.entity));
	ETG_TRACE_USR4(("getEndStationDetailsStruct configuration is %d", stEntityDetails.configuration));

	
	}
	else{
	ETG_TRACE_FATAL(("getEndStationDetailsStruct not found element for %s", strEntityName.c_str()));
	}
	return bfound;
}

bool clPluginDataProvider::getEndStationDetails(std::string strEntityName, vector<stEntityDesc> &stEntityDetails)
{
	ETG_TRACE_FATAL(("clPluginDataProvider: getEndStationDetails EntityName is %s", strEntityName.c_str()));

	//To get the end station details, the recieved entity name will be matched against
	//the entityname found in each entity detail structure from the vector
	//and if found the entire entity structure is copied and status is sent as TRUE
	bool bfound = false;
	ETG_TRACE_FATAL(("clPluginDataProvider: getEndStationDetailsVect: vector Count: %d ", m_vecEndStations.size()));
	pthread_mutex_lock(&m_thMutexLock);
	std::for_each(m_vecEndStations.begin(), m_vecEndStations.end(),
		[strEntityName, &stEntityDetails, &bfound](const stEntityDesc& s) {
		ETG_TRACE_USR4(("clPluginDataProvider: getEndStationDetailsVect:Inside strEntityName is %s",s.strEntityName.c_str()));	
		if (strcmp(s.strEntityName.c_str(),strEntityName.c_str()) == 0) {
			bfound = true;
			ETG_TRACE_FATAL(("clPluginDataProvider: getEndStationDetailsVect:Element Found "));
			stEntityDetails.push_back(s);
			
		}
		
	});
	pthread_mutex_unlock(&m_thMutexLock);
	
	if(bfound)
	{
		std::for_each(stEntityDetails.begin(), stEntityDetails.end(),
		[](const stEntityDesc ReqEnt){
			ETG_TRACE_USR4(("getEndStationDetailsVect u64EntityID is %d", ReqEnt.u64EntityID));
			ETG_TRACE_USR4(("getEndStationDetailsVect u32EntityIndex is %d", ReqEnt.u32EntityIndex));
			ETG_TRACE_USR4(("getEndStationDetailsVect strEndStationName is %s", ReqEnt.strEndStationName.c_str()));
			ETG_TRACE_USR4(("getEndStationDetailsVect strEntityName is %s",ReqEnt.strEntityName.c_str()));
			ETG_TRACE_USR4(("getEndStationDetailsVect end_station is %d",ReqEnt.end_station));
			ETG_TRACE_USR4(("getEndStationDetailsVect entity is %d", ReqEnt.entity));
			ETG_TRACE_USR4(("getEndStationDetailsVect configuration is %d", ReqEnt.configuration));
		});
	}
	else{
	ETG_TRACE_FATAL(("getEndStationDetailsVect not found element for %s", strEntityName.c_str()));
	}
	return bfound;
}

/************************************************************************
*FUNCTION: 		setEndStationDetails()
*DESCRIPTION: 	This Function will store the End station Details in the vector m_vecEndStations.
*				This function will be called by the clPluginThread class which extracting the
*				Notification callback endtstion details.
*PARAMETER:		stEntityDesc &tEntityDetails is the End station details to be stored
*RETURNVALUE: 	none
*HISTORY:
*revision 0.1	ess7kor	08/05/2020
*************************************************************************/
void clPluginDataProvider::setEndStationDetails(stEntityDesc &tEntityDetails)
{

	ETG_TRACE_USR4(("setEndStationDetails u64EntityID is %d", tEntityDetails.u64EntityID));
	ETG_TRACE_USR4(("setEndStationDetails u32EntityIndex is %d", tEntityDetails.u32EntityIndex));
	ETG_TRACE_USR4(("setEndStationDetails strEndStationName is %s", tEntityDetails.strEndStationName.c_str()));
	ETG_TRACE_USR4(("setEndStationDetails strEntityName is %s",tEntityDetails.strEntityName.c_str()));
	ETG_TRACE_USR4(("setEndStationDetails end_station is %d",tEntityDetails.end_station));
	ETG_TRACE_USR4(("setEndStationDetails entity is %d", tEntityDetails.entity));
	ETG_TRACE_USR4(("setEndStationDetails configuration is %d", tEntityDetails.configuration));

	//Push it to the list of End station vector
	pthread_mutex_lock(&m_thMutexLock);
	m_vecEndStations.push_back(tEntityDetails);
	pthread_mutex_unlock(&m_thMutexLock);
	ETG_TRACE_USR4(("setEndStationDetails before removing vecReadComplete"));
	eraseVecReadComplete(tEntityDetails.u64EntityID);
	ETG_TRACE_USR4(("setEndStationDetails size of vecReadComplete %d", clPluginDataProvider::vecReadComplete.size()));

}
/************************************************************************
*FUNCTION: 		clearEndStationDetails()
*DESCRIPTION: 	This Function clears the vector which has all the end station Details
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	ess7kor	29/05/2020
*************************************************************************/
void clPluginDataProvider::clearEndStationDetails()
{
	pthread_mutex_lock(&m_thMutexLock);
	m_vecEndStations.clear();
	pthread_mutex_unlock(&m_thMutexLock);

}
/************************************************************************
*FUNCTION: 		vCreatePluginHelpers()
*DESCRIPTION: 	This Function creates all the plugin helper objects
* 				and stores it in the vector m_PluginHelpers
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	ess7kor	06/17/2020
*************************************************************************/
void clPluginDataProvider::vCreatePluginHelpers()
{
	m_PluginHelpers.push_back(clCameraPlugin::getInstance());	
	m_PluginHelpers.push_back(clExternalMediaPlayerPlugin::getInstance());	
	m_PluginHelpers.push_back(clSettingsPlugin::getInstance());
	m_PluginHelpers.push_back(clMasterControlPlugin::getInstance());
	m_PluginHelpers.push_back(clRouterPlugin::getInstance());
	
}

/************************************************************************
*FUNCTION: 		vNotifyESReadComplete()
*DESCRIPTION: 	This Function Notifies the End Station Read Complete
*				Event to all the plugin Helpers
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	ess7kor	06/17/2020
*************************************************************************/
void clPluginDataProvider::vNotifyESReadComplete()
{
    #if 0
    //Clear the Mapping table so that all plugin handlers can re-register
	//callbacks based on the availablity of the end stations.
	clControlMapping *p_ControlMap = clControlMapping::getInstance();
	if(NULL != p_ControlMap)
	{
		p_ControlMap->clearMappingContent();
	}
    #endif
	//Notify all the Plugin Helpers to resgster for the callbacks
	for_each(m_PluginHelpers.begin(), m_PluginHelpers.end(), [](clPluginBase *ptr)
	{
		ptr->vEventESReadComplete();
	});	
}

/************************************************************************
*FUNCTION: 		vecGetEndStationVector()
*DESCRIPTION: 	Getter function for m_vecEndStations
*PARAMETER:		None
*RETURNVALUE: 	vector<stEntityDesc>
*HISTORY:
*revision 0.1	ess7kor	06/17/2020
*************************************************************************/
vector<stEntityDesc> clPluginDataProvider::vecGetEndStationVector()
{
	pthread_mutex_lock(&m_thMutexLock);
	vector<stEntityDesc> m_vecEndStationsLocal = m_vecEndStations;
	pthread_mutex_unlock(&m_thMutexLock);
	return m_vecEndStationsLocal;
}

bool clPluginDataProvider::isEntityDisconnected()
{
	return clPluginDataProvider::entityDisconnected;
}
std::vector<stCamDesc> clPluginDataProvider::vecGetCamDesc()
{
	return m_pEndStationThread->vecGetCamDesc();
}

std::vector<uint64_t> clPluginDataProvider::vecGetReadComplete()
{
	return clPluginDataProvider::vecReadComplete;
}

void clPluginDataProvider::eraseVecReadComplete(uint64_t u64EntityID)
{
	for(auto itr = clPluginDataProvider::vecReadComplete.begin(); itr != clPluginDataProvider::vecReadComplete.end(); itr++)
	{
		ETG_TRACE_USR4(("setEndStationDetails Inside for loop"));
		if(*itr == u64EntityID)
		{
			ETG_TRACE_USR4(("setEndStationDetails removing vecReadComplete"));
			pthread_mutex_lock(&m_thMutexLock);
			clPluginDataProvider::vecReadComplete.erase(itr);
			pthread_mutex_unlock(&m_thMutexLock);
			break;
		}
	}
}
