/* ***************************************************************************************
* FILE:          clEndStationThread.cpp
* SW-COMPONENT:  avdecc_appl_plugins
* DESCRIPTION:  clEndStationThread.cpp is part of avdecc_appl_plugins library
* COPYRIGHT:  (c) 2020-21 Robert Bosch Car Multimedia GmbH
* HISTORY: Initial Version - 29/05/2020
* AUTHOR:  Supriya Seshadri
* REVISION: 
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
/*****************************************************************
| includes
|----------------------------------------------------------------*/
#include <clEndStationThread.h>
#include "clCommunicationProtocol.h"
#include "plugin_trace.h"
#include "end_station.h"
#include "entity_descriptor.h"
#include "configuration_descriptor.h"
#include "audio_unit_descriptor.h"
#include "stream_input_descriptor.h"
#include "stream_output_descriptor.h"
#include "control_descriptor.h"
#include <iostream>
#include "CommandToPlugin/CommandToPlugin.h"
#include "PluginConstants.h"
#include "PluginServerHandler.h"
using namespace std;
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_PLUGIN_MAIN
#define ETG_I_FILE_PREFIX AVDECCPLUGINS::ENDSTATIONTHREAD::
#include "trcGenProj/Header/clEndStationThread.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


clPluginDataProvider * clEndStationThread::m_pPluginMain = NULL;
/************************************************************************
*FUNCTION: 		clEndStationThread()
*DESCRIPTION   :Constructor of the class.
*PARAMETER:		None
*RETURNVALUE:  	None
*HISTORY:
*revision 0.1	ess7kor	29/05/2020
*************************************************************************/
clEndStationThread::clEndStationThread(clPluginDataProvider *pPluginMain) {
ETG_TRACE_USR4(("clEndStationThread: clEndStationThread() "));
	m_pPluginMain = pPluginMain;
	pthread_cond_init(&m_thWaitCond,NULL);
	m_threadID = 0;
	m_pEntityInfo = tclAvRoutingParser::pGetInstance();
	m_bParseState = false;
	vEndStationList.clear();
	vecStEntityDetailsToAppl.clear();
}


/************************************************************************
*FUNCTION: 		~clEndStationThread()
*DESCRIPTION   :Destructor of the class.
*PARAMETER:		None
*RETURNVALUE:  	None
*HISTORY:
*revision 0.1	ess7kor	29/05/2020
*************************************************************************/
clEndStationThread::~clEndStationThread(){
ETG_TRACE_USR4(("clEndStationThread: ~clEndStationThread() "));
vEndStationList.clear();
vecStEntityDetailsToAppl.clear();
}

/************************************************************************
*FUNCTION: 		bCreateEndStationthread()
*DESCRIPTION   :Function that creates the End Station Thread.
*PARAMETER:		None
*RETURNVALUE: 	True if Thread creation is successful, false otherwise
*HISTORY:
*revision 0.1	ess7kor	29/05/2020
*************************************************************************/
bool clEndStationThread::bCreateEndStationthread()
{
	ETG_TRACE_USR4(("clEndStationThread: bCreateEndStationthread() "));
	int status = pthread_create(&m_threadID, NULL, (THREADFUNCPTR)&clEndStationThread::thThreadStart,(VOIDPTR)this);
	if( status != 0 )
	{
		ETG_TRACE_FATAL(("Thread creation failed!!"));
		return false;	
	}
	m_bParseState = m_pEntityInfo->bParseXml();
	return true;
}
/************************************************************************
*FUNCTION: 		thThreadStart()
*DESCRIPTION   :Thread Main function that Waits for the thread event
*				and calls the extraction of end stations.
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	ess7kor	29/05/2020
*************************************************************************/
void * clEndStationThread::thThreadStart(void * args)
{
    ETG_TRACE_USR4(("thThreadStart this is %d", this));
	
	while(true)
	{
		//Wait for the message through signal
		pthread_mutex_lock(&m_thMutexLock);
 		if (0 != pthread_cond_wait(&m_thWaitCond, &m_thMutexLock)){
			break;
		} 
		pthread_mutex_unlock(&m_thMutexLock);
		if(bExtractEndStationInfo()){
		   ETG_TRACE_FATAL(("bExtractEndStationInfo returned false"));
		}
	}
}

/************************************************************************
*FUNCTION: 		setEndStationEvent()
*DESCRIPTION:	Sets the event for thread Execution
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	ess7kor	29/05/2020
*************************************************************************/
void clEndStationThread::setEndStationEvent()
{
	ETG_TRACE_USR4(("setEndStationEvent"));
	pthread_cond_signal(&m_thWaitCond);
}

/************************************************************************
*FUNCTION: 		bExtractEndStationInfo()
*DESCRIPTION   :Extracts the End stations 
				and stores the details in the Plugin Main Vector.
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	ess7kor	29/05/2020
*************************************************************************/
bool clEndStationThread::bExtractEndStationInfo()
{
	
	ETG_TRACE_USR4(("bExtractEndStationInfo"));
	clCommunicationProtocol objCommProto;
	ACClientInterface *clientIf = objCommProto.getACClientInterface();
	ETG_TRACE_USR4(("bExtractEndStationInfo clientIf %d", clientIf));
	bool bMatchFound = false;
	vector<uint64_t> vecReadComplete = m_pPluginMain-> vecGetReadComplete();
	
	if(!(m_pPluginMain->isEntityDisconnected())){
		while(!vecReadComplete.empty()){
			ETG_TRACE_USR4(("bExtractEndStationInfo size of vecReadComplete %d", vecReadComplete.size()));	
			if(NULL != clientIf && !(m_pPluginMain->isEntityDisconnected()))
    {
       uint32_t end_station_count = clientIf->get_end_station_count();
       ETG_TRACE_USR4(("bExtractEndStationInfo End station count: %d", end_station_count));
	   
	   if (vEndStationList.size() > end_station_count)
	   {
		 ETG_TRACE_USR4(("bExtractEndStationInfo vEndStationList count is > than end_station_count: %d", vEndStationList.size()));  
	   }
	   if(!vecReadComplete.empty()){
	   if(!(m_pEntityInfo->bIsEntityIdExist(vecReadComplete.at(0)))){
		   m_pPluginMain->eraseVecReadComplete(vecReadComplete.at(0));
		   vecReadComplete.erase(vecReadComplete.begin());
	   }
	   else
	   {
		   bMatchFound = false;

       for (uint32_t endStationIndex = 0; endStationIndex < end_station_count; endStationIndex++)
       {
           avdecc_lib::end_station* end_station = clientIf->get_end_station_by_index(endStationIndex);
           if(NULL != end_station)
           {
              uint32_t entity_count = end_station->entity_desc_count();
              ETG_TRACE_USR4(("bExtractEndStationInfo, Entity count: %d", entity_count));

              for (uint32_t entityIndex = 0; entityIndex < entity_count; entityIndex++)
              {
                  avdecc_lib::entity_descriptor* entity = end_station->get_entity_desc_by_index(entityIndex);
                  if(NULL != entity)
                  {
                     uint32_t config_count = entity->config_desc_count();
                     ETG_TRACE_USR4(("Configuration count : %d", config_count));

                     for (uint32_t configIndex = 0; configIndex < config_count; configIndex++)
                     {
                         avdecc_lib::entity_descriptor_response* entity_resp = entity->get_entity_response();
                         if(NULL != entity_resp)
                         {
                            uint8_t* obj_name = entity_resp->entity_name();
                            uint64_t uint64EntityID = entity_resp->entity_id();
							if(!vecReadComplete.empty()){
							if((m_pEntityInfo->bIsEntityIdExist(uint64EntityID)) && (vecReadComplete.at(0) == uint64EntityID))
							{
							bMatchFound = true;
							stEntityDesc tEntityDetails;
							stCamDesc tCamDetails;
							tCamDetails.position = 0;
							tEntityDetails.u64EntityID = entity_resp->entity_id();						
							tEntityDetails.strEntityName = m_pEntityInfo->sGetEntityName(tEntityDetails.u64EntityID);
							tEntityDetails.strEndStationName = "";
							tEntityDetails.end_station = end_station;
							tEntityDetails.entity = entity;
							tEntityDetails.configuration = entity->get_config_desc_by_index(configIndex);
							if(strcmp(tEntityDetails.strEntityName.c_str(), COACHMEDIA_CAMERAPORT_1) == 0)
							{
								tCamDetails.CamEntityDesc = tEntityDetails;
								tCamDetails.position = m_pEntityInfo->u16GetCamPos(tEntityDetails.u64EntityID);
								if(!m_pEntityInfo->bIsPosTagAvailable() && m_pEntityInfo->vectorGetEntityID(CMP).size()==1)
								{
								    ETG_TRACE_USR4(("bExtractEndStationInfo, single CMP without POS tag"));
									tCamDetails.position = PRIMARY_CAMPORT;
								}
								vecStCamDesc.push_back(tCamDetails);
							}

							if(tCamDetails.position == CAMPORT_POSITION_UNAVAILABLE) // 0 indicates not CMP checking for 1 indicates adding CMP only if its primary CMP
							{
								//need to add all the entities and need to add CMP, only if its primary
								ETG_TRACE_USR4(("bExtractEndStationInfo, Entity name pushed: %s", tEntityDetails.strEntityName.c_str()));
								vEndStationList.push_back(tEntityDetails.strEntityName);
							}
							else if(tCamDetails.position == PRIMARY_CAMPORT)
							{
								if(m_pEntityInfo->bIsPosTagAvailable())
								{
									ETG_TRACE_USR4(("bExtractEndStationInfo, normal usecase"));
									vEndStationList.push_back(tEntityDetails.strEntityName); //Normal usecase
								}
								else if(!m_pEntityInfo->bIsPosTagAvailable() && m_pEntityInfo->vectorGetEntityID(CMP).size()>1)
								{
									ETG_TRACE_USR4(("bExtractEndStationInfo, INVALID CONDITION"));
								}
								else if(!m_pEntityInfo->bIsPosTagAvailable() && m_pEntityInfo->vectorGetEntityID(CMP).size()==1)
								{
									ETG_TRACE_USR4(("bExtractEndStationInfo, for backward compatibility"));
									vEndStationList.push_back(tEntityDetails.strEntityName); //for backward compatibility where pos tag is not available
								}
							}
							ETG_TRACE_USR4(("End station index : %d, Entity index : %d, Config index : %d, Entity ID : %d", endStationIndex, entityIndex, configIndex, uint64EntityID));
							ETG_TRACE_USR4(("Entity name is %s", tEntityDetails.strEntityName.c_str()));
							//push the end station details in Plugin main endstation Vector.
							m_pPluginMain->setEndStationDetails(tEntityDetails);
							setEntityDetailsToAppl(tEntityDetails);
							ETG_TRACE_USR4(("bExtractEndStationInfo: after setEndStationDetails"));
							vecReadComplete.erase(vecReadComplete.begin());
							ETG_TRACE_USR4(("bExtractEndStationInfo size of vecReadComplete %d", vecReadComplete.size()));
							ETG_TRACE_USR4(("bExtractEndStationInfo: after erasing vecReadComplete"));
							}
							}
                         }
                     }
                  }
              }
           }
       }
	   if(!bMatchFound && !vecReadComplete.empty())
	   {
		   m_pPluginMain->eraseVecReadComplete(vecReadComplete.at(0));
		   vecReadComplete.erase(vecReadComplete.begin());
	   }
	   }
	   }
    } 
	vecReadComplete = m_pPluginMain-> vecGetReadComplete();
   }
	}

	if(NULL != m_pPluginMain){
		
	
	if(m_pPluginMain->isEntityDisconnected())
	{
		vector<stEntityDesc> vecEndStations = m_pPluginMain->vecGetEndStationVector();
		stCamDesc tCamDetails;
		vEndStationList.clear();
		vecStCamDesc.clear();
		for_each(vecEndStations.begin(), vecEndStations.end(),[&tCamDetails, this](stEntityDesc Ent){
			tCamDetails.position = CAMPORT_POSITION_UNAVAILABLE;
			if(strcmp(Ent.strEntityName.c_str(), COACHMEDIA_CAMERAPORT_1) == 0)
			{
				tCamDetails.position = m_pEntityInfo->u16GetCamPos(Ent.u64EntityID);
				tCamDetails.CamEntityDesc = Ent;
				vecStCamDesc.push_back(tCamDetails);
			}
			if(tCamDetails.position == CAMPORT_POSITION_UNAVAILABLE || tCamDetails.position == PRIMARY_CAMPORT)
			{
				vEndStationList.push_back(Ent.strEntityName);
			}
		});
	}
	}

    //EndStationToDataUtility(vEndStationList);
	
	ETG_TRACE_USR4(("End station Read Completed"));
	//notification to all helpers for read_completed.
	if(NULL != m_pPluginMain){
		m_pPluginMain->vNotifyESReadComplete();
	}
	vecReadComplete = m_pPluginMain-> vecGetReadComplete();
	if(!vecReadComplete.empty())
	{
		ETG_TRACE_USR4(("Extract endstation info called after ESReadComplete"));
		bExtractEndStationInfo();
	}
	return true;
}

/************************************************************************
*FUNCTION: 		setEntityDetailsToAppl()
*DESCRIPTION   :Method to  add the newly connected entity to the vector 
*               containing entity details to be sent to HMI.If entity is 
*               already present in vector then connection status is updated
*               to 1 
*PARAMETER:		structure containing details of entity
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	dhs1kor	02/09/2021
*************************************************************************/
void clEndStationThread::setEntityDetailsToAppl(stEntityDesc &tEntityDetails)
{
    uint64_t entity_id = tEntityDetails.u64EntityID; 
	ETG_TRACE_USR4(("clEndStationThread::setEntityDetailsToAppl entity ID: %llx",entity_id));
    pthread_mutex_lock(&m_thMutexLock);
    auto it = std::find_if(vecStEntityDetailsToAppl.begin(), vecStEntityDetailsToAppl.end(),
                       [entity_id] (const stEntityDetailsToAppl& v) { 
                          return v.u64EntityID == entity_id; 
                       });
                       
    if (it != vecStEntityDetailsToAppl.end())
	{                    
        ETG_TRACE_USR4(("clEndStationThread::setEntityDetailsToAppl, entity found in vector"));                   
        it->bConnectionStatus = TRUE;    
    }
	else
	{
	    ETG_TRACE_USR4(("clEndStationThread::setEntityDetailsToAppl, entity not found in vector"));
	    stEntityDetailsToAppl tEntityDetailsToAppl;
	    tEntityDetailsToAppl.strEntityName = tEntityDetails.strEntityName;
	    tEntityDetailsToAppl.u64EntityID = tEntityDetails.u64EntityID; 
	    tEntityDetailsToAppl.bConnectionStatus = TRUE;
	    vecStEntityDetailsToAppl.push_back(tEntityDetailsToAppl);
	    ETG_TRACE_USR4(("setEntityDetailsToAppl size of vecStEntityDetailsToAppl %d", vecStEntityDetailsToAppl.size()));
    }
    pthread_mutex_unlock(&m_thMutexLock);
    EntityDetailsToListDataUtility(vecStEntityDetailsToAppl);

}
/************************************************************************
*FUNCTION: 		EntityDisConnected()
*DESCRIPTION   :Method to remove the disconnected entity from the vector 
*				containing entity detils to be sent to HMI
*PARAMETER:		entity ID which is disconnected
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	dhs1kor	02/09/2021
*************************************************************************/
void clEndStationThread::EntityDisConnected(uint64_t entity_id)
{
    ETG_TRACE_USR4(("clEndStationThread::EntityDisConnected entity ID: %llx",entity_id));
    pthread_mutex_lock(&m_thMutexLock);
    bool bVecChanged = FALSE;
    auto it = std::find_if(vecStEntityDetailsToAppl.begin(), vecStEntityDetailsToAppl.end(),
                       [entity_id] (const stEntityDetailsToAppl& v) { 
                          return v.u64EntityID == entity_id; 
                       });
                       
    if (it != vecStEntityDetailsToAppl.end())
    {                        
        it->bConnectionStatus = FALSE;
        bVecChanged = TRUE;
        pthread_mutex_unlock(&m_thMutexLock);
        ETG_TRACE_USR4(("clEndStationThread::EntityDisConnected, entity found in vector"));
    }
    else
    {
	    pthread_mutex_unlock(&m_thMutexLock);
	    ETG_TRACE_USR4(("clEndStationThread::EntityDisConnected, entity not found in vector")); 
    }
    if (TRUE == bVecChanged)
    {
        EntityDetailsToListDataUtility(vecStEntityDetailsToAppl);
    }
}

/************************************************************************
*FUNCTION: 		EndStationToDataUtility()
*DESCRIPTION   :Method to add the data in the event bus format to send 
				the data by eventbus mechanism
*PARAMETER:		vector<string> 
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	ess7kor	29/05/2020
*************************************************************************/
void clEndStationThread::EndStationToDataUtility(std::vector<std::string> vEndStationList)
{
	//sending the endstation details to HMI
	ETG_TRACE_USR4(("End station list vector size: %d",vEndStationList.size()));
	
	boost::shared_ptr<EventDataUtility> dataUtility(EventDataUtility::newEventDataUtility());

	if(NULL != dataUtility.get())
	{
		for (uint32_t index = 0; index < vEndStationList.size(); index++)
		{
			ETG_TRACE_USR4(("Entity name in vEndStationList is %s", vEndStationList[index].c_str()));
			dataUtility->addEventData(vEndStationList[index]);
		}
	}
	//send the dataUtility to fire event
	RecvMsgFromEndStation(dataUtility);	
}

/************************************************************************
*FUNCTION: 		EntityDetailsToListDataUtility()
*DESCRIPTION   :Method to add the data in the event bus format to send 
*				the listdata by eventbus mechanism
*				Convert vector containing entity details to be send to
*               HMI to EventListData
*PARAMETER:		std::vector<stEntityDetailsToAppl> vStEntityDetails
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	dhs1kor	02/09/2021
*************************************************************************/

void clEndStationThread::EntityDetailsToListDataUtility(std::vector<stEntityDetailsToAppl> vStEntityDetails)
{
	ETG_TRACE_USR4(("EntityDetailsToListDataUtility, Vector size: %d", vStEntityDetails.size()));
    for (auto it : vStEntityDetails)
    {
        ETG_TRACE_USR4(("EntityDetailsToListDataUtility, EntityDetails sent to HMI u64EntityID: %llx, bConnectionStatus: %d, EntityName: %s",
                        it.u64EntityID, it.bConnectionStatus, it.strEntityName.c_str() ));
    }

    boost::shared_ptr<EventListDataUtility> pluginListData(EventListDataUtility::newEventListDataUtility());
    if(NULL != pluginListData.get())
    {	
		for(auto ct : vStEntityDetails)
		{
			EventDataUtility *eventData = EventDataUtility::newEventDataUtility();
			if(NULL != eventData)
			{
			eventData->addEventData(ct.u64EntityID);
			eventData->addEventData(ct.strEntityName);
			eventData->addEventData(ct.bConnectionStatus);
			}
			pluginListData->addEventListData(eventData);
			//delete eventData; //check on target to prevent data leak
		}
		
     //RecvListMsgFromEndstation(pluginData, pluginListData);
    }
    PluginServerHandler* ptrPluginServer = PluginServerHandler::getInstance();
    if (NULL != ptrPluginServer) {
        ptrPluginServer->EntityDetailsListUpdateProperty(pluginListData);
    }
 
}
/************************************************************************
*FUNCTION: 		RecvListMsgFromEndstation()
*DESCRIPTION   :method to fire the event for listdata for the HMI component to access
*Send entitydetails including entity id, entity name and connection status to HMI
*PARAMETER:		shared_ptr<EventDataUtility>, shared_ptr<EventListDataUtility>
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	dhs1kor	02/09/2021
*************************************************************************/
void clEndStationThread::RecvListMsgFromEndstation(boost::shared_ptr<EventDataUtility> dataUtility,
		const boost::shared_ptr<EventListDataUtility> eListData)
{
	 
     ETG_TRACE_USR4(("clEndStationThread::RecvListMsgFromEndstation enter"));
	 CommandToPlugin* _commandFromPlugin = CommandToPlugin::poGetInstance();
	 HandlerRegistration* pluginReg = nullptr;
	 //create object of common plugin
	 PluginEventBase<CommonPlugin> pluginEvent(_commandFromPlugin, CommonPlugin::getInstance(), CTRLBLOCK_ENTITY_DATA, dataUtility,eListData,ENTITY_LIST_DATA);

	 //create object of Plugin Listener
	 PluginEvtListenerBase<CommonPlugin>* pluginEventListener = PluginEvtListenerBase<CommonPlugin>::getInstance();

	 //add handler
     pthread_mutex_lock(&m_thMutexLock);
	 if(pluginEventListener != NULL)
	 {
		 pluginReg = EventBus::AddHandler<PluginEventBase<CommonPlugin>>(pluginEventListener);
	 }

	 //fire the event
	 EventBus::FireEvent(pluginEvent);
	 //remove handler after event fire else handlers keep incrementing everytime and multiple events are fired
	 pluginReg->removeHandler();
     pthread_mutex_unlock(&m_thMutexLock);
	 delete pluginReg;
     ETG_TRACE_USR4(("clEndStationThread::RecvListMsgFromEndstation exit"));
}



/************************************************************************
*FUNCTION: 		EndStationToDataUtility()
*DESCRIPTION   :method to fire the event for the HMI component to access
*PARAMETER:		shared_ptr<EventDataUtility>
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	ess7kor	29/05/2020
*************************************************************************/
void clEndStationThread::RecvMsgFromEndStation(boost::shared_ptr<EventDataUtility> dataUtility)
{
    ETG_TRACE_USR4(("clEndStationThread::RecvMsgFromEndStation enter"));
	CommandToPlugin* _commandFromPlugin = CommandToPlugin::poGetInstance();

	 HandlerRegistration* pluginReg = nullptr;
	 //create object of common plugin
	 PluginEventBase<CommonPlugin> pluginEvent(_commandFromPlugin, CommonPlugin::getInstance(), CTRLBLOCK_STATION_DATA, dataUtility, STATION_LIST_DATA);

	 //create object of Plugin Listener
	 PluginEvtListenerBase<CommonPlugin>* pluginEventListener = PluginEvtListenerBase<CommonPlugin>::getInstance();

	 //add handler
     pthread_mutex_lock(&m_thMutexLock);
	 if(pluginEventListener != NULL)
	 {
		 pluginReg = EventBus::AddHandler<PluginEventBase<CommonPlugin>>(pluginEventListener);
	 }

	 //fire the event
	 EventBus::FireEvent(pluginEvent);
	 //remove handler after event fire else handlers keep incrementing everytime and multiple events are fired
	 pluginReg->removeHandler();
     pthread_mutex_unlock(&m_thMutexLock);

	 delete pluginReg;
     ETG_TRACE_USR4(("clEndStationThread::RecvMsgFromEndStation exit"));
}

/************************************************************************
*FUNCTION: 		vecGetCamDesc()
*DESCRIPTION   :To provide the filled camera description vector
*PARAMETER:		None
*RETURNVALUE: 	vector<stCamDesc>
*HISTORY:
*revision 0.1	hhd1kor	03/12/2021
*************************************************************************/
std::vector<stCamDesc> clEndStationThread::vecGetCamDesc()
{
	return vecStCamDesc;
}
