/* ***************************************************************************************
* FILE:          clSettingsPlugin.cpp
* SW-COMPONENT:  avdecc_appl_plugins
* DESCRIPTION:  clSettingsPlugin.cpp is part of avdecc_appl_plugins library
* COPYRIGHT:  (c) 2020-21 Robert Bosch Car Multimedia GmbH
* HISTORY: 
* AUTHOR:  
* REVISION: 
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
/*****************************************************************
| includes
|----------------------------------------------------------------*/
#include <clSettingsPlugin.h>
#include "plugin_trace.h"
#include "MediaGlass.h"
#include "clControlMapping.h"
#include "clPluginDataProvider.h"
#include "PluginDataType.h"
#include "AudioDefines.h"

#include "PluginEvtListenerBase.h"
#include "PluginEventBase.h"
//#include "XMLRead.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_PLUGIN_SETTINGS
#include "trcGenProj/Header/clSettingsPlugin.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

#define CABINA_SINKID 4
#define SETTING_CMG_SWIVELDOOR_NAME "SwivelDoor"
#define SETTING_CMG_SWIVELDOOR_VALUE_DRIVER_DOOR "Door 1"
#define SETTING_CMG_SWIVELDOOR_VALUE_PASSENGER_DOOR "Door 2"
#define Number_OF_SWIVEL_ASSOCIATED_to_Door 1

#define DEV_NAME_CMG "CMG"
#define GLASS_A_REGION_1 "Glass_A1"
#define GLASS_A_REGION_2 "Glass_A2"
#define GLASS_B_REGION_1 "Glass_B1"
#define GLASS_B_REGION_2 "Glass_B2"
#define OBJ_NAME_CMG "CM-G"
#define MATCH 0

clSettingsPlugin* clSettingsPlugin::poSelf = NULL;
clSettingsPlugin::clSettingsPlugin(){
	setMapFuncs();
	//Communucation Protocol Object creation logic
	m_pCommControl = new clCommunicationProtocol();
	if(m_pCommControl != NULL)
	{
		ETG_TRACE_USR4(("clSettingsPlugin::clCommunicationProtocol is created"));
	}
	m_u16LastExecutedSetCommand = 0;
	m_pEntityInfo = tclAvRoutingParser::pGetInstance();
	m_bParseState = m_pEntityInfo->bParseXml();
	m_CabinAChannelinfo.clear();
        m_CabinBChannelinfo.clear();
	m_CMGEntityDetails.clear();
        m_A1MediaGlassList.clear();
        m_A2MediaGlassList.clear();
        m_CABINAGlasses.clear();
       IntializeAttenuationValuesMap();
       /*if(TRUE ==  m_pEntityInfo->bParseXml())
       {
             VExtractCabinChannelInfo();
       }*/
//       vRegisterPluginCallbacks(); // Commenting out to stop the Cyclic Reset
//       Behaviour . Bug 1384322
}
void clSettingsPlugin::IntializeAttenuationValuesMap()
{
        attenationvalues.clear();
        attenationvalues.insert(pair<int32_t,int32_t>(-9,136));
        attenationvalues.insert(pair<int32_t,int32_t>(-8,221));
        attenationvalues.insert(pair<int32_t,int32_t>(-7,231));
        attenationvalues.insert(pair<int32_t,int32_t>(-6,241));
        attenationvalues.insert(pair<int32_t,int32_t>(-5,246));
        attenationvalues.insert(pair<int32_t,int32_t>(-4,248));
        attenationvalues.insert(pair<int32_t,int32_t>(-3,250));
        attenationvalues.insert(pair<int32_t,int32_t>(-2,252));
        attenationvalues.insert(pair<int32_t,int32_t>(-1,254));
        attenationvalues.insert(pair<int32_t,int32_t>(0,0));
        attenationvalues.insert(pair<int32_t,int32_t>(1,254));
        attenationvalues.insert(pair<int32_t,int32_t>(2,252));
        attenationvalues.insert(pair<int32_t,int32_t>(3,250));
        attenationvalues.insert(pair<int32_t,int32_t>(4,248));
        attenationvalues.insert(pair<int32_t,int32_t>(5,246));
        attenationvalues.insert(pair<int32_t,int32_t>(6,241));
        attenationvalues.insert(pair<int32_t,int32_t>(7,231));
        attenationvalues.insert(pair<int32_t,int32_t>(8,221));
        attenationvalues.insert(pair<int32_t,int32_t>(9,136));

        m_frontAttenuationValue = 0;
        m_rearAttenuationValue = 0;

}

clSettingsPlugin::~clSettingsPlugin(){
	m_CMGEntityDetails.clear();
        m_A1MediaGlassList.clear();
        m_A2MediaGlassList.clear();
        m_CABINAGlasses.clear();
	m_pEntityInfo = nullptr;
	delete m_pCommControl;
}

clSettingsPlugin* clSettingsPlugin::getInstance()
{
	ETG_TRACE_USR4(("clSettingsPlugin::getInstance"));
	if(poSelf == NULL)
	{
		ETG_TRACE_USR4(("Creating object clSettingsPlugin"));
		poSelf = new clSettingsPlugin();
	}
	ETG_TRACE_USR4(("Return object clSettingsPlugin %d", poSelf));
	return poSelf;
	
}
void clSettingsPlugin::deleteInstance()
{
	if(NULL != poSelf)
	{
		delete poSelf;
	}
}

void clSettingsPlugin::vEventESReadComplete()
{
	//extract the details of the entities required
	clPluginDataProvider *ptrMain = clPluginDataProvider::getInstance();
	if(NULL != ptrMain)
	{
		std::vector<stEntityDesc> stEntityDetails;
		stEntityDesc stGlassDetails;
		bool found = ptrMain->getEndStationDetails(CMG, stEntityDetails);
		if(true == found)
		{
			ETG_TRACE_USR4(("clSettingsPlugin::vEventESReadComplete::Copying Entity details"));
			vExtractESData(stEntityDetails);
            
            if(NULL != m_pCommControl)
            {
                m_pCommControl->RegisterUnSolicitedResponse(m_CMGEntityDetails.at(0).end_station);
            }
			
		}
		stEntityDetails.clear();
		found = ptrMain->getEndStationDetails(CMA, stEntityDetails);
		if (found)
		{
			ETG_TRACE_USR4(("clSettingsPlugin::vEventESReadComplete::Copying CMA Entity details"));
			updateCMAEntityDetails(stEntityDetails);
			VExtractCabinChannelInfo();
		}
		else 
		{
			ETG_TRACE_USR4(("NO Amplifier End station details "));
		}		
		//VExtractCabinChannelInfo();
	}
	//VExtractCabinChannelInfo();
}

void clSettingsPlugin::updateCMAEntityDetails(std::vector<stEntityDesc> stEntityDetails)	
{		
	ETG_TRACE_USR4(("clSettingsPlugin::updateCMAEntityDetails::Copying Entity details"));	
	m_CMAEntityDetails.clear();
	m_CMAEntityDetails = stEntityDetails;
		
	ETG_TRACE_USR4(("vEventESReadComplete m_CMAEntityDetails size is %d", m_CMAEntityDetails.size()));
}

void clSettingsPlugin::vRegisterPluginCallbacks()
{
	ETG_TRACE_USR4(("clSettingsPlugin::vRegisterPluginCallbacks"));
	//register for callbacks
	clControlMapping *p_controlmap = clControlMapping::getInstance();
	if(NULL != p_controlmap)
	{
		p_controlmap->setRespCallBack(CMG, SWIVEL_UNIT_CONTROL, this, (FUNCPTR)(&clSettingsPlugin::vHandleSwivelUpDownStatus));
		p_controlmap->setRespCallBack(CMG, LCD_PANEL_ON_OFF, this, (FUNCPTR)(&clSettingsPlugin::vHandleLCDOnOffStatus));
	}

}
/*! ***********************************************************************
 *FUNCTION: 		VExtractCabinChannelInfo
 *@details: 		Extracting the channel info which are mapped to Front and Rear
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		NA
 *@return: 		void
 *@details
 *HISTORY:
 *08.02.2020		0.1		Sathiya		Initial Version
 ************************************************************************/
void clSettingsPlugin::VExtractCabinChannelInfo()
{
	ETG_TRACE_USR4(("clSettingsPlugin::VExtractCabinChannelInfo"));
	tCabinChannelMap channeldetails = m_pEntityInfo->getCabinChannelMap();
        
	//std::map<std::string, std::string>::iterator itr;
	std::vector<std::pair<std::string, std::string> >::iterator itr;
	for(itr = channeldetails.begin();itr != channeldetails.end();itr++)
	{
		ETG_TRACE_USR4(("clSettingsPlugin::VExtractCabinChannelInfo %s",(itr->first).c_str()));
		ETG_TRACE_USR4(("clSettingsPlugin::VExtractCabinChannelInfo %s",(itr->second).c_str()));
		if(MATCH == strcmp((itr->first).c_str(),CABINA_GROUP))
		{
			if(MATCH == strcmp((itr->second).c_str(),FRONT))
			{
				ETG_TRACE_USR4(("clSettingsPlugin::VExtractCabinChannelInfo MAtched"));
				m_CabinAChannelinfo.push_back(FRONT_CHANNEL);
			}
			else if(MATCH == strcmp((itr->second).c_str(),REAR))
			{
				m_CabinAChannelinfo.push_back(REAR_CHANNEL);
			}
		}
		if(MATCH == strcmp((itr->first).c_str(),CABINB_GROUP))
		{
			if(MATCH == strcmp((itr->second).c_str(),FRONT))
			{
				m_CabinBChannelinfo.push_back(FRONT_CHANNEL);
			}
			else if(MATCH == strcmp((itr->second).c_str(),REAR))
			{
				m_CabinBChannelinfo.push_back(REAR_CHANNEL);
			}
		}
	}
	std::vector<int>::iterator it;
	for(it = m_CabinAChannelinfo.begin();it != m_CabinAChannelinfo.end(); it++)
	{
		ETG_TRACE_USR4(("Settings Plugin Channel info is %d", *it));
	}
        
}

void clSettingsPlugin::vExtractESData(std::vector<stEntityDesc> stEntityDetails)
{
	ETG_TRACE_USR4(("clSettingsPlugin::vExtractESData stEntityDetails size is %d", stEntityDetails.size()));
	m_CMGEntityDetails.clear();
	m_A1MediaGlassList.clear();
	m_A2MediaGlassList.clear();
	m_CABINAGlasses.clear();
	m_CMGEntityDetails = stEntityDetails;
	for(int i = 0; i < m_CMGEntityDetails.size();i++)
	{
		uint64_t u64EntityID = stEntityDetails.at(i).u64EntityID;		
		std::string ipaddress = m_pEntityInfo->sGetEntityIP(u64EntityID);
		std::string entLinkName = m_pEntityInfo->sGetEntitySinkLinkName(u64EntityID);
		
		if (!(entLinkName.empty()))
		{
			uint32_t entityIndex;
			bool isEntityGlass = false;
		
			if ( (strcmp(entLinkName.c_str(), CABINA_GROUP_GLASS) == 0))
			{
					m_A1MediaGlassList.push_back(m_CMGEntityDetails.at(i));
					isEntityGlass = true;
					ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation(), GlassA or GlassA1 added, entity name , %s", entLinkName.c_str()));
					ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation(), GlassA or GlassA1 vector size: %d ",m_A1MediaGlassList.size()));
					ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation: Storing CabinA ipaddress %s", ipaddress.c_str()));
					m_CABINAGlasses.push_back(ipaddress);
			}
			else if ( (strcmp(entLinkName.c_str(), CABINA2_GROUP_GLASS) == 0) )
			{
					m_A2MediaGlassList.push_back(m_CMGEntityDetails.at(i));
					isEntityGlass = true;
					ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation(), GlassA2 added, entity name , %s", entLinkName.c_str()));
					ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation(), GlassA2 vector size: %d ",m_A2MediaGlassList.size()));
					ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation: Storing CabinA2 ipaddress %s", ipaddress.c_str()));
					m_CABINAGlasses.push_back(ipaddress);
			}
		/*	else if ( (strcmp(entName.c_str(), COACHMEDIA_GLASS_B) == 0) || (strcmp(entName.c_str(), COACHMEDIA_GLASS_B1) == 0) )
			{
				if (!bFindEntityInList(u64EntityID, m_B1MediaGlassList, entityIndex))
				{
					m_B1MediaGlassList.push_back(m_CMGEntityDetails.at(i));
					isEntityGlass = true;
					ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation(), GlassB or GlassB1 added, entity name , %s", entName.c_str()));
					ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation(), GlassB or GlassB1 vector size: %d ",m_B1MediaGlassList.size()));
				}
			}
			else if ( (strcmp(entName.c_str(), COACHMEDIA_GLASS_B2) == 0) )
			{
				if (!bFindEntityInList(u64EntityID, m_B2MediaGlassList, entityIndex))
				{
					m_B2MediaGlassList.push_back(m_CMGEntityDetails.at(i));
					isEntityGlass = true;
					ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation(), GlassB2 added, entity name , %s", entName.c_str()));
					ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation(), GlassB2 vector size: %d ",m_B2MediaGlassList.size()));
				}
			}*/
			else
			{
				ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation(), else part "));
			}
		
			if ( isEntityGlass )
			{
                // By default, it is not a swivel unit
                m_IsEntitySwivel[u64EntityID] = false;
                m_IsEntityPassengerDoorSwivel[u64EntityID] = false;
                m_IsEntityDriverDoorSwivel[u64EntityID] = false;

				// Check if swivel is associated with door
                tSettingNameValueMap mapMediaGlassSetting = m_pEntityInfo->mapGetMediaGlassSetting(u64EntityID);
                if (!(mapMediaGlassSetting.empty()))
                {
                    const char* swivelDoorSettingValue = mapMediaGlassSetting[SETTING_CMG_SWIVELDOOR_NAME].c_str();
    
                    if ( NULL != swivelDoorSettingValue )
                    {
                        ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation: swivelDoor setting : %s", swivelDoorSettingValue));
    
                        // Entity is a swivel unit
                        m_IsEntitySwivel[u64EntityID] = true;

                        if (
                            ( strncmp(swivelDoorSettingValue, SETTING_CMG_SWIVELDOOR_VALUE_PASSENGER_DOOR, strlen(SETTING_CMG_SWIVELDOOR_VALUE_PASSENGER_DOOR)) == 0 )
                            )
                        {
                            ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation: Swivel associated to passenger door"));
                            m_IsEntityPassengerDoorSwivel[u64EntityID] = true;
                        }
                        else if (
                            ( strncmp(swivelDoorSettingValue, SETTING_CMG_SWIVELDOOR_VALUE_DRIVER_DOOR, strlen(SETTING_CMG_SWIVELDOOR_VALUE_DRIVER_DOOR)) == 0 )
                            )
                        {
                            ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation: Swivel associated to driver door"));
                            m_IsEntityDriverDoorSwivel[u64EntityID] = true;
                        }
                        else
                        {
                            ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation: Swivel not associated to any door"));
                        }
                    }
                    else
                    {
                        ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation: SwivelDoor setting value is NULL"));
                    }
                }
                else
                {
                    ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation: SwivelDoor setting not available"));
                }
    
					ETG_TRACE_USR4(("clSettingsPlugin::vAddDeleteEndStation(),added, entity name , %s", entLinkName.c_str()));
                ETG_TRACE_USR4(("clSettingsPlugin:;ADDDeleteEndStation,value of m_IsEntitySwivel[u64EntityID] %d", m_IsEntitySwivel[u64EntityID]));
		
		
				// Request for unsolicited respose
				// vRegisterUnSolicitedResponse(clientIf, end_station);
		
			}
		}
	}
}

/*****************************************************************************************************
*NAME        : setMapFuncs
*DESCRIPTION : add fucntion pointers map
******************************************************************************************************/
void clSettingsPlugin::setMapFuncs()
{
	ETG_TRACE_USR4(("clSettingsPlugin::setMapFuncs"));
	m_MemberFunctionMap[CTRLBLOCK_NAME_SETTINGS_SWIVELUPDOWN] = &clSettingsPlugin::swivelUpDown;
    m_MemberFunctionMap[CTRLBLOCK_NAME_SETTINGS_GLASSLCD] = &clSettingsPlugin::lcdOnOff;
    m_MemberFunctionMap[CTRLBLOCK_NAME_SETTINGS_CABINAGLASSES] = &clSettingsPlugin::CabinAGlasses;
    m_MemberFunctionMap[CTRLBLOCK_NAME_SETTINGS_CABINBGLASSES] = &clSettingsPlugin::CabinBGlasses;
    m_MemberFunctionMap[CTRLBLOCK_NAME_SETTINGS_SWIVELCALIBRATION] = &clSettingsPlugin::SwivelCalibration;
	m_MemberFunctionMap[CTRLBLOCK_NAME_SETTINGS_FADERSETTINGS] = &clSettingsPlugin::FaderSetting;
}

/******************************************************************************
*NAME          : addPluginEventHandler
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get plugin event handler, part of evenbus mechanism
******************************************************************************/
std::unique_ptr<HandlerRegistration> clSettingsPlugin::addPluginEventHandler()
{
    ETG_TRACE_USR4(("clSettingsPlugin::addPluginEventHandler"));
    PluginEvtListenerBase<clSettingsPlugin>* cameraPlugEventListener = PluginEvtListenerBase<clSettingsPlugin>::getInstance();
    std::unique_ptr<HandlerRegistration> handle (EventBus::AddHandler<PluginEventBase<clSettingsPlugin>>(cameraPlugEventListener));
    return std::move(handle);
}

/******************************************************************************
*NAME          : setPluginData
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Based on the key, the corresponding function is invoked
******************************************************************************/
void clSettingsPlugin::setPluginData(std::string key, const boost::shared_ptr<EventDataUtility>& eValue)
{
    ETG_TRACE_USR4(("clSettingsPlugin::setPluginData map size %d ", m_MemberFunctionMap.size()));
	ETG_TRACE_USR4(("clSettingsPlugin::setPluginData string key %s", key.c_str()));
	
	void(clSettingsPlugin::*FnPtr)(const boost::shared_ptr<EventDataUtility>&) = m_MemberFunctionMap.at(key);
    if(NULL != FnPtr)
    {
        (*this.*FnPtr)(eValue);
    }	
}

/******************************************************************************
*NAME          : setPluginListData
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Based on the key, the corresponding function is invoked
******************************************************************************/
void clSettingsPlugin::setPluginListData(std::string key, const boost::shared_ptr<EventDataUtility>& eData,
        const boost::shared_ptr<EventListDataUtility>& eListData){
   ETG_TRACE_USR4(("clSettingsPlugin::setPluginListData"));
    //To Do, based on whether camera_control_plugin needs list data or not
}

/******************************************************************************185
*NAME          : addPluginEventHandler
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get plugin event, part of evenbus mechanism
******************************************************************************/
std::unique_ptr<Event> clSettingsPlugin::getPluginEvent(Object* sender, const std::string msg, boost::shared_ptr<EventDataUtility>evalue)
{
    ETG_TRACE_USR4(("clSettingsPlugin::getPluginEvent"));
    std::unique_ptr<PluginEventBase<clSettingsPlugin>> cameraPluginEvent(new PluginEventBase<clSettingsPlugin>(sender,this,msg,evalue));
    return std::move(cameraPluginEvent);
}


/******************************************************************************
*NAME          : addPluginEventHandler
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get plugin list event, part of evenbus mechanism
******************************************************************************/
std::unique_ptr<Event> clSettingsPlugin::getPluginListEvent(Object* sender, const std::string msg, boost::shared_ptr<EventDataUtility>eData,
                                                                const boost::shared_ptr<EventListDataUtility>& eListData)
{
    ETG_TRACE_USR4(("clSettingsPlugin::getPluginListEvent"));
    std::unique_ptr<PluginEventBase<clSettingsPlugin>> cameraPluginEvent(new PluginEventBase<clSettingsPlugin>(sender,this,msg,eData,eListData));
    return std::move(cameraPluginEvent);
}

/******************************************************************************
*NAME          : vCreateSetControlCommand
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Send the SetControl command to the Pluginbase by Communication control
******************************************************************************/
void clSettingsPlugin::vCreateSetControlCommand(uint16_t u16ControlId,uint16_t u16ControlVal)
{
	ETG_TRACE_USR4(("clSettingsPlugin::vCreateSetControlCommand  u16ControlId=%d  u16ControlVal=%d",u16ControlId,u16ControlVal));
	
	// m_u16LastExecutedSetCommand = u16ControlId;//Set value received from HMI
	
	// ControlDataElement controlElement;
	// controlElement.configuration = m_CMGEntityDetails.configuration;
	// controlElement.u32ControlDescIdx = u16ControlId;
	// controlElement.uint8Value = u16ControlVal;
		
	// if(m_pCommControl != NULL)
	// {
		// ETG_TRACE_USR4(("clSettingsPlugin SetControl"));
		// m_pCommControl->setControl(controlElement);
	// }
	// else
	// {
		// ETG_TRACE_USR4(("clSettingsPlugin::Control Pointer is NULL"));
	// }
}

/************************************************************************
*NAME        : vSetControlUINT8
*DESCRIPTION : Design section 8.4.1
************************************************************************/
//void clSettingsPlugin::vSetControlUINT8(const char *cacEntityName, uint32_t u32EndStationIdx, uint32_t u32EntityDescIdx, uint32_t u32ConfigDescIdx, uint32_t u32ControlDescIdx, uint8_t uint8Value)
void clSettingsPlugin::vSetControlUINT8(uint32_t u32ControlDescIdx, uint64_t entityid, uint8_t uint8Value)
{
    ETG_TRACE_USR4(("clSettingsPlugin::vSetControlUINT8 u16ControlId=%d  u16ControlVal=%d  entityid=%d",u32ControlDescIdx,uint8Value,entityid));
	//ETG_TRACE_USR4(("clSettingsPlugin::vSetControlUINT8 m_CMGEntityDetails Size %d", m_CMGEntityDetails.size()));
	//if( m_CMGEntityDetails.size() > 0 )
	//{
		//ETG_TRACE_USR4(("clSettingsPlugin::m_CMGEntityDetails size  > 0 "));
                ControlDataElement controlElement;	
		for (uint8_t index = 0 ;index <  m_CMGEntityDetails.size() ; index++)
		{
                    if(entityid == m_CMGEntityDetails.at(index).u64EntityID)
                    {
                        ETG_TRACE_USR4(("clSettingsPlugin::vSetControlUINT8 Entityid found"));
                        if(m_CMGEntityDetails.at(index).configuration)
                        {
                        controlElement.configuration = m_CMGEntityDetails.at(index).configuration;
                                 
			controlElement.u32ControlDescIdx = u32ControlDescIdx;
			controlElement.uint8Value = uint8Value;
				
			if(m_pCommControl != NULL)
			{
				ETG_TRACE_USR4(("clSettingsPlugin SetControl"));
				m_pCommControl->setControl(controlElement);
			}
			else
			{
				ETG_TRACE_USR4(("clSettingsPlugin::Control Pointer is NULL"));
			}
                        }
                        else
                        {
                               ETG_TRACE_USR4(("clSettingsPlugin Entity Configuration is NULL"));
                        }
                        break;
                    }
                }
		//}
	//}
	//else
	//{
		//ETG_TRACE_USR4(("clSettingsPlugin::m_CMGEntityDetails is empty"));
	//}
}




/************************************************************************
*NAME        : sendCommandToSwivel
*SWFLs       : 2839
************************************************************************/
void clSettingsPlugin::sendCommandToSwivel(uint8_t sinkId, uint8_t swivelSelected, uint8_t swivelCommand, uint8_t lcdOnOffCommand)
{
    ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel(), Sink ID: %d, swivelSelected: %d, swivelCommand: %d, lcdOnOffCommand: %d" , sinkId, swivelSelected, swivelCommand, lcdOnOffCommand));

    std::vector<stEntityDesc>* mediaGlassList = NULL;
    char* entityName = NULL;

    // Choose the CMG based on sink
    selectEntityBasedOnSink(sinkId, mediaGlassList, entityName);

    if ( ( NULL != mediaGlassList ) && ( NULL != entityName ) )
    {
        ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel(), Entity name : %s", entityName));
        
        for (std::vector<stEntityDesc>::iterator iter = mediaGlassList->begin(); iter != mediaGlassList->end(); iter++)
        {
            bool isSwivelUnit = false;
            bool isPassengerDoorSwivel = false;
            bool isDriverDoorSwivel = false;

            // Check if swivel is available for this entity
            if ( m_IsEntitySwivel.count((*iter).u64EntityID) == Number_OF_SWIVEL_ASSOCIATED_to_Door )
            {
                isSwivelUnit = m_IsEntitySwivel[(*iter).u64EntityID];
                if ( isSwivelUnit )
                {
                    ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel: entity %d is a swivel unit", (*iter).u64EntityID));
                }
                else
                {
                    ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel: entity %d is not a swivel unit", (*iter).u64EntityID));
                }
            }
            else
            {
                ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel: entity %d is not a swivel unit", (*iter).u64EntityID));
            }

            if ( m_IsEntityPassengerDoorSwivel.count((*iter).u64EntityID) == Number_OF_SWIVEL_ASSOCIATED_to_Door )
            {
                isPassengerDoorSwivel = m_IsEntityPassengerDoorSwivel[(*iter).u64EntityID];
                if ( isPassengerDoorSwivel )
                {
                    ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel: entity %d is a passenger door swivel", (*iter).u64EntityID));
                }
                else
                {
                    ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel: entity %d is not a passenger door swivel", (*iter).u64EntityID));
                }
            }
            else
            {
                ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel: entity %d is not a passenger door swivel", (*iter).u64EntityID));
            }

            if ( m_IsEntityDriverDoorSwivel.count((*iter).u64EntityID) == Number_OF_SWIVEL_ASSOCIATED_to_Door )
            {
                isDriverDoorSwivel = m_IsEntityDriverDoorSwivel[(*iter).u64EntityID];
                if ( isDriverDoorSwivel )
                {
                    ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel: entity %d is a driver door swivel", (*iter).u64EntityID));
                }
                else
                {
                    ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel: entity %d is not a driver door swivel", (*iter).u64EntityID));
                }
            }
            else
            {
                ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel: entity %d is not a driver door swivel", (*iter).u64EntityID));
            }

            if (
                (
                    ( ( swivelSelected == SWIVEL_SELECTED_ALL_EXCEPT_DOOR ) && ( !isPassengerDoorSwivel && !isDriverDoorSwivel ) ) || // All swivels except those associated to door or on path way
                    ( ( swivelSelected == SWIVEL_SELECTED_ONLY_PASSENGER_DOOR ) && ( isPassengerDoorSwivel ) ) ||  // Only swivels associated to passenger door or on path way
                    ( ( swivelSelected == SWIVEL_SELECTED_ALL_EXCEPT_PASSENGER_DOOR ) && ( !isPassengerDoorSwivel ) ) || // All swivels except those associated to passenger door or on path way
                    ( ( swivelSelected == SWIVEL_SELECTED_ONLY_DRIVER_DOOR ) && ( isDriverDoorSwivel ) ) ||  // Only swivels associated to driver door or on path way
                    ( ( swivelSelected == SWIVEL_SELECTED_ALL_EXCEPT_DRIVER_DOOR ) && ( !isDriverDoorSwivel ) ) || // All swivels except those associated to driver door or on path way
                    ( swivelSelected == SWIVEL_SELECTED_ALL )// All swivels
                )
               )
            {
                if (
                    ( isSwivelUnit ) && // Send only to swivels and not to all glasses
                    (
                        ( COMMAND_SWIVEL_DOWN == swivelCommand ) ||
                        ( COMMAND_SWIVEL_UP == swivelCommand )
                    )
                   )
                {
                    // Send swivel command
                   // ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel %d, %d %d %d %d", (*iter).u32EndStationIdx, (*iter).u32EntityIndex, (*iter).u32ConfigIdx, SWIVEL_UNIT_CONTROL, swivelCommand));
                    ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel %d, %d", SWIVEL_UNIT_CONTROL, swivelCommand));
					vSetControlUINT8(SWIVEL_UNIT_CONTROL,(*iter).u64EntityID,swivelCommand);
                    m_SwivelStatus[(*iter).u64EntityID] = swivelCommand;
                }

                if ( ( COMMAND_LCD_ON == lcdOnOffCommand ) || ( COMMAND_LCD_OFF == lcdOnOffCommand ) )
                {
                    // Send LCD on/off command
                    // ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel %d, %d %d %d %d", (*iter).u32EndStationIdx, (*iter).u32EntityIndex, (*iter).u32ConfigIdx, LCD_PANEL_ON_OFF, lcdOnOffCommand));
                    // vSetControlUINT8(entityName, (*iter).u32EndStationIdx, (*iter).u32EntityIndex, (*iter).u32ConfigIdx, LCD_PANEL_ON_OFF, lcdOnOffCommand);
					ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel %d, %d", LCD_PANEL_ON_OFF, lcdOnOffCommand));
					vSetControlUINT8(LCD_PANEL_ON_OFF,(*iter).u64EntityID,lcdOnOffCommand);
                }
            }
        
        }
    }
    else 
    {
        ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel(), no glasses associated with sink"));
    } 
}

void clSettingsPlugin::lcdOnOff(const boost::shared_ptr<EventDataUtility>& eValue)
{
	ETG_TRACE_USR4(("clSettingsPlugin::lcdOnOff()"));

    uint8_t selectionValues[2] = {0xFF, 0xFF};

    if ( NULL != eValue.get() )
    {
        uint8_t expectedNumberOfValues = sizeof(selectionValues)/ sizeof(selectionValues[0]);
        const std::vector<EventDataItem*>& dataItems = eValue->getData();

        if ( dataItems.size() == expectedNumberOfValues )
        {
            for ( uint8_t index = 0; index < expectedNumberOfValues; index++ )
            {
                EventDataItem* eventDataItem =  dataItems.at(index);

                if ( NULL != eventDataItem )
                {
                    const EventDataItem::Data& eventData = eventDataItem->getData();

                    if ( EventDataItem::UINT8 == eventData._type )
                    {
                        selectionValues[index] = eventData._value._uint8Value;
                        ETG_TRACE_USR4(("clSettingsPlugin::lcdOnOff(), Value extracted is : %d", selectionValues[index]));
                    }
                }
            }

            sendCommandToLCD(selectionValues[0], selectionValues[1]);
        }
        else
        {
            ETG_TRACE_USR4(("clSettingsPlugin::lcdOnOff(), received wrong number of values from application, expected = %d, received = %d", expectedNumberOfValues, dataItems.size()));
        }
    }
}
/************************************************************************
*NAME        : swivelUpDown
*SWFLs       : 2839
************************************************************************/
void clSettingsPlugin::swivelUpDown(const boost::shared_ptr<EventDataUtility>& eValue)
{
	 ETG_TRACE_USR4(("clSettingsPlugin::swivelUpDown()"));

    uint8_t selectionValues[4] = {0xFF, 0xFF, 0xFF, 0xFF};

    if ( NULL != eValue.get() )
    {
        uint8_t expectedNumberOfValues = sizeof(selectionValues)/ sizeof(selectionValues[0]);
        const std::vector<EventDataItem*>& dataItems = eValue->getData();

        if ( dataItems.size() == expectedNumberOfValues )
        {
            for ( uint8_t index = 0; index < expectedNumberOfValues; index++ )
            {
                EventDataItem* eventDataItem =  dataItems.at(index);

                if ( NULL != eventDataItem )
                {
                    const EventDataItem::Data& eventData = eventDataItem->getData();

                    if ( EventDataItem::UINT8 == eventData._type )
                    {
                        selectionValues[index] = eventData._value._uint8Value;
                        ETG_TRACE_USR4(("clSettingsPlugin::swivelUpDown(), Value extracted is : %d", selectionValues[index]));
                    }
                }
            }

            sendCommandToSwivel(selectionValues[0], selectionValues[1], selectionValues[2], selectionValues[3]);
        }
        else
        {
            ETG_TRACE_USR4(("clSettingsPlugin::swivelUpDown(), received wrong number of values from application, expected = %d, received = %d", expectedNumberOfValues, dataItems.size()));
        }
    }
//VExtractCabinChannelInfo();	
}

/************************************************************************
*NAME        : sendCommandToLCD
*SWFLs       : 2839
************************************************************************/
void clSettingsPlugin::sendCommandToLCD(uint8_t sinkId, uint8_t lcdOnOffCommand)
{
    ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToLCD() sinkId : %d" , sinkId));
    ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToLCD() LCD command : %d" , lcdOnOffCommand));

    if ( ( COMMAND_LCD_ON == lcdOnOffCommand ) || ( COMMAND_LCD_OFF == lcdOnOffCommand ) )
    {
        std::vector<stEntityDesc>* mediaGlassList = NULL;
        char* entityName = NULL;

        // Choose the CMG based on sink
        selectEntityBasedOnSink(sinkId, mediaGlassList, entityName);

        if ( ( NULL != mediaGlassList ) && ( NULL != entityName ) )
        {
            ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToLCD(), Entity name : %s", entityName));
           
            for (std::vector<stEntityDesc>::iterator iter = mediaGlassList->begin(); iter != mediaGlassList->end(); iter++)
            {
                bool isSwivelUnit = false;
                uint8_t swivelStatus = COMMAND_SWIVEL_UP;

                // Check if swivel is available for this entity
                if ( m_IsEntitySwivel.count((*iter).u64EntityID) == 1 )
                {
                    isSwivelUnit = m_IsEntitySwivel[(*iter).u64EntityID];
                    if ( isSwivelUnit )
                    {
                        ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToLCD: entity %d is a swivel unit", (*iter).u64EntityID));
                        if ( m_SwivelStatus.count((*iter).u64EntityID) == 1 )
                        {
                            swivelStatus = m_SwivelStatus[(*iter).u64EntityID];
                            ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToLCD: swivel status: %d", swivelStatus));
                        }
                    }
                    else
                    {
                        ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToLCD: entity %d is not a swivel unit", (*iter).u64EntityID));
                    }
                }
                else
                {
                    isSwivelUnit = false;
                    ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToLCD: entity %d is not a swivel unit", (*iter).u64EntityID));
                }

                if (
                    ( !isSwivelUnit ) || // Send to all fixed holders
                    ( isSwivelUnit && ( swivelStatus != COMMAND_SWIVEL_UP ) ) // Send to swivel holders when it is not in up position
                   )
                {
                    // ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToLCD()  LCD_PANEL_ON_OFF command: %d %d", LCD_PANEL_ON_OFF, lcdOnOffCommand));
                    // vSetControlUINT8(entityName, (*iter).u32EndStationIdx, (*iter).u32EntityIndex, (*iter).u32ConfigIdx, LCD_PANEL_ON_OFF, lcdOnOffCommand);
					ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToSwivel %d, %d", LCD_PANEL_ON_OFF, lcdOnOffCommand));
					vSetControlUINT8(LCD_PANEL_ON_OFF,(*iter).u64EntityID,lcdOnOffCommand);
                }
               
            }
        }
        else
        {
            ETG_TRACE_USR4(("clSettingsPlugin::sendCommandToLCD(), no glasses associated with sink"));
        }
    }
}

/************************************************************************
*NAME        : selectEntityBasedOnSink
*SWFLs       : 2839
************************************************************************/
void clSettingsPlugin::selectEntityBasedOnSink(uint8_t sinkId, std::vector<stEntityDesc>* &mediaGlassList, char* &entityName)
{
    ETG_TRACE_USR4(("clSettingsPlugin::selectEntityBasedOnSink(), Sink ID : %d" , sinkId));

    switch ( sinkId )
    {
        case CABIN_A_SINK_ID:
        case CABIN_A_MEDIAGLASS1_SINK_ID:
        {
            mediaGlassList = &m_A1MediaGlassList;
            entityName = COACHMEDIA_GLASS_A;   // to be changed later to A1
            ETG_TRACE_USR4(("clSettingsPlugin::selectEntityBasedOnSink() A1 mediaGlassList: %d %s", mediaGlassList, entityName));
            break;
        }

        case CABIN_A_MEDIAGLASS2_SINK_ID:
        {
            mediaGlassList = &m_A2MediaGlassList;
            entityName = COACHMEDIA_GLASS_A;  // to be changed later to A2
            ETG_TRACE_USR4(("clSettingsPlugin::selectEntityBasedOnSink() A2 mediaGlassList: %d %s", mediaGlassList, entityName));
            break;
        }

        case CABIN_B_SINK_ID:
        case CABIN_B_MEDIAGLASS1_SINK_ID:
        {
            mediaGlassList = &m_B1MediaGlassList;
            entityName = COACHMEDIA_GLASS_B;   // to be changed later to B1
            ETG_TRACE_USR4(("clSettingsPlugin::selectEntityBasedOnSink() B1 mediaGlassList: %d %s", mediaGlassList, entityName));
            break;
        }

        case CABIN_B_MEDIAGLASS2_SINK_ID:
        {
            mediaGlassList = &m_B2MediaGlassList;
            entityName = COACHMEDIA_GLASS_B;  // to be changed later to B2
            ETG_TRACE_USR4(("clSettingsPlugin::selectEntityBasedOnSink() B2 mediaGlassList: %d %s", mediaGlassList, entityName));
            break;
        }

        default:
        {
            mediaGlassList = NULL;
            entityName = NULL;
            ETG_TRACE_USR4(("clSettingsPlugin::selectEntityBasedOnSink() Sink ID : %d not handled", sinkId));
            break;
        }
    }
}	
/************************************************************************
*NAME        : CabinAGlasses
*SWFLs       : 
************************************************************************/
void clSettingsPlugin::CabinAGlasses(const boost::shared_ptr<EventDataUtility>& eValue)
{
    ETG_TRACE_USR4(("clSettingsPlugin:: CabinAGlasses"));
	UpdateIpAddressToHmi(CABINA_SINKID,m_CABINAGlasses); 
}

/************************************************************************
*NAME        : CabinBGlasses
*SWFLs       : 
************************************************************************/
void clSettingsPlugin::CabinBGlasses(const boost::shared_ptr<EventDataUtility>& eValue)
{
}


/************************************************************************
*NAME        : UpdateIpAddressToHmi
*SWFLs       : 
************************************************************************/
void clSettingsPlugin:: UpdateIpAddressToHmi(uint8_t sinkId,std::vector<std::string> ipaddress)
{
	ETG_TRACE_USR4(("==== clSettingsPlugin UpdateIpAddressToHmi  =SINKID=%d",sinkId));
	boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
	if(NULL != pluginData.get())
	{
		pluginData->addEventData(sinkId);
                pluginData->addEventData(m_CABINAGlasses.size());
                for(int i = 0; i<ipaddress.size();i++)
                {
                   ETG_TRACE_USR4(("==== UpdateIpAddressToHmi Added Ipaddress=%s",ipaddress[i].c_str()));
	               pluginData->addEventData(ipaddress[i].c_str());
                }
	}
	RecvMsgFromPlugin(PLUGIN_NAME_SETTINGS,CTRLBLOCK_NAME_SETTINGS_CABINAGLASSES,pluginData);
}

/************************************************************************
*NAME        : SwivelCalibration
*SWFLs       : 
************************************************************************/
void clSettingsPlugin::SwivelCalibration(const boost::shared_ptr<EventDataUtility>& eValue)
{
ETG_TRACE_USR4(("clSettingsPlugin::SwivelCalibration() Entered"));
uint8_t Sink_id = 0;
std::string recievedip = "";
const std::vector<EventDataItem*>& dataItems = eValue->getData();
if ( dataItems.size() == 2 )
{
 for ( uint8_t index = 0; index < 2; index++ )
 {
   EventDataItem* eventDataItem =  dataItems.at(index);
   if ( NULL != eventDataItem )
   {
	const EventDataItem::Data& eventData = eventDataItem->getData();
	if ( EventDataItem::UINT8 == eventData._type )
        {
             Sink_id = eventData._value._uint8Value;
             ETG_TRACE_USR4(("clSettingsPlugin::SwivelCalibration(), Sinkid Value extracted is : %d", Sink_id));
        }
	if ( EventDataItem::STRING == eventData._type )
        {
            recievedip = (eventData._value._stringValue[0]);
            ETG_TRACE_USR4(("clSettingsPlugin::SwivelCalibration(), Sinkid Value extracted is : %s", recievedip));
        }
    }
 }
 
 
 for(int i = 0; i < m_CMGEntityDetails.size();i++)
	{
		uint64_t u64EntityID = m_CMGEntityDetails.at(i).u64EntityID;		
		std::string ipaddress = m_pEntityInfo->sGetEntityIP(u64EntityID);
		
		ETG_TRACE_USR4(("clSettingsPlugin::SwivelCalibration(), received ip %s", recievedip.c_str()));
		ETG_TRACE_USR4(("clSettingsPlugin::SwivelCalibration(), entity   ip %s", ipaddress.c_str()));
		
		
		if(strcmp(recievedip.c_str(),ipaddress.c_str()) == 0)
		{
			ETG_TRACE_USR4(("clSettingsPlugin::SwivelCalibration(),Sending Calibration Command for the Glass ip %s", ipaddress.c_str()));
	        vSetControlUINT8(SWIVEL_UNIT_CONTROL,u64EntityID,SWIVEL_CALIBRATION);
		}
	}
		
 // std::vector<stEntityDesc>* mediaGlassList = NULL;
 // char* entityName = NULL;

 // // Choose the CMG based on sink
 // selectEntityBasedOnSink(4, mediaGlassList, entityName);

 // if ( ( NULL != mediaGlassList ) && ( NULL != entityName ) )
 // {
   // for ( std::vector<stEntityDesc>::iterator iter = mediaGlassList->begin(); iter != mediaGlassList->end(); iter++ )
   // {
       // if(strcmp(recievedip.c_str(),((*iter).ipaddress).c_str()) == 0)
       // {
		 // ETG_TRACE_USR4(("clSettingsPlugin::SwivelCalibration(),Sending Calibration Command for the Glass ip %s", (*iter).ipaddress));
	     // //vSetControlUINT8(entityName, (*iter).u32EndStationIdx, (*iter).u32EntityIdx, (*iter).u32ConfigIdx, SWIVEL_UNIT_CONTROL,SWIVEL_CALIBRATION);
		 // vSetControlUINT8(SWIVEL_UNIT_CONTROL,SWIVEL_CALIBRATION);
		 
                    // // Request status of swivel unit
         // //vRequestSwivelStatus((*iter).u32EndStationIdx, (*iter).u32EntityIdx, (*iter).u32ConfigIdx);
		 // vRequestSwivelStatus(0, 0, 0);
         // break;
       // }

   // }
// }

}
}
/************************************************************************
*NAME        : FaderSetting
*SWFLs       : 
************************************************************************/
void clSettingsPlugin::FaderSetting(const boost::shared_ptr<EventDataUtility>& eValue)
{
	ETG_TRACE_USR4(("clSettingsPlugin::FaderSetting() Entered"));
	uint8_t Sink_id = 0;
	uint8_t stepvalue = 0;
	uint8_t factor = 0;
	int stepval = 0;
	const std::vector<EventDataItem*>& dataItems = eValue->getData();
    if ( dataItems.size() == 3)
    {
         for ( uint8_t index = 0; index < 3; index++ )
         {
              EventDataItem* eventDataItem =  dataItems.at(index);
              if ( NULL != eventDataItem )
              {
	             const EventDataItem::Data& eventData = eventDataItem->getData();
	             if ((EventDataItem::UINT8 == eventData._type) && (index == 0))
                 {
                      Sink_id = eventData._value._uint8Value;
                      ETG_TRACE_USR4(("clSettingsPlugin::FaderSetting(), Sinkid Value extracted is : %d", Sink_id));
                 }
	             if ((EventDataItem::UINT8 == eventData._type) && (index == 1))
                 {
                      stepvalue = eventData._value._uint8Value;
                      ETG_TRACE_USR4(("clSettingsPlugin::FaderSetting(), Step Value extracted is : %d", stepvalue));
                 }
	             if ((EventDataItem::UINT8 == eventData._type) && (index == 2))
                 {
                      factor = eventData._value._uint8Value;
                      ETG_TRACE_USR4(("clSettingsPlugin::FaderSetting(), factor Value extracted is : %d", factor));
                 }
              }
          }
		  if(factor == 1)    //Just to understand whether we have recieved Negative ticks from HMI, so if factor is 1 then it is has to be multiplied by -1
		  {
			  stepval = stepvalue * -1;
		  }
                  else
                  {
                         stepval = stepvalue;
                  }
		  ExtractAttenuationValues(stepval);

    }
	else
	{
		ETG_TRACE_USR4(("clSettingsPlugin::FaderSetting(),Expected values are not recieved from HMI"));
	}
}
/************************************************************************
*NAME        : ExtractAttenuationValues
*DESCRIPTION : Requests swivel status
************************************************************************/
void clSettingsPlugin::ExtractAttenuationValues(int stepvalue)
{
	ETG_TRACE_USR4(("clSettingsPlugin::ExtractAttenuationValues step value is %d",stepvalue));
	 std::map<int32_t,int32_t>::iterator it = attenationvalues.find(stepvalue);
	 if (it != attenationvalues.end())
	 {
		 int atenuationvalue = it->second;
		 ETG_TRACE_USR4(("clSettingsPlugin::Extracted Attenation value is %d",atenuationvalue));
		 if(stepvalue < 0)
		 { 
			 m_frontAttenuationValue = atenuationvalue;
			 m_rearAttenuationValue = 0;
		 }
		 if(stepvalue > 0)
		 {
			  m_frontAttenuationValue = 0;
			  m_rearAttenuationValue = atenuationvalue;
		 }
		 if(stepvalue == 0)
		 {
			 m_frontAttenuationValue = 0;
			 m_rearAttenuationValue = 0;
		 }
		 ETG_TRACE_USR4(("clSettingsPlugin::Extracted Front Attenation value is %d",m_frontAttenuationValue));
		 ETG_TRACE_USR4(("clSettingsPlugin::Extracted Rear Attenation value is %d",m_rearAttenuationValue));
		 sendfadersettingcommand();

	}

}
/************************************************************************
*NAME        : sendfadersettingcommand
*DESCRIPTION : Requests swivel status
************************************************************************/
void clSettingsPlugin::sendfadersettingcommand()
{
	ETG_TRACE_USR4(("clSettingsPlugin::Entered SendFaderSettings Command"));
	if( m_CMAEntityDetails.size() > 0 )
	{
		ETG_TRACE_USR4(("clSettingsPlugin::MediaAmplifer size  > 0 "));	
		if(m_CabinAChannelinfo.size() > 0)
		{
		
        if(m_CabinAChannelinfo[0] == FRONT_CHANNEL)
		{
		   vSetControl(  m_CMAEntityDetails.at(0).configuration , CABINACHANNEL1 , m_frontAttenuationValue );
		}
		else
		{
			vSetControl(  m_CMAEntityDetails.at(0).configuration , CABINACHANNEL1 , m_rearAttenuationValue );
		}
		if(m_CabinAChannelinfo[1] == FRONT_CHANNEL)
		{
			vSetControl( m_CMAEntityDetails.at(0).configuration ,  CABINACHANNEL2 , m_frontAttenuationValue );
		}
		else
		{
			vSetControl( m_CMAEntityDetails.at(0).configuration ,  CABINACHANNEL2 , m_rearAttenuationValue );
		}
		if(m_CabinAChannelinfo[2] == FRONT_CHANNEL)
		{
			vSetControl( m_CMAEntityDetails.at(0).configuration ,  CABINACHANNEL3 , m_frontAttenuationValue );
		}
		else
		{
			vSetControl( m_CMAEntityDetails.at(0).configuration ,  CABINACHANNEL3 , m_rearAttenuationValue);
		}
		}
	}

		
	else
	{
		ETG_TRACE_USR4(("No Amplifier connected "));
	}

}

/************************************************************************
*NAME        : vSetControl
*DESCRIPTION : To Send Fader Settings value to CMA
************************************************************************/
void clSettingsPlugin::vSetControl(avdecc_lib::configuration_descriptor * configuration, uint32_t u32ControlDescIdx, uint8_t uint8Value)
{
     ETG_TRACE_USR4(("clSettingsPlugin::vSetControlUINT8(), Control index : %d, Value : %d",u32ControlDescIdx, uint8Value));

 	ControlDataElement controlElement;
	controlElement.configuration = configuration;
	controlElement.u32ControlDescIdx = u32ControlDescIdx;
	controlElement.uint8Value = uint8Value;


	if(m_pCommControl != NULL)
	{
		ETG_TRACE_USR4(("clSettingsPlugin SetControl"));
		m_pCommControl->setControl(controlElement);
	}
	else
	{
		ETG_TRACE_USR4(("clSettingsPlugin::Control Pointer is NULL"));
	}
}

/************************************************************************
*NAME        : vRequestSwivelStatus
*DESCRIPTION : Requests swivel status
************************************************************************/
void clSettingsPlugin::vRequestSwivelStatus(uint32_t u32EndStationIdx, uint32_t u32EntityDescIdx, uint32_t u32ConfigDescIdx)
{
	swivelStatusControlElem.configuration = m_CMGEntityDetails.at(0).configuration;
	swivelStatusControlElem.u32ControlDescIdx = SWIVEL_UNIT_STATUS;
		
	if(m_pCommControl != NULL)
	{
		ETG_TRACE_USR4(("clSettingsPlugin vRequestSwivelStatus getControl"));
		m_pCommControl->getControl(swivelStatusControlElem);
	}

}


/************************************************************************
*NAME        : bFindEntityInList
*DESCRIPTION : Design section 8.4.1
************************************************************************/
bool clSettingsPlugin::bFindEntityInList(uint64_t u64EntityID, std::vector<stEntityDesc> MediaGlass, uint32_t &entityIndex)
{
    bool found = false;

    for (uint32_t index = 0; index < MediaGlass.size(); index++)
    {
        if ( u64EntityID == MediaGlass[index].u64EntityID )
        {
            found = true;
            entityIndex = index;
            break;
        }
    }

    return found;
}

/************************************************************************
 *FUNCTION: 		vRegisterUnSolicitedResponse
 *DESCRIPTION: 		Registering with Camport for UnsolicitedResponse
 *REQUIREMENT ID:	SWFL-2972
 *PARAMETER:		NA
 *RETURNVALUE: 		NA
 *HISTORY:
 *02.25.2019		1.0		Sunil Sagar		Initial Version
 ************************************************************************/
void clSettingsPlugin::vRegisterUnSolicitedResponse(ACClientInterface *daemonClient, avdecc_lib::end_station * end_station)
{
    //avdecc_lib::end_station *end_station = end_station;
    if( nullptr != daemonClient)
    {
        ACNotificationId *cmd_notification_id = daemonClient->get_async_notification_id();
        if( nullptr != cmd_notification_id && nullptr != end_station)
        {
            end_station->send_register_unsolicited_cmd((void *)cmd_notification_id);
        }
    }
}

/************************************************************************
*NAME        : vRequestSwivelAvailability
*DESCRIPTION : Requests swivel availability
************************************************************************/
void clSettingsPlugin::vRequestSwivelAvailability(uint32_t u32EndStationIdx, uint32_t u32EntityDescIdx, uint32_t u32ConfigDescIdx)
{
    // swivelAvailabilityControlElem.u32EndStationIdx = u32EndStationIdx;
    // swivelAvailabilityControlElem.u32EntityDescIdx = u32EntityDescIdx;
    // swivelAvailabilityControlElem.u32ConfigDescIdx = u32ConfigDescIdx;
    // swivelAvailabilityControlElem.u32ControlDescIdx = SWIVEL_UNIT_AVAILABILITY;
    // vGetControl(clientIf,swivelAvailabilityControlElem);
	
	swivelAvailabilityControlElem.configuration = m_CMGEntityDetails.at(0).configuration;
	swivelAvailabilityControlElem.u32ControlDescIdx = SWIVEL_UNIT_AVAILABILITY;
	//swivelStatusControlElem.uint8Value = uint8Value;
		
	if(m_pCommControl != NULL)
	{
		ETG_TRACE_USR4(("clSettingsPlugin vRequestSwivelAvailability getControl"));
		m_pCommControl->getControl(swivelAvailabilityControlElem);
	}
}

/************************************************************************
*NAME        : bFindEntityID
*DESCRIPTION : Design section 8.4.1 - PluginBase
************************************************************************/

bool clSettingsPlugin::bFindEntityID(uint64_t u64EntID)
{
	bool bEntityIdForCMG = false;
	   if( m_bParseState ) {
		   vector<uint64_t>vecEntityIDCMG = m_pEntityInfo->vectorGetEntityID(DEV_NAME_CMG);
		   for (vector<uint64_t>::iterator iter = vecEntityIDCMG.begin(); iter != vecEntityIDCMG.end(); iter++)
		   {
		      if((*iter)== u64EntID )
		      {
		    	  ETG_TRACE_USR4(("clSettingsPlugin::bFindEntityID(), Entity Id found for CMG %lx",(*iter)));
		    	  bEntityIdForCMG = true;
		    	  break;
		      }
		   }
	   }
	   return bEntityIdForCMG;
}


/************************************************************************
 *FUNCTION: 		sGetEntityCabinName
 *DESCRIPTION: 		Cabin mapping for
 *REQUIREMENT ID:	SWFL-2972
 *PARAMETER:		NA
 *RETURNVALUE: 		NA
 *HISTORY:
 *02.25.2019		1.0		Sunil Sagar		Initial Version
 ************************************************************************/
std::string clSettingsPlugin::sGetEntityCabinName(uint64_t u64EntityID)
{
	ETG_TRACE_USR4(("clSettingsPlugin::sGetEntityCabinName(), entityId : %lx ",u64EntityID ));
	std::string sentityCabin;
	if( bFindEntityID(u64EntityID) ) {
		std::string slinkName = m_pEntityInfo->sGetEntitySinkLinkName(u64EntityID);
		tSinkLinkCabinMap mapData = m_pEntityInfo->mapGetSinkCabinLinkList();
		tSinkLinkCabinMap::iterator it = mapData.find(slinkName);
		if (it != mapData.end()){
			std::string sinkName = it->second;
			if( GLASS_A_REGION_1 == sinkName ){
				sentityCabin = COACHMEDIA_GLASS_A1;
			}
			else if(GLASS_A_REGION_2 == sinkName ){
				sentityCabin = COACHMEDIA_GLASS_A2;
			}
			else if(GLASS_B_REGION_1 == sinkName ){
				sentityCabin = COACHMEDIA_GLASS_B1;
			}
			else if(GLASS_B_REGION_2 == sinkName ){
				sentityCabin = COACHMEDIA_GLASS_B2;
			}
		}
	}
    return sentityCabin;
}


void clSettingsPlugin::vHandleSwivelUpDownStatus(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clSettingsPlugin::vHandleSwivelUpDownStatus"));
	boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
	
    uint16_t u16DescIndex = SWIVEL_UNIT_CONTROL;
	if(NULL != pluginData.get())
	{
		pluginData->addEventData(u16DescIndex);
		pluginData->addEventData(m_u16LastExecutedSetCommand);
	}

	RecvMsgFromPlugin(PLUGIN_NAME_SETTINGS,CTRLBLOCK_NAME_SETTINGS_SWIVELUPDOWN, pluginData);
}

void clSettingsPlugin::vHandleLCDOnOffStatus(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clSettingsPlugin::vHandleLCDOnOffStatus"));
	boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
	
    uint16_t u16DescIndex = LCD_PANEL_ON_OFF;
	if(NULL != pluginData.get())
	{
		pluginData->addEventData(u16DescIndex);
		pluginData->addEventData(m_u16LastExecutedSetCommand);
	}

	RecvMsgFromPlugin(PLUGIN_NAME_SETTINGS,CTRLBLOCK_NAME_SETTINGS_GLASSLCD, pluginData);
}
