/* ***************************************************************************************
* FILE:          clMasterControlPlugin.cpp
* SW-COMPONENT:  avdecc_appl_plugins
* DESCRIPTION:  clMasterControlPlugin.cpp is part of avdecc_appl_plugins library
* COPYRIGHT:  (c) 2020-21 Robert Bosch Car Multimedia GmbH
* HISTORY: 
* AUTHOR:  vta5kor
* REVISION: 
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
/*****************************************************************
| includes
|----------------------------------------------------------------*/
#include <clMasterControlPlugin.h>
#include "plugin_trace.h"
#include "MediaCameraPort.h"
#include "clControlMapping.h"
#include "clPluginDataProvider.h"
#include "PluginDataType.h"
#include "PluginEvtListenerBase.h"
#include "PluginEventBase.h"
#include "PluginDataType.h"
#include "AudioDefines.h"
#include "MediaGlass.h"



#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_PLUGIN_MASTER
#include "trcGenProj/Header/clMasterControlPlugin.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN



clMasterControlPlugin* clMasterControlPlugin::poSelf = NULL;
clMasterControlPlugin::clMasterControlPlugin(){
	setMapFuncs();
	//Communucation Protocol Object creation logic
	m_pCommControl = new clCommunicationProtocol();
	if(m_pCommControl != NULL)
	{
		ETG_TRACE_USR4(("clMasterControlPlugin::clCommunicationProtocol is created"));
	}
	m_CabinApreviousStreamType = AUDIOCONTROL_STREAM_NONE;
	m_CabinA2previousStreamType = AUDIOCONTROL_STREAM_NONE;
	m_pEntityInfo = tclAvRoutingParser::pGetInstance();
	vIntializeVolumeMap();
	m_U16BeepVolume = 16;

}

clMasterControlPlugin::~clMasterControlPlugin(){
	delete m_pCommControl;
}

clMasterControlPlugin* clMasterControlPlugin::getInstance()
{
	ETG_TRACE_USR4(("clMasterControlPlugin::getInstance"));
	if(poSelf == NULL)
	{
		ETG_TRACE_USR4(("Creating object clMasterControlPlugin"));
		poSelf = new clMasterControlPlugin();
	}
	ETG_TRACE_USR4(("Return object clMasterControlPlugin %d", poSelf));
	return poSelf;
	
}
void clMasterControlPlugin::deleteInstance()
{
	if(NULL != poSelf)
	{
		delete poSelf;
	}
}

void clMasterControlPlugin::vEventESReadComplete()
{
	//extract the details of the entities required
	clPluginDataProvider *ptrMain = clPluginDataProvider::getInstance();
	if(NULL != ptrMain)
	{
		std::vector<stEntityDesc> stEntityDetails;
  		std::vector<stEntityDesc> stGlassDetails;
		// Read Camera port details
		bool found = ptrMain->getEndStationDetails(CMP, stEntityDetails);
		if(true == found)
		{
			UpdateCMPEntityDetails(stEntityDetails);			
		}
		else 
		{
			
			ETG_TRACE_USR4(("NO CMP End station details "));
		}

		//Read Amplifier Details
		stEntityDetails.clear();
		found = ptrMain->getEndStationDetails(CMA, stEntityDetails);
		if (found)
		{
			UpdateCMAEntityDetails(stEntityDetails);
		}
		else 
		{
			ETG_TRACE_USR4(("NO Amplifier End station details "));
		}		
		// REad Glass details
		found = ptrMain->getEndStationDetails(CMG, stGlassDetails);	
		if (found)
		UpdateCMGEntityDetails(stGlassDetails);
		else
		ETG_TRACE_USR4(("NO Glass End station details "));
		
	}			
			
}		
		
void clMasterControlPlugin::UpdateCMPEntityDetails(std::vector<stEntityDesc> stEntityDetails)	
{		
	ETG_TRACE_USR4(("clMasterControlPlugin::UpdateCMPEntityDetails::Copying Entity details"));	
    	m_CMPEntityDetails.clear();	
	m_CMPEntityDetails = stEntityDetails;		
	
        ETG_TRACE_USR4(("vEventESReadComplete m_CMPEntityDetails size is %d", m_CMPEntityDetails.size()));


	//register for callbacks
	clControlMapping *p_controlmap = clControlMapping::getInstance();

	if(NULL != p_controlmap)
	{
	//	p_controlmap->setRespCallBack(CMP, STREAM1MAXTRANSITTIME, this, (FUNCPTR)(&clMasterControlPlugin::onSetMaxTransitTime));
		
	}	
}

void clMasterControlPlugin::UpdateCMAEntityDetails(std::vector<stEntityDesc> stEntityDetails)	
{		
	ETG_TRACE_USR4(("clMasterControlPlugin::UpdateCMAEntityDetails::Copying Entity details"));	
	m_CMAEntityDetails.clear();
	m_CMAEntityDetails = stEntityDetails;
	SetBeepVolume(0,m_U16BeepVolume);
		
	ETG_TRACE_USR4(("vEventESReadComplete m_CMAEntityDetails size is %d", m_CMAEntityDetails.size()));
	//register for callbacks
	clControlMapping *p_controlmap = clControlMapping::getInstance();

	if(NULL != p_controlmap)
	{
	//	p_controlmap->setRespCallBack(CMP, STREAM1MAXTRANSITTIME, this, (FUNCPTR)(&clMasterControlPlugin::onSetMaxTransitTime));
		
	}	
}


void clMasterControlPlugin::UpdateCMGEntityDetails(std::vector<stEntityDesc> stEntityDetails)
{
    ETG_TRACE_USR4(("clMasterControlPlugin::UpdateCMGEntityDetails() size is %d", stEntityDetails.size()));
	m_AMediaGlassList.clear();
	m_A2MediaGlassList.clear();
	
	for (tU8 u8index =0;u8index < stEntityDetails.size(); u8index++ )
	{
     uint64_t u64EntityID = stEntityDetails[u8index].u64EntityID;		
     std::string entName = m_pEntityInfo->sGetEntitySinkLinkName(u64EntityID);
	 ETG_TRACE_USR4(("clMasterControlPlugin::sGetEntitySinkLinkName() is %s", entName));

        if (!(entName.empty()))
        {
            uint32_t entityIndex;
            if ( (strcmp(entName.c_str(), CABINA_GROUP_GLASS) == 0) )
            {
                //if (!bFindEntityInList(u64EntityID, m_AMediaGlassList, entityIndex))
                {
                    m_AMediaGlassList.push_back(stEntityDetails[u8index]);
                    ETG_TRACE_USR4(("clMasterControlPlugin::UpdateCMGEntityDetails(), GlassA   added, entity name , %s", entName.c_str()));
                    ETG_TRACE_USR4(("clMasterControlPlugin::UpdateCMGEntityDetails(), GlassA  vector size: %d ",m_AMediaGlassList.size()));
                }
            }
            else if ( (strcmp(entName.c_str(), CABINA2_GROUP_GLASS) == 0) )
            {
               // if (!bFindEntityInList(u64EntityID, m_A1MediaGlassList, entityIndex))
                {
                    m_A2MediaGlassList.push_back(stEntityDetails[u8index]);
                    ETG_TRACE_USR4(("clMasterControlPlugin::UpdateCMGEntityDetails(), GlassA1 added, entity name , %s", entName.c_str()));
                    ETG_TRACE_USR4(("clMasterControlPlugin::UpdateCMGEntityDetails()), GlassA1 vector size: %d ",m_A2MediaGlassList.size()));
                }
            }
            else
            {
                ETG_TRACE_USR4(("clMasterControlPlugin::UpdateCMGEntityDetails(), glass group unknown "));

			}

		}
	}
	
}


/************************************************************************
*NAME        : bFindEntityInList
*DESCRIPTION : Design section 8.4.1
************************************************************************/
bool clMasterControlPlugin::bFindEntityInList(uint64_t u64EntityID, std::vector<stEntityDesc> MediaGlass, uint32_t &entityIndex)
{
    bool found = false;

    for (uint32_t index = 0; index < MediaGlass.size(); index++)
    {
        if ( u64EntityID == MediaGlass[index].u64EntityID )
        {
            found = true;
            entityIndex = index;
            break;
      }
    }

    return found;
}

/*****************************************************************************************************
*NAME        : setMapFuncs
*DESCRIPTION : add fucntion pointers map
******************************************************************************************************/
void clMasterControlPlugin::setMapFuncs()
{
	ETG_TRACE_USR4(("clMasterControlPlugin::setMapFuncs"));
 
    m_MemberFunctionMap[CTRLBLOCK_NAME_MASTER_STREAM_SELECT] = &clMasterControlPlugin::onStreamSelection;
	m_MemberFunctionMap[CTRLBLOCK_NAME_CABIN_VOLUME_CHANGE] = &clMasterControlPlugin::onVolumeControlChange;
	m_MemberFunctionMap[CTRLBLOCK_NAME_MIX_SOURCE_CHANGE] = &clMasterControlPlugin::onMixSourceChange;
	m_MemberFunctionMap[CTRBLOCK_NAME_PRESENTATION_TIME_OFFSET] = &clMasterControlPlugin::onSetPresentationTimeOffset;
	m_MemberFunctionMap[CTRBLOCK_NAME_MAX_TRANSIT_TIME] = &clMasterControlPlugin::onSetMaxTransitTime;
	m_MemberFunctionMap[CTRBLOCK_NAME_BEEP_VOLUME] = &clMasterControlPlugin::onBeepVolumeChange;
	m_MemberFunctionMap[CTRBLOCK_NAME_BEEP_OUTPUT] = &clMasterControlPlugin::onBeepOutputUpdate;	
	
}

/************************************************************************
*NAME        : onStreamSelection
*SWFLs       : SWFL-2858, SWFL-2859
************************************************************************/
void clMasterControlPlugin::onStreamSelection(const boost::shared_ptr<EventDataUtility>& eValue)
{
   ETG_TRACE_USR4(("MasterControlPlugin::onStreamSelection()"));
   uint8_t selectionValues[2] = {0xFF};
   if(NULL != eValue.get())
   {
      const std::vector<EventDataItem*>& dataItems = eValue->getData();
      for(uint8_t index = 0; index < (sizeof(selectionValues) / sizeof(uint8_t)); index++)
      {
         EventDataItem* eventDataItem =  dataItems.at(index);
         if(NULL != eventDataItem)
         {
            const EventDataItem::Data& eventData = eventDataItem->getData();
            if(EventDataItem::UINT8 == eventData._type)
            {
               selectionValues[index] = eventData._value._uint8Value;
               ETG_TRACE_USR4(("MasterControlPlugin::onStreamSelection(), Value extracted is : %d", selectionValues[index]));
            }
         }
      }
   }
   switch(selectionValues[0])
   {
	   case CABIN_A_MEDIAGLASS1_SINK_ID:
	   {
		   if (selectionValues[1] != m_CabinApreviousStreamType)
		   selectStream(selectionValues[0], selectionValues[1]);
		   else
			ETG_TRACE_USR4(("MasterControlPlugin::onStreamSelection(), same Sink ID : %d, Stream type : %d", selectionValues[0], selectionValues[1]));   
			break;
	   }
	   case CABIN_A_MEDIAGLASS2_SINK_ID:
	   {
		   if (selectionValues[1] != m_CabinA2previousStreamType)
		   selectStream(selectionValues[0], selectionValues[1]);
		   else
			ETG_TRACE_USR4(("MasterControlPlugin::onStreamSelection(), same Sink ID : %d, Stream type : %d", selectionValues[0], selectionValues[1]));   
			break;
	   }
	   default:
	   ETG_TRACE_USR4(("MasterControlPlugin::onStreamSelection() default "));
	   break;
   }
   
}

/************************************************************************
*NAME        : selectStream
*SWFLs       : SWFL-2858, SWFL-2859
onStreamSelection************************************************************************/
void clMasterControlPlugin::selectStream(uint8_t sinkId, uint8_t streamType)
{
	//sinkId = CABIN_A_MEDIAGLASS1_SINK_ID;
   ETG_TRACE_USR4(("MasterControlPlugin::selectStream(), Sink ID : %d, Stream type : %d", sinkId, streamType));
    

   if((AUDIOCONTROL_AVB_STREAM == streamType) || (AUDIOCONTROL_RTP_STREAM == streamType))
   {
      std::vector<stEntityDesc>* entityDescList = NULL;
      const char* entityName = NULL;

      switch(sinkId)
      {
         case CABIN_A_MEDIAGLASS1_SINK_ID:
         {
            entityDescList = &m_AMediaGlassList;
            entityName = COACHMEDIA_GLASS_A;   // to be changed later to A1
			m_CabinApreviousStreamType = streamType;
            break;
         }
         case CABIN_A_MEDIAGLASS2_SINK_ID:
         {
            entityDescList = &m_A2MediaGlassList;
            entityName = COACHMEDIA_GLASS_A;  // to be changed later to A2
			m_CabinA2previousStreamType = streamType;
            break;
         }
         default:
         {
            ETG_TRACE_USR4(("MasterControlPlugin::selectStream(), Sink ID : %d not handled", sinkId));
         }
      }
      ETG_TRACE_USR4(("MasterControlPlugin::selectStream(), Entity name : %s", entityName));

      if(NULL != entityDescList)
      {
         for (std::vector<stEntityDesc>::iterator iter = entityDescList->begin(); iter != entityDescList->end(); iter++)
         {
            //Send Set_CONTROL Command to mediaGlass to change it to take RTP/AVB Streaming
			ETG_TRACE_USR4(("MasterControlPlugin::selectStream(), Entity ID : %lx", (*iter).u64EntityID));
            vSetControlUINT8((*iter).configuration,SOURCE, streamType);
			//ETG_TRACE_USR4(("MasterControlPlugin::selectStream(), Entity name : %s", entityName  ));
         }
      }
   }
}


/************************************************************************
*NAME        : vSetControlUINT8
*DESCRIPTION : Design section 8.4.1 - PluginBase
************************************************************************/
void clMasterControlPlugin::vSetControlUINT8(avdecc_lib::configuration_descriptor * configuration, uint32_t u32ControlDescIdx, uint8_t uint8Value)
{
     ETG_TRACE_USR4(("MasterControlPlugin::vSetControlUINT8(), Control index : %d, Value : %d",u32ControlDescIdx, uint8Value));

 	ControlDataElement controlElement;
	controlElement.configuration = configuration;
	controlElement.u32ControlDescIdx = u32ControlDescIdx;
	controlElement.uint8Value = uint8Value;


	if(m_pCommControl != NULL)
	{
		ETG_TRACE_USR4(("MasterControlPlugin SetControl"));
		m_pCommControl->setControl(controlElement);
	}
	else
	{
		ETG_TRACE_USR4(("MasterControlPlugin::Control Pointer is NULL"));
	}
}


/************************************************************************
*NAME        : onSetMaxTransitTime
*DESCRIPTION : Called on every source change to set the transit time to Camport
************************************************************************/
void clMasterControlPlugin::onSetMaxTransitTime(const boost::shared_ptr<EventDataUtility>& eValue)
{
	ETG_TRACE_USR4(("MasterControlPlugin::onSetMaxTransitTime()"));
	uint8_t selectionValues[2] = {0xFF};
	if(NULL != eValue.get())
   {
      const std::vector<EventDataItem*>& dataItems = eValue->getData();
      for(uint8_t index = 0; index < (sizeof(selectionValues) / sizeof(uint8_t)); index++)
      {
		 EventDataItem* eventDataItem =  dataItems.at(index);
         if(NULL != eventDataItem)
         {
			const EventDataItem::Data& eventData = eventDataItem->getData();
            if(EventDataItem::UINT8 == eventData._type)
			{
				selectionValues[index] = eventData._value._uint8Value;
			}
		 }
	  }
   }
	  SetTransitTimeControl(selectionValues[0],selectionValues[1]);
}

/************************************************************************
*NAME        : SetTransitTimeControl
*DESCRIPTION : Function to determine the Control to be set
************************************************************************/
  
void clMasterControlPlugin::SetTransitTimeControl(uint8_t source,uint8_t value)
{
	ETG_TRACE_USR4(("MasterControlPlugin::SetTransitTimeControl()"));
	uint8_t ControlIndex = 0xFF;
	switch(source)
	{
		case CMP1_IPCAM:		
		   ControlIndex = STREAM2MAXTRANSITTIME;
		   break;
		case CMP1_USB1 :
		case CMP1_HDMI:
		case CMP1_CAMERA1:
		case CMP1_CAMERA2:
		case CMP1_CAMERA3:
		case CMP1_CAMERA4:	
		case CMP1_SPLITSCREEN:
		   ControlIndex = STREAM1MAXTRANSITTIME;
		   break;
		default:
			break;
	}
	 if (0xFF != ControlIndex)
	{
		if(m_CMPEntityDetails.size() != 0)
		{
			vSetControlUINT8( m_CMPEntityDetails.at(0).configuration,ControlIndex,value );
		}
		else
		{
			ETG_TRACE_USR4(("MasterControlPlugin::SetTransitTimeControl Cameraport is not part of AVB network"));
		}
	}
	else
	{
		ETG_TRACE_USR4(("MasterControlPlugin::SetTransitTimeControl ControlIndex 0xFF"));
	}
}

/************************************************************************
*NAME        : onSetPresentationTimeOffset
*DESCRIPTION : Called on every source change to set the Presentation Time 
************************************************************************/
void clMasterControlPlugin::onSetPresentationTimeOffset(const boost::shared_ptr<EventDataUtility>& eValue)
{
	ETG_TRACE_USR4(("MasterControlPlugin::onSetPresentationTimeOffset()"));
	uint8_t selectionValues[3] = {0xFF};
  	uint16_t presentationOffset = 0;
	
	if(NULL != eValue.get())
   {
      const std::vector<EventDataItem*>& dataItems = eValue->getData();
      for(uint8_t index = 0; index < (dataItems.size()); index++)
      {
		 EventDataItem* eventDataItem =  dataItems.at(index);
         if(NULL != eventDataItem)
         {
			const EventDataItem::Data& eventData = eventDataItem->getData();
            if(EventDataItem::UINT8 == eventData._type)
			{
              			//ETG_TRACE_USR4(("MasterControlPlugin::EventDataItem::UINT8 == eventData._type"));
				selectionValues[index] = eventData._value._uint8Value;
			}
           else if (EventDataItem::UINT16 == eventData._type)
			{
				//ETG_TRACE_USR4(("MasterControlPlugin::EventDataItem::UINT16 == eventData._type"));
				presentationOffset = eventData._value._uint16Value;
			}
		
	  }
	  
   }
	ETG_TRACE_USR4(("MasterControlPlugin::onSetPresentationTimeOffset() %d %d %d ",selectionValues[0],selectionValues[1], presentationOffset));
	SetPresentationTimeControl(selectionValues[0],selectionValues[1], presentationOffset);
}
}

/************************************************************************
*NAME        : SetPresentationTimeControl
*DESCRIPTION : Function to determine the Control to be set
************************************************************************/
  
void clMasterControlPlugin::SetPresentationTimeControl(uint8_t source,uint8_t sinkId,uint16_t Offset)
{
	ETG_TRACE_USR4(("MasterControlPlugin::SetPresentationTimeControl(): Offset :%d", Offset));
	
	std::vector<stEntityDesc>* entityDescList = NULL;
    	const char* entityName = NULL;
	uint8_t ControlIndex = PRESENTATIONTIMEOFFSET;
	uint8_t PTOffset = Offset/10;
	ETG_TRACE_USR4(("MasterControlPlugin::Presentation() PTOffset - %d ", PTOffset));

	switch (sinkId)
	{
		case CABIN_A_SINK_ID:
		case CABIN_B_SINK_ID:
		{
			entityDescList = &m_CMAEntityDetails;
			entityName = COACHMEDIA_AMPLIFIER;
			ControlIndex = AUDIOCONTROL_CMA_PRESENTATION;	
			break;
		}
		 case CABIN_A_MEDIAGLASS1_SINK_ID:
         {
            entityDescList = &m_AMediaGlassList;
            entityName = COACHMEDIA_GLASS_A;   // to be changed later to A1
            break;
         }
         case CABIN_A_MEDIAGLASS2_SINK_ID:
         {
            entityDescList = &m_A2MediaGlassList;
            entityName = COACHMEDIA_GLASS_A;  // to be changed later to A2
            break;
         }
        /*  case CABIN_B_MEDIAGLASS1_SINK_ID:
         {
            entityDescList = &m_A1MediaGlassList;
            entityName = COACHMEDIA_GLASS_B;   // to be changed later to B1
            break;
         }
         case CABIN_B_MEDIAGLASS2_SINK_ID:
         {
            entityDescList = &m_A1MediaGlassList;
            entityName = COACHMEDIA_GLASS_B;  // to be changed later to B2
            break;
         } */
         default:
         {
            ETG_TRACE_USR4(("MasterControlPlugin::Presentation(), Sink ID : %d not handled", sinkId));
         } 
	}	
	
       
    if(entityDescList != NULL)
    {
        if(entityDescList->size() != 0)
        {
             for (std::vector<stEntityDesc>::iterator iter = entityDescList->begin(); iter != entityDescList->end(); iter++)
            {
                ETG_TRACE_USR4(("MasterControlPlugin::Presentation time offset(), Entity ID : %lx", (*iter).u64EntityID));
                vSetControlUINT8((*iter).configuration,ControlIndex,PTOffset);
            }
        }
    }
}

void clMasterControlPlugin::onVolumeControlChange(const boost::shared_ptr<EventDataUtility>& eValue)
{
   ETG_TRACE_USR4(("clMasterControlPlugin::onVolumeControlChange()"));
   cout<<"clMasterControlPlugin::onVolumeControlChange()";

   uint16_t selectionValues[1] = {0xFF};
   if(NULL != eValue.get())
   {
      const std::vector<EventDataItem*>& dataItems = eValue->getData();
      for(uint8_t index = 0; index < (sizeof(selectionValues) / sizeof(uint16_t)); index++)
      {
		 EventDataItem* eventDataItem =  dataItems.at(index);
         if(NULL != eventDataItem)
         {
			const EventDataItem::Data& eventData = eventDataItem->getData();
            if(EventDataItem::UINT16 == eventData._type)
            {
				selectionValues[index] = eventData._value._uint16Value;
			   
			   if( m_CMAEntityDetails.size() > 0 )
			   {
				   //ETG_TRACE_USR4(("MasterControlPlugin::onVolumeControlChange() m_CMAEntityDetails.size() > 0"));
				  vSetControlUINT8( m_CMAEntityDetails.at(0).configuration, VOLUMECONTROL_INPUT , selectionValues[0] );
			   }
			   
               ETG_TRACE_USR4(("MasterControlPlugin::onVolumeControlChange(), Value extracted is : %d", selectionValues[index]));
            }
         }
      }
   }
   
}
void clMasterControlPlugin::onMixSourceChange(const boost::shared_ptr<EventDataUtility>& eValue)
{
	ETG_TRACE_USR4(("MasterControlPlugin::onMixSourceChange()"));
	uint8_t selectionValues[2] = {0xFF};
   if(NULL != eValue.get())
   {
      const std::vector<EventDataItem*>& dataItems = eValue->getData();
      for(uint8_t index = 0; index < (dataItems.size()); index++)
      {
		 EventDataItem* eventDataItem =  dataItems.at(index);
         if(NULL != eventDataItem)
         {
			const EventDataItem::Data& eventData = eventDataItem->getData();
            if(EventDataItem::UINT8 == eventData._type)
            {
				selectionValues[index] = eventData._value._uint8Value;
			   
               ETG_TRACE_USR4(("MasterControlPlugin::onMixSourceChange(), Value extracted is : %d", selectionValues[index]));
			   
			  
            }
         }
      }

	ETG_TRACE_USR4(("MasterControlPlugin::Calling SetMixSourceValues  sink :%d , source : %d ", selectionValues[0],selectionValues[1]));	
	SetMixSourceValues(selectionValues[0],selectionValues[1]);
   }
	
}

void clMasterControlPlugin::SetMixSourceValues(uint8_t sinkid,uint8_t source)
{
	ETG_TRACE_USR4(("MasterControlPlugin::SetMixSourceValues -  Sinkid :%d, source - %d", sinkid , source));
        // Sinkid to be used when Cabin B is available- SOP2
	if( m_CMAEntityDetails.size() > 0 )
	{
		ETG_TRACE_USR4(("MasterControlPlugin::MediaAmplifer size  > 0 "));	
		for (uint8_t index = 0 ;index <  m_CMAEntityDetails.size() ; index++)
		{	

		vSetControlUINT8(  m_CMAEntityDetails.at(index).configuration , MIX_SOURCE_1 , source );

		vSetControlUINT8( m_CMAEntityDetails.at(index).configuration ,  MIX_SOURCE_2 , source );

		vSetControlUINT8( m_CMAEntityDetails.at(index).configuration  , MIX_SOURCE_3 , source );
		}
	}
	else
	{
		ETG_TRACE_USR4(("No Amplifier connected "));
	}

}


void clMasterControlPlugin::vIntializeVolumeMap()
{

ETG_TRACE_USR4(("Intializing Volume Map  "));
	m_mapVolume[0]= 136;
	m_mapVolume[1]= 177;
	m_mapVolume[2]= 185;
	m_mapVolume[3]= 193;
	m_mapVolume[4]= 199;
	m_mapVolume[5]= 203;
	m_mapVolume[6]= 206;
	m_mapVolume[7]= 209;
	m_mapVolume[8]= 211;
	m_mapVolume[9]= 213;
	m_mapVolume[10]= 215;
	m_mapVolume[11]= 217;
	m_mapVolume[12]= 219;
	m_mapVolume[13]= 221;
	m_mapVolume[14]= 222;
	m_mapVolume[15]= 224;
	m_mapVolume[16]= 225;
	m_mapVolume[17]= 226;
	m_mapVolume[18]= 227;
	m_mapVolume[19]= 229;
	m_mapVolume[20]= 230;
	m_mapVolume[21]= 231;
	m_mapVolume[22]= 233;
	m_mapVolume[23]= 234;
	m_mapVolume[24]= 235;
	m_mapVolume[25]= 236;
	m_mapVolume[26]= 237;
	m_mapVolume[27]= 239;
	m_mapVolume[28]= 241;
	m_mapVolume[29]= 242;
	m_mapVolume[30]= 243;
	m_mapVolume[31]= 244;
	m_mapVolume[32]= 246;
	m_mapVolume[33]= 247;
	m_mapVolume[34]= 248;
	m_mapVolume[35]= 249;
	m_mapVolume[36]= 250;
	m_mapVolume[37]= 252;
	m_mapVolume[38]= 254;
	m_mapVolume[39]= 255;
	m_mapVolume[40]= 0;

}

/******************************************************************************
*NAME          : u8GetVolume_dBValue
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get the Volume decibel value from the mapping table
******************************************************************************/

uint8_t clMasterControlPlugin::u8GetVolume_dBValue (uint8_t u8Volume)
{
  
   ETG_TRACE_USR4(("clMasterControlPlugin::u8GetVolume_dBValue  Volume Value : %d , DB value : %d ",u8Volume , m_mapVolume.find(u8Volume)->second));
	return m_mapVolume.find(u8Volume)->second;
}


void clMasterControlPlugin::onBeepVolumeChange(const boost::shared_ptr<EventDataUtility>& eValue)
{
	ETG_TRACE_USR4(("MasterControlPlugin::onBeepVolumeChange()"));
	uint8_t selectionValues[2] = {0xFF};
  	 if(NULL != eValue.get())
  	 {
      	const std::vector<EventDataItem*>& dataItems = eValue->getData();
     	 for(uint8_t index = 0; index < (dataItems.size()); index++)
      	{
		 EventDataItem* eventDataItem =  dataItems.at(index);
         if(NULL != eventDataItem)
         {
		const EventDataItem::Data& eventData = eventDataItem->getData();
            if(EventDataItem::UINT8 == eventData._type)
            {
				selectionValues[index] = eventData._value._uint8Value;
			   
               ETG_TRACE_USR4(("MasterControlPlugin::onBeepVolumeChange(), Value extracted is : %d", selectionValues[index]));
			   
			  
            }
         }
      }

	ETG_TRACE_USR4(("MasterControlPlugin::Calling SetMixSourceValues  sink :%d , Volume : %d ", selectionValues[0],selectionValues[1]));	
	m_U16BeepVolume = selectionValues[1];
	SetBeepVolume(selectionValues[0],selectionValues[1]);
   }
	
}

void clMasterControlPlugin::SetBeepVolume(uint8_t sinkid,uint8_t volume)
{
	ETG_TRACE_USR4(("MasterControlPlugin::SetBeepVolume -  Sinkid :%d, volume - %d", sinkid , volume));
	uint8_t VolumeInDecibel = u8GetVolume_dBValue(volume);
	ETG_TRACE_USR4(("MasterControlPlugin::VolumeInDecibel :%d ", VolumeInDecibel));

        // Sinkid to be used when Cabin B is available- SOP2
	if( m_CMAEntityDetails.size() > 0 )
	{
		ETG_TRACE_USR4(("MasterControlPlugin::MediaAmplifer size  > 0 "));	
		for (uint8_t index = 0 ;index <  m_CMAEntityDetails.size() ; index++)
		{
			vSetControlUINT8(m_CMAEntityDetails.at(index).configuration ,BEEP_VOLUME, VolumeInDecibel);
		}
	}
	else
	{
		ETG_TRACE_USR4(("No Amplifier connected "));
	}

}


void clMasterControlPlugin::onBeepOutputUpdate(const boost::shared_ptr<EventDataUtility>& eValue)
{
	ETG_TRACE_USR4(("MasterControlPlugin::onBeepOutputUpdate()"));
	uint8_t selectionValues[2] = {0xFF};
  	 if(NULL != eValue.get())
  	 {
      	const std::vector<EventDataItem*>& dataItems = eValue->getData();
     	 for(uint8_t index = 0; index < (dataItems.size()); index++)
      	{
		 EventDataItem* eventDataItem =  dataItems.at(index);
         if(NULL != eventDataItem)
         {
		const EventDataItem::Data& eventData = eventDataItem->getData();
            if(EventDataItem::UINT8 == eventData._type)
            {
				selectionValues[index] = eventData._value._uint8Value;
			   
               ETG_TRACE_USR4(("MasterControlPlugin::onBeepOutputUpdate(), Value extracted is : %d", selectionValues[index]));
			   
			  
            }
         }
      }

	ETG_TRACE_USR4(("MasterControlPlugin::Calling onBeepOutputUpdate  sink :%d , value : %d ", selectionValues[0],selectionValues[1]));	
	SetBeepOutput(selectionValues[0],selectionValues[1]);
   }
	
}

void clMasterControlPlugin::SetBeepOutput(uint8_t sinkid,uint8_t value)
{
	ETG_TRACE_USR4(("MasterControlPlugin::SetBeepOutput -  Sinkid :%d, value - %d", sinkid , value));
	uint8_t BeepValue = 7;

        // Sinkid to be used when Cabin B is available- SOP2
	if( m_CMAEntityDetails.size() > 0 )
	{
		ETG_TRACE_USR4(("MasterControlPlugin::MediaAmplifer size  > 0 "));	
		for (uint8_t index = 0 ;index <  m_CMAEntityDetails.size() ; index++)
		{
			vSetControlUINT8(m_CMAEntityDetails.at(index).configuration ,BEEP_OUTPUT, BeepValue);
		}
	}
	else
	{
		ETG_TRACE_USR4(("No Amplifier connected "));
	}

}


/******************************************************************************
*NAME          : addPluginEventHandler
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get plugin event handler, part of evenbus mechanism
******************************************************************************/
std::unique_ptr<HandlerRegistration> clMasterControlPlugin::addPluginEventHandler()
{
    ETG_TRACE_USR4(("clMasterControlPlugin::addPluginEventHandler"));
    PluginEvtListenerBase<clMasterControlPlugin>* MasterPlugEventListener = PluginEvtListenerBase<clMasterControlPlugin>::getInstance();
    std::unique_ptr<HandlerRegistration> handle (EventBus::AddHandler<PluginEventBase<clMasterControlPlugin>>(MasterPlugEventListener));
    return std::move(handle);
}

/******************************************************************************
*NAME          : setPluginData
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Based on the key, the corresponding function is invoked
******************************************************************************/
void clMasterControlPlugin::setPluginData(std::string key, const boost::shared_ptr<EventDataUtility>& eValue)
{
    ETG_TRACE_USR4(("clMasterControlPlugin::setPluginData map size %d ", m_MemberFunctionMap.size()));
	ETG_TRACE_USR4(("clMasterControlPlugin::setPluginData string key %s", key.c_str()));

    void(clMasterControlPlugin::*FnPtr)(const boost::shared_ptr<EventDataUtility>&) = m_MemberFunctionMap.at(key);
    if(FnPtr!=NULL)
    {
       (*this.*FnPtr)(eValue);
    }
}

/******************************************************************************
*NAME          : setPluginListData
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Based on the key, the corresponding function is invoked
******************************************************************************/
void clMasterControlPlugin::setPluginListData(std::string key, const boost::shared_ptr<EventDataUtility>& eData,
        const boost::shared_ptr<EventListDataUtility>& eListData){
   ETG_TRACE_USR4(("clMasterControlPlugin::setPluginListData"));
    //To Do, based on whether camera_control_plugin needs list data or not
}

/******************************************************************************185
*NAME          : addPluginEventHandler
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get plugin event, part of evenbus mechanism
******************************************************************************/
std::unique_ptr<Event> clMasterControlPlugin::getPluginEvent(Object* sender, const std::string msg, boost::shared_ptr<EventDataUtility>evalue)
{
    ETG_TRACE_USR4(("clMasterControlPlugin::getPluginEvent"));
    std::unique_ptr<PluginEventBase<clMasterControlPlugin>> MasterPluginEvent(new PluginEventBase<clMasterControlPlugin>(sender,this,msg,evalue));
    return std::move(MasterPluginEvent);
}


/******************************************************************************
*NAME          : addPluginEventHandler
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get plugin list event, part of evenbus mechanism
******************************************************************************/
std::unique_ptr<Event> clMasterControlPlugin::getPluginListEvent(Object* sender, const std::string msg, boost::shared_ptr<EventDataUtility>eData,
                                                                const boost::shared_ptr<EventListDataUtility>& eListData)
{
    ETG_TRACE_USR4(("clMasterControlPlugin::getPluginListEvent"));
    std::unique_ptr<PluginEventBase<clMasterControlPlugin>> MasterPluginEvent(new PluginEventBase<clMasterControlPlugin>(sender,this,msg,eData,eListData));
    return std::move(MasterPluginEvent);
}




