/* ***************************************************************************************
* FILE:          clCameraPlugin.cpp
* SW-COMPONENT:  avdecc_appl_plugins
* DESCRIPTION:  clCameraPlugin.cpp is part of avdecc_appl_plugins library
* COPYRIGHT:  (c) 2020-21 Robert Bosch Car Multimedia GmbH
* HISTORY: 
* AUTHOR:  
* REVISION: 
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
/*****************************************************************
| includes
|----------------------------------------------------------------*/
#include <clCameraPlugin.h>
#include "plugin_trace.h"
#include "MediaCameraPort.h"
#include "clControlMapping.h"
#include "clPluginDataProvider.h"
#include "PluginDataType.h"

#include "PluginEvtListenerBase.h"
#include "PluginEventBase.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_PLUGIN_CAMERA
#define ETG_I_TRACE_CHANNEL      TR_TTFIS_PLUGIN
#define ETG_I_TTFIS_CMD_PREFIX   "CameraPlugin_"
#define ETG_I_FILE_PREFIX        clCameraPlugin::
#include "trcGenProj/Header/clCameraPlugin.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

clCameraPlugin* clCameraPlugin::poSelf = NULL;
clCameraPlugin::clCameraPlugin(){
	setMapFuncs();
	//Communucation Protocol Object creation logic
	m_pCommControl = new clCommunicationProtocol();
	if(m_pCommControl != NULL)
	{
		ETG_TRACE_USR4(("clCameraPlugin::clCommunicationProtocol is created"));
	}
	m_u16LastExecutedSetCommand = 0;
	m_CMPEntityDetails.clear();
	vecStCamDesc.clear();
	vecCamDataToHmi.clear();
	vecCamDataOnHmiRequest.clear();
	m_numberofCMPS = 0;
	//make pair
	mapCmpSrc_AvdeccValue.insert(pair<std::string, int>("CMP1_CAM1", 1));
	mapCmpSrc_AvdeccValue.insert(pair<std::string, int>("CMP1_CAM2", 2));
	mapCmpSrc_AvdeccValue.insert(pair<std::string, int>("CMP1_CAM3", 4));
	mapCmpSrc_AvdeccValue.insert(pair<std::string, int>("CMP1_CAM4", 8));
	mapCmpSrc_AvdeccValue.insert(pair<std::string, int>("CMP2_CAM1", 1));
	mapCmpSrc_AvdeccValue.insert(pair<std::string, int>("CMP2_CAM2", 2));
	mapCmpSrc_AvdeccValue.insert(pair<std::string, int>("CMP2_CAM3", 4));
	mapCmpSrc_AvdeccValue.insert(pair<std::string, int>("CMP2_CAM4", 8));
	mapCmpSrc_AvdeccValue.insert(pair<std::string, int>("CMP3_CAM1", 1));
	mapCmpSrc_AvdeccValue.insert(pair<std::string, int>("CMP3_CAM2", 2));
	mapCmpSrc_AvdeccValue.insert(pair<std::string, int>("CMP3_CAM3", 4));
	mapCmpSrc_AvdeccValue.insert(pair<std::string, int>("CMP3_CAM4", 8));
	
	_avRoutingParser = tclAvRoutingParser::pGetInstance();
	if(_avRoutingParser != NULL)
	{
		bool m_bParseState = _avRoutingParser->bParseXml();
		ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig _avRoutingParser m_bParseState %d", m_bParseState));
		//if(m_bParseState == true){readCameraConfig();}
	}
	bPrimaryDetected = false;
    vRegisterResponseCallbacks();
}

clCameraPlugin::~clCameraPlugin(){
	delete m_pCommControl;
	//clear all vectors
	m_CMPEntityDetails.clear();
	vecStCamDesc.clear();
	vecCamDataToHmi.clear();
	vecCamDataOnHmiRequest.clear();
	bPrimaryDetected = false;
}

clCameraPlugin* clCameraPlugin::getInstance()
{
	ETG_TRACE_USR4(("clCameraPlugin::getInstance"));
	if(poSelf == NULL)
	{
		ETG_TRACE_USR4(("Creating object clCameraPlugin"));
		poSelf = new clCameraPlugin();
	}
	ETG_TRACE_USR4(("Return object clCameraPlugin %d", poSelf));
	return poSelf;
	
}
void clCameraPlugin::deleteInstance()
{
	if(NULL != poSelf)
	{
		delete poSelf;
	}
}

void clCameraPlugin::vRegisterResponseCallbacks()
{
    //register for callbacks
    clControlMapping *p_controlmap = clControlMapping::getInstance();

    if(NULL != p_controlmap)
    {
        p_controlmap->setRespCallBack(CMP, CAMERA1STATUS, this, (FUNCPTR)(&clCameraPlugin::vHandleCamera1Status));
        p_controlmap->setRespCallBack(CMP, CAMERA2STATUS, this, (FUNCPTR)(&clCameraPlugin::vHandleCamera2Status));
        p_controlmap->setRespCallBack(CMP, CAMERA3STATUS, this, (FUNCPTR)(&clCameraPlugin::vHandleCamera3Status));
        p_controlmap->setRespCallBack(CMP, CAMERA4STATUS, this, (FUNCPTR)(&clCameraPlugin::vHandleCamera4Status));
        p_controlmap->setRespCallBack(CMP, CVBSSOURCE, this, (FUNCPTR)(&clCameraPlugin::vHandleCameraCVBSStatus));
        p_controlmap->setRespCallBack(CMP, SPLTSCRNCAMS, this, (FUNCPTR)(&clCameraPlugin::vHandleCameraSplitCamStatus));
        p_controlmap->setRespCallBack(CMP, IPCAMSTATUS, this, (FUNCPTR)(&clCameraPlugin::vHandleCameraIPCamStatus));
    }

}
void clCameraPlugin::vEventESReadComplete()
{
	//extract the details of the entities required
	std::vector<stEntityDesc> stEntityDetails;
	clPluginDataProvider *ptrMain = clPluginDataProvider::getInstance();
	stEntityDetails.clear();
	if(NULL != ptrMain)
	{
		bool found = ptrMain->getEndStationDetails(CMP, stEntityDetails);
		if(found == true)
		{
			ETG_TRACE_USR4(("clCameraPlugin vEventESReadComplete CMP found. Check the size and copy %d %d",m_CMPEntityDetails.size(), stEntityDetails.size()));
			if(m_CMPEntityDetails.size() != stEntityDetails.size())
			{
				vecStCamDesc.clear();     //First clear, then copy.
				m_CMPEntityDetails = stEntityDetails;
				vecStCamDesc = ptrMain->vecGetCamDesc();
				vValidateCamInfo();	
				ETG_TRACE_USR4(("clCameraPlugin vEventESReadComplete Desriptor copied to vector"));
			}
			
			for_each(m_CMPEntityDetails.begin(), m_CMPEntityDetails.end(), [](stEntityDesc Ent){
	        ETG_TRACE_USR4(("clCameraPlugin vEventESReadComplete u64EntityID is %d", Ent.u64EntityID));
	        ETG_TRACE_USR4(("clCameraPlugin vEventESReadComplete u32EntityIndex is %d", Ent.u32EntityIndex));
	        ETG_TRACE_USR4(("clCameraPlugin vEventESReadComplete strEndStationName is %s", Ent.strEndStationName.c_str()));
	        ETG_TRACE_USR4(("clCameraPlugin vEventESReadComplete strEntityName is %s", Ent.strEntityName.c_str()));
	        ETG_TRACE_USR4(("clCameraPlugin vEventESReadComplete end_station is %d", Ent.end_station));
	        ETG_TRACE_USR4(("clCameraPlugin vEventESReadComplete entity is %d", Ent.entity));
	        ETG_TRACE_USR4(("clCameraPlugin vEventESReadComplete configuration is %d", Ent.configuration));			
			});	
		}
		else			
		{
			ETG_TRACE_USR4(("clCameraPlugin vEventESReadComplete Disconnected. Clear all data"));
			m_CMPEntityDetails.clear();
			vecStCamDesc.clear();
			vecCamDataToHmi.clear();
			vecCamDataOnHmiRequest.clear();
			bPrimaryDetected = false;
		}
	}
	

}

tVoid clCameraPlugin::vValidateCamInfo()
{
	vecCamDataToHmi.clear();
	for(auto it1 : CameraList) // from xml
	{
		for(auto it2 : vecStCamDesc) // vec<pos, stentitydesc>
		{
			if(it1.pos == it2.position)
			{
				vecCamDataToHmi.push_back(it1);
				break;
			}
		}
	}
	for(auto it:vecCamDataToHmi)
	{
		ETG_TRACE_USR4(("clCameraPlugin::vecValidateCamInfo pos is %d", it.pos));
		ETG_TRACE_USR4(("clCameraPlugin::vecValidateCamInfo srcNam is %s", it.srcNam.c_str()));
		ETG_TRACE_USR4(("clCameraPlugin::vecValidateCamInfo camtextid is %d", it.camtextid));	
		if(it.pos == PRIMARY_CAMPORT)
		{
			bPrimaryDetected = true;
			//sendCameraListDataToHMI();	 // update to HMI the camera list only after Primary is available.
		}
		if(((it.pos == SECOND_CAMPORT) || (it.pos == THIRD_CAMPORT)) && (bPrimaryDetected == true))
		{
           	ETG_TRACE_USR4(("clCameraPlugin::Send update for 2nd or 3rd CMP after Primary already detected"));
			//sendCameraListDataToHMI();	 // update to HMI the if Primay was earlier detected and list sent
		}
	}
	
}
/*****************************************************************************************************
*NAME        : setMapFuncs
*DESCRIPTION : add fucntion pointers map
******************************************************************************************************/
void clCameraPlugin::setMapFuncs()
{
	ETG_TRACE_USR4(("clCameraPlugin::setMapFuncs"));
}

/******************************************************************************
*NAME          : addPluginEventHandler
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get plugin event handler, part of evenbus mechanism
******************************************************************************/
std::unique_ptr<HandlerRegistration> clCameraPlugin::addPluginEventHandler()
{
    ETG_TRACE_USR4(("clCameraPlugin::addPluginEventHandler"));
    PluginEvtListenerBase<clCameraPlugin>* cameraPlugEventListener = PluginEvtListenerBase<clCameraPlugin>::getInstance();
    std::unique_ptr<HandlerRegistration> handle (EventBus::AddHandler<PluginEventBase<clCameraPlugin>>(cameraPlugEventListener));
    return std::move(handle);
}

/******************************************************************************
*NAME          : setPluginData
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Based on the key, the corresponding function is invoked
******************************************************************************/
void clCameraPlugin::setPluginData(std::string key, const boost::shared_ptr<EventDataUtility>& eValue)
{
    ETG_TRACE_USR4(("clCameraPlugin::setPluginData map size %d ", m_MemberFunctionMap.size()));
	ETG_TRACE_USR4(("clCameraPlugin::setPluginData string key %s", key.c_str()));
	
	if(NULL != eValue.get())
	{
		uint16_t u16ControlVal = 0;
		uint16_t u16ControlId = 0;
		std::string strCameraSrc = "";
		std::vector<EventDataItem*> dataItems = eValue->getData();
		ETG_TRACE_USR4(("clCameraPlugin::setPluginData dataItems %d ", dataItems.size()));
		if(dataItems.size()>1)
		{
			EventDataItem* eventDataItemControl =  dataItems.at(0);  //17
			EventDataItem* eventDataItemControlVal =  dataItems.at(1); //2
			EventDataItem* eventDataItemControlString =  dataItems.at(2); //CMP2_CAM3
			
			ETG_TRACE_USR4(("clCameraPlugin::setPluginData eventDataItemControl %d ", dataItems.at(0)));
			ETG_TRACE_USR4(("clCameraPlugin::setPluginData eventDataItemControlVal %d ", dataItems.at(1)));
			ETG_TRACE_USR4(("clCameraPlugin::setPluginData eventDataItemControlVal %s ", dataItems.at(2)));
			
			if(NULL != eventDataItemControl)
			{
				EventDataItem::Data eventData = eventDataItemControl->getData();
				if(EventDataItem::UINT16 == eventData._type)
				{
					u16ControlId = eventData._value._uint16Value;
				}
			}
			if(NULL != eventDataItemControlVal)
			{
				EventDataItem::Data eventData = eventDataItemControlVal->getData();
				if(EventDataItem::UINT16 == eventData._type)
				{
					u16ControlVal = eventData._value._uint16Value;
				}
			}
			if(NULL != eventDataItemControlString)
			{
				EventDataItem::Data eventData = eventDataItemControlString->getData();
				if(EventDataItem::STRING == eventData._type)
				{
					strCameraSrc = eventData._value._stringValue->c_str();
				}
			}
			// here calls to fetch avdecc Value can be added.
			ETG_TRACE_USR4(("clCameraPlugin::setPluginData u16ControlId=%d  u16ControlVal=%d  ",u16ControlId,u16ControlVal));
			if(CAMLIST_STARTUP != u16ControlId)
			{
				vCreateSetControlCommand(u16ControlId,u16ControlVal, strCameraSrc);//call to setcontrol	
			}
			else
			{	//workaround fix for camera list reset, when HMI requests list.
				if(vecStCamDesc.size() > 0)
				{
					ETG_TRACE_USR4(("clCameraPlugin::vecStCamDesc size is > 0 "));
					vecCamDataOnHmiRequest.clear();
	                for(auto it1 : CameraList) // from xml
	                {
	                	for(auto it2 : vecStCamDesc) // vec<pos, stentitydesc>
	                	{
	                		if(it1.pos == it2.position)
	                		{
	                			vecCamDataOnHmiRequest.push_back(it1);
	                			break;
	                		}
	                	}
	                }
					
					//update to HMI the camera list
					//sendCameraListOnHMIRequest();	
	                // for(auto it:vecCamDataOnHmiRequest)
	                // {
	                	// ETG_TRACE_USR4(("clCameraPlugin::vecCamDataOnHmiRequest pos is %d", it.pos));
	                	// ETG_TRACE_USR4(("clCameraPlugin::vecCamDataOnHmiRequest srcNam is %s", it.srcNam.c_str()));
	                	// ETG_TRACE_USR4(("clCameraPlugin::vecCamDataOnHmiRequest camtextid is %d", it.camtextid));	
	                	// //if(it.pos == PRIMARY_CAMPORT)
	                	// {
	                		// sendCameraListOnHMIRequest();	 // update to HMI the camera list only after Primary is available.
	                	// }
	                // }
				}
			}
		}
	}	
}

/******************************************************************************
*NAME          : setPluginListData
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Based on the key, the corresponding function is invoked
******************************************************************************/
void clCameraPlugin::setPluginListData(std::string key, const boost::shared_ptr<EventDataUtility>& eData,
        const boost::shared_ptr<EventListDataUtility>& eListData){
   ETG_TRACE_USR4(("clCameraPlugin::setPluginListData"));
    //To Do, based on whether camera_control_plugin needs list data or not
}

/******************************************************************************185
*NAME          : addPluginEventHandler
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get plugin event, part of evenbus mechanism
******************************************************************************/
std::unique_ptr<Event> clCameraPlugin::getPluginEvent(Object* sender, const std::string msg, boost::shared_ptr<EventDataUtility>evalue)
{
    ETG_TRACE_USR4(("clCameraPlugin::getPluginEvent"));
    std::unique_ptr<PluginEventBase<clCameraPlugin>> cameraPluginEvent(new PluginEventBase<clCameraPlugin>(sender,this,msg,evalue));
    return std::move(cameraPluginEvent);
}


/******************************************************************************
*NAME          : addPluginEventHandler
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get plugin list event, part of evenbus mechanism
******************************************************************************/
std::unique_ptr<Event> clCameraPlugin::getPluginListEvent(Object* sender, const std::string msg, boost::shared_ptr<EventDataUtility>eData,
                                                                const boost::shared_ptr<EventListDataUtility>& eListData)
{
    ETG_TRACE_USR4(("clCameraPlugin::getPluginListEvent"));
    std::unique_ptr<PluginEventBase<clCameraPlugin>> cameraPluginEvent(new PluginEventBase<clCameraPlugin>(sender,this,msg,eData,eListData));
    return std::move(cameraPluginEvent);
}

/******************************************************************************
*NAME          : vCreateSetControlCommand
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Send the SetControl command to the Pluginbase by Communication control
******************************************************************************/
void clCameraPlugin::vCreateSetControlCommand(uint16_t u16ControlId,uint16_t u16CMPPosition, std::string strCameraSource) 
{
	ETG_TRACE_USR4(("clCameraPlugin::vCreateSetControlCommand  u16ControlId=%d  u16CMPPosition=%d",u16ControlId,u16CMPPosition));	
	m_u16LastExecutedSetCommand = u16ControlId;//Set value received from HMI
	ETG_TRACE_USR4(("clCameraPlugin::vCreateSetControlCommand  getAvdeccValue for camSrc =%s",strCameraSource.c_str()));
	
	int avdeccValue = 0;
	if((strCameraSource == strSplitCamSrcName[0]) || (strCameraSource == strSplitCamSrcName[1]) || (strCameraSource == strSplitCamSrcName[2]))
	{
		avdeccValue = getSplitAvdeccValue(u16CMPPosition, strCameraSource);
	}
	else
	{	
	getAvdeccValue(strCameraSource, avdeccValue);  //get the AVDECC value from mapping table returns 1/2/4/8
	}
	ETG_TRACE_USR4(("clCameraPlugin::vCreateSetControlCommand  AVDECC value is = %d",avdeccValue ));
	
	if(vecStCamDesc.size() > 0)
	{
		ControlDataElement controlElement;	
		for(auto itr : vecStCamDesc) // vec<pos, stentitydesc>   
		{
			if(itr.position == u16CMPPosition)
			{
				controlElement.configuration = itr.CamEntityDesc.configuration; 
				controlElement.u32ControlDescIdx = u16ControlId;
				controlElement.uint8Value = avdeccValue;
				
				if(m_pCommControl != NULL)   /// && (m_CMPEntityDetails.at(actualPosition).configuration != 0))
				{
					ETG_TRACE_USR4(("clCameraPlugin SetControl for selected camera"));
					m_pCommControl->setControl(controlElement);
				}		
				break;
			}
		}
	
		if(u16CMPPosition > 1)     // requests for cascadded camport. 3
		{
			for(int i=0; i < u16CMPPosition; ++i)   //
			{
				ETG_TRACE_USR4(("clCameraPlugin::Index for cascadded camport %d", u16CMPPosition));
				sendSetControlForCascadedCamera(u16ControlId ,u16CMPPosition-1);   // add validation checks
				--u16CMPPosition;
			}
		}
	}
	else	
	{
		ETG_TRACE_USR4(("clCameraPlugin::vecStCamDesc size is 0. No setControl sent."));
	}
}

void clCameraPlugin::sendSetControlForCascadedCamera(uint16_t u16ControlId, int position)
{
    ETG_TRACE_USR4(("clCameraPlugin sendSetControlForCascadedCamera %d", position));
	int avdeccCMPValue = CMP_CAM4_AVDECC_VALUE;
	if(vecStCamDesc.size() > 0)
	{	
		ControlDataElement controlElement;
		for(auto itr : vecStCamDesc) // vec<pos, stentitydesc>
		{
			if(itr.position == position)
			{
				controlElement.configuration = itr.CamEntityDesc.configuration; 
				controlElement.u32ControlDescIdx = u16ControlId;
				controlElement.uint8Value = avdeccCMPValue;
				
				if(m_pCommControl != NULL)   /// && (m_CMPEntityDetails.at(actualPosition).configuration != 0))
				{
					ETG_TRACE_USR4(("clCameraPlugin SetControl for cascaded camera"));
					m_pCommControl->setControl(controlElement);
				}
				break;
			}
		}
	}	
}

void clCameraPlugin::vHandleCamera1Status(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clCameraPlugin::vHandleCamera1Status"));
}

void clCameraPlugin::vHandleCamera2Status(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clCameraPlugin::vHandleCamera2Status"));
}
void clCameraPlugin::vHandleCamera3Status(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clCameraPlugin::vHandleCamera3Status"));
}
void clCameraPlugin::vHandleCamera4Status(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clCameraPlugin::vHandleCamera4Status"));
}
void clCameraPlugin::vHandleCameraCVBSStatus(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clCameraPlugin::vHandleCameraCVBSStatus"));
	boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
	
    uint16_t u16DescIndex = CVBSSOURCE;
	if(NULL != pluginData.get())
	{
		pluginData->addEventData(u16DescIndex);
		pluginData->addEventData(m_u16LastExecutedSetCommand);
	}

	RecvMsgFromPlugin(PLUGIN_NAME_CAMERA,CTRLBLOCK_NAME_CAMERA_CONNECT, pluginData);
}

void clCameraPlugin::vHandleCameraSplitCamStatus(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clCameraPlugin::vHandleCameraSplitCamStatus"));
	boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
	
    uint16_t u16DescIndex = SPLTSCRNCAMS;
	if(NULL != pluginData.get())
	{
		pluginData->addEventData(u16DescIndex);
		pluginData->addEventData(m_u16LastExecutedSetCommand);
	}

	RecvMsgFromPlugin(PLUGIN_NAME_CAMERA,CTRLBLOCK_NAME_CAMERA_SPLITVIEW_SEL, pluginData);
}

void clCameraPlugin::vHandleCameraIPCamStatus(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clCameraPlugin::vHandleCameraIPCamStatus"));
}


void clCameraPlugin::sendCameraListDataToHMI()
{

	ETG_TRACE_USR4(("clCameraPlugin::sendCameraListDataToHMI, Vector size: %d", vecCamDataToHmi.size()));
    for (auto it : vecCamDataToHmi)
    {
        ETG_TRACE_USR4(("clCameraPlugin::sendCameraListDataToHMI,%d -->  it.camtextid: %d, it.camSrcName: %s",
                        it.pos, it.camtextid, it.srcNam.c_str() ));
    }
    
	uint16_t tempData = 0;
    boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
    if(NULL != pluginData.get())
    {
        pluginData->addEventData(tempData);
    }

    boost::shared_ptr<EventListDataUtility> pluginListData(EventListDataUtility::newEventListDataUtility());
    if(NULL != pluginListData.get())
    {	
        pthread_mutex_lock(&m_thMutexLock);
		for(auto ct : vecCamDataToHmi)
		{
			EventDataUtility *eventData = EventDataUtility::newEventDataUtility();
			if(NULL != eventData){
			eventData->addEventData(ct.pos);
			eventData->addEventData(ct.camtextid);
			eventData->addEventData(ct.srcNam);
			}
			pluginListData->addEventListData(eventData);
			//delete eventData; //check on target to prevent data leak
		}
        pthread_mutex_unlock(&m_thMutexLock);
	 RecvListMsgFromPlugin(PLUGIN_NAME_CAMERA, CTRLBLOCK_NAME_CAMERA_CAMERA_LIST, pluginData, pluginListData);	
    }  

}

void clCameraPlugin::getAvdeccValue(std::string camSrc, int &avdeccValue)
{
   ETG_TRACE_USR4(("clCameraPlugin::getAvdeccValue for camera source %s", camSrc.c_str()));
   if (mapCmpSrc_AvdeccValue.size() != 0)
   {
      std::map<std::string, int>::iterator itr = mapCmpSrc_AvdeccValue.find(camSrc);
	  if (itr != mapCmpSrc_AvdeccValue.end())
	  {
		avdeccValue = itr->second;
      }
	  else
	  {
	  }
   }
   ETG_TRACE_USR4(("clCameraPlugin::getAvdeccValue for selected source is %d", avdeccValue));
}
 
void clCameraPlugin::readCameraConfig()
{
 ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig"));
 
	if(_avRoutingParser != NULL)
	{  
		ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig _avRoutingParser is not null"));
		// bool m_bParseState = _avRoutingParser->bParseXml();
		// ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig _avRoutingParser m_bParseState %d", m_bParseState));
		m_caminfo.clear();
		CameraList.clear();
		m_caminfo = _avRoutingParser->vecGetCamSrc();
	    m_camSplitData = _avRoutingParser->vecGetSplitCamSrc();
	    m_splitCaminfo = _avRoutingParser->vecGetSplitCamInfo();
		ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig m_splitCaminfo size %d", m_splitCaminfo.size()));
		if(m_splitCaminfo.size() > 0)
		{
			for(tU8 u8Index= 0 ; u8Index < m_splitCaminfo.size();u8Index++ )
			{	
				ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig Split Pos is %d", m_splitCaminfo[u8Index].pos));
				ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig Split name is %s", m_splitCaminfo[u8Index].name.c_str()));
				ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig Split view is %s",    m_splitCaminfo[u8Index].view.c_str()));			
				ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig Split source_1 is %s", m_splitCaminfo[u8Index].source_1.c_str()));
				ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig Split source_2 is %s", m_splitCaminfo[u8Index].source_2.c_str()));
				ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig Split srcName is %s", m_splitCaminfo[u8Index].srcNam.c_str()));
				ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig Split camTextid is %d", m_splitCaminfo[u8Index].camtextid));
			}
		}
	   int numberOfCMPs = _avRoutingParser->vectorGetEntityID("CMP").size();
	   ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig number of camports is %d", numberOfCMPs));		
	   m_numberofCMPS = numberOfCMPs;
	   vPrepareAvdeccValuesForSplitCameras();   // this function will fill the avdecc values for each of the split cameras.
	   if(m_camSplitData.size() > 0)
	   {
	    for(tU8 u8Index= 0 ; u8Index < m_camSplitData.size();u8Index++ )
		{
			ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig split Pos is %d", m_camSplitData[u8Index].pos));
			ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig split srcName is %s", m_camSplitData[u8Index].srcNam.c_str()));
			ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig split camTextID is %d [%d]",u8Index, m_camSplitData[u8Index].camtextid));			
		}
	   }
	   
	   if(m_caminfo.size() > 0)
	   {
		// Print for tracing 
	    for(tU8 u8Index= 0 ; u8Index < m_caminfo.size();u8Index++ )
		{
			ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig Pos is %d", m_caminfo[u8Index].pos));
			ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig srcName is %s", m_caminfo[u8Index].srcNam.c_str()));
			ETG_TRACE_USR4(("clCameraPlugin:readCameraConfig camTextID is %d [%d]",u8Index, m_caminfo[u8Index].camtextid));			
		}
		

		tU8 positionCMP = 0;
		tU8 u8Index = 0;

		switch(numberOfCMPs)
		{
			case ONE:
					positionCMP = getCamportIndex(1); fillDataInVector(positionCMP, 4);   // Pick 1st 4 analogue camera and add IP camera.
			break;
			
			case TWO:
					positionCMP = getCamportIndex(1); fillDataInVector(positionCMP, 3);     // Pick 1st 3 analogue camera.
					positionCMP = getCamportIndex(2); fillDataInVector(positionCMP, 4);   // Pick  4 analogue camera and add IP camera.
			break;
			
			case THREE:
					positionCMP = getCamportIndex(1); fillDataInVector(positionCMP, 3);      // Pick 1st 3 analogue camera.
					positionCMP = getCamportIndex(2); fillDataInVector(positionCMP, 3);  // Pick  3 analogue camera
					positionCMP = getCamportIndex(3); fillDataInVector(positionCMP, 4);  // Pick  4 analogue camera and add IP camera.
			
			break;
			default:
			break;
		}
		//add IP camera at the end of the list. Add the IP cam from the 1st camport only.
		//get the position of primay CMP, and add 4 to get IP CAM index.
		// positionCMP = getCamportIndex(1);
		// CL = m_caminfo[positionCMP + 4];					// 4th item in list belongs to IP camera.
		// CameraList.push_back(CL);
		
		vCheckAndAddIPCamera(numberOfCMPs);
		vCheckAndAddSplitCamera(numberOfCMPs);
	   }
	}
	vPrintCamInfo();
}

void clCameraPlugin::vCheckAndAddIPCamera(int numberOfCMPs)
{
	ETG_TRACE_USR4(("clCameraPlugin:vCheckAndAddIPCamera number of camports is %d", numberOfCMPs));		
	tU8 positionCMP = 0;
	
	switch(numberOfCMPs)
	{
		case ONE:
				// positionCMP = getCamportIndex(1);
				// CL = m_caminfo[positionCMP + 4];					// 4th item in list belongs to IP camera.
				// CameraList.push_back(CL);
				
				positionCMP = getIPCameraIndexFromCamport("CMP1_IPCAM");
				CL = m_caminfo[positionCMP];
				CameraList.push_back(CL);
				
		break;
		
		case TWO:
				// positionCMP = getCamportIndex(1);
				// CL = m_caminfo[positionCMP + 4];					// 4th item in list belongs to IP camera.
				// CameraList.push_back(CL);
				// positionCMP = getCamportIndex(2);
				// CL = m_caminfo[positionCMP + 4];					// 4th item in list belongs to IP camera.
				// CameraList.push_back(CL);
				
				positionCMP = getIPCameraIndexFromCamport("CMP1_IPCAM");
				CL = m_caminfo[positionCMP];
				CameraList.push_back(CL);
				positionCMP = getIPCameraIndexFromCamport("CMP2_IPCAM");
				CL = m_caminfo[positionCMP];
				CameraList.push_back(CL);
				
		break;
		case THREE:
				// positionCMP = getCamportIndex(1);
				// CL = m_caminfo[positionCMP + 4];					// 4th item in list belongs to IP camera.	
				// CameraList.push_back(CL);
			    // positionCMP = getCamportIndex(2);
				// CL = m_caminfo[positionCMP + 4];					// 4th item in list belongs to IP camera.
				// CameraList.push_back(CL);
			    // positionCMP = getCamportIndex(3);
				// CL = m_caminfo[positionCMP + 4];					// 4th item in list belongs to IP camera.
				// CameraList.push_back(CL);
				
				positionCMP = getIPCameraIndexFromCamport("CMP1_IPCAM");
				CL = m_caminfo[positionCMP];
				CameraList.push_back(CL);
				positionCMP = getIPCameraIndexFromCamport("CMP2_IPCAM");
				CL = m_caminfo[positionCMP];
				CameraList.push_back(CL);
				positionCMP = getIPCameraIndexFromCamport("CMP3_IPCAM");
				CL = m_caminfo[positionCMP];
				CameraList.push_back(CL);
		break;
		default:
		break;
	}
}

void clCameraPlugin::fillDataInVector(int indexCMP, int loopSize)
{
	int u8Index = indexCMP;
	for(tU8 loop = 0 ; loop < loopSize; ++loop )  
	{
		CL = m_caminfo[u8Index];   
		CameraList.push_back(CL);
		++u8Index;
	}
} 
 
 
 
  int clCameraPlugin::getCamportIndex(int position){

	ETG_TRACE_USR4(("clCameraPlugin:getCamportIndex %d", position));
	int positionIndex = 0;
	for (int index = 0; index < m_caminfo.size(); index++)
	{
		if(m_caminfo[index].pos == position )
		{
			positionIndex = index;
			break;
		}
	}
	ETG_TRACE_USR4(("clCameraPlugin:getCamportIndex index %d", positionIndex));
    return positionIndex; 
} 


 int clCameraPlugin::getIPCameraIndexFromCamport(std::string ipcamstr){

	ETG_TRACE_USR4(("clCameraPlugin:getIPCameraIndexFromCamport find %s", ipcamstr));
	int positionIndex = 0;
	for (int index = 0; index < m_caminfo.size(); index++)
	{
		if(m_caminfo[index].srcNam == ipcamstr )
		{
			positionIndex = index;
			break;
		}
	}
	ETG_TRACE_USR4(("clCameraPlugin:getIPCameraIndexFromCamport found at %d", positionIndex));
    return positionIndex; 
			
}

void clCameraPlugin::sendCameraListOnHMIRequest()
{
	ETG_TRACE_USR4(("clCameraPlugin::sendCameraListOnHMIRequest, Vector size: %d", vecCamDataOnHmiRequest.size()));

    for (auto it : vecCamDataOnHmiRequest)
    {
        ETG_TRACE_USR4(("clCameraPlugin::sendCameraListOnHMIRequest,%d -->  it.camtextid: %d, it.camSrcName: %s",
                        it.pos, it.camtextid, it.srcNam.c_str() ));
    }
    
	uint16_t tempData = 0;
    boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
    if(NULL != pluginData.get())
    {
        pluginData->addEventData(tempData);
    }

    boost::shared_ptr<EventListDataUtility> pluginListData(EventListDataUtility::newEventListDataUtility());
    if(NULL != pluginListData.get())
    {	
        pthread_mutex_lock(&m_thMutexLock);
		for(auto ct : vecCamDataOnHmiRequest)
		{
			EventDataUtility *eventData = EventDataUtility::newEventDataUtility();
			if(NULL != eventData){
			eventData->addEventData(ct.pos);
			eventData->addEventData(ct.camtextid);
			eventData->addEventData(ct.srcNam);
			}
			pluginListData->addEventListData(eventData);
			//delete eventData; //check on target to prevent data leak
		}
        pthread_mutex_unlock(&m_thMutexLock);
		
	 RecvListMsgFromPlugin(PLUGIN_NAME_CAMERA, CTRLBLOCK_NAME_CAMERA_CAMERA_LIST, pluginData, pluginListData);	
    }  
	
}


void clCameraPlugin::vCheckAndAddSplitCamera(int numberOfCMPs)
{
	ETG_TRACE_USR4(("clCameraPlugin:vCheckAndAddSplitCamera number of camports is %d", numberOfCMPs));		
	tU8 positionCMP = 0;
	switch(numberOfCMPs)
		{
			case ONE:
					positionCMP = getCamportIndex(1); fillSplitDataInVector(positionCMP+6, 3);   // Pick 3 split config from camport 1
			break;
			case TWO:
					positionCMP = getCamportIndex(1); fillSplitDataInVector(positionCMP+6, 3);   // Pick 3 split config from camport 1
					positionCMP = getCamportIndex(2); fillSplitDataInVector(positionCMP+6, 3);   // Pick 3 split config from camport 2
			break;
			case THREE:
					positionCMP = getCamportIndex(1); fillSplitDataInVector(positionCMP+6, 3);  // Pick  3 split config from camport 1
					positionCMP = getCamportIndex(2); fillSplitDataInVector(positionCMP+6, 3);  // Pick  3 split config from camport 2
					positionCMP = getCamportIndex(3); fillSplitDataInVector(positionCMP+6, 3);  // Pick  3 split config from camport 3
			break;
			default:
			break;
		}
}
tVoid clCameraPlugin::vPrintCamInfo()
{
	ETG_TRACE_USR4(("clCameraPlugin::vPrintCamInfo -------------> "));
	for(auto it : CameraList) // from xml
	{
		ETG_TRACE_USR4(("clCameraPlugin::vPrintCamInfo pos is %d", it.pos));
		ETG_TRACE_USR4(("clCameraPlugin::vPrintCamInfo srcNam is %s", it.srcNam.c_str()));
		ETG_TRACE_USR4(("clCameraPlugin::vPrintCamInfo camtextid is %d", it.camtextid));	
	}
	ETG_TRACE_USR4(("clCameraPlugin::vPrintCamInfo <------------- "));
}
void clCameraPlugin::fillSplitDataInVector(int indexCMP, int loopSize)
{
	tCamData CD;
	int u8Index = indexCMP;
	for(tU8 loop = 0 ; loop < loopSize; ++loop )  
	{
		CD.pos = m_camSplitData[u8Index].pos; 
		CD.srcNam = m_camSplitData[u8Index].srcNam; 
		CD.camtextid = m_camSplitData[u8Index].camtextid; 
		CameraList.push_back(CD);
		++u8Index;
	}
} 
int clCameraPlugin::getSplitAvdeccValue(int position, std::string cameraSrc)
{
	int splitval = 0;
	for(tU8 u8Index = 0 ; u8Index < m_splitAvdeccinfo.size();u8Index++ )
	{
		if(( m_splitAvdeccinfo.at(u8Index).pos == position) && (m_splitAvdeccinfo.at(u8Index).name == cameraSrc))
		{
			splitval = m_splitAvdeccinfo.at(u8Index).avdeccValue;
		}
	}
	return splitval;
}
 void clCameraPlugin::vPrepareAvdeccValuesForSplitCameras()
 {
	   tCamPortAvdeccSplitInfo splitInfo;
	   int avdeccValueSrc1 = 0;
	   int avdeccValueSrc2 = 0;
	   int splitAvdeccValue = 0;
	   for(int l = 1; l <= m_numberofCMPS; l++)
	   {
		   for(int m = 0; m < 3; m++)  // fpr now hardcoded to 3. will be changed to array size.
		   {
			   std::vector<std::string> vecSrNamesSplit =  _avRoutingParser->getSourceNamesForSplitView(strSplitCamSrcName[m],l );
			   if(vecSrNamesSplit.size() > 0)
			   {
					for(tU8 u8Index= 0 ; u8Index < vecSrNamesSplit.size();u8Index++ )
					{	
						ETG_TRACE_USR4(("clCameraPlugin:vPrepareAvdeccValuesForSplitCameras Split name src is %s", vecSrNamesSplit.at(u8Index).c_str()));
						getAvdeccValue(vecSrNamesSplit.at(u8Index).c_str(), avdeccValueSrc1); 
					}
					getAvdeccValue(vecSrNamesSplit.at(0).c_str(), avdeccValueSrc1); 
					getAvdeccValue(vecSrNamesSplit.at(1).c_str(), avdeccValueSrc2); 
					splitAvdeccValue = avdeccValueSrc1 | avdeccValueSrc2;
					ETG_TRACE_USR4(("clCameraPlugin:vPrepareAvdeccValuesForSplitCameras AV is %d | %d -> %d", avdeccValueSrc1,avdeccValueSrc2,splitAvdeccValue));
				}
				splitInfo.pos = l;
				splitInfo.name = strSplitCamSrcName[m];
				splitInfo.avdeccValue = splitAvdeccValue;
				m_splitAvdeccinfo.push_back(splitInfo);
		   }
	   }
	    if(m_splitAvdeccinfo.size() > 0)
		{
			for(tU8 u8Index= 0 ; u8Index < m_splitAvdeccinfo.size();u8Index++ )
			{	
				ETG_TRACE_USR4(("clCameraPlugin:vPrepareAvdeccValuesForSplitCameras Split avdecc value is %d", m_splitAvdeccinfo.at(u8Index).pos));
				ETG_TRACE_USR4(("clCameraPlugin:vPrepareAvdeccValuesForSplitCameras Split avdecc value is %s", m_splitAvdeccinfo.at(u8Index).name.c_str()));
				ETG_TRACE_USR4(("clCameraPlugin:vPrepareAvdeccValuesForSplitCameras Split avdecc value is %d", m_splitAvdeccinfo.at(u8Index).avdeccValue));
			}
		} 
 }
ETG_I_CMD_DEFINE((TraceCmd_sendSetControlCommand, "setControlCommand %d %d %d", int, int, int))
void clCameraPlugin::TraceCmd_sendSetControlCommand(uint16_t u16ControlId,uint16_t u16ControlVal, uint16_t position)
{
   ETG_TRACE_USR4(("clCameraPlugin::fillTestDataToHMI"));
   //poSelf->fillTestDataToHMI();
}

