/* ***************************************************************************************
 * FILE:          clAsfLCMServiceAdaptor.cpp
 * SW-COMPONENT:  HMI-BASE
 * DESCRIPTION:  clAsfLCMServiceAdaptor.cpp is part of NanoMsgMasterApp
 * COPYRIGHT:  (c) 2020-21 Robert Bosch Car Multimedia GmbH
 * Author: Supriya Seshadri
 * Date: 03.02.2020
 * Revision: Supriya S 18.03.2020
 *		Removed ACKevent , Extended all interfaces to incorporate addition of URL-IP
 * Revision: Supriya S 01/04/2020
 *        Added NULL Checks in requests and responses
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */

#include "AsfLCMServiceAdaptor.h"
#include "plugin_trace.h"
//#include "clNanoMsgCoreThread.h"
#include <stdio.h>
#include "NanoMsgReceiver.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_NANOMSGMASTERASF
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_NAVRES6
#include "trcGenProj/Header/AsfLCMServiceAdaptor.cpp.trc.h"
#endif

using namespace ::asf::core;
// ========================================================================
namespace NanoMsg_MasterApp{
	namespace NanoMsgServer{

#define LCM_PORT_NAME "NanoMsgLCMPort"

	/* Pointer to the singleton class object*/
	clAsfLCMServiceAdaptor* clAsfLCMServiceAdaptor::p_AsfLCMInstance = NULL;

/************************************************************************
*FUNCTION: 		 clAsfLCMServiceAdaptor()
*DESCRIPTION: 	 Constructor of the class
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
clAsfLCMServiceAdaptor::clAsfLCMServiceAdaptor() :
	NanoMsgLCMInterfaceStub(LCM_PORT_NAME)
{
	ETG_TRACE_USR4(("clAsfLCMServiceAdaptor() constructor"));
	init();
	m_pcoreSenderThread = clSenderThread::getInstance();
}



/************************************************************************
*FUNCTION: 		init()
*DESCRIPTION: 	Initialization of Callbacks from Core is handled here
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
bool clAsfLCMServiceAdaptor::init()
{
	ETG_TRACE_USR4(("clAsfLCMServiceAdaptor init()"));

	clMsgIDMapping *_instance;
	_instance = clMsgIDMapping::getInstance();

	if (NULL != _instance)
	{
		ETG_TRACE_USR4(("clAsfLCMServiceAdaptor Initializing the Callbacks from Core..."));

		////Please look into document the CIS - system management protocol message definition 
		////and CIS Intercom message definition for the Below Mapping


		_instance->setCallback(0x0004, (FUNCPTR)&clAsfLCMServiceAdaptor::sendLCMRspSetPowerModeSignal);
		_instance->setCallback(0x0006, (FUNCPTR)&clAsfLCMServiceAdaptor::sendLCMRspGetPowerModeSignal);
		_instance->setCallback(0x0007, (FUNCPTR)&clAsfLCMServiceAdaptor::sendLCMEventPowerModeChange);
		_instance->setCallback(0x0009, (FUNCPTR)&clAsfLCMServiceAdaptor::sendLCMRspGetWakeupReason);
		_instance->setCallback(0x00011, (FUNCPTR)&clAsfLCMServiceAdaptor::sendLCMRspShutdown);
		_instance->setCallback(0x00019, (FUNCPTR)&clAsfLCMServiceAdaptor::sendRspSetChainMode);
                 _instance->setCallback(0x00031, (FUNCPTR)&clAsfLCMServiceAdaptor::sendLCMRspHealthCareSignal);
                _instance->setCallback(0x00032, (FUNCPTR)&clAsfLCMServiceAdaptor::sendLCMEventDtcOccuranceSignal);
                return true;
	}
	return false;
}


/************************************************************************
*FUNCTION: 		getInstance()
*DESCRIPTION: 	GetInstance will return the pointer to the clAsfLCMServiceAdaptor
*				Singleton Object
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
clAsfLCMServiceAdaptor* clAsfLCMServiceAdaptor::getInstance()
{

	ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::getInstance()"));
	if (p_AsfLCMInstance == NULL)
	{
		ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::getInstance Creating Instance"));
		p_AsfLCMInstance = new clAsfLCMServiceAdaptor();
	}
	return p_AsfLCMInstance;
}


/************************************************************************
*FUNCTION: 		vDeleteInstance()
*DESCRIPTION: 	vDeleteInstance will delete clAsfLCMServiceAdaptor
*				Singleton Object
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfLCMServiceAdaptor::vDeleteInstance()
{
	if (p_AsfLCMInstance != NULL)
	{
		ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::vDeleteInstance()"));
		delete p_AsfLCMInstance;
		p_AsfLCMInstance = NULL;
	}
}

/************************************************************************
*FUNCTION: 		~clAsfLCMServiceAdaptor()
*DESCRIPTION: 	clAsfLCMServiceAdaptor class Destructor
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
clAsfLCMServiceAdaptor::~clAsfLCMServiceAdaptor()
{
}

/************************************************************************
*FUNCTION: 		sendLCMRspSetPowerModeSignal(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the Sig_RspSetPowerModeSignal
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfLCMServiceAdaptor::sendLCMRspSetPowerModeSignal(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendLCMRspSetPowerModeSignal"));
	if(NULL == data)
	{
		ETG_TRACE_USR4(("Data Received is NULL"));
		return;
	}
	//extract the data	

	uint16 msg_ID = data->MsgID;
	enDevID dev_ID = static_cast<enDevID>(data->DevID);
	
	uint16 status = data->data[0];
	uint8 dev_Nr = data->DevNr;
	//uint16 msg_ID = 0;
	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		dev_ID, dev_Nr, msg_ID));
	ETG_TRACE_USR4(("Recieved Message URL-IP is %s", data->strURL.c_str()));

	sendSig_RspSetPowerModeSignal(status, dev_ID, dev_Nr, msg_ID, data->strURL);
}

/************************************************************************
*FUNCTION: 		sendLCMRspGetPowerModeSignal(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the Sig_RspGetPowerModeSignal
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfLCMServiceAdaptor::sendLCMRspGetPowerModeSignal(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendLCMRspGetPowerModeSignal"));
	if(NULL == data)
	{
		ETG_TRACE_USR4(("Data Received is NULL"));
		return;
	}
	//extract the data	

	enDevID dev_ID = static_cast<enDevID>(data->DevID);
	//0xff ff ff ff
	uint16_t Data = ((data->data[0]) | (data->data[1] << 8));
	Data = ntohs(Data);
	entPowerMode powerMode = static_cast<entPowerMode>(Data);
	uint8 dev_Nr = data->DevNr;
	uint16 msg_ID = data->MsgID;

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		dev_ID, dev_Nr, msg_ID));

	ETG_TRACE_USR4(("PowerMode value in stNanoMsgData.data[0] = %d, stNanoMsgData.data[1] = %d",
		data->data[0], data->data[1]));

	ETG_TRACE_USR4(("PowerMode value is %d", powerMode));
	ETG_TRACE_USR4(("Recieved Message URL-IP is %s", data->strURL.c_str()));
	sendSig_RspGetPowerModeSignal(powerMode, dev_ID, dev_Nr, msg_ID, data->strURL);
}

/************************************************************************
*FUNCTION: 		sendLCMEventPowerModeChange(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the Sig_EventPowerModeChange
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfLCMServiceAdaptor::sendLCMEventPowerModeChange(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendLCMEventPowerModeChange"));
	if(NULL == data)
	{
		ETG_TRACE_USR4(("Data Received is NULL"));
		return;
	}
	//extract the data	

	enDevID dev_ID = static_cast<enDevID>(data->DevID);
	//0xff ff ff ff
	uint16_t Data = ((data->data[0]) | (data->data[1] << 8));
	Data = ntohs(Data);
	entPowerMode powerMode = static_cast<entPowerMode>(Data);
	uint8 dev_Nr = data->DevNr;
	uint16 msg_ID = data->MsgID;

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		dev_ID, dev_Nr, msg_ID));

	ETG_TRACE_USR4(("PowerMode value in stNanoMsgData.data[0] = %d, stNanoMsgData.data[1] = %d",
		data->data[0], data->data[1]));

	ETG_TRACE_USR4(("PowerMode value is %d", powerMode));
	ETG_TRACE_USR4(("Recieved Message URL-IP is %s", data->strURL.c_str()));
	sendSig_EventPowerModeChangeSignal(powerMode, dev_ID, dev_Nr, msg_ID, data->strURL);
}



/************************************************************************
*FUNCTION: 		sendLCMRspGetWakeupReason(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the Sig_RspGetWakeupReason
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfLCMServiceAdaptor::sendLCMRspGetWakeupReason(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendLCMRspGetWakeupReason"));
	if(NULL == data)
	{
		ETG_TRACE_USR4(("Data Received is NULL"));
		return;
	}
	//extract the data
	enDevID dev_ID = static_cast<enDevID>(data->DevID);
	//0xff ff ff ff
	uint32_t Data = ((data->data[0]) | (data->data[1] << 8) | (data->data[2] << 16) | (data->data[3] << 24));
	//Change the Endianess of the received data
	Data = ntohl(Data);
	entWakeupReason enWakeupreason = static_cast<entWakeupReason>(Data);
	
	uint8 dev_Nr = data->DevNr;
	uint16 msg_ID = data->MsgID;

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		dev_ID, dev_Nr, msg_ID));

	ETG_TRACE_USR4(("Wakeupreason value in stNanoMsgData.data[0] = %d, stNanoMsgData.data[1] = %d",
		data->data[0], data->data[1]));

	ETG_TRACE_USR4(("Wakeupreason value is %d", enWakeupreason));
	ETG_TRACE_USR4(("Recieved Message URL-IP is %s", data->strURL.c_str()));
	sendSig_RspGetWakeupReasonSignal(enWakeupreason, dev_ID, dev_Nr, msg_ID, data->strURL);
}


/************************************************************************
*FUNCTION: 		sendLCMRspShutdown(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the Sig_RspShutdown
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfLCMServiceAdaptor::sendLCMRspShutdown(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendLCMRspShutdown"));
		if(NULL == data)
	{
		ETG_TRACE_USR4(("Data Received is NULL"));
		return;
	}
	//Extract the data
	
	enDevID dev_ID = static_cast<enDevID>(data->DevID);

	uint16 status = data->data[0];

	uint8 dev_Nr = data->DevNr;
	uint16 msg_ID = data->MsgID;

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		dev_ID, dev_Nr, msg_ID));
	ETG_TRACE_USR4(("Recieved Message URL-IP is %s", data->strURL.c_str()));
	sendSig_RspShutdownSignal(status, dev_ID, dev_Nr, msg_ID, data->strURL);
}

/************************************************************************
*FUNCTION: 		sendRspSetChainMode(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the RspSetChainMode
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfLCMServiceAdaptor::sendRspSetChainMode(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendRspSetChainMode"));
	if(NULL == data)
	{
		ETG_TRACE_USR4(("Data Received is NULL"));
		return;
	}
	//Extract the data
	enDevID dev_ID = static_cast<enDevID>(data->DevID);
	//0xff ff ff ff
	uint16_t Data = ((data->data[0]));
	eniRet eniRetvalue = static_cast<eniRet>(Data);

	uint8 dev_Nr = data->DevNr;
	uint16 msg_ID = data->MsgID;

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		dev_ID, dev_Nr, msg_ID));

	ETG_TRACE_USR4(("return value in stNanoMsgData.data[0] is %d",
		data->data[0]));

	ETG_TRACE_USR4(("Return Value being sent is %d", eniRetvalue));
	ETG_TRACE_USR4(("Recieved Message URL-IP is %s", data->strURL.c_str()));
	sendSig_RspSetChainModeSignal(eniRetvalue, dev_ID, dev_Nr, msg_ID, data->strURL);
}

/************************************************************************
*FUNCTION: 		sendLCMRspHealthCareSignal(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the RspHealthCare
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	DRM2KOR	05/03/2021
*************************************************************************/
void clAsfLCMServiceAdaptor::sendLCMRspHealthCareSignal(stNanoMsgData *data)
{
   ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMRspHealthCareSignal()"));
   if(NULL == data)
   {
      ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMRspHealthCareSignal(),Data Received is NULL"));
      return;
   }
 
  //extract the data	
   // uint16_t dtcTableSize = 0;

   enDevID dev_ID = static_cast<enDevID>(data->DevID);
   uint16_t Data = ((data->data[0]) | (data->data[1] << 8));
   Data = ntohs(Data);
   entPowerMode powerMode = static_cast<entPowerMode>(Data);

  // std ::vector<uint8_t> DtcTableData;

   // datalength is length of DevID + DevNr + message type + data
   uint16_t dataLength = data -> DataLength;
   ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMRspHealthCareSignal() dataLength is :%d",dataLength));
   // dtcTable array size would be dataLength- 1(DevID) -1(DevNr ) - 2(message type)- 2(powerMode)
   
 // dtcTableSize= dataLength; //dataLength - DATA_HEADER_LENGTH_WITHPOWERMODE;
    
 /* for(int index = 2; index < dtcTableSize; index++)
  {  
     ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMRspHealthCareSignal() DtcTable data to be added is :%d",data->data[index]));
     DtcTableData.push_back(data->data[index]); 
  } */
 // }
    /*
    for (auto itr = DtcTableData.begin();itr!= DtcTableData.end(); ++itr)
    {
    ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMRspHealthCareSignal() DTCTable data at buffer [%d] is :%d",itr, *itr);
    } */
   uint8 dev_Nr = data->DevNr;
   uint16 msg_ID = data->MsgID;

   ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMRspHealthCareSignal(), Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		dev_ID, dev_Nr, msg_ID));
        
   ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMRspHealthCareSignal(), PowerMode value in stNanoMsgData.data[0] = %d, stNanoMsgData.data[1] = %d",
		data->data[0], data->data[1]));

   ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMRspHealthCareSignal(), PowerMode value is %d", powerMode));
   ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMRspHealthCareSignal(), Recieved Message URL-IP is %s", data->strURL.c_str()));
   sendSig_RspHealthCareSignal(powerMode, dev_ID, dev_Nr, msg_ID, data->strURL);
}


/************************************************************************
*FUNCTION: 		sendLCMEventDtcOccuranceSignal(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the EventDtcOccurance
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	NDW5kor	27/10/2021
*************************************************************************/
void clAsfLCMServiceAdaptor::sendLCMEventDtcOccuranceSignal(stNanoMsgData *data)
{
   ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMEventDtcOccuranceSignal()"));
   if(NULL == data)
   {
      ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMEventDtcOccuranceSignal(),Data Received is NULL"));
      return;
   }
 
  //extract the data	
    uint16_t dtcTableSize = 0;
	
   enDevID dev_ID = static_cast<enDevID>(data->DevID);
   ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::enDevID dev_ID is :%d",dev_ID));
      
   std ::vector<uint8_t> DtcTableData;
   // datalength is length of DevID + DevNr + message type + data
   uint16_t dataLength = data -> DataLength;
   ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMEventDtcOccuranceSignal() dataLength is :%d",dataLength));
   // dtcTable array size would be dataLength- 1(DevID) -1(DevNr ) - 2(message type)
   
  dtcTableSize= dataLength; 
  ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMEventDtcOccuranceSignal() dtcTableSize is :%d",dtcTableSize));
    
  for(int index = 0; index < dtcTableSize; index++)
  {  
     ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMEventDtcOccuranceSignal() DtcTable data to be added is :%d",data->data[index]));
     DtcTableData.push_back(data->data[index]); 
  } 

   uint8 dev_Nr = data->DevNr;
   uint16 msg_ID = data->MsgID;

   ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMEventDtcOccuranceSignal(), Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		dev_ID, dev_Nr, msg_ID));
        
   ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMEventDtcOccuranceSignal(), value in stNanoMsgData.data[0] = %d, stNanoMsgData.data[1] = %d",
		data->data[0], data->data[1]));

   ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::sendLCMEventDtcOccuranceSignal(), Recieved Message URL-IP is %s", data->strURL.c_str()));
   sendSig_EventDtcOccuranceSignal(DtcTableData, dev_ID, dev_Nr, msg_ID, data->strURL);
}
/************************************************************************
*FUNCTION: 		onReqSetPowerModeRequest(const ::boost::shared_ptr< ReqSetPowerModeRequest >& request)
*DESCRIPTION: 	The method onReqSetPowerMode() is called by ASF when a client invokes
* the method 'ReqSetPowerMode'.
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfLCMServiceAdaptor::onReqSetPowerModeRequest(const ::boost::shared_ptr< ReqSetPowerModeRequest >& request)
{
	ETG_TRACE_USR4(("onReqSetPowerModeRequest"));
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//Extract the data into stNanoMsgData structure
	stNanoMsgData data;
	data.MsgID = static_cast<uint16_t>(request->getMsg_ID());
	data.DevID = static_cast<uint8_t>(request->getDev_ID());
	data.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	data.strURL = request->getDestIP().c_str();
	uint16_t tempdata = request->getPowerMode();
	
	tempdata = htons(tempdata);

	//32 bit data 
	data.data[0] = 0x00FF & tempdata;
	data.data[1] = (0xFF00 & tempdata) >> 8;
	data.DataLength = 2;
	
	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));

	ETG_TRACE_USR4(("PowerMode value is %d", request->getPowerMode()));

	ETG_TRACE_USR4(("Sending PowerMode value in stNanoMsgData.data[0] = %d, stNanoMsgData.data[1] = %d",
		data.data[0], data.data[1]));
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", data.strURL.c_str()));
	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(data);
		//jon1kor: send the reply back to daemon for clearing the buffer
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
		sendReqSetPowerModeResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
	else{
		sendReqSetPowerModeResponse(true);
	}
}
/************************************************************************
*FUNCTION: 		onReqGetPowerModeRequest(const ::boost::shared_ptr< ReqGetPowerModeRequest >& request)
*DESCRIPTION:   The method onReqGetPowerMode() is called by ASF when a client invokes
*				the method 'ReqGetPowerMode'.
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfLCMServiceAdaptor::onReqGetPowerModeRequest(const ::boost::shared_ptr< ReqGetPowerModeRequest >& request)
{
	ETG_TRACE_USR4(("onReqGetPowerModeRequest"));
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//Extract the data into stNanoMsgData structure
	stNanoMsgData data;
	data.MsgID = static_cast<uint16_t>(request->getMsg_ID());
	data.DevID = static_cast<uint8_t>(request->getDev_ID());
	data.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	data.strURL = request->getDestIP().c_str();
	
	//32 bit data 
	//todo; check the message length, and the data
	data.DataLength = 0;

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", data.strURL.c_str()));


	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(data);
		
		//jon1kor: send the reply back to daemon for clearing the buffer
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
		sendReqGetPowerModeResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
	else{
		sendReqGetPowerModeResponse(true);
	}

}

/************************************************************************
*FUNCTION: 		onReqGetWakeupReasonRequest(const ::boost::shared_ptr< ReqGetWakeupReasonRequest >& request)
*DESCRIPTION:   The method onReqGetWakeupReason() is called by ASF when a client
* invokes the method 'ReqGetWakeupReason'
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfLCMServiceAdaptor::onReqGetWakeupReasonRequest(const ::boost::shared_ptr< ReqGetWakeupReasonRequest >& request)
{
	ETG_TRACE_USR4(("onReqGetWakeupReasonRequest"));
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());
	nm.strURL = request->getDestIP().c_str();
	//todo: Check if the length is 0
	nm.DataLength = 0;

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));
	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
		
		//jon1kor: send the reply back to daemon for clearing the buffer
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
		sendReqGetWakeupReasonResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
	else{
		sendReqGetWakeupReasonResponse(true);
	}
	

}

/************************************************************************
*FUNCTION: 		onReqShutdownRequest(const ::boost::shared_ptr< ReqShutdownRequest >& request)
*DESCRIPTION:    The method onReqShutdown() is called by ASF when a client invokes the
* method 'ReqShutdown'.
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfLCMServiceAdaptor::onReqShutdownRequest(const ::boost::shared_ptr< ReqShutdownRequest >& request)
{
	ETG_TRACE_USR4(("onReqShutdownRequest"));
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());
	nm.strURL = request->getDestIP().c_str();
	nm.DataLength = 2;
	uint16 data = request->getTimeout();
	data = htons(data);

	//construct the data part of stNanoMsgData,
	nm.data[0] = 0x00FF & data;
	nm.data[1] = (0xFF00 & data) >> 8;


	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));

	ETG_TRACE_USR4(("Timeout value is %d", request->getTimeout()));

	ETG_TRACE_USR4(("Sending Timeout value in stNanoMsgData.data[0] = %d, stNanoMsgData.data[1] = %d", 
					 nm.data[0],nm.data[1]));

	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));
	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
		
		//jon1kor: send the reply back to daemon for clearing the buffer
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
		sendReqShutdownResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
	else{
		sendReqShutdownResponse(true);
	}
}

/************************************************************************
*FUNCTION: 		onReqSetChainModeRequest(const ::boost::shared_ptr< ReqSetChainModeRequest >& request)
*DESCRIPTION: 	The method onReqSetChainModeRequest() is called by ASF fremework for requesting setchainmode.
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfLCMServiceAdaptor::onReqSetChainModeRequest(const ::boost::shared_ptr< ReqSetChainModeRequest >& request)
{
	ETG_TRACE_USR4(("onReqSetChainModeRequest"));
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());
	nm.strURL = request->getDestIP().c_str();
	
	//Data length is 4 , 2 bytes of powermode data, 2 bytes of chain data
	nm.DataLength = 4;
	uint16 chaindata = request->getTChain();
	chaindata = htons(chaindata);
	uint16 powerdata = request->getPowerMode();
	powerdata = htons(powerdata);

	//construct the data part of stNanoMsgData,
	nm.data[0] = 0x00FF & powerdata;
	nm.data[1] = (0xFF00 & powerdata) >> 8;
	nm.data[2] = 0x00FF & chaindata;
	nm.data[3] = (0xFF00 & chaindata) >> 8;

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));

	ETG_TRACE_USR4(("PowerMode value is %d", request->getPowerMode()));
	ETG_TRACE_USR4(("Chain value is %d", request->getTChain()));

	ETG_TRACE_USR4(("Sending Data value in stNanoMsgData.data[0] = %d, stNanoMsgData.data[1] = %d",
		nm.data[0], nm.data[1]));

	ETG_TRACE_USR4(("Sending Data value in stNanoMsgData.data[2] = %d, stNanoMsgData.data[3] = %d",
		nm.data[2], nm.data[3]));
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));

	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
		
		//jon1kor: send the reply back to daemon for clearing the buffer
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
		sendReqSetChainModeResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
	else{
		sendReqSetChainModeResponse(true);
	}
}

/************************************************************************
*FUNCTION: 		onReqHealthCareRequest(const ::boost::shared_ptr< ReqHealthCareRequest >& request)
*DESCRIPTION:    The method onReqHealthCareRequest() is called by ASF when a client invokes the
                 method 'ReqHealthCare'.
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	DRM2KOR	05/03/2021
*************************************************************************/
void clAsfLCMServiceAdaptor::onReqHealthCareRequest(const ::boost::shared_ptr< ReqHealthCareRequest >& request)
{
    ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::onReqHealthCareRequest"));
    if(NULL == request)
    {
       ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::onReqHealthCareRequest() Received NULL data"));
       //Send Method error, Post an assert, will not be done for SOP1
       return;
    }

    //fill the stNanoMsgData structure
    stNanoMsgData nm;
    nm.DevID = static_cast<uint8_t>(request->getDev_ID());
    nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
    nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());
    nm.strURL = request->getDestIP().c_str();
    nm.DataLength = 0;

    ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::onReqHealthCareRequest() Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
           request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));
    ETG_TRACE_USR4(("clAsfLCMServiceAdaptor::onReqHealthCareRequest() Sending Message URL-IP is %s", nm.strURL.c_str()));
    bool beventSuccess = false;
    // Pass the data to the sender thread and call a NanoMsg event
    if (NULL != m_pcoreSenderThread){
	beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
    }	
    if (!beventSuccess){
        ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
        sendReqHealthCareResponse(false);
        //Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
    }
   else{
	sendReqHealthCareResponse(true);
    }
}

}//Namespace
}




