/* ***************************************************************************************
 * FILE:          clAsfIntercomServiceAdaptor.cpp
 * SW-COMPONENT:  HMI-BASE
 * DESCRIPTION:  clAsfIntercomServiceAdaptor.cpp is part of NanoMsgMasterApp
 * COPYRIGHT:  (c) 2020-21 Robert Bosch Car Multimedia GmbH
 * Author: Supriya Seshadri
 * Date: 03.02.2020
 * Revision: Supriya S 18.03.2020
 *		Removed ACKevent , Extended all interfaces to incorporate addition of URL-IP
 * Revision: Supriya S 01/04/2020
 *        Added NULL Checks in requests and responses
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */

#include "AsfIntercomServiceAdaptor.h"
//#include "clNanoMsgCoreThread.h"
#include "clSenderThread.h"
#include "plugin_trace.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_NANOMSGMASTERASF
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_NAVRES6
#include "trcGenProj/Header/AsfIntercomServiceAdaptor.cpp.trc.h"
#endif


using namespace ::asf::core;
// ========================================================================
namespace NanoMsg_MasterApp {
	namespace NanoMsgServer {
#define INTERCOM_PORT_NAME "NanoMsgIntercomPort"

	clAsfIntercomServiceAdaptor* clAsfIntercomServiceAdaptor::p_clAsfIntercomServiceAdaptor = NULL;
/************************************************************************
*FUNCTION: 		clAsfIntercomServiceAdaptor()
*DESCRIPTION: 	Constructor of the class
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18./02/2020
*************************************************************************/
clAsfIntercomServiceAdaptor::clAsfIntercomServiceAdaptor() :
	NanoMsgIntercomInterfaceStub(INTERCOM_PORT_NAME),
	m_pcoreSenderThread(NULL)
{

	ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor::Constructor Called"));
	init();
	m_pcoreSenderThread = clSenderThread::getInstance();
}

/************************************************************************
*FUNCTION: 		getInstance()
*DESCRIPTION: 	singleton implementation, creates an object
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18/02/2020
*************************************************************************/
clAsfIntercomServiceAdaptor* clAsfIntercomServiceAdaptor::getInstance()
{
	ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor::getInstance()"));
	if (p_clAsfIntercomServiceAdaptor == NULL)
	{
		ETG_TRACE_USR4(("p_clAsfIntercomServiceAdaptor::getInstance Creating Instance"));
		p_clAsfIntercomServiceAdaptor = new clAsfIntercomServiceAdaptor();
	}
	return p_clAsfIntercomServiceAdaptor;
}
/************************************************************************
*FUNCTION: 		~clAsfIntercomServiceAdaptor()
*DESCRIPTION: 	Destructor of the class
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18/02/2020
*************************************************************************/
// ------------------------------------------------------------------------
clAsfIntercomServiceAdaptor::~clAsfIntercomServiceAdaptor()
{
}

/************************************************************************
*FUNCTION: 		vDeleteInstance()
*DESCRIPTION: 	vDeleteInstance will delete clAsfIntercomServiceAdaptor
*				Singleton Object
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18/02/2020
*************************************************************************/
void clAsfIntercomServiceAdaptor::vDeleteInstance()
{
	if (p_clAsfIntercomServiceAdaptor != NULL)
	{
		ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor::vDeleteInstance()"));
		delete p_clAsfIntercomServiceAdaptor;
		p_clAsfIntercomServiceAdaptor = NULL;
	}
	else
	{
		ETG_TRACE_FATAL(("clAsfIntercomServiceAdaptor::vDeleteInstance() is ALREADY CALLED"));
	}
}


/************************************************************************
*FUNCTION: 		Init()
*DESCRIPTION: 	Initialization of Callbacks from Core is handled here
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
bool clAsfIntercomServiceAdaptor::init()
{
	ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor Init()"));

	clMsgIDMapping *_instance;
	_instance = clMsgIDMapping::getInstance();

		if (NULL != _instance)
	{
		ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor Initializing the Callbacks from Core..."));

		//Please look into document the CIS - system management protocol message definition 
		//and CIS Intercom message definition for the Below Mapping


		_instance->setCallback(0x0101, (FUNCPTR)&clAsfIntercomServiceAdaptor::sendIntercomRspPhoneStatus);
		_instance->setCallback(0x0102, (FUNCPTR)&clAsfIntercomServiceAdaptor::sendIntercomEventPhoneStatusChanged);
		_instance->setCallback(0x0104, (FUNCPTR)&clAsfIntercomServiceAdaptor::sendIntercomEventIntercomCall);
		_instance->setCallback(0x0108, (FUNCPTR)&clAsfIntercomServiceAdaptor::sendIntercomEventCancelIntercomCall);
		_instance->setCallback(0x0106, (FUNCPTR)&clAsfIntercomServiceAdaptor::sendIntercomEventAcceptIntercomCall);
		_instance->setCallback(0x010E, (FUNCPTR)&clAsfIntercomServiceAdaptor::sendIntercomRspGetVolume);
		_instance->setCallback(0x0110, (FUNCPTR)&clAsfIntercomServiceAdaptor::sendIntercomEventSetVolume);
		_instance->setCallback(0x0112, (FUNCPTR)&clAsfIntercomServiceAdaptor::sendIntercomRspGetVersion);
                _instance->setCallback(0x0113, (FUNCPTR)&clAsfIntercomServiceAdaptor::sendIntercomEventIntercomAvailable);
                _instance->setCallback(0x0115, (FUNCPTR)&clAsfIntercomServiceAdaptor::sendIntercomRspGetIntercomAvailable);
                return true;
	}
	return false;
}
/************************************************************************
*FUNCTION: 		onEventHUCallRequest(const ::boost::shared_ptr< EventHUCallRequest >& request)
*DESCRIPTION: 	The method onEventHUCallRequest() is called by ASF fremework for requesting a Call.
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18/02/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfIntercomServiceAdaptor::onEventHUCallRequest(const ::boost::shared_ptr< EventHUCallRequest >& request)
{
	ETG_TRACE_USR4(("onEventHUCallRequest"));
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());

	//Data length is 1 , 1 byte of phonenumber
	nm.DataLength = 1;
	uint8_t phoneno = request->getPhoneNumber();
	nm.strURL = request->getDestIP().c_str();
	
	//construct the data part of stNanoMsgData,
	nm.data[0] = 0x00FF & phoneno;

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));

	ETG_TRACE_USR4(("Phone Number value is %d", request->getPhoneNumber()));
	
	ETG_TRACE_USR4(("Sending Data value in stNanoMsgData.data[0] = %d, stNanoMsgData.data[1] = %d",
		nm.data[0], nm.data[1]));
		
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));

	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendEventHUCallResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
	else{
		 sendEventHUCallResponse(true);
	}
}

/************************************************************************
*FUNCTION: 		onEventAcceptHUCallRequest(const ::boost::shared_ptr< EventAcceptHUCallRequest >& request)
*DESCRIPTION: 	The method onEventAcceptHUCallRequest() is called by ASF fremework for requesting to accept a Call.
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18/02/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfIntercomServiceAdaptor::onEventAcceptHUCallRequest(const ::boost::shared_ptr< EventAcceptHUCallRequest >& request) 
{
	ETG_TRACE_USR4(("onEventAcceptHUCallRequest"));
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());

	//Data length is 0 , no data
	nm.DataLength = 0;
	nm.strURL = request->getDestIP().c_str();

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));
		
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));

	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendEventAcceptHUCallResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		 sendEventAcceptHUCallResponse(true);
	}
}

/************************************************************************
*FUNCTION: 		onEventCancelHUCallRequest(const ::boost::shared_ptr< EventCancelHUCallRequest >& request)
*DESCRIPTION: 	The method onEventCancelHUCallRequest() is called by ASF fremework for requesting to cancel a Call.
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18/02/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfIntercomServiceAdaptor::onEventCancelHUCallRequest(const ::boost::shared_ptr< EventCancelHUCallRequest >& request) 
{
	ETG_TRACE_USR4(("onEventCancelHUCallRequest"));
	
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());

	//Data length is 0, No data
	nm.DataLength = 0;
	nm.strURL = request->getDestIP().c_str();
	

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));
		
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));

	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendEventCancelHUCallResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		 sendEventCancelHUCallResponse(true);
	}
}

/************************************************************************
*FUNCTION: 		onReqPhoneStatusRequest(const ::boost::shared_ptr< ReqPhoneStatusRequest >& request)
*DESCRIPTION: 	The method onReqPhoneStatusRequest() is called by ASF fremework for phone status request
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18/02/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfIntercomServiceAdaptor::onReqPhoneStatusRequest(const ::boost::shared_ptr< ReqPhoneStatusRequest >& request) 
{
	ETG_TRACE_USR4(("onReqPhoneStatusRequest"));
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());
	//Data length is 0, No data
	nm.DataLength = 0;

	nm.strURL = request->getDestIP().c_str();

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));

	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));

	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendReqPhoneStatusResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		 sendReqPhoneStatusResponse(true);
	}
}

/************************************************************************
*FUNCTION: 		onReqSetVolumeUpRequest(const ::boost::shared_ptr< ReqSetVolumeUpRequest >& request)
*DESCRIPTION: 	The method onReqSetVolumeUpRequest() is called by ASF fremework for volume up request
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18/02/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfIntercomServiceAdaptor::onReqSetVolumeUpRequest(const ::boost::shared_ptr< ReqSetVolumeUpRequest >& request) 
{
	ETG_TRACE_USR4(("onReqSetVolumeUpRequest"));
	
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());

	//Data length is 0, No data
	nm.DataLength = 0;
	
	nm.strURL = request->getDestIP().c_str();

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));
		
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));

	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendReqSetVolumeUpResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		 sendReqSetVolumeUpResponse(true);
	}
}


/************************************************************************
*FUNCTION: 		onReqSetVolumeDownRequest(const ::boost::shared_ptr< ReqSetVolumeDownRequest >& request)
*DESCRIPTION: 	The method onReqSetVolumeDownRequest() is called by ASF fremework for volume down request
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18/02/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfIntercomServiceAdaptor::onReqSetVolumeDownRequest(const ::boost::shared_ptr< ReqSetVolumeDownRequest >& request) 
{
	ETG_TRACE_USR4(("onReqSetVolumeDownRequest"));
	
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());

	//Data length is 0, No data
	nm.DataLength = 0;

	nm.strURL = request->getDestIP().c_str();
	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));

	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendReqSetVolumeDownResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		 sendReqSetVolumeDownResponse(true);
	}
}


/************************************************************************
*FUNCTION: 		onReqGetVolumeRequest(const ::boost::shared_ptr< ReqGetVolumeRequest >& request)
*DESCRIPTION: 	The method onReqGetVolumeRequest() is called by ASF fremework for volume get request
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18/02/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfIntercomServiceAdaptor::onReqGetVolumeRequest(const ::boost::shared_ptr< ReqGetVolumeRequest >& request)
{
	ETG_TRACE_USR4(("onReqGetVolumeRequest"));
	
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());

	//Data length is 0, No data
	nm.DataLength = 0;

	nm.strURL = request->getDestIP().c_str();

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));
		
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));

	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendReqGetVolumeResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		 sendReqGetVolumeResponse(true);
	}
}


/************************************************************************
*FUNCTION: 		onReqSetVolumeRequest(const ::boost::shared_ptr< ReqSetVolumeRequest >& request)
*DESCRIPTION: 	The method onReqSetVolumeRequest() is called by ASF fremework for volume set request
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18/02/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfIntercomServiceAdaptor::onReqSetVolumeRequest(const ::boost::shared_ptr< ReqSetVolumeRequest >& request) 
{
	ETG_TRACE_USR4(("onReqSetVolumeRequest"));
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());
	nm.strURL = request->getDestIP().c_str();
	

	//Data length is 1 , 1 byte of phonenumber
	nm.DataLength = 1;
	uint8_t volumevalue = request->getVolumevalue();

	//construct the data part of stNanoMsgData,
	nm.data[0] = volumevalue;

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));

	ETG_TRACE_USR4(("Volume value requested is %d", request->getVolumevalue()));

	ETG_TRACE_USR4(("Sending Data value in stNanoMsgData.data[0] = %d",
		nm.data[0]));
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));

	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendReqSetVolumeResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		 sendReqSetVolumeResponse(true);
	}
}


/************************************************************************
*FUNCTION: 		onReqGetVersionRequest(const ::boost::shared_ptr< ReqGetVersionRequest >& request)
*DESCRIPTION: 	The method onReqGetVersionRequest() is called by ASF fremework for version get request
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18/02/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfIntercomServiceAdaptor::onReqGetVersionRequest(const ::boost::shared_ptr< ReqGetVersionRequest >& request)
{
	ETG_TRACE_USR4(("onReqGetVersionRequest"));
	
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());
	nm.strURL = request->getDestIP().c_str();

	//Data length is 0, No data
	nm.DataLength = 0;

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));
		
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));

	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendReqGetVersionResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		 sendReqGetVersionResponse(true);
	}
}

/************************************************************************
*FUNCTION: 		onReqGetIntercomAvailableRequest(const ::boost::shared_ptr< ReqGetIntercomAvailableRequest >& request)
*DESCRIPTION: 	The method onReqGetIntercomAvailableRequest() is called by ASF fremework for Intercom Availablity request
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Madhuree	26/10/2020
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfIntercomServiceAdaptor::onReqGetIntercomAvailableRequest(const ::boost::shared_ptr< ReqGetIntercomAvailableRequest >& request) 
{
	ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor::onReqGetIntercomAvailableRequest"));
	if(NULL == request)
	{
		ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor::onReqGetIntercomAvailableRequest() Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());

	//Data length is 0, No data
	nm.DataLength = 0;
	nm.strURL = request->getDestIP().c_str();

	ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor::onReqGetIntercomAvailableRequest(): Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));
		
	ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor::onReqGetIntercomAvailableRequest(): Sending Message URL-IP is %s", nm.strURL.c_str()));

	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
        if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
        //jon1kor: send the reply back to daemon for clearing the buffer		
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
               sendReqGetIntercomAvailableResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		 sendReqGetIntercomAvailableResponse(true);
	}
}

/************************************************************************
*FUNCTION: 		sendIntercomEventIntercomAvailable(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the Sig_EventIntercomAvailable
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Madhuree	26/10/2020
*************************************************************************/
void clAsfIntercomServiceAdaptor::sendIntercomEventIntercomAvailable(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor::sendIntercomEventIntercomAvailable"));
	if(NULL == data)
	{
		ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor::sendIntercomEventIntercomAvailable():Data Received is NULL"));
		return;
	}
	//Extract the data

	eTIntercomAvailable availability = static_cast<eTIntercomAvailable>(data->data[0]);

	ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor::sendIntercomEventIntercomAvailable() Recieved availability is = %d",
		availability));
		
	ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor::sendIntercomEventIntercomAvailable() Received Message URL-IP is %s", data->strURL.c_str()));

	//Send the Recived data to ASF clients
	sendSig_EventIntercomAvailableSignal(availability, data->strURL);

}

/************************************************************************
*FUNCTION: 		sendIntercomRspGetIntercomAvailable(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the Sig_RspGetIntercomAvailable
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Madhuree	26/10/2020
*************************************************************************/
void clAsfIntercomServiceAdaptor::sendIntercomRspGetIntercomAvailable(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor::sendIntercomRspGetIntercomAvailable()"));
	if(NULL == data)
	{
		ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor::sendIntercomRspGetIntercomAvailable() Data Received is NULL"));
		return;
	}
	//Extract the data
	
	eTIntercomAvailable availabilityValue = static_cast<eTIntercomAvailable>(data->data[0]);

	ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor::sendIntercomRspGetIntercomAvailable(): Recieved availability Value = %d",
		availabilityValue));
	
	ETG_TRACE_USR4(("clAsfIntercomServiceAdaptor::sendIntercomRspGetIntercomAvailable() : Received Message URL-IP is %s", data->strURL.c_str()));

	//Send the Recived data to ASF clients
	sendSig_RspGetIntercomAvailableSignal(availabilityValue, data->strURL);
}

/************************************************************************
*FUNCTION: 		sendIntercomRspPhoneStatus(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the Sig_RspPhoneStatus
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfIntercomServiceAdaptor::sendIntercomRspPhoneStatus(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendIntercomRspPhoneStatus"));
	if(NULL == data)
	{
		ETG_TRACE_USR4(("Data Received is NULL"));
		return;
	}
	
	//Extract the data
	uint8_t phonestatus1 = data->data[0];
	uint8_t phonestatus2 = data->data[1];
	uint8_t phonestatus3 = data->data[2];


	ETG_TRACE_USR4(("Recieved Phone Status phonestatus1 = %d, phonestatus2 = %d, phonestatus3 = %d ",
		phonestatus1, phonestatus1, phonestatus1));
		
	ETG_TRACE_USR4(("Received Message URL-IP is %s", data->strURL.c_str()));

	//Send the Recived data to ASF clients
	sendSig_RspPhoneStatusSignal(phonestatus1, phonestatus2, phonestatus3, data->strURL);
}

/************************************************************************
*FUNCTION: 		sendIntercomEventPhoneStatusChanged(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the Sig_EventPhoneStatusChanged
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfIntercomServiceAdaptor::sendIntercomEventPhoneStatusChanged(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendIntercomEventPhoneStatusChanged"));
	if(NULL == data)
	{
		ETG_TRACE_USR4(("Data Received is NULL"));
		return;
	}
	//Extract the data
	
	eTPhoneNumber Phoneno = static_cast<eTPhoneNumber>(data->data[0]);
	uint8_t phonestatus = data->data[1];


	ETG_TRACE_USR4(("Recieved Phone Number = %d, phonestatus1 = %d",
		Phoneno, phonestatus));
	
	ETG_TRACE_USR4(("Received Message URL-IP is %s", data->strURL.c_str()));

	//Send the Recived data to ASF clients
	sendSig_EventPhoneStatusChangedSignal(Phoneno, phonestatus, data->strURL);

}

/************************************************************************
*FUNCTION: 		sendIntercomEventIntercomCall(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the Sig_EventIntercomCall
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfIntercomServiceAdaptor::sendIntercomEventIntercomCall(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendIntercomEventIntercomCall"));
	if(NULL == data)
	{
		ETG_TRACE_USR4(("Data Received is NULL"));
		return;
	}
	//Extract the data
	
	eTPhoneNumber Phoneno = static_cast<eTPhoneNumber>(data->data[0]);

	ETG_TRACE_USR4(("Recieved Phone Number = %d",
		Phoneno));
		
	ETG_TRACE_USR4(("Received Message URL-IP is %s", data->strURL.c_str()));

	//Send the Recived data to ASF clients
	sendSig_EventIntercomCallSignal(Phoneno, data->strURL);

}

/************************************************************************
*FUNCTION: 		sendIntercomEventCancelIntercomCall(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the Sig_EventCancelIntercomCall
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfIntercomServiceAdaptor::sendIntercomEventCancelIntercomCall(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendIntercomEventCancelIntercomCall"));
	if(NULL == data)
	{
		ETG_TRACE_USR4(("Data Received is NULL"));
		return;
	}
	//Extract the data
	bool bResponse = static_cast<bool>(data->data[0]);

	ETG_TRACE_USR4(("Recieved CancelIntercomCall value = %d",
						bResponse));
						
	ETG_TRACE_USR4(("Received Message URL-IP is %s", data->strURL.c_str()));

	//Send the Recived data to ASF clients
	sendSig_EventCancelIntercomCallSignal(bResponse, data->strURL);

}


/************************************************************************
*FUNCTION: 		sendIntercomEventAcceptIntercomCall(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the Sig_EventAcceptIntercomCall
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfIntercomServiceAdaptor::sendIntercomEventAcceptIntercomCall(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendIntercomEventAcceptIntercomCall"));
		if(NULL == data)
	{
		ETG_TRACE_USR4(("Data Received is NULL"));
		return;
	}
	//Extract the data
	eTPhoneNumber Phoneno = static_cast<eTPhoneNumber>(data->data[0]);

	ETG_TRACE_USR4(("Recieved Phone no value = %d",
		Phoneno));

	ETG_TRACE_USR4(("Received Message URL-IP is %s", data->strURL.c_str()));
	//Send the Recived data to ASF clients
	sendSig_EventAcceptIntercomCallSignal(Phoneno, data->strURL);

}


/************************************************************************
*FUNCTION: 		sendIntercomRspGetVolume(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the Sig_RspGetVolume
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfIntercomServiceAdaptor::sendIntercomRspGetVolume(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendIntercomRspGetVolume"));
	if(NULL == data)
	{
		ETG_TRACE_USR4(("Data Received is NULL"));
		return;
	}
	//Extract the data
	
	uint8 volumevalue = data->data[0];

	ETG_TRACE_USR4(("Recieved Get Volume Value = %d",
		volumevalue));
	
	ETG_TRACE_USR4(("Received Message URL-IP is %s", data->strURL.c_str()));

	//Send the Recived data to ASF clients
	sendSig_RspGetVolumeSignal(volumevalue, data->strURL);
}

/************************************************************************
*FUNCTION: 		sendIntercomEventSetVolume(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the Sig_EventSetVolume
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfIntercomServiceAdaptor::sendIntercomEventSetVolume(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendIntercomEventSetVolume"));
	
	if(NULL == data)
	{
		ETG_TRACE_USR4(("Data Received is NULL"));
		return;
	}
	
	//Extract the data
	uint8 volumevalue = data->data[0];

	ETG_TRACE_USR4(("Recieved Set Volume Value = %d",
		volumevalue));
		
	ETG_TRACE_USR4(("Received Message URL-IP is %s", data->strURL.c_str()));

	//Send the Recived data to ASF clients
	sendSig_EventSetVolumeSignal(volumevalue, data->strURL);
}

/************************************************************************
*FUNCTION: 		sendIntercomRspGetVersion(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the Sig_EventSetVolume
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfIntercomServiceAdaptor::sendIntercomRspGetVersion(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendIntercomRspGetVersion"));
	
	if(NULL == data)
	{
		ETG_TRACE_USR4(("Data Received is NULL"));
		return;
	}
	//Extract the data
	

	uint32_t swversion = data->data[0] | ((data->data[1]) << 8) | ((data->data[2]) << 16) | ((data->data[3]) << 24);
	uint32_t hwversion = data->data[4] | ((data->data[5]) << 8) | ((data->data[6]) << 16) | ((data->data[7]) << 24);	

	
	swversion = ntohl(swversion);
	hwversion = ntohl(hwversion);
	ETG_TRACE_USR4(("Recieved SWversion is %d, HW version is %d",
		swversion, hwversion));
	ETG_TRACE_USR4(("Received Message URL-IP is %s", data->strURL.c_str()));

	sendSig_RspGetVersionSignal(swversion, hwversion, data->strURL);
}

}}
