/* ***************************************************************************************
 * FILE:          clAsfGenericServiceAdaptor.cpp
 * SW-COMPONENT:  NanoMsg_Master_Application
 * DESCRIPTION:  clAsfGenericServiceAdaptor.cpp is part of NanoMsgMasterApp
 * COPYRIGHT:  (c) 2020-21 Robert Bosch Car Multimedia GmbH
 * Author: Madhuree
 * Date: 14.02.2020
 * Revision: Supriya S
 *		Added the Method Starts and the Broadcast messages with the functionality to provide to the clients
 * Revision: Supriya S 01/04/2020
 *        Added NULL Checks in requests and responses
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */

#include "AsfGenericServiceAdaptor.h"


#include "plugin_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_NANOMSGMASTERASF
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_NAVRES6

#include "trcGenProj/Header/AsfGenericServiceAdaptor.cpp.trc.h"

#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


using namespace ::asf::core;
// ========================================================================
namespace NanoMsg_MasterApp {
	namespace NanoMsgServer {
clAsfGenericServiceAdaptor* clAsfGenericServiceAdaptor::p_clAsfGenericServiceAdaptor = NULL;

#define GENERIC_PORT_NAME "NanoMsgGenericPort" 

/************************************************************************
*FUNCTION: 		clAsfGenericServiceAdaptor()
*DESCRIPTION: 	Constructor of the class
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	drm2kor	14/02/2020
*************************************************************************/
clAsfGenericServiceAdaptor::clAsfGenericServiceAdaptor() :
	NanoMsgGenericInterfaceStub(GENERIC_PORT_NAME),
	m_pcoreSenderThread(NULL)
{
	ETG_TRACE_USR4(("clAsfGenericServiceAdaptor::Constructor Called"));
	init();
	m_pcoreSenderThread = clSenderThread::getInstance();

}

/************************************************************************
*FUNCTION: 		getInstance()
*DESCRIPTION: 	singleton implementation, creates an object
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	drm2kor	14/02/2020
*************************************************************************/
clAsfGenericServiceAdaptor* clAsfGenericServiceAdaptor::getInstance()
{
	ETG_TRACE_USR4(("clAsfGenericServiceAdaptor::getInstance()"));
	if (p_clAsfGenericServiceAdaptor == NULL)
	{
		ETG_TRACE_USR4(("p_clAsfGenericServiceAdaptor::getInstance Creating Instance"));
		p_clAsfGenericServiceAdaptor = new clAsfGenericServiceAdaptor();
	}
	return p_clAsfGenericServiceAdaptor;
}
/************************************************************************
*FUNCTION: 		~clAsfGenericServiceAdaptor()
*DESCRIPTION: 	Destructor of the class
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	drm2kor	14/02/2020
*************************************************************************/
// ------------------------------------------------------------------------
clAsfGenericServiceAdaptor::~clAsfGenericServiceAdaptor()
{
}


/************************************************************************
*FUNCTION: 		Init()
*DESCRIPTION: 	Initialization of Callbacks from Core is handled here
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
bool clAsfGenericServiceAdaptor::init()
{
	ETG_TRACE_USR4(("clAsfGenericServiceAdaptor Init()"));

	clMsgIDMapping *_instance;
	_instance = clMsgIDMapping::getInstance();

	if (NULL != _instance)
	{
		ETG_TRACE_USR4(("clAsfGenericServiceAdaptor Initializing the Callbacks from Core..."));

		////Please look into document the CIS - system management protocol message definition 
		////and CIS Intercom message definition for the Below Mapping
		
		//Digital Input Messages
		_instance->setCallback(0x0015, (FUNCPTR)&clAsfGenericServiceAdaptor::sendGenericEventDigitalInput);
		_instance->setCallback(0x0017, (FUNCPTR)&clAsfGenericServiceAdaptor::sendGenericRspGetDigitalInput);
		_instance->setCallback(0x0050, (FUNCPTR)&clAsfGenericServiceAdaptor::sendGenericEventDigitalInputCMP);
		_instance->setCallback(0x0052, (FUNCPTR)&clAsfGenericServiceAdaptor::sendGenericRspGetDigitalInputCMP);
		_instance->setCallback(0x0060, (FUNCPTR)&clAsfGenericServiceAdaptor::sendGenericEventDigitalInputCMG);
		_instance->setCallback(0x0062, (FUNCPTR)&clAsfGenericServiceAdaptor::sendGenericRspGetDigitalInputCMG);
		_instance->setCallback(0x0021, (FUNCPTR)&clAsfGenericServiceAdaptor::sendGenericRspSetAnnouncementOut);
                _instance->setCallback(0x0065, (FUNCPTR)&clAsfGenericServiceAdaptor::sendGenericRspCMGSvivelPositionSignal);
                _instance->setCallback(0x0063, (FUNCPTR)&clAsfGenericServiceAdaptor::sendGenericEventCMGSwivelPositionSignal);

		//Thermal and Voltage Messages
		_instance->setCallback(0x0043, (FUNCPTR)&clAsfGenericServiceAdaptor::sendGenericEventThermalSignal);
		_instance->setCallback(0x0045, (FUNCPTR)&clAsfGenericServiceAdaptor::sendGenericRspGetThermalSignal);
		_instance->setCallback(0x0012, (FUNCPTR)&clAsfGenericServiceAdaptor::sendGenericEventVoltageSignal);
		_instance->setCallback(0x0014, (FUNCPTR)&clAsfGenericServiceAdaptor::sendGenericRspGetVoltageSignal);
		return true;
	}
	return false;
}
/************************************************************************
*FUNCTION: 		vDeleteInstance()
*DESCRIPTION: 	vDeleteInstance will delete clAsfGenericServiceAdaptor
*				Singleton Object
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	drm2kor	14/02/2020
*************************************************************************/
void clAsfGenericServiceAdaptor::vDeleteInstance()
{
	if (p_clAsfGenericServiceAdaptor != NULL)
	{
		ETG_TRACE_USR4(("clAsfGenericServiceAdaptor::vDeleteInstance()"));
		delete p_clAsfGenericServiceAdaptor;
		p_clAsfGenericServiceAdaptor = NULL;
	}
	else
	{
		ETG_TRACE_FATAL(("clAsfGenericServiceAdaptor::vDeleteInstance() is ALREADY CALLED"));
	}
}
/************************************************************************
*FUNCTION: 		onReqGetDigitalInputRequest(const ::boost::shared_ptr< ReqGetDigitalInputRequest >& request)
*DESCRIPTION: 	The method onReqGetDigitalInputRequest() is called by ASF fremework for requesting DigitalInput
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18/02/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfGenericServiceAdaptor::onReqGetDigitalInputRequest(const ::boost::shared_ptr< ReqGetDigitalInputRequest >& request)
{
	ETG_TRACE_USR4(("onReqGetDigitalInputRequest"));

	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}
	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());

	//Data length is 0, No data
	nm.DataLength = 0;
	
	nm.strURL = request->getDestIP().c_str();

	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));		

	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendReqGetDigitalInputResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		sendReqGetDigitalInputResponse(true);
	}
}


/************************************************************************
*FUNCTION: 		onReqGetDigitalInputCMPRequest(const ::boost::shared_ptr< ReqGetDigitalInputCMPRequest >& request)
*DESCRIPTION: 	The method onReqGetDigitalInputCMPRequest() is called by ASF fremework for requesting DigitalInputCMP
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	09/03/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfGenericServiceAdaptor::onReqGetDigitalInputCMPRequest (const ::boost::shared_ptr< ReqGetDigitalInputCMPRequest >& request)
{
		ETG_TRACE_USR4(("onReqGetDigitalInputCMPRequest"));
		
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());
	nm.strURL = request->getDestIP().c_str();

	//Data length is 0, No data
	nm.DataLength = 0;
	
	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));
	// Pass the data to the sender thread and call a NanoMsg event

	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendReqGetDigitalInputCMPResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		sendReqGetDigitalInputCMPResponse(true);
	}

}

/************************************************************************
*FUNCTION:     onReqSetAnnouncementOutRequest (const ::boost::shared_ptr< ReqSetAnnouncementOut >& request)
*DESCRIPTION: 	The method onReqSetAnnouncementOutRequest() is called by ASF fremework for requesting Annoucement
*PARAMETER:	None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Madhuree    06/10/2020   as part of SWFL-3715/RTC – 900223
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfGenericServiceAdaptor::onReqSetAnnouncementOutRequest (const ::boost::shared_ptr< ReqSetAnnouncementOutRequest >& request)
{
    ETG_TRACE_USR4(("clAsfGenericServiceAdaptor::onReqSetAnnouncementOutRequest()"));
    if(NULL == request)
      {
	ETG_TRACE_USR4(("onReqSetAnnouncementOutRequest(): Received NULL data"));
	return;
      }
      //fill the stNanoMsgData structure
      stNanoMsgData nm;
      nm.DevID = static_cast<uint8_t>(request->getDev_ID());
      nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
      nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());
      //1 byte data
      nm.DataLength = 1;
      uint8_t announcementState= request->getAnnouncementState();
      nm.strURL = request->getDestIP().c_str();
      
      //construct the data part of stNanoMsgData,
      nm.data[0] = announcementState;
      ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));

      ETG_TRACE_USR4(("Announcement state value is %d", request->getAnnouncementState()));
      ETG_TRACE_USR4(("Sending Data value in stNanoMsgData.data[0] = %d, stNanoMsgData.data[1] = %d",
		nm.data[0], nm.data[1]));
		
      ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));
      bool beventSuccess = false;
      // Pass the data to the sender thread and call a NanoMsg event
      if (NULL != m_pcoreSenderThread)
           //Pass it to the sender thread of core
	   beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendReqSetAnnouncementOutResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		sendReqSetAnnouncementOutResponse(true);
	}
}


/************************************************************************
*FUNCTION: 		onReqGetDigitalInputCMGRequest(const ::boost::shared_ptr< ReqGetDigitalInputCMGRequest >& request)
*DESCRIPTION: 	The method onReqGetDigitalInputCMPRequest() is called by ASF fremework for requesting DigitalInputCMP
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	09/03/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfGenericServiceAdaptor::onReqGetDigitalInputCMGRequest (const ::boost::shared_ptr< ReqGetDigitalInputCMGRequest >& request)
{
		ETG_TRACE_USR4(("onReqGetDigitalInputCMGRequest"));
		
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());
	nm.strURL = request->getDestIP().c_str();

	//Data length is 0, No data
	nm.DataLength = 0;
	
	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));
	
	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendReqGetDigitalInputCMGResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		sendReqGetDigitalInputCMGResponse(true);
	}
}

/************************************************************************
*FUNCTION: 		onReqCMGSvivelPositionRequest(const ::boost::shared_ptr< ReqCMGSvivelPositionRequest >& request)
*DESCRIPTION: 	The method onReqCMGSvivelPositionRequest() is called by ASF when a client invokes
* the method 'ReqCMGSvivelPosition'.
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	DRM2KOR	16/10/2020
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfGenericServiceAdaptor::onReqCMGSvivelPositionRequest(const ::boost::shared_ptr< ReqCMGSvivelPositionRequest >& request)
{
	ETG_TRACE_USR4(("clAsfGenericServiceAdaptor::onReqCMGSvivelPositionRequest"));
	
	if(NULL == request)
	{
		ETG_TRACE_USR4(("clAsfGenericServiceAdaptor::onReqCMGSvivelPositionRequest() Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

        //Extract the data into stNanoMsgData structure
	stNanoMsgData data;
	data.MsgID = static_cast<uint16_t>(request->getMsg_ID());
	data.DevID = static_cast<uint8_t>(request->getDev_ID());
	data.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	data.strURL = request->getDestIP().c_str();
	//8 bit data
	data.DataLength = 1;
	uint8_t svivelData = request->getSvivelRequest();
	//construct the data as part of stNanoMsgData
	data.data[0] = svivelData;
        ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));

	ETG_TRACE_USR4(("svivelData value is %d", request->getSvivelRequest()));

	ETG_TRACE_USR4(("Sending Message URL-IP is %s", data.strURL.c_str()));
	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
        if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(data);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendReqCMGSvivelPositionResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		sendReqCMGSvivelPositionResponse(true);
	}
}

/************************************************************************
*FUNCTION: 		sendGenericRspGetDigitalInput(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the sig_RspGetDigitalInput
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfGenericServiceAdaptor::sendGenericRspGetDigitalInput(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendGenericRspGetDigitalInput"));
	//Extract the data
	
	uint32_t digitalInput = ((data->data[0]) | ((data->data[1]) << 8) | ((data->data[2]) << 16) | ((data->data[3]) << 24));
	digitalInput = ntohl(digitalInput);
	
	ETG_TRACE_USR4(("Recieved digitalInput in data[0], data[1], data[2], data[3] is %d,%d,%d,%d ",
		data->data[0],data->data[1],data->data[2],data->data[3]));
		
	ETG_TRACE_USR4(("Sending digitalInput is %d, ",
		digitalInput));

	ETG_TRACE_USR4(("Recieved Message URL-IP is %s", data->strURL.c_str()));
	sendSig_RspGetDigitalInputSignal(digitalInput, data->strURL);

}

/************************************************************************
*FUNCTION: 		sendGenericEventDigitalInput(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the sig_RspGetDigitalInput
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfGenericServiceAdaptor::sendGenericEventDigitalInput(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendGenericEventDigitalInput"));
	//Extract the data
	

	uint32_t digitalInput = ((data->data[0]) | ((data->data[1]) << 8) | ((data->data[2]) << 16) | ((data->data[3]) << 24));
	
	digitalInput = ntohl(digitalInput);
	
	entBool response = static_cast<entBool>(data->data[3]);
	
	ETG_TRACE_USR4(("Recieved digitalInput in data[0], data[1], data[2], data[3] is %d,%d,%d,%d ",
		data->data[0],data->data[1],data->data[2],data->data[3]));
		
	ETG_TRACE_USR4(("Sending digitalInput is %d",
		digitalInput));
		
	ETG_TRACE_USR4(("Recieved response in data[5] is %d ",
		data->data[5]));
		
	ETG_TRACE_USR4(("Sending response is %d",
		response));
	ETG_TRACE_USR4(("Recieved Message URL-IP is %s", data->strURL.c_str()));
	
	sendSig_EventDigitalInputSignal(digitalInput, response, data->strURL);

}


/************************************************************************
*FUNCTION: 		sendGenericRspGetDigitalInputCMP(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the sig_RspGetDigitalInputCMP
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfGenericServiceAdaptor::sendGenericRspGetDigitalInputCMP(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendGenericRspGetDigitalInputCMP"));
	//Extract the data
	

	uint32_t digitalInput = ((data->data[0]) | ((data->data[1]) << 8) | ((data->data[2]) << 16) | ((data->data[3]) << 24));
	digitalInput = ntohl(digitalInput);
	
	ETG_TRACE_USR4(("Recieved digitalInputCMP in data[0], data[1], data[2], data[3] is %d,%d,%d,%d ",
		data->data[0],data->data[1],data->data[2],data->data[3]));
		
	ETG_TRACE_USR4(("Sending digitalInputCMP is %d, ",
		digitalInput));

	ETG_TRACE_USR4(("Recieved Message URL-IP is %s", data->strURL.c_str()));
	sendSig_RspGetDigitalInputCMPSignal(digitalInput, data->strURL);
}

/************************************************************************
*FUNCTION: 		sendGenericRspGetDigitalInputCMG(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the sig_RspGetDigitalInputCMG
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfGenericServiceAdaptor::sendGenericRspGetDigitalInputCMG(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendGenericRspGetDigitalInputCMG"));
	//Extract the data
	

	uint32_t digitalInput = ((data->data[0]) | ((data->data[1]) << 8) | ((data->data[2]) << 16) | ((data->data[3]) << 24));
	digitalInput = ntohl(digitalInput);
	
	ETG_TRACE_USR4(("Recieved digitalInputCMG in data[0], data[1], data[2], data[3] is %d,%d,%d,%d ",
		data->data[0],data->data[1],data->data[2],data->data[3]));
		
	ETG_TRACE_USR4(("Sending digitalInputCMP %d, ",
		digitalInput));
	
	ETG_TRACE_USR4(("Recieved Message URL-IP is %s", data->strURL.c_str()));

	sendSig_RspGetDigitalInputCMGSignal(digitalInput, data->strURL);
}

/************************************************************************
*FUNCTION: 		sendGenericEventDigitalInputCMP(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the sig_RspGetDigitalInputCMP
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfGenericServiceAdaptor::sendGenericEventDigitalInputCMP(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendGenericEventDigitalInputCMP"));
	//Extract the data
	

	uint32_t digitalInput = ((data->data[0]) | ((data->data[1]) << 8) | ((data->data[2]) << 16) | ((data->data[3]) << 24));
	
	digitalInput = ntohl(digitalInput);
	
	
	ETG_TRACE_USR4(("Recieved digitalInputCMP in data[0], data[1], data[2], data[3] is %d,%d,%d,%d ",
		data->data[0],data->data[1],data->data[2],data->data[3]));
		
	ETG_TRACE_USR4(("Sending digitalInputCMP is %d",
		digitalInput));

	ETG_TRACE_USR4(("Recieved Message URL-IP is %s", data->strURL.c_str()));
	
	sendSig_EventDigitalInputCMPSignal(digitalInput, data->strURL);

}

/************************************************************************
*FUNCTION: 		sendGenericEventDigitalInputCMG(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the sig_RspGetDigitalInputCMG
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfGenericServiceAdaptor::sendGenericEventDigitalInputCMG(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendGenericEventDigitalInputCMG"));
	//Extract the data
	

	uint32_t digitalInput = ((data->data[0]) | ((data->data[1]) << 8) | ((data->data[2]) << 16) | ((data->data[3]) << 24));
	
	digitalInput = ntohl(digitalInput);
	
	
	ETG_TRACE_USR4(("Recieved digitalInputCMG in data[0], data[1], data[2], data[3] is %d,%d,%d,%d ",
		data->data[0],data->data[1],data->data[2],data->data[3]));
		
	ETG_TRACE_USR4(("Sending digitalInputCMG is %d",
		digitalInput));
		
	ETG_TRACE_USR4(("Recieved Message URL-IP is %s", data->strURL.c_str()));

	sendSig_EventDigitalInputCMGSignal(digitalInput , data->strURL);
}

/************************************************************************
*FUNCTION: 		ReqGetThermalStateRequest(const ::boost::shared_ptr< ReqGetThermalStateRequest >& request)
*DESCRIPTION: 	The method ReqGetThermalStateRequest() is called by ASF fremework for requesting DigitalInputCMP
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18/03/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfGenericServiceAdaptor::onReqGetThermalStateRequest (const ::boost::shared_ptr< ReqGetThermalStateRequest >& request)
{
	ETG_TRACE_USR4(("ReqGetThermalStateRequest"));

	//fill the stNanoMsgData structure
	stNanoMsgData nm;
	nm.DevID = static_cast<uint8_t>(request->getDev_ID());
	nm.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	nm.MsgID = static_cast<uint16_t>(request->getMsg_ID());
	nm.strURL = request->getDestIP().c_str();

	//Data length is 0, No data
	nm.DataLength = 0;
	
	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));
		
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", nm.strURL.c_str()));

	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(nm);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendReqGetThermalStateResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		sendReqGetThermalStateResponse(true);
	}
}
/************************************************************************
*FUNCTION: 		onReqGetVoltageRequest(const ::boost::shared_ptr< ReqGetVoltageRequest >& request)
*DESCRIPTION: 	The method onReqGetVoltageRequest() is called by ASF fremework for requesting DigitalInputCMP
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	18/03/2020
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue
*Bugfix:        Shiva Kumar J (MS/ECG5-XC) issue fixed 1085389
*************************************************************************/
void clAsfGenericServiceAdaptor::onReqGetVoltageRequest (const ::boost::shared_ptr< ReqGetVoltageRequest >& request)
{
	ETG_TRACE_USR4(("onReqGetVoltageRequest"));	
		
	if(NULL == request)
	{
		ETG_TRACE_USR4(("Received NULL data"));
		//Send Method error, Post an assert, will not be done for SOP1
		return;
	}

	//fill the stNanoMsgData structure
	stNanoMsgData data;
	data.DevID = static_cast<uint8_t>(request->getDev_ID());
	data.DevNr = static_cast<uint8_t>(request->getDev_Nr());
	data.MsgID = static_cast<uint16_t>(request->getMsg_ID());
	data.strURL = request->getDestIP().c_str();
	
	uint16_t tempdata = static_cast<uint16_t>(request->getVoltageSensor());
	
	ETG_TRACE_USR4(("Recieved Voltage Value in %d", tempdata));
	//Change the endianess needed to send in network
	tempdata = htons(tempdata);

	//16 bit data 
	data.data[0] = 0x00FF & tempdata;
	data.data[1] = (0xFF00 & tempdata) >> 8;
	data.DataLength = 2;
	
	ETG_TRACE_USR4(("Recieved Message Data is devid = %d, devnr = %d, msgid = %d ",
		request->getDev_ID(), request->getDev_Nr(), request->getMsg_ID()));
		
	ETG_TRACE_USR4(("Sending Voltage value in stNanoMsgData.data[0] = %d, stNanoMsgData.data[1] = %d",
		data.data[0], data.data[1]));
		
	ETG_TRACE_USR4(("Sending Message URL-IP is %s", data.strURL.c_str()));

	bool beventSuccess = false;
	// Pass the data to the sender thread and call a NanoMsg event
	if (NULL != m_pcoreSenderThread)
		//Pass it to the sender thread of core
		beventSuccess = m_pcoreSenderThread->setNanoMsgEvent(data);
	//jon1kor: send the reply back to daemon for clearing the buffer	
	if (!beventSuccess){
		ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
                sendReqGetVoltageResponse(false);
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}
        else{
		sendReqGetVoltageResponse(true);
	}
}
/************************************************************************
*FUNCTION: 		sendGenericRspGetVoltageSignal(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the sendRspGetVoltageSignal
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfGenericServiceAdaptor::sendGenericRspGetVoltageSignal(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendGenericRspGetVoltageSignal"));
	//Extract the data
	
	

	uint16_t voltageLevel = ((data->data[0]) | (data->data[1] << 8));
	voltageLevel = ntohs(voltageLevel);
	entVoltageLevel envoltageLevel = static_cast<entVoltageLevel>(voltageLevel);
	
	
	uint16_t voltageSensor = ((data->data[2]) | (data->data[3] << 8));
	voltageSensor = ntohs(voltageSensor);
	entVoltageSensor envoltageSensor = static_cast<entVoltageSensor>(voltageSensor);
	
	
	uint16 voltageValue = ((data->data[4]) | (data->data[5] << 8));
	voltageValue = ntohs(voltageValue);
	
	ETG_TRACE_USR4(("Recieved Data values in data[0], data[1], data[2], data[3] is %d,%d,%d,%d ",
		data->data[0],data->data[1],data->data[2],data->data[3]));
		
	ETG_TRACE_USR4(("Recieved Data values in data[4], data[5] %d,%d ",
		data->data[4],data->data[5]));
		
	ETG_TRACE_USR4(("Sending  Volatage -- level %d , Sensor %d and Value %d ",
		voltageLevel, voltageSensor, voltageValue));
		
	ETG_TRACE_USR4(("Received Message URL-IP is %s", data->strURL.c_str()));

	sendRspGetVoltageSignal(envoltageLevel, envoltageSensor, voltageValue, data->strURL);
}

/************************************************************************
*FUNCTION: 		sendGenericEventVoltageSignal(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the sendEventVoltageSignal
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfGenericServiceAdaptor::sendGenericEventVoltageSignal(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendGenericEventVoltageSignal"));
	//Extract the data
	
	

	uint16_t voltageLevel = ((data->data[0]) | (data->data[1] << 8));
	voltageLevel = ntohs(voltageLevel);
	entVoltageLevel envoltageLevel = static_cast<entVoltageLevel>(voltageLevel);
	
	
	uint16_t voltageSensor = ((data->data[2]) | (data->data[3] << 8));
	voltageSensor = ntohs(voltageSensor);
	entVoltageSensor envoltageSensor = static_cast<entVoltageSensor>(voltageSensor);
	
	
	uint16 voltageValue = ((data->data[4]) | (data->data[5] << 8));
	voltageValue = ntohs(voltageValue);
	
	ETG_TRACE_USR4(("Recieved Data values in data[0], data[1], data[2], data[3] is %d,%d,%d,%d ",
		data->data[0],data->data[1],data->data[2],data->data[3]));
		
	ETG_TRACE_USR4(("Recieved Data values in data[4], data[5] %d,%d ",
		data->data[4],data->data[5]));
		
	ETG_TRACE_USR4(("Sending  Volatage -- level %d , Sensor %d and Value %d ",
		voltageLevel, voltageSensor, voltageValue));
	
	ETG_TRACE_USR4(("Received Message URL-IP is %s", data->strURL.c_str()));

	sendEventVoltageSignal(envoltageLevel, envoltageSensor, voltageValue, data->strURL);
}

/************************************************************************
*FUNCTION: 		sendGenericRspSetAnnouncementOut(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the RspSetAnnouncementOut
*				broadcast signal to all registered clients
*PARAMETER:	None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Madhuree	06/10/2020
*************************************************************************/
void clAsfGenericServiceAdaptor::sendGenericRspSetAnnouncementOut(stNanoMsgData *data)
{
    ETG_TRACE_USR4(("clAsfGenericServiceAdaptor::sendGenericRspSetAnnouncementOut"));
    //Extract the data
    if(NULL == data)
    {
         ETG_TRACE_USR4(("clAsfGenericServiceAdaptor::sendGenericRspSetAnnouncementOut(): Data Received is NULL"));
         return;
    }
    //extract the data	
    uint16 status = data->data[0];
    ETG_TRACE_USR4(("sendGenericRspSetAnnouncementOut():Recieved Message msgid is = %d ",
		 data->MsgID));
    ETG_TRACE_USR4(("sendGenericRspSetAnnouncementOut(): Recieved Message URL-IP is %s", data->strURL.c_str()));
    sendRspSetAnnouncementOutSignal(status, data->strURL);
}

/************************************************************************
*FUNCTION: 		sendGenericRspGetThermalSignal(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the RspGetThermalSignal
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfGenericServiceAdaptor::sendGenericRspGetThermalSignal(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendGenericRspGetThermalSignal"));
	//Extract the data
	
	

	uint16_t thermalState = ((data->data[0]) | (data->data[1] << 8));
	thermalState = ntohs(thermalState);
	entThermalState enthermalState = static_cast<entThermalState>(thermalState);
	
	
	uint16_t thermalSensorID = ((data->data[2]) | (data->data[3] << 8));
	thermalSensorID = ntohs(thermalSensorID);
	entThermalSensorID enthermalSensorID = static_cast<entThermalSensorID>(thermalSensorID);
	
	
	int16 temperature = ((data->data[4]) | (data->data[5] << 8));
	temperature = ntohs(temperature);
	
	ETG_TRACE_USR4(("Recieved Data values in data[0], data[1], data[2], data[3] is %d,%d,%d,%d ",
		data->data[0],data->data[1],data->data[2],data->data[3]));
		
	ETG_TRACE_USR4(("Recieved Data values in data[4], data[5] %d,%d ",
		data->data[4],data->data[5]));
		
	ETG_TRACE_USR4(("Sending  Thermal -- State %d , Sensor %d and temperature %d ",
		thermalState, thermalSensorID, temperature));
		
	ETG_TRACE_USR4(("Received Message URL-IP is %s", data->strURL.c_str()));

	sendRspGetThermalStateSignal(enthermalState, enthermalSensorID, temperature, data->strURL);
}

/************************************************************************
*FUNCTION: 		sendGenericEventThermalSignal(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the EventThermalStateSignal
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
*************************************************************************/
void clAsfGenericServiceAdaptor::sendGenericEventThermalSignal(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("sendGenericEventThermalSignal"));
	//Extract the data
	
	

	uint16_t thermalState = ((data->data[0]) | (data->data[1] << 8));
	thermalState = ntohs(thermalState);
	entThermalState enthermalState = static_cast<entThermalState>(thermalState);
	
	
	uint16_t thermalSensorID = ((data->data[2]) | (data->data[3] << 8));
	thermalSensorID = ntohs(thermalSensorID);
	entThermalSensorID enthermalSensorID = static_cast<entThermalSensorID>(thermalSensorID);
	
	
	int16 temperature = ((data->data[4]) | (data->data[5] << 8));
	temperature = ntohs(temperature);
	
	ETG_TRACE_USR4(("Recieved Data values in data[0], data[1], data[2], data[3] is %d,%d,%d,%d ",
		data->data[0],data->data[1],data->data[2],data->data[3]));
		
	ETG_TRACE_USR4(("Recieved Data values in data[4], data[5] %d,%d ",
		data->data[4],data->data[5]));
		
	ETG_TRACE_USR4(("Sending  Thermal -- State %d , Sensor %d and temperature %d ",
		thermalState, thermalSensorID, temperature));
	
	ETG_TRACE_USR4(("Received Message URL-IP is %s", data->strURL.c_str()));

	sendEventThermalStateSignal(enthermalState, enthermalSensorID, temperature, data->strURL);
}

/************************************************************************
*FUNCTION: 		sendGenericRspCMGSvivelPositionSignal(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the sig_RspCMGSvivelPosition
*				broadcast signal to all registered clients
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	Madhuree	16/10/2020
*************************************************************************/
void clAsfGenericServiceAdaptor::sendGenericRspCMGSvivelPositionSignal(stNanoMsgData *data)
{
	ETG_TRACE_USR4(("clAsfGenericServiceAdaptor::sendGenericRspCMGSvivelPositionSignal"));
	if(NULL == data)
	{
		ETG_TRACE_USR4(("clAsfGenericServiceAdaptor::sendGenericRspCMGSvivelPositionSignal() : Data Received is NULL"));
		return;
	}
	//extract the data	
        
	uint16 status = data->data[0];
	
	ETG_TRACE_USR4(("sendGenericRspCMGSvivelPositionSignal(): status is :%d,Recieved Message URL-IP is %s", status, data->strURL.c_str()));
        ETG_TRACE_USR4(("sendGenericRspCMGSvivelPositionSignal():Recieved Message msgid is = %d ",
		 data->MsgID));
	sendSig_RspCMGSvivelPositionSignal(status, data->strURL);
}
/************************************************************************
*FUNCTION: 		sendGenericEventCMGSwivelPositionSignal(stNanoMsgData *data)
*DESCRIPTION: 	This is a callback function from Core, this will send the sig_EventCMGSvivelPosition
*				broadcast signal to all registered clients. This event is sent from subsystem
*               when swivel position has changed.
*PARAMETER:		Data to be sent, obtained from subsystem
*RETURNVALUE: 	None
*HISTORY:       Dhanya Elza Sunny (MS/ECG5-XC) 02/06/2022  Initial version
*************************************************************************/
void clAsfGenericServiceAdaptor::sendGenericEventCMGSwivelPositionSignal(stNanoMsgData *data)
{
    ETG_TRACE_USR4(("clAsfGenericServiceAdaptor::sendGenericEventCMGSwivelPositionSignal"));
    if(NULL == data)
    {
        ETG_TRACE_USR4(("clAsfGenericServiceAdaptor::sendGenericEventCMGSwivelPositionSignal() : Data Received is NULL"));
        return;
    }
    //extract the data

    entSwivelPosition enSwivelPos = static_cast<entSwivelPosition>(data->data[0]);

    ETG_TRACE_USR4(("sendGenericEventCMGSwivelPositionSignal(): SwivelPosition is :%d,Recieved Message URL-IP is %s", enSwivelPos, data->strURL.c_str()));
    ETG_TRACE_USR4(("sendGenericEventCMGSwivelPositionSignal():Recieved Message msgid is = %d ",
         data->MsgID));
    sendSig_EventCMGSwivelPositionSignal(enSwivelPos, data->strURL);
}
}}
