/* ***************************************************************************************
 * FILE:          NanoMsgProtocolSetup.h
 * SW-COMPONENT:  NanoMsg_Master_Application
 * DESCRIPTION:  NanoMsgProtocolSetup.h is part of NanoMsg_MasterApplication
 * 				 This is the protocol part of the application, has the nanomsg library interface implementation
 * COPYRIGHT:  (c) 2020-21 	Robert Bosch Car Multimedia GmbH
 * HISTORY: 
 * AUTHOR:  Shivakumar J(RBEI/ECH2)  12.02.2020
 * Revision:  Shivakumar J(RBEI/ECH2)  27.03.2020 
 *				Review comments incorporated
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */

#include "NanoMsgProtocolSetup.h"
#include "plugin_trace.h"
#include "protocoldefines.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_NANOMSGMASTERPROTOCOL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_NAVRES6
#include "trcGenProj/Header/NanoMsgProtocolSetup.cpp.trc.h"
#endif
void fatal(const char *func)
{
        fprintf(stderr, "%s: %s\n", func, nn_strerror(nn_errno()));
        ETG_TRACE_USR4((" fatal: %s",nn_strerror(nn_errno())));
}

/************************************************************************
*FUNCTION: 		clConnection()
*DESCRIPTION:   Constructor of the Connection class which calls nn_connect or nn_bind with the URL
*PARAMETER:		int socket id, string Url, type = n_connect or nn_bind
*RETURNVALUE: 	None
*History:		Shiva Kumar J (RBEI/ECH2)   Initial version
*Revision:      Shiva Kumar J(RBEI/ECH2) Added connection status to know if connection is sucess
*************************************************************************/
clConnection::clConnection(int socket_id, std::string const& url, Type type)
    : socket_id(socket_id)
    , _connection_id(type == Type::Connect ? nn_connect(socket_id, url.c_str()) : nn_bind(socket_id, url.c_str()))
{
	ETG_TRACE_USR4(("Connection construtor socket-id: %d,_connection_id: %d, url %s ",socket_id,_connection_id, url.c_str()));

		if((_connection_id > 0) && (socket_id >= 0))
		{
			m_bconnectionStatus = true;
		}
		else
		{
			m_bconnectionStatus = false;
		}

}


/************************************************************************
*FUNCTION: 		~Connection()
*DESCRIPTION:   Destrcutor of the Connection class
*PARAMETER:		void *
*RETURNVALUE: 	None
*History:		Shiva Kumar J (RBEI/ECH2)   Initial version
*************************************************************************/
clConnection::~clConnection() { /*nn_shutdown(socket_id, connection_id);*/
 }

/************************************************************************
*FUNCTION: 		clSocket()
*DESCRIPTION:   Constrctor of the clSocket class calls nn_socket to get the socket_id
*PARAMETER:		int Type -- which protocol to be used eg:NN_PUB, NN_PAIR, NN_REQ etc..
*RETURNVALUE: 	None
*History:		Shiva Kumar J (RBEI/ECH2)   Initial version
*************************************************************************/
clSocket::
clSocket(int type)
    : socket_id(nn_socket(AF_SP, type))
{
	ETG_TRACE_USR4(("Socket constructor called type: %d, socket_id: %d",type, socket_id));
}

/************************************************************************
*FUNCTION: 		~clSocket()
*DESCRIPTION:   Destrcutor of the clSocket class which calls nn_close for the created socket_id
*PARAMETER:		void *
*RETURNVALUE: 	None
*History:		Shiva Kumar J (RBEI/ECH2)   Initial version
*************************************************************************/
clSocket::~clSocket() 
{ 
	if(socket_id >= 0)
		nn_close(socket_id); 
}

/************************************************************************
*FUNCTION: 		bind()
*DESCRIPTION:   binds the sokcet to URL..binds to end station
*PARAMETER:		std::string 
*RETURNVALUE: 	bool states bind is success
*History:		Shiva Kumar J (RBEI/ECH2)   Initial version
*Revision:      Shiva Kumar J (RBEI/ECH2) added return value to bool
*************************************************************************/
bool clSocket::bind(const std::string& url) {
	ETG_TRACE_USR4(("Socket::bind,socket_id : %d, url : %s,",socket_id, url.c_str()));
    connection.reset(new clConnection(socket_id, url, clConnection::Type::Bind));
	return connection.get()->getConnectionStatus(); // returns true if connection is success else false
}

/************************************************************************
*FUNCTION: 		connect()
*DESCRIPTION:   connects the sokcet to URL..connect to end station
*PARAMETER:		std::string 
*RETURNVALUE: 	bool states connect is success
*History:		Shiva Kumar J (RBEI/ECH2)   Initial version
*Revision:      Shiva Kumar J (RBEI/ECH2) added return value to bool
*************************************************************************/
bool clSocket::connect(const std::string& url) {
	ETG_TRACE_USR4(("Socket::connect,socket_id : %d, url : %s,",socket_id, url.c_str()));
    connection.reset(new clConnection(socket_id, url, clConnection::Type::Connect));
	return connection.get()->getConnectionStatus(); // returns true if connection is success else false
}

/************************************************************************
*FUNCTION: 		setSocketOptionSender()
*DESCRIPTION:   Set socket option for sending the message with timeout
*PARAMETER:		void *
*RETURNVALUE: 	int < 0 states options are not set
*History:		Shiva Kumar J (RBEI/ECH2)   Initial version
*Revision:      Shiva Kumar J (RBEI/ECH2) connection status is checked
*************************************************************************/
int clSocket::setSocketOptionSender()
{
	if (!(connection.get()->getConnectionStatus())){
  		ETG_TRACE_FATAL(("Socket is not connected"));
		return -1;
	}
    int rec_timeout = 1000;
	int sen_timeout = 100;
   
    if (nn_setsockopt(socket_id, NN_SOL_SOCKET, NN_RCVTIMEO, &rec_timeout,sizeof (rec_timeout)) < 0) {
        return -1;
    }

    if (nn_setsockopt(socket_id, NN_SOL_SOCKET, NN_SNDTIMEO, &sen_timeout,sizeof (sen_timeout)) < 0) {
        return -1;
    }
    return 0;
}

/************************************************************************
*FUNCTION: 		setSocketOptionReciever()
*DESCRIPTION:   Set socket option for Receiving the message with timeout
*PARAMETER:		void *
*RETURNVALUE: 	int < 0 states options are not set
*History:		Shiva Kumar J (RBEI/ECH2)   Initial version
*Revision:      Shiva Kumar J (RBEI/ECH2) connection status is checked
*************************************************************************/
int clSocket::setSocketOptionReciever()
{
	if (!(connection.get()->getConnectionStatus())){
  		ETG_TRACE_FATAL(("Socket is not connected"));
		return -1;
	}
    int timeout = 10000;
	if (nn_setsockopt(socket_id, NN_SOL_SOCKET, NN_RCVTIMEO, &timeout,sizeof (timeout)) < 0) {
	    //fatal("nn_setsockopt");
	    return -1;
	}
	return 0;
}

/************************************************************************
*FUNCTION: 		disconnect()
*DESCRIPTION:   removes the managed object from unique pointer
*PARAMETER:		void *
*RETURNVALUE: 	void
*History:		Shiva Kumar J (RBEI/ECH2)   Initial version
*************************************************************************/
void clSocket::disconnect() {
    connection.reset();
}

/************************************************************************
*FUNCTION: 		send()
*DESCRIPTION:   calls nn_send to transfer message over network
*PARAMETER:		char * , int
*RETURNVALUE: 	bool
*History:		Shiva Kumar J (RBEI/ECH2)   Initial version
*Revision:		Shiva Kumar J (RBEI/ECH2) return type changes to bool from void
*Revision:      Shiva Kumar J (RBEI/ECH2) connection status is checked
*************************************************************************/
bool clSocket::send(const tNanoDataPack message, int MessageSize, int &send_bytes) {
    if (!(connection.get()->getConnectionStatus())){
  		ETG_TRACE_FATAL(("Socket is not connected"));
		return false;
	}
	
    send_bytes = nn_send(socket_id, &message, MessageSize, 0);
	
    //todo
	if (send_bytes <= 0) 
	{
        ETG_TRACE_USR4(("Socket send sent failed"));
        return false;
	}
		
	ETG_TRACE_USR4(("Socket send sent bytes is %d", send_bytes));
	//if(bytes != sizeof(struct tNanoDataPack))
	//	ETG_TRACE_FATAL(("Sent less than expected")); // retry not done in SOP1
	return true;
}

/************************************************************************
*FUNCTION: 		receive()
*DESCRIPTION:   calls nn_recv to receive message over network
*PARAMETER:		 void
*RETURNVALUE: 	char *
*History:		Shiva Kumar J (RBEI/ECH2)   Initial version
*Revision:      Supriya S(RBEI/ECG5) added error conditions if bytes received is not correct
*Revision:      Shiva Kumar J (RBEI/ECH2) connection status is checked and buff!=NULL
*************************************************************************/
unsigned char * clSocket::receive(int &recv_bytes) {
    if (!(connection.get()->getConnectionStatus())){
  		ETG_TRACE_FATAL(("Socket is not connected"));
		return NULL;
	}
    unsigned char* buf = NULL;
	
    recv_bytes = nn_recv(socket_id, &buf, NN_MSG, 0);
	if(recv_bytes <= 0)
	{
		ETG_TRACE_USR4(("Socket receive recv_bytes <= 0, setting buffer to NULL"));
        fatal("nn_recv");
		buf = NULL;
	}
	if(NULL != buf)
	{		//Todo revisit and check if can be done on receiver classes
			/* based on the way the received message is processed, 
			it is exepcted that the bytes received is atleast >=INDEX_RAW_DATA (payload could be 0 length), 
			hence this condition will check this */
			//(buf[0] != NULL) buff[0] is a first element which can be 0 since it represents version and not an pointer to check for NULL
			if((recv_bytes < INDEX_RAW_DATA) || ((recv_bytes >= 1) /* && (buf[0] != NULL) */ && (buf[0] == 0xff)))
			{
				ETG_TRACE_USR4(("Socket:: Msg cannot be processed(recv_bytes < INDEX_RAW_DATA) %d", recv_bytes));
				ETG_TRACE_USR4(("Socket:: Msg cannot be processed (buf[0] == 0xff) %x", buf[0]));
				freeBuffer(buf);
				buf = NULL;
			}
	}
	else
	{
		ETG_TRACE_USR4(("Received bytes are NULL %d", recv_bytes));
	}
    return buf;
}

/************************************************************************
*FUNCTION: 		freeBuffer()
*DESCRIPTION:   calls nn_freemsg to clear the buffer
*PARAMETER:		char * 
*RETURNVALUE: 	bool
*History:		Shiva Kumar J (RBEI/ECH2)  Initial version
*Revision:      Shiva Kumar J (RBEI/ECH2) Added checks and traces
*************************************************************************/
bool clSocket::freeBuffer(unsigned char * buf){
	bool result = false;
	if(buf != NULL){
		ETG_TRACE_USR4(("Socket::Buffer cleared in freebuffer"));
		if (0 == nn_freemsg(buf))
		result = true;
	}
    return result;
}

/************************************************************************
*FUNCTION: 		send()
*DESCRIPTION:   calls nn_send to transfer ACK message over network
*PARAMETER:		void
*RETURNVALUE: 	void
*History:		Supriya S(RBEI/ECG5)   Initial version
*************************************************************************/
bool clSocket::SendAck() {

	ETG_TRACE_USR4(("Socket::SendEventMsgAck()"));
	bool bReturn = true;
	uint8_t data = 0xFF;
	//todo need to create a mutex lock
	
	setSocketOptionSender();
	
	int bytes = nn_send(socket_id, &data, 1, 0);
	
	setSocketOptionReciever();

	if (bytes < 0)
	bReturn = false;

	return(bReturn);

}

