/* ***************************************************************************************
 * FILE:          NanoMsgReceiver.cpp
 * SW-COMPONENT:  NanoMsg_Master_Application
 * DESCRIPTION:  clRecvNanoMsg.cpp is part of NanoMsg_MasterApplication
 * 				 This is the protocol part of the application, has the socket receiever implementation
 * COPYRIGHT:  (c) 2020-21 	Robert Bosch Car Multimedia GmbH
 * HISTORY: 
 * AUTHOR:  Shivakumar J(RBEI/ECH2)  
 * REVISION: Supriya S 18.03.2020
 *		Added the extraction of IP URL in the recieve logic
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */


#include "NanoMsgReceiver.h"
#include "plugin_trace.h"
#include <arpa/inet.h>
#include "clMsgIDMapping.h"
#include "DataPacket.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_NANOMSGMASTERPROTOCOL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_NAVRES6
#include "trcGenProj/Header/NanoMsgReceiver.cpp.trc.h"
#endif

clRecvNanoMsg* clRecvNanoMsg::m_poSelf = NULL;

/************************************************************************
*FUNCTION: 		thStartFunction(void *arg);
*DESCRIPTION:   Static Function, Protocol Receiver thread entry point
*PARAMETER:		void *
*RETURNVALUE: 	None
*History:		Supriya Seshadri(RBEI/ECG5)   Initial version
*Revision:      Shiva Kumar J (RBEI/ECH2) changed clRecvNanoMsg from singleton
*************************************************************************/
void* clSocketthread::thStartFunction(void *arg) 
{
		 ETG_TRACE_USR4(("thStartFunction "));
 		 clRecvNanoMsg* p_instance = clRecvNanoMsg::getInstance();

		 if(NULL != p_instance)
		 {
			ETG_TRACE_USR4(("thStartFunction inside while "));
			p_instance->vRecieveinLoop();
			ETG_TRACE_FATAL(("clRecvNanoMsg::Exiting the Protocol Receiver Thread..."));
		 }
}

/************************************************************************
*FUNCTION: 		~clSocketthread()
*DESCRIPTION:   Destructor of the class clSocketthread, 
*PARAMETER:		None
*RETURNVALUE: 	None
*History:		Supriya Seshadri(RBEI/ECG5)   Initial version
*************************************************************************/
clSocketthread::~clSocketthread()
{
	clRecvNanoMsg::vDeleteInstance();
	
}

/************************************************************************
*FUNCTION: 		clSocketthread()
*DESCRIPTION:   Constructor of the class clSocketthread, Cretes the the 
*				protocol receiver thread
*PARAMETER:		None
*RETURNVALUE: 	None
*History:		Supriya Seshadri(RBEI/ECG5)   Initial version
*Revision:      Shiva Kumar J (RBEI/ECH2) done error handling for Thread creation
*************************************************************************/
clSocketthread::clSocketthread()
 {
	ETG_TRACE_USR4(("clSocketthread called "));
	//p_instance = new clRecvNanoMsg();
	pthread_attr_t attr;
	pthread_attr_init(&attr);
	try
	{
		int error = pthread_create(&m_thHandle, NULL,&(clSocketthread::thStartFunction), NULL);
		
			if( error != 0 )
			{
//				setThreadState(enmThStateFault);
				ETG_TRACE_FATAL(("clSocketthread::Thread creation failed!!"));
				return ;	
			}
	}
	catch (char *s)
	{
		exit(-1); //to be commented out when we enable sd_notify
		//sd_notify(0, "ERRNO=2"); // No tested so far, so commented out.
	}
	
}

/************************************************************************
*FUNCTION: 		processRecvMsg()
*DESCRIPTION: 	the receieved socket message is processed and the structure stNanoMsgData is filled
*PARAMETER:		None
*RETURNVALUE: 	unsigned char* buffer, int recv_bytes
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*Revision:		Supriya S(RBEI/ECG5) Added the Data Length correction , IP extarction , 
*Revision:		Supriya S(RBEI/ECG5) Added the recv_bytes and logic to copy payload data only when sufficient
				data is available.
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue and added few more checks
*************************************************************************/
/************************************************************************
Packet structure according to the specification:
Field Name              Byte(s)				Description
 Version				 2                  Protocol version(Not used) 
 SourceIP				 4					IP of sending station(CM-A, CM-P etc..)
 MAC					 6                  MAC adress of sending station (Not used) 
 Length					 2                  MsgID + DevID + DevNr + Data
 DevID					 1					Source DevID
 DevNr					 1                  Source DevNr
 MsgID					 2                  Specific command or event
 Data					 xx(Max 8)	        command specific data
************************************************************************/
//processRecvMsg function is also present in NanoMsgReceiver.
// todo: put this as a helper function in datapacket.cpp
void clRecvNanoMsg::processRecvMsg(unsigned char* buffer, int recv_bytes){
      ETG_TRACE_USR4((" clRecvNanoMsg::processRecvMsg() "));
      stNanoMsgData recvBufferData;
      if (NULL != buffer && recv_bytes > 0) {  			  
	
			//if we have come to this point, it means recv_bytes >= INDEX_RAW_DATA
			//So memcpy can be done till buffer[INDEX_RAW_DATA]
			
			/*****copy source IP to recvBufferData***********************************/
			uint8_t URL[4];
			memcpy(URL, (buffer + INDEX_RAW_IP), DATA_SIZE_FOUR_BYTE);
			clDataPacket::vPackURL(URL, recvBufferData.strURL);
			ETG_TRACE_USR4(("Msg received IP is %s", recvBufferData.strURL.c_str())); 
			/*************************************************************************/
			
			/*****copy MSGID to recvBufferData***********************************/
            memcpy(&recvBufferData.MsgID, (buffer + INDEX_RAW_MSGID), DATA_SIZE_TWO_BYTE);
			recvBufferData.MsgID = ntohs(recvBufferData.MsgID);
			ETG_TRACE_USR4(("Msg received MsgID is %d", recvBufferData.MsgID )); 
			/*************************************************************************/
			
			/*****copy DevID and DevNr to recvBufferData***********************************/
            recvBufferData.DevID = buffer[INDEX_DEV_ID];
            recvBufferData.DevNr = buffer[INDEX_DEV_NUMBER];
			/*************************************************************************/
			
			/*****copy DataLength to recvBufferData***********************************/
			memcpy(&recvBufferData.DataLength, (buffer + INDEX_RAW_LENGTH), DATA_SIZE_TWO_BYTE);
            recvBufferData.DataLength = ntohs(recvBufferData.DataLength);
			ETG_TRACE_USR4(("Msg received DataLength is after ntohs %d", recvBufferData.DataLength )); 
			/*************************************************************************/
			
			/*****Data length correction if not received as expected****************/
			if(recvBufferData.DataLength >= DATA_HEADER_LENGTH)
			recvBufferData.DataLength = recvBufferData.DataLength - DATA_HEADER_LENGTH;
						
			if((recvBufferData.DataLength > MAX_PAYLOAD_LENGTH) || (recv_bytes < (INDEX_RAW_DATA+ recvBufferData.DataLength))){
			ETG_TRACE_USR4(("Recieved payload is not supported %d", recvBufferData.DataLength ));
			return;			
			}
			/*************************************************************************/
			
			/*****copy Data to recvBufferData***********************************/
			if((recvBufferData.DataLength > 0) && (recv_bytes >= INDEX_RAW_DATA+ recvBufferData.DataLength))
			memcpy(recvBufferData.data, &buffer[INDEX_RAW_DATA], recvBufferData.DataLength);
            /*************************************************************************/

            /*************************Print the whole buffer**************************/
			for(int itr = 0; itr < recv_bytes ; itr++)
                  ETG_TRACE_USR4(("value at buffer[%d] is %d", itr, buffer[itr] ));
			/*************************************************************************/
			
			bool beventSuccess = false;
            if(NULL != m_pCoreReceiver)
			beventSuccess = m_pCoreReceiver->setNanoMsgEvent(recvBufferData);
		
			if (!beventSuccess){
				ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
				//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
			}
      }
  }

/************************************************************************
*FUNCTION: 		processRecvMsg()
*DESCRIPTION:   Thread while(1) of the protocol receiver thread implementation 
				For socket receive
*PARAMETER:		None
*RETURNVALUE: 	None
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*Revision:      Shiva Kumar J(RBEI/ECH2)   added some more checks
*************************************************************************/
void clRecvNanoMsg::vRecieveinLoop()
{
    ETG_TRACE_USR4(("clRecvNanoMsg::vRecieveinLoop"));
    std::string login_socket_endpoint{ "tcp://192.168.0.1:2227" };

    bool bindstatus = FALSE;
    setThreadRunStatus(true);
    int temp_Msg = 0;
    int recv_bytes = 0;
    while (getThreadRunStatus()) {

        if (FALSE == bindstatus)
        {
            clSocket recv_socket(NN_PAIR);
            bindstatus = recv_socket.bind(login_socket_endpoint);
            if(!bindstatus)
            {
                ETG_TRACE_FATAL(("clRecvNanoMsg::vRecieveinLoop Binding Failed"));
                // No re-try mechanism added in SOP1
                sleep(50);
                continue;
            }
            else
            {
              ETG_TRACE_USR4(("clRecvNanoMsg::vRecieveinLoop Binding Succeesful"));
            }
            int rcv = recv_socket.setSocketOptionReciever();
            ETG_TRACE_USR4(("clRecvNanoMsg::vRecieveinLoop setSocketOptionReciever : %d", rcv));
            while (bindstatus)
            {
                recv_bytes = 0;
				unsigned char* recv_packet;
                //This is a blocking call.
                recv_packet = recv_socket.receive(recv_bytes);
                if ((recv_packet != NULL) && (recv_bytes > 0) ) {
                    memcpy(&temp_Msg, (recv_packet + INDEX_RAW_MSGID), DATA_SIZE_TWO_BYTE);
                    temp_Msg = ntohs(temp_Msg);
                    // check if the MSGID is of type event and send ACK
                    if (SendAckForEventMessage(temp_Msg)){
                        if(!recv_socket.SendAck())
                        {
                            ETG_TRACE_FATAL(("sending ACK failed"));
                        }
                    }

                     processRecvMsg(recv_packet,recv_bytes);
                     //free buffer after data read
                    if(!recv_socket.freeBuffer(recv_packet)){
                        ETG_TRACE_USR4(("clNanoMsgSender::Buffer NULL"));
                    }
                    
                }
                else
                {
                    recv_socket.disconnect();
                    ETG_TRACE_USR4(("clRecvNanoMsg::vRecieveinLoop Socket Disconnected"));
                    bindstatus = FALSE;
                }
            }
        }
    }
}
/************************************************************************
*FUNCTION: 		clRecvNanoMsg()
*DESCRIPTION:   Constructor of the clRecvNanoMsg class
*PARAMETER:		None
*RETURNVALUE: 	None
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
clRecvNanoMsg::clRecvNanoMsg():m_pCoreReceiver(NULL),m_threadRunStatus(false)
{
	m_pCoreReceiver = clReceiverThread::getInstance();
	ETG_TRACE_USR4(("clRecvNanoMsg Constructor"));
}

/************************************************************************
*FUNCTION: 		clRecvNanoMsg()
*DESCRIPTION:   Destructor of the clRecvNanoMsg class
*PARAMETER:		None
*RETURNVALUE: 	None
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
clRecvNanoMsg::~clRecvNanoMsg() {
	m_pCoreReceiver = NULL;
	setThreadRunStatus(false);
}

/************************************************************************
*FUNCTION: 		SendAckForEventMessage()
*DESCRIPTION:   Function checked if the Message ID is an event or not (Response)
*PARAMETER:		None
*RETURNVALUE: 	None
*History:		Supriya Seshadri(RBEI/ECG5)   Initial version
*************************************************************************/
bool clRecvNanoMsg::SendAckForEventMessage(int msgid) {

//check for Events and send a ACK
bool bAckNeeded = FALSE;
	clMsgIDMapping *IDMappingInstance = clMsgIDMapping::getInstance();
	if (NULL != IDMappingInstance)
		{
			ETG_TRACE_USR4((" clRecvNanoMsg::processRecvMsg() checking if ACK is needed"));
			//check if ACK is needed
			bAckNeeded = IDMappingInstance->bcheckforEvent(msgid);
			}

return 	 bAckNeeded;
}

/************************************************************************
*FUNCTION: 		getInstance()
*DESCRIPTION:   Static Function, Singleton implementation, return the ptr of the
*				Singleton class clRecvNanoMsg
*PARAMETER:		None
*RETURNVALUE: 	clRecvNanoMsg *
*History:		Supriya Seshadri(RBEI/ECG5)   Initial version
*************************************************************************/
clRecvNanoMsg* clRecvNanoMsg::getInstance()
{
	ETG_TRACE_USR4((" clRecvNanoMsg::getInstance() "));
	if (NULL == m_poSelf)
	{
		m_poSelf = new clRecvNanoMsg();
	}
	return m_poSelf;
}


/************************************************************************
*FUNCTION: 		vDeleteInstance()
*DESCRIPTION:   Singleton implementation, Deletes the singleton
*				object of the class clSocketthread
*PARAMETER:		None
*RETURNVALUE: 	None
*History:		Supriya Seshadri(RBEI/ECG5)   Initial version
*************************************************************************/
void clRecvNanoMsg::vDeleteInstance()
{
	if (m_poSelf != NULL)
	{		
		ETG_TRACE_USR4(("clRecvNanoMsg::vDeleteInstance()"));
		delete m_poSelf;
		m_poSelf = NULL;
	}
}


