/* ***************************************************************************************
 * FILE:          clMsgIDMapping.cpp
 * SW-COMPONENT:  NanoMsg_Master_Application
 * DESCRIPTION:  clMsgIDMapping.cpp is part of NanoMsg_MasterApplication
 * 				 This class would have msgid mapping table mapped to function pointers
 * COPYRIGHT:  (c) 2020-21 	Robert Bosch Car Multimedia GmbH
 * HISTORY: 
 * AUTHOR:  Shivakumar J(RBEI/ECH2)  12.02.2020
 *				Initial version
 * Revision: Supriya S(RBEI/ECG5) 16.04.2020
 *				Added Digital Input CMP and CMG messages.
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */


/*****************************************************************
| includes
|----------------------------------------------------------------*/

#include "clMsgIDMapping.h"
#include "plugin_trace.h"
#include "AsfLCMServiceAdaptor.h"
#include "AsfIntercomServiceAdaptor.h"
#include "AsfGenericServiceAdaptor.h"

using namespace NanoMsg_MasterApp::NanoMsgServer;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_NANOMSGMASTERCMPCORE
#include "trcGenProj/Header/clMsgIDMapping.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN
#include <iostream>

 clMsgIDMapping* clMsgIDMapping::m_poSelf = NULL;
/************************************************************************
*FUNCTION: 		getInstance()
*DESCRIPTION:   Static Function, Singleton implementation, return the ptr of the
*				Singleton class clMsgIDMapping
*PARAMETER:		None
*RETURNVALUE: 	clMsgIDMapping*
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
clMsgIDMapping* clMsgIDMapping::getInstance()
{
   if (NULL == m_poSelf)
   {
      m_poSelf = new clMsgIDMapping();
   }
   return m_poSelf;
}

/************************************************************************
*FUNCTION: 		vDeleteInstance()
*DESCRIPTION:   Static Function, Singleton implementation, Deletes the Singleton Object 
*PARAMETER:		None
*RETURNVALUE: 	None
*History:		Supriya Seshadri(RBEI/ECG5)   Initial version
*************************************************************************/
void clMsgIDMapping::vDeleteInstance()
{
   if (NULL != m_poSelf)
   {
	   ETG_TRACE_USR4(("clMsgIDMapping::vDeleteInstance(), deleting the obj"));
		delete m_poSelf;
		m_poSelf = NULL;
   }
}

/************************************************************************
*FUNCTION: 		clMsgIDMapping()
*DESCRIPTION:   Constructor
*PARAMETER:		None
*RETURNVALUE: 	None
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
clMsgIDMapping::clMsgIDMapping():m_MsgType(enmMsgType_None){

	setSMap();
	setReqRspMap();
}

/************************************************************************
*FUNCTION: 		~clMsgIDMapping()
*DESCRIPTION:   Destructor
*PARAMETER:		None
*RETURNVALUE: 	None
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
clMsgIDMapping::~clMsgIDMapping(){
  m_MsgType = enmMsgType_None;
}

/************************************************************************
*FUNCTION: 		setSMap()
*DESCRIPTION:   Static Map with Key as MSGID and value to differentiate component
*				EG: LCM, Intercom etc..Ref: CIS system message definition_v0.8.4
*PARAMETER:		None
*RETURNVALUE: 	void
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*Revision		Supriya Seshadri(RBEI/ECG5) Unblocked DigitalInput CMP, CMG messages
*************************************************************************/
void clMsgIDMapping::setSMap(){

	//General MSgG's
	m_sMap[0x0001] = enmMsgType_Request_LCM;
	m_sMap[0x0002] = enmMsgType_Response_LCM;
	//LCM MSG's
	m_sMap[0x0003] = enmMsgType_Request_LCM;
	m_sMap[0x0004] = enmMsgType_Response_LCM;
	m_sMap[0x0005] = enmMsgType_Request_LCM;
	m_sMap[0x0006] = enmMsgType_Response_LCM;
	m_sMap[0x0007] = enmMsgType_Event_LCM;
	m_sMap[0x0008] = enmMsgType_Request_LCM;
	m_sMap[0x0009] = enmMsgType_Response_LCM;
	m_sMap[0x0010] = enmMsgType_Request_LCM;
	m_sMap[0x0011] = enmMsgType_Response_LCM;
	m_sMap[0x0018] = enmMsgType_Request_LCM;
	m_sMap[0x0019] = enmMsgType_Response_LCM;
	//VOltage MSG's
	m_sMap[0x0012] = enmMsgType_Event_GENERIC;
	m_sMap[0x0013] = enmMsgType_Request_GENERIC;
	m_sMap[0x0014] = enmMsgType_Response_GENERIC;
	m_sMap[0x0015] = enmMsgType_Event_GENERIC;
	m_sMap[0x0016] = enmMsgType_Request_GENERIC;
	m_sMap[0x0017] = enmMsgType_Response_GENERIC;
        m_sMap[0x0020] = enmMsgType_Request_GENERIC;
        m_sMap[0x0021] = enmMsgType_Response_GENERIC;
        //Healthcare MSG's
        m_sMap[0x0030] = enmMsgType_Request_LCM;
	m_sMap[0x0031] = enmMsgType_Response_LCM;
        m_sMap[0x0032] = enmMsgType_Event_LCM;
	//Thermal MSG's
	m_sMap[0x0043] = enmMsgType_Event_GENERIC;
	m_sMap[0x0044] = enmMsgType_Request_GENERIC;
	m_sMap[0x0045] = enmMsgType_Response_GENERIC;
	//Intercom MSG's
	m_sMap[0x0100] = enmMsgType_Request_INTERCOM;
	m_sMap[0x0101] = enmMsgType_Response_INTERCOM;
	m_sMap[0x0102] = enmMsgType_Event_INTERCOM;
	m_sMap[0x0103] = enmMsgType_Event_INTERCOM;
	m_sMap[0x0104] = enmMsgType_Event_INTERCOM;
	m_sMap[0x0105] = enmMsgType_Event_INTERCOM;
	m_sMap[0x0106] = enmMsgType_Event_INTERCOM;
	m_sMap[0x0107] = enmMsgType_Event_INTERCOM;
	m_sMap[0x0108] = enmMsgType_Event_INTERCOM;
	m_sMap[0x0109] = enmMsgType_Blocked;
	m_sMap[0x010A] = enmMsgType_Blocked;
	m_sMap[0x010B] = enmMsgType_Request_INTERCOM;
	m_sMap[0x010C] = enmMsgType_Request_INTERCOM;
	m_sMap[0x010D] = enmMsgType_Request_INTERCOM;
	m_sMap[0x0110] = enmMsgType_Blocked;
	m_sMap[0x0111] = enmMsgType_Blocked;
	m_sMap[0x0112] = enmMsgType_Blocked;
        m_sMap[0x0113] = enmMsgType_Event_INTERCOM;
        m_sMap[0x0114] = enmMsgType_Request_INTERCOM;
        m_sMap[0x0115] = enmMsgType_Response_INTERCOM;
//Digitial Input CMP, CMG	
	m_sMap[0x0050] = enmMsgType_Event_GENERIC;
	m_sMap[0x0051] = enmMsgType_Request_GENERIC;
	m_sMap[0x0052] = enmMsgType_Response_GENERIC;
	m_sMap[0x0060] = enmMsgType_Event_GENERIC;
	m_sMap[0x0061] = enmMsgType_Request_GENERIC;
	m_sMap[0x0062] = enmMsgType_Response_GENERIC;	
        m_sMap[0x0063] = enmMsgType_Event_GENERIC;
        m_sMap[0x0064] = enmMsgType_Request_GENERIC;
        m_sMap[0x0065] = enmMsgType_Response_GENERIC;
	
	m_sMap[0x0070] = enmMsgType_Blocked;
	m_sMap[0x0071] = enmMsgType_Blocked;
	m_sMap[0x0072] = enmMsgType_Blocked;
}

/************************************************************************
*FUNCTION: 		setReqRspMap()
*DESCRIPTION:   Static Map with Key Req MsgID and second parameter as
*				RspMsgID. Only Msg Ids which has just status as return value
*               is added.
*PARAMETER:		None
*RETURNVALUE: 	void
*History:		Dhanya Elza Sunny (MS/ECG5-XC) 02/06/2022  Initial version
*Revision
*************************************************************************/
void clMsgIDMapping::setReqRspMap(){

    m_ReqRspMap[0x0003] = 0x0004;  //ReqSetPowerMode
    m_ReqRspMap[0x0010] = 0x0011;  //ReqShutdown
    m_ReqRspMap[0x0018] = 0x0019;  //ReqSetChainMode
    m_ReqRspMap[0x0020] = 0x0021;  //ReqSetAnnouncementOut
    m_ReqRspMap[0x0063] = 0x0064;  //ReqCMGSwivelPosition
}


/************************************************************************
*FUNCTION: 		bcheckforEvent()
*DESCRIPTION:   checks if Msgid is of type event to send ACK or to process response
*PARAMETER:		MsgID
*RETURNVALUE: 	bool
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
bool clMsgIDMapping::bcheckforEvent(int MsgID)
{
	
	if(m_sMap.empty()){
		setSMap();
	}
	bool bEvent = FALSE;
	if(clMsgIDMapping::m_sMap.count(MsgID) <= 0){
		ETG_TRACE_USR4(("clMsgIDMapping::bcheckforEvent  MSGid %d Not valid",MsgID));
		return bEvent;
	}	
	//todo
	//check for out of range Mesgids and default return
	//bool bEvent = FALSE;
	switch(clMsgIDMapping::m_sMap.at(MsgID))
	{
		case enmMsgType_Event_GENERIC:
		case enmMsgType_Event_INTERCOM:
		case enmMsgType_Event_LCM:
				bEvent = TRUE;
				break;
		default:
		 break;
	}
	
	return bEvent;
}

/************************************************************************
*FUNCTION: 		u16GetRspMsgID(()
*DESCRIPTION:   Value of Rsp MsgID correposnding to the ReqMsgID is sent
*PARAMETER:		MsgID
*RETURNVALUE: 	uint16_t (MsgID)
*History:		Dhanya Elza Sunny (MS/ECG5-XC) 02/06/2022  Initial version
*************************************************************************/
uint16_t clMsgIDMapping::u16GetRspMsgID(uint16_t MsgID)
{
	uint16_t u16RetVal = 0xFFFF;
	if(m_ReqRspMap.empty()){
		setReqRspMap();
	}	
    map<uint16_t, uint16_t>::iterator it;
    it = m_ReqRspMap.find(MsgID);
    if (it != m_ReqRspMap.end())
	{
	  	u16RetVal = it->second;
	}
    ETG_TRACE_USR4(("clMsgIDMapping::u16GetRspMsgID  ReqMSGid %d RspMSGid %d",MsgID, u16RetVal));
	return u16RetVal;
}

/************************************************************************
*FUNCTION: 		setCallback()
*DESCRIPTION:   function which fills map with Key-MsgID, Value-functionPtr
*				to find a call back for events and responses				
*PARAMETER:		MsgID
*RETURNVALUE: 	bool
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*Revision:      Shiva Kumar J(RBEI/ECH2) Null check added
*************************************************************************/
void clMsgIDMapping::setCallback(int MsgID, FUNCPTR funcptr){

	if(m_sMap.empty()){
		setSMap();
	}
	
	if(clMsgIDMapping::m_sMap.count(MsgID) <= 0){
		ETG_TRACE_USR4(("clMsgIDMapping::setCallback  MSGid: %d Not valid",MsgID));
		return ;
	}
	if(funcptr != NULL)
		clMsgIDMapping::m_mapingTable[MsgID] = funcptr;

}

/************************************************************************
*FUNCTION: 		getInstanceForMsgID()
*DESCRIPTION:   to find the instance of which service component to 
*				be called with function ptr
*PARAMETER:		MsgID
*RETURNVALUE: 	IBackendInterface*
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
IBackendInterface* clMsgIDMapping::getInstanceForMsgID(int MsgID){

	if(m_sMap.empty()){
		setSMap();
	}

	IBackendInterface *backendInstance = NULL;
	if(clMsgIDMapping::m_sMap.count(MsgID) <= 0){
		ETG_TRACE_USR4(("clMsgIDMapping::getInstanceForMsgID  MSGid: %d Not valid",MsgID));
		return backendInstance;
	}	
	switch(clMsgIDMapping::m_sMap.at(MsgID))
	{

	case enmMsgType_Request_LCM :
			// request from CMC to CM-X so no call back. case for future handling if any
		break;
	case enmMsgType_Request_INTERCOM :
	       // request from CMC to CM-X so no call back. case for future handling if any
		break;
	case enmMsgType_Response_LCM:
	case enmMsgType_Event_LCM:
		//return LCM service adapter instance
		backendInstance = clAsfLCMServiceAdaptor::getInstance();
		break;
	case enmMsgType_Response_INTERCOM :
	case enmMsgType_Event_INTERCOM :
		//return Intercom service adapter instance
		backendInstance = clAsfIntercomServiceAdaptor::getInstance();
		break;
	case enmMsgType_Request_GENERIC:
		// request from CMC to CM-X so no call back. case for future handling if any
		break;
	case enmMsgType_Response_GENERIC :
	case enmMsgType_Event_GENERIC :
		//return Generic service adapter instance 
		backendInstance = clAsfGenericServiceAdaptor::getInstance();
		break;
	case enmMsgType_Blocked:
		//currently blocked as per spec, can enable based on handling
		break;
	default:
		break;
	}
	return backendInstance;
}

/************************************************************************
*FUNCTION: 		getFunctionPtr()
*DESCRIPTION:   It finds the funcptr for MsgID passed to make a callback
*PARAMETER:		MsgID
*RETURNVALUE: 	FUNCPTR*
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
FUNCPTR clMsgIDMapping::getFunctionPtr(int MsgID){
	FUNCPTR func = NULL;
	if(!m_mapingTable.empty()){
		if(m_mapingTable.count(MsgID)> 0) // check if funcptr added for MsgID
		{
			func = 	m_mapingTable.at(MsgID);
			ETG_TRACE_USR4(("clMsgIDMapping::getFunctionPtr MSGid: %d is valid",MsgID));
		}
		else{
			ETG_TRACE_USR4(("clMsgIDMapping::getFunctionPtr MSGid: %d Not valid",MsgID));
		}
	}
	return func;
}

