/*
 * Copyright (c) 2010 Gracenote.
 *
 * This software may not be used in any way or distributed without
 * permission. All rights reserved.
 *
 * Some code herein may be covered by US and international patents.
 */

/*
 * gn_phonetic_conversion_utility.c - 
 *
 * Is a dynamic string implementation that grows the memory of the
 * character pointer when adding a new character or character string.
 * It only grows memory when needed.  This was created to encapsulate
 * this behavior to make the conversion algorithm easier to understand.
 */

#include "gn_phonetic_conversion_utility.h"

/* EPAL Headers */
#include "gnepal_memory.h"
#include "gnepal_string.h"

/*
 * Public API Implementation
 */

/* 
 * gn_dynamic_string_create
 */
gnex_error_t
gn_dynamic_string_create(
	gn_dynamic_string_t** dynamic_string,
	gn_uint32_t size
	)
{
	gn_uint32_t buffer_size = 0;

	if ((dynamic_string == GN_NULL) || (*dynamic_string != GN_NULL))
	{
		return GNEX_PHOCVRT_InvalidArg;
	}

	*dynamic_string = (gn_dynamic_string_t*) gnepal_mem_malloc(sizeof(gn_dynamic_string_t));
	if (*dynamic_string == GN_NULL)
	{
		return GNEX_PHOCVRT_NoMemory;
	}

	buffer_size = sizeof(gn_uchar_t) * size + 1;
	(*dynamic_string)->buffer = (gn_uchar_t*) gnepal_mem_malloc(buffer_size);
	if ((*dynamic_string)->buffer == GN_NULL)
	{
		return GNEX_PHOCVRT_NoMemory;
	}

	gnepal_mem_memset((*dynamic_string)->buffer, 0, buffer_size);
	(*dynamic_string)->size = size;
	(*dynamic_string)->used = 0;
	
	return GNEX_PHOCVRT_NoError;
}

/* 
 * gn_dynamic_string_copy
 */
gnex_error_t
gn_dynamic_string_copy(
	gn_dynamic_string_t** dst_dynamic_string,
	const gn_dynamic_string_t* src_dynamic_string
	)
{
	gnex_error_t error = GNEX_PHOCVRT_NoError;

	if (	(dst_dynamic_string == GN_NULL)
		||	(*dst_dynamic_string != GN_NULL)
		||	(src_dynamic_string == GN_NULL)
		)
	{
		return GNEX_PHOCVRT_InvalidArg;
	}

	error = gn_dynamic_string_create(dst_dynamic_string, src_dynamic_string->size);
	if (error == GNEX_PHOCVRT_NoError)
	{
		error = gn_dynamic_string_append_string(*dst_dynamic_string, src_dynamic_string->buffer);
	}
	
	if (error != GNEX_PHOCVRT_NoError)
	{
		gn_dynamic_string_free(dst_dynamic_string);
	}
	
	return error;
}

/* 
 * gn_dynamic_string_free
 */
gnex_error_t
gn_dynamic_string_free(
	gn_dynamic_string_t** dynamic_string
	)
{
	if (dynamic_string == GN_NULL)
	{
		return GNEX_PHOCVRT_InvalidArg;
	}

	if (*dynamic_string != GN_NULL)
	{
		if ((*dynamic_string)->buffer != GN_NULL)
		{
			gnepal_mem_free((*dynamic_string)->buffer);
		}

		gnepal_mem_free(*dynamic_string);
	}
	
	*dynamic_string = GN_NULL;
	return GNEX_PHOCVRT_NoError;
}

/* 
 * gn_dynamic_string_append_segment
 */
gnex_error_t
gn_dynamic_string_append_segment(
	gn_dynamic_string_t* dynamic_string,
	const gn_uchar_t* append,
	gn_uint32_t size
	)
{
	gn_uchar_t* buffer = GN_NULL;
	gn_uint32_t new_size = 0;
	gn_uint32_t alloc_size = 0;

	if ((dynamic_string == GN_NULL) || (dynamic_string->buffer == GN_NULL))
	{
		return GNEX_PHOCVRT_InvalidArg;
	}

	/* Calculate new size and see if we need to grow */
	new_size = (dynamic_string->used + size);
	if (new_size > dynamic_string->size)
	{
		/* adding one for trailing '\0' character */
		alloc_size = new_size + 1;
		buffer = (gn_uchar_t*) gnepal_mem_realloc(dynamic_string->buffer, sizeof(gn_uchar_t) * alloc_size);
		if (buffer == GN_NULL)
		{
			return GNEX_PHOCVRT_NoMemory;
		}

		/* Set up internals of dynamci_string and clear new data */
		dynamic_string->buffer = buffer;
		dynamic_string->size = new_size;
		gnepal_mem_memset(dynamic_string->buffer + dynamic_string->used, 0, sizeof(gn_uchar_t) * (alloc_size - dynamic_string->used));
	}

	/* Append data to dynamic string */
	gnepal_mem_memcpy(dynamic_string->buffer + dynamic_string->used, append, sizeof(gn_uchar_t) * size);
	dynamic_string->used += size;

	return GNEX_PHOCVRT_NoError;
}

/* 
 * gn_dynamic_string_append_string
 */
gnex_error_t
gn_dynamic_string_append_string(
	gn_dynamic_string_t* dynamic_string,
	const gn_uchar_t* append
	)
{
	return gn_dynamic_string_append_segment(dynamic_string, append, gnepal_strlen(append));
}
	
/* gn_dynamic_string_append_char
 *
 */
gnex_error_t
gn_dynamic_string_append_char(
	gn_dynamic_string_t* dynamic_string,
	gn_uchar_t append
	)
{
	return gn_dynamic_string_append_segment(dynamic_string, &append, 1);
}

/* 
 * gn_dynamic_string_clear
 */
gnex_error_t
gn_dynamic_string_clear(
	gn_dynamic_string_t* dynamic_string
	)
{
	if (dynamic_string == GN_NULL)
	{
		return GNEX_PHOCVRT_InvalidArg;
	}
	
	dynamic_string->used = 0;
	if (dynamic_string->buffer != GN_NULL)
	{
		gnepal_mem_memset(dynamic_string->buffer, 0, sizeof(gn_uchar_t) * dynamic_string->size);
	}
	
	return GNEX_PHOCVRT_NoError;
}
