/*
 * Copyright (c) 2010 Gracenote, Inc. All rights reserved.
 *
 * This software may not be used in any way or distributed without
 * Gracenote's written permission. 
 *
 * One or more patents owned by Gracenote apply to this software.
 */

/*
 * gn_phonetic_conversion_data.h - 
 *
 * This contains the the X-SAMPA to L&H+ conversion map and well as
 * the language specific L&H+ post-processing conversion map.  Not all
 * languages need L&H+ post-processing, so some languages don’t contain
 * L&H+ conversion post-processing map.  It also provides two functions
 * to retrieve this data.
 */

#ifndef _GN_PHONETIC_CONVERSION_DATA_H_
#define _GN_PHONETIC_CONVERSION_DATA_H_

#include "gn_abs_types.h"

/* Error-Code Headers */
#include "gnex_errors.h"
#include "gnex_error_values.h"

#ifdef __cplusplus
extern "C"{
#endif

/* supported speech languages */
#define SPEECH_LANG_USA_ENGLISH				0
#define SPEECH_LANG_MEXICO_SPANISH			1
#define SPEECH_LANG_CANADA_FRENCH			2
#define SPEECH_LANG_BRITAIN_ENGLISH			3
#define SPEECH_LANG_FRANCE_FRENCH			4
#define SPEECH_LANG_SPAIN_SPANISH			5
#define SPEECH_LANG_GERMANY_GERMAN			6
#define SPEECH_LANG_ITALY_ITALIAN			7
#define SPEECH_LANG_JAPAN_JAPANESE			8
#define SPEECH_LANG_CHINA_MANDARIN			9
#define SPEECH_LANG_RUSSIA_RUSSIAN			10
#define SPEECH_LANG_NETHERLANDS_DUTCH		11
#define SPEECH_LANG_KOREA_KOREAN			12
#define SPEECH_LANG_BRAZILIAN_PORTUGUESE	13
#define SPEECH_LANG_PORTUGAL_PORTUGUESE		14
#define SPEECH_LANG_TURKEY_TURKISH			15
#define SPEECH_LANG_AUSTRALIA_ENGLISH		16
#define SPEECH_LANG_COUNT					17

/* Phonetic Algorithm ID */
#define	PHONETIC_MAP_ALGORITHM_NONE	0
#define	PHONETIC_MAP_ALGORITHM_1	1
#define	PHONETIC_MAP_ALGORITHM_2	2

/*
 * Type Declarations
 */

/* Phonetic Converion Rule Struct */
typedef struct gn_phonetic_conversion_rule_s
{
	gn_uchar_t* src_token;									/* Source phonetic token */
	gn_uchar_t* trg_token;									/* Target phonetic token */
} gn_phonetic_conversion_rule_t;

/* Phonetic Converion Map Struct */
typedef struct gn_phonetic_conversion_map_s
{
	gn_uint32_t algorithm_id;								/* Algorithm to use during conversion */
	gn_phonetic_conversion_rule_t* rules;					/* Array of rules */
} gn_phonetic_conversion_map_t;

/*
 * API Declarations
 */

/* gn_phonetic_conversion_lnh_map_get
 *
 * Get the language agnostic xsampa to lnh map.
 *
 * Parameters:
 *	OUT		: conversion_map	: An un-initialize Phonetic Conversion Map object
 *  IN		: lang_idx			: language index.
 *
 * Return Values:
 *  On success:
 *   gnex_error_t				: GNEX_PHOCVRT_NoError
 *   conversion_map				: An Phonetic Conversion Map object.
 *
 *	On error:
 *   gnex_error_t				: Gracenote Extension Error Value
 *   conversion_map				: Will not have been modified.
 *
 *  Errors include:
 *   GNEX_PHOCVRT_InvalidArg	: already initialized
 */
gnex_error_t
gn_phonetic_conversion_lnh_map_get(
	gn_phonetic_conversion_map_t** conversion_map,
	gn_uint32_t lang_idx
	);

/* gn_phonetic_conversion_post_process_map_get
 *
 * Get the post process xsampa to lnh map by language string.
 *
 * Parameters:
 *	OUT		: conversion_map	: An un-initialize Phonetic Conversion Map object
 *  IN		: lang_idx			: language index.
 *
 * Return Values:
 *  On success:
 *   gnex_error_t				: GNEX_PHOCVRT_NoError
 *   conversion_map				: An Phonetic Conversion Map object.
 *
 *	On error:
 *   gnex_error_t				: Gracenote Extension Error Value
 *   conversion_map				: Will not have been modified.
 *
 *  Errors include:
 *   GNEX_PHOCVRT_InvalidArg	: already initialized
 */
gnex_error_t
gn_phonetic_conversion_lnh_post_process_map_get(
	gn_phonetic_conversion_map_t** conversion_map,
	gn_uint32_t lang_idx
	);

#ifdef __cplusplus
};
#endif

#endif
