#define __PLACEMENT_NEW_INLINE
#include "GracenoteWrapperInterface.h"
#include "GracenoteWrapperLib.h"

#include <string> // std::string
#include <iostream>
#include <fstream>

#include <stdio.h>      // popen() printf() feof() fgets() pclose() fprintf()
#include <stdlib.h>     // system()
#include <unistd.h>     // chdir() sync()
#include <sys/stat.h>   // mkdir
#include <sys/types.h>  // mkdir

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

// Program version number.
#define VERSION       "1.04"
#define ERR_EXIT_CODE (1)


using namespace std;
class tclGracenoteWrapper GNWL;
static FILE* errLog = 0;
static int iVerbose = 0;

static bool bIsTraceActive(int lvl);


#define TR_CLASS_GAL_UPDATER (TR_COMP_DEMO + 0x10)

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "etg_adit_linux_platform.h"
//#define ETG_I_FILE_PREFIX ccademo_tclApp::
//#define ETG_I_TTFIS_CMD_PREFIX "GAL_"
//#define ETG_I_TRACE_CHANNEL TR_TTFIS_DEMO_SERVER
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GAL_UPDATER // TODO change to separate class
#include "trcGenProj/Header/gal_updater.cpp.trc.h"
#else
#define OSAL_vProcessExit() // remove the OSAL exit if not compiled with OSAL
// ETG PRINTF case
// put verbose level condition into macro
#undef _ETG_TRACE_GEN
//#define _ETG_TRACE_GEN(lvl,format) if(bIsTraceActive(lvl)) { __ETG_TRACE_HEADER(lvl) printf format; __ETG_TRACE_FOOTER; }
#define _ETG_TRACE_GEN(lvl, format) if (bIsTraceActive(lvl)) { __ETG_TRACE_HEADER(lvl) printf format; printf("\n"); }
#endif

/* Typedefs */
typedef struct {
	string checksum;
	string filename;
}MD5SUM_INFO;

typedef struct {
	string srcDir;
	string destDir;
	string workDir;
	string tempDir;
	string lfsId;
	string instMode;
	string traceLevel;
	string removeTempDirContent;
}GAL_ARGS;


/* **************************************************************************
 * Function  bIsTraceActive
 * *************************************************************************/
/**
 * The global variable "iVerbose" is set between 0 and 8 using the -v parameter
 * when the gal_updater_out is started.  Whether tracing should be on can be
 * queried using this function.
 *
 * \param[in] lvl - tracing level to compare
 *
 * \return true if verbose tracing level from command line is high enough
 *
 */
static bool bIsTraceActive(int lvl)
{
	return lvl <= iVerbose;
}

/* **************************************************************************
 * Function  displayExitError
 * *************************************************************************/
/**
 * If tracing is disabled, the program can stop on error without
 * displaying on the console to the user what the error was.   This message tells
 * the user that there was an error and how to turn on tracing.
 *
 * The message is only displayed if error tracing isn't already active.
 *
 */
static void displayExitError( void )
{
	if ( iVerbose < 1 ) {
		printf("Exited with error.  Run again adding parameter \"-v 8\" for tracing.\n");
	}
}


/* **************************************************************************
 * Function  execute
 * *************************************************************************/
/**
 * Invoke the command process to execute the specified command.
 *
 * \param[in] cmd - command to execute
 *
 * \return status code returned by the called command
 *
 */
static int execute(const string cmd)
{
	ETG_TRACE_USR1(("Executing command %s", cmd.c_str()));
	return system(cmd.c_str());
}

/* **************************************************************************
 * Function  chomp
 * *************************************************************************/
/**
 * Remove '\r' and '\n'.   '\n' is used in unix but not '\r'.   So when removing
 * '\n' it is also advisable to check for the existence of '\r'.
 * '\r' can exist when text editors on windows based systems are used to edit
 * text files on unix based systems.
 *
 * \param[in/out] str - string to process
 *
 * \return true if the string is not empty after removing \n and \r.
 *
 */
static bool chomp(string& str)
{
	str.erase( remove(str.begin(), str.end(), '\r'), str.end() );
	str.erase( remove(str.begin(), str.end(), '\n'), str.end() );
	return !str.empty();
}

/* **************************************************************************
 * Function  convertTabsToSpaces
 * *************************************************************************/
/**
 * Convert tabs to spaces.   When parsing a string looking for spaces, it can
 * go wrong if a tab has been used instead.   This can occur when using editors
 * that have not been setup to insert spaces when the tab key is pressed.  This
 * problem can occur when text editors on windows based systems are used to edit
 * text files on unix based systems.
 *
 * \param[in/out] str - string to process
 *
 * \return number of tabs replaced.
 *
 */
static unsigned int convertTabsToSpaces(string& str)
{
	unsigned int tabs_found = 0;
	size_t found = str.find_first_of('\t');

	while ( found != string::npos ) {
		tabs_found++;
		str[found] = ' ';
		found = str.find_first_of('\t');
	}
	return tabs_found;
}

/* **************************************************************************
 * Function  checkFileExists
 * *************************************************************************/
/* Check the file specified exists.
 *
 * \param[in]  filename    - file to check.
 *
 * \return true if file exists, otherwise false.
 */
static bool checkFileExists( string filename )
{
	bool result = false;
	struct stat statFile;

	if (0 == stat(filename.c_str(), &statFile) && S_ISREG(statFile.st_mode)) {
		result = true;
	}
	return result;
}

/* **************************************************************************
 * Function  checkDirExists
 * *************************************************************************/
/* Check the dir specified exists.
 *
 * \param[in]  dirname    - dir to check.
 *
 * \return true if dir exists, otherwise false.
 */
static bool checkDirExists( string dirname )
{
	bool result = false;
	struct stat statDir;

	if (0 == stat(dirname.c_str(), &statDir) && S_ISDIR(statDir.st_mode)) {
		result = true;
	}
	return result;
}

/* **************************************************************************
 * Function  deleteFile
 * *************************************************************************/
/* Delete file.
 *
 * \param[in]  filename    - file to delete.
 *
 * \return true if file deleted, otherwise false.
 */
static bool deleteFile( string filename )
{
	bool result = true;

	if (0 != remove(filename.c_str())) {
		ETG_TRACE_ERR(("Cannot remove \"%s\" for recreation  errno=%d", filename.c_str(), errno ));
		result = false;
	}
	return result;
}

/* **************************************************************************
 * Function  makeFileWritable
 * *************************************************************************/
/* Make specified file writable using chmod.
 *
 * \param[in]  filename    - file to make writable.
 *
 * \return true if successful, otherwise false.
 */
static bool makeFileWritable( string filename )
{
	bool result = true;

	if ( checkFileExists(filename) ) {
		if (0 != chmod(filename.c_str(), 0777) ) {
			ETG_TRACE_ERR(("chmod() error %d for file: %s", errno, filename.c_str()));
			displayExitError();
			result = false;
		}
	}
	return result;
}

/* **************************************************************************
 * Function  changeDir
 * *************************************************************************/
/* Change directory
 *
 * \param[in]  dirname    - directory name
 *
 * \return true if successful, otherwise false.
 */
static bool changeDir( string dirname )
{
	bool result = true;

	ETG_TRACE_COMP(("Change dir to %s", dirname.c_str()));
	if (0 != chdir(dirname.c_str())) {
		ETG_TRACE_ERR(("Error: command chdir returns with errno %d for dir: %s", errno, dirname.c_str()));
		result = false;
	}
	return result;
}


/* **************************************************************************
 * Function  execute_ret
 * *************************************************************************/
/**
 * Execute the command specified as a child process and store what is displayed
 * by the child process in a string.
 *
 * \param[in] cmd - command to execute
 *
 * \return text produced by child process.
 *
 */
static string execute_ret(string cmd)
{
	FILE* pipe = popen(cmd.c_str(), "r");

	if (!pipe) {
		return string();
	}
	char buffer[255];
	string result = "";
	while (!feof(pipe)) {
		if (fgets(buffer, 255, pipe) != NULL) {
			result += buffer;
		}
	}
	pclose(pipe);
	return result;
}

/* **************************************************************************
 * Function  extract_gn
 * *************************************************************************/
/**
 * Extract the content of the Gracenote DB zip file into "destDir/lfsId/"
 *
 * \param[in] lfsId       - List file set ID (i.e. "734").
 * \param[in] zipFileName - Gracenote DB zip (i.e. "ce2022_rev6792_edb13560001_CN_le_20150304093957.zip").
 * \param[in] destDir     - destination directory (i.e. "db/crypted")
 *
 * \return 0 if successful.  Otherwise ERR_EXIT_CODE.
 *
 */
static int extract_gn(string lfsId, string zipFileName, string destDir)
{

	ETG_TRACE_COMP(("extract_gn(lfsId: \"%s\", zipFileName: \"%s\", destDir: \"%s\")", lfsId.c_str(), zipFileName.c_str(), destDir.c_str()));
	if (0 == lfsId.length()) {
		ETG_TRACE_ERR(("Manifest LFS_ID unknown in zip: %s", zipFileName.c_str()));
		return ERR_EXIT_CODE;
	}
	string lfsidDestDir = (destDir + "/") + lfsId + "/";
	int res = GNWL.mkdir_p(lfsidDestDir.c_str());
	if (!res) {
		ETG_TRACE_ERR(("Command mkdir(%s) returns with error %d", lfsidDestDir.c_str(), errno ));
		return ERR_EXIT_CODE;
	}
	res = execute(string("unzip -o ") + zipFileName + string(" -d ") + lfsidDestDir.c_str());
	if (0 != res) {
		ETG_TRACE_ERR(("Problem with extracting errno %d the zip %s", errno, (zipFileName + string(" to ") + lfsidDestDir).c_str()));
		return ERR_EXIT_CODE;
	}
	// done
	return 0;
}

/* **************************************************************************
 * Function  progress
 * *************************************************************************/
/**
 * Registered with 'tclGracenoteWrapper::progress' to report progress
 * during Gracenote DB processing.
 *
 * \param[in] percent - progress in %.
 *
 */
static void progress(int percent)
{
	static bool bStopping = false;
	static bool bStarting = false;

	// either via trace or as printf...
	if (percent > 100) {
		if (percent > 200) {
			if (bStopping) {
				ETG_TRACE_USR1(("                           %3d", percent - 200));
			} else {
				ETG_TRACE_USR1(("Installer progress stop DB %3d", percent - 200));
				bStopping = true;
			}
		} else {
			if (bStarting) {
				ETG_TRACE_USR1(("                            %3d", percent - 100));
			} else {
				ETG_TRACE_USR1(("Installer progress start DB %3d", percent - 100));
				bStarting = true;
			}
		}
	} else {
		bStopping = false;
		bStarting = false;
		ETG_TRACE_COMP(("Installer progress %3d", percent));
	}
}

/* **************************************************************************
 * Function  problem
 * *************************************************************************/
/**
 * Registered with 'tclGracenoteWrapper::problemreport' to report problems
 * during Gracenote DB processing.
 *
 * \param[in] str - problem to report.
 *
 */
static void problem(const char* str)
{
	ETG_TRACE_SYS(("Warning - %s", str));
	if (errLog) {
		fprintf(errLog, "%s\n", str);
	}
}

/* **************************************************************************
 * Function  usage
 * *************************************************************************/
/**
 * Display usage message.  Used when incorrect parameters are given.
 *
 * \param[in] argc     - Number of parameters.
 * \param[in] argv     - Arguments.
 *
 */
void usage( int argc, char *argv[] )
{
	printf("Called with %d parameters: ", argc - 1);
	for (int i = 1; i < argc; ++i ) {
		printf("%s ", argv[i]);
	}
	printf("\n");
	printf("Analyses the source zip-files from GN-WebSite and prepares data for software update in the device.\n");
	printf("Usage:\n");
	printf("%s -s <sourceDir> -w <workDir> -t <tempDir> [-d <destDir>] [-v <level>] [-l <lfsId>] [-n] [-i <tgz|raw>]\n", argv[0]);
	printf("<sourceDir>, <workDir>, <tempDir> and <destDir> must be unique.\n");
	printf("The content of <tempDir> will be deleted unless -n is specified.\n");
	printf("  s ourceDir must contain a md5.txt with lines of <md5sum> <space> <space> <zip-filename> (in \"-i tgz\" mode \"GracenoteDB.md5\").\n");
	printf("  w orkdir is used to store the decrypted data (shall be volatile and not copied).\n");
	printf("  t empDir is used to intermediately unzip the GN DB (shall be volatile and not used with \"-i raw\".\n");
	printf("  d estDir is used to copy the final tgz-file for release (not used with \"-i raw|tgz\".\n");
	printf("  l fsId is used in target configuration (use with '-i').\n");
	printf("  v : activates verbose mode on level 0 - 8.\n");
	printf("  n : tells not to remove tempDir content.\n");
	printf("  i : tells which installer mode (tgz|raw) for target configuration is to be used (use with '-l').\n");
}


/* **************************************************************************
 * Function  calcFileMD5
 * *************************************************************************/
/**
 * Read MD5 checksum(s) and file(s) from the specified file.
 *
 * \param[in] file         - File to perform md5sum on.
 * \param[out] md5Checksum - where to store the md5 checksum calculated.
 *
 * \return true if MD5 checksum was calculated ok.  Otherwise false.
 *
 */
static bool calcFileMD5(string file, string& md5Checksum )
{
	string md5ChecksumCalc;
	bool result = true;
	string cmd = string("md5sum ") + file + string(" 2>/dev/null");

	md5ChecksumCalc = execute_ret(cmd);
	string::size_type pos = md5ChecksumCalc.find(' ');
	if ( pos != string::npos ) {
		md5Checksum = md5ChecksumCalc.substr(0, pos);
		if ( string::npos != md5Checksum.find_first_not_of("abcdefABCDEF0123456789")) {
			/* If a space or non-checksum characters are found in the result,
			 * the result is not as expected and it could be that md5sum was not found.
			 */
			result = false;
		}
	}else {
		/* If a space cannot be found in the result, the result is not as expected,
		 * could an error message with no spaces.
		 */
		result = false;
	}

	if ( !result ) {
		ETG_TRACE_ERR(("Error processing \"%s\"\nInvalid checksum: \"%s\"", cmd.c_str(), md5Checksum.c_str()));
	}

	return result;
}

/* **************************************************************************
 * Function  readMD5file
 * *************************************************************************/
/**
 * Read MD5 checksum(s) and file(s) from the specified file.
 *
 * \param[in] md5File  - Contains the MD5 checksums and filenames.
 * \param[out] md5Info - Memory for the read MD5 checksums and filenames.
 *
 * \return true if MD5 checksums and files were read ok.  Otherwise false.
 *
 */
static bool readMD5file( string md5File, vector<MD5SUM_INFO>& md5Info )
{
	ifstream infile(md5File.c_str());
	string md5Line;
	int line_num = 0;
	bool result = true;

	/* Confirm the existence of the MD5 file. */
	if (checkFileExists(md5File.c_str())) {
		ETG_TRACE_USR1(("Reading: \"%s\"", md5File.c_str()));

		while (getline(infile, md5Line)) {
			(void)chomp(md5Line);
			(void)convertTabsToSpaces(md5Line);

			if ( !md5Line.empty() ) {
				MD5SUM_INFO md5Info_element;
				string::size_type pos = md5Line.find(' ');
				line_num++;
				if ( pos != string::npos ) {
					md5Info_element.checksum = md5Line.substr(0, pos);
					md5Info_element.filename = md5Line.substr(pos);
					md5Info_element.filename.erase( remove(md5Info_element.filename.begin(), md5Info_element.filename.end(), ' '), md5Info_element.filename.end() );

					if ( (md5Info_element.filename.empty()) ||
					     (string::npos != md5Info_element.filename.find_first_of(" @\\/:*?|<>\"\0;=+%&\t"))) {
						ETG_TRACE_ERR(("Error processing \"%s\" on line %d, invalid filename: \"%s\"", md5File.c_str(), line_num, md5Info_element.filename.c_str()));
						result = false;
					}

					if ( string::npos != md5Info_element.checksum.find_first_not_of("abcdefABCDEF0123456789")) {
						ETG_TRACE_ERR(("Error processing \"%s\" on line %d, invalid checksum: \"%s\"", md5File.c_str(), line_num, md5Info_element.checksum.c_str()));
						result = false;
					}

					ETG_TRACE_COMP(("Found: \"%s\" \"%s\"", md5Info_element.checksum.c_str(), md5Info_element.filename.c_str()));
					if ( result ) {
						md5Info.push_back(md5Info_element);
					}
				}else {
					/* If a space cannot be found in the result, the md5 file is on the wrong format. */
					ETG_TRACE_ERR(("Error processing \"%s\" on line %d: \"%s\"", md5File.c_str(), line_num, md5Line.c_str()));
					result = false;
				}
			}
		}
		infile.close();
	}else {
		ETG_TRACE_ERR(("MD5 file not found: \"%s\"", md5File.c_str()));
		result = false;
	}

	if ( 0 == md5Info.size() ) {
		/* It was an empty file. */
		result = false;
	}else {
		ETG_TRACE_COMP(("Number files in \"%s\": %d", md5File.c_str(), md5Info.size()));
	}
	return result;
}

/* **************************************************************************
 * Function  verifyMD5checksums
 * *************************************************************************/
/**
 * Verify MD5 checksums of files and checksums in md5Info.
 *
 * \param[in] md5Info - Contains the MD5 checksums and filenames read
 *                      from the md5 file (see function readMD5file).
 * \return true if MD5 checksums are ok.  Otherwise false.
 *
 */
static bool verifyMD5checksums(vector<MD5SUM_INFO> md5Info)
{
	bool result = true;

	for (vector<MD5SUM_INFO>::const_iterator it = md5Info.begin(); (it != md5Info.end()); ++it) {
		string calc_checksum;
		if (checkFileExists(it->filename.c_str())) {
			if ( calcFileMD5(it->filename, calc_checksum) ) {
				if ( 0 == it->checksum.compare(calc_checksum) ) {
					ETG_TRACE_COMP(("Checksum passed: \"%s\"", it->filename.c_str()));
				}else {
					string error_str;
					ETG_TRACE_ERR(("Checksum failed: \"%s\"", it->filename.c_str()));
					error_str = string("Calculated md5 checksum: \"") + calc_checksum + string("\" <> \"") + it->checksum + string("\"");
					ETG_TRACE_ERR(("%s", error_str.c_str()));
					result = false;
				}
			}else {
				result = false;
			}
		}else {
			ETG_TRACE_ERR(("File not found: \"%s\"", it->filename.c_str()));
			result = false;
		}
	}
	return result;
}

/* **************************************************************************
 * Function  extractXmlField
 * *************************************************************************/
/**
 * Extracts the LFS_ID from the XML manifest info.
 *
 * \param[in] xml_in - XML string.
 * \param[in] fieldname - field value to extract from XML string.
 * \return field value.
 *
 */
static string extractXmlField( string xml_in, string fieldname )
{
	string result;
	size_t found_start = string::npos;
	size_t found_end = string::npos;

	found_start = xml_in.find(fieldname);
	found_end = xml_in.find(fieldname, found_start + fieldname.length());
	result = xml_in.substr(found_start + fieldname.length(), found_end - (found_start + fieldname.length()));
	found_start = result.find_first_of('>');
	found_end = result.find_first_of('<');
	result = result.substr(found_start + 1, found_end - (found_start + 1));
	return fieldname + string("=") + result;
}


/* **************************************************************************
 * Function  checkLfsId
 * *************************************************************************/
/**
 * Check LFS Id is only numbers and is the correct length.
 *
 * \param[in] lfsId  - string to check
 *
 * \return true if ok.  Otherwise false.
 *
 */
static bool checkLfsId( string lfsId )
{
	bool result = true;

	if ( (lfsId.find_first_not_of("0123456789") != string::npos) || lfsId.empty()) {
		printf("LFS_ID \"%s\" must be numeric.\n", lfsId.c_str());
		result = false;
	}else if ( lfsId.length() != 3 ) {
		printf("LFS_ID \"%s\" must be 3 digits.\n", lfsId.c_str());
		result = false;
	}

	return result;
}


/* **************************************************************************
 * Function  extractLfsId
 * *************************************************************************/
/**
 * Extracts the LFS_ID from the XML manifest info.
 *
 * \param[in] manifest_info - XML string
 * \param[out] resultLfsId - where to store LfsId
 * \return true if LFS_ID was decoded successfully.
 *
 */
static bool extractLfsId( string manifest_info, string& resultLfsId )
{
	bool result = true;

	resultLfsId = extractXmlField( manifest_info, "LFS_ID" );
	resultLfsId = resultLfsId.substr( 7, 3 );
	if ( !checkLfsId( resultLfsId ) ) {
		result = false;
	}
	return result;
}

/* **************************************************************************
 * Function  extractDbInfo
 * *************************************************************************/
/**
 * Extracts the EXPORT_TIME, DB_REVISION and LFS_ID from the XML manifest info.
 *
 * \param[in] manifest_info - XML string
 * \return DB Info in the format shown below (example):
 *
 * EXPORT_TIME=2015-03-04 17:29:43Z
 * DB_REVISION=6792
 * LFS_ID=734
 *
 */
static string extractDbInfo( string manifest_info )
{
	string result;

	result = extractXmlField( manifest_info, "EXPORT_TIME" ) + string("\n");
	result += extractXmlField( manifest_info, "DB_REVISION" ) + string("\n");
	result += extractXmlField( manifest_info, "LFS_ID" ) + string("\n");
	return result;
}

/* **************************************************************************
 * Function  checkTraceLevel
 * *************************************************************************/
/**
 * Check the trace level is within range.
 *
 * \param[in] traceLevel - string to test.
 *
 * \return true if successful, otherwise false.
 *
 */
static bool checkTraceLevel(string traceLevel)
{
	bool result = true;

	if ( traceLevel.empty() ||
	     (traceLevel.find_first_not_of("012345678") != string::npos) ||
	     (traceLevel.length() > 1)) {
		printf("Parameter specified for \"-v\" must be between 0 and 8.\n");
		result = false;
	}
	return result;
}

/* **************************************************************************
 * Function  checkInstMode
 * *************************************************************************/
/**
 * Check Install mode is "raw" or "tgz".
 *
 * \param[in] instMode  - string to check
 *
 * \return true if ok.  Otherwise false.
 *
 */
static bool checkInstMode( string instMode )
{
	bool result = true;

	if ( instMode != "raw" && instMode != "tgz" ) {
		printf("Parameter specified for \"-i\" must be \"tgz\" or \"raw\"\n");
		result = false;
	}
	return result;
}

/* **************************************************************************
 * Function  checkDirName
 * *************************************************************************/
/**
 * Check directory names contain valid characters, is not empty and does
 * not have a trailing '/'.
 *
 * \param[in] dir_name  - string to check
 *
 * \return true if ok.  Otherwise false.
 *
 */
static bool checkDirName( string dir_name )
{
	bool result = true;

	if ( dir_name.empty() ) {
		printf("Directory name cannot be empty\n");
		result = false;
	}else if ( string::npos != dir_name.find_first_of(", ~-@\\:*?|<>\";(){}[]£€#`=+%%&\t\n\r\0") ) {
		printf("Directory name \"%s\" cannot contain: \",& ~-@\\:*?|<>\";(){}[]£€#`=+%%\\t\\n\\r\\0\".\n", dir_name.c_str());
		result = false;
	}
	/* Check for trailing slash. */
	else if ( (dir_name.length() > 1) && (dir_name[dir_name.length() - 1] == '/') ) {
		printf("Directory name \"%s\" cannot have a trailing '/'.\n", dir_name.c_str());
		result = false;
	}

	return result;
}

/* **************************************************************************
 * Function  checkValidDirArgs
 * *************************************************************************/
/**
 * Check directory names are not the same and if appropriate, are not empty.
 *
 * \param[in] galArgs  - Directory parameters to check.
 *
 * \return true if ok.  Otherwise false.
 *
 */
static bool checkValidDirArgs( GAL_ARGS galArgs )
{
	bool result = true;

	/* Always need source and working directories. */
	if ( galArgs.srcDir.empty() || galArgs.workDir.empty() ||
	     (0 == galArgs.srcDir.compare(galArgs.workDir))) {
		printf("Check dir names:  SourceDir(-s)=\"%s\", WorkDir(-w)=\"%s\"\n", galArgs.srcDir.c_str(), galArgs.workDir.c_str());
		result = false;
	}

	if ( galArgs.instMode == "raw") {
		if ( !galArgs.tempDir.empty()) {
			printf("Temp dir not used in \"i- raw\" mode.\n");
			result = false;
		}
	}
	/* Check for empty directories depending on install mode. */
	/* "raw" checked above. */
	else{
		if ( galArgs.instMode == "tgz" ) {
			if (galArgs.tempDir.empty() ||
			    (0 == galArgs.srcDir.compare(galArgs.tempDir)) ||
			    (0 == galArgs.workDir.compare(galArgs.tempDir))) {
				printf("Check temp dir name:  TempDir(-t)=\"%s\"\n", galArgs.tempDir.c_str());
				result = false;
			}
		}else {
			/* This is a normal install mode. */
			if (galArgs.tempDir.empty() || galArgs.destDir.empty() ||
			    (0 == galArgs.srcDir.compare(galArgs.tempDir)) ||
			    (0 == galArgs.workDir.compare(galArgs.tempDir)) ||
			    (0 == galArgs.srcDir.compare(galArgs.destDir)) ||
			    (0 == galArgs.workDir.compare(galArgs.destDir)) ||
			    (0 == galArgs.destDir.compare(galArgs.tempDir)) ) {
				printf("Parameter error.\nTempDir(-t)=\"%s\", SourceDir(-s)=\"%s\", WorkDir(-w)=\"%s\", destDir(-d)=\"%s\"\n", galArgs.tempDir.c_str(), galArgs.srcDir.c_str(), galArgs.workDir.c_str(), galArgs.destDir.c_str());
				printf("Please ensure directory names are unique.\n");
				result = false;
			}
		}
	}
	return result;
}

/* **************************************************************************
 * Function  checkValidModeArgs
 * *************************************************************************/
/**
 * Check that the installation mode and lfs id parameters are used correctly.
 * "-i tgz|raw" and "-l lfsid" should be only be used together.
 * And "-n" should not with be used with "-i raw"
 *
 * \param[in] galArgs  - instMode and lfsId parameters to check.
 *
 * \return true if ok.  Otherwise false.
 *
 */
static bool checkValidModeArgs(GAL_ARGS galArgs)
{
	bool result = true;

	if ( galArgs.instMode.empty() ) {
		if ( !galArgs.lfsId.empty()) {
			printf("LFS ID specified, but not with option \"-i\"?\n");
			result = false;
		}
	}else {
		if ( galArgs.lfsId.empty()) {
			printf("Missing LFS ID, use option \"-l\"\n");
			result = false;
		}else if ( galArgs.instMode == "raw" ) {
			if ( !galArgs.tempDir.empty() && !galArgs.destDir.empty() ) {
				printf("Target installation \"raw\" mode, but tempDir=\"%s\" and destDir=\"%s\" specified.\n", galArgs.tempDir.c_str(), galArgs.destDir.c_str());
				result = false;
			}else if ( !galArgs.removeTempDirContent.empty() ) {
				printf("Parameter \"-n\" not applicable when in \"-i raw\" mode.\n");
				result = false;
			}
		}else if ( galArgs.instMode == "tgz" && !galArgs.destDir.empty() ) {
			printf("Target installation \"tgz\" mode, but destDir=\"%s\" specified.\n", galArgs.destDir.c_str());
			result = false;
		}
	}
	return result;
}

/* **************************************************************************
 * Function  checkOptionArgs
 * *************************************************************************/
/**
 * Check the option args (i.e. parameter starting with '-').
 *
 * \param[in] OptionArg  - string to check
 *
 * \return true if ok.  Otherwise false.
 *
 */
static bool checkOptionArgs( string OptionArg )
{
	bool result = true;

	if ( OptionArg.empty() ) {
		result = false;
		printf("Missing parameter\n", OptionArg.c_str());
	}else if ( OptionArg.length() != 2 ) {
		printf("Parameter is the wrong length: \"%s\"\n", OptionArg.c_str());
		result = false;
	}else {
		switch ( (int)OptionArg[1] ) {
		case 's':
		case 'd':
		case 't':
		case 'w':
		case 'l':
		case 'i':
		case 'v':
		case 'n':
			break;
		default:
			printf("Unknown option parameter: \"%s\".\n", OptionArg.c_str());
			result = false;
			break;
		}
	}
	return result;
}

/* **************************************************************************
 * Function  processOptionArgs
 * *************************************************************************/
/**
 * Process the option args (i.e. parameter following parameter starting with '-').
 *
 * \param[out] galArgs   - processed arguments.
 * \param[in]  option    - parameter starting with '-'.
 * \param[in]  param     - parameter following parameter starting with '-'.
 *
 * \return true if ok.  Otherwise false.
 *
 */
static bool processOptionArgs( GAL_ARGS& galArgs, string option, string param )
{
	bool result = false;

	if ( param.empty() ) {
		printf("No parameter following \"%s\".\n", option.c_str());
	}else {
		switch ( (int)option[1] ) {
		case 's':
			if ( !galArgs.srcDir.empty()) {
				printf("Duplicate source dir parameter \"%s\" before \"%s\"\n.", galArgs.srcDir.c_str(), param.c_str());
			}else {
				galArgs.srcDir = param;
				result = checkDirName(galArgs.srcDir);
			}
			break;

		case 'd':
			if ( !galArgs.destDir.empty()) {
				printf("Duplicate destination dir parameter \"%s\" before \"%s\"\n.", galArgs.destDir.c_str(), param.c_str());
			}else {
				galArgs.destDir = param;
				result = checkDirName(galArgs.destDir);
			}
			break;

		case 't':
			if ( !galArgs.tempDir.empty()) {
				printf("Duplicate temporary dir parameter \"%s\" before \"%s\"\n.", galArgs.tempDir.c_str(), param.c_str());
			}else {
				galArgs.tempDir = param;
				result = checkDirName(galArgs.tempDir);
			}
			break;

		case 'w':
			if ( !galArgs.workDir.empty()) {
				printf("Duplicate work dir parameter \"%s\" before \"%s\"\n.", galArgs.workDir.c_str(), param.c_str());
			}else {
				galArgs.workDir = param;
				result = checkDirName(galArgs.workDir);
			}
			break;

		case 'l':
			if ( !galArgs.lfsId.empty()) {
				printf("Duplicate LFS_ID parameter \"%s\" before \"%s\"\n.", galArgs.lfsId.c_str(), param.c_str());
			}else {
				galArgs.lfsId = param;
				result = checkLfsId(galArgs.lfsId);
			}
			break;

		case 'i':
			if ( !galArgs.instMode.empty()) {
				printf("Duplicate installation mode parameter \"%s\" before \"%s\"\n.", galArgs.instMode.c_str(), param.c_str());
			}else {
				galArgs.instMode = param;
				result = checkInstMode(galArgs.instMode);
			}
			break;

		case 'v':
			if ( !galArgs.traceLevel.empty()) {
				printf("Duplicate trace level parameter \"%s\" before \"%s\"\n.", galArgs.traceLevel.c_str(), param.c_str());
			}else {
				galArgs.traceLevel = param;
				result = checkTraceLevel(galArgs.traceLevel);
			}
			break;

		default:
			printf("Unknown parameter : \"%s\".\n", param.c_str());
			break;
		}
	}
	return result;
}

/* **************************************************************************
 * Function  extractArgs
 * *************************************************************************/
/* Check the parameters
 *
 * \param[in]  argc    - number of parameters
 * \param[in]  argv    - parameters
 * \param[out] galArgs - extracted parameters
 *
 * \return true if parameters are ok.  Otherwise false.
 */
bool extractArgs( int argc, char *argv[], GAL_ARGS& galArgs )
{
	bool result = true;
	bool display_help_option = true;

	if ( argc < 3 ) {
		usage( argc, argv );
		result = false;
		display_help_option = false;
	}else {
		for (int i = 1; (i < argc) && result; ++i ) {
			string option;
			if ( '-' == argv[i][0] ) {
				option = string( argv[i]);
				if ( !checkOptionArgs(option) ) {
					result = false;
					continue;
				}

				if ( option == "-n") {
					galArgs.removeTempDirContent = "no";
					continue;
				}

				if ( (i + 1) >= argc ) {
					/* There is a missing parameter. */
					printf("Missing parameter after \"%s\"\n", option.c_str());
					result = false;
					continue;
				}
				i++;
				result = processOptionArgs(galArgs, option, string( argv[i]));
			}else {
				/* There is a missing parameter. */
				printf("Parameter order error, expected a parameter starting with '-', not \"%s\"\n", argv[i]);
				result = false;
			}
		}
	}

	if ( result ) {
		/* Check the Target install and LFS ID args. */
		result = checkValidModeArgs(galArgs);
	}

	if ( result ) {
		/* Check for duplicate and empty directory names. */
		result = checkValidDirArgs(galArgs);
	}else if ( display_help_option ) {
		printf("Use \"-h\" option for usage.\n");
	}

	return result;
}

/* **************************************************************************
 * Function  isPathRelative
 * *************************************************************************/
/* When using tar, it is sometimes necessary to know whether the path is
 * absolute or relative to ensure the zip file can be accessed from the target
 * directory.
 *
 * \param[in]  dirPath    - path to check.
 *
 * \return true if path is relative (not starting with '/'), otherwise false.
 */
bool isPathRelative( string dirPath )
{
	bool result = true;
	size_t found = dirPath.find('/');

	if ( 0 == found ) {
		// Path is absolute.
		result = false;
	}
	return result;
}

/* **************************************************************************
 * Function  main
 * *************************************************************************/
/**
 * Main program.
 *
 * \param[in] argc - number of parameters
 * \param[in] argv - parameters
 *
 * \return 0 on success, or ERR_EXIT_CODE
 *
 */
int main( int argc, char *argv[] )
{
	printf("Installer for Gracenote data bases. v"VERSION " (Built on %s %s).\n", __DATE__, __TIME__);
	printf("-----------------------------------------\n");


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
	//   init trace backend
	vInitPlatformEtg();
#endif

	string sourceDir;
	string workDir;
	string tempDir;
	string destDir;
	string lfsId;
	string instMode;
	string curDir = execute_ret("pwd");
	bool bRemoveTempDirContent = true;
	bool bTargetInstall = false;
	GAL_ARGS galArgs;

	if ( !extractArgs( argc, argv, galArgs ) ) {
		return ERR_EXIT_CODE;
	}

	if (false == chomp(curDir)) {
		printf("General fault. pwd returned zero string.\n");
		return ERR_EXIT_CODE;
	}

	sourceDir = galArgs.srcDir;
	workDir = galArgs.workDir;
	destDir = galArgs.destDir;
	tempDir = galArgs.tempDir;

	lfsId = galArgs.lfsId;
	instMode = galArgs.instMode;

	if ( !galArgs.traceLevel.empty() ) {
		iVerbose = atoi(galArgs.traceLevel.c_str());
	}

	if ( "no" == galArgs.removeTempDirContent ) {
		bRemoveTempDirContent = false;
	}

	bTargetInstall = !galArgs.instMode.empty();

	if (bTargetInstall) {
		string lfsidSourceDir = sourceDir + string("/") + lfsId;
		string lfsidDestDir = destDir + string("/") + lfsId;
		string lfsidWorkDir = workDir + string("/") + lfsId;
		/* For this to work, the directory sourceDir/LfsId must exist. */
		if (!checkDirExists(lfsidSourceDir)) {
			// lfsidSourceDir is not available or not a directory
			ETG_TRACE_ERR(("LFS_ID directory \"%s\" is not available: ", lfsidSourceDir.c_str()));
			displayExitError();
			return ERR_EXIT_CODE;
		}

		if (!changeDir(lfsidSourceDir)) {
			displayExitError();
			return ERR_EXIT_CODE;
		}
		// target use case
		ETG_TRACE_COMP(("Target installer for ONE Gracenote DB lfsid %d from %s",
				atoi(lfsId.c_str()), (lfsidSourceDir + string(" to ") + workDir).c_str()));

		if (instMode == "tgz") {
			// Read md5sum file and remove trailing new lines, carriage returns and empty lines.
			vector<MD5SUM_INFO> md5DBInfo;
			if ( !readMD5file("GracenoteDB.md5", md5DBInfo)) {
				displayExitError();
				return ERR_EXIT_CODE;
			}

			if (md5DBInfo.size() > 1) {
				ETG_TRACE_ERR(("Expected only 1 tgz in GracenoteDB.md5, found %d.", md5DBInfo.size()));
				displayExitError();
				return ERR_EXIT_CODE;
			}
			if ( !verifyMD5checksums(md5DBInfo) ) {
				displayExitError();
				return ERR_EXIT_CODE;
			}
		}else {
			/* Target update is in RAW mode, check the existence of at least the manifest file. */
			if (!checkFileExists("manifest.gn")) {
				ETG_TRACE_ERR(("Target update in RAW mode, \"manifest.gn\" file not found."));
				displayExitError();
				return ERR_EXIT_CODE;
			}
		}

		if (!changeDir(curDir)) {
			displayExitError();
			return ERR_EXIT_CODE;
		}

		// remove dbv1_revision
		string cmdline = string("rm -rf ") + lfsidWorkDir + string("/") + DB_REVISION_FILE;
		int res = execute(cmdline);
		if (0 != res) {
			ETG_TRACE_ERR(("Problem with remove version info errno %d", errno));
			displayExitError();
			return ERR_EXIT_CODE;
		}

		// ensure file system is finished
		ETG_TRACE_COMP(("Syncing file system..."));
		sync();

		// init the DB
		tclGracenoteWrapper GNWL;
		GNWL.cb.progress = progress;
		GNWL.cb.problemreport = problem;

		// if tgz install do prepare temp
		if (instMode == "tgz") {
			string lfsidTempDir = tempDir + string("/") + lfsId;
			ETG_TRACE_COMP(("Prepare temp dir for tgz install"));
			int res = GNWL.mkdir_p(lfsidTempDir.c_str());
			if (!res) {
				ETG_TRACE_ERR(("Command mkdir(%s) returns with error %d", lfsidTempDir.c_str(), errno ));
				displayExitError();
				return ERR_EXIT_CODE;
			}
			cmdline = string("rm -rf ") + lfsidTempDir + string("/*");
			res = execute(cmdline);
			if (0 != res) {
				ETG_TRACE_ERR(("Problem with remove old temp files errno %d", errno));
				displayExitError();
				return ERR_EXIT_CODE;
			}

			if (!changeDir(lfsidTempDir)) {
				displayExitError();
				return ERR_EXIT_CODE;
			}

			if ( isPathRelative(lfsidSourceDir) ) {
				cmdline = string("tar -xf ") + curDir + string("/") + lfsidSourceDir + string("/GracenoteDB.tgz");
			}else {
				cmdline = string("tar -xf ") + lfsidSourceDir + string("/GracenoteDB.tgz");
			}
			res = execute(cmdline);
			if (0 != res) {
				ETG_TRACE_ERR(("Problem with extracting errno %d", errno));
				displayExitError();
				return ERR_EXIT_CODE;
			}
		}

		if (!changeDir(curDir)) {
			displayExitError();
			return ERR_EXIT_CODE;
		}

		res = GNWL.mkdir_p(lfsidWorkDir.c_str());
		if (!res) {
			ETG_TRACE_ERR(("Command mkdir(workDir) returns with error %d", errno ));
			displayExitError();
			return ERR_EXIT_CODE;
		}
		ETG_TRACE_COMP(("GN wrapper start(%d, %s)", atoi(lfsId.c_str()), workDir.c_str()));
		GNWL.start(false, atoi(lfsId.c_str()), (workDir + string("/")).c_str());
		// decrypt the data base
		if (instMode == "tgz") {
			ETG_TRACE_COMP(("GN wrapper updateDb(%s)", tempDir.c_str()));
			GNWL.updateDb(tempDir.c_str());
		} else {
			ETG_TRACE_COMP(("GN wrapper updateDb(%s)", sourceDir.c_str()));
			GNWL.cb.progress = progress;
			GNWL.cb.problemreport = problem;
			GNWL.updateDb(sourceDir.c_str());
		}
		ETG_TRACE_COMP(("GN wrapper stop()"));
		GNWL.stop();
		// ensure file system is finished
		ETG_TRACE_COMP(("Install finished. Syncing file system..."));
		sync();

		ETG_TRACE_COMP(("Testing DB Status..."));
		if (GNWL.start(false, atoi(lfsId.c_str()), (workDir + string("/")).c_str())) {
			ETG_TRACE_COMP(("DB Status: Success"));
			GNWL.stop();
		} else {
			ETG_TRACE_ERR(("DB Status: Error"));
			GNWL.stop();
			displayExitError();
			return ERR_EXIT_CODE;
		}

		// copy version file
		cmdline = string("cp ") + lfsidSourceDir + string("/") + string(DB_REVISION_FILE) + string(" ") + lfsidWorkDir + string("/");
		res = execute(cmdline);
		if (0 != res) {
			ETG_TRACE_ERR(("Problem to copy the %20s: errno %d cmd %s", DB_REVISION_FILE, errno, cmdline.c_str()));
			displayExitError();
			return ERR_EXIT_CODE;
		}

		if (bRemoveTempDirContent && (instMode == "tgz")) {
			cmdline = string("rm -rf ") + tempDir + "/*";
			res = execute(cmdline);
			if (0 != res) {
				ETG_TRACE_ERR(("Command \"%s\" return with errno %d", cmdline.c_str(), errno ));
				displayExitError();
				return ERR_EXIT_CODE;
			}
		}

		// ensure file system is finished
		ETG_TRACE_COMP(("Revision file copy finished. Syncing file system..."));
		sync();

		return 0;
	} // if (bTargetInstall)

	if (!changeDir(sourceDir)) {
		displayExitError();
		return ERR_EXIT_CODE;
	}

	// Read md5sum file and remove trailing new lines, carriage returns and empty lines.
	vector<MD5SUM_INFO> md5Info;
	if (!readMD5file("md5.txt", md5Info)) {
		displayExitError();
		return ERR_EXIT_CODE;
	}

	if ( !verifyMD5checksums(md5Info) ) {
		displayExitError();
		return ERR_EXIT_CODE;
	}

	if (!changeDir(curDir)) {
		displayExitError();
		return ERR_EXIT_CODE;
	}

	// init the DB
	tclGracenoteWrapper GNWL;
	// connect wrapper library progress with call backs
	GNWL.cb.progress = progress;
	GNWL.cb.problemreport = problem;

	int res = GNWL.mkdir_p(workDir.c_str());
	if (!res) {
		ETG_TRACE_ERR(("Command mkdir(%s) returns with error %d", workDir.c_str(), errno ));
		displayExitError();
		return ERR_EXIT_CODE;
	}

	res = GNWL.mkdir_p(destDir.c_str());
	if (!res) {
		ETG_TRACE_ERR(("Command mkdir(%s) returns with error %d", destDir.c_str(), errno ));
		displayExitError();
		return ERR_EXIT_CODE;
	}
	string errlogname = destDir + string("/") + ERR_LOG_FILE;
	if (!makeFileWritable(errlogname)) {
		return ERR_EXIT_CODE;
	}
	errLog = fopen(errlogname.c_str(), "w");
	if (0 == errLog) {
		ETG_TRACE_ERR(("Could not open the error %d log file for writing on %s", errno, errlogname.c_str()));
		displayExitError();
		return ERR_EXIT_CODE;
	}
	fprintf(errLog, "Analyze report for GN DB. \nIf this file keeps empty, nothing is to be done. Otherwise please seek for comments from Gracenote!\n");

	res = GNWL.mkdir_p(tempDir.c_str());
	if (!res) {
		ETG_TRACE_ERR(("Command mkdir(%s) returns with error %d", tempDir.c_str(), errno ));
		displayExitError();
		return ERR_EXIT_CODE;
	}

	// repeats unpack and decrypt for each zip file being a marketing area (i.e. different lfsid)
	for (vector<MD5SUM_INFO>::const_iterator it = md5Info.begin(); it != md5Info.end(); ++it) {
		lfsId.clear();
		ETG_TRACE_COMP(("Process file: \"%s\"", it->filename.c_str()));
		// read the DB_REVISION|EXPORT_TIME|LFS_ID from manifest file
		string cmd = string("unzip -p ") + sourceDir +  string("/") + it->filename.c_str() + string(" manifest.gn ");
		string manifest_info = execute_ret(cmd);
		string db_info = extractDbInfo(manifest_info);

		ETG_TRACE_COMP(("Working on %s", (it->filename.c_str() + string(" -> ") + db_info).c_str()));
		// put version info in error report
		fprintf(errLog, "\nWorking on %s:\n%s\n", it->filename.c_str(), db_info.c_str());

		if (!extractLfsId( manifest_info, lfsId )) {
			ETG_TRACE_ERR(("Couldn't find the LFS_ID in manifest.gn."));
			displayExitError();
			return ERR_EXIT_CODE;
		}
		ETG_TRACE_COMP(("LFS_ID from manifest.gn: %s", lfsId.c_str()));
		if (0 != extract_gn(lfsId, sourceDir +  string("/") + it->filename.c_str(), tempDir)) {
			ETG_TRACE_ERR(("Problem in extract_gn"));
			displayExitError();
			return ERR_EXIT_CODE;
		}

		// the genre file is available only once but becomes copied into each lfsid dir
		// to clean up the gracenote base dir
		string genreFile = string(sourceDir) + string("/") + GENRE_PHONEM_FILE;
		if (checkFileExists(genreFile)) {
			cmd = string("cp ") + genreFile + string(" ");
			cmd += tempDir + string("/") + lfsId + string("/");
			res = execute(cmd.c_str());
			if (0 != res) {
				ETG_TRACE_ERR(("Command returns %d: %s ", res, cmd.c_str()));
				displayExitError();
				return ERR_EXIT_CODE;
			}
		}else {
			ETG_TRACE_ERR(("Cannot find: \"%s\"\n ", genreFile.c_str()));
			displayExitError();
			return ERR_EXIT_CODE;
		}

		// start the wrapper on selected market region
		GNWL.start(false, atoi(lfsId.c_str()), (workDir + "/").c_str());
		// decrypt the data base
		GNWL.updateDb(tempDir);
		// start test
		ETG_TRACE_COMP(("Starting test and create exception list on database %s", (lfsId + string(" in ") + workDir).c_str()));
		GNWL.start();
		GNWL.test();
		GNWL.stop();
		ETG_TRACE_COMP(("Test finished on database %s", workDir.c_str()));
		// copy ex list from dest to temp
		string src = string("cp ") + workDir + string("/") + lfsId + "/";
		string destTmp = tempDir + string("/") + lfsId + "/";

		cmd = src + string("exceptionlist.lst ") + destTmp;
		res = execute(cmd);
		if (0 != res) {
			ETG_TRACE_ERR(("problem to copy the exceptionlist: errno %d cmd %s", errno, cmd.c_str()));
			displayExitError();
			return ERR_EXIT_CODE;
		}
		cmd = src + string(SNAPSHOT_DATE_FILE) + string(" ") + destTmp;
		res = execute(cmd);
		if (0 != res) {
			ETG_TRACE_ERR(("problem to copy the %20s: errno %d cmd %s", SNAPSHOT_DATE_FILE, errno, cmd.c_str()));
			displayExitError();
			return ERR_EXIT_CODE;
		}
		cmd = src + string(DB_REVISION_FILE) + string(" ") + destTmp;
		res = execute(cmd);
		if (0 != res) {
			ETG_TRACE_ERR(("problem to copy the %20s: errno %d cmd %s", DB_REVISION_FILE, errno, cmd.c_str()));
			displayExitError();
			return ERR_EXIT_CODE;
		}
		// zip temp and copy to destination dir
		if (!changeDir(destTmp)) {
			displayExitError();
			return ERR_EXIT_CODE;
		}

		if (checkFileExists("GracenoteDB.tgz")) {
			if (!deleteFile("GracenoteDB.tgz")) {
				displayExitError();
				return ERR_EXIT_CODE;
			}
		}
		if (checkFileExists("GracenoteDB.md5")) {
			if (!deleteFile("GracenoteDB.tgz")) {
				displayExitError();
				return ERR_EXIT_CODE;
			}
		}
		// zip the content of tmp
		cmd = string("tar -zcvf ./GracenoteDB.tgz *");
		res = execute(cmd);
		if (0 != res) {
			ETG_TRACE_ERR(("Command \"%s\" returns %d", cmd.c_str(), res ));
			displayExitError();
			return ERR_EXIT_CODE;
		}
		cmd = string("md5sum GracenoteDB.tgz > GracenoteDB.md5");
		res = execute(cmd);
		if (0 != res) {
			ETG_TRACE_ERR(("Command \"%s\" returns %d", cmd.c_str(), res ));
			displayExitError();
			return ERR_EXIT_CODE;
		}

		if (!changeDir(curDir)) {
			displayExitError();
			return ERR_EXIT_CODE;
		}
		string dest = destDir + string("/") + lfsId + "/";
		// ensure overriding in version control is possible fail will be realized in copy command
		(void)makeFileWritable((dest + "GracenoteDB.tgz"));
		(void)makeFileWritable((dest + "GracenoteDB.md5"));
		(void)makeFileWritable((dest + DB_REVISION_FILE));

		ETG_TRACE_COMP(("Make dir to %s", dest.c_str()));
		GNWL.mkdir_p(dest.c_str());

		cmd = string("mv ") + string(destTmp) + string("GracenoteDB.* ") + dest;
		res = execute(cmd);
		if (0 != res) {
			ETG_TRACE_ERR(("Problem to copy the GracenoteDB: errno %d, cmd \"%s\"", errno, cmd.c_str()));
			displayExitError();
			return ERR_EXIT_CODE;
		}
		cmd = string("cp ") + string(destTmp) + string(DB_REVISION_FILE) + string(" ") + dest;
		res = execute(cmd);
		if (0 != res) {
			ETG_TRACE_ERR(("Problem to copy the %20s: errno %d, cmd \"%s\"", DB_REVISION_FILE, errno, cmd.c_str()));
			displayExitError();
			return ERR_EXIT_CODE;
		}
	}

	GNWL.cb.progress = 0;
	GNWL.cb.problemreport = 0;

	fclose(errLog);
	errLog = 0;

	if (bRemoveTempDirContent) {
		res = execute(string("rm -rf ") + tempDir + "/*");
		if (0 != res) {
			ETG_TRACE_ERR(("Command rm -rf return with errno %d", errno ));
			displayExitError();
			return ERR_EXIT_CODE;
		}
	}

	// ensure copying is finished
	ETG_TRACE_COMP(("Syncing file system..."));
	sync();

	if (!changeDir(curDir)) {
		displayExitError();
		return ERR_EXIT_CODE;
	}

	printf("Release process finished! GN Update Data: %s/\n", destDir.c_str() );
	printf("GN work data for image production: %s/\n", workDir.c_str() );
	printf("Please check-in to version control and inform integration and data carrier team.\n" );

	OSAL_vProcessExit();
	return 0;

}
