/// \file GracenoteWrapperLib.h
///
/// Header file for Gracenote integration library
///
/// Copryright: (c) 2011 Robert Bosch GmbH
///
/// \author Ingo Reise CM-AI/PJ-GM28 (external.ingo.reise@de.bosch.com)
///
#ifndef _GRACENOTE_WRAPPER_LIB_H_
#define _GRACENOTE_WRAPPER_LIB_H_ 
// ignore these warnings: warning: deprecated conversion from string constant to ‘char*’
#pragma GCC diagnostic ignored "-Wwrite-strings"
#define __PLACEMENT_NEW_INLINE
#include <iostream>
#include <fstream>
#include <vector>
#include <set>
#include <map>
#include <list>
#include <algorithm>
#include "GracenoteWrapperInterface.h"
extern "C"
{
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include "gn_string.h"
#include "gn_validate.h"
#include "gn_init_emms.h"
#include "gn_abs_errors.h"
#include "gn_error_display.h"
#include "gn_textid_lookup.h"
#include "gn_textid_cache.h"
#include "gn_transcript_accessors.h"
#include "gn_lists_variations.h"
#include "xsampa_to_lnh_conversion.h"
#include "gnepal_string.h"
#include "gnepal_memory.h"
}

using namespace std;

#define EXCEPTIONLISTFILE "exceptionlist.lst"
#define SNAPSHOT_DATE_FILE "dbv1_snapshot_date"
#define DB_REVISION_FILE "dbv1_revision"
#define ERR_LOG_FILE "errorlog.txt"
#define GENRE_PHONEM_FILE "genre-phonemes.utf8" // Unicode-Text export from Excel delimited by TABs

// this value is used in the target environment if SDS TS inits the GNWL
static const char* gnBasePath = "/var/opt/bosch/gracenote/";

class tclGracenoteWrapper
{

private:
    /// Definition whether original or original phonemes are prioritized.
    ///
    /// Abstract: prioritizeOfficial => Do normalization
    /// E.g.: Query for "Wacko Jacko"
    /// When prioritizeOfficial is active "Michael Jackson" as official name
    /// will be used for TTS and as first SR entry
    /// When prioritizeOfficial is NOT active "Wacko Jacko" is prioritized and
    /// used for TTS. The official "Michael Jackson" will be set on the second
    /// place
    static const bool prioritizeOfficial = false;

    // Run GN subroutine and log error in case of failure
    static bool gnSuccess(string error_message, gn_error_t gn_status);

    /// Path where the Gracenote database resides
    string gnDbPath;

    // Display languages specifieres consist of 3 letters
    static const int displayLanguageLength = 3;
    char displayLanguage[displayLanguageLength + 1];

    /// Configuration of the GN library
    gn_emms_configuration_t gnConfig;

    /// Flag: GN database access is initialized
    bool gnEmmsIsInitialized;

    /// Flag: GN database configuration is initialized (i.e. cleared)
    bool gnConfigIsInitialized;

    // list file set id of the currently running database
    int lfsid;

    /// Cache for the extra delivery of genre phonemes
    map<pair<string, string>, list<string> > genrePhonemeCache;

    /// Internal method to reduce duplicate code
    bool gnQueryObject(tclGracenoteWrapperInterface::QueryType type,
                       gn_uchar_t* const queryString,
                       gn_textid_presult_data_t* result,
                       gn_pgenre_arr_t *genre_array,
                       gn_prep_array_t *representation_array) const;


    // Get (alternate) official representation
    gn_prepresentation_t getOfficialRepresentation(gn_spklangid_t const query_lang,
            gn_prep_array_t const representation_array) const;

    // Get matching representation
    gn_prepresentation_t getMatchingRepresentation(gn_prep_array_t const representation_array,
            const string &queryString) const;


    // Get readable display string of representation
    const string displayString(gn_prepresentation_t const representation) const;

    /// \brief Query representation_array for the first entry where the lowered
    /// display string exactly matches the lowered query_string
    int getMatchingRepresentationIndex(gn_prep_array_t const representationArray,
                                       string query_string,
                                       double maxDifference = 1.0) const;

    gn_uint32_t getOfficialRepresentationIndex(gn_prep_array_t const representation_array,
            const tclLanguageId &qLanguage) const;

    /// \brief Mapping pseudo numeric written language identifier (used by GN)
    /// codes to textual representations
    ///
    /// Spoken languages are specified by Gracenote by country and (written)
    /// language, e.g. DEU_ger. Written languages are specified using numbers
    /// (formatted as strings). This map allows to get the written language
    /// identifier for a spoken language:
    ///
    /// Example: \code gnWrittenLanguageMap["ger"] = 30 \endcode
    map<string, string> gnWrittenLanguageMap;

    // Look for written language for spoken language
    bool gnWrittenLanguage(const string &spokenLanguage, string &writtenLanguage) const;

    /// Convert X-Sampa to L&H+
    static string GN_XSAMPA2LHplus(const string &xsampastring, const tclLanguageId query_lang);
    static string XSAMPA2LHplus(const string src, const tclLanguageId lang);

    // map of supported languages per region
    /// This map is used for testing purposes only, the wrapper supports also
    /// languages not listed in this map (but there will be no tests)
    map<int, set<tclLanguageId > > supportedLanguages;
	
	// map containing invalid  strings detected at database delivery test
	// map<transcription,language>
	map<string,string> exceptions_map;

    ///  When looking for genres, the result depends on the configured display language
    void checkAndSetDisplayLangauge(tclLanguageId spokenLanguage);
    
    /// Calculate the Levenshtein Distance of two strings normalized to the 
    /// length of source
    static double levenshteinDistance(string source, string target);

public:
    struct {
       void (*progress)(int percent);
       void (*problemreport)(const char* report);
    } cb;
    void progress(int percent);
    void problemreport(const char * report);
    // Start/initialize GN database access
    bool start(bool SkipInitConfig = false, int lfsidParam = 0, const char* basePath = gnBasePath);

    // Stop GN database access
    void stop(void);

    // Query GN database for speech recognition
    int srQuery(tclGracenoteWrapperInterface::QueryType type,
                const tclLanguageId &qlang,
                tString qstring,
                size_t maxHits,
                tString phonemes[],
                bool returnDisplayString = false);


    // Query GN database for text to speech
    int ttsQuery(tclGracenoteWrapperInterface::QueryType type,
                 const tclLanguageId &qlang,
                 tString qstring,
                 tString phoneme,
                 bool returnDisplayString = false);

    // Default Constructor
    tclGracenoteWrapper();

    // Destructor
    ~tclGracenoteWrapper();

    // Get timestamp of last update
    tLong getUpdateTime(void) const;

    // Get timestamp of last update
    tLong getSnapshotTime(void) const;

    /*! \cond ALL */
    // create a directory including subdirs
    bool mkdir_p(const char *dir, mode_t mode = 0775);
    /// Install or update a new database from sourcedir
    bool installDatabase(const string &sourcedir);
    /// Update database from \code <basePath>/<lfsid>/* \endcode with calling installDatabase()
    bool updateDb(const string &basePath);
    /// Extract the DB from static and call updateDb() afterwards
    bool extractDb(const string& zipFileName, const string& destDir, const string& tempDir) ;

    /*! \endcond */


    /// test various features af the active database
    void test(void);

    void timing(void);



    int getContent(tclGracenoteWrapperInterface::QueryType type,
                   set<string> &elements) const;

    const string getOfficialName(tclGracenoteWrapperInterface::QueryType type,
                                 const tclLanguageId &qLanguage,
                                 const string &qstring);

    const string getCompleteInfo(tclGracenoteWrapperInterface::QueryType type,
                                 const tclLanguageId &qLanguage,
                                 const string &qstring) const;

    void makeMP3 ( tclGracenoteWrapperInterface::QueryType type,
                   const tclLanguageId &qLanguage,
                   const string &query_string,
                   const string &basefile ) const ;

    void getAllGenres(const tclLanguageId &qLanguage);

    static const list<int> getInstalledLfids(void);

    const char *getDefaultGnBasePath(){  return gnBasePath; }
    bool retrieveLFSIDfromKDS( int &lfsid );
public:
    void countPhonemes(void);

    /// Verify L&H+-phonemes against the allowed codeset for the specific language
    static bool verifyLHP(string phoneme, tclLanguageId lang);


};
#endif

