/// \file GracenoteWrapperLib.cpp
///
/// Gracenote integration library
///
/// This file contains the basic functionality
/// \see See GracenoteDBTests.cpp for debugging functions
///
/// Copryright: (c) 2011 Robert Bosch GmbH
///
/// \author Ingo Reise CM-AI/PJ-GM28 (external.ingo.reise@de.bosch.com)
///

// Note: gn_*_free_ functions are setting the pointer to NULL

#include "GracenoteWrapperLib.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <unistd.h> // exec
#include <sstream>
#include <map>
#include <fstream>
#include <iomanip>

#ifndef _LINUXX86MAKE_
#define DP_S_IMPORT_INTERFACE_BASE
#include "dp_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#endif

using namespace std;

static const int lfsidUninitialized = -1;
static const int lfsidInvalid = -2;

#define MAX_REGION_TYPE     (0x1C)

/* **************************************************************************
 * Function Default Constructor 
 * *************************************************************************/ 
/**
* Only variable initializations are done here, no file access
*/
tclGracenoteWrapper::tclGracenoteWrapper()
{
    // Initialize flags for the status of the GN-library
    gnEmmsIsInitialized = false;
    gnConfigIsInitialized = false;
    cb.progress = 0;
    cb.problemreport = 0;

    lfsid = lfsidUninitialized;

    // List of supported languages per region
    // Used only for testing purposes

    /// List of registered ListFileSetIDs;
    /// LFSID 698, North America 5.5
    supportedLanguages[698].insert(tclLanguageId("USA_eng"));
    supportedLanguages[698].insert(tclLanguageId("CAN_fre"));
    supportedLanguages[698].insert(tclLanguageId("MEX_spa"));

    /// LFSID 727, North America 5.6
    supportedLanguages[727].insert(tclLanguageId("USA_eng"));
    supportedLanguages[727].insert(tclLanguageId("CAN_fre"));
    supportedLanguages[727].insert(tclLanguageId("MEX_spa"));

    /// LFSID 735, Latin America 5.5
    supportedLanguages[705].insert(tclLanguageId("USA_eng"));
    supportedLanguages[705].insert(tclLanguageId("MEX_spa"));
    supportedLanguages[735].insert(tclLanguageId("USA_eng"));
    supportedLanguages[735].insert(tclLanguageId("MEX_spa"));
    supportedLanguages[735].insert(tclLanguageId("BRA_por"));

    /// LFSID 704, China 5.5
    supportedLanguages[704].insert(tclLanguageId("USA_eng"));
    supportedLanguages[704].insert(tclLanguageId("CHN_qad"));

    /// LFSID 734, China 5.6
    supportedLanguages[734].insert(tclLanguageId("USA_eng"));
    supportedLanguages[734].insert(tclLanguageId("CHN_qad"));

    /// LFSID 733, Korea 5.6
    /*
    supportedLanguages[733].insert(tclLanguageId("KOR_kor"));
	*/

    /// LFSID 731, Europe 5.5
    supportedLanguages[701].insert(tclLanguageId("GBR_eng"));
    supportedLanguages[701].insert(tclLanguageId("FRA_fre"));
    supportedLanguages[701].insert(tclLanguageId("DEU_ger"));
    supportedLanguages[701].insert(tclLanguageId("RUS_rus"));
    supportedLanguages[701].insert(tclLanguageId("ESP_spa"));
    supportedLanguages[731].insert(tclLanguageId("GBR_eng"));
    supportedLanguages[731].insert(tclLanguageId("FRA_fre"));
    supportedLanguages[731].insert(tclLanguageId("DEU_ger"));
    supportedLanguages[731].insert(tclLanguageId("RUS_rus"));
    supportedLanguages[731].insert(tclLanguageId("ITA_ita"));
    supportedLanguages[731].insert(tclLanguageId("ESP_spa"));
    supportedLanguages[731].insert(tclLanguageId("NLD_dut"));
    supportedLanguages[731].insert(tclLanguageId("PRT_por"));
    supportedLanguages[731].insert(tclLanguageId("TUR_tur"));

    /// LFSID 692, 5.5 Global-DB, not for productive use
    supportedLanguages[692].insert(tclLanguageId("USA_eng"));
    supportedLanguages[692].insert(tclLanguageId("GBR_eng"));
    supportedLanguages[692].insert(tclLanguageId("AUS_eng"));
    supportedLanguages[692].insert(tclLanguageId("FRA_fre"));
    supportedLanguages[692].insert(tclLanguageId("CAN_fre"));
    supportedLanguages[692].insert(tclLanguageId("ESP_spa"));
    supportedLanguages[692].insert(tclLanguageId("MEX_spa"));
    supportedLanguages[692].insert(tclLanguageId("PRT_por"));
    supportedLanguages[692].insert(tclLanguageId("BRA_por"));
    supportedLanguages[692].insert(tclLanguageId("ITA_ita"));
    supportedLanguages[692].insert(tclLanguageId("DEU_ger"));
    supportedLanguages[692].insert(tclLanguageId("NLD_dut"));
    supportedLanguages[692].insert(tclLanguageId("RUS_rus"));
    supportedLanguages[692].insert(tclLanguageId("TUR_tur"));
    supportedLanguages[692].insert(tclLanguageId("CHN_qad"));
    supportedLanguages[692].insert(tclLanguageId("JPN_jpn"));

    /// LFSID 827, SANZA 5.6
    /*
    supportedLanguages[827].insert(tclLanguageId("GBR_eng"));
    supportedLanguages[827].insert(tclLanguageId("AUS_eng"));
	*/
	
    /// LFSID 831, SANZA+KOR 5.6
    supportedLanguages[831].insert(tclLanguageId("USA_eng"));
    supportedLanguages[831].insert(tclLanguageId("AUS_eng"));
	supportedLanguages[831].insert(tclLanguageId("GBR_eng"));
    supportedLanguages[831].insert(tclLanguageId("KOR_kor"));

}

/* **************************************************************************
 * Function Destructor 
 * *************************************************************************/ 
/**
* Includes shutdown if necessary
*/
tclGracenoteWrapper::~tclGracenoteWrapper()
{
    stop();
}

/* **************************************************************************
 * Function gnSuccess 
 * *************************************************************************/ 
/**
 * Run GN subroutine and log error in case of failure
 * 
 * \param[in] error_message Text to display when subroutine failed
 * \param[in] gn_status Return value of GN routine
 * \return true if command succeeded, false else
 * 
 */
bool tclGracenoteWrapper::gnSuccess(string error_message, gn_error_t gn_status)
{
   if (GN_SUCCESS != gn_status)
   {
      string s = string("Failed to ") + error_message + ":";
      s += (const char*)gnerr_get_code_desc(GNERR_ERROR_CODE(gn_status));
      //cb.problemreport(s.c_str()); // TODO improve...
#ifndef ARM
      printf("%s\n", s.c_str());
#endif
      return false;
   } else {
      return true;
   }
}

void tclGracenoteWrapper::progress(int percent) {
   if (cb.progress) {
      cb.progress(percent);
   }
}
void tclGracenoteWrapper::problemreport(const char * report) {
   if (cb.problemreport) {
      cb.problemreport(report);
   }
}

/* **************************************************************************
 * Function retrieveLFSIDfromKDS
 * *************************************************************************/
/**
 * The region needs to determined from the KDS.
 *
 * \param[out]  lfsid      Gracenote LFSid region found.
 *
 * \return true if found, otherwise false.
 */
bool tclGracenoteWrapper::retrieveLFSIDfromKDS( int &lfsid )
{
    bool result = false;
#ifndef _LINUXX86MAKE_
    static const int allianceRegionToGracenoteRegion[MAX_REGION_TYPE] = {
            727, // US    [0x00]
            727, // CAN   [0x01]
            735, // MEX   [0x02]
            731, // UK    [0x03]
            731, // TKY   [0x04]
            731, // RUS   [0x05]
            731, // other EUR [0x06]
            734, // PRC   [0x07]
            734, // TWN   [0x08]
            734, // HKG   [0x09]
            731, // GCC   [0x0A]
            731, // EGP   [0x0B]e
            831, // ASR/NZE [0x0C]
            735, // BRA   [0x0D]
            735, // AGT   [0x0E]
            735, // other LAC [0x0F]
            831, // SAF   [0x10]
            831, // THI   [0x11]
            831, // SGP - ASIA [0x12]
            831, // MLY - ASIA [0x13]
            831, // BRN - ASIA [0x14]
            831, // INN - ASIA [0x15]
            831, // VNM - ASIA [0x16]
            831, // PHL - ASIA [0x17]
            731, // IND - ASIA [0x18]
            831, // JPN [0x19]
            831, // KOR [0x1A]
            831 // Other GOM [0x1B]
           };

    tU8 regionType;
    tS32 error;

    error = DP_s32GetConfigItem("VehicleInformation", "DestinationRegion1", &regionType, 1);
    if (DP_S32_NO_ERR == error)
    {
       if( regionType < MAX_REGION_TYPE )
       {
    	   lfsid = allianceRegionToGracenoteRegion[regionType];
       }
       else
       {
           problemreport("Region type from KDS out of range\n");
           lfsid = 727; //NAR
       }
       result = true;
    }
    else
    {
        problemreport("Region could not be read from KDS.");
    }
#endif
   return result;
}

/* **************************************************************************
 * Function start 
 * *************************************************************************/ 
/**
* Start/initialize GN database access
* 
* This will open all needed files
* Start/Stop session is needed for performance reasons. Opening a session for 
* each call would take much longer for each query
* Queries will not work in case of failure
* \return bool for success
*/
bool tclGracenoteWrapper::start(bool SkipInitConfig, int lfsidParam, const char* basePath)
{
    ostringstream DbPath;
    const string marketingRegionToken("MARKETING_REGION");
    progress(101);

    stop();

    progress(102);

    if (lfsidParam != 0)
    {
        lfsid = lfsidParam;
        // This is where the (installed) database is stored.
        // Final "/" is mandatory here!
        DbPath << basePath << lfsid << "/";
        gnDbPath = DbPath.str();
        progress(103);
    }
    // lfsid is only set once, so that a manually specified lfsid remains
    // unchanged at stop()/start() cycles.
    else if (lfsid == lfsidUninitialized)
    {
        if( !retrieveLFSIDfromKDS(lfsid))
        {
            // Default to NorthAmerica
            lfsid = 727;
        }
        progress(104);

        // This is where the (installed) database is stored.
        // Final "/" is mandatory here!
        DbPath << basePath << lfsid << "/";
        gnDbPath = DbPath.str();
        progress(106);
    }
    progress(107);
    mkdir_p(gnDbPath.c_str());

    progress(108);
    if (!SkipInitConfig) {
        if (!gnSuccess("initialize config",
                       gnconf_initialize_configuration(&gnConfig))){
            return false;
        }
    }
    progress(109);
    gnConfigIsInitialized = true;
    gnConfig.base_database_path = (gn_uchar_t*) gnDbPath.c_str();
    gnConfig.set_base_database_path = GN_TRUE;

    gnConfig.enable_playlist = (gn_uchar_t*) GN_STR_TRUE;
    gnConfig.set_enable_playlist = GN_TRUE;

    gnConfig.enable_textid = (gn_uchar_t*) GN_STR_TRUE;
    gnConfig.set_enable_textid = GN_TRUE;

    gnConfig.enable_textidalb = (gn_uchar_t*) GN_STR_TRUE;
    gnConfig.set_enable_textidalb = GN_TRUE;

    gnConfig.enable_mediavocs = (gn_uchar_t*) GN_STR_TRUE;
    gnConfig.set_enable_mediavocs = GN_TRUE;

    // These magic codes are communicated by Don Ross by mail 17.02.2011
    // INFO: Don Ross left Gracenote meanwhile
    gnConfig.client_id = (gn_uchar_t*) "5045504";
    gnConfig.set_client_id = GN_TRUE;

    gnConfig.client_id_tag = (gn_uchar_t*) "FD930221DE043389EB239A63FCE42598";
    gnConfig.set_client_id_tag = GN_TRUE;

    gnConfig.LDAC_key = (gn_uchar_t*) "LDAC000100109AFD2F42B16A47EC8557B500939240EA";
    gnConfig.set_LDAC_key = GN_TRUE;

    progress(110);
    // Start Gracenote system now
    if (gnSuccess("initialize emms",
                  gninit_initialize_emms(&gnConfig)))
    {
        gn_uint32_t count;
        gn_available_language_t *language_array;

        // Emms started successfully, ...
        gnEmmsIsInitialized = true;

        progress(111);
        // ..build up list of written languages
        gnSuccess("get available languages",
                  gn_list_get_available_languages(&count, &language_array));

        // Example: gnWrittenLanguageMap["ger"] = 30
        for (gn_uint32_t i = 0; i < count ; ++i)
            gnWrittenLanguageMap[string((char *)language_array[i].lang_id)] =
                string((char *) language_array[i].lang_id_code);

        progress(112);
        gnSuccess("free available languages",
                  gn_list_free_available_languages(count, &language_array));
    } else {
        gnEmmsIsInitialized = false;
    }

    progress(113);

    if (gnEmmsIsInitialized && genrePhonemeCache.size() == 0)
    {
        // Read extra genre phoneme table delivered by Gracennote
        // This is a fixed table, so we'll read it in a fixed manner
        char* genreLangs[] =
            { "USA_eng", "DEU_ger", "FRA_fre", "CAN_fre", "ITA_ita", "XXX_XXX",
              "MEX_spa", "BRA_por", "PRT_por", "RUS_rus", "CHN_qad", NULL
            };

        // Unicode-Text export from Excel delimited by TABs
        char delimiter = '\t';
        genrePhonemeCache.clear();
        ostringstream genreCacheFileNameS;
        genreCacheFileNameS << gnDbPath << GENRE_PHONEM_FILE;
        ifstream genreCacheFile(genreCacheFileNameS.str().c_str());

        progress(114);
        if (genreCacheFile.good())
        {
            string line, lineElement, genre;

            progress(115);
            // Skip first three lines
            for (int i = 0; i < 3; ++i)
            {
                getline(genreCacheFile, line);
                if (0 == line.compare(0, 5, "Genre"))
                    break;
            }
            while (true)
            {
                getline(genreCacheFile, line);
                if (!genreCacheFile.good())
                    break;

                // Tokenize each line by delimiters
                stringstream lineStream(line);

                // First column: English genre name
                getline(lineStream, genre, delimiter);

                for (char **lang = genreLangs; *lang != NULL; ++lang)
                {
                    string localName, xsampa;

                    // Now two columns each for each language
                    getline(lineStream, localName, delimiter);
                    getline(lineStream, xsampa, delimiter);

                    if (xsampa[xsampa.size() - 1] == '"' &&
                            xsampa[0] == '"')
                    {
                        xsampa.erase(xsampa.size() - 1, 1);
                        xsampa.erase(0, 1);
                    }

                    if (xsampa.size() > 0)
                    {
                        // Alternatives phonemes are separated by ","
                        stringstream xsampaStream(xsampa);
                        string xsampaElement;
                        while (getline(xsampaStream, xsampaElement, ','))
                        {
                            // Remove quotes added by the Excel-Export
                            size_t qIndex;


                            while (string::npos != (qIndex = xsampaElement.find("\"\"")))
                                xsampaElement.replace(qIndex, 2, "\"");
                            // Trim leading/trailing blanks
                            while (xsampaElement[0] == ' ')
                                xsampaElement.erase(0, 1);
                            while (xsampaElement[xsampaElement.size() - 1] == ' ')
                                xsampaElement.erase(xsampaElement.size() - 1, 1);
                            // Add this phonemes to cache
                            if (string(*lang) != string("XXX_XXX"))
                            {
#ifdef NEVER_LOCAL_COMPILE
                                // LH+-conversion is quite slow, so check phonemes only during development
                                // (the genre-phoneme list won't be updated by the user)
                                string LHplus = XSAMPA2LHplus(xsampaElement, (gn_spklangid_t) * lang);
                                //                                if (string(*lang) == string("DEU_ger"))
                                //                                    cout << localName << SPA << LHP << LHplus << LHP << LPA << endl;
                                if (!verifyLHP(LHplus, *lang))
                                    cerr << "X-SAMPA: " << xsampaElement << endl;
                                else
#endif

                                    genrePhonemeCache[make_pair(*lang, localName)].push_back(xsampaElement);
                            }
                        }
                    }
                }
            }
        }
    }
	
    progress(116);
	// load exception list if it exists
	ifstream exceptions((string(gnDbPath)+EXCEPTIONLISTFILE).c_str());
	if(exceptions.good())
	{
      progress(117);
		while (true)
		{
			string line;
			getline(exceptions, line);
			if (!exceptions.good())
			{
				break;
			}
			// cout << "Exception added: " << line <<  endl;
			exceptions_map.insert(pair<string,string>(line," "));
		}
	}
	exceptions.close();
   progress(118);

	return gnEmmsIsInitialized;
}

/* **************************************************************************
 * Function stop 
 * *************************************************************************/ 
/**
 * Stop GN database access
 * 
 * Close all open file handles and free memory 
 */
void tclGracenoteWrapper::stop()
{

   progress(201);
    // Clear WrittenLanguageMap
    gnWrittenLanguageMap.clear();

   progress(202);
// Clear Exceptions
	exceptions_map.clear();
	
   progress(203);
    // Stop system
    if (gnEmmsIsInitialized){
        gninit_shutdown_emms();
        gnEmmsIsInitialized = false;
    }

   progress(204);
    // Note: The config remains initialized, even if unused
}

/* **************************************************************************
 * Function   gnWrittenLanguage
 * *************************************************************************/ 
/**
 * Look for written language for spoken language
 * \see See gnWrittenLanguageMap for details
 * 
 * \param[in] spokenLanguage e.g. DEU_ger
 * \param[out] writtenLanguage found written language e.g. "30" 
 * \return true if written language was found
 */
bool tclGracenoteWrapper::gnWrittenLanguage (
    const string &spokenLanguage,
    string &writtenLanguage
) const
{
    map<string, string>::const_iterator wlmIterator =
        gnWrittenLanguageMap.find(spokenLanguage.substr(4));

    if (wlmIterator != gnWrittenLanguageMap.end())
    {
        writtenLanguage = wlmIterator->second;
        return true;
    }
    return false;
}

/* **************************************************************************
 * Function  getUpdateTime 
 * *************************************************************************/ 
/**
 * Gets the timestamp of the last database update
 * 
 * This is the time when the database was installed on the target
 * 
 * \return seconds since 1970; 0 in case of an error
 */
tLong tclGracenoteWrapper::getUpdateTime(void) const
{
    struct stat statBuffer;
    ostringstream lfsidFileName;
    lfsidFileName << gnDbPath << SNAPSHOT_DATE_FILE;
    if (0 == stat(lfsidFileName.str().c_str(), &statBuffer))
        return statBuffer.st_mtime;
    else
        return 0;
}

/* **************************************************************************
 * Function  getSnapshotTime 
 * *************************************************************************/ 
/**
 * Gets the timestamp of the database snapshot
 * 
 * This is the time when the database content was filtered from the master 
 * database at Gracenote
 * 
 * \return seconds since 1970; 0 in case of an error
 */
tLong tclGracenoteWrapper::getSnapshotTime(void) const
{
    gn_sys_install_profile_t* installedProfile = NULL;
    // Example for snapshotDate:
    // 2011-03-18 00:40:02Z
    const char *gnFormat = "%Y-%m-%d %H:%M:%SZ";
    struct tm tm;

    // Ask GN-API if library is up
    if (gnConfigIsInitialized &&
            gnSuccess("gn_sys_get_install_profile",
                      gn_sys_get_install_profile(&installedProfile)))
    {
        strptime((char *) installedProfile->data_snapshot_date, gnFormat, &tm);
        return mktime(&tm);

    }
    else
    {
        // Look for textfile if lib is not yet up
        ifstream snapshotDateFile((string(gnDbPath) + SNAPSHOT_DATE_FILE).c_str());
        if (snapshotDateFile.good())
        {
            string snapshotDate;
            getline(snapshotDateFile, snapshotDate);
            strptime(snapshotDate.c_str(), gnFormat, &tm);
            return mktime(&tm);

        }
    }
    return 0;
}

/* **************************************************************************
 * Function  displayString
 * *************************************************************************/ 
/**
 * Get (readable) display string of representation
 * 
 * \param[in] representation related representation pointer
 * \return readable string
  */
const string tclGracenoteWrapper::displayString(gn_prepresentation_t const representation) const
{
    gn_uchar_t *display_string;
    if (GN_SUCCESS ==
            gn_representation_get_display_string(representation, &display_string))
        return string((char *)display_string);
    else
        return "error in get_display_string";
}

/* **************************************************************************
 * Function levenshteinDistance 
 * *************************************************************************/ 
/**
* Calculate the levenshtein distance of two strings. The result is normalized to 
* the size of the first string
* 
* \see http://en.wikipedia.org/wiki/Levenshtein_distance
* 
* \param[in] source First string to compare
* \param[in] target Second string to compare
* \return Levenshtein distance normalized to length(source)
*/
double tclGracenoteWrapper::levenshteinDistance(string source, string target)
{
    // Transform both string to lowercase to ignore case duing calculation
    std::transform(source.begin(),
                   source.end(),
                   source.begin(),
                   static_cast < int (*) (int) > (tolower));
    std::transform(target.begin(),
                   target.end(),
                   target.begin(),
                   static_cast < int (*) (int) > (tolower));

    const int n = source.length();
    const int m = target.length();
    if (n == 0 || m == 0)
        return 1.0;

    typedef std::vector< std::vector<int> > Tmatrix;
    Tmatrix matrix(n + 1);

    // Size the vectors in the 2.nd dimension. Unfortunately C++ doesn't
    // allow for allocation on declaration of 2.nd dimension of vec of vec
    for (int i = 0; i <= n; i++)
        matrix[i].resize(m + 1);

    for (int i = 0; i <= n; i++)
        matrix[i][0] = i;

    for (int j = 0; j <= m; j++)
        matrix[0][j] = j;

    for (int i = 1; i <= n; i++)
    {
        const char s_i = source[i - 1];

        for (int j = 1; j <= m; j++)
        {
            const char t_j = target[j - 1];

            int cost = (s_i == t_j ? 0 : 1 );

            const int above = matrix[i - 1][j];
            const int left = matrix[i][j - 1];
            const int diag = matrix[i - 1][j - 1];
            int cell = min( above + 1, min(left + 1, diag + cost));

            // Cover transposition, in addition to deletion,
            // insertion and substitution. This step is taken from:
            // Berghel, Hal ; Roach, David : "An Extension of Ukkonen's
            // Enhanced Dynamic Programming ASM Algorithm"
            // (http://www.acm.org/~hlb/publications/asm/asm.html)

            if (i > 2 && j > 2)
            {
                int trans = matrix[i - 2][j - 2] + 1;
                if (source[i - 2] != t_j)
                    trans++;
                if (s_i != target[j - 2])
                    trans++;
                if (cell > trans)
                    cell = trans;
            }

            matrix[i][j] = cell;
        }
    }
    return (double) matrix[n][m] / (double) source.length();
}

bool tclGracenoteWrapper::mkdir_p(const char *dir, mode_t mode) {
   size_t len;
   bool ret = false;
   if(dir && (len = strlen(dir))) {
      ret = true;
      char *tmp = new char[len+1];
      char *p = NULL;
      (void)memcpy(tmp, dir, len+1);

      if(tmp[len - 1] == '/') {
         tmp[len - 1] = 0; // remove trailing "/"
      }
      for(p = tmp + 1; *p; p++) {  // 1st char to avoid zero mkdir for root
         if(*p == '/') {
            *p = 0;
            if(0 != mkdir(tmp, mode)) {
               if(EEXIST != errno) {
                  ret = false;
                  break;
               }
            }
            *p = '/';
         }
      }
      if(0 != mkdir(tmp, mode)) {
         if(EEXIST != errno) {
            ret = false;
         }
      }
      delete[] tmp;
   }
   return ret;
}

/* **************************************************************************
 * Function  installDatabase
 * *************************************************************************/ 
/**
 * Install or update a new database from sourcedir
 * 
 * \see This method has to be called by updateDb
 * 
 * \remark Run start() prior and after installDatabase
 * \param[in] sourcedir Path (with terminating /) where the new database content 
 * is located
 * \return true for success, 
 * \return false is case of errors. State of DB is unclear. Try update again
 * 
 */
bool tclGracenoteWrapper::installDatabase(const string &sourcedir)
{
    if (!gnConfigIsInitialized){
       string s = string("gnConfig is not initialized!");
       problemreport(s.c_str());
       return false;
    }
    progress(10);
    // Check if the directory exists at all
    struct stat statBuf;
    if (!(0 == stat(sourcedir.c_str(), &statBuf) && S_ISDIR(statBuf.st_mode)))
    {
        // Sourcedir is not available or not a directory
        string s = string("Sourcedir is not available or not a directory: ") + sourcedir.c_str();
        problemreport(s.c_str());
        return false;
    }

    progress(11);

    // Don't try installing on a running system
    stop();
    progress(20);

    string newSnapshotDateFileName(string((char *) gnConfig.base_database_path) + SNAPSHOT_DATE_FILE);
    string baseDBRevisionFileName(string((char *) gnConfig.base_database_path) + DB_REVISION_FILE);

    if (0 == stat(gnDbPath.c_str(), &statBuf) && S_ISDIR(statBuf.st_mode))
    {
        // Remove existing database first
        gninit_uninstall_local_db(&gnConfig, GNINIT_INSTALL_ALL);
        progress(21);
        // Remove private side files also
        remove(newSnapshotDateFileName.c_str());
        remove(baseDBRevisionFileName.c_str());
    } else {
       progress(22);
       if(!mkdir_p(gnDbPath.c_str())) {
          string s = string("Can't create the gnDbPath: ") + gnDbPath;
          problemreport(s.c_str());
          return false;
       }
       progress(23);
    }

    progress(30);

    // Install the new one
    if (!gnSuccess("install local db",
                   gninit_install_local_db (&gnConfig,
                                            (gn_uchar_t*) sourcedir.c_str(),
                                            lfsid,
                                            GNINIT_INSTALL_ALL)))
    {
        return false;
    }
    progress(90);
    // Add file containing snapshot_date
    {
        string newSnapshotDate;
        ofstream newSnapshotDateFile (newSnapshotDateFileName.c_str());

        ifstream inputSnapshotDateFile(string(sourcedir + SNAPSHOT_DATE_FILE).c_str());
        // A file provided on the data carrier overrides the internal date
        if (inputSnapshotDateFile.good())
        {
           progress(91);
           getline(inputSnapshotDateFile, newSnapshotDate);
           newSnapshotDateFile << newSnapshotDate << endl;
        }
        else
        {
           gn_sys_install_profile_t* installedProfile = NULL;

           progress(92);
           // Temporarily Start GN lib to ...
           start();

            // ... Retrieve current installation profile
            if (gnSuccess("gn_sys_get_install_profile",
                          gn_sys_get_install_profile(&installedProfile)))
            {
                newSnapshotDateFile << (char *)installedProfile->data_snapshot_date << endl;
                gn_sys_destroy_install_profile (&installedProfile);
            }
            progress(93);
            stop();
        }
    }

    progress(94);
    // Add file containing base_db_revision
    {
        ofstream baseDBRevisionFile (baseDBRevisionFileName.c_str());
        gn_sys_install_profile_t* installedProfile = NULL;

        // Temporarily Start GN lib to ...
        start();
        progress(95);
        // ... Retrieve current installation profile
        if (gnSuccess("gn_sys_get_install_profile",
                      gn_sys_get_install_profile(&installedProfile)))
        {
            baseDBRevisionFile << (int)installedProfile->base_db_revision << endl;
            gn_sys_destroy_install_profile (&installedProfile);
        }
        progress(96);
        stop();
    }
    progress(97);
    return true;
}

/* **************************************************************************
 * Function  updateDb
 * *************************************************************************/ 
/**
 * Update database from \code <basePath>/<lfsid>/<data> \endcode
 * 
 * This method includes stopping the Service
 * 
 * \param[in] basePath full qualified path 
 * \return true for success else false
 */
bool tclGracenoteWrapper::updateDb(const string &basePath)
{
	bool returnvalue = true;
    ostringstream sourcePath;
    sourcePath << basePath << "/" << lfsid << "/";
    progress(1);
    if(!mkdir_p(sourcePath.str().c_str())) {
       return false;
    }
    progress(2);
    // copy exceptionlist
    string exceptionFile = sourcePath.str() + EXCEPTIONLISTFILE;
    struct stat statBuffer;
    if (0 == stat(exceptionFile.c_str(), &statBuffer) && S_ISREG(statBuffer.st_mode)) {
       ostringstream copyCommand;
       copyCommand << "cp " << exceptionFile.c_str() << " " << gnDbPath;
       progress(3);
       system(copyCommand.str().c_str());
    }
    progress(4);

    string genreFile = sourcePath.str() + GENRE_PHONEM_FILE;
    if (0 == stat(genreFile.c_str(), &statBuffer) && S_ISREG(statBuffer.st_mode)) {
       ostringstream copyCommand;
       copyCommand << "cp " << genreFile.c_str() << " " << gnDbPath;
       progress(5);
       system(copyCommand.str().c_str());
    }
    progress(6);

    // call GN wrapper func
    returnvalue = installDatabase(sourcePath.str());
    progress(100);

    return returnvalue;
}

/* **************************************************************************
 * Function  extractDb
 * *************************************************************************/
/**
 * Unzip the encrypted data base from \code <basePath> \endcode to tempPath
 * and call updateDB with destDir afterwards.
 *
 * This method includes stopping the Service
 *
 * \param[in]
 * \return true for success else false
 */
bool tclGracenoteWrapper::extractDb(const string& zipFileName, const string& destDir, const string& tempDir) {
   // extract the content of the ip file into one lfsid directory

   int res = mkdir(tempDir.c_str(), S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH);
   if ((0 != res) && (EEXIST != errno)) {
      return false;
   }
   //unpack the DB
   res = execl("/bin/tar", "-xzf", zipFileName.c_str(),"-C ", tempDir.c_str(), (const char*)NULL);
   if(0 != res){
      char str[1000];
      sprintf(str, "Problem with extracting errno %d the zip %s to %s\n",errno ,zipFileName.c_str(), tempDir.c_str());
      problemreport(str);
      //return false; // disabled because system reports 10 = No child processes but all is OK
   }
   return updateDb(destDir);
}


/* **************************************************************************
 * Function  getMatchingRepresentationIndex
 * *************************************************************************/ 
/**
 * Look for the best matching representation in representation_array
 * 
 * The Levenshtein Distance is calculated to find the best match.
 * 
 * \param[in] representationArray array with all representations 
 * \param[in] queryString displayString to look for 
 * \param[in] maxDifference (optional) Match only item with a maximum weighted 
 * levenshtein distance. This prevents matches on completely different strings.
 *  
 * \return Index of the best match, -1 if no match (better than maxDifference)
 * 
 */
int tclGracenoteWrapper::getMatchingRepresentationIndex(gn_prep_array_t const representationArray,
        string queryString,
        double maxDifference) const
{
    gn_uint32_t repArraySize;
    gn_prepresentation_t representation = NULL;
    double dist;
    // The Levenshtein distance is normalized to string length. So 1 should be max
    double minDist = 10.0;
    // index of the best matching element
    int minDistIndex = -1;

    gnSuccess("get element count of representation_array",
              gn_representation_get_element_count(representationArray,
                                                  &repArraySize));

    // lowercase queryString to be case insensitive
    std::transform(queryString.begin(),
                   queryString.end(),
                   queryString.begin(),
                   static_cast < int (*) (int) > (tolower));

    for (gn_uint32_t repArrayElement = 0; repArrayElement < repArraySize; ++repArrayElement)
    {
        gnSuccess("gn_representation_get_rep",
                  gn_representation_get_rep(representationArray,
                                            repArrayElement,
                                            &representation));

        string representationString = displayString(representation);

        // lowercase representationString to be case insensitive
        std::transform(representationString.begin(),
                       representationString.end(),
                       representationString.begin(),
                       static_cast < int (*) (int) > (tolower));

        if ((dist = levenshteinDistance(queryString, representationString)) < minDist)
        {
            minDist = dist;
            minDistIndex = repArrayElement;
        }
    }

    if (minDist < maxDifference)
        return minDistIndex;
    else
        return -1;
}


/* **************************************************************************
 * Function  getMatchingRepresentation
 * *************************************************************************/ 
/**
 * Get matching representation
 * 
 * Returns the first representation where the display string 
 * matches the queryString 
 * 
 * \param[in] representationArray representation_array
 * \param[in] queryString String to look for
 * \return Official representation or NULL when no match available
 * 
 */
gn_prepresentation_t tclGracenoteWrapper::getMatchingRepresentation(
    gn_prep_array_t const representationArray,
    const string &queryString) const
{
    int index;
    gn_prepresentation_t representation = GN_NULL;

    index = getMatchingRepresentationIndex(representationArray, queryString);
    if (index != -1)
        gn_representation_get_rep(representationArray,
                                  index,
                                  &representation);
    return representation;
}


/* **************************************************************************
 * Function  getOfficialRepresentationIndex
 * *************************************************************************/ 
/**
 * Look for the index of the official representation
 * \param[in] representationArray array with all representations 
 * \param[in] qLanguage In some rare cases there are different official
 * representations. Up to now seen only for RUS_rus 
 * \return Index of the best match, -1 if no match (better than maxDifference)
 */
gn_uint32_t tclGracenoteWrapper::getOfficialRepresentationIndex(
    gn_prep_array_t const representationArray,
    const tclLanguageId &qLanguage) const
{
    gn_uint32_t repArraySize;
    gn_prepresentation_t officialRepresentation = GN_NULL;
    gn_prepresentation_t representation = GN_NULL;
    int index = -1;
    string representationString;

    officialRepresentation = getOfficialRepresentation(
                                 (gn_spklangid_t) qLanguage,
                                 representationArray
                             );

    gnSuccess("get element count of representation_array",
              gn_representation_get_element_count(representationArray,
                                                  &repArraySize));

    for (gn_uint32_t repArrayElement = 0; repArrayElement < repArraySize; ++repArrayElement)
    {
        gnSuccess("gn_representation_get_rep",
                  gn_representation_get_rep(representationArray,
                                            repArrayElement,
                                            &representation));
        // Both representation and officialRepresentation are just pointers in
        // same array. So == is just comparing pointers but this is ok
        if (representation == officialRepresentation)
        {
            index = repArrayElement;
            break;
        }
    }
    return index;
}

/* **************************************************************************
 * Function getOfficialRepresentation  
 * *************************************************************************/ 
/**
 * Get (alternate) official representation
 * 
 * There may be multiple official representations corresponding to
 * different written languages. The queryLanguage parameter specifies the
 * language desired.
 * 
 * \param[in] queryLanguage Spoken language to look for
 * \param[out] representationArray representation_array
 * \return Official representation
 * 
 */
gn_prepresentation_t tclGracenoteWrapper::getOfficialRepresentation(
    gn_spklangid_t const queryLanguage,
    gn_prep_array_t const representationArray) const
{
    string writtenLanguage;
    gn_prepresentation_t representation = GN_NULL;

    gn_uint32_t repArraySize;
    gnSuccess("get element count of representation_array",
              gn_representation_get_element_count(representationArray,
                                                  &repArraySize));

    if (repArraySize == 0)
        return GN_NULL;

    // If there's a official representation for the requested spoken language
    // take this one, ...
    if (gnWrittenLanguage((char *) queryLanguage, writtenLanguage))
        gnSuccess("get official representation",
                  gn_representation_get_official(representationArray,
                                                 &representation,
                                                 (gn_uchar_t*) writtenLanguage.c_str())
                 );

    // ... otherwise search for an official representation in the
    // default language of the artist/album
    if (GN_NULL == representation)
        gnSuccess("get official representation",
                  gn_representation_get_official(representationArray,
                                                 &representation,
                                                 (gn_uchar_t*) GN_NULL)
                 );

    return representation;
}

/* **************************************************************************
 * Function  gnQueryObject
 * *************************************************************************/ 
/**
 * Internal method to reduce duplicate code
 * 
 * Query GN database for an artist/album/genre and return representation array
 * 
 * \return true for success, false else
 */
bool tclGracenoteWrapper::gnQueryObject(tclGracenoteWrapperInterface::QueryType queryType,
                                        gn_uchar_t * const queryString,
                                        gn_textid_presult_data_t *result,
                                        gn_pgenre_arr_t *genreArray,
                                        gn_prep_array_t *repArray) const
{
    gn_error_t gn_status;
    gn_textid_match_source_t match_source;
    gn_textid_match_type_t match_type;
    gn_uint32_t match_count;
    gn_pfile_data_t pfile_data = NULL;

    // Internal function, no need to check for initialized system here

    // Note: The representationArray is part of the genreArray. So the genreArray
    // may not be cleared as this would kill the representation array

    if (!gnSuccess("initialize TextID",
                   gn_textid_file_data_init(&pfile_data)))
        return false;  /* Nothing to free here */

    if (tclGracenoteWrapperInterface::ALBUM == queryType)
        gnSuccess("gn_textid_file_data_set_disc_title",
                  gn_textid_file_data_set_disc_title (pfile_data, queryString));
    else if (tclGracenoteWrapperInterface::ARTIST == queryType)
        gnSuccess("gn_textid_file_data_set_disc_artist",
                  gn_textid_file_data_set_disc_artist (pfile_data, queryString));
    else
        gnSuccess("set genre",
                  gn_textid_file_data_set_genre (pfile_data, queryString));

    /* Errors may happen here, e.g. due to incorrect chars in query string
     * so do not report errors */
    gn_textid_local_lookup (pfile_data,
                            GN_TEXTID_LU_FLAG_EXACT_ONLY,
                            //                           GN_TEXTID_LU_FLAG_DFLT,
                            &match_source,
                            &match_type,
                            &match_count);
    if (0 == match_count)
    {
        // Nothing found for this query
        gn_textid_file_data_smart_free(&pfile_data);
        return false;
    }

    // Yes, this list is 1-based
    // Note: I've never seen more than one element
    if (!gnSuccess("get result",
                   gn_textid_get_result(1, result)))
    {
        gn_textid_file_data_smart_free(&pfile_data);
        return false;
    }

    if (tclGracenoteWrapperInterface::ALBUM == queryType)
        gn_status = gn_textid_get_album_title_representation_array(*result,
                    repArray);
    else if (tclGracenoteWrapperInterface::ARTIST == queryType)
        gn_status = gn_textid_get_primary_artist_representation_array(*result,
                    repArray);
    else
    {
        // For genres the API delivers an genre_array of representation_arrays of
        // transcription_arrays
        gn_pgenre_t genre = GN_NULL;
        gn_uint32_t numberOfGenreLevels = 0;
        gn_status = GNERR_MemInvalid; // Dummy for NOT GN_SUCCESS

        if (gnSuccess("gn_textid_get_genre_array",
                      gn_textid_get_genre_array(*result,
                                                genreArray)))
        {
            gnSuccess("gn_genre_array_get_element_count",
                      gn_genre_array_get_element_count(*genreArray, &numberOfGenreLevels));
            for (gn_uint32_t genreArrayLevel = 0;
                    genreArrayLevel < numberOfGenreLevels;
                    ++genreArrayLevel)
            {
                // In each genre level look if there's a matching genre string
                if (gnSuccess("gn_genre_array_get_genre",
                              gn_genre_array_get_genre(*genreArray, genreArrayLevel, &genre)))
                {
                    gnSuccess("gn_genre_get_representation_array",
                              gn_genre_get_representation_array(genre,
                                                                repArray));
                    // Match only genre title which are similar to the original one
                    // 0.3 is a value found by heuristic tests for english titles
                    if ( -1 != getMatchingRepresentationIndex(*repArray,
                            (char *)queryString, 0.3))
                    {
                        gn_status = GN_SUCCESS;
                        break;
                    }
                }
            }
            if (GN_SUCCESS != gn_status)
                gnSuccess("gn_genre_array_smart_free",
                          gn_genre_array_smart_free(genreArray));
        }
    }

    gn_textid_file_data_smart_free(&pfile_data);

    return (GN_SUCCESS == gn_status);
}



/* **************************************************************************
 * Function srQuery 
 * *************************************************************************/ 
/** Query GN database for speech recognition
* 
* This method queries the database for all available phonemes for a specific
* artist/album
* It is ensured, that the official representation is always the first item
* of the returned vector (if a official representation is available at all)
* 
* \see Please read comments for prioritizeOfficial
* 
* \param[in] queryType Look for album, artist or genre
* \param[in] qLanguage Requested spoken language for the phonemes. Only phonemes
* for this language will be delivered
* \param[in] queryString Written name of artist or album
* \param[in] maxHits Maximum number of phonemes returned 
* (use a high limit (e.g. 12) to get all)
* Use at least 2 results to catch both the official and original representation
* \param[out] phonemesResult result of this method
* \param[in] returnDisplayString (optional, default false) Return readable strings
* instead of phonemes (for debugging purposes)
* 
* \return Number of phonemes found. 0=no hit at all
*/
int tclGracenoteWrapper::srQuery(tclGracenoteWrapperInterface::QueryType queryType,
                                 const tclLanguageId &qLanguage,
                                 tString queryString,
                                 size_t maxHits,
                                 tString phonemesResult[],
                                 bool returnDisplayString)
{
#define HEX( x ) setw(2) << setfill('0') << hex << (int)( x )

//    {
//        ofstream tout("/tmp/gal.txt", ios_base::app);
//        tout << "tclGracenoteWrapper::srQuery("
//        << (int) queryType << ", "
//        << (char* ) (unsigned char *) qLanguage << ", "
//        << ">" << queryString << "<, "
//        << maxHits << ","
//        << phonemesResult << ","
//        << returnDisplayString << ")" << endl;
//        for (char *cp = queryString; *cp != '\0'; ++cp)
//        {
//            tout << "QS: >" << (char) *cp << "< " << HEX ((char) *cp & 0xff) << " ";
//            for (int i = 256; i != 1; i /= 2)
//                tout << (((int) * cp & (i/2)) ? "1" : "0");
//            tout << endl;
//        }
//        tout << "UTF8-test: レディヘ" << endl;
//    }

    list<pair<int, string> > ratedPhonemes;
    gn_textid_presult_data_t result = NULL;
    gn_prep_array_t repArray = NULL;
    gn_uint32_t repArraySize;
    gn_prepresentation_t representation = NULL;
    gn_ptranscript_arr_t tsArray = NULL;
    gn_uint32_t tsArraySize;
    gn_pgenre_arr_t genreArray = GN_NULL;

    if (!gnEmmsIsInitialized)
        return 0;

    // Query special genre phonemes first
    if (tclGracenoteWrapperInterface::GENRE == queryType)
    {
        pair<string, string> cacheKey = make_pair(string((char*) (unsigned char*)qLanguage), queryString);
        if (genrePhonemeCache.end() != genrePhonemeCache.find(cacheKey))
        {
            size_t count = 0;
            for (list<string>::const_iterator li = genrePhonemeCache[cacheKey].begin();
                    li != genrePhonemeCache[cacheKey].end();
                    ++li)
            {
                string phonemeString;
                if (returnDisplayString)
                    phonemeString = queryString;
                else
                    phonemeString = XSAMPA2LHplus(li->c_str(), qLanguage);

                ostringstream searchString;
                searchString << string((char*) (unsigned char*)qLanguage) << " " << phonemeString;
                if(exceptions_map.end() == exceptions_map.find(searchString.str()))
				{
					strncpy(phonemesResult[count], phonemeString.c_str(), tclGracenoteWrapperInterface::MAX_PHONEME_LENGTH);
					phonemesResult[count][tclGracenoteWrapperInterface::MAX_PHONEME_LENGTH - 1] = '\0';
					if (++count >= maxHits)
						break;
				}
            }
            return count;
        }
    }

    if (tclGracenoteWrapperInterface::GENRE == queryType)
        checkAndSetDisplayLangauge(qLanguage);

    // Clear result set
    for (size_t i = 0; i < maxHits; ++i)
        phonemesResult[i][0] = '\0';

    if (!gnQueryObject(queryType,
                       (gn_uchar_t*) queryString,
                       &result,
                       &genreArray,
                       &repArray))
    {
        gn_textid_free_result(&result);
        gnSuccess("gn_genre_array_smart_free",
                  gn_genre_array_smart_free(&genreArray));
        //        {
        //            ofstream tout("/tmp/gal.txt", ios_base::app);
        //            tout << "tclGracenoteWrapper gnQueryObject failed " << endl;
        //        }

        return 0;
    }

    if (!gnSuccess("get element count of representation_array",
                   gn_representation_get_element_count(repArray,
                                                       &repArraySize)))
        repArraySize = 0;

    for (gn_uint32_t repArrayIndex = 0; repArrayIndex < repArraySize; repArrayIndex++)
    {
        if (!gnSuccess("get representation",
                       gn_representation_get_rep(repArray,
                                                 repArrayIndex,
                                                 &representation)))
            continue;

        // Get transcriptions for this representation
        if (!gnSuccess("get transcript array",
                       gn_representation_get_transcript_array(representation,
                                                              qLanguage,
                                                              &tsArray))
                || tsArray == NULL
                || !gnSuccess("get element count of transcript_array",
                              gn_transcript_array_get_array_size(tsArray,
                                                                 &tsArraySize)))
            tsArraySize = 0;

        for (gn_uint32_t tsArrayIndex = 0;
                tsArrayIndex < tsArraySize;
                tsArrayIndex++)
        {
            gn_ptranscript_t ts = GN_NULL;
            gn_uchar_t* tsString = NULL;
            gn_bool_t isOrigin = GN_FALSE;
            gn_bool_t isOfficial = GN_FALSE;
            gn_rep_type_t reptype;
            gn_bool_t isCorrect = GN_FALSE;
            bool isSimplified = false;
            bool isTraditional = false;
            int rating = 0;

            if (!gnSuccess("get transcription array element",
                           gn_transcript_array_get_element(tsArray,
                                                           tsArrayIndex,
                                                           &ts))
                    || !gnSuccess("get transcription string",
                                  gn_transcript_get_transcript_string(ts,
                                                                      &tsString)))
                continue;

            gnSuccess("gn_transcript_is_correct_pronunciation",
                      gn_transcript_is_correct_pronunciation(ts,
                                                             &isCorrect));
            gn_langid_t origLang;
            gnSuccess("gn_representation_get_language_id",
                      gn_representation_get_language_id(representation, &origLang));
            if (origLang != GN_NULL)
            {
                isSimplified = (string((char*)origLang) == string("13"));
                isTraditional = (string((char*)origLang) == string("55"));
            }

            gnSuccess("gn_representation_get_rep_type",
                      gn_representation_get_rep_type(representation, &reptype));
            gnSuccess("gn_transcript_is_origin_language",
                      gn_transcript_is_origin_language(ts, &isOrigin));
            gn_representation_is_official(representation, &isOfficial);


            // Phonemes are rated to their relevance. Phonemes are used in
            // decreasing order of relevance.

            // Handling of Chinese phonemes is according to the algorithm
            // provided by Jonathan Ford. 
            // See https://hi-dms.de.bosch.com/docushare/dscgi/ds.py/Get/File-372923
            if (tclLanguageId("CHN_qad") == qLanguage)
            {
                if (reptype == GN_REPRESENTATION_OFFICIAL_VARIANT &&
                        isCorrect &&
                        isSimplified)
                    rating = 128;
                else if (reptype == GN_REPRESENTATION_OFFICIAL_VARIANT &&
                         isCorrect &&
                         isTraditional)
                    rating = 64;
                else if (reptype == GN_REPRESENTATION_OFFICIAL &&
                         isOrigin &&
                         isCorrect)
                    rating = 32;
                else if (reptype == GN_REPRESENTATION_OFFICIAL &&
                         !isOrigin &&
                         isCorrect)
                    rating = 8;
            }
            else
            {
                // Phonemes are rated to their likeness to the queried string
                // 100% matches give 128, completely different words 0
                rating = (int) (prioritizeOfficial ? 100.0 : 128.0 * (1.0 - min(1.0, levenshteinDistance(queryString, displayString(representation)))));
                if (isOfficial)
                    rating = max(prioritizeOfficial ? 128 : 100, rating);

                // Correct pronunciations are rated better
                if (isCorrect)
                    rating += 64;

                // Same for official representations
                if (reptype == GN_REPRESENTATION_OFFICIAL)
                    rating += 8;
                    
                // Conclusion:
                // A Official, correct, 100% matching phoneme shoult win 
            }

            // Add phonemes and their rating to list
            // list will contain elements in order of with decreasing relevance
            bool inserted = false;
            string retVal = (returnDisplayString ? displayString(representation) : string((char*)tsString));
            // cout << rating << "\t: " << retVal << endl;
            for (list<pair<int, string> >::iterator rpi = ratedPhonemes.begin();
                    rpi != ratedPhonemes.end();
                    ++rpi)
                if (rpi->first < rating)
                {
                    ratedPhonemes.insert(rpi, make_pair(rating, retVal));
                    inserted = true;
                    break;
                }
            if (!inserted)
                ratedPhonemes.push_back(make_pair(rating, retVal));
        }
        gn_transcript_array_destroy(&tsArray);
    }
    gn_textid_free_result(&result);
    gn_genre_array_smart_free(&genreArray);

    size_t count = 0;
    for (list<pair<int, string> >::iterator rpi = ratedPhonemes.begin();
            rpi != ratedPhonemes.end() && count < maxHits;
            ++rpi)
    {
        if (returnDisplayString)
            strncpy(phonemesResult[count], rpi->second.c_str(), tclGracenoteWrapperInterface::MAX_PHONEME_LENGTH);
        else
		{
#ifdef VARIANT_LINUX_X86
        	if(rpi->second.size() > tclGracenoteWrapperInterface::MAX_PHONEME_LENGTH)
        	{
        		cout << rpi->second.size() << "char phoneme for: " << queryString << endl;
        	}
#endif
        	string phoneme_lhp = XSAMPA2LHplus(rpi->second, qLanguage);
        	if(phoneme_lhp.size()!=0)
        	{
#ifdef VARIANT_LINUX_X86
        		bool check_result = verifyLHP(phoneme_lhp, qLanguage);
        		if(false==check_result)
        		{
            		cout << qLanguage << " Wrong LH+ conversion queryType " << queryType << " string: " << queryString << " XSAMPA: " << rpi->second << " LH+: "<< phoneme_lhp << endl;
        		}
#endif
				ostringstream searchString;
				searchString << string((char*) (unsigned char*)qLanguage) << " " << phoneme_lhp;
				if(exceptions_map.end() == exceptions_map.find(searchString.str()))
				{
					strncpy(phonemesResult[count], phoneme_lhp.c_str(), tclGracenoteWrapperInterface::MAX_PHONEME_LENGTH);
					phonemesResult[count][tclGracenoteWrapperInterface::MAX_PHONEME_LENGTH - 1] = '\0';
					++count;
				}
        	}
#ifdef VARIANT_LINUX_X86
        	else
        	{
        		cout << qLanguage << " LH+ conversion failed for queryType " << queryType << " string: " << queryString << " XSAMPA: " << rpi->second << endl;
        	}
#endif
		}
    }

    //    {
    //        ofstream tout("/tmp/gal.txt", ios_base::app);
    //        tout << "tclGracenoteWrapper::srQuery for " << queryString << " returns" << endl;
    //        for (int i = 0; i < count; ++i)
    //            tout << "\t" << i << ": >" << phonemesResult[i] << "<" << endl;
    //    }

    return count;
}


/* **************************************************************************
 * Function ttsQuery 
 * *************************************************************************/ 
/**
* Query GN database for text to speech
* 
* This method queries the database for the first official or original phoneme 
* for a specific artist/album
* 
* \see Please read comments for prioritizeOfficial
* 
* \param[in] queryType Look for album or artist
* \param[in] qLanguage Requested spoken language for the phonemes. Only phonemes 
* for this language will be delivered
* \param[in] queryString Written name of artist or album
* \param[out] phoneme result of this method
* \param[in] returnDisplayString (optional, default false) Return readable strings
* instead of phonemes (for debugging purposes)
* \return Number of phonemes found: 0/1
*/
int tclGracenoteWrapper::ttsQuery(tclGracenoteWrapperInterface::QueryType queryType,
                                  const tclLanguageId &qLanguage,
                                  tString queryString,
                                  tString phoneme,
                                  bool returnDisplayString)
{
    tString phonemesResult[1];
    phonemesResult[0] = phoneme;

    return srQuery(queryType,
                   qLanguage,
                   queryString,
                   1,
                   phonemesResult,
                   returnDisplayString);
}


/* **************************************************************************
 * Function getOfficialName 
 * *************************************************************************/ 
/**
* Query GN database for an official name
* 
* This is as costly as a complete ttsQuery, so use this for testing purposes 
* only. In opposite to the ttsQuery, this function returns an official name 
* even if there's no phoneme for this name available
* 
* \param[in] queryType Look for album or artist
* \param[in] qLanguage Requested spoken language for official name.
* \param[in] queryString Written name of artist or album
* \return official name. empty string if nothing is found
*/
const string tclGracenoteWrapper::getOfficialName(tclGracenoteWrapperInterface::QueryType queryType,
        const tclLanguageId &qLanguage,
        const string &queryString)
{
    if (!gnEmmsIsInitialized)
        return string();

    if (tclGracenoteWrapperInterface::GENRE == queryType)
        checkAndSetDisplayLangauge(qLanguage);

    gn_textid_presult_data_t result = NULL;
    gn_prep_array_t representation_array = NULL;
    gn_prepresentation_t representation = NULL;
    gn_pgenre_arr_t genre_array = GN_NULL;
    string retval;

    if (!gnQueryObject(queryType,
                       (gn_uchar_t*) queryString.c_str(),
                       &result,
                       &genre_array,
                       &representation_array))
    {
        gn_textid_free_result(&result);
        gn_genre_array_smart_free(&genre_array);
        return retval;
    }

    // Get primary representation
    if (tclGracenoteWrapperInterface::GENRE == queryType)
        representation = getMatchingRepresentation(representation_array,
                         queryString);
    else
        representation = getOfficialRepresentation(qLanguage,
                         representation_array);

    if (GN_NULL != representation)
    {
        retval = displayString(representation);
    }
    gn_textid_free_result(&result);
    gn_genre_array_smart_free(&genre_array);
    return retval;
}


/* **************************************************************************
 * Function checkAndSetDisplayLangauge
 * *************************************************************************/ 
/**
* When looking for genres, the result depends on the configured display language
* This functions checks whether the requested language is configured.
* If not, it is configured and the service is restarted 
* 
* \param[in] spokenLanguage Requested spoken language for official name.
*/
void tclGracenoteWrapper::checkAndSetDisplayLangauge(tclLanguageId spokenLanguage)
{
    gn_uchar_t *currentDLang;
    string requestedLang;

    gn_list_get_display_language_id_code(&currentDLang);

    if (!gnWrittenLanguage(string((char *) (gn_uchar_t *) spokenLanguage), requestedLang))
        return ;


    if (string((char *) currentDLang) != requestedLang)
    {
        strncpy(displayLanguage, requestedLang.c_str(),displayLanguageLength);
        displayLanguage[displayLanguageLength] = '\0';
        gnConfig.display_language_id = (gn_uchar_t*) displayLanguage;
        gnConfig.set_display_language_id = GN_TRUE;
        start(true);
    }
}


const list<int> tclGracenoteWrapper::getInstalledLfids(void)
{
    struct dirent *dit;
    list<int> retVal;
    DIR *dip = opendir(gnBasePath);
    if (dip != NULL)
    {
        while ((dit = readdir(dip)))
            if (dit->d_type == DT_DIR)
            {
                int id = atoi(dit->d_name);
                if (id != 0)
                    retVal.push_back(id);
            }
    }
    closedir(dip);
    return retVal;
}
