/// \file GracenoteWrapperInterface.cpp
///
/// Gracenote integration library
///
/// \mainpage Gracenote integration
/// This is a wrapper to integrate the Gracenote Playlist Plus/TextID and
/// MediaVOCS Services into the Transcription Service
///
/// <BR> Copyright: (c) 2011 Robert Bosch GmbH
///
/// \author Ingo Reise CM-AI/PJ-GM28 (external.ingo.reise@de.bosch.com)
///
/// \version 06.09.2011 ttsQuery is now just a call to srQuery<br> 
/// \version 06.09.2011 Replaced GN-conversion routine to LH+: overall 30% faster
/// \version 06.09.2011 Implemented a TTS-selection strategy for CHN_qad 
/// \version 10.07.2011 Supports now /var/opt/bosch/dynamic/speech/config/SDP_CONF.INI
///                     to select the GN database according to the configured
///                     marketing region
/// \version 29.06.2011 Improved conformity to coding guidelines and worked
///                     on lint findings
/// \version 17.06.2011 Edited chinese phoneme A%r to a%r
/// \version 14.06.2011 Integrated special lookup table for genre phonemes
/// \version 10.06.2011 PartNumberProliferation: Now all regian databases
///                     are available on the system. Lib defaults to NAR
/// \version 30.05.2011 Added tclGracenoteWrapperInterface::getUpdateTime,
///                     tclGracenoteWrapperInterface::getSnapshotTime
/// \version 23.05.2011 Interface is completely STL-clean now
/// \version 19.05.2011 Better support for non-english genre phonemes
/// \version 11.05.2011 Original names are prioritized over official ones,
///                     see prioritizeOfficial
/// \version 14.04.2011 Created Interface class using the PIMPL idiom
/// \version 14.04.2011 Transition to system_types for interface
/// \version 04.04.2011 Interface agreed with Madhuri Raya
/// \version 28.03.2011 support for sds_fi_tcl_e16_ISOLanguageCode and
///                     sds_fi_tcl_e16_ISOCountryCode added
/// \version 23.03.2011 L&H+ postprocessing integrated
/// \version 18.03.2011 Limited GENRE support added
/// \version 17.03.2011 Changes language encoding (remove fixed lists)
///
/// \section Languages Supported spoken languages
/// <UL>
/// <LI> North America: "USA_eng", "CAN_fre", "MEX_spa"
/// <LI> Latin America: "USA_eng", "MEX_spa"
/// <LI> China: "USA_eng", "CHN_qad" (Chinse mandarin)
/// <LI> Europe: "GBR_eng" "FRA_fre" "DEU_ger" "RUS_rus" "ESP_spa"
/// </UL>
/// Glossar:
/// <UL>
/// <LI> USA_eng English North America
/// <LI> GBR_eng English United Kingdom
/// <LI> AUS_eng English Australia
/// <LI> FRA_fre French France
/// <LI> CAN_fre French Canada
/// <LI> ESP_spa Spanish Spain
/// <LI> MEX_spa Spanish Mexico
/// <LI> PRT_por Portuguese Portugal
/// <LI> BRA_por Portuguese Brazil
/// <LI> ITA_ita Italian Italy
/// <LI> DEU_ger German Germany
/// <LI> NLD_dut Dutch Netherlands
/// <LI> RUS_rus Russian Russia
/// <LI> TUR_tur Turkish Turkey
/// <LI> CHN_qad Mandarin Chinese China
/// <LI> JPN_jpn Japanese Japan
/// </UL>
///
/// \section lfsid ListFileSetID assignment
/// The ListFileSetID is a Gracenote specific identifier specifying a database 
/// for a certain region and a certain SDK (5.5/5.6)
/// See tclGracenoteWrapper::tclGracenoteWrapper() (Link available in internal 
/// documentation only) for a list of lfsids
/// 
/// \section kb Known issues
/// \note Genre support is available but due to architectural differences with
/// sparse results only,
///  As the Gracenote-API, this wrapper is \b NOT thread safe
///
/// \bug GMNGA-9105: Problems in GN-database: In some cases there is no phoneme
/// available from GN for the official/original representation. This is detected
/// and alternate phonemes suppressed in this case.
/// \see tclGracenoteWrapperInterface::srQuery
///
/// \bug GMNGA-10031: Problems in GN-database: Phoneme "fix" or "one" for some
/// artists/ambums with differing names
///
/// \bug GMNGA-9103: Problem with official names (does only apply for
/// prioritizeOfficial) There is a different "official name" for an already
/// official name
///
/// \section Example Example
/// \code
///   #include "GracenoteWrapperInterface.h"
///   // more code
///   // Define used language
///   tclLanguageId german(sds_fi_tcl_e16_ISOLanguageCode::FI_EN_ISO_639_1_DE,
///                        sds_fi_tcl_e16_ISOCountryCode::FI_EN_ISO_ALPHA_3_DEU);
///
///   // single phoeneme
///   tChar phoneme[tclGracenoteWrapperInterface::MAX_PHONEME_LENGTH];
///
///   // array of phonemes
///   tInt max = 5;
///   tString phonemes[max];
///   for (int i = 0; i < max; ++i)
///      phonemes[i] = new tChar[tclGracenoteWrapperInterface::MAX_PHONEME_LENGTH];
///
///   // Create and start GracenoteWrapperInterface
///   tclGracenoteWrapperInterface GWI;
///   GWI.start();
///
///   // TTS query
///   if (GWI.ttsQuery(tclGracenoteWrapperInterface::ARTIST,
///                    german,
///                    (tString) "Michael Jackson",
///                    phoneme))
///      cout << "Phoneme: >" << phoneme << "<" << endl;
///
///   // SR Query
///   tInt n = GWI.srQuery(tclGracenoteWrapperInterface::ARTIST,
///                        german,
///                        (tString) "Michael Jackson",
///                        max,
///                        phonemes);
///   for (tInt i = 0 ; i < n; ++i)
///      cout << i << ": " << phonemes[i] << endl;
///
///   // free array of phonemes
///   for (int i = 0; i < max; ++i)
///      delete phonemes[i];
/// \endcode
///
/// \section Architecture Example Call
/// \msc
/// tclGracenoteWrapperInterface,tclGracenoteWrapperLib, gn_api;
/// tclGracenoteWrapperInterface=>tclGracenoteWrapperLib [label="start()"];
/// tclGracenoteWrapperLib=>gn_api [label="gninit_initialize_emms()"];
/// gn_api rbox gn_api [label="open files, ..."];
/// --- [label="ready to run"];
/// tclGracenoteWrapperInterface->tclGracenoteWrapperLib [label="ttyQuery()"];
/// tclGracenoteWrapperLib=>gn_api [label="gn_textid_file_data_init()"];
/// tclGracenoteWrapperLib=>gn_api [label="gn_textid_file_data_set_disc_title ()"];
/// tclGracenoteWrapperLib=>gn_api [label="gn_textid_local_lookup ()"];
/// tclGracenoteWrapperLib=>gn_api [label="gn_textid_get_result()"];
/// tclGracenoteWrapperLib=>gn_api [label="gn_textid_get_album_title_representation_array()"];
/// tclGracenoteWrapperLib=>gn_api [label="gn_textid_file_data_smart_free()"];
/// tclGracenoteWrapperLib=>gn_api [label="get transcript array()"];
/// tclGracenoteWrapperLib=>gn_api [label="get element count of transcript array()"];
/// tclGracenoteWrapperLib=>gn_api [label="get transcription array element()"];
/// tclGracenoteWrapperLib=>gn_api [label="get transcription is_correct()"];
/// tclGracenoteWrapperLib=>gn_api [label="get transcription string()"];
/// tclGracenoteWrapperInterface<<tclGracenoteWrapperLib [label="L&H+ phoeneme"];
/// \endmsc

// Note: gn_*_free_ functions are setting the pointer to NULL
#include "GracenoteWrapperLib.h"
#include "GracenoteWrapperInterface.h"

/* **************************************************************************
 * Function start 
 * *************************************************************************/ 
/**
* Start/initialize GN database access
* 
* This will open all needed files
* Start/Stop session is needed for performance reasons. Opening a session for 
* each call would take much longer for each query
* Queries will not work in case of failure
* \return true for success, false for failure 
*/
tBool tclGracenoteWrapperInterface::start(void)
{
    return (tBool) pImpl->start();
}

/* **************************************************************************
 * Function stop 
 * *************************************************************************/ 
/**
 * Stop GN database access
 * 
 * Close all open file handles and free memory
 * Further queries will return empty reults 
 */
void tclGracenoteWrapperInterface::stop(void)
{
    pImpl->stop();
}

/* **************************************************************************
 * Function Default Constructor 
 * *************************************************************************/ 
/**
* Only variable initializations are done here, no file access
* Call start() prior to querying the database
*/
tclGracenoteWrapperInterface::tclGracenoteWrapperInterface() :
        pImpl(new tclGracenoteWrapper)
{}

/* **************************************************************************
 * Function Destructor 
 * *************************************************************************/ 
/**
* Includes shutdown if necessary. Normally not called directly
*/
tclGracenoteWrapperInterface::~tclGracenoteWrapperInterface()
{
    delete pImpl;
    pImpl = NULL;
}

/* **************************************************************************
 * Function Copy Constructor 
 * *************************************************************************/ 
/**
* Creates a new instance
* \warning Do not use this. The GN-library is single threaded and does not 
* support multiple instances
*/
tclGracenoteWrapperInterface::tclGracenoteWrapperInterface(const tclGracenoteWrapperInterface & other) :
        pImpl(new tclGracenoteWrapper(*(other.pImpl)))
{
    std::cerr << "WARNING: tclGracenoteWrapperInterface Copy constructor called."
    << endl << "Please take care to use Gracenote only in one thread" << endl;
}

/* **************************************************************************
 * Function Assignment operator 
 * *************************************************************************/ 
/**
* Assigns an instance to another
* \warning Do not use this. The GN-library is single threaded and does not 
* support multiple instances
*/
tclGracenoteWrapperInterface& tclGracenoteWrapperInterface::operator=(tclGracenoteWrapperInterface other)
{
    std::cerr << "WARNING: tclGracenoteWrapperInterface assigned."
    << endl << "Please take care to use Gracenote only in one thread" << endl;
    std::swap(this->pImpl, other.pImpl);
    return *this;
}

/* **************************************************************************
 * Function ttsQuery 
 * *************************************************************************/ 
/**
* Query GN database for text to speech
* 
* This method queries the database for the first official phoneme for a specific 
* artist/album
* 
* \param[in] type Look for album or artist
* \param[in] qlang Requested spoken language for the phonemes. Only phonemes 
* for this language will be delivered
* \param[in] qstring Written name of artist or album
* \param[out] phoneme result of this method
* \return Number of phonemes found: 0/1
* 
* \note Memory management is under responsibility of the user <br>
* phoneme must point to a tString of minimum 
* tclGracenoteWrapperInterface::MAX_PHONEME_LENGTH chars
*/
tInt tclGracenoteWrapperInterface::ttsQuery(QueryType type,
        tclLanguageId qlang,
        tString qstring,
        tString phoneme)
{
#ifdef VARIANT_S_FTR_ENABLE_GRACENOTE_DB

    return (tInt) pImpl->ttsQuery(type,
                                  qlang,
                                  qstring,
                                  phoneme);
#else
   return (tInt) 0;
#endif
}

/* **************************************************************************
 * Function srQuery 
 * *************************************************************************/ 
/** Query GN database for speech recognition
* 
* This method queries the database for all available phonemes for a specific
* artist/album
* It is ensured, that the official representation is always the first item
* of the returned vector (if a official representation is available at all)
* 
* \param[in] type Look for album, artist or genre
* \param[in] qLang Requested spoken language for the phonemes. Only phonemes
* for this language will be delivered
* \param[in] qString Written name of artist or album
* \param[in] maxHits Maximum number of phonemes returned 
* (use a high limit (e.g. 10) to get all)
* \param[out] phonemes result of this method
* \return Number of phonemes found. 0=no hit at all
* 
* \note Memory management is under responsibility of the user <br>
* phonemes must point to an array of at least maxHits tString elements with 
* tclGracenoteWrapperInterface::MAX_PHONEME_LENGTH chars each
* 
* \remark It is ensured, that srQuery will only return results if ttsQuery would
* do this also. So if there's no correct official/original phoneme available 
* then srQuery will return 0, even if alternative or incorrect phonemes are
* available. Thus it's ensured that TTS is always available for recognized
* titles.  
*/
tInt tclGracenoteWrapperInterface::srQuery(QueryType type,
        tclLanguageId qLang,
        tString qString,
        tUInt maxHits,
        tString phonemes[])
{
#ifdef VARIANT_S_FTR_ENABLE_GRACENOTE_DB

    return (tInt) pImpl->srQuery(type,
                                 qLang,
                                 qString,
                                 maxHits,
                                 phonemes);
#else
    return (tInt) 0;
#endif
}

/* **************************************************************************
 * Function  getUpdateTime 
 * *************************************************************************/ 
/**
 * Gets the timestamp of the last database update
 * 
 * This is the time when the database was installed on the target. 
 * Use this to check whether the database has been updated
 * 
 * This function may be called without calling 
 * start() before
 * 
 * \return seconds since 1970; 0 in case of an error
 */
tLong tclGracenoteWrapperInterface::getUpdateTime(void)
{
    return (tLong) pImpl->getUpdateTime();
}


/* **************************************************************************
 * Function  getSnapshotTime 
 * *************************************************************************/ 
/**
 * Gets the timestamp of the database snapshot
 * 
 * This is the time when the database content was filtered from the master 
 * database at Gracenote
 * Use this to display the version of the installed database.
 * 
 * \return seconds since 1970; 0 in case of an error
 */
tLong tclGracenoteWrapperInterface::getSnapshotTime(void)
{
    return (tLong) pImpl->getSnapshotTime();
}

