/// \file GracenoteDBTests.cpp
///
/// Gracenote integration library, testing and debugging aids
///
/// This file contains additional methods, mainly used for (internal) testing
/// and debugging. These functions should not be used during normal system
/// operation
/// \see See GracenoteWrapperLib.cpp for basic functions
///
/// Copryright: (c) 2011 Robert Bosch GmbH
///
/// \author Ingo Reise CM-AI/PJ-GM28 (external.ingo.reise@de.bosch.com)
///

#include "GracenoteWrapperLib.h"
#include <sys/time.h>
#include <sstream>

// Convenience arrays to iterate over all three query types.

/// Number of different query types
const int queryTypeCount = 3;

/// Array containing all query types
tclGracenoteWrapperInterface::QueryType qtList[] = {
            tclGracenoteWrapperInterface::ALBUM,
            tclGracenoteWrapperInterface::ARTIST,
            tclGracenoteWrapperInterface::GENRE
        };


/// readable names for the query types
string qtName[] = {
                      "album",
                      "artist",
                      "genre"
                  };

/* **************************************************************************
 * Function test
 * *************************************************************************/ 
/**
 * Tests if the installed database is working and look for known issues
 * 
 * This function is intended to be used for series care and when updates of 
 * the database appear. It's available via gal_test
 * 
 * Diagnostic output is printed to stdout, so run this in terminal
 * 
 */
void tclGracenoteWrapper::test(void)
{
    // Don't try this, it would be reverse engineering ...
    // strings i000010.gn | grep -B 1  "^..._...\$" | grep -v "^--" | tr -d "\"" | tr -d "\t" | gawk ' /_/ { print $0" "l} { l = $0 } ' | sort
    // it would give you 141 language specifieres for NAR (ZZZs deleted -> 120)
    // This list could be used to check which of the languages use the ISO
    // standards, e.g.:
    //
    //   string gnLanguages[] = { "AFG_pus", "ALB_alb", "ARG_spa", "ARM_arm", "AUS_eng", "AZE_aze", "BEL_dut", "BGR_bul", "BIH_bos", "BLR_bel", "BRA_por", "CAN_fre", "CHE_gsw", "CHN_chi", "CHN_qab", "CHN_qac", "CHN_qad", "CHN_qaf", "CHN_qag", "CHN_qah", "CZE_cze", "DEU_ger", "DNK_dan", "DZA_ara", "DZA_tmh", "EGY_ara", "ESP_baq", "ESP_cat", "ESP_glg", "ESP_spa", "EST_est", "ETH_amh", "FIN_fin", "FRA_fre", "GBR_eng", "GBR_wel", "GEO_geo", "GRC_gre", "GTM_spa", "HRV_scr", "HTI_fre", "HUN_hun", "IDN_ind", "IDN_jav", "IDN_sun", "IND_asm", "IND_ben", "IND_bho", "IND_eng", "IND_guj", "IND_hin", "IND_kan", "IND_kas", "IND_kok", "IND_mai", "IND_mal", "IND_mar", "IND_mni", "IND_ori", "IND_pan", "IND_snd", "IND_tam", "IND_tel", "IRL_gla", "IRN_per", "IRQ_ara", "ISL_ice", "ISR_heb", "ITA_ita", "JPN_jpn", "KHM_khm", "KOR_kor", "LAO_lao", "LKA_sin", "LTU_lit", "LVA_lav", "MAR_ara", "MDG_mlg", "MEX_spa", "MMR_bur", "MYS_may", "NER_hau", "NGA_yor", "NLD_dut", "NOR_nor", "NPL_nep", "PAK_urd", "PHL_tgl", "POL_pol", "PRT_por", "PRY_grn", "ROU_rum", "RUS_rus", "SAU_ara", "SDN_ara", "SEN_wol", "SOM_som", "SRB_scc", "SVK_slo", "SVN_slv", "SWE_swe", "SYR_ara", "THA_tha", "TUN_ara", "TUR_tur", "TWN_qae", "TZA_swa", "UKR_ukr", "USA_chr", "USA_eng", "USA_haw", "USA_nai", "USA_nav", "UZB_uzb", "VNM_vie", "YEM_ara", "ZAF_afr", "ZAF_xho", "ZAF_zul", "ZWE_sna" };
    //   for (int li = 0; li < 120; ++li )
    //      if (gnLanguages[li] != tclLanguageId(gnLanguages[li]).getGNCode())
    //         cout << "Language not according to standard: " << gnLanguages[li] << endl;

    gn_sys_install_profile_t* installedProfile = NULL;

    // Provide info about the tested item
    cout << "Checking Gracenote Database at " << gnDbPath << endl;
    cout << "Gracenote API: " << gn_get_build_summary() << endl;
    cout << "This wrapper-library prioritizes "
    << (prioritizeOfficial ? "official" : "original") << " names" << endl;

    cout << "List File Set Id: " << lfsid << endl;

    cout << "List File Set Revision ";
    if (gnSuccess("gn_sys_get_install_profile",
                  gn_sys_get_install_profile(&installedProfile)))
    {
        cout << (int) installedProfile->lfs_revision << endl;
        gn_sys_destroy_install_profile (&installedProfile);
    }
    else
    {
        cout << "NOT AVAILABLE" << endl;
    }

    cout << "Base Revision ";
    if (gnSuccess("gn_sys_get_install_profile",
                  gn_sys_get_install_profile(&installedProfile)))
    {
        cout << (int) installedProfile->base_db_revision << endl;
        gn_sys_destroy_install_profile (&installedProfile);
    }
    else
    {
        cout << "NOT AVAILABLE" << endl;
    }

    // Print snapshot date from file (.../SNAPSHOT_DATE_FILE) and by API
    cout << "Snapshot date FILE: ";
    string snapshotDate;
    ifstream snapshotDateFile((string(gnDbPath) + SNAPSHOT_DATE_FILE).c_str());
    if (snapshotDateFile.good())
    {
        getline(snapshotDateFile, snapshotDate);
        cout << snapshotDate << endl;
    }
    else
    {
        cout << "NOT AVAILABLE" << endl;
    }

    cout << "Snapshot date API : ";
    if (gnSuccess("gn_sys_get_install_profile",
                  gn_sys_get_install_profile(&installedProfile)))
    {
        cout << (char *) installedProfile->data_snapshot_date << endl;
        gn_sys_destroy_install_profile (&installedProfile);
    }
    else
    {
        cout << "NOT AVAILABLE" << endl;
    }

    gn_uchar_t * hierarchy_name;
    if (gnSuccess("gn_list_get_genre_hierarchy_name", gn_list_get_genre_hierarchy_name(&hierarchy_name)))
        cout << "Default Genre Hierarchy: " << (char *)hierarchy_name << endl;

    cout << "Supported languages (wrapper): ";
    for (set<tclLanguageId>::const_iterator lii = supportedLanguages[lfsid].begin();
            lii != supportedLanguages[lfsid].end();
            ++ lii)
        cout << *lii << " ";
    cout << endl;


    //    // Look for available languages
    //
    //    char * allLangs[] = { "AFG_pus", "ALB_alb", "ARG_spa", "ARM_arm", "AUS_eng", "AZE_aze", "BEL_dut", "BGR_bul", "BIH_bos", "BLR_bel", "BRA_por", "CAN_fre", "CHE_gsw", "CHN_chi", "CHN_qab", "CHN_qac", "CHN_qad", "CHN_qaf", "CHN_qag", "CHN_qah", "CZE_cze", "DEU_ger", "DNK_dan", "DZA_ara", "DZA_tmh", "EGY_ara", "ESP_baq", "ESP_cat", "ESP_glg", "ESP_spa", "EST_est", "ETH_amh", "FIN_fin", "FRA_fre", "GBR_eng", "GBR_wel", "GEO_geo", "GRC_gre", "GTM_spa", "HRV_scr", "HTI_fre", "HUN_hun", "IDN_ind", "IDN_jav", "IDN_sun", "IND_asm", "IND_ben", "IND_bho", "IND_eng", "IND_guj", "IND_hin", "IND_kan", "IND_kas", "IND_kok", "IND_mai", "IND_mal", "IND_mar", "IND_mni", "IND_ori", "IND_pan", "IND_snd", "IND_tam", "IND_tel", "IRL_gla", "IRN_per", "IRQ_ara", "ISL_ice", "ISR_heb", "ITA_ita", "JPN_jpn", "KHM_khm", "KOR_kor", "LAO_lao", "LKA_sin", "LTU_lit", "LVA_lav", "MAR_ara", "MDG_mlg", "MEX_spa", "MMR_bur", "MYS_may", "NER_hau", "NGA_yor", "NLD_dut", "NOR_nor", "NPL_nep", "PAK_urd", "PHL_tgl", "POL_pol", "PRT_por", "PRY_grn", "ROU_rum", "RUS_rus", "SAU_ara", "SDN_ara", "SEN_wol", "SOM_som", "SRB_scc", "SVK_slo", "SVN_slv", "SWE_swe", "SYR_ara", "THA_tha", "TUN_ara", "TUR_tur", "TWN_qae", "TZA_swa", "UKR_ukr", "USA_chr", "USA_eng", "USA_haw", "USA_nai", "USA_nav", "UZB_uzb", "VNM_vie", "YEM_ara", "ZAF_afr", "ZAF_xho", "ZAF_zul", "ZWE_sna", NULL };
    //    cout << "Supported langauges (database): ";
    //    supportedLanguages[lfsid].clear();
    //    for (char **testLang = allLangs; *testLang != NULL; ++testLang)
    //    {
    //        string requestedLang;
    //        if (gnWrittenLanguage(string(*testLang), requestedLang))
    //        {
    //            strcpy(displayLanguage, requestedLang.c_str());
    //            gnConfig.display_language_id = (gn_uchar_t*) displayLanguage;
    //            gnConfig.set_display_language_id = GN_TRUE;
    //            if (start(true)) {
    //                cout << *testLang << " ";
    //                supportedLanguages[lfsid].insert(tclLanguageId(*testLang));
    //            }
    //        }
    //    }
    //    cout << endl;
    //    checkAndSetDisplayLangauge(*(supportedLanguages[lfsid].begin()));

    // Look for available entries for different spoken languages
    set<string> elements[queryTypeCount];
    set<string> officialElements;
    string official;
    int withTTS = 0;
    // array of phonemes
    tInt maxPhonemeCount = 10;
    tString phonemes[maxPhonemeCount];
    for (int i = 0; i < maxPhonemeCount; ++i)
        phonemes[i] = new tChar[tclGracenoteWrapperInterface::MAX_PHONEME_LENGTH];

    for (int i = 0; i < queryTypeCount; i++)
    {
        elements[i].clear();
        getContent(qtList[i], elements[i]);
        cout << "Found " << elements[i].size() << " " << qtName[i] << "s" << endl;
    }


    bool firstRun = true;
	ofstream outputfile((string(gnDbPath)+EXCEPTIONLISTFILE).c_str());
	cout << "Outputfile: " << (string(gnDbPath)+EXCEPTIONLISTFILE).c_str() << endl;
	
    for (set<tclLanguageId>::const_iterator languageIterator = supportedLanguages[lfsid].begin();
            languageIterator != supportedLanguages[lfsid].end();
            ++languageIterator)
    {
        tclLanguageId lang(*languageIterator);
        cout << "Checking spoken language " << lang << endl;

        for (int i = 0; i < queryTypeCount; i++)
        {
            officialElements.clear();
            for (set<string>::const_iterator ei = elements[i].begin(); ei != elements[i].end(); ++ei)
            {
                string officialName = getOfficialName(qtList[i], lang, *ei);
                if (firstRun &&                                                                                             // Report these issues only once
                        *ei != officialName &&
                        officialName != getOfficialName(qtList[i], lang, officialName))
                {
                   if(cb.problemreport){
                      string rep = string("Problem with ") + qtName[i] + string(" ") + *ei
                                 + string(" -official-> ") + officialName
                                 + string(" -official-> ") + getOfficialName(qtList[i], lang, officialName);
                      cb.problemreport(rep.c_str());
                   } else {
                      cout << "\tGMNGA-9103 Problem with " << qtName[i] << " " << *ei
                      << " -official-> " << officialName
                      << " -official-> " << getOfficialName(qtList[i], lang, officialName)
                      << endl;
                   }
                }

                officialElements.insert(officialName);
            }
            int countOfficial = officialElements.size();
            int maxPhonemeSize = 0;
            withTTS = 0;
            unsigned countExceptions = 0;
            for (set<string>::const_iterator ei = officialElements.begin(); ei != officialElements.end(); ++ei)
            {
                int srResult = srQuery(qtList[i], lang, (tString) ei->c_str(), maxPhonemeCount, phonemes, false);
                if (srResult > 0)
                    withTTS++;
                for (int iPhoneme = 0; iPhoneme < srResult; ++iPhoneme)
                {
                    if(false == verifyLHP(phonemes[iPhoneme], lang))
					{
                       ++countExceptions;
						if(outputfile.is_open())
						{
							cerr << "exception written in exception file " << endl;
							outputfile << lang << " " << phonemes[iPhoneme] << endl;
						}
						else
						{
							cerr << "Exception found but exception file closed???" << endl;
						}
					}
					maxPhonemeSize = max(maxPhonemeSize, (int) strlen(phonemes[iPhoneme]));
                }
            }
            cout << "Count of exceptions: " << countExceptions << endl;
            cout << "Found ";
            cout.width(6);
            cout << withTTS << " " << lang << " phonemes for ";
            cout.width(6);
            cout << countOfficial << " unique (alternate) official "
            << qtName[i] << " names" << endl;
            cout << "Maximum length of a phoneme: " << maxPhonemeSize << endl;
        }
        firstRun = false;
    }
	outputfile.close();

    for (int i = 0; i < maxPhonemeCount; ++i)
        delete[] phonemes[i];
}

/// Local convenience function providing microseconds to calculate time differences
static long msec(void)
{
    struct timeval tv;
    gettimeofday(&tv, NULL);
    return tv.tv_sec*1000000 + tv.tv_usec;
}

/* **************************************************************************
 * Function timing
 * *************************************************************************/ 
/**
 * Runs various benchmarks on the installed database
 * 
 * This function is intended to be used for series care and when updates of 
 * the database appear. It's available via gal_test
 * 
 * Diagnostic output is printed to stdout, so run this in terminal
 * 
 */
void tclGracenoteWrapper::timing(void)
{
    // Just take the first of the supported languages
    tclLanguageId lang = *(supportedLanguages[lfsid].begin());
    tChar phoneme[tclGracenoteWrapperInterface::MAX_PHONEME_LENGTH];
    tInt maxPhonemeCount = 10;
    tString phonemes[maxPhonemeCount];
    for (int i = 0; i < maxPhonemeCount; ++i)
        phonemes[i] = new tChar[tclGracenoteWrapperInterface::MAX_PHONEME_LENGTH];

    // Charset fro random names
    string charset("abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz1234567890   -!?/()[]");

    cout << "Running benchmark on lfsid " << lfsid << " for " << lang << endl;

    set<string> elements[queryTypeCount];
    time_t t0 = time(NULL);

    for (int mode = 0; mode < 2; ++mode)
    {
        if (mode == 0)
        {
            cout << "Using random strings" << endl;
            for (int i = 0; i < queryTypeCount; i++)
            {
                elements[i].clear();
                // Create 25000 random strings (150 for genres)
                for (int j = 0; j < (i == 2 ? 150 : 25000); ++j)
                {
                    int size = (random() % 50) + 2;
                    string xyz("");
                    for (int var = 0; var < size; ++var)
                        xyz.push_back(charset[random() % charset.size()]);

                    elements[i].insert(xyz);
                }
                cout << "Created " << elements[i].size() << " random " << qtName[i] << "s" << endl;
            }
        }
        else
        {
            cout << "Using existing strings" << endl;
            for (int i = 0; i < queryTypeCount; i++)
            {
                elements[i].clear();
                getContent(qtList[i], elements[i]);
                cout << "Found " << elements[i].size() << " " << qtName[i] << "s" << endl;
            }
        }

        // ofstream output("/tmp/timing.dat");
        for (int i = 0; i < queryTypeCount; i++)
        {
            time_t t1 = time(NULL);
            long maxTime = 0;
            for (set<string>::const_iterator ei = elements[i].begin(); ei != elements[i].end(); ++ei)
            {
                long mt1 = msec();
                ttsQuery(qtList[i], lang, (tString)ei->c_str(), phoneme);
                long mdiff = msec() - mt1;
                //  if (mdiff > 20000)
                //    output << mdiff << "ms " << qtName[i] << ": " << *ei << endl;
                if (mdiff > maxTime)
                    maxTime = mdiff;
            }
            cout << "Time per TTS query (" << qtName[i] << "s): "
            << (double) (time(NULL) - t1) / (double) elements[i].size() * 1000.0
            << "ms" << endl;

            t1 = time(NULL);
            for (set<string>::const_iterator ei = elements[i].begin(); ei != elements[i].end(); ++ei)
            {
                srQuery(qtList[i], lang, (tString)ei->c_str(), maxPhonemeCount, phonemes);
            }
            cout << "Time per SR  query (" << qtName[i] << "s): "
            << (double) (time(NULL) - t1) / (double) elements[i].size() * 1000.0
            << "ms" << endl;
        }
    }
    cout << "Total time: " << (int) (time(NULL) - t0) << "s" << endl;
    for (int i = 0; i < maxPhonemeCount; ++i)
        delete[] phonemes[i];
}

/* **************************************************************************
 * Function getCompleteInfo 
 * *************************************************************************/ 
/**
* Query GN database for comprehensive complete info on an album
* 
* Not for extenal use
*/
const string tclGracenoteWrapper::getCompleteInfo(tclGracenoteWrapperInterface::QueryType type,
        const tclLanguageId &qLanguage,
        const string &qstring) const
{
    if (!gnEmmsIsInitialized)
        return "";

    gn_uchar_t *query_string = (gn_uchar_t*) qstring.c_str();
    gn_textid_presult_data_t result = NULL;
    gn_prep_array_t representation_array = NULL;
    gn_pgenre_arr_t genre_array = GN_NULL;
    ostringstream retval;
    gn_uint32_t year;
    const gn_uchar_t *text;

    retval << "Query: " << qstring << " ";

    if (!gnQueryObject(type,
                       query_string,
                       &result,
                       &genre_array,
                       &representation_array))
    {
        gn_textid_free_result(&result);
        retval << "no result";
        return retval.str();
    }

    if (GN_SUCCESS == gn_textid_get_album_title_representation_array(result,
            &representation_array))
        retval << "Album: >"
        << displayString(getOfficialRepresentation(qLanguage,
                         representation_array))
        << "< ";

    if (GN_SUCCESS == gn_textid_get_album_release_year(result,
            &year))
        retval << "Year: " << (int) year << " ";

    //   if (GN_SUCCESS == gn_textid_get_album_tagid(result, &text))
    //      retval << "Album-ID: " << (char *) text << " ";

    if (GN_SUCCESS == gn_textid_get_primary_artist_representation_array(result,
            &representation_array))
        retval << "Artist: >"
        << displayString(getOfficialRepresentation(qLanguage,
                         representation_array))
        << "< ";


    if (GN_SUCCESS == gn_textid_get_contributor_era(result, &text))
        retval << "Artist-Era: " << (char *) text << " ";

    if (GN_SUCCESS == gn_textid_get_contributor_type(result, &text))
        retval << "Artist-Type: " << (char *) text << " ";

    if (GN_SUCCESS == gn_textid_get_contributor_origin(result, &text))
        retval << "Artist-Origin: " << (char *) text << " ";

    if (GN_SUCCESS == gn_textid_get_album_genre(result, &text))
        retval << "Album-Genre: " << (char *) text << " ";

    if (GN_SUCCESS == gn_textid_get_contributor_genre(result, &text))
        retval << "Artist-Genre: " << (char *) text << " ";

    if (GN_SUCCESS == gn_textid_get_mapped_genre_string(result, &text))
        retval << "Genre: " << (char *) text << " ";

    gn_textid_free_result(&result);
    return retval.str();
}


/* **************************************************************************
 * Function  getContent
 * *************************************************************************/ 
/**
 * This is intentionally undocumented
 */
int tclGracenoteWrapper::getContent(tclGracenoteWrapperInterface::QueryType type,
                                    set<string> &elements) const
{
    FILE *in;
    long offset;
    char buffer[1000];
    char marker[] = "\0\0\0@\0\0\0";
    string filename;

    if (tclGracenoteWrapperInterface::ARTIST == type)
    {
        filename = string(gnDbPath) + "contrib.tbl";
    }
    else if (tclGracenoteWrapperInterface::ALBUM == type)
    {
        filename = string(gnDbPath) + "album.tbl";
    }
    else
    {
        filename = string(gnDbPath) + "genres.tbl";
    }

    if ((in = fopen(filename.c_str(), "r")))
    {
        int size;
        for (offset = 2161; ; offset += 32)
        {
            if (0 != fseek(in, offset, SEEK_SET))
                break;
            if (0 == fread(buffer, 999, 1, in))
                break;
            if (0 == memcmp(buffer, marker, 7))
            {
                size = (int) (* ((char *) (buffer + 7)));
                if (size > 0 )
                {
                    buffer[47 + size] = '\0';
                    elements.insert(string(buffer + 47));
                }
            }
        }
        fclose(in);
    }

    return elements.size();
}

static int trackCount = 0;

void tclGracenoteWrapper::makeMP3 ( tclGracenoteWrapperInterface::QueryType type,
                                    const tclLanguageId &qLanguage,
                                    const string &queryString,
                                    const string &basefile ) const
{

    if (!gnEmmsIsInitialized)
        return ;

    gn_uchar_t *query_string = (gn_uchar_t*) queryString.c_str();
    gn_textid_presult_data_t result = NULL;
    gn_prep_array_t representation_array = NULL;
    gn_pgenre_arr_t genre_array = GN_NULL;
    ostringstream retval;
    const gn_uchar_t *text;
    string artist("unknown-artist"), album("unknown-album"), genre("unknown-genre");

    if (!gnQueryObject(type,
                       query_string,
                       &result,
                       &genre_array,
                       &representation_array))
    {
        gn_textid_free_result(&result);
        return ;
    }

    if (GN_SUCCESS == gn_textid_get_album_title_representation_array(result,
            &representation_array))
        album = displayString(getOfficialRepresentation(qLanguage,
                              representation_array));


    if (GN_SUCCESS == gn_textid_get_primary_artist_representation_array(result,
            &representation_array))
        artist = displayString(getOfficialRepresentation(qLanguage,
                               representation_array));


    if (GN_SUCCESS == gn_textid_get_mapped_genre_string(result, &text))
        genre = string((char *) text);

    gn_textid_free_result(&result);

    string fname = artist + album + genre;
    for (size_t i = 0; i < fname.size(); ++i)
        if (!isalnum(fname[i]))
            fname[i] = '_';
    fname += ".mp3";


    ostringstream command;

    cout << "cp " << basefile << " " << fname << endl;

    command << "/usr/bin/eyeD3 --v2 "
    << "--artist=\"" << artist << "\" "
    << "--album=\"" << album << "\" "
    << "--genre=\"" << genre << "\" "
    << "--title=\"Track " << ++trackCount << "\" "
    << fname;

    cout << command.str() << endl;

    return ;
}

void tclGracenoteWrapper::getAllGenres(const tclLanguageId &qLanguage)
{
    set<string> albums;
    getContent(tclGracenoteWrapperInterface::ALBUM, albums);
    gn_textid_presult_data_t result = NULL;
    gn_pgenre_arr_t genreArray = GN_NULL;
    gn_prep_array_t repArray = NULL;
    gn_uint32_t repArraySize;
    gn_prepresentation_t representation = NULL;
    gn_ptranscript_arr_t tsArray = NULL;
    gn_uint32_t tsArraySize;

    checkAndSetDisplayLangauge(qLanguage);

    for (set<string>::const_iterator ei = albums.begin(); ei != albums.end(); ++ei)
    {
        gn_error_t gn_status;
        gn_textid_match_source_t match_source;
        gn_textid_match_type_t match_type;
        gn_uint32_t match_count;
        gn_pfile_data_t pfile_data = NULL;

        // Internal function, no need to check for initialized system here

        // Note: The representationArray is part of the genreArray. So the genreArray
        // may not be cleared as this would kill the representation array

        if (!gnSuccess("initialize TextID",
                       gn_textid_file_data_init(&pfile_data)))
            return ;  /* Nothing to free here */

        gnSuccess("gn_textid_file_data_set_disc_title",
                  gn_textid_file_data_set_disc_title (pfile_data, (gn_uchar_t *) ei->c_str()));

        gn_textid_local_lookup (pfile_data,
                                GN_TEXTID_LU_FLAG_EXACT_ONLY,
                                &match_source,
                                &match_type,
                                &match_count);

        if (match_count == 0)
        {
            // Nothing found for this query
            gn_textid_file_data_smart_free(&pfile_data);
            continue;
        }

        if (!gnSuccess("get result",
                       gn_textid_get_result(1, &result)))
        {
            gn_textid_file_data_smart_free(&pfile_data);
            continue;
        }

        // For genres the API delivers an genre_array of representation_arrays of
        // transcription_arrays
        gn_pgenre_t genre = GN_NULL;
        gn_uint32_t numberOfGenreLevels = 0;
        gn_status = GNERR_MemInvalid; // Dummy for NOT GN_SUCCESS

        if (gnSuccess("gn_textid_get_genre_array",
                      gn_textid_get_genre_array(result,
                                                &genreArray)))
        {
            numberOfGenreLevels = 0;
            if (genreArray != GN_NULL)
                gnSuccess("gn_genre_array_get_element_count",
                          gn_genre_array_get_element_count(genreArray, &numberOfGenreLevels));
            for (gn_uint32_t genreArrayLevel = 0;
                    genreArrayLevel < numberOfGenreLevels;
                    ++genreArrayLevel)
            {
                // In each genre level look if there's a matching genre string
                if (gnSuccess("gn_genre_array_get_genre",
                              gn_genre_array_get_genre(genreArray, genreArrayLevel, &genre)))
                {
                    gnSuccess("gn_genre_get_representation_array",
                              gn_genre_get_representation_array(genre,
                                                                &repArray));

                    if (!gnSuccess("get element count of representation_array",
                                   gn_representation_get_element_count(repArray,
                                                                       &repArraySize)))
                        repArraySize = 0;


                    // Iterate over all representations
                    for (gn_uint32_t index = 0; index < repArraySize ; ++index)
                    {
                        gnSuccess("get representation",
                                  gn_representation_get_rep(repArray,
                                                            index,
                                                            &representation));

                        // Get transcriptions for this representation
                        if (!gnSuccess("get transcript array",
                                       gn_representation_get_transcript_array(representation,
                                                                              qLanguage,
                                                                              &tsArray))
                                || tsArray == NULL
                                || !gnSuccess("get element count of transcript_array",
                                              gn_transcript_array_get_array_size(tsArray,
                                                                                 &tsArraySize)))
                            tsArraySize = 0;

                        for (gn_uint32_t ts_iterator = 0;
                                ts_iterator < tsArraySize;
                                ts_iterator++)
                        {
                            gn_ptranscript_t ts;
                            gn_uchar_t* tsString;

                            if (!gnSuccess("get transcription array element",
                                           gn_transcript_array_get_element(tsArray,
                                                                           ts_iterator,
                                                                           &ts))

                                    || !gnSuccess("get transcription string",
                                                  gn_transcript_get_transcript_string(ts,
                                                                                      &tsString)))
                                continue;

                            cout << displayString(representation) << endl;
                        }
                    }
                }

            }
            if (gn_status != GN_SUCCESS)
                gnSuccess("gn_genre_array_smart_free",
                          gn_genre_array_smart_free(&genreArray));
        }
    }
}


void tclGracenoteWrapper::countPhonemes(void)
{
    list<int> lfsids = getInstalledLfids();
    gn_sys_install_profile_t* installedProfile = NULL;
    tInt maxPhonemeCount = 10;
    tString phonemes[maxPhonemeCount];
    for (int i = 0; i < maxPhonemeCount; ++i)
        phonemes[i] = new tChar[tclGracenoteWrapperInterface::MAX_PHONEME_LENGTH];

    tChar phoneme[tclGracenoteWrapperInterface::MAX_PHONEME_LENGTH];

    lfsids.remove(692);

    for (list<int>::const_iterator lid = lfsids.begin();
            lid != lfsids.end();
            ++lid)
    {
        start(false, *lid);
        int baseDbRev = 0;
        if (gnSuccess("gn_sys_get_install_profile",
                      gn_sys_get_install_profile(&installedProfile)))
        {
            baseDbRev = (int) installedProfile->base_db_revision;
            gn_sys_destroy_install_profile (&installedProfile);
        }

        set<string> names[2];
        cout << *lid << "/" << baseDbRev << "\ttotal unique";
        for (int i = 0; i < 2; ++i)
        {
            getContent(qtList[i], names[i]);
            set<string> offNames;
            for (set<string>::const_iterator ni = names[i].begin();
                    ni != names[i].end();
                    ++ni)
                offNames.insert(getOfficialName(qtList[i], *supportedLanguages[*lid].begin(), *ni));

            cout << "\t" << offNames.size() / 1000.0 << "\t";
        }
        cout << endl;

        for (set<tclLanguageId>::const_iterator lii = supportedLanguages[*lid].begin();
                lii != supportedLanguages[lfsid].end();
                ++ lii)
        {
            cout << "\t" << *lii;
            for (int i = 0; i < 2; ++i)
            {
                int results = 0;
                set<string> offNames;
                for (set<string>::const_iterator ni = names[i].begin();
                        ni != names[i].end();
                        ++ni)
                    offNames.insert(getOfficialName(qtList[i], tclLanguageId(*lii), *ni));

                for (set<string>::const_iterator ni = offNames.begin();
                        ni != offNames.end();
                        ++ni)
                    //                    results += srQuery(qtList[i],
                    //                                       tclLanguageId(*lii),
                    //                                       (tString) ni->c_str(),
                    //                                       maxPhonemeCount,
                    //                                       phonemes);
                    results += ttsQuery(qtList[i],
                                        tclLanguageId(*lii),
                                        (tString) ni->c_str(),
                                        phoneme);
                cout << "\t" << results / 1000.0 << "\t" << (int) (100.0 * (double) results / (double) offNames.size());
            }
            cout << endl;
        }
    }
    for (int i = 0; i < maxPhonemeCount; ++i)
        delete phonemes[i];
}


