/* ***************************************************************************************
* FILE:          HealthcareMonDtcProcess.h
* SW-COMPONENT:  HMI-BASE
 * DESCRIPTION:  HealthCareMonServer.h is part of HealtCareMonApplication
 * COPYRIGHT:  (c) 2020-21 Robert Bosch Car Multimedia GmbH
 * Author: Sathiya
 * Date: 23.06.2021
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "HealthcareMonDtcProcess.h"
#include "../Trace/HealthMontrace.h"

#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#endif 

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HEALTHCAREMONASF
#define ETG_I_TRACE_CHANNEL               TR_COMP_FC_TELEMATIC
#define ETG_I_TTFIS_CMD_PREFIX            "HEALTHCAREMONAPP"
#define ETG_I_FILE_PREFIX                 HEALTHCAREMONAPP::SERVICE
#include "trcGenProj/Header/HealthcareMonDtcProcess.cpp.trc.h"
#endif


HealthcareMonDtcProcess* HealthcareMonDtcProcess::p_HealthCareDtcProcess = NULL;

void msSleep( unsigned int msec )
{
	usleep(msec*1000);
}
/************************************************************************
*FUNCTION: 		HealthcareMonDtcProcess()
*DESCRIPTION: 	Constructor of the class
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	rsa8cob	23/06/2021
*************************************************************************/
HealthcareMonDtcProcess::HealthcareMonDtcProcess()
{
ETG_TRACE_USR4(("HealthcareMonDtcProcess Constructor called"));
u16textid = 0x0;                      // This variable will be updated with Parser output once implemented. For now we shall have some default code
default_text = "";        //This variable will be updated with Parser output once implemented. For now we shall have default text
dtcTableInformation.clear();
u8XmlSeverity = 0;
dtcXmlContent.clear();
CMA_dtctable.clear();
CMG_dtctable.clear();
CMR_dtctable.clear();
CMP_dtctable.clear();
HDTV_dtctable.clear();
CMX_dtctable.clear();
Intercom_dtctable.clear();
Swivel_dtctable.clear();
TestBed_dtctable.clear();
CMC_ParsedContent.clear();
CMA_ParsedContent.clear();
CMG_ParsedContent.clear();
CMR_ParsedContent.clear();
CMP_ParsedContent.clear();
populateXmlContents();
Initialize();
readKdsDTCSeverity();
m_threadID = 0x0;
pthread_mutexattr_t mattr;
pthread_mutexattr_init(&mattr);
pthread_mutex_init(&m_thMutexLock,&mattr);
pthread_cond_init(&m_thWaitCond,NULL);
powermode = (::bosch::cm::di::NanoMsgLCMInterface::entPowerMode)0x0;
ipaddress = "";
enHMIPopupState = enmPOPUPIDLE;
m_thDtcEventCurrentContainer = NULL;
m_DtcEventQueue = new vector<std::shared_ptr<dtcEvent>>(0);
m_thStopTimer = 30;
if(!m_DtcEventQueue)
{
   setThreadState(enmThStateFault);
   ETG_TRACE_FATAL(("DTCEvent::Queue creation failed, hence thread not created!!"));
   return; 
}
pThreadStart();
}
/************************************************************************
*FUNCTION:       Initialize()
*DESCRIPTION:  Initialization of the class members
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 12/07/2021
*************************************************************************/
void HealthcareMonDtcProcess::Initialize()
{
ETG_TRACE_USR4(("HealthcareMonDtcProcess Inititalize called"));
dtcTableInformation.insert({::bosch::cm::di::NanoMsgLCMInterface::enDevID::enDevID__enDevID_Media_Apm,{CMA_dtctable}});
dtcTableInformation.insert({::bosch::cm::di::NanoMsgLCMInterface::enDevID::enDevID__enDevID_Media_Glass,{CMG_dtctable}});
dtcTableInformation.insert({::bosch::cm::di::NanoMsgLCMInterface::enDevID::enDevID__enDevID_Media_Router,{CMR_dtctable}});
dtcTableInformation.insert({::bosch::cm::di::NanoMsgLCMInterface::enDevID::enDevID__enDevID_Media_CamPort,{CMP_dtctable}});
dtcTableInformation.insert({::bosch::cm::di::NanoMsgLCMInterface::enDevID::enDevID__enDevID_Media_HDTV,{HDTV_dtctable}});
dtcTableInformation.insert({::bosch::cm::di::NanoMsgLCMInterface::enDevID::enDevID__enDevID_Media_ControlPanel,{CMX_dtctable}});
dtcTableInformation.insert({::bosch::cm::di::NanoMsgLCMInterface::enDevID::enDevID__enDevID_Intercom,{Intercom_dtctable}});
dtcTableInformation.insert({::bosch::cm::di::NanoMsgLCMInterface::enDevID::enDevID__enDevID_Swivel_Holder,{Swivel_dtctable}});
dtcTableInformation.insert({::bosch::cm::di::NanoMsgLCMInterface::enDevID::enDevID__enDevID_TestBed,{TestBed_dtctable}});
dtcXmlContent.insert({::bosch::cm::di::NanoMsgLCMInterface::enDevID::enDevID__enDevID_Media_Apm,{CMA_ParsedContent}});
dtcXmlContent.insert({::bosch::cm::di::NanoMsgLCMInterface::enDevID::enDevID__enDevID_Media_Glass,{CMG_ParsedContent}});
dtcXmlContent.insert({::bosch::cm::di::NanoMsgLCMInterface::enDevID::enDevID__enDevID_Media_Router,{CMR_ParsedContent}});
dtcXmlContent.insert({::bosch::cm::di::NanoMsgLCMInterface::enDevID::enDevID__enDevID_Media_CamPort,{CMP_ParsedContent}});
}
/************************************************************************
*FUNCTION:            readkdsDTCSeverity()
*DESCRIPTION:   reads the DTC Severity from Data Pool
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 16/08/2021
*************************************************************************/
void HealthcareMonDtcProcess::readKdsDTCSeverity()
{
#ifdef KDS_CONFIG
 if (DP_S32_NO_ERR == DP_s32GetConfigItem("CIS_Configuration","HealthcareDTCPopupSeverity",(uint8_t*) &u8KdsSeverity,1))
 {
       ETG_TRACE_USR4(("DTC KdsSeverity read from kds %d",u8KdsSeverity));
 }
 else
 {
      ETG_TRACE_FATAL(("Error in Reading DTC Severity config from Kds"));
 }
#else
u8KdsSeverity = DEFAULT_DTC_SEVERITY;
#endif
bool status = checkAndAssignSeverity(u8KdsSeverity);
if(status == false)
{
 u8KdsSeverity = DEFAULT_DTC_SEVERITY;
}

}
/************************************************************************
*FUNCTION:          pThreadStart()
*DESCRIPTION:   Starting the DTC Processing thread
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 18/10/2021
*************************************************************************/
bool HealthcareMonDtcProcess::pThreadStart()
{
	try 
	{
		pthread_mutex_lock(&m_thMutexLock);
		if( getThreadRunStatus() )
		{
			pthread_mutex_unlock(&m_thMutexLock);
			return TRUE;
		}

		pthread_mutex_unlock(&m_thMutexLock);

		pthread_attr_t attr;
		pthread_attr_init(&attr);
	        int error = pthread_create(&m_thHandle,&attr,(THREADFUNCPTR)&HealthcareMonDtcProcess::thStartRoutine,(void*)this);
        	if( error != 0 )
		{
			setThreadState(enmThStateFault);
			ETG_TRACE_FATAL(("clNanoMsgCoreThread::Thread creation failed!!"));
			return FALSE;	
		}
	}
	catch (char *s)
	{
		exit(-1); 
	}
	return true;
}
/************************************************************************
*FUNCTION:          pThreadStop()
*DESCRIPTION:   Stopping the thread during shutdown
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 21/10/2021
*************************************************************************/
bool HealthcareMonDtcProcess::pThreadStop()
{
	try
	{
		pthread_mutex_lock(&m_thMutexLock);
		setThreadRunStatus(FALSE);
		pthread_mutex_unlock(&m_thMutexLock);

		pthread_cond_signal(&m_thWaitCond);

		int ticks = (m_thStopTimer*1000)/100;
		for( int i=0; i<ticks; i++ )
		{
			msSleep(100);// Need to find a solution, currently sleep is used until signal is processed
			pthread_mutex_lock(&m_thMutexLock);
			if( getThreadState() == enmThStateDown )
			{
				pthread_mutex_unlock(&m_thMutexLock);
				return TRUE;
			}
			pthread_mutex_unlock(&m_thMutexLock);
		}
	}
	catch (char *s)
	{
		ETG_TRACE_FATAL(("HealthcareMonDtcProcess::Thread Exit failed!!"));
	}
	return FALSE;
}

/************************************************************************
*FUNCTION:        thStartRoutine ()
*DESCRIPTION:   Thread function to process Dtc Events
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 18/10/2021
*************************************************************************/
void* HealthcareMonDtcProcess::thStartRoutine(void* dtcEventdata)
{
     ETG_TRACE_USR4(("Entered Thread function"));
     pthread_mutex_lock(&m_thMutexLock);
     setThreadState(enmThStateWaiting);
     setThreadRunStatus(TRUE);
     pthread_mutex_unlock(&m_thMutexLock);
     while(getThreadRunStatus())
     {
	pthread_mutex_lock(&m_thMutexLock);
	if (0 != pthread_cond_wait(&m_thWaitCond, &m_thMutexLock))
	{
	   ETG_TRACE_USR4(("Failure in Thread wait condition"));
	    break;
	}
       pthread_mutex_unlock(&m_thMutexLock);
       if(!processDtcEvent() )
       {
	        ETG_TRACE_FATAL(("HealthCareMonDtcProcess::ProcessDtcEvent retured a FALSE"));
       }
     }
     pthread_mutex_lock(&m_thMutexLock);
     setThreadState(enmThStateDown);
     setThreadRunStatus(FALSE);
     pthread_mutex_unlock(&m_thMutexLock);
     return (void*)0;
}
/************************************************************************
*FUNCTION:       processDtcEvent()
*DESCRIPTION:   This function will be called to process the Dtc Event
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 20/10/2021
*************************************************************************/
bool HealthcareMonDtcProcess::processDtcEvent()
{
  ETG_TRACE_USR4(("ProcessDTCEvent()"));
  pthread_mutex_lock(&m_thMutexLock);
  setThreadState(enmThStateBusy);
  if( !getThreadRunStatus() )
  {
    setThreadState(enmThStateShuttingDown);
    pthread_mutex_unlock(&m_thMutexLock);
    return FALSE;
  }
  pthread_mutex_unlock(&m_thMutexLock);
  if( !isQueueEmpty() )
  {
     while( !isQueueEmpty() )
     {
	getdtcEventfromQueue();
	if(NULL != m_thDtcEventCurrentContainer)
	{
          processReceivedDtclist(powermode,m_thDtcEventCurrentContainer->Dev_ID,m_thDtcEventCurrentContainer->u8devNr,m_thDtcEventCurrentContainer->u16msgid,ipaddress,m_thDtcEventCurrentContainer->receivedDtcEvent);
	   pthread_mutex_lock(&m_thMutexLock);
           m_thDtcEventCurrentContainer.reset();
	   pthread_mutex_unlock(&m_thMutexLock);
	}
     }
  }
  pthread_mutex_lock(&m_thMutexLock);
  if(m_thDtcEventCurrentContainer != NULL)
  {
	  m_thDtcEventCurrentContainer.reset();
  }
  setThreadState(enmThStateWaiting);
  pthread_mutex_unlock(&m_thMutexLock);
  return true;
	
}
/************************************************************************
*FUNCTION:      isQueueEmpty()
*DESCRIPTION:   This function will return false if the Queue size is
greater than 0
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 20/10/2021
*************************************************************************/
bool HealthcareMonDtcProcess::isQueueEmpty()
{
        pthread_mutex_lock(&m_thMutexLock);
	if(m_DtcEventQueue->size() <= 0 )
	{
		pthread_mutex_unlock(&m_thMutexLock);
		return true;
	}
	pthread_mutex_unlock(&m_thMutexLock);
	return false;
}
/************************************************************************
*FUNCTION:      getdtcEventfromQueue()
*DESCRIPTION:   Get Dtc Event to process
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 19/10/2021
*************************************************************************/
bool HealthcareMonDtcProcess::getdtcEventfromQueue()
{
  ETG_TRACE_USR4(("GetDtcEvent from Queue"));
  pthread_mutex_lock(&m_thMutexLock);
  if(m_DtcEventQueue->size() == 0)
  {
	pthread_mutex_unlock(&m_thMutexLock);
      	return FALSE;
  }
  if(m_DtcEventQueue->size() > 0)
  {
     m_thDtcEventCurrentContainer = m_DtcEventQueue->at(0);
     m_DtcEventQueue->erase(m_DtcEventQueue->begin());
  }
  pthread_mutex_unlock(&m_thMutexLock);
  return TRUE;

}
/************************************************************************
*FUNCTION:       setDtcEvent ()
*DESCRIPTION:   Set Dtc Event
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 19/10/2021
*************************************************************************/
bool HealthcareMonDtcProcess::setDtcEvent(dtcEvent event)
{
   ETG_TRACE_USR4(("clNanoMsgCoreThread::setDTCEvent"));
   pthread_mutex_lock(&m_thMutexLock);
   // return FALSE if thread is not running;
   if( !getThreadRunStatus() )
   {
 
      pthread_mutex_unlock(&m_thMutexLock);
      ETG_TRACE_USR4(("HealthcareMonDtcProcess::Thread is not in Running State"));
      return FALSE;
   }
   std::shared_ptr<dtcEvent> sPa(new dtcEvent);
   sPa->Dev_ID = event.Dev_ID;
   sPa->u8devNr = event.u8devNr;
   sPa->u16msgid = event.u16msgid;
   sPa->RecevedIP = event.RecevedIP;
   sPa->receivedDtcEvent = event.receivedDtcEvent;
   pthread_mutex_unlock(&m_thMutexLock);
   if (!addDtcEventToQueue(sPa))
   {
	return FALSE;
   }
   //signal waiting thread
   pthread_cond_signal(&m_thWaitCond);
   return TRUE;
}
/************************************************************************
*FUNCTION:       addDtcEventToQueue ()
*DESCRIPTION:   Will add the new DtcEvent to Queue
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 18/10/2021
*************************************************************************/
bool HealthcareMonDtcProcess::addDtcEventToQueue(std::shared_ptr<dtcEvent> event)
{
	ETG_TRACE_USR4(("ADD the New Dtc Event to Queue"));
	pthread_mutex_lock(&m_thMutexLock);
	if(event)
	{
           m_DtcEventQueue->push_back(event);
	   pthread_mutex_unlock(&m_thMutexLock);
	   return true;
	}
	else
	{
	   pthread_mutex_unlock(&m_thMutexLock);
	   ETG_TRACE_USR4(("Invalid Dtc Event"));
	   return false;
	}

}
/************************************************************************
*FUNCTION:       checkandAssignSeverity()
*DESCRIPTION:  Will Check for the Valid Severity Range and Assign
*PARAMETER:       Reference parameter
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 17/08/2021
*************************************************************************/
bool HealthcareMonDtcProcess::checkAndAssignSeverity(uint8_t& severity)
{
  ETG_TRACE_USR4(("HealthcareMonDtcProcess CheckAndAssignSeverity for Received Severity %d",severity));
  if((severity <= MAX_KDS_SEVERITY))
  {
    return true;
  }
  else
  {
    ETG_TRACE_USR4(("HealthcareMonDtcProcess Severity is out of Range"));
    return false;
  }
}
/************************************************************************
*FUNCTION: 		getInstance()
*DESCRIPTION: 	Object creation of the class
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	rsa8cob	23/06/2021
*************************************************************************/
HealthcareMonDtcProcess* HealthcareMonDtcProcess ::getInstance()
{
	ETG_TRACE_USR4(("HealthcareMonDtcProcess getInstance called"));
	if(p_HealthCareDtcProcess == NULL)
	{
		p_HealthCareDtcProcess = new HealthcareMonDtcProcess();
		ETG_TRACE_USR4(("HealthCareDtcProcess Instance created"));
	}
	return p_HealthCareDtcProcess ;
}
/************************************************************************
*FUNCTION:           populatexmlcontents()
*DESCRIPTION:  Will Populate the Xml Content for each Subsystem
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 30/07/2021
*************************************************************************/
void HealthcareMonDtcProcess::populateXmlContents()
{
     ETG_TRACE_USR4(("HealthcareMonDtcProcess PopulateXmlContens"));
     if(NULL == tclHealthcareMonParser::pGetInstance())
     {
        ETG_TRACE_USR4(("HealthcareMonDtcProcess could not get XML Parser Instance"));
     }
     else
     {
        if(TRUE == (tclHealthcareMonParser::pGetInstance()->bParseXml()))
        {
           ETG_TRACE_USR4(("HealthcareMonDtcProcess Parsed Xml Content"));
           CMC_ParsedContent = tclHealthcareMonParser::pGetInstance()->vectorGetDTCListInfo("CMC01B");
           CMA_ParsedContent = tclHealthcareMonParser::pGetInstance()->vectorGetDTCListInfo("Amplifier");
           CMG_ParsedContent = tclHealthcareMonParser::pGetInstance()->vectorGetDTCListInfo("Glasses");
           CMP_ParsedContent = tclHealthcareMonParser::pGetInstance()->vectorGetDTCListInfo("CameraPort");
           CMR_ParsedContent = tclHealthcareMonParser::pGetInstance()->vectorGetDTCListInfo("Router");
           tDTCInfo::iterator itr; 
           
           ETG_TRACE_USR4(("HealthcareMonDtcProcess XML Size of Parsed CMC Content is %d",CMC_ParsedContent.size()));
           ETG_TRACE_USR4(("HealthcareMonDtcProcess XML Parsed Content DtcCode for CMA is %x",CMA_ParsedContent.size()));
           ETG_TRACE_USR4(("HealthcareMonDtcProcess XML Parsed Content DtcCode for CMG is %x",CMG_ParsedContent.size()));
           ETG_TRACE_USR4(("HealthcareMonDtcProcess XML Parsed Content DtcCode for CMP is %x",CMP_ParsedContent.size()));
           for(itr = CMA_ParsedContent.begin();itr != CMA_ParsedContent.end();itr++)
           {
                 ETG_TRACE_USR4(("HealthcareMonDtcProcess XML Parsed Content DtcCode for CMA is %x",itr->m_sCode));
                 ETG_TRACE_USR4(("HealthcareMonDtcProcess XML Parsed Content textid for CMA is %d",itr->m_textid));
                 ETG_TRACE_USR4(("HealthcareMonDtcProcess XML Parsed Content Error name for CMA is %s",itr->m_errname));
              
           }
           for(itr = CMG_ParsedContent.begin();itr != CMG_ParsedContent.end();itr++)
           {
                 ETG_TRACE_USR4(("HealthcareMonDtcProcess XML Parsed Content DtcCode for CMGis %x",itr->m_sCode));
                 ETG_TRACE_USR4(("HealthcareMonDtcProcess XML Parsed Content textid for CMG is %d",itr->m_textid));
                 ETG_TRACE_USR4(("HealthcareMonDtcProcess XML Parsed Content Error name for CMG is %s",itr->m_errname));

           }

        }
        else
        {
           ETG_TRACE_USR4(("HealthcareMonDtcProcess could not fetch Xml contents"));
        }         
     }
     
}

/************************************************************************
*FUNCTION: 		vDeleteInstance()
*DESCRIPTION: 	Destruction of the object class
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	rsa8cob	23/06/2021
*************************************************************************/
void HealthcareMonDtcProcess ::vDeleteInstance()
{
   if(p_HealthCareDtcProcess != NULL)
   {
          delete p_HealthCareDtcProcess ;
	  p_HealthCareDtcProcess = NULL;
   }
}
/************************************************************************
*FUNCTION: 		~HealthCareMonClient()
*DESCRIPTION: 	Destruction of the object class
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	rsa8cob	23/06/2021
*************************************************************************/
HealthcareMonDtcProcess::~HealthcareMonDtcProcess ()
{
	ETG_TRACE_USR4(("HealthcareMonDtcProcess destructor"));
	if( getThreadRunStatus() ) // gracefull termination
	{
		try
		{
			if( !pThreadStop() )
			{
				ETG_TRACE_FATAL(("HealthcareMonDtcProcess::Cannot stop thread!!"));
			}
		}
		catch( char *s )
		{
			//FATAL trace:
		}
	}
	m_DtcEventQueue->clear();
}
/************************************************************************
*FUNCTION:      processReceivedDtclist
*DESCRIPTION:   Destruction of the object class
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 10/07/2021
*************************************************************************/
void HealthcareMonDtcProcess::processReceivedDtclist(::bosch::cm::di::NanoMsgLCMInterface::entPowerMode powermode,::bosch::cm::di::NanoMsgLCMInterface::enDevID devId,uint8_t devNumber,uint16_t msgId,string receivedIp,rawDtcInfo rawdtc)
{
	if(((rawdtc.size() - 1)%4) == 0)
	{
    	  Dtclist dtclist = vecGetAdaptedDTCList(rawdtc);
          dtcinfotable *dtctab = getdtctable(devId);
          tDTCInfo *dtcXmltab = getdtcXmlContenttable(devId);
          auto itr = dtcTableInformation.find(devId);
          if(dtctab && dtcXmltab && (itr != dtcTableInformation.end()))
          {
            checkUpdateDtclistWithDtctable(devId,devNumber,*dtctab,*dtcXmltab,dtclist);
          }
	}
       else
       {
	ETG_TRACE_USR4(("HealthcareMonDtcProcess DTC list is having incorrect size %d",rawdtc.size()));
          
       }

}
/************************************************************************
*FUNCTION:      formdtclist
*DESCRIPTION:   Form the Dtclist from Raw Dtcinfo
*PARAMETER:             None
*RETURNVALUE:   Dtclist
*HISTORY:
*revision 0.1   rsa8cob 12/07/2021
************************************************************************/
Dtclist HealthcareMonDtcProcess::vecGetAdaptedDTCList(rawDtcInfo rawdtc)
{
     Dtclist dtclist;
     for(int itr = 1;itr<rawdtc.size();itr++)
     {
         dtcInfo dtcElement;
         dtcElement.u32Dtccode = 0;
         dtcElement.u8Status = 0;
         dtcElement.u32Dtccode = rawdtc[itr] << 16 | rawdtc[itr+1] << 8 | rawdtc[itr+2];
         itr = itr + 3;
         if((rawdtc[itr] & 1) && (rawdtc[itr] & (1<<3)))
         {
           dtcElement.u8Status = ACTIVE;
         }
         else
         {
           dtcElement.u8Status = INACTIVE;
         }
         dtclist .push_back(dtcElement);   
          
         ETG_TRACE_USR4(("Dtclist dtccode is %x", dtcElement.u32Dtccode));
         ETG_TRACE_USR4(("Dtclist dtccode is %d", dtcElement.u8Status));
      }
     return dtclist;
}
/************************************************************************
*FUNCTION:       getdtctable()
*DESCRIPTION:  Will return the required dtctable
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 12/07/2021
*************************************************************************/
dtcinfotable* HealthcareMonDtcProcess::getdtctable(::bosch::cm::di::NanoMsgLCMInterface::enDevID devId)
{
    auto itr = dtcTableInformation.find(devId);
    if(itr != dtcTableInformation.end())
    {
      ETG_TRACE_USR4(("Returning the Dtc Table"));
      return &(itr->second);
    }
    else
    {
      ETG_TRACE_USR4(("DTC table not found with this DevId %d", devId));
      return NULL;
    }
}
/************************************************************************
*FUNCTION:      getdtcXmlContenttable()
*DESCRIPTION:  Will return the Xml Contents for a particular Subsystem
*PARAMETER:    DevID for which the XML content is required
*RETURNVALUE:  tDTCInfo
*HISTORY:
*revision 0.1   rsa8cob 2/08/2021
*************************************************************************/
tDTCInfo* HealthcareMonDtcProcess::getdtcXmlContenttable(::bosch::cm::di::NanoMsgLCMInterface::enDevID devId)
{
    auto itr = dtcXmlContent.find(devId);
    if(itr != dtcXmlContent.end())
    {
      ETG_TRACE_USR4(("Returning the Dtc XML Content Table"));
      return &(itr->second);
    }
    else
    {
      ETG_TRACE_USR4(("DTC XML Content table not found with this DevId %d", devId));
      return NULL;
    }
}

/************************************************************************
*FUNCTION:       checkDtclistWithDtctable()
*DESCRIPTION:  Will return the dtccode to be procesed
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 13/07/2021
*************************************************************************/
void HealthcareMonDtcProcess::checkUpdateDtclistWithDtctable(::bosch::cm::di::NanoMsgLCMInterface::enDevID devId,uint8_t devNumber,dtcinfotable& dtctab,tDTCInfo& dtcparsedtable,Dtclist dtclist)
{
dtcinfotable::iterator dtcinfoitr;
Dtclist::iterator dtclistitr;
uint8_t dtcfoundstatus = DTC_NOTFOUND;
for(dtclistitr = dtclist.begin();dtclistitr < dtclist.end(); dtclistitr++)
{
  dtcfoundstatus = DTC_NOTFOUND;
  if(true ==  IsDTCCodeValid(dtclistitr->u32Dtccode,dtcparsedtable))
  {
 	 for(dtcinfoitr = dtctab.begin();dtcinfoitr < dtctab.end();dtcinfoitr++)
  	{
    		if((dtclistitr->u32Dtccode == dtcinfoitr->u32DtcCode))
    		{
          	  dtcfoundstatus = DTC_FOUND;
          	  break;
    		}
  	}
 	if(dtcfoundstatus == DTC_NOTFOUND)
  	{
           enum_dtcAvailability status = checkDtcStatusInHMIQueue(dtclistitr->u32Dtccode,dtclistitr->u8Status);
	   if(status == enmNewDTC_tobe_Added)
	   {
	      if(dtclistitr->u8Status == ACTIVE)
	      {
                 PopulateHMIQueue(devId,devNumber,dtclistitr->u32Dtccode,dtctab);
	      }
           }
        }
   }
}
}
/************************************************************************
*FUNCTION:       checkDtcStatusInHMIQueue()
*DESCRIPTION: Will check for the DTC Status in HMI Queue and will delete
that dtccode if there is a status change from Active to Inactive
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 25/10/2021
*************************************************************************/
enum_dtcAvailability HealthcareMonDtcProcess::checkDtcStatusInHMIQueue(uint32_t u32dtccode,uint8_t u8status)
{
   ETG_TRACE_USR4(("CheckDtcStatusInHMIQueue"));
   enum_dtcAvailability  dtcAvalability = enmNewDTC_tobe_Added;
   vector<dtcHMIQueue>::iterator itr;
   pthread_mutex_lock(&m_thMutexLock);
   if(m_DtcHMIQueue.size() >= 1)
   {
     for(itr = m_DtcHMIQueue.begin();itr != m_DtcHMIQueue.end(); itr++)
     {
       if(itr->u32dtccode == u32dtccode)
       {
           if(u8status == ACTIVE)
	   {
              dtcAvalability = enmDTCAvailable_And_Active;
              break;
           }	      
           if((u8status == INACTIVE) && (itr != m_DtcHMIQueue.begin()))
           {
	       m_DtcHMIQueue.erase(itr);
               ETG_TRACE_USR4(("Removed from HMI Queue since DTC status changed from Active to Inactive"));
               dtcAvalability = enmDTCAvailable_And_Inactive;
	       break;
           }
           if((u8status == INACTIVE) && (itr == m_DtcHMIQueue.begin()))
	   {
               dtcAvalability = enmDTCAvailable_And_Inactive;
               ETG_TRACE_USR4(("First Element in the HMI Queue is Inactive now"));
	       break;
	   }
      }
     }
   }
  pthread_mutex_unlock(&m_thMutexLock);
   return dtcAvalability;
}
/************************************************************************
*FUNCTION:     checkforValidSeverity
*DESCRIPTION:  Will Check for the Valid Severity
*PARAMETER:    severity
*RETURNVALUE:   Bool
*HISTORY:
*revision 0.1   rsa8cob 18/11/2021
*************************************************************************/
bool HealthcareMonDtcProcess::checkforValidSeverity(uint8_t severity)
{
        ETG_TRACE_USR4(("Check for Valid Severity %d",severity));
	if(severity >= u8KdsSeverity)
	{
		return true;
	}
	else
	{
		return false;
	}
}

/************************************************************************
*FUNCTION:     IsDTCCodeValid
*DESCRIPTION:  Will Check whether the Dtc code is vaild one or not
*PARAMETER:    dtccode,dtcXmlContenttable
*RETURNVALUE:   Bool
*HISTORY:
*revision 0.1   rsa8cob 02/08/2021
*************************************************************************/
bool HealthcareMonDtcProcess::IsDTCCodeValid(uint32_t u32dtccode,tDTCInfo dtcparsedtable)
{
    
   ETG_TRACE_USR4(("IsDTCCodeValid"));
   tDTCInfo::iterator itr;
   u16textid = 0x0;
   u8XmlSeverity = 0x0;
   default_text = "";
   bool returnstatus = false;;
   
   for(itr = dtcparsedtable.begin();itr != dtcparsedtable.end();itr++)
   {
         if(u32dtccode == itr->m_sCode)
         {
             ETG_TRACE_USR4(("Its a Valid DtcCode"));
             u16textid = itr->m_textid;
             default_text = itr->m_errname;
             u8XmlSeverity = itr->m_severity;
             bool status = checkforValidSeverity(u8XmlSeverity);
             if(status == true)
             {
               returnstatus = (u8XmlSeverity >= u8KdsSeverity)?true:false;
             }
             break;
          }
             
    }
ETG_TRACE_USR4(("Reutnstatus value is %d",returnstatus));
return returnstatus;
}
/************************************************************************
*FUNCTION:     PopulateHMIQueue
*DESCRIPTION:  Will add the new event to the HMI Queue
*PARAMETER:    devId,devNumber,dtccode,dtctable
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 13/07/2021
*************************************************************************/
void HealthcareMonDtcProcess::PopulateHMIQueue(::bosch::cm::di::NanoMsgLCMInterface::enDevID devId,uint8_t devNumber,uint32_t dtccode,dtcinfotable& dtctab)
{
	ETG_TRACE_USR4(("Populate HMI QUeue"));
        dtcHMIQueue newevent;
	newevent.Dev_ID = devId;
	newevent.u8devNr = devNumber;
	newevent.u32dtccode = dtccode;
	newevent.u16textid = u16textid;
	newevent.default_text = default_text;
	AddDTCEventtoHMIQueue(newevent);
	checkHMIQueueandUpdateHMI();
}
/************************************************************************
*FUNCTION:     AddDtcToDtctable
*DESCRIPTION:  Will add the new dtccode to the Dtctable
*PARAMETER:    devId,devNumber,dtccode,dtctable
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 13/07/2021
*************************************************************************/
void HealthcareMonDtcProcess::AddDtcToDtctable(::bosch::cm::di::NanoMsgLCMInterface::enDevID devId,uint8_t devNumber,uint32_t dtccode)
{
	ETG_TRACE_USR4(("AddToDtcTable"));
        dtctable newentry;
        newentry.Dev_ID = devId;
        newentry.u8devNr = devNumber;
        newentry.u32DtcCode = dtccode;
        newentry.u8DtcActiveStatus = ACTIVE;
        newentry.u8AckStatus = WAIT_FOR_ACK;
	pthread_mutex_lock(&m_thMutexLock);
	dtcinfotable *dtctab = getdtctable(devId);
	if(dtctab)
	{
	   dtctab->push_back(newentry);
	   pthread_mutex_unlock(&m_thMutexLock);
           ETG_TRACE_USR4(("Pushed to Dtc table"));
	}
	   pthread_mutex_unlock(&m_thMutexLock);
}
/************************************************************************
*FUNCTION:     AddDTCEventtoHMIQueue
*DESCRIPTION:  Will add the new event to dtcHMIQueue
*PARAMETER:    dtcinfotable,Dtclist
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 20/10/2021
*************************************************************************/
void HealthcareMonDtcProcess::AddDTCEventtoHMIQueue(dtcHMIQueue newevent)
{
	pthread_mutex_lock(&m_thMutexLock);
	m_DtcHMIQueue.push_back(newevent);
	pthread_mutex_unlock(&m_thMutexLock);
}
/************************************************************************
*FUNCTION:     checkHMIQueueandUpdateHMI
*DESCRIPTION:  Will update to HMI after checking from the HMI Queue
*PARAMETER:    
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 20/10/2021
*************************************************************************/
void HealthcareMonDtcProcess::checkHMIQueueandUpdateHMI()
{
     ETG_TRACE_USR4(("CheckHMI Queue and Update")); 
     pthread_mutex_lock(&m_thMutexLock);
     ::bosch::cm::di::NanoMsgLCMInterface::enDevID Dev_ID;
     uint8_t u8devNr;
     uint32_t u32dtccode;
     uint16_t u16textid;
     string default_text;
     if((enHMIPopupState == enmPOPUPIDLE) && (m_DtcHMIQueue.size() > 0))
     { 
	     enHMIPopupState = enmPOPUPBUSY;
	     Dev_ID = m_DtcHMIQueue[0].Dev_ID;
	     u8devNr = m_DtcHMIQueue[0].u8devNr;
	     u32dtccode = m_DtcHMIQueue[0].u32dtccode;
	     u16textid = m_DtcHMIQueue[0].u16textid;
	     default_text = m_DtcHMIQueue[0].default_text;
             pthread_mutex_unlock(&m_thMutexLock);
	     AddDtcToDtctable(Dev_ID,u8devNr,u32dtccode);
             HealthCareMonServer::getInstance()->vSendToHMI(static_cast<::bosch::cm::ai::HealthcareMonService::enDevID>(Dev_ID),u8devNr,u32dtccode,u16textid,default_text);
     }
     pthread_mutex_unlock(&m_thMutexLock);
}
/************************************************************************
*FUNCTION:     HandleHMIStatus
*DESCRIPTION:  Will add the new dtccode to the Dtctable
*PARAMETER:    dtcinfotable,Dtclist
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 13/07/2021
*************************************************************************/
void HealthcareMonDtcProcess::HandleHMIStatus(::bosch::cm::di::NanoMsgLCMInterface::enDevID devId,uint8_t devNumber,uint32_t dtccode,uint8_t ackstatus)
{
     ETG_TRACE_USR4(("HandleHMIStatus")); 
    dtcinfotable *dtctab = getdtctable(devId);
    if(dtctab)
    {
      dtcinfotable::iterator dtcinfoitr;
      for(dtcinfoitr = dtctab->begin();dtcinfoitr < dtctab->end();dtcinfoitr++)
      { 
         if((dtccode == dtcinfoitr->u32DtcCode))
         {
           dtcinfoitr->u8AckStatus = ackstatus;
           break;					   
         }
      }
    }
    pthread_mutex_lock(&m_thMutexLock);
    if(m_DtcHMIQueue.size() > 0)
    {
	   m_DtcHMIQueue.erase(m_DtcHMIQueue.begin());
    }
     enHMIPopupState = enmPOPUPIDLE;
     pthread_mutex_unlock(&m_thMutexLock);
     checkHMIQueueandUpdateHMI();

}
