/* ***************************************************************************************
* FILE:          HealthCareDiaglogClientHandler.cpp
* SW-COMPONENT:  HMI-BASE
 * DESCRIPTION:  HealthCareDiaglogClientHandler.cpp is part of HealtCareMonApplication
 * COPYRIGHT:  (c) 2020-21 Robert Bosch Car Multimedia GmbH
 * Author: Sathiya
 * Date: 21.09.2021
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "HealthCareDiaglogClientHandler.h"
#include "../Trace/HealthMontrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HEALTHCAREMONASF
#define ETG_I_TRACE_CHANNEL               TR_COMP_FC_TELEMATIC
#define ETG_I_TTFIS_CMD_PREFIX            "HEALTHCAREMONAPP"
#define ETG_I_FILE_PREFIX                 HEALTHCAREMONAPP::SERVICE
#include "trcGenProj/Header/HealthCareDiaglogClientHandler.cpp.trc.h"
#endif


using namespace ::asf::core;
using namespace ::diaglog_main_fi;
using namespace ::diaglog_main_fi_types;

HealthCareDiaglogClient* HealthCareDiaglogClient::p_HealthCareDiaglogClient = NULL;

#define DIAGLOG_PORT "diaglogFiPort"
#define CARRIERFILE_PATH "/sys/class/net/ethernet/carrier"

/************************************************************************
*FUNCTION: 		HealthCareDiaglogClient()
*DESCRIPTION: 	Constructor of the class
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	rsa8cob	21/09/2021
*************************************************************************/
HealthCareDiaglogClient::HealthCareDiaglogClient()
{
      InitializeTimerCallbacks();
      u64counter = 0;
      u32_EthernetTimer =  ETHERNETSTATUSTIMER;
      pFile =0;
      m_poDiaglogProxy = Diaglog_main_fiProxy::createProxy(DIAGLOG_PORT,*this);
      if(m_poDiaglogProxy == NULL)
      {
            ETG_TRACE_USR4(("HealthCareDiaglogClient Proxy creation failed"));

      }
      ETG_TRACE_USR4(("HealthCareDiaglogClient Constructor called"));
}

/************************************************************************
*FUNCTION: 		getInstance()
*DESCRIPTION: 	Object creation of the class
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	rsa8cob	21/09/2021
*************************************************************************/
HealthCareDiaglogClient* HealthCareDiaglogClient::getInstance()
{
	ETG_TRACE_USR4(("HealthCareDiaglogClient getInstance called"));
	if(p_HealthCareDiaglogClient == NULL)
	{
		p_HealthCareDiaglogClient = new HealthCareDiaglogClient();
                if(p_HealthCareDiaglogClient)
                {
	        	   ETG_TRACE_USR4(("HealthCareDiaglogClient Instance created"));
                }
	}
	return p_HealthCareDiaglogClient;
}

/************************************************************************
*FUNCTION: 		vDeleteInstance()
*DESCRIPTION: 	Destruction of the object class
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	rsa8cob	21/09/2021
*************************************************************************/
void HealthCareDiaglogClient::vDeleteInstance()
{
	if(p_HealthCareDiaglogClient != NULL)
   {
      delete p_HealthCareDiaglogClient;
	  p_HealthCareDiaglogClient = NULL;
   }
}
/************************************************************************
*FUNCTION: 		~HealthCareNanoMsgClient()
*DESCRIPTION: 	Destruction of the object class
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	rsa8cob	21/09/2021
*************************************************************************/
HealthCareDiaglogClient::~HealthCareDiaglogClient()
{
   ETG_TRACE_USR4(("HealthCareDiaglogClient Destructor"));
   if(pFile)
   {
     fclose(pFile);
   }
   p_HealthCareDiaglogClient = NULL;

}
/************************************************************************
*FUNCTION:      InitializeTimerCallbacks ()
*DESCRIPTION:   Will Intialize the timeer Callbacks
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 21/09/2021
*************************************************************************/
void HealthCareDiaglogClient::InitializeTimerCallbacks()
{
   uint8_t numofDtcElements = sizeof(dtconf)/sizeof(dtconf[0]);
   ETG_TRACE_USR4(("HealthCareDiaglogClient Number of DtcElements is %d",numofDtcElements));
   for(uint8_t itr = 0;itr<numofDtcElements;itr++)
   {
   
      TimerCallbackMapping obj;
      obj.timerfuncptr = dtconf[itr].timerfuncptr;
      obj.u8_timeinterval = dtconf[itr].u8_timeinterval;
      obj.u32_dtcCode = dtconf[itr].u32_dtcCode;
      obj.u8_dTCStatus = T_e8_TestResult__NoResult;
      obj.u8SetItcFlag = DONTSETITC;
      m_callbackinfo.push_back(obj);
   }

}
/************************************************************************
*FUNCTION: 		onAvailable()
*DESCRIPTION: 	This function gets once the DIAGLOG client Service is 
available
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	rsa8cob	21/09/2021
*************************************************************************/
void HealthCareDiaglogClient::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
    ETG_TRACE_USR4(("HealthCareDiaglogClient OnAvailable"));
    if(proxy == m_poDiaglogProxy)
    {
      ethernettimerObj.start(*this,u32_EthernetTimer,0);
      ETG_TRACE_USR4(("HealthCareDiaglogClient  Timer Started"));
    }
}
/************************************************************************
*FUNCTION: 		onUnavailable()
*DESCRIPTION: 	This function gets once the DIAGLOG client Service is 
unavailable
*PARAMETER:		None
*RETURNVALUE: 	None
*HISTORY:
*revision 0.1	rsa8cob	16/06/2021
*************************************************************************/
void HealthCareDiaglogClient::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
	ETG_TRACE_USR4(("HealthCareDiaglogClient onUnavailable"));
}
/************************************************************************
*FUNCTION:             onSaveTestResultError()
*DESCRIPTION:   This function gets called from Diaglog when there is
an Error while trying to set the ITC
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 21/09/2021
*************************************************************************/
void HealthCareDiaglogClient::onSaveTestResultError(const ::boost::shared_ptr< Diaglog_main_fiProxy >& proxy, const ::boost::shared_ptr< SaveTestResultError >& error)
{
	 ETG_TRACE_USR4(("HealthCareDiaglogClient onSaveTestResultError"));
}
/************************************************************************
*FUNCTION:            onSaveTestResultResult()
*DESCRIPTION:   This function gets called from Diaglog after setting
the required ITC in DIAGLOG
*PARAMETER:             None
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 21/09/2021
*************************************************************************/
void HealthCareDiaglogClient::onSaveTestResultResult(const ::boost::shared_ptr< Diaglog_main_fiProxy >& proxy, const ::boost::shared_ptr< SaveTestResultResult >& result)
{
	 ETG_TRACE_USR4(("HealthCareDiaglogClient onSaveTestResultResult"));
}
/************************************************************************
*FUNCTION:            onExpired()
*DESCRIPTION:   This function gets called from Asf once the timer
is expired
*PARAMETER:            
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 21/09/2021
*************************************************************************/
void HealthCareDiaglogClient::onExpired(asf::core::Timer & timer, boost::shared_ptr <asf::core::TimerPayload> payload)	
{
//   (tVoid)timer;
 //  (tVoid)payload;
   ETG_TRACE_USR4(("HealthCareDiaglogClient::onExpired entered. "));
   u64counter++;
   for(timercallbackitr = m_callbackinfo.begin();timercallbackitr != m_callbackinfo.end();timercallbackitr++)
   {
	   
	   if((u64counter % timercallbackitr->u8_timeinterval) == 0)
	   {
		  if(timercallbackitr->timerfuncptr)
		  {
                    ETG_TRACE_USR4(("HealthCareDiaglogClient::Calling Callback function"));
	            executeptr = NULL;
		    executeptr = timercallbackitr->timerfuncptr;
		    (this->*executeptr)();
		    if(timercallbackitr->u8SetItcFlag == SETITC)
		    {
                         SetItc(timercallbackitr->u32_dtcCode,(T_e8_TestResult)timercallbackitr->u8_dTCStatus);
			 timercallbackitr->u8SetItcFlag = DONTSETITC;

		    }
		  }
	   }
   }
}
/************************************************************************
*FUNCTION:           readfile()
*DESCRIPTION:   This function will read the content of the file
*PARAMETER:
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 21/09/2021
*************************************************************************/
void HealthCareDiaglogClient::readfile()
{

ETG_TRACE_USR4(("HealthCareDiaglogClient::ReadFile Entered"));
uint32_t u32filecontent = 0;
timercallbackitr->u8SetItcFlag = DONTSETITC;
      pFile = fopen(CARRIERFILE_PATH,"r");
 if(pFile)
 {
	 if(fseek(pFile,0,SEEK_SET) != 0)
		ETG_TRACE_USR4(("fseek() fail"));
	 fscanf(pFile,"%d",&u32filecontent);
         ETG_TRACE_USR4(("HealthCareDiaglogClient::File Content is %d",u32filecontent));
	 if((u32filecontent == 0) && (timercallbackitr->u8_dTCStatus != T_e8_TestResult__Failed ))
	 {
	   timercallbackitr->u8SetItcFlag = SETITC;
           timercallbackitr->u8_dTCStatus = T_e8_TestResult__Failed;
         }
	 else if((u32filecontent == 1) && (timercallbackitr->u8_dTCStatus != T_e8_TestResult__Passed))
	 {
		   timercallbackitr->u8SetItcFlag = SETITC;
                   timercallbackitr->u8_dTCStatus = T_e8_TestResult__Passed;
	 }
	fclose(pFile);
 }
 else
 {
         ETG_TRACE_USR4(("HealthCareDiaglogClient File Descriptor is NULL"));
 }
 

}
/************************************************************************
*FUNCTION:       SetItc()
*DESCRIPTION:   This function will be called based on File Content
*PARAMETER:
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 21/09/2021
*************************************************************************/
void HealthCareDiaglogClient::SetItc(uint32_t u32_dtccode,T_e8_TestResult u8_status)
{
  ETG_TRACE_USR4(("HealthCareDiaglogClient SetItc Code %x", u32_dtccode));
  ETG_TRACE_USR4(("HealthCareDiaglogClient SetItc Status %d", u8_status));
  T_TestResult  oTestResult;
  oTestResult.setTroubleCode(u32_dtccode);
  oTestResult.setResult(u8_status);
  T_TestResultList oTestResultList;
  oTestResultList.reserve(1);
  oTestResultList.push_back(oTestResult);
  m_poDiaglogProxy->sendSaveTestResultStart(*this,oTestResultList);

}
/************************************************************************
*FUNCTION:      _bGetGPIOState_Reverse()
*DESCRIPTION:   This function will be called based on Internal timer
*PARAMETER:
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 4/10/2021
*************************************************************************/
void HealthCareDiaglogClient::checkGPIOState_MCP()
{
   ETG_TRACE_USR4(("CheckGPIOState_MCP() entered."));
             tBool  bMcpRet = FALSE;
	     tBool  bMcpDiagRet = FALSE;
   OSAL_trGPIOData  gpioData;
              tU32  u32OSALError;
   OSAL_tIODescriptor m_hGpioHandle_Mcp;
  
       m_hGpioHandle_Mcp = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE);
   if(m_hGpioHandle_Mcp != OSAL_ERROR)   {
       gpioData.tId = OSAL_EN_U140_SW_ENABLE;

       if (OSAL_ERROR == OSAL_s32IOControl(m_hGpioHandle_Mcp, OSAL_C_32_IOCTRL_GPIO_IS_STATE_ACTIVE, (intptr_t)(&gpioData)))   {
           u32OSALError = OSAL_u32ErrorCode();
           ETG_TRACE_ERR(("CheckGPIO_MCP() Get GPIO State failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
       }
       else   {
           //state
           bMcpRet = gpioData.unData.bState;
           ETG_TRACE_USR3(("MCP Pin State = %u", bMcpRet));
       }
       gpioData.tId = OSAL_EN_U140_SW_DIAG;

       if (OSAL_ERROR == OSAL_s32IOControl(m_hGpioHandle_Mcp,OSAL_C_32_IOCTRL_GPIO_IS_STATE_ACTIVE, (intptr_t)(&gpioData)))   {
           u32OSALError = OSAL_u32ErrorCode();
           ETG_TRACE_ERR(("checkGPIO_DIAGMCP() Get GPIO State failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
       }
       else   {
           //state
           bMcpDiagRet = gpioData.unData.bState;
           ETG_TRACE_USR3(("MCP Diag Pin State = %u",bMcpDiagRet));
       }
       if (OSAL_s32IOClose(m_hGpioHandle_Mcp) != OSAL_OK){
			ETG_TRACE_ERR(("Error detected closing MCP GPIO Handle!!!!!!"));
		}

      checkItcStateforMCP(bMcpRet,bMcpDiagRet);
     }
     else
     {
	     ETG_TRACE_ERR(("CheckGPIO_MCP() OSAL_IOOPEN Failed"));
     }
   }
/************************************************************************
*FUNCTION:      checkItcStateforMCP()
*DESCRIPTION:   This function will be determine the ITC to set or clear
for MCP pin voltage condition
*PARAMETER:
*RETURNVALUE:   None
*HISTORY:
*revision 0.1   rsa8cob 4/10/2021
*************************************************************************/
void HealthCareDiaglogClient::checkItcStateforMCP(bool bMcpValue,bool bMcpDiagValue)
{
   ETG_TRACE_USR4(("checkItcStateforMCP() entered with values %d, %d",bMcpValue,bMcpDiagValue));
   
   if((bMcpValue == bMcpDiagValue) && (timercallbackitr->u8_dTCStatus != T_e8_TestResult__Failed ))
   {
	
	 if( (timercallbackitr->u32_dtcCode == ITC_CMC_SWITCHABLE_OUTPUT_VOLTAGE_SHORT_TO_BATTERY) && ((0 == bMcpValue) && (0 == bMcpDiagValue) ) )
	 {	 
           ETG_TRACE_USR4(("checkItcStateforMCP() entered Short to Battery for MCP"));
	   timercallbackitr->u8SetItcFlag = SETITC;
           timercallbackitr->u8_dTCStatus = T_e8_TestResult__Failed;
	 }
	 else if( (timercallbackitr->u32_dtcCode == ITC_CMC_SWITCHABLE_OUTPUT_VOLTAGE_SHORT_TO_GROUND) && ((1 == bMcpValue ) && (1 == bMcpDiagValue) ) )
	 {
           ETG_TRACE_USR4(("checkItcStateforMCP() entered Short to Ground for MCP"));
	   timercallbackitr->u8SetItcFlag = SETITC;
           timercallbackitr->u8_dTCStatus = T_e8_TestResult__Failed;
	 }
   }
   else if((bMcpValue != bMcpDiagValue) && (timercallbackitr->u8_dTCStatus != T_e8_TestResult__Passed))
   {
           ETG_TRACE_USR4(("checkItcStateforMCP() entered for clearing ITC for MCP"));
	   timercallbackitr->u8_dTCStatus = T_e8_TestResult__Passed;
	   timercallbackitr->u8SetItcFlag = SETITC;
   }
}
