/****************************************************************************
* Copyright (c) 2019-2020 Robert Bosch Car Multimedia GmbH
* duplication and disclosure to third parties is prohibited.
*
* FILE                : TFTPClientCMR.cpp
* COMPONENT Name      : di_middlewareserver
* DESCRIPTION         : Handles downloading of DB or Albumart via TFTP protocol for external media player List
* AUTHOR              : Madhuree
* Date                : 31.08.2020
* Revision History    : 0.1
* Date 31.08.2020     : Initial version
****************************************************************************/


#include "TFTPClientCMR.h"
#include "tClAlbumArt.h"
#include "ListHandlerLogic.h"
#include <cstdlib>
#include <thread>
#include <curl/curl.h>
#include <iostream>
#include <cstring>
#include <cstdio>
#include <cstdlib>


/* Header files required for Tracing out log statements */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "EMPListTraceConfig.h" // For using the Trace Class ID
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_LISTHANDLER_TFTPCLIENT
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_NAVRES1
#include "trcGenProj/Header/TFTPClientCMR.cpp.trc.h"
#endif
using namespace std;

TFTPClientCMR* TFTPClientCMR::m_poSelf = NULL;

/************************************************************************
*FUNCTION: 		my_fwrite()
*DESCRIPTION:   write a file from TFTPServer to client
*PARAMETER:		None
*RETURNVALUE: 	size_t
*************************************************************************/
size_t TFTPClientCMR::my_fwrite(void *buffer, size_t size, size_t nmemb, void *stream)
{
  TftpFile *out = (TftpFile *)stream;
  if(!out->stream) {
    /* open file for writing */
    out->stream = fopen(out->filename, "wb");
    if(!out->stream)
      return -1; /* failure, can't open file to write */
  }
  return fwrite(buffer, size, nmemb, out->stream);
}

/************************************************************************
*FUNCTION: 		TFTPClientCMR()
*DESCRIPTION:   Constructor of the class TFTPClientCMR
*PARAMETER:		None
*RETURNVALUE: 	None
*************************************************************************/
TFTPClientCMR::TFTPClientCMR()
{   
	ETG_TRACE_USR4((" TFTPClientCMR::TFTPClientCMR() "));
	m_pEntityInfo = tclAvRoutingParser::pGetInstance();
    if (initializeXMLParser ())
	{
		ETG_TRACE_USR4((" TFTPClientCMR::TFTPClientCMR() Parser status true"));
	}
    createDirectory();
}

void TFTPClientCMR::createDirectory()
{
   ETG_TRACE_USR4((" TFTPClientCMR::createDirectory()"));
   std::string DownloadAlbumart = "mkdir -m 777 -p ";
    DownloadAlbumart.append(CMR_ALBUMART_PATH);
    m_albumartDirCreation= system(DownloadAlbumart.c_str());
    ETG_TRACE_USR4(("TFTPClientCMR::createDirectory(), AA Directory :%d",m_albumartDirCreation));
    std::string DownloadDBPath = "mkdir -m 777 -p ";
    DownloadDBPath.append(CMR_DATABASE_PATH);
    m_databaseDirCreation= system(DownloadDBPath.c_str());
     ETG_TRACE_USR4(("TFTPClientCMR::createDirectory(), DB Directory :%d",m_databaseDirCreation));
    system("sync");
}

/************************************************************************
*FUNCTION: 		~TFTPClientCMR()
*DESCRIPTION:   Destructor  of the class TFTPClientCMR
*PARAMETER:		None
*RETURNVALUE: 	None
*************************************************************************/
TFTPClientCMR::~TFTPClientCMR()
{
    ETG_TRACE_USR4((" TFTPClientCMR::~TFTPClientCMR() "));
}

/************************************************************************
*FUNCTION: 		initializeXMLParser()
*DESCRIPTION:           parse AvRoutingAdapter.xml and get Ip address of CMR
*PARAMETER:		None
*RETURNVALUE: 	       bool
*************************************************************************/

bool TFTPClientCMR::initializeXMLParser()
{
    ETG_TRACE_USR4(("TFTPClientCMR::initializeXMLParser()"));
    m_bParseState = m_pEntityInfo->bParseXml();
    bool status = false ;
    if ( m_bParseState  )
    {
        vector<std::string> vecEntityIpCMR = m_pEntityInfo->vectorGetEntityIP( DEV_NAME_CMR );
        if ( vecEntityIpCMR.size() > 0  )
        {
		CmrIpAddress = vecEntityIpCMR[0];
		status = true ;
	}
        ETG_TRACE_USR4(("TFTPClientCMR::initializeXMLParser(), IP address of CMR is :%s",CmrIpAddress.c_str()));
        
    }
    else
    {
       ETG_TRACE_USR4(("TFTPClientCMR::initializeXMLParser(), Parser failed"));
    }
    return status ;

}

/************************************************************************
*FUNCTION: 		curlDownloadFile()
*DESCRIPTION:   LibCurl call for downloading file from TFTP server
*PARAMETER:		TftpFile *tftpFile, std::string url
*RETURNVALUE: 	bool
*************************************************************************/
bool TFTPClientCMR::curlDownloadFile(TftpFile *tftpFile, std::string url)
{
  CURL *curl;
  CURLcode res{CURLE_GOT_NOTHING};
  curl_global_init(CURL_GLOBAL_DEFAULT);
  curl = curl_easy_init();
  bool isDownloadsuccess = false;
  if(curl) {
    ETG_TRACE_USR4(("TFTPClientCMR::curlDownloadFile URL:%s",url.c_str()));
    if(CURLE_OK != curl_easy_setopt(curl, CURLOPT_URL,url.c_str()))
	isDownloadsuccess = false;
    if(CURLE_OK != curl_easy_setopt(curl, CURLOPT_LOCALPORT, 5784L))
	isDownloadsuccess = false;
         /* and try 10 more ports following that */
    if(CURLE_OK != curl_easy_setopt(curl, CURLOPT_LOCALPORTRANGE, 5L))
	isDownloadsuccess = false;
	//printf("curl set opt completed \n");
    /*callback to get called when there's data to be written */
    if(CURLE_OK != curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, my_fwrite))
	isDownloadsuccess = false;
	/* Set a pointer to our struct to pass to the callback */
    if(CURLE_OK != curl_easy_setopt(curl, CURLOPT_WRITEDATA, tftpFile))
	isDownloadsuccess = false;
	//timeout set as 5sec
    if(CURLE_OK != curl_easy_setopt(curl, CURLOPT_TIMEOUT, 15L))
	isDownloadsuccess = false;
    /* Switch on full protocol/debug output */
    //curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);
    ETG_TRACE_USR4(("TFTPClientCMR::curlDownloadFile() starting downloading for URL:%s",url.c_str()));
    res = curl_easy_perform(curl);
	/* always cleanup */
    curl_easy_cleanup(curl);

    if(CURLE_OK != res) {
	  ETG_TRACE_USR4((" Curl perform failed: %d ",res));
	  ETG_TRACE_FATAL(("Curl perform failed stderr: %s", curl_easy_strerror(res)));
	  isDownloadsuccess = false;
    }	
    }
  if(tftpFile->stream)
  {
    fclose(tftpFile->stream); /* close the local file */
    if(CURLE_OK == res)
        isDownloadsuccess = true;
  }
  curl_global_cleanup();
  return isDownloadsuccess;
}

/************************************************************************
*FUNCTION: 		downloadAlbumart()
*DESCRIPTION:   prepare URL for downloading Albumart and store filename 
                for removing previously downloaded file
*PARAMETER:		std::string IP, uint16_t songID, uint16_t FileType
*RETURNVALUE: 	Bool
*************************************************************************/
bool TFTPClientCMR::downloadAlbumart(std::string IP, uint16_t songID, uint16_t FileType)
{	
	ETG_TRACE_USR4(("TFTPClientCMR::downloadAlbumart(), songid is:%d, ip is: %s", songID,IP.c_str()));
	ETG_TRACE_USR4(("TFTPClientCMR::downloadAlbumart(), FileType %d", FileType));
	bool ret = false;
	std::string albumArtFileName;
    if(FileType == EnumConst::entFileType_Video)
    {
        albumArtFileName = std::to_string(songID) + ".video";
    }
    else if(FileType == EnumConst::entFileType_Audio)
    {
        albumArtFileName = std::to_string(songID) + ".audio";
    }
    else
    {
        return ret;
    }
	std::string url = "tftp://"+ IP + ":69/" + albumArtFileName;
	std::string rcvdABFile ="/tmp/CMR/AlbumArt/AlbumArt";
	char *cstr = new char[rcvdABFile.length() + 1];
    strcpy(cstr, rcvdABFile.c_str());
	TftpFile tftpAlbumartfile= {cstr,NULL};
	if(!savedAlbumartName.empty())
	{
	   removePrevAlbumart();
	}
        if (!m_albumartDirCreation)
        {ret = curlDownloadFile(&tftpAlbumartfile, url);}
         else {ETG_TRACE_USR4(("TFTPClientCMR::downloadAlbumart(), albumart directory not available"));}
	if (ret)
	{
		savedAlbumartName=rcvdABFile;
	}
	return ret;
}

/************************************************************************
*FUNCTION: 		downloadDB()
*DESCRIPTION:   prepare URL for downloading DB
*PARAMETER:		std::string IP
*RETURNVALUE: 	Bool
*************************************************************************/
bool TFTPClientCMR::downloadDB(std::string IP)
{	
    ETG_TRACE_USR4(("TFTPClientCMR::downloadDB(), IP is: %s",IP.c_str()));
    bool ret = false;
    std::string dBFileName(CMR_DB_FILENAME);
    std::string url ="tftp://" + IP + ":69/" + dBFileName;
    std::string rcvdFile ="/tmp/CMR/Database/USB_DB.sqlite";
    char *cstr = new char[rcvdFile.length() + 1];
    strcpy(cstr, rcvdFile.c_str());
    TftpFile tftpDbFile= {cstr,NULL};
    if (!m_databaseDirCreation)
    {
        if(!(ret = curlDownloadFile(&tftpDbFile, url)))
        {
                RemoveDBFile(EnumConst::entSourceType_CMR);
        }
    }
    else {ETG_TRACE_USR4(("TFTPClientCMR::downloadDB(), database directory not available"));}
    return ret;
}   

/************************************************************************
*FUNCTION: 		reqDBDownload()
*DESCRIPTION:   to initiate DB download request
*PARAMETER:		uint16_t sourceType
*RETURNVALUE: 	EnumConst::entDownloadState
*************************************************************************/
EnumConst::entDownloadState TFTPClientCMR::reqDBDownload(uint16_t sourceType)
{
     ETG_TRACE_USR4(("TFTPClientCMR::reqDBDownload()"));
     std::string IP= getCmrIpAddress();
     bool ret = downloadDB(IP);
     ETG_TRACE_USR4(("TFTPClientCMR::reqDBDownload() result of DB download is: %d",ret)); 
     EnumConst::entDownloadState dbDownloadStatus = ret? EnumConst::entDownloadState_Downloadsuccess : EnumConst::entDownloadState_DownloadError;
     ETG_TRACE_USR4(("TFTPClientCMR::reqDBDownload()DB download result : %d",dbDownloadStatus));
     return dbDownloadStatus;
}

/************************************************************************
*FUNCTION: 		reqAlbumArtDownload
*DESCRIPTION:   to initiate Albumart download request
*PARAMETER:		uint16_t sourceType, uint16_t songId, uint16_t FileType
*RETURNVALUE: 	EnumConst::entDownloadState
*************************************************************************/
EnumConst::entDownloadState TFTPClientCMR::reqAlbumArtDownload(uint16_t sourceType, uint16_t songId, uint16_t FileType)
{
        ETG_TRACE_USR4(("TFTPClientCMR::reqAlbumArtDownload"));
        std::string IP= getCmrIpAddress();
	bool ret = downloadAlbumart(IP, songId, FileType);
	ETG_TRACE_USR4(("TFTPClientCMR::reqAlbumArtDownload() result of AlbumArt download is: %d",ret));
        EnumConst::entDownloadState albumartDownloadResult= ret? EnumConst::entDownloadState_Downloadsuccess : EnumConst::entDownloadState_DownloadError;
       ETG_TRACE_USR4(("TFTPClientCMR::reqAlbumArtDownload()Albumart download result: %d",albumartDownloadResult));
        return albumartDownloadResult;
}

/************************************************************************
*FUNCTION: 		reqAlbumArtDownload
*DESCRIPTION:   to remove previously downloaded albumart file
*PARAMETER:		None
*RETURNVALUE: 	None
*************************************************************************/
void TFTPClientCMR::removePrevAlbumart()
{
   ETG_TRACE_USR4(("TFTPClientCMR::removePrevAlbumart()"));
   char *filename = new char[savedAlbumartName.length() + 1];
    strcpy(filename, savedAlbumartName.c_str());
    ETG_TRACE_USR4(("TFTPClientCMR::removePrevAlbumart() file to be deleted : %s", filename));
    if( remove( filename ) != 0 )
     {
        ETG_TRACE_USR4(("Error deleting file: %s", filename));
     }
    else
     {
        ETG_TRACE_USR4(( "File: %s, successfully deleted",filename ));
     }
     delete[] filename;
}

/************************************************************************
*FUNCTION: 		RemoveDBFile
*DESCRIPTION:   to remove previously downloaded DB file
*PARAMETER:		uint16_t sourceType
*RETURNVALUE: 	None
*************************************************************************/
void TFTPClientCMR::RemoveDBFile(uint16_t sourceType)
{ 
    ETG_TRACE_USR4(("TFTPClientCMR::RemoveDBFile() sourcetype is :%d", sourceType));
    if(sourceType == EnumConst::entSourceType_CMR)
    {
        std::string DBFile ="/tmp/CMR/Database/USB_DB.sqlite";
        char *filename = new char[DBFile.length() + 1];
        strcpy(filename, DBFile.c_str());
        ETG_TRACE_USR4(("TFTPClientCMR::RemoveDBFile() file to be deleted : %s", filename));
        if( remove( filename ) != 0 )
         {
           ETG_TRACE_USR4(("TFTPClientCMR::RemoveDBFile(), Error deleting file: %s", filename));
         }
         else
         {
            ETG_TRACE_USR4(( "TFTPClientCMR::RemoveDBFile(), File: %s, successfully deleted",filename ));
         }
	 delete[] filename;
     }
     
}

std::string  TFTPClientCMR::getCmrIpAddress()
{
	return  CmrIpAddress  ;
}
