/* ***************************************************************************************
 * FILE:          clListHandlerCoreThread.cpp
 * SW-COMPONENT:  EMP_List_Handler_Application
 * DESCRIPTION:  clListHandlerCoreThread.cpp is part of EMP List Handler Application
 * 				 This is thread base class with queue and wait logic, class that 
 *				 inherits from this would have to implement _ListRequestCallBack(std::shared_ptr<stListRequestData> data)
 *				 function to get a msg update. 
 * COPYRIGHT:  (c) 2020-21 	Robert Bosch Car Multimedia GmbH
 * HISTORY: 
 * AUTHOR:  Vasuki T(RBEI/ECG5)  21.12.2020
 *				Initial version
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */

#include "clListHandlerCoreThread.h"
#include "ListHandlerTypes.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "EMPListTraceConfig.h" // For using the Trace Class ID
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_LISTHANDLER_LOGIC
#define ETG_I_TRACE_CHANNEL     TR_TTFIS_NAVRES1
#include "trcGenProj/Header/clListHandlerCoreThread.cpp.trc.h"
#endif


using namespace listHandlerTypes;

void msSleep( unsigned int msec )
{
	usleep(msec*1000);
}
/************************************************************************
*FUNCTION: 		clListHandlerCoreThread()
*DESCRIPTION:   Constructor, initializes queue for adding msg's
*PARAMETER:		None
*RETURNVALUE: 	None
*History:		Vasuki T (RBEI/ECG5)   Initial version
*************************************************************************/
clListHandlerCoreThread::clListHandlerCoreThread()
:m_thRunStatus(FALSE)
,m_threadID(0L)
,m_thCurrentState(enmThStateDown)
,m_thListReqQueue(NULL)
,m_thListReqCurrentContainer(NULL)
,m_thStackSize(0)
,m_thStopTimer(30)
,m_thStatusTime(100)
{
	ETG_TRACE_USR4(("clListHandlerCoreThread::Constructor!!"));
    m_thListReqQueue = new vector<std::shared_ptr<stWindowSliceRequest>>(0);
	if( !m_thListReqQueue )
	{
		setThreadState(enmThStateFault);
		ETG_TRACE_USR4(("clListHandlerCoreThread::Queue creation failed, hence thread not created!!"));
		//sd_notify(0, "ERRNO=2"); // No tested so far, so commented out.
		return; //to be commented out when we enable sd_notify
	}
	// m_plistHandler = tclListHandlerLogic::GetInstance();
	pthread_mutexattr_t mattr;
	pthread_mutexattr_init(&mattr);
	pthread_mutex_init(&m_thMutexLock,&mattr);
	pthread_cond_init(&m_thWaitCond,NULL);
	pThreadStart();
}

/************************************************************************
*FUNCTION: 		~clListHandlerCoreThread()
*DESCRIPTION:   Destructor
*PARAMETER:		None
*RETURNVALUE: 	None
*History:		Vasuki T (RBEI/ECG5)   Initial version
*************************************************************************/
clListHandlerCoreThread::~clListHandlerCoreThread(void)
{
	if( getThreadRunStatus() ) // gracefull termination
	{
		try
		{
			if( !pThreadStop() )
			{
				ETG_TRACE_USR4(("clListHandlerCoreThread::Cannot stop thread!!"));
			}
		}
		catch( char *s )
		{
			//FATAL trace:
		}
	}
	m_thListReqQueue->clear();
}

/************************************************************************
*FUNCTION: 		thStartFunction(void *arg);
*DESCRIPTION:   Thread routine function, core threads(Receiver & Sender) thread entry point
*PARAMETER:		void *
*RETURNVALUE: 	None
*History:		Vasuki T (RBEI/ECG5)   Initial version
*************************************************************************/
VOIDPTR clListHandlerCoreThread::thStartRoutine( VOIDPTR List )
{
	ETG_TRACE_USR4(("clListHandlerCoreThread::thStartRoutine"));
	pthread_mutex_lock(&m_thMutexLock);

	setThreadState(enmThStateWaiting);
	setThreadRunStatus(TRUE);

	pthread_mutex_unlock(&m_thMutexLock);
	while( getThreadRunStatus() )
	{
		pthread_mutex_lock(&m_thMutexLock);
		//Wait for the message through signal
		if (0 != pthread_cond_wait(&m_thWaitCond, &m_thMutexLock))
		{
			break;
		}
		pthread_mutex_unlock(&m_thMutexLock);
		if(!processListRequest() )
		{
			//if(!getThreadRunStatus()){
			  // ETG_TRACE_FATAL(("clListHandlerCoreThread::processListRequest retured a FALSE"));
			  // break;
			//}
		}
	}
	pthread_mutex_lock(&m_thMutexLock);
	setThreadState(enmThStateDown);
	setThreadRunStatus(FALSE);
	pthread_mutex_unlock(&m_thMutexLock);

	return (VOIDPTR)0;
}

/************************************************************************
*FUNCTION: 		setListRequestEvent
*DESCRIPTION:   struct msg would be added to the queue and signal wait thread
*				Called by Dbus server and protocol threads
*PARAMETER:		stListRequestData  struct
*RETURNVALUE: 	bool
*History:		Vasuki T (RBEI/ECG5)   Initial version
*************************************************************************/
bool clListHandlerCoreThread::setListRequestEvent(stWindowSliceRequest listData)
{
	ETG_TRACE_USR4(("clListHandlerCoreThread::setListRequestEvent"));
	pthread_mutex_lock(&m_thMutexLock);
 // return FALSE if thread is not running;
	if( !getThreadRunStatus() )
	{
		pthread_mutex_unlock(&m_thMutexLock);
		return FALSE;
	}
/* 	create shared pointer of stListRequestData struct and copy data 
	to ensure data is not lost between threads */
	std::shared_ptr<stWindowSliceRequest> sPa(new stWindowSliceRequest);
	sPa->ListType = listData.ListType;
	sPa->startIndex = listData.startIndex;
	sPa->windowSize = listData.windowSize;
	sPa->Region = listData.Region;
	sPa->SourceType = listData.SourceType;
	sPa->FolderPath = listData.FolderPath;
		

	pthread_mutex_unlock(&m_thMutexLock);
	// Add Msg to queue
	if (!addListRequestToQueue(sPa))
	{
		return FALSE;
	}
	//signal waiting thread
	pthread_cond_signal(&m_thWaitCond);

	return TRUE;
}

/************************************************************************
*FUNCTION: 		processNanoMsg
*DESCRIPTION:   After getting signal to waiting thread, this function would
*				would get called to process the message added to queue
*PARAMETER:		void
*RETURNVALUE: 	bool
*History:		Vasuki T (RBEI/ECG5)   Initial version
*************************************************************************/
bool clListHandlerCoreThread::processListRequest()
{
	ETG_TRACE_USR4(("clListHandlerCoreThread::processListRequest()"));
	pthread_mutex_lock(&m_thMutexLock);
	setThreadState(enmThStateBusy);
	if( !getThreadRunStatus() )
	{
		setThreadState(enmThStateShuttingDown);
		pthread_mutex_unlock(&m_thMutexLock);
		return FALSE;
	}

	pthread_mutex_unlock(&m_thMutexLock);
	if( !isQueueEmpty() )
	{
		while( !isQueueEmpty() )
		{
			getListRequestFromQueue(); //getListRequestFromQueue updates m_thListReqCurrentContainer
			if(NULL != m_thListReqCurrentContainer)
			{
				if( !_listRequestCallBack(m_thListReqCurrentContainer) )
				{
					pthread_mutex_lock(&m_thMutexLock);
					m_thListReqCurrentContainer.reset();
					//setThreadState(enmThStateShuttingDown);
					pthread_mutex_unlock(&m_thMutexLock);
				    ETG_TRACE_USR4(("clListHandlerCoreThread::processListRequest retured a FALSE"));
			//		//Todo error handling in SOP2
					//return FALSE;
				}
			}
		}
		pthread_mutex_lock(&m_thMutexLock);
		if(m_thListReqCurrentContainer != NULL)
		{
		 m_thListReqCurrentContainer.reset();// delete managed object
		}
		setThreadState(enmThStateWaiting);
		pthread_mutex_unlock(&m_thMutexLock);
	}
	return TRUE;
}

// bool clListHandlerCoreThread::_listRequestCallBack(std::shared_ptr<stWindowSliceRequest> listData)
// {
	// ETG_TRACE_USR4(("clListHandlerCoreThread::_listRequestCallBack"));
	// if (NULL != listData)
	// m_plistHandler->vRequestMediaPlayerListSlice(*(listData.get()));
	// return true;
	// //vRequestMediaPlayerListSlice(listType, sourceType, region, startingIndex, windowSize);
// }


/************************************************************************
*FUNCTION: 		isQueueEmpty
*DESCRIPTION:   Checks the size of the queue and sends true if >0 or false
*PARAMETER:		void
*RETURNVALUE: 	bool
*History:		Vasuki T (RBEI/ECG5)   Initial version
*************************************************************************/
bool clListHandlerCoreThread::isQueueEmpty()
{
	ETG_TRACE_USR4(("clListHandlerCoreThread::isQueueEmpty"));
	ETG_TRACE_USR4(("clListHandlerCoreThread::isQueueEmpty queue size %d", m_thListReqQueue->size()));
	pthread_mutex_lock(&m_thMutexLock);
	if( m_thListReqQueue->size() <= 0 )
	{
		pthread_mutex_unlock(&m_thMutexLock);
		return TRUE;
	}
	pthread_mutex_unlock(&m_thMutexLock);
	return FALSE;
}

/************************************************************************
*FUNCTION: 		addListRequestToQueue
*DESCRIPTION:   adds msg's to the queue and checks if size> 100.
*				current size of the queue is kept to 100 per thread
*PARAMETER:		void
*RETURNVALUE: 	bool
*History:		Vasuki T (RBEI/ECG5)   Initial version
*************************************************************************/
bool clListHandlerCoreThread::addListRequestToQueue(std::shared_ptr<stWindowSliceRequest> nanoData)
{
	ETG_TRACE_USR4(("clListHandlerCoreThread::addListRequestToQueue"));
	if( !nanoData ) return TRUE;

	pthread_mutex_lock(&m_thMutexLock);
	if( m_thListReqQueue->size() >= NANO_MSG_QUEUE_SIZE )
	{
		pthread_mutex_unlock(&m_thMutexLock);
		ETG_TRACE_USR4(("clListHandlerCoreThread::addListRequestToQueue,size() >= NANO_MSG_QUEUE_SIZE"));
		return FALSE;
	}
	//Push into vector
	m_thListReqQueue->push_back(nanoData);
	pthread_mutex_unlock(&m_thMutexLock);
	if( m_thListReqQueue->size() >= NANO_MSG_QUEUE_SIZE )
	{
		ETG_TRACE_USR4(("clListHandlerCoreThread::addListRequestToQueue, stack full"));
	}
	return TRUE;
}

/************************************************************************
*FUNCTION: 		getListRequestFromQueue
*DESCRIPTION:   Picks the 1st item from queue and puts on current container
*				and removes item from queue
*PARAMETER:		void
*RETURNVALUE: 	bool
*History:		Vasuki T (RBEI/ECG5)   Initial version
*************************************************************************/
bool clListHandlerCoreThread::getListRequestFromQueue()
{
	ETG_TRACE_USR4(("clListHandlerCoreThread::getNanoMsgFromQueue()"));
	pthread_mutex_lock(&m_thMutexLock);
	if( m_thListReqQueue->size() == 0 )
	{
		pthread_mutex_unlock(&m_thMutexLock);
		return FALSE;
	}

	if(m_thListReqQueue->size() > 0)
	{
	  m_thListReqCurrentContainer = m_thListReqQueue->at(0);
	  m_thListReqQueue->erase(m_thListReqQueue->begin());
	}
	pthread_mutex_unlock(&m_thMutexLock);
	return TRUE;
}

/************************************************************************
*FUNCTION: 		pThreadStop
*DESCRIPTION:   Makes thread run status to false to exit from while and stops thread
*PARAMETER:		void
*RETURNVALUE: 	bool
*History:		Vasuki T (RBEI/ECG5)   Initial version
*************************************************************************/
bool clListHandlerCoreThread::pThreadStop()
{
	ETG_TRACE_USR4(("clListHandlerCoreThread::pThreadStop!!"));
	try 
	{
		pthread_mutex_lock(&m_thMutexLock);
		setThreadRunStatus(FALSE);
		pthread_mutex_unlock(&m_thMutexLock);

		pthread_cond_signal(&m_thWaitCond);

		int ticks = (m_thStopTimer*1000)/100;
		for( int i=0; i<ticks; i++ )
		{
			msSleep(100);// Need to find a solution, currently sleep is used until signal is processed
			pthread_mutex_lock(&m_thMutexLock);
			if( getThreadState() == enmThStateDown )
			{
				pthread_mutex_unlock(&m_thMutexLock);
				return TRUE;
			}
			pthread_mutex_unlock(&m_thMutexLock);
		} 
	}
	catch (char *s)
	{
		ETG_TRACE_USR4(("clListHandlerCoreThread::Thread Exit failed!!"));
	}
	return FALSE;
}

/************************************************************************
*FUNCTION: 		pThreadStop
*DESCRIPTION:   thread create function 
*PARAMETER:		void
*RETURNVALUE: 	bool
*History:		Vasuki T (RBEI/ECG5)   Initial version
*************************************************************************/
bool clListHandlerCoreThread::pThreadStart()
{
	ETG_TRACE_USR4(("clListHandlerCoreThread::pThreadStart!!"));
	try 
	{
		pthread_mutex_lock(&m_thMutexLock);
		if( getThreadRunStatus() )
		{
			pthread_mutex_unlock(&m_thMutexLock);
			return TRUE;
		}

		pthread_mutex_unlock(&m_thMutexLock);

		pthread_attr_t attr;
		pthread_attr_init(&attr);
		if( m_thStackSize != 0 )
		{
			pthread_attr_setstacksize(&attr,m_thStackSize);
		}

		int error = pthread_create(&m_thHandle,&attr,(THREADFUNCPTR)&clListHandlerCoreThread::thStartRoutine,(VOIDPTR)this);
		
		if( error != 0 )
		{
			setThreadState(enmThStateFault);
			ETG_TRACE_USR4(("clListHandlerCoreThread::Thread creation failed!!"));
			return FALSE;	
		}
		
/****************** Thread real time priority setting********************
      // struct sched_param is used to store the scheduling priority
     struct sched_param params;
 
     // We'll set the priority to the maximum.
     params.sched_priority = sched_get_priority_max(SCHED_FIFO);
	 
	   // Attempt to set thread real-time priority to the SCHED_FIFO policy
     ret = pthread_setschedparam(m_thHandle, SCHED_FIFO, &params);
     if (ret != 0) {
         ETG_TRACE_FATAL(("Unsuccessful in setting thread realtime priority!!"));
      } 
****************************No tested so far, so commented out**********/
	}
	catch (char *s)
	{
		exit(-1); //to be commented out when we enable sd_notify
		//sd_notify(0, "ERRNO=2"); // No tested so far, so commented out.
	}
	return TRUE;
}

/************************************************************************
*FUNCTION: 		checkThreadRunStatus
*DESCRIPTION:   In case if any msg comes before thread creation, this function 
*				waits for some time for thread to get created.
*				This function will be introduced on need basis
*PARAMETER:		LONG
*RETURNVALUE: 	bool
*History:		Vasuki T (RBEI/ECG5)   Initial version
*************************************************************************/
/* bool clListHandlerCoreThread::checkThreadRunStatus(LONG timeout )
{
	LONG totalTime = 0;
    ETG_TRACE_USR4((" clListHandlerCoreThread::checkThreadRunStatus %d",getThreadRunStatus()));
	while(TRUE)
	{
		if( totalTime > timeout && timeout > 0 )
			return FALSE;

		pthread_mutex_lock(&m_thMutexLock);
		if( getThreadRunStatus() )
		{
			pthread_mutex_unlock(&m_thMutexLock);
			return TRUE;
		}
		totalTime += m_thStatusTime;

		pthread_mutex_unlock(&m_thMutexLock);
		Sleep(m_thStatusTime);
	}
	return FALSE;
} */

