/************************************************************************
*FILE: 			ListHandlerLogic.cpp
*SW-COMPONENT:
*DESCRIPTION:   This is list handler used to provide mediaplayer list data slices to HMI
*AUTHOR: 		Deepak Babu K
*COPYRIGHT: 	(c) 2020 - 2021 Robert Bosch Car Multimedia GmbH
*HISTORY:
*06.10.2020 Rev. 0.2 Deepak Babu K
*			bRequestDBDownload function added 
*28.07.2020 Rev. 0.1 Deepak Babu K
* 			Initial Revision.
************************************************************************/

/*****************************************************************
| includes
|----------------------------------------------------------------*/

#include "ListHandlerLogic.h" // For defining the member functions
#include "../../DBManager/inc/DAO/SelectQueryDAO.h"
#include "AvdeccClientHandler.h"
#include "../../DbusComponent/EMPServer/AsfEmpListHmiServiceAdaptor.h"
#include "bosch/cm/di/HmiEmpListInterface.h"
#include <map>
#include <vector>
#include <algorithm>
#include <random>

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"     // interface to the trace component

/* Header files required for Tracing out log statements */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "EMPListTraceConfig.h" // For using the Trace Class ID
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_LISTHANDLER_LOGIC
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_NAVRES1
#include "trcGenProj/Header/ListHandlerLogic.cpp.trc.h"
#endif

/*****************************************************************
| function implementation (scope: global)
|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION:      tclListHandlerLogic
*DESCRIPTION:   Constructor. Initializes the member variables
*PARAMETER:     None
*RETURNVALUE:   NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
***********************************************************************/

tclListHandlerLogic::tclListHandlerLogic()
{
	ETG_TRACE_USR4(("tclListHandlerLogic::Constructor "));	


	m_enRequestedListSourceType = EnumConst::entSourceType_DEFAULT;
	m_u8RequestedListRegion =0xFF;
	m_enRequestedListType = EnumConst::entListType_Invalid;
	m_u16count =0;
	m_u16CurrTrackIndex = 0;
    m_enCurrentTrackFileType = EnumConst::entFileType_unknown;
    m_sFolderPath.clear();
	m_enCurrListType = EnumConst::entListType_Video;
    vClearShuffleList();
    vClearFirstAndLastTrackIndex();
}

/************************************************************************
*FUNCTION: 		 ~tclListHandlerLogic
*DESCRIPTION: 	 Destructor. Clears the memory allocated
*PARAMETER:		 None
*RETURNVALUE: 	 NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/

tclListHandlerLogic::~tclListHandlerLogic()
{
	ETG_TRACE_USR4(("tclListHandlerLogic : Destructor"));
	vclearMap();
	m_EMPList.clear();
	m_enRequestedListSourceType = EnumConst::entSourceType_DEFAULT;
	m_u8RequestedListRegion =0xFF;
    m_sFolderPath.clear();
	m_enRequestedListType = EnumConst::entListType_Invalid;
	m_u16count =0;
	m_u16CurrTrackIndex = 0;
    m_enCurrentTrackFileType = EnumConst::entFileType_unknown;
	// if (NULL != poListHandlerLogic_)
	// {
	   // delete poListHandlerLogic_;
	   // poListHandlerLogic_ = NULL;
	// }
}

/************************************************************************
*FUNCTION: 		 vRequestCreateMediaPlayerList
*DESCRIPTION: 	 Request for downloading EMP Database and create given list type
*PARAMETER:		 entListType,entSourceType,Region,AbsoluteIndex
*RETURNVALUE: 	 NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vRequestCreateMediaPlayerList(const EnumConst::entListType eListType,
													const EnumConst::entSourceType eSourceType,
													const uint8_t u8Region,
													std::string sFolderPath)
{
	ETG_TRACE_USR4(("vRequestCreateMediaPlayerList , eListType Should never be called " ));
}

/************************************************************************
*FUNCTION: 		 vUpdateMediaPlayerListCount
*DESCRIPTION: 	 Function used by BDMgr to update the list count
*			 	 Response to vRequestCreateMediaPlayerList
*PARAMETER:		 entDownloadState
*RETURNVALUE: 	 NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vUpdateMediaPlayerListCount(const EnumConst::entListType eListType)
{
	ETG_TRACE_USR4(("tclListHandlerLogic::vUpdateMediaPlayerListCount"));
    ETG_TRACE_USR4(("vUpdateMediaPlayerListCount , eListType : %d ",eListType ));
	// Get the List instance to update the size
	typename listHandlerTypes::mEMPDataMap::iterator listIterator = m_EMPList.find(eListType);
	if(listIterator != m_EMPList.end())
	{
		tclListHandlerInterface* poListInstance  = listIterator->second;
		poListInstance->vSetTotalListSize(m_u16count);
		ETG_TRACE_USR4(("List Created  , List Size : %d",poListInstance->u16GetTotalListSize() ));
		vUpdateListCount(poListInstance->u16GetTotalListSize(),poListInstance->eGetListType());
	}
    else
    {
        ETG_TRACE_USR4(("tclListHandlerLogic::vUpdateMediaPlayerListCount list not created"));
        vUpdateListCount(0, m_enRequestedListType);
    }

}

/************************************************************************
*FUNCTION: 		 vUpdateListCount
*DESCRIPTION: 	 Function usedto update the list count to HMI
*PARAMETER:		 entDownloadState
*RETURNVALUE: 	 NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vUpdateListCount(const uint16_t u16count,const EnumConst::entListType eListType)
 {
	ETG_TRACE_USR4(("tclListHandlerLogic::vUpdateListCount"));

	EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor *poHMIServiceAdaptor = EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor::getInstance();
	
	if(nullptr != poHMIServiceAdaptor)
	{
		poHMIServiceAdaptor->setListSize(u16count,m_enRequestedListType, \
                    m_enRequestedListSourceType, m_u8RequestedListRegion);
	}
 }
 
/************************************************************************
*FUNCTION: 		 vGetListCount
*DESCRIPTION: 	 
*PARAMETER:		 ListType, source type, folderpath
*RETURNVALUE: 	 NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vGetListCount(EnumConst::entListType elListType, EnumConst::entSourceType srcType, std::string sFolderPath)
{
	ETG_TRACE_USR4(("tclListHandlerLogic::vGetListCount "));
	SelectQueryDAO *poSelectQueryDAO = SelectQueryDAO::getInstance();
	sRequestCount sRequestListCount;
	sRequestListCount.setType((eListType)elListType);
	sRequestListCount.setSourceType((eSourceType)srcType);
	if(elListType == EnumConst::entListType_Folder)
    {
        sRequestListCount.setFolderPath(sFolderPath);
    }
	EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor *poHMIServiceAdaptor = EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor::getInstance();
    if(nullptr != poHMIServiceAdaptor)
	{
		Langlist vecSelectedLanguages = poHMIServiceAdaptor->vecGetSelectedLanguages(srcType);
        sRequestListCount.setLang(vecSelectedLanguages);
	}
	bool bRet =  false;
    m_u16count = 0;
	if(nullptr != poSelectQueryDAO)
	{
		ETG_TRACE_USR4(("Request GetList Count for list type %d , " , sRequestListCount.getType()));
		bRet = poSelectQueryDAO->getListCount(sRequestListCount,m_u16count);	
	}
	if(bRet)
	{
		ETG_TRACE_USR4(("Request GetList Count for list type %d , and list type :  %d  " , sRequestListCount.getType(),m_u16count));
		vUpdateMediaPlayerListCount(elListType);
	}
	else if(!bRet)
	{
		ETG_TRACE_USR4(("Request GetList Count for list type %d " , m_u16count));
		m_u16count = 0;
        vUpdateListCount(0, m_enRequestedListType);
	}
}

/************************************************************************
*FUNCTION: 		 vCreateListhandlerInstance
*DESCRIPTION:	 this method used to crete new ListHandler interface
*DESCRIPTION:	 when HMI request to create new  list type
*PARAMETER:		 NA
*RETURNVALUE: 	 NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
bool tclListHandlerLogic::vCreateListhandlerInstance(const EnumConst::entListType eNewListType)
{
	ETG_TRACE_USR4(("tclListHandlerLogic::vCreateListhandlerInstance "));
	bool bRetVal = FALSE;
	tclListHandlerInterface* poListInstance = new tclListHandlerInterface(eNewListType);
	if(NULL != poListInstance)
	{
		ETG_TRACE_USR4(("Request GetList Count for list type %d , " , eNewListType));
		poListInstance->vSetTotalListSize(ZERO);
		poListInstance->vSetListType(eNewListType);
		m_EMPList.insert( std::pair<EnumConst::entListType,tclListHandlerInterface *> (eNewListType,poListInstance));
		bRetVal = TRUE;
	}
	else
		bRetVal = TRUE;

	return bRetVal;

}

/************************************************************************
*FUNCTION: 		vRequestMediaPlayerListSlice
*DESCRIPTION:	HMI request for List slices
*PARAMETER:		entListType,entSourceType,Region,StartingIndex,WindowSize
*RETURNVALUE: 	NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vRequestMediaPlayerListSlice(stWindowSliceRequest listData)
{
	ETG_TRACE_USR4(("tclListHandlerLogic::vRequestMediaPlayerListSlice() Should never be called"));
}

/************************************************************************
*FUNCTION: 		 vUpdateMediaPlayerListSlice
*DESCRIPTION:
*PARAMETER:		 dbManagerTypes::ItemList, const EnumConst::entSourceType
*                EnumConst::entListType enListType
*RETURNVALUE: 	 NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vUpdateMediaPlayerListSlice(dbManagerTypes::ItemList vSliceList, const EnumConst::entSourceType eSourceType, EnumConst::entListType enListType)
{
	ETG_TRACE_USR4(("tclListHandlerLogic:vUpdateMediaPlayerListSlice"));
	/*Update the slice list to HMI */

	EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor *poHMIServiceAdaptor = EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor::getInstance();
	if(nullptr != poHMIServiceAdaptor)
	{
		poHMIServiceAdaptor->UpdateMediaPlayerListSliceResponse(vSliceList, eSourceType, enListType);
	}
}

/************************************************************************
*FUNCTION: 		 vHandleClearListData
*DESCRIPTION:
*PARAMETER:		 eListType ,eSourceType, u8Region
*RETURNVALUE: 	 NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vHandleClearListData(const EnumConst::entListType eListType,const EnumConst::entSourceType eSourceType,
													const uint8_t u8Region)
{
	
	ETG_TRACE_USR4(("vHandleClearListData vHandleClearListData , list type %d , RequestedList type %d ",m_enRequestedListType,eListType));
	if(m_enRequestedListSourceType!=eSourceType || m_u8RequestedListRegion!=u8Region || static_cast<int>(entListType_Invalid) !=eListType )
	{
		typename listHandlerTypes::mEMPDataMap::iterator listIterator = m_EMPList.find(m_enRequestedListType);
		if(listIterator != m_EMPList.end())
		{
			tclListHandlerInterface* poListInstance  = listIterator->second;
			ETG_TRACE_USR4(("vHandleClearListData m_EMPList size = %d List Size : %d ",m_EMPList.size(),poListInstance->u16GetTotalListSize()));
			poListInstance->vClearListData();
			vclearMap();
			m_EMPList.clear();
		}
		else
		{
			ETG_TRACE_USR4(("listIterator reahced EOL"));
		}
		
	}
	else	
	{
		ETG_TRACE_USR4((" Invalid list type request "));
	}
}

/************************************************************************
*FUNCTION: 		 vHandleClearListData
*DESCRIPTION:	 	 overloaded vHandleClearListData to handle USB Disconnection
*PARAMETER:		 void
*RETURNVALUE: 	 void
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vHandleClearListData()
{
	ETG_TRACE_USR4(("vHandleClearListData on USBDisconnection "));
	
	typename listHandlerTypes::mEMPDataMap::iterator listIterator = m_EMPList.find(m_enRequestedListType);
	if(listIterator != m_EMPList.end())
	{
		tclListHandlerInterface* poListInstance  = listIterator->second;
		poListInstance->vClearListData();
	}
	vclearMap();
	m_EMPList.clear();
	m_enRequestedListSourceType = EnumConst::entSourceType_DEFAULT;
	m_u8RequestedListRegion =0xFF;
	m_enRequestedListType = EnumConst::entListType_Invalid;
	m_u16count =0;
	ETG_TRACE_USR4(("vHandleClearListData on USBDisconnection cleared all member variables m_EMPList size = %d",m_EMPList.size()));
}

/************************************************************************
*FUNCTION: 		 vCurrentTrackIndexUpdate
*DESCRIPTION:	 Request for Next/Previous song 
*PARAMETER:		 u8TrackIndexAction
*RETURNVALUE: 	 void
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vCurrentTrackIndexUpdate(uint16_t u16CurrentTrackIndex, EnumConst::entFileType enFileType)
{
	ETG_TRACE_USR4(("tclListHandlerLogic vCurrentTrackIndexUpdate  CurrentTrackIndex %d , m_u16CurrTrackIndex %d",u16CurrentTrackIndex,m_u16CurrTrackIndex));
	m_u16CurrTrackIndex = u16CurrentTrackIndex;
    m_enCurrentTrackFileType = enFileType;
}

/************************************************************************
*FUNCTION: 		 vRequestTrackIndex
*DESCRIPTION:	 Request for Next/Previous song 
*PARAMETER:		 u8TrackIndexAction
*RETURNVALUE: 	 void
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vRequestTrackIndex(uint8_t u8TrackIndexAction, uint16_t srcType, \
                            EnumConst::entListType enListType, std::string sFolderPath, uint8_t u8_Region )
{
	ETG_TRACE_USR4(("tclListHandlerLogic::vRequestTrackIndex() TrackIndexAction = %d , CurrTrackIndex = %d srcType = %d ",u8TrackIndexAction, m_u16CurrTrackIndex, srcType));
    m_sFolderPath.clear();
    if(!sFolderPath.empty())
    {
        m_sFolderPath = sFolderPath;
        ETG_TRACE_USR4(("tclListHandlerLogic::vRequestTrackIndex() Folderpath: %s", m_sFolderPath.c_str()));
    }
    vSetRegion(u8_Region);
    if(u8TrackIndexAction == MEDIA_CURRENTTRACKINDEXREQ)
	{	//media play request to fetch current song ID
        ResponseSongInfo songInfo;
        songInfo.setUpdatedSongInfo(m_u16CurrTrackIndex);
        songInfo.setFileType(static_cast<eFileType>(m_enCurrentTrackFileType));
		vUpdateTrackIndex(true, songInfo, srcType, enListType);//true, not fetching from DB
	}
    else if(u8TrackIndexAction == MEDIA_FIRSTTRACKINDEXREQ)
    {
		m_enCurrListType = enListType;
		vSendFirstTrackIndex(srcType);
    }
    else if((u8TrackIndexAction == MEDIA_NEXTTRACKINDEXREQ || u8TrackIndexAction == MEDIA_PREVTRACKINDEXREQ))
	{	
        vSendNextPrevTrackIndex(u8TrackIndexAction, srcType, enListType);
	}
    else if(u8TrackIndexAction == MEDIA_SELECTTRACKINDEXREQ)
    {
        //vCurrentListTypeUpdate(enListType);
    }
}

/************************************************************************
*FUNCTION      : 		vSendFirstTrackIndex
*DESCRIPTION   :	    Fetch the first song in the current list and update track index
*PARAMETER     :	    sourcetype
*RETURNVALUE   : 	    void
*HISTORY       :
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vSendFirstTrackIndex(uint16_t srcType)
{
    ETG_TRACE_USR4(("tclListHandlerLogic::vSendFirstTrackIndex()"));
	// request first Song ID
	SelectQueryDAO *poSelectQueryDAO = SelectQueryDAO::getInstance();
	bool bRet =  false;
	ResponseSongInfo sCurrTrackIndexInfo;

    sRequestFrontRearSong sReqFrontRearSong;
    sReqFrontRearSong.setType(static_cast<eListType>(m_enCurrListType));
    sReqFrontRearSong.setSourceType(static_cast<eSourceType>(srcType));
    if(m_enCurrListType == EnumConst::entListType_Folder)
    {
        sReqFrontRearSong.setFolderPath(m_sFolderPath);
    }
	EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor *poHMIServiceAdaptor = EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor::getInstance();
    if(nullptr != poHMIServiceAdaptor)
	{
		Langlist vecSelectedLanguages = poHMIServiceAdaptor->vecGetSelectedLanguages(static_cast<EnumConst::entSourceType>(srcType));
        sReqFrontRearSong.setLang(vecSelectedLanguages);
	}
	if(nullptr != poSelectQueryDAO)
	{       
		ETG_TRACE_USR4(("tclListHandlerLogic getFirstSongId"));
		bRet = poSelectQueryDAO->getFirstSongId(sReqFrontRearSong, sCurrTrackIndexInfo);	
	}
	vUpdateTrackIndex(bRet,sCurrTrackIndexInfo, srcType, static_cast<EnumConst::entListType>(m_enCurrListType));
}

/************************************************************************
*FUNCTION      : 		vSendNextPrevTrackIndex
*DESCRIPTION   :	    Fetch the first song in the current list and update track index
*PARAMETER     :	    NA
*RETURNVALUE   : 	    void
*HISTORY       :
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vSendNextPrevTrackIndex(uint8_t u8TrackIndexAction, uint16_t srcType, EnumConst::entListType enListType)
{
    ETG_TRACE_USR4(("tclListHandlerLogic::vSendNextPrevTrackIndex()"));
	bool bRet =  false;
	ResponseSongInfo sNewTrackInfo;
    EnumConst::enPlayback_Modes enPlayMode;
	//request for next & prev song ID
    
    //Read playback mode
	EMP_ListHandler::DbusComponent::EMPServer::AvdeccClientHandler* poAvdeccClientHandler= EMP_ListHandler::DbusComponent::EMPServer::AvdeccClientHandler::poGetInstance();
	if(nullptr !=poAvdeccClientHandler)
	{
	    enPlayMode = poAvdeccClientHandler->enGetPlayBackMode(enDeduceCurrentSubSourceType(static_cast <EnumConst::entSourceType>(srcType)));
        ETG_TRACE_USR4(("tclListHandlerLogic::vSendNextPrevTrackIndex()  PlayBackMode = %d", enPlayMode));

	}

    if( (enPlayMode == EnumConst::Repeat_Off) || (enPlayMode == EnumConst::Repeat_List) || (enPlayMode == EnumConst::Repeat_Track) )
    {
        
        if((enPlayMode != EnumConst::Repeat_Off) && (m_u32RepeatListLastTrackindex == TRACKINDEX_UNAVAILABLE) && (m_u32RepeatListFirstTrackindex == TRACKINDEX_UNAVAILABLE))
        {
            vFetchFirstAndLastTrackIndex(srcType); //m_u32RepeatListLastTrackindex and m_u32RepeatListFirstTrackindex is updated
        }
        else if(enPlayMode == EnumConst::Repeat_Off)
        {
            vClearFirstAndLastTrackIndex();
        }

	if( ((u8TrackIndexAction == MEDIA_PREVTRACKINDEXREQ) && ((m_u16CurrTrackIndex != m_u32RepeatListFirstTrackindex) || (m_enCurrentTrackFileType != m_u32RepeatListFirstTrackindexFileType)))
            || ((u8TrackIndexAction == MEDIA_NEXTTRACKINDEXREQ) && ((m_u16CurrTrackIndex != m_u32RepeatListLastTrackindex) || (m_enCurrentTrackFileType != m_u32RepeatListLastTrackindexFileType))) )
        {
            SelectQueryDAO *poSelectQueryDAO = SelectQueryDAO::getInstance();
            sRequestSongId sRequestNextPrevSongId;
            sRequestNextPrevSongId.setCurrentSongId(m_u16CurrTrackIndex);
            sRequestNextPrevSongId.setType(static_cast<eListType>(m_enCurrListType));
            sRequestNextPrevSongId.setDirection(static_cast<Direction>(u8TrackIndexAction));
            sRequestNextPrevSongId.setSourceType(static_cast<eSourceType>(srcType));
            sRequestNextPrevSongId.setFileType(static_cast<eFileType>(m_enCurrentTrackFileType));
            if(m_enCurrListType == EnumConst::entListType_Folder)
            {
                sRequestNextPrevSongId.setFolderPath(m_sFolderPath);
            }
          	EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor *poHMIServiceAdaptor = EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor::getInstance();
            if(nullptr != poHMIServiceAdaptor)
            {
                Langlist vecSelectedLanguages = poHMIServiceAdaptor->vecGetSelectedLanguages(static_cast<EnumConst::entSourceType>(srcType));
                sRequestNextPrevSongId.setLang(vecSelectedLanguages);
            }
            if(nullptr != poSelectQueryDAO)
            {       
                ETG_TRACE_USR4(("tclListHandlerLogic::vSendNextPrevTrackIndex() fetch NextPreviousSongId from db"));
                bRet = poSelectQueryDAO->getNextPreviousSongId(sRequestNextPrevSongId, sNewTrackInfo);	
            }
        }
        else if((u8TrackIndexAction == MEDIA_NEXTTRACKINDEXREQ) && (m_u16CurrTrackIndex == m_u32RepeatListLastTrackindex))
        {
            ETG_TRACE_USR4(("tclListHandlerLogic::vSendNextPrevTrackIndex() in lastTrack and  action is next, Sending first song"));
            bRet = true; //already fetched from DB
            sNewTrackInfo.setUpdatedSongInfo(m_u32RepeatListFirstTrackindex);
            sNewTrackInfo.setFileType(static_cast<eFileType>(m_u32RepeatListFirstTrackindexFileType));
        }
        else if((u8TrackIndexAction == MEDIA_PREVTRACKINDEXREQ) && (m_u16CurrTrackIndex == m_u32RepeatListFirstTrackindex))
        {
            ETG_TRACE_USR4(("tclListHandlerLogic::vSendNextPrevTrackIndex() in lastTrack and  action is next, Sending first song"));
            bRet = true; //already fetched from DB
            sNewTrackInfo.setUpdatedSongInfo(m_u32RepeatListLastTrackindex);
            sNewTrackInfo.setFileType(static_cast<eFileType>(m_u32RepeatListLastTrackindexFileType));        }
    }
    else if(enPlayMode == EnumConst::Shuffle_List)
    {
        if(m_Curr_List.size() == 0)
        {
            bRet = vCreateShuffleList(m_enCurrListType, srcType);
        }
        else
        {
            ETG_TRACE_USR4(("tclListHandlerLogic::vSendNextPrevTrackIndex() Shuffled list is available"));
            bRet = true;
        }
		if(bRet)
		{
            bRet = bGetNextPreviousTrackIndexFromShuffledList(u8TrackIndexAction, m_u16CurrTrackIndex, sNewTrackInfo);
		}
        else
        {
            ETG_TRACE_USR4(("tclListHandlerLogic::vSendNextPrevTrackIndex() failed to fetch track"));
        }
    }

	vUpdateTrackIndex(bRet, sNewTrackInfo, srcType, enListType);
}

/************************************************************************
*FUNCTION: 		 vUpdateTrackIndex
*DESCRIPTION:	 
*PARAMETER:		 isAvailable, songInfo
*RETURNVALUE: 	 void
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vUpdateTrackIndex(bool isAvailable , ResponseSongInfo songInfo, uint16_t srcType, EnumConst::entListType enListType)
{
	ETG_TRACE_USR4(("tclListHandlerLogic::vUpdateTrackIndex() "));
	// if DB manager provided the requested track index
	ETG_TRACE_USR4(("tclListHandlerLogic::vUpdateTrackIndex TrackIndex =  %d ", songInfo.getUpdatedSongId()));
	ETG_TRACE_USR4(("tclListHandlerLogic::vUpdateTrackIndex isAvailable =  %d ", isAvailable));
	ETG_TRACE_USR4(("tclListHandlerLogic::vUpdateTrackIndex FileType =  %d ", songInfo.getFileType()));
	EMP_ListHandler::DbusComponent::EMPServer::AvdeccClientHandler* poAvdeccClientHandler= EMP_ListHandler::DbusComponent::EMPServer::AvdeccClientHandler::poGetInstance();

    if(nullptr !=poAvdeccClientHandler)
	{
	    ETG_TRACE_USR4(("tclListHandlerLogic::vUpdateTrackIndex() call vsetListAbsoluteTrackIndex()"));
	    poAvdeccClientHandler->vsetListAbsoluteTrackIndex(isAvailable, songInfo.getUpdatedSongId(), \
                                static_cast<uint16_t>(srcType), static_cast<uint16_t>(songInfo.getFileType()));
	}
	if(!isAvailable)
	{
		ETG_TRACE_USR4(("tclListHandlerLogic::vUpdateTrackIndex() failed to fetch song"));	
	}
}



/************************************************************************
*FUNCTION: 		 vCurrentListTypeUpdate
*DESCRIPTION:    update list type of current playing song 
*PARAMETER:		 enCurrListType
*RETURNVALUE: 	 void
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vCurrentListTypeUpdate(EnumConst::entListType enCurrListType)
{
	ETG_TRACE_USR4(("tclListHandlerLogic::vCurrentListTypeUpdate()  new list type %d , existing list type %d",enCurrListType,m_enCurrListType));
	if(m_enCurrListType != enCurrListType)
    {
        ETG_TRACE_USR4(("tclListHandlerLogic::vCurrentListTypeUpdate() Clearing Shuffled list and repeatList Indexes"));
        vClearShuffleList();
		vClearFirstAndLastTrackIndex();
    }
    m_enCurrListType = enCurrListType;
}

/************************************************************************
*FUNCTION: 		 getListCount
*DESCRIPTION:    get getListCount // used in unit test now
*PARAMETER:		 
*RETURNVALUE: 	 uint16_t
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
uint16_t tclListHandlerLogic::getListCount()
{
	ETG_TRACE_USR4(("tclListHandlerLogic::getListCount()"));
    return m_u16count;
}

/************************************************************************
*FUNCTION: 		 enGetCurrentListType
*DESCRIPTION:	 returns current list type
*PARAMETER:		 None
*RETURNVALUE: 	 EnumConst::entListType
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
EnumConst::entListType tclListHandlerLogic::enGetCurrentListType()
{
	ETG_TRACE_USR4(("tclListHandlerLogic::enGetCurrentListType CurrListType: %d", m_enCurrListType));
	return m_enCurrListType;
}

/************************************************************************
*FUNCTION: 		 vFetchFirstAndLastTrackIndex
*DESCRIPTION:	 Fetch the first and last track in the current list.
*PARAMETER:		 None
*RETURNVALUE:
*HISTORY:        Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vFetchFirstAndLastTrackIndex(uint16_t SrcType)
{
    ETG_TRACE_USR4(("tclListHandlerLogic::vFetchFirstAndLastTrackIndex "));
    sRequestFrontRearSong sRequest_lastTrackIndex;
    SelectQueryDAO *poSelectQueryDAO = SelectQueryDAO::getInstance();
	bool bRet =  false;
    ResponseSongInfo sFirstsongInfo;
    ResponseSongInfo sLastsongInfo;
    if(nullptr != poSelectQueryDAO)
	{
        sRequest_lastTrackIndex.setType(static_cast <eListType>(enGetCurrentListType()));
        sRequest_lastTrackIndex.setSourceType(static_cast <eSourceType>(SrcType));
        if(enGetCurrentListType() == EnumConst::entListType_Folder)
        {
            sRequest_lastTrackIndex.setFolderPath(m_sFolderPath);
        }
        EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor *poHMIServiceAdaptor = EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor::getInstance();
        Langlist vecSelectedLanguages;
        if(nullptr != poHMIServiceAdaptor)
        {
            vecSelectedLanguages = poHMIServiceAdaptor->vecGetSelectedLanguages(static_cast <EnumConst::entSourceType>(SrcType));
            sRequest_lastTrackIndex.setLang(vecSelectedLanguages);
        }
        ETG_TRACE_USR4(("tclListHandlerLogic::vFetchFirstAndLastTrackIndex() Requesting to DB"));
		bRet = poSelectQueryDAO->getLastSongId(sRequest_lastTrackIndex, sLastsongInfo);
        if(bRet)
        {
            m_u32RepeatListLastTrackindex = sLastsongInfo.getUpdatedSongId();
            m_u32RepeatListLastTrackindexFileType = static_cast<EnumConst::entFileType>(sLastsongInfo.getFileType());
		    ETG_TRACE_USR4(("tclListHandlerLogic::vFetchFirstAndLastTrackIndex() LastSongID : %d \n", m_u32RepeatListLastTrackindex));
		    ETG_TRACE_USR4(("tclListHandlerLogic::vFetchFirstAndLastTrackIndex() LastSongFileType : %d \n", m_u32RepeatListLastTrackindexFileType));
        }
		else
		{
		    ETG_TRACE_USR4(("tclListHandlerLogic::vFetchFirstAndLastTrackIndex Failed to fetch last song"));
		}
		bRet =  false;
        sRequestFrontRearSong sReqFrontRearSong;
        sReqFrontRearSong.setType(static_cast<eListType>(m_enCurrListType));
        sReqFrontRearSong.setSourceType(static_cast<eSourceType>(SrcType));
        sReqFrontRearSong.setLang(vecSelectedLanguages);
        if(m_enCurrListType == EnumConst::entListType_Folder)
        {
            sReqFrontRearSong.setFolderPath(m_sFolderPath);
        }
        bRet = poSelectQueryDAO->getFirstSongId(sReqFrontRearSong, sFirstsongInfo);	
        if(bRet)
        {
            m_u32RepeatListFirstTrackindex = sFirstsongInfo.getUpdatedSongId();
            m_u32RepeatListFirstTrackindexFileType = static_cast<EnumConst::entFileType>(sFirstsongInfo.getFileType());
		    ETG_TRACE_USR4(("tclListHandlerLogic::vFetchFirstAndLastTrackIndex() FirstsongId : %d \n", m_u32RepeatListFirstTrackindex));
		    ETG_TRACE_USR4(("tclListHandlerLogic::vFetchFirstAndLastTrackIndex() FirstsongFileType : %d \n", m_u32RepeatListFirstTrackindexFileType));
        }
		else
		{
		    ETG_TRACE_USR4(("tclListHandlerLogic::vFetchFirstAndLastTrackIndex Failed to fetch First song"));
		}
    }
}

/************************************************************************
*FUNCTION: 		 vClearFirstAndLastTrackIndex
*DESCRIPTION:	 clears member variables related to repeat list
*PARAMETER:		 None
*RETURNVALUE:
*HISTORY:        Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vClearFirstAndLastTrackIndex()
{
    ETG_TRACE_USR4(("tclListHandlerLogic::vClearFirstAndLastTrackIndex "));
    m_u32RepeatListLastTrackindex = TRACKINDEX_UNAVAILABLE;
    m_u32RepeatListFirstTrackindex = TRACKINDEX_UNAVAILABLE;
    m_u32RepeatListLastTrackindexFileType = EnumConst::entFileType_unknown;
    m_u32RepeatListFirstTrackindexFileType = EnumConst::entFileType_unknown;
}

/************************************************************************
*FUNCTION: 		 enGetRegion
*DESCRIPTION:	 returns Region
*PARAMETER:		 None
*RETURNVALUE: 	 uint8_t
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
uint8_t tclListHandlerLogic::u8GetRegion()
{
	ETG_TRACE_USR4(("tclListHandlerLogic::enGetRegion region: %d", m_u8Region));
	return m_u8Region;
}

/************************************************************************
*FUNCTION: 		 vSetRegion
*DESCRIPTION:	 Store Region
*PARAMETER:		 uint8_t
*RETURNVALUE: 	 void
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vSetRegion(uint8_t u8_Region)
{
	ETG_TRACE_USR4(("tclListHandlerLogic::vSetRegion prev region value: %d, current region value: %d", m_u8Region, u8_Region));
	m_u8Region = u8_Region;
}

/************************************************************************
*FUNCTION: 		 vclearMap
*DESCRIPTION: 	 to clear the pointers used in Map since STD::map clear() function wouldn't do it
*PARAMETER:		 None
*RETURNVALUE: 	 NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vclearMap()
{
     ETG_TRACE_USR4(("tclListHandlerLogic::vclearMap"));
	 for (  listHandlerTypes::mEMPDataMap::iterator it = m_EMPList.begin(); it != m_EMPList.end(); ++it ) { 
	   delete it->second;
	 }
}

/************************************************************************
*FUNCTION: 		 vCreateShuffleList
*DESCRIPTION:	 creates shuffled list of the current listType
*PARAMETER:		 current list type and source type
*RETURNVALUE:    bool
*HISTORY:        Rev 0.1 Initial Revision.
************************************************************************/
bool tclListHandlerLogic::vCreateShuffleList(EnumConst::entListType enCurrListType, uint16_t SrcType)
{
    ETG_TRACE_USR4(("tclListHandlerLogic::vCreateShuffleList()"));
    ReqListInfoMessage sRequestListInfoMessage;
	SelectQueryDAO *poSelectQueryDAO = SelectQueryDAO::getInstance();
	bool bRet =  false;
	if(nullptr != poSelectQueryDAO)
	{
        sRequestListInfoMessage.setType(static_cast <eListType> ( enCurrListType ));
        sRequestListInfoMessage.setSourceType(static_cast <eSourceType > ( SrcType ));
        if(enCurrListType == EnumConst::entListType_Folder)
        {
            sRequestListInfoMessage.setFolderPath(m_sFolderPath);
        }
        EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor *poHMIServiceAdaptor = EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor::getInstance();
        if(nullptr != poHMIServiceAdaptor)
        {
            Langlist vecSelectedLanguages = poHMIServiceAdaptor->vecGetSelectedLanguages(static_cast <EnumConst::entSourceType>(SrcType));
            sRequestListInfoMessage.setLang(vecSelectedLanguages);
        }
        ETG_TRACE_USR4(("tclListHandlerLogic::vCreateShuffleList() Requesting to DB"));
		bRet = poSelectQueryDAO->getListInfo((sRequestListInfoMessage), m_Curr_List);
		ETG_TRACE_USR4(("tclListHandlerLogic::vCreateShuffleList() Requested to DB - Request Status  : %d \n",bRet));
	}
    
	if(bRet)
	{
        ETG_TRACE_USR4(("tclListHandlerLogic::vCreateShuffleList() Shuffle Start"));
        std::random_device rd;
        std::mt19937 g(rd());
		std::shuffle(m_Curr_List.begin(), m_Curr_List.end(), g);
        for (sItemData i: m_Curr_List) //to print shuffled list , delete for release
        {
            ETG_TRACE_USR4(("tclListHandlerLogic::vCreateShuffleList() items of list: %d", i.getAbsoluteIndex() ));
        }
        ETG_TRACE_USR4(("tclListHandlerLogic::vCreateShuffleList() Shuffle End"));
	}
	return bRet;
}

/************************************************************************
*FUNCTION: 		 vClearShuffleList
*DESCRIPTION:	 clears shuffled list
*PARAMETER:		 None
*RETURNVALUE:    None
*HISTORY:        Rev 0.1 Initial Revision.
************************************************************************/
void tclListHandlerLogic::vClearShuffleList()
{
    ETG_TRACE_USR4(("tclListHandlerLogic::vClearShuffleList()"));
    m_Curr_List.clear();
}

/************************************************************************
*FUNCTION: 		 bGetNextPreviousTrackIndexFromShuffledList
*DESCRIPTION:	 based on trackIndexAction fetches the Next/Previous element
*                in the shuffled list
*PARAMETER:		 trackIndexAction, CurrentTrackIndex, sNewTrackInfo;
*RETURNVALUE:    bool(Success/failure)
*HISTORY:        Rev 0.1 Initial Revision.
************************************************************************/
bool tclListHandlerLogic::bGetNextPreviousTrackIndexFromShuffledList(uint8_t u8TrackIndexAction, \
                                    uint16_t m_u16CurrTrackIndex, ResponseSongInfo &sNewTrackInfo)
{
    ETG_TRACE_USR4(("tclListHandlerLogic::bGetNextPreviousTrackIndexFromShuffledList() BEGIN"));
    //Search for the m_u16CurrTrackIndex in m_Curr_List and based on the u8TrackIndexAction fill sNewTrackInfo
    //return false if m_Curr_List is empty.  
    sNewTrackInfo.setUpdatedSongInfo(TRACKINDEX_UNAVAILABLE);
    sNewTrackInfo.setFileType(static_cast<eFileType>(EnumConst::entFileType_unknown));
    ItemList::iterator itr;
    if( m_Curr_List.size() == 0 )
	{
		return false;
	}
	for(itr = m_Curr_List.begin() ; itr != m_Curr_List.end() ; itr++)
	{
		if(((*itr).absoluteIndex == m_u16CurrTrackIndex) && ((*itr).fileType = static_cast<eFileType>(m_enCurrentTrackFileType)))
		{
			if(u8TrackIndexAction == MEDIA_NEXTTRACKINDEXREQ)
			{
                ETG_TRACE_USR4(("tclListHandlerLogic::bGetNextPreviousTrackIndexFromShuffledList() Next"));
				if(itr == m_Curr_List.end()-1)
				{
					sNewTrackInfo.setUpdatedSongInfo((*(m_Curr_List.begin())).absoluteIndex);
					sNewTrackInfo.setFileType((*(m_Curr_List.begin())).fileType);
				}
				else 
				{
					sNewTrackInfo.setUpdatedSongInfo((*(itr+1)).absoluteIndex);
					sNewTrackInfo.setFileType((*(itr+1)).fileType);
				}
			}
			else if(u8TrackIndexAction == MEDIA_PREVTRACKINDEXREQ)
			{
                ETG_TRACE_USR4(("tclListHandlerLogic::bGetNextPreviousTrackIndexFromShuffledList() Previous"));
				if(itr == m_Curr_List.begin())
				{
					sNewTrackInfo.setUpdatedSongInfo((*(m_Curr_List.end()-1)).absoluteIndex);
					sNewTrackInfo.setFileType((*(m_Curr_List.end()-1)).fileType);
				}
				else 
				{
					sNewTrackInfo.setUpdatedSongInfo((*(itr-1)).absoluteIndex);
					sNewTrackInfo.setFileType((*(itr-1)).fileType);
				}
			}	
			else
			{
                ETG_TRACE_USR4(("tclListHandlerLogic::bGetNextPreviousTrackIndexFromShuffledList()Invalid action here"));
				return false;
			}
		}
	}
    if(sNewTrackInfo.getUpdatedSongId() != TRACKINDEX_UNAVAILABLE)
	{
        ETG_TRACE_USR4(("tclListHandlerLogic::bGetNextPreviousTrackIndexFromShuffledList() index = %d", sNewTrackInfo.getUpdatedSongId()));
        ETG_TRACE_USR4(("tclListHandlerLogic::bGetNextPreviousTrackIndexFromShuffledList() FileType = %d", sNewTrackInfo.getFileType()));
        return true;
	}
	return false;
}

/************************************************************************
*FUNCTION: 		 enDeduceCurrentSubSourceType
*DESCRIPTION:	 based on current list type and source, subsource is calculated
*PARAMETER:		 SourceType
*RETURNVALUE:    current SubSourceType
*HISTORY:        Rev 0.1 Initial Revision.
************************************************************************/
EnumConst::enSubSourceType tclListHandlerLogic::enDeduceCurrentSubSourceType(EnumConst::entSourceType srcType)
{
    ETG_TRACE_USR4(("tclListHandlerLogic::enDeduceCurrentSubSourceType()"));
    if(enGetCurrentListType() == EnumConst::entListType_Song)
    {
        if(srcType == EnumConst::entSourceType_CMP)
        {
            return EnumConst::enSubSourceType_CMP_AUDIO;
        }
        if(srcType == EnumConst::entSourceType_CMR)
        {
            return EnumConst::enSubSourceType_CMR_AUDIO;
        }
    }
    else
    {
        if(srcType == EnumConst::entSourceType_CMP)
        {
            return EnumConst::enSubSourceType_CMP_VIDEO;
        }
        if(srcType == EnumConst::entSourceType_CMR)
        {
            return EnumConst::enSubSourceType_CMR_VIDEO;
        }
     }
     return EnumConst::enSubSourceType_UNKNOWN;
}

/******************************************************************************
| EOF
|----------------------------------------------------------------------------*/
