/************************************************************************
*FILE: 			ListHandler.cpp
*SW-COMPONENT:
*DESCRIPTION:   This is list handler used to keep  list and its opetations
*AUTHOR: 		Deepak Babu K
*COPYRIGHT: 	(c) 2020 - 2021 Robert Bosch Car Multimedia GmbH
*HISTORY:
*23.12.2020 Rev. 0.2 Shraddha S K
*16.09.2020 Rev. 0.1 Deepak Babu K
* 				Initial Revision.
************************************************************************/

/*****************************************************************
| includes
|----------------------------------------------------------------*/

#include "ListHandler.h"
#include "../../DBManager/inc/DAO/SelectQueryDAO.h"
#include "../../DbusComponent/EMPServer/AsfEmpListHmiServiceAdaptor.h"
#include <algorithm>
#include <iostream>
#include <vector>

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"  

// interface to the trace component

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "EMPListTraceConfig.h" // For using the Trace Class ID
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_LISTHANDLER_LOGIC
#define ETG_I_TRACE_CHANNEL     TR_TTFIS_NAVRES1
#include "trcGenProj/Header/ListHandler.cpp.trc.h"
#endif

/*****************************************************************
| function implementation (scope: global)
|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION:      tclListHandlerInterface
*DESCRIPTION:   Constructor. Initializes the member variables
*PARAMETER:     None
*RETURNVALUE:   NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
***********************************************************************/
tclListHandlerInterface::tclListHandlerInterface(EnumConst::entListType u8ListType)
{
	ETG_TRACE_USR4(("tclListHandlerInterface : constructor "));
	m_u8WindowSize = DEFAULT_WINDOW_SIZE;
	m_u16TotalListSize = ZERO;
	m_u16CurrStartIndex = ZERO;
	m_u16PrevStartIndex = ZERO;
	m_eListType = EnumConst::entListType_Invalid;
	m_u16WindowIndex = ZERO;
	m_u16WindowIndexSent = ZERO;
	
}
/************************************************************************
*FUNCTION:      ~tclListHandlerInterface
*DESCRIPTION:   Destructor. 
*PARAMETER:     None
*RETURNVALUE:   NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
***********************************************************************/
tclListHandlerInterface::~tclListHandlerInterface()
{
	ETG_TRACE_USR4(("tclListHandlerInterface :Destructor "));
	m_u16TotalListSize = 0;
	m_u8WindowSize = 0;
	m_u16CurrStartIndex = 0;
	m_u16PrevStartIndex = 0;
	m_u16WindowIndex = 0;
	m_u16WindowIndexSent = 0;
	m_Curr_Data.clear();
	m_Prev_Data.clear();
	m_Next_Data.clear();
}



/************************************************************************
*FUNCTION:      vReqListSlice
*DESCRIPTION:   Handles slice request from HMI
*PARAMETER:     Start index , window size
*RETURNVALUE:   NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
***********************************************************************/
dbManagerTypes::ItemList tclListHandlerInterface::vReqListSlice(uint16_t u16StartIndex,uint16_t u8WindowSize, EnumConst::entSourceType SrcType, std::string sFolderpath)
{
	m_u16CurrStartIndex = u16StartIndex;
	m_u8WindowSize = u8WindowSize;
	ETG_TRACE_USR4(("tclListHandlerInterface : m_u16PrevStartIndex = %d u16StartIndex = %d , m_u8WindowSize =%d source type = %d", m_u16PrevStartIndex,u16StartIndex,m_u8WindowSize, SrcType));
	//On first request of slice, request current buffer, update slice as once current data is available,request for next slice and keep it ready for next requests.
	vRequestListBuffer(m_u16CurrStartIndex, m_u8WindowSize, SrcType, m_Curr_Data, sFolderpath); 
    // vUpdateListSlice(m_Curr_Data);
	m_u16PrevStartIndex = m_u16CurrStartIndex;
    return m_Curr_Data;
}

/************************************************************************
*FUNCTION:      u16GetRelativeStart()
*DESCRIPTION:   Handle Current, next, previous  buffers on Scroll up
*PARAMETER:     
*RETURNVALUE:   NA
*
*HISTORY:
*Rev 0.1 Initial Revision. 
***********************************************************************/
uint16_t tclListHandlerInterface::u16GetRelativeStart()
{	
	uint16_t u16RelativeStart = 0;
	//Relative start = startIndex for Song Index from 1-60
	if(m_u16WindowIndex <= TOTAL_SLICES ) 
		u16RelativeStart = m_u16CurrStartIndex;
	else
		u16RelativeStart = m_u16CurrStartIndex - ( m_u16WindowIndex - TOTAL_SLICES ) * DEFAULT_WINDOW_SIZE;

	ETG_TRACE_USR4(( "tclListHandlerInterface - vHandleSendData u16RelativeStart %d", u16RelativeStart ));
	
	return u16RelativeStart;
	
}

/************************************************************************
*FUNCTION:      vGetWindowIndexSent()
*DESCRIPTION:   Handle Current, next, previous  buffers on Scroll up
*PARAMETER:     
*RETURNVALUE:   NA
*
*HISTORY:
*Rev 0.1 Initial Revision. 
***********************************************************************/
void tclListHandlerInterface::vGetWindowIndexSent()
{
	//last songIndex sent to HMI
	uint16_t u16RelativeEnd = m_u16CurrStartIndex+m_u8WindowSize; 
	//check u16RelativeEnd belongs to which WindowIndex 
		for(int i = m_u16WindowIndex;i>0;i--) 
		{
			ETG_TRACE_USR4(( "tclListHandlerInterface - i =  %d",i ));
		
			if(u16RelativeEnd > DEFAULT_WINDOW_SIZE*i)
			{	
				m_u16WindowIndexSent = i+1;
				ETG_TRACE_USR4(( "tclListHandlerInterface - m_u16WindowIndexSent =  %d",m_u16WindowIndexSent));
				break;
			}
			else 
			{
				ETG_TRACE_USR4(( "tclListHandlerInterface u16RelativeEnd does not belongs to i = %d",i));
			}
		}
}




/************************************************************************
*FUNCTION:      vRequestListBuffer()
*DESCRIPTION:   Request List Slice from DB manager
*PARAMETER:     
*RETURNVALUE:   NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
***********************************************************************/
void tclListHandlerInterface::vRequestListBuffer(uint16_t u16StartIndex, uint16_t u8WindowSize , \
                EnumConst::entSourceType SrcType, dbManagerTypes::ItemList& vItemList, std::string sFolderpath)
{
	ETG_TRACE_USR4(("tclListHandlerInterface::vRequestListBuffer"));
		sRequestMessage stReqWindowMessgae;
		stReqWindowMessgae.setType((eListType)m_eListType);
		stReqWindowMessgae.setStartIndex(u16StartIndex);
		stReqWindowMessgae.setWindowSize(u8WindowSize);
		stReqWindowMessgae.setSourceType((eSourceType)SrcType);
        if(m_eListType == EnumConst::entListType_Folder)
        {
            stReqWindowMessgae.setFolderPath(sFolderpath);
        }
        EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor *poHMIServiceAdaptor = EMP_ListHandler::DbusComponent::EMPServer::clAsfEmpListHmiServiceAdaptor::getInstance();
        if(nullptr != poHMIServiceAdaptor)
        {
            Langlist vecSelectedLanguages = poHMIServiceAdaptor->vecGetSelectedLanguages(SrcType);
            stReqWindowMessgae.setLang(vecSelectedLanguages);
        }
		ETG_TRACE_USR4((" vRequestListBuffer type   : %d, start index : %d , windowSize : %d \n",stReqWindowMessgae.getType(),stReqWindowMessgae.getStartIndex(),stReqWindowMessgae.getWindowSize()));
		SelectQueryDAO *poSelectQueryDAO = SelectQueryDAO::getInstance();
		bool bRet =  false;
		if(nullptr != poSelectQueryDAO)
		{
			bRet = poSelectQueryDAO->getWindowListSliced((stReqWindowMessgae),vItemList);
			ETG_TRACE_USR4((" RRequestd to DB - Request Status  : %d \n",bRet));
		}
		else
		{
			ETG_TRACE_USR4((" RRequestd to DB - Request Status  : %d \n",bRet));
		}
}

/************************************************************************
*FUNCTION:      vClearListBuffers
*DESCRIPTION:	clear list buffers on request for new list type
*PARAMETER:     startindex : window size
*RETURNVALUE:   NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
***********************************************************************/
void tclListHandlerInterface::vClearListData()
{
	ETG_TRACE_USR4(("tclListHandlerLogic :vClearListBuffers "));
	m_u16TotalListSize = 0;
	m_u8WindowSize = 0;
	m_u16CurrStartIndex = 0;
	m_u16PrevStartIndex = 0;
	m_u16WindowIndex = 0;
	m_u16WindowIndexSent = 0;
	m_Curr_Data.clear();
	m_Prev_Data.clear();
	m_Next_Data.clear();
	m_eListType = EnumConst::entListType_Invalid;
		      

}

