/*
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2020
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 *
 * FILE                : QueryManager.cpp
 * COMPONENT Name      : EMP Query Module
 * DESCRIPTION         :
 * AUTHOR              : Monojit Chatterjee
 * Date                : 7.07.2020
 * Revision History    : 1.0
 * Date 7.07.2020     : Initial version
 *
*/

#include <QueryManager.h>
#include <iostream>
using namespace std;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "EMPListTraceConfig.h" // For using the Trace Class ID
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_LISTHANDLER_DBMANAGER
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_NAVRES1
#include "trcGenProj/Header/QueryManager.cpp.trc.h" // cpp filename.cpp.trc.h
#endif

ItemList QueryManager::mItemList {}  ;
int      QueryManager::count {};

QueryManager* QueryManager::instance = nullptr;


QueryMap< eListType , string > QueryListCountType =
{
    pair<eListType, string > ( entListType_Genre , "SELECT COUNT(*) FROM Genres"   ),
    pair<eListType, string > ( entListType_Artist,   "SELECT COUNT(*) FROM Artists" ),
    pair<eListType, string > ( entListType_Album,   "SELECT COUNT(*) FROM Albums" ),
    pair<eListType, string > ( entListType_Song,   "SELECT COUNT(*) FROM AudioFiles WHERE Playable=1" ),
    // pair<eListType, string > ( entListType_Artist_Album,   "" ),
    // pair<eListType, string > ( entListType_Artist_Album_Song,  "" ),
    pair<eListType, string > ( entListType_Artist_Song,  "SELECT COUNT(*) FROM AudioFiles WHERE IdArtist=?"),
    pair<eListType, string > ( entListType_Album_Song,  "SELECT COUNT(*) FROM AudioFiles WHERE IdAlbum=?" ),
    //pair<eListType, string > ( entListType_Genre_Album,  "" ),
    //pair<eListType, string > ( entListType_Genre_Album_Song,  "" ),
    pair<eListType, string > ( entListType_Genre_Song,  "SELECT COUNT(*) FROM AudioFiles WHERE IdGenre=?" ),
    pair<eListType, string > ( entListType_Video,  "SELECT COUNT(*) from VideoFiles  INNER JOIN Languages on VideoFiles.LangId = Languages.IdLanguage  where  ( Playable=1 ) AND ( LangId  in (" ),

    pair<eListType, string > ( entListType_Video_NoFilter,  "SELECT COUNT(*) FROM VideoFiles WHERE Playable=1" ),
    //pair<eListType, string > ( entListType_Video_episode,  "" ),
    //pair<eListType, string > ( entListType_Playlist,  "" ),
    //pair<eListType, string > ( entListType_Playlist_song,  "" ),
    pair<eListType, string > ( entListType_Folder, FOLDER_COUNT_REQUEST_QUERY  ),
    pair<eListType, string > ( entListType_Video_Languages,  "SELECT COUNT(*) FROM Languages" ),
    //pair<eListType, string > ( entListType_Folder_file,  "" ),
    pair<eListType, string > ( entListType_Invalid,  "" ),
};

QueryMap< eListType , string > QueryListType =
{
    pair<eListType, string > ( entListType_Genre , "SELECT Id,Genre FROM Genres LIMIT ?,?"   ),
    pair<eListType, string > ( entListType_Artist,   "SELECT Id,Artist FROM Artists LIMIT ?,?" ),
    pair<eListType, string > ( entListType_Album,   "SELECT Id,Album FROM Albums LIMIT ?,?" ),
    pair<eListType, string > ( entListType_Song,   "SELECT Id,Filename,2 as FileType FROM AudioFiles WHERE Playable=1 LIMIT ?,?" ),
    //  pair<eListType, string > ( entListType_Artist_Album,   "" ),
    //  pair<eListType, string > ( entListType_Artist_Album_Song,  "" ),
    pair<eListType, string > ( entListType_Artist_Song,  "SELECT Title FROM AudioFiles WHERE IdArtist=? LIMIT ?,?"),
    pair<eListType, string > ( entListType_Album_Song,  "SELECT Title FROM AudioFiles WHERE IdAlbum=? LIMIT ?,?" ),
    //pair<eListType, string > ( entListType_Genre_Album,  "" ),
    //pair<eListType, string > ( entListType_Genre_Album_Song,  "" ),
    pair<eListType, string > ( entListType_Genre_Song,  "SELECT Title FROM AudioFiles WHERE IdGenre=? LIMIT ?,?" ),
    pair<eListType, string > ( entListType_Video,  "SELECT Id,Filename,ISO,1 as FileType from VideoFiles INNER JOIN Languages on VideoFiles.LangId = Languages.IdLanguage  where ( Playable=1 ) AND ( LangId  in (" ),
    pair<eListType, string > ( entListType_Video_NoFilter,  "SELECT Id,Filename,ISO,1 as FileType from VideoFiles  INNER JOIN Languages on VideoFiles.LangId = Languages.IdLanguage  where   Playable=1 LIMIT ?,?" ),
    //pair<eListType, string > ( entListType_Video_episode,  "" ),
    //pair<eListType, string > ( entListType_Playlist,  "" ),
    //pair<eListType, string > ( entListType_Playlist_song,  "" ),
    pair<eListType, string > ( entListType_Folder,  FOLDER_SLICE_REQUEST_QUERY ),
    pair<eListType, string > ( entListType_Video_Languages,  "SELECT IdLanguage,Language FROM Languages LIMIT ?,?" ),
    //pair<eListType, string > ( entListType_Folder_file,  "" ),
    pair<eListType, string > ( entListType_Invalid,  "" ),
};

QueryMap< eListType , string > QueryGetFirstSongType =
{
    pair<eListType, string > ( entListType_Genre , "SELECT Id FROM Genres LIMIT 0,1"   ),
    pair<eListType, string > ( entListType_Artist,   "SELECT Id FROM Artists LIMIT 0,1" ),
    pair<eListType, string > ( entListType_Album,   "SELECT Id FROM Albums LIMIT 0,1" ),
    pair<eListType, string > ( entListType_Song,   "SELECT Id as IndexItem , 2 as IndexFileType FROM AudioFiles WHERE Playable=1 LIMIT 0,1" ),
    //  pair<eListType, string > ( entListType_Artist_Album,   "" ),
    //  pair<eListType, string > ( entListType_Artist_Album_Song,  "" ),
    pair<eListType, string > ( entListType_Artist_Song,  "SELECT Id as IndexItem , 2 as IndexFileType FROM AudioFiles WHERE IdArtist=? LIMIT 0,1"),
    pair<eListType, string > ( entListType_Album_Song,  "SELECT Id as IndexItem , 2 as IndexFileType FROM AudioFiles WHERE IdAlbum=? LIMIT 0,1" ),
    //pair<eListType, string > ( entListType_Genre_Album,  "" ),
    //pair<eListType, string > ( entListType_Genre_Album_Song,  "" ),
    pair<eListType, string > ( entListType_Genre_Song,  "SELECT Id as IndexItem , 2 as IndexFileType FROM AudioFiles WHERE IdGenre=? LIMIT 0,1" ),
    pair<eListType, string > ( entListType_Video_NoFilter,  "SELECT Id as IndexItem , 1 as IndexFileType FROM VideoFiles WHERE Playable=1 LIMIT 0,1" ),

    pair<eListType, string > ( entListType_Video,  "SELECT Id as IndexItem , 1 as IndexFileType from VideoFiles  INNER JOIN Languages on VideoFiles.LangId = Languages.IdLanguage where  ( Playable=1 ) AND ( LangId  in (" ),

    //pair<eListType, string > ( entListType_Video_episode,  "" ),
    //pair<eListType, string > ( entListType_Playlist,  "" ),
    //pair<eListType, string > ( entListType_Playlist_song,  "" ),
    pair<eListType, string > ( entListType_Folder,  FOLDER_FIRST_SONG ),
    //pair<eListType, string > ( entListType_Folder_file,  "" ),
    pair<eListType, string > ( entListType_Invalid,  "" ),
};

QueryMap< eListType , string > QueryGetLastSongType =
{
    pair<eListType, string > ( entListType_Genre , "SELECT Id FROM Genres LIMIT ?,1"   ),
    pair<eListType, string > ( entListType_Artist,   "SELECT Id FROM Artists LIMIT ?,1" ),
    pair<eListType, string > ( entListType_Album,   "SELECT Id FROM Albums LIMIT ?,1" ),
    pair<eListType, string > ( entListType_Song,   "SELECT Id as IndexItem , 2 as IndexFileType FROM AudioFiles WHERE Playable=1 LIMIT ?,1" ),
    //  pair<eListType, string > ( entListType_Artist_Album,   "" ),
    //  pair<eListType, string > ( entListType_Artist_Album_Song,  "" ),
    pair<eListType, string > ( entListType_Artist_Song,  "SELECT Id as IndexItem , 1 as IndexFileType FROM AudioFiles WHERE IdArtist=? LIMIT ?,1"),
    pair<eListType, string > ( entListType_Album_Song,  "SELECT Id as IndexItem , 2 as IndexFileType FROM AudioFiles WHERE IdAlbum=? LIMIT ?,1" ),
    //pair<eListType, string > ( entListType_Genre_Album,  "" ),
    //pair<eListType, string > ( entListType_Genre_Album_Song,  "" ),
    pair<eListType, string > ( entListType_Genre_Song,  "SELECT Id as IndexItem , 2 as IndexFileType FROM AudioFiles WHERE IdGenre=? LIMIT ?,1" ),
    pair<eListType, string > ( entListType_Video,  "SELECT Id as IndexItem , 1 as IndexFileType from VideoFiles  INNER JOIN Languages on VideoFiles.LangId = Languages.IdLanguage where  ( Playable=1 ) AND ( LangId  in (" ),
    pair<eListType, string > ( entListType_Video_NoFilter,  "SELECT Id as IndexItem , 1 as IndexFileType FROM VideoFiles WHERE Playable=1 LIMIT ?,1" ),
    //pair<eListType, string > ( entListType_Video_episode,  "" ),
    //pair<eListType, string > ( entListType_Playlist,  "" ),
    //pair<eListType, string > ( entListType_Playlist_song,  "" ),
    pair<eListType, string > ( entListType_Folder,  FOLDER_LAST_SONG ),
    //pair<eListType, string > ( entListType_Folder_file,  "" ),
    pair<eListType, string > ( entListType_Invalid,  "" ),
};

multimap < eListType , string > QueryNextPreviousType =
{

    pair<eListType, string > ( entListType_Song,  "SELECT Id as IndexItem , 2 as IndexFileType  FROM AudioFiles WHERE Id > ? AND Playable = 1 ORDER BY Id  LIMIT 1" ),

    pair<eListType, string > ( entListType_Song,  "SELECT Id as IndexItem , 2 as IndexFileType  FROM AudioFiles WHERE Id < ? AND Playable = 1 ORDER BY Id DESC  LIMIT 1" ),

    pair<eListType, string > ( entListType_Video_NoFilter,  "SELECT Id as IndexItem , 1 as IndexFileType  FROM VideoFiles WHERE Id > ? AND Playable = 1 ORDER BY Id  LIMIT 1" ),

    pair<eListType, string > ( entListType_Video_NoFilter,  "SELECT Id as IndexItem , 1 as IndexFileType  FROM VideoFiles WHERE Id < ? AND Playable = 1 ORDER BY Id DESC  LIMIT 1" ),

    pair<eListType, string > ( entListType_Video,  "SELECT Id as IndexItem , 1 as IndexFileType  FROM VideoFiles  INNER JOIN Languages on VideoFiles.LangId = Languages.IdLanguage where  ( Playable=1 AND Id > ? ) AND ( LangId  in (" ),

    pair<eListType, string > ( entListType_Video,  "SELECT Id as IndexItem , 1 as IndexFileType  FROM VideoFiles  INNER JOIN Languages on VideoFiles.LangId = Languages.IdLanguage where  ( Playable=1 AND Id < ? ) AND ( LangId  in (" ),

    pair<eListType, string > ( entListType_Folder, FOLDER_NEXT_QUERY  ),

    pair<eListType, string > ( entListType_Folder, FOLDER_PREVIOUS_QUERY   ),

    pair<eListType, string > ( entListType_Invalid,  "" ),
};

QueryMap< eListType , string > QueryCompleteList =
{

    pair<eListType, string > ( entListType_Song,  "SELECT Id,Filename FROM AudioFiles  WHERE Playable=1" ),
    pair<eListType, string > ( entListType_Video_NoFilter ,  "SELECT Id,Filename FROM VideoFiles WHERE Playable=1" ),
    pair<eListType, string > ( entListType_Genre,  "SELECT Id,Filename FROM Genres WHERE Playable=1" ),
    pair<eListType, string > ( entListType_Album,  "SELECT Id,Filename FROM Albums WHERE Playable=1" ),
    pair<eListType, string > ( entListType_Artist_Song,  "SELECT Id,Filename FROM AudioFiles WHERE IdArtist=? WHERE Playable=1" ),
    pair<eListType, string > ( entListType_Genre_Song,  "SELECT Id,Filename FROM AudioFiles WHERE IdGenre=? WHERE Playable=1" ),
    pair<eListType, string > ( entListType_Album_Song,  "SELECT Id,Filename FROM AudioFiles WHERE IdAlbum=? WHERE Playable=1" ),
    pair<eListType, string > ( entListType_Folder,  FOLDER_COMPLETE_LIST ),

};

QueryMap  < eColumnList , const char* > QueryColumnList =
{

    pair<eColumnList, const char* > ( eFILENAME,              "Filename" ),
    pair<eColumnList, const char* > ( eID,                    "Id"       ),
    pair<eColumnList, const char* > ( eISO,                   "ISO"      ),
    pair<eColumnList, const char* > ( eLANGNAME,              "Language" ),
    pair<eColumnList, const char* > ( eLANGID,                "LangId"   ),
    pair<eColumnList, const char* > ( eROOTPATH,              "Path"     ),
    pair<eColumnList, const char* > ( eFILETYPE,              "FileType" ),
    pair<eColumnList, const char* > ( eINDEX_ITEM,           "IndexItem" ),
    pair<eColumnList, const char* > ( eINDEX_FILETYPE,   "IndexFileType" ),


};




/*! ***********************************************************************
 *FUNCTION: 		QueryManager
 *@details: 		Constr
 *REQUIREMENT ID:	NA
 *@param:		NA
 *@return: 		NA
 *@details
 *HISTORY:
 *05.28.2020		1.0		Monojit Chatterjee		Reviewed
 ************************************************************************/
QueryManager::QueryManager()
{
    ETG_TRACE_USR4((" QueryManager constructor initialize map with null started"  ));
    rc = 0;
    mapSourceList [ entSourceType_CMP ] = nullptr;

    mapSourceList [ entSourceType_CMR ] = nullptr;


    ETG_TRACE_USR4((" QueryManager constructor initialize map with null complete"  ));
}

/*! ***********************************************************************
 *FUNCTION: 		QueryManager
 *@details: 		Destructor
 *REQUIREMENT ID:	NA
 *@param:		NA
 *@return: 		NA
 *@details
 *HISTORY:
 *05.28.2020		1.0		Monojit Chatterjee		Reviewed
 ************************************************************************/
QueryManager::~QueryManager()
{
    ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
    ETG_TRACE_USR4((" %d", __LINE__ ));
}

/*! ***********************************************************************
 *FUNCTION: 		getSqlQuery
 *@details: 		get the sql query based on request for windowList
 *REQUIREMENT ID:	NA
 *@param:		sRequestMessage
 *@return: 		string
 *@details
 *HISTORY:
 *05.28.2020		1.0		Monojit Chatterjee		Reviewed
 ************************************************************************/

string QueryManager::getSqlQuery ( sRequestMessage& vRequestMessage )
{
    ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
    ETG_TRACE_USR4((" %d", __LINE__ ));
    eListType type   = vRequestMessage.getType  ();
    string query {};

    if ( type != entListType_Invalid )
    {
        ETG_TRACE_USR4((" getSqlQuery type %d  ", type  ));

        if (QueryListType.find ( type ) != QueryListType.end() )
        {
            query = ( type == entListType_Video  ) ? getQuerySliceVideoLangList ( vRequestMessage, QueryListType [ type ]  ) : QueryListType [ type ];
        }

    }


    ETG_TRACE_USR4((" [ query ]%s ", query.c_str() ));

    return query;
}


/*! ***********************************************************************
 *FUNCTION: 		getInstance
 *@details: 		
 *REQUIREMENT ID:	NA
 *@param:		sRequestMessage
 *@return: 		string
 *@details
 *HISTORY:
 *05.28.2020		1.0		Monojit Chatterjee		Reviewed
 ************************************************************************/

QueryManager *QueryManager::getInstance()
    {
        if (!instance)
            instance = new QueryManager;
        return instance;
    }

/*! ***********************************************************************
 *FUNCTION: 		getSqlQueryCount
 *@details: 		get the sql query based on request for count
 *REQUIREMENT ID:	NA
 *@param:		sRequestMessage
 *@return: 		string
 *@details
 *HISTORY:
 *05.28.2020		1.0		Monojit Chatterjee		Reviewed
 ************************************************************************/

string QueryManager::getSqlQueryCount ( sRequestCount& vRequestCount)
{
    eListType type   = vRequestCount.getType  ();
    ETG_TRACE_USR4((" getSqlQueryCount [ ListType ]%d ", type ));
    string query {};

    if ( type != entListType_Invalid )
    {
        if ( QueryListCountType.find ( type ) != QueryListCountType.end() )
        {
            query = ( type == entListType_Video  ) ? getQueryCountVideoLangList ( vRequestCount, QueryListCountType [ type ]  ) : QueryListCountType [ type ];
            ETG_TRACE_USR4((" getSqlQueryCount query found %s ", query .c_str()));
			ETG_TRACE_USR4((" getSqlQueryCount string length %d ", query .length()));
			ETG_TRACE_USR4((" getSqlQueryCount string max_size %d ", query .capacity()));
			ETG_TRACE_USR4((" getSqlQueryCount string capasity %d ", query .max_size()));
        }

    }

    ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
    ETG_TRACE_USR4((" %d", __LINE__ ));

    ETG_TRACE_USR4((" [ query ]%s ", query.c_str() ));

    return query;
}



/*! ***********************************************************************
*FUNCTION: 		init
*@details: 		initialize the querymanager
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::init ( eSourceType& sourceType )
{
    ETG_TRACE_USR4(("QueryManager::init: sourceType %d ", sourceType ));
    bool status = false;
    if ( mapSourceList [ sourceType ] != nullptr )
    {
        ETG_TRACE_USR4(("QueryManager::init: sourceType %d found in table ", sourceType ));
        status = true;
    }
    else
    {
        mapSourceList [ sourceType ] = getSourceFactory ( sourceType );

        ETG_TRACE_USR4(("QueryManager::init: sourceType %d not found in table getting from sourceFactory ", sourceType ));
        if ( mapSourceList [ sourceType ] != nullptr )
        {
            status = true;
            ETG_TRACE_USR4(("QueryManager::init: sourceType %d got from sourceFactory and calling init ", sourceType ));
            mapSourceList [ sourceType ]->init();

        }
    }

    ETG_TRACE_USR4(("QueryManager::init: updating the root folder path for srcType %d ", sourceType ));
    status =  ( sourceType ==  entSourceType_CMR ) ? updateRootFolderPath ( sourceType ) : true ;

    status = updateLanguageTable ( sourceType );
    ETG_TRACE_USR4(("QueryManager::init: updateLanguageTable %d ", status ));

    return status ;
}


IBaseSource* QueryManager::getSourceFactory ( eSourceType srcType )
{
    IBaseSource* mSourceObject = nullptr;

    //  ETG_TRACE_USR4(("QueryManager::getSourceFactory: srcType %d", srcType ));
    if ( mapSourceList [ srcType ] != nullptr )
    {
        // do nothing
    }
    else
    {
        if ( srcType == entSourceType_CMP )
        {
            ETG_TRACE_USR4(("QueryManager::getSourceFactory: sourceType CMP creating new objects  "));
            mSourceObject =  new   CMP_Source ;
            mapSourceList [ srcType ] = mSourceObject ;
        }
        else if ( srcType == entSourceType_CMR )
        {
            ETG_TRACE_USR4(("QueryManager::getSourceFactory: sourceType CMR creating new objects  " ));
            mSourceObject =  new   CMR_Source ;
            mapSourceList [ srcType ] = mSourceObject;
        }
        else
        {
            ETG_TRACE_USR4(("QueryManager::getSourceFactory: invalid sourceType" ));
            // logs should be added saying invalid sourceType
            mapSourceList [ srcType ] = nullptr;
        }


    }

    mSourceObject =  mapSourceList [ srcType ];

    return mSourceObject;

}





/*! ***********************************************************************
*FUNCTION: 		prepareQuery
*@details: 		prepare the Query
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::prepareQuery ( string query, eSourceType srcType  )
{
    bool status = true;
    sqlite3_stmt *stmt;
    ETG_TRACE_USR4(("QueryManager::prepareQuery  [ query %s  ] ", query.c_str() ));

    ETG_TRACE_USR4(("QueryManager::prepareQuery  [ srcType %d  ] ", srcType ));

    if ( !getSourceFactory( srcType )->getDBStatus () )
    {
        ETG_TRACE_USR4(("QueryManager::prepareQuery DBStatus false " ));
        return false;
    }
    else
    {
        ETG_TRACE_USR4(("QueryManager::prepareQuery DBStatus true " ));
    }

    if (sqlite3_prepare_v2( getSourceFactory( srcType )->getDB (), query.c_str(),  -1, &stmt, NULL) != SQLITE_OK )
    {


        ETG_TRACE_USR4(("QueryManager::prepareQuery sqlite error [ errormessage %s ] ", sqlite3_errmsg(getSourceFactory( srcType )->getDB () ) ));
        //ETG_TRACE_USR4((" %d", __LINE__ ));
        status = false;
    }

    ETG_TRACE_USR4(("QueryManager::prepareQuery [ query  %s   ] end ", query.c_str() ));
    ETG_TRACE_USR4(("QueryManager::prepareQuery [ status  %d   ] ", status ));


    getSourceFactory( srcType )->setQueryStateMent ( stmt );


    return status;

}


/*! ***********************************************************************
*FUNCTION: 		fetchResult
*@details: 		process the Result
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::fetchResult ( ItemList& vItemList, eSourceType srcType, eListType type )
{
    ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
    bool status = true;
    if ( type == entListType_Video  || type == entListType_Video_NoFilter )
    {
        status = fetchSliceVideoList (  vItemList, srcType );

        return status ;
    }
    else if ( type == entListType_Folder )
    {
        status = fetchSliceFolderList ( vItemList, srcType );

        return status;
    }
    vItemList.clear();
    ETG_TRACE_USR4((" Before Statement steps size %d ", vItemList.size() ));
    while ( sqlite3_step( getSourceFactory( srcType )->getQueryStateMent ()) != SQLITE_DONE)
    {
        ETG_TRACE_USR4((" inside sqlite3_step " ));
        int index ;
        int num_cols = sqlite3_column_count( getSourceFactory( srcType )->getQueryStateMent ());
        //printf(" column count %d \n", num_cols );

        ETG_TRACE_USR4((" num of columns %d ",  num_cols ));

        sItemData vItemData;

        for (index = 0; index < num_cols; index++)
        {

            switch (sqlite3_column_type( getSourceFactory( srcType )->getQueryStateMent (), index))
            {
            case (SQLITE3_TEXT):
                printf("[ column name : %s] = %s \n",sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index),  sqlite3_column_text( getSourceFactory( srcType )->getQueryStateMent (), index));

                ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
                ETG_TRACE_USR4((" %d", __LINE__ ));

                ETG_TRACE_USR4((" [ column text %s ] ", sqlite3_column_text( getSourceFactory( srcType )->getQueryStateMent (), index)));
                vItemData.setFirstString ( (const char* ) sqlite3_column_text( getSourceFactory( srcType )->getQueryStateMent (), index) );
                break;
            case (SQLITE_INTEGER):
                printf("[ column name : %s ] , index =  %d, \n", sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index), sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index));
		
		if ( !strcmp ( (char*)sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index), QueryColumnList [ eFILETYPE ] ) )
        	{
        		vItemData.setFileType ( static_cast <eFileType> (sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index) ) );
                	ETG_TRACE_USR4((" [ FileType %d ] ", sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index)));
                    	ETG_TRACE_USR4((" fetchSliceVideolist FileType [ %d ] ",vItemData.getFileType()));
                }
		else
		{
                	ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
                	ETG_TRACE_USR4((" %d", __LINE__ ));
                	ETG_TRACE_USR4((" [ id %d ] ", sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index)));
                	vItemData.setAbsoulteIndex ( sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index) );
		}	
                break;
            case (SQLITE_FLOAT):
                printf("[ column float %g, ] \n", sqlite3_column_double( getSourceFactory( srcType )->getQueryStateMent (), index ));
                break;
            default:
                break;
            }
        }
        //, printf("\n");

        vItemList.push_back ( vItemData );

    }

    if ( SQLITE_OK != sqlite3_finalize( getSourceFactory( srcType )->getQueryStateMent ()) )
    {
        ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
        //  ETG_TRACE_USR4((" %d", __LINE__ ));
        status = false;
    }
    else
    {
        ETG_TRACE_USR4((" Most recent query exucuted "  ));
    }

    ETG_TRACE_USR4((" After Statement steps size %d ", vItemList.size() ));

    return status;

}

/*! ***********************************************************************
*FUNCTION: 		fetchCount
*@details: 		process the count
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::fetchCount ( uint16_t&  count, eSourceType srcType )
{
  
    ETG_TRACE_USR4((" fetchcount sourceType is %d ", srcType  ));
	ETG_TRACE_USR4((" fetchcount count is %d ", count  ));
   
    bool status = false;

    count = 0;
	
	

    while (sqlite3_step( getSourceFactory( srcType )->getQueryStateMent ()) != SQLITE_DONE)
    {
        int index ;
        int num_cols = sqlite3_column_count( getSourceFactory( srcType )->getQueryStateMent ());
        //printf(" column count %d \n", num_cols );

        for ( index = 0; index < num_cols; index++)
        {
            switch (sqlite3_column_type( getSourceFactory( srcType )->getQueryStateMent (), index))
            {

            case (SQLITE_INTEGER):
                string columnName = sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index );
                if ( columnName == COUNT_COLUMN )
                {
                    count += sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index );
                    ETG_TRACE_USR4(("fetchCount count is %d ", sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index )  ));

                    status = true;
                }
                break;
            }
        }
        //, printf("\n");

    }

    if ( SQLITE_OK != sqlite3_finalize( getSourceFactory( srcType )->getQueryStateMent ()) )
    {
        ETG_TRACE_USR4((" sqlite3_finalize success"  ));
        ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
        // ETG_TRACE_USR4((" %d", __LINE__ ));
        status = false;
    }
    else
    {
        ETG_TRACE_USR4((" Most recent query exucuted "  ));
    }

    ETG_TRACE_USR4(("fetchCount final count is %d ", count  ));
	
	

    return status ;
}

/*! ***********************************************************************
*FUNCTION: 		countOccuances
*@details: 		count of occurances of ?
*REQUIREMENT ID:	NA
*@param:		string
*@return: 		int
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

int QueryManager::countOccuances ( string sQuery )
{
    int res  = 0;

    for ( int c = 0; c < sQuery.length (); c++ )
    {
        if ( sQuery [ c ] == '?' )
            res++;
    }

    return res;
}

/*! ***********************************************************************
*FUNCTION: 		prepareQueryCount
*@details:
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::prepareQueryCount ( sRequestCount& vRequestCount  )
{
    ETG_TRACE_USR4((" prepareQueryCount ListType %d ", vRequestCount.getType()  ));
    bool status = true;
    switch ( vRequestCount.getType() )
    {
    case entListType_Genre:
    case entListType_Artist:
    case entListType_Album:
    case entListType_Song:
    case entListType_Video_NoFilter:
    {
        ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
        ETG_TRACE_USR4((" %d", __LINE__ ));
        status = true;
        break;
    }
    case entListType_Artist_Song:
    case entListType_Album_Song:
    case entListType_Genre_Song:
    {
        ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
        ETG_TRACE_USR4((" %d", __LINE__ ));
        if ( sqlite3_bind_int ( getSourceFactory( vRequestCount.getSourceType() )->getQueryStateMent () , FIRST_PARAM, vRequestCount.getAbsoluteId_1 () ) != SQLITE_OK )
        {
            status = false;
        }
        break;
    }

    case entListType_Video:
    {
        int index = 0 ;

        status = prepareBind ( vRequestCount.getLang(),getSourceFactory( vRequestCount.getSourceType() )->getQueryStateMent (), index );

        break;
    }

    case entListType_Folder:
    {
        ETG_TRACE_USR4((" prepareQueryCount FolderPath %s ", vRequestCount.getFolderPath().c_str()  ));
        int paramIndex = 0;
        status = prepareQueryFolder ( vRequestCount.getSourceType(), vRequestCount.getFolderPath(), paramIndex );

        break;
    }
    }

    return status;
}

/*! ***********************************************************************
*FUNCTION: 		prepareQueryData
*@details:
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::prepareQueryData( sRequestMessage& vRequestMessage )
{
    bool status = true;
    switch ( vRequestMessage.getType() )
    {
    case entListType_Genre:
    case entListType_Artist:
    case entListType_Album:
    case entListType_Song:
    case entListType_Video_NoFilter:
    case entListType_Video_Languages:
    {
        ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
        ETG_TRACE_USR4((" %d", __LINE__ ));
        ETG_TRACE_USR4((" StartIndex %d",   vRequestMessage.getStartIndex () ));
        ETG_TRACE_USR4((" WindowSize %d",   vRequestMessage.getWindowSize () ));
        ETG_TRACE_USR4((" ListType %d",   vRequestMessage.getType() ));
        int rc = sqlite3_bind_int ( getSourceFactory( vRequestMessage.getSourceType() )->getQueryStateMent () , FIRST_PARAM, vRequestMessage.getStartIndex ()  );
        rc = sqlite3_bind_int ( getSourceFactory( vRequestMessage.getSourceType() )->getQueryStateMent () , SECOND_PARAM, vRequestMessage.getWindowSize () );

        if ( rc != SQLITE_OK )
        {
            status = false;
        }
        break;
    }
    case entListType_Artist_Song:
    case entListType_Album_Song:
    case entListType_Genre_Song:
    {
        ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
        ETG_TRACE_USR4((" %d", __LINE__ ));
        int rc = sqlite3_bind_int ( getSourceFactory( vRequestMessage.getSourceType() )->getQueryStateMent () , FIRST_PARAM, vRequestMessage.getAbsoluteId_1 ()  );
        rc = sqlite3_bind_int ( getSourceFactory( vRequestMessage.getSourceType() )->getQueryStateMent () , SECOND_PARAM, vRequestMessage.getStartIndex ()  );
        rc = sqlite3_bind_int ( getSourceFactory( vRequestMessage.getSourceType() )->getQueryStateMent () , THIRD_PARAM , vRequestMessage.getWindowSize () );

        if ( rc != SQLITE_OK )
        {
            status = false;
        }
        break;
    }

    case entListType_Video:
    {
        int index = 0;
        status = prepareBind ( vRequestMessage.getLang(),getSourceFactory( vRequestMessage.getSourceType() )->getQueryStateMent (), index );

        sqlite3_bind_int ( getSourceFactory( vRequestMessage.getSourceType() )->getQueryStateMent (), ++index , vRequestMessage.getStartIndex () );
        ETG_TRACE_USR4((" index %d",   index ));
        sqlite3_bind_int ( getSourceFactory( vRequestMessage.getSourceType() )->getQueryStateMent (), ++index , vRequestMessage.getWindowSize () );
        ETG_TRACE_USR4((" index %d",   index ));
        ETG_TRACE_USR4((" StartIndex %d",   vRequestMessage.getStartIndex () ));
        ETG_TRACE_USR4((" WindowSize %d",   vRequestMessage.getWindowSize () ));
        ETG_TRACE_USR4((" ListType %d",   vRequestMessage.getType() ));
        break;
    }

    case entListType_Folder :
    {
        int paramIndex = 0;
        status = prepareQueryFolder ( vRequestMessage.getSourceType(), vRequestMessage.getFolderPath(), paramIndex );

        sqlite3_bind_int ( getSourceFactory( vRequestMessage.getSourceType() )->getQueryStateMent (), ++paramIndex , vRequestMessage.getStartIndex () );
        ETG_TRACE_USR4((" paramIndex %d",   paramIndex ));
        sqlite3_bind_int ( getSourceFactory( vRequestMessage.getSourceType() )->getQueryStateMent (), ++paramIndex , vRequestMessage.getWindowSize () );
        break;
    }
    }

    return status;
}

/*! ***********************************************************************
*FUNCTION: 		deInit
*@details:
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::deInit ( eSourceType& sourceType )
{
    bool status = false;
    if ( mapSourceList [ sourceType ] != nullptr )
    {
        mapSourceList.erase ( sourceType );

        mapSourceList [ sourceType ] = nullptr;

        ETG_TRACE_USR4((" QueryManager deInit initialize map sourceTpe %d with null ",   sourceType ));

        status = true ;
    }

    return status;
}

/*! ***********************************************************************
*FUNCTION: 		getNextQuery
*@details:
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/


string QueryManager::getNextQuery ( eListType listType )
{
    return ((QueryNextPreviousType.equal_range( listType ).first))->second;
}

/*! ***********************************************************************
*FUNCTION: 		getPreviousQuery
*@details:
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

string QueryManager::getPreviousQuery ( eListType listType )
{
    return (++(QueryNextPreviousType.equal_range( listType ).first))->second;
}

/*! ***********************************************************************
*FUNCTION: 		existNextPreviousQuery
*@details:
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/


bool QueryManager::existNextPreviousQuery ( eListType listType )
{
    return ( QueryNextPreviousType.count ( listType ) > 1 );
}

/*! ***********************************************************************
*FUNCTION: 		getQueryNextPrevious
*@details:
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/



string QueryManager::getQueryNextPrevious ( sRequestSongId& vRequestSongId )
{
    eListType type  = vRequestSongId.getType ();
    if ( !existNextPreviousQuery ( type ) )
        return "";

    string query = ( vRequestSongId.getDirection () == eNEXT ) ? getNextQuery ( type ) :  getPreviousQuery ( type );

    query = ( type == entListType_Video ) ? getLangQuery ( query, vRequestSongId ) : query ;

    ETG_TRACE_USR4((" query executed %s  ",   query.c_str() ));

    return query;

}

/*! ***********************************************************************
*FUNCTION: 		prepareQueryNextPrevious
*@details:
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/


bool QueryManager::prepareQueryNextPrevious ( sRequestSongId& vRequestSongId )
{
    ETG_TRACE_USR4((" current song id %d and listtype %d ",   vRequestSongId.getCurrentSongId (), vRequestSongId.getType() ));
    bool status = true;
    switch ( vRequestSongId.getType() )
    {
    case entListType_Genre:
    case entListType_Artist:
    case entListType_Album:
    case entListType_Song:
    case entListType_Video_NoFilter:
    {

        if ( sqlite3_bind_int ( getSourceFactory( vRequestSongId.getSourceType() )->getQueryStateMent () , FIRST_PARAM, vRequestSongId.getCurrentSongId () ) != SQLITE_OK )
        {
            status = false;
        }
        break;
    }
    case entListType_Artist_Song:
    case entListType_Album_Song:
    case entListType_Genre_Song:
    {
        ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
        ETG_TRACE_USR4((" %d", __LINE__ ));
        if ( sqlite3_bind_int ( getSourceFactory( vRequestSongId.getSourceType() )->getQueryStateMent () , FIRST_PARAM, vRequestSongId.getCurrentSongId () ) != SQLITE_OK )
        {
            status = false;
        }
        break;
    }

    case entListType_Video:
    {
        status = prepareQueryVideoNextPrevious ( vRequestSongId );
        break;
    }

    case  entListType_Folder:
    {
        int paramIndex = 0;
        status = prepareQueryFolderNextPrevious ( vRequestSongId, paramIndex );
        break;
    }
    }

    return status;
}

/*! ***********************************************************************
*FUNCTION: 		updateSongId
*@details:
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/


bool QueryManager::updateSongId (  sResponseSongInfo& vResponseSongInfo, eSourceType srcType )
{
    IBaseSource* vSourceReference = nullptr;

    if ( getSourceFactory ( srcType ) != nullptr )
    {
        ETG_TRACE_USR4((" updateSongId  song Id %d ", vResponseSongInfo.getUpdatedSongId()  ));
        ETG_TRACE_USR4((" updateSongId srcType %d ", srcType ));
        ETG_TRACE_USR4((" updateSongId dbPath %s ", getSourceFactory ( srcType )->getDBPath ().c_str() ));
        vSourceReference = getSourceFactory ( srcType );
    }
    else
    {
        ETG_TRACE_USR4((" updateSongId srcType %d failed to get factory object ", srcType ));
        return false;
    }

    bool status = false;

    while (sqlite3_step( vSourceReference->getQueryStateMent ()) != SQLITE_DONE)
    {
        ETG_TRACE_USR4(("updateSongId  sqlite3_step is SQLITE done "));
        int index ;
        int num_cols = sqlite3_column_count( vSourceReference->getQueryStateMent ());


        for ( index = 0; index < num_cols; index++)
        {
            switch (sqlite3_column_type( vSourceReference->getQueryStateMent (), index))
            {

            case (SQLITE_INTEGER):
                string columnName = sqlite3_column_name( vSourceReference->getQueryStateMent (), index );

                if ( !strcmp ( sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index), QueryColumnList [ eINDEX_ITEM ] ) )
                {
                    vResponseSongInfo.setUpdatedSongInfo ( sqlite3_column_int( vSourceReference->getQueryStateMent (), index ) );
					ETG_TRACE_USR4(("inside indexItem  updateSongId song Id  is %d ", vResponseSongInfo.getUpdatedSongId()  ));
                }
                else if ( !strcmp ( sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index), QueryColumnList [ eINDEX_FILETYPE ] ) )
                {
                    vResponseSongInfo.setFileType ( static_cast <eFileType> ( sqlite3_column_int( vSourceReference->getQueryStateMent (), index ) ) );
					ETG_TRACE_USR4(("inside indexFileType  updateSongId filetype %d ", vResponseSongInfo.getFileType()  ));
                }

                ETG_TRACE_USR4(("updateSongId song Id  is %d ", sqlite3_column_int( vSourceReference->getQueryStateMent (), index )  ));

                status = true;

                break;
            }
        }
        //, printf("\n");

    }

    if ( SQLITE_OK != sqlite3_finalize( vSourceReference->getQueryStateMent ()) )
    {
        printf("sqlite error \n");
        ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
        // ETG_TRACE_USR4((" %d", __LINE__ ));
        status = false;
    }
    else
    {
        ETG_TRACE_USR4((" Most recent query exucuted "  ));
    }

    return status ;
}

/*! ***********************************************************************
*FUNCTION: 		getFirstSongQuery
*@details: get the query of first song
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

string QueryManager::getFirstSongQuery ( sRequestFrontRearSong& vRequestFirstSong )
{
    ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
    ETG_TRACE_USR4((" %d", __LINE__ ));

    eListType type =  vRequestFirstSong.getType();

    string query {};

    if ( type != entListType_Invalid )
    {
        ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
        ETG_TRACE_USR4((" %d", __LINE__ ));
        if (QueryGetFirstSongType.find ( type ) != QueryGetFirstSongType.end() )
        {

            query = ( type == entListType_Video ) ? ( getVideoFirstSongQuery ( QueryGetFirstSongType [ type ] , vRequestFirstSong) ) : QueryGetFirstSongType [ type ];
        }

    }

    ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
    ETG_TRACE_USR4((" %d", __LINE__ ));

    ETG_TRACE_USR4((" [ query ]%s ", query.c_str() ));

    return query;
}

/*! ***********************************************************************
*FUNCTION: 		getSqlQueryCompleteList
*@details: get the query of getting complete list based on listType
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

string QueryManager::getSqlQueryCompleteList ( sReqListInfoMessage& vReqListMessage )
{
    ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
    ETG_TRACE_USR4((" %d", __LINE__ ));

    eListType type = vReqListMessage.getType();

    string query {};

    if ( type != entListType_Invalid )
    {
        ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
        ETG_TRACE_USR4((" %d", __LINE__ ));
        if (QueryCompleteList.find ( type ) != QueryCompleteList.end() )
        {
            query = ( type == entListType_Video ) ? getVideoCompleteListQuery (QueryCompleteList [ type ] , vReqListMessage ) : QueryCompleteList [ type ] ;
        }

    }

    ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
    ETG_TRACE_USR4((" %d", __LINE__ ));

    ETG_TRACE_USR4((" [ query ]%s ", query.c_str() ));

    return query;
}

/*! ***********************************************************************
*FUNCTION: 		getSqlLastSongQuery
*@details: get the query of last song
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

string QueryManager::getSqlLastSongQuery ( sRequestFrontRearSong& vRequestLastSong )
{
    ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
    ETG_TRACE_USR4((" %d", __LINE__ ));

    eListType type = vRequestLastSong.getType();

    string query {};

    if ( type != entListType_Invalid )
    {
        ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
        ETG_TRACE_USR4((" %d", __LINE__ ));
        if (QueryGetLastSongType.find ( type ) != QueryGetLastSongType.end() )
        {

            query = ( type == entListType_Video ) ? getVideoLastSongQuery ( QueryGetLastSongType [ type ] , vRequestLastSong ) : QueryGetLastSongType [ type ];
        }

    }

    ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
    ETG_TRACE_USR4((" %d", __LINE__ ));

    ETG_TRACE_USR4((" [ query ]%s ", query.c_str() ));

    return query;
}

bool QueryManager::prepareQueryLastSong ( sRequestFrontRearSong& vRequestLastSong, uint16_t& vLastSongPrepareCount )
{
    bool status = true;
    switch ( vRequestLastSong.getType() )
    {
    case entListType_Genre:
    case entListType_Artist:
    case entListType_Album:
    case entListType_Song:
    case entListType_Video_NoFilter:
    {
        ETG_TRACE_USR4((" do Nothing since no preparation required complete list listype %d ", vRequestLastSong.getType() ));

        int rc = sqlite3_bind_int ( getSourceFactory( vRequestLastSong.getSourceType() )->getQueryStateMent () , FIRST_PARAM, vLastSongPrepareCount  );

        if ( rc != SQLITE_OK )
        {
            status = false;
        }
        break;
    }
    case entListType_Artist_Song:
    case entListType_Album_Song:
    case entListType_Genre_Song:
    {
        ETG_TRACE_USR4((" preparation required for complete list listype %d ", vRequestLastSong.getType() ));
        int rc = sqlite3_bind_int ( getSourceFactory( vRequestLastSong.getSourceType() )->getQueryStateMent () , FIRST_PARAM, vRequestLastSong.getAbsoluteId_1 ()  );
        rc = sqlite3_bind_int ( getSourceFactory( vRequestLastSong.getSourceType() )->getQueryStateMent () , SECOND_PARAM, vLastSongPrepareCount  );
        if ( rc != SQLITE_OK )
        {
            status = false;
        }
        break;
    }

    case entListType_Video:
    {

        int index = 0 ;

        status = prepareBind ( vRequestLastSong.getLang(),getSourceFactory( vRequestLastSong.getSourceType() )->getQueryStateMent (), index );

        int rc = sqlite3_bind_int ( getSourceFactory( vRequestLastSong.getSourceType() )->getQueryStateMent () , ++index, vLastSongPrepareCount  );
        if ( rc != SQLITE_OK )
        {
            status = false;
        }

        ETG_TRACE_USR4((" prepareQueryLastSong entListType_Video  status %d ", status ));
	break;
    }

    case entListType_Folder:
    {
        int paramIndex = 0;
        status = prepareQueryFrontRearSong ( vRequestLastSong, paramIndex );
        status = ( sqlite3_bind_int ( getSourceFactory( vRequestLastSong.getSourceType() )->getQueryStateMent () , ++paramIndex, vLastSongPrepareCount  ) == SQLITE_OK ) ? true : false ;

        break;
    }
    default:{}
    }

    return status;
}

/*! ***********************************************************************
*FUNCTION: 		prepareQueryData
*@details:      prepare the query for listType
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*05.28.2020		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::prepareQueryData( sReqListInfoMessage& vReqListMessage )
{
    bool status = true;
    switch ( vReqListMessage.getType() )
    {
    case entListType_Genre:
    case entListType_Artist:
    case entListType_Album:
    case entListType_Song:
    case entListType_Video_NoFilter:
    {
        ETG_TRACE_USR4((" do Nothing since no preparation required complete list listype %d ", vReqListMessage.getType() ));

        break;
    }
    case entListType_Artist_Song:
    case entListType_Album_Song:
    case entListType_Genre_Song:
    {
        ETG_TRACE_USR4((" preparation required for complete list listype %d ", vReqListMessage.getType() ));
        int rc = sqlite3_bind_int ( getSourceFactory( vReqListMessage.getSourceType() )->getQueryStateMent () , FIRST_PARAM, vReqListMessage.getAbsoluteId_1 ()  );

        if ( rc != SQLITE_OK )
        {
            status = false;
        }
        break;
    }

    case entListType_Video:
    {
        int index = 0;



        status = this->prepareBind ( vReqListMessage.getLang(),getSourceFactory( vReqListMessage.getSourceType() )->getQueryStateMent (), index );

        ETG_TRACE_USR4((" prepareQueryData entListType_Video  status %d ", status ));


        break;
    }

    case entListType_Folder:
    {
        int paramIndex = 0;

        status = prepareQueryCompleteList ( vReqListMessage , paramIndex );

        break;
    }
    }

    return status;
}

/*! ***********************************************************************
*FUNCTION: 		getQueryVideoLangList
*@details:      function is used to get the query for count video list
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*03.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

string QueryManager::getQueryCountVideoLangList ( sRequestCount& vRequestCount, string query )
{
    string paramList = prepareParamList ( vRequestCount.getLang() );

    query = query + paramList + "))";

    ETG_TRACE_USR4((" getQueryCountVideoLangList query %s ", query.c_str() ));

    return query;
}


/*! ***********************************************************************
*FUNCTION: 		getQuerySliceVideoLangList
*@details:      function is used to get the query for video list
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*03.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

string QueryManager::getQuerySliceVideoLangList ( sRequestMessage& vRequestMessage, string query )
{
    string paramList = prepareParamList ( vRequestMessage.getLang() );

    query = query + paramList + "))" +" LIMIT ?,?";

    ETG_TRACE_USR4((" getQuerySliceVideoLangList query %s ", query.c_str() ));

    return query;
}

/*! ***********************************************************************
*FUNCTION: 		fetchSliceVideoList
*@details:      fetch the Slice of VideoList
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*3.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::fetchSliceVideoList ( ItemList& vItemList, eSourceType srcType )
{

    bool status = true;
    vItemList.clear();
    ETG_TRACE_USR4((" fetchSliceVideoList Before Statement steps size %d ", vItemList.size() ));
    while ( sqlite3_step( getSourceFactory( srcType )->getQueryStateMent ()) != SQLITE_DONE)
    {
        int index ;
        int num_cols = sqlite3_column_count( getSourceFactory( srcType )->getQueryStateMent ());

        ETG_TRACE_USR4((" fetchSliceVideoList [ num_cols ]  %d ", num_cols ));

        sItemData vItemData;

        for (index = 0; index < num_cols; index++)
        {
            ETG_TRACE_USR4((" fetchSliceVideoList [ column name : %s ]   ", sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index) ));

            switch (sqlite3_column_type( getSourceFactory( srcType )->getQueryStateMent (), index))
            {
            case (SQLITE3_TEXT):

                if ( !strcmp ( sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index), QueryColumnList [ eFILENAME ] ) )
                {
                    vItemData.setFirstString ( (const char* ) sqlite3_column_text( getSourceFactory( srcType )->getQueryStateMent (), index) );

                    ETG_TRACE_USR4((" fetchSliceVideoList [ firstString ]  %s ", vItemData.getFirstString().c_str() ));
                }
                else if ( !strcmp ( (char*)sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index), QueryColumnList [ eISO ] ) )
                {
                    vItemData.setSecondString ( (const char* ) sqlite3_column_text( getSourceFactory( srcType )->getQueryStateMent (), index) );

                    ETG_TRACE_USR4((" fetchSliceVideoList [ secondString ]  %s ", vItemData.getSecondString().c_str() ));
                }

                ETG_TRACE_USR4((" [ column text %s ] ", sqlite3_column_text( getSourceFactory( srcType )->getQueryStateMent (), index)));

                break;
            case (SQLITE_INTEGER):
                printf("[ column name : %s ] , index =  %d, \n", sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index), sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index));

                if ( !strcmp ( (char*)sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index), QueryColumnList [ eID ] ) )
                {
                    vItemData.setAbsoulteIndex ( sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index) );
                    ETG_TRACE_USR4((" [ id %d ] ", sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index)));

                    ETG_TRACE_USR4((" fetchSliceVideoList [ Id ]  %d ", vItemData.getAbsoluteIndex() ));
                }
                else if ( !strcmp ( (char*)sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index), QueryColumnList [ eFILETYPE ] ) )
                {
                    vItemData.setFileType ( static_cast <eFileType> (sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index) ) );
                    ETG_TRACE_USR4((" [ FileType %d ] ", sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index)));

                    ETG_TRACE_USR4((" fetchSliceVideolist FileType [ %d ] ",vItemData.getFileType()));
                }





                break;
            case (SQLITE_FLOAT):
                printf("[ column float %g, ] \n", sqlite3_column_double( getSourceFactory( srcType )->getQueryStateMent (), index ));
                break;
            default:
                break;
            }
        }
        //, printf("\n");

        vItemList.push_back ( vItemData );

    }

    if ( SQLITE_OK != sqlite3_finalize( getSourceFactory( srcType )->getQueryStateMent ()) )
    {
        ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
        //  ETG_TRACE_USR4((" %d", __LINE__ ));
        status = false;
    }
    else
    {
        ETG_TRACE_USR4((" Most recent query exucuted "  ));
    }

    ETG_TRACE_USR4((" After Statement steps size %d ", vItemList.size() ));

    return status  ;
}

/*! ***********************************************************************
*FUNCTION: 		prepareParamList
*@details:      prepare  the param List
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*3.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

string QueryManager::prepareParamList ( Langlist&  lang )
{
    string paramList = "?";

    for ( int index = 1; index < lang.size(); index++ )
    {
        paramList +=  " , ? ";
    }

    return paramList;
}

/*! ***********************************************************************
*FUNCTION: 		prepareBind
*@details:      prepare  the Bind
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*3.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/
bool QueryManager::prepareBind ( Langlist&         lang,  sqlite3_stmt *stmt, int& index  )
{
    bool status = false;

    auto  it = lang.begin();

    for ( ; it != lang.end(); it++ )
    {
        sqlite3_bind_int ( stmt ,  index =  ( distance (lang.begin(), it ) ) + 1 , *it );

        ETG_TRACE_USR4((" prepareBind param  %d ", index ));
        ETG_TRACE_USR4((" prepareBind value  %d ", *it ));


    }

    if ( it == lang.end() )  status = true;

    ETG_TRACE_USR4((" prepareBind status  %d ", status ));

    return status;

}

/*! ***********************************************************************
*FUNCTION: 		CheckIfSongIdExist
*@details:      Check If SongId Exist
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*3.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::CheckIfSongIdExist (  sRequestSongId& vRequestSongId )
{
    bool status = false;

    if ( vRequestSongId.getType () == entListType_Video && !vRequestSongId.getLang().empty() )
    {
        string paramList = prepareParamList ( vRequestSongId.getLang() );
        string query = QUERY_ID_EXIST + paramList + ID_EXIST_EXT;

        ETG_TRACE_USR4((" CheckIfSongIdExist query  %s ", query.c_str() ));

        status = this->prepareQuery ( query, vRequestSongId.getSourceType() );

        status = this->prepareQueryVideoIdExist( vRequestSongId );

        status = this->fetchIdStatus ( vRequestSongId.getSourceType () );

        ETG_TRACE_USR4((" CheckIfSongIdExist status %d ", status ));

    }

    return status;
}

/*! ***********************************************************************
*FUNCTION: 		prepareQueryIdExist
*@details:      prepare Query Id Exist
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*3.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::prepareQueryVideoIdExist ( sRequestSongId& vRequestSongId )
{
    bool status = false;
    int index = 0;

    sqlite3_bind_int ( getSourceFactory( vRequestSongId.getSourceType() )->getQueryStateMent (),  index = index  + 1 , vRequestSongId.getCurrentSongId() );

    status = this->prepareBind ( vRequestSongId.getLang(),getSourceFactory( vRequestSongId.getSourceType() )->getQueryStateMent (), index );

    ETG_TRACE_USR4((" fetchIdStatus dbPath %s ", getSourceFactory ( vRequestSongId.getSourceType() )->getDBPath ().c_str() ));



    return status;
}

/*! ***********************************************************************
*FUNCTION: 		fetchIdStatus
*@details:      fetch Id Status
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*3.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::fetchIdStatus ( eSourceType srcType )
{

    bool status = false;

    IBaseSource* vSourceReference = nullptr;

    if ( getSourceFactory ( srcType ) != nullptr )
    {

        ETG_TRACE_USR4((" fetchIdStatus dbPath %s ", getSourceFactory ( srcType )->getDBPath ().c_str() ));
        vSourceReference = getSourceFactory ( srcType );
    }
    else
    {
        ETG_TRACE_USR4((" fetchIdStatus srcType %d failed to get factory object ", srcType ));
        return false;
    }

    while (sqlite3_step( vSourceReference->getQueryStateMent ()) != SQLITE_DONE)
    {
        ETG_TRACE_USR4(("fetchIdStatus  sqlite3_step is SQLITE done "));
        int index ;
        int num_cols = sqlite3_column_count( vSourceReference->getQueryStateMent ());


        for ( index = 0; index < num_cols; index++)
        {
            switch (sqlite3_column_type( vSourceReference->getQueryStateMent (), index))
            {

            case (SQLITE_INTEGER):
                string columnName = sqlite3_column_name( vSourceReference->getQueryStateMent (), index );

                status = sqlite3_column_int( vSourceReference->getQueryStateMent (), index );

                ETG_TRACE_USR4(("fetchIdStatus status  is %d , column details %s ", sqlite3_column_int( vSourceReference->getQueryStateMent (), index ), columnName.c_str()  ));



                break;
            }
        }
        //, printf("\n");

    }

    if ( SQLITE_OK != sqlite3_finalize( vSourceReference->getQueryStateMent ()) )
    {
        printf("sqlite error \n");
        ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
        // ETG_TRACE_USR4((" %d", __LINE__ ));
        status = false;
    }
    else
    {
        ETG_TRACE_USR4((" Most recent query exucuted "  ));
    }

    return status ;
}

/*! ***********************************************************************
*FUNCTION: 		getLangQuery
*@details:      get Lang Query
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*3.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

string QueryManager::getLangQuery ( string query, sRequestSongId& vRequestSongId  )
{
    string paramList = prepareParamList ( vRequestSongId.getLang() );

    query = query + paramList;

    query += ( vRequestSongId.getDirection () == eNEXT ) ? ( NEXT_EXT ) : (PREVIOUS_EXT);

    ETG_TRACE_USR4((" QueryManager getLangQuery query %s ", query.c_str()  ));

    return query;
}

/*! ***********************************************************************
*FUNCTION: 		prepareQueryVideoNextPrevious
*@details:      prepare Query for Video NextPrevious
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*3.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::prepareQueryVideoNextPrevious ( sRequestSongId& vRequestSongId )
{
    bool status = false;
    int index = 0;

    sqlite3_bind_int ( getSourceFactory( vRequestSongId.getSourceType() )->getQueryStateMent (),  index = index  + 1 , vRequestSongId.getCurrentSongId() );

    ETG_TRACE_USR4((" QueryManager prepareQueryVideoNextPrevious currentSongId %d ", vRequestSongId.getCurrentSongId()  ));

    status = this->prepareBindNextPrevious ( vRequestSongId.getLang(),getSourceFactory( vRequestSongId.getSourceType() )->getQueryStateMent (), index );

    ETG_TRACE_USR4((" QueryManager prepareQueryVideoNextPrevious index %d ", index  ));

    return status;
}

/*! ***********************************************************************
*FUNCTION: 		getVideoFirstSongQuery
*@details:      get Video First Song Query
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*3.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

string QueryManager::getVideoFirstSongQuery ( string query, sRequestFrontRearSong& vRequestFirstSong )
{
    query = query + prepareParamList ( vRequestFirstSong.getLang() ) + FIRST_SONG_EXT;

    ETG_TRACE_USR4((" QueryManager getVideoFirstSongQuery query %s ", query.c_str()  ));

    return query;
}

/*! ***********************************************************************
*FUNCTION: 		prepareQueryFirstSong
*@details:      prepare Query First Song
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*3.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::prepareQueryFirstSong ( sRequestFrontRearSong& vRequestFirstSong )
{
    bool status = true;
    switch ( vRequestFirstSong.getType() )
    {
    case entListType_Genre:
    case entListType_Artist:
    case entListType_Album:
    case entListType_Song:
    case entListType_Video_NoFilter:
    {

        status = true;
        break;
    }
    case entListType_Artist_Song:
    case entListType_Album_Song:
    case entListType_Genre_Song:
    {
        ETG_TRACE_USR4((" preparation required for complete list listype %d ", vRequestFirstSong.getType() ));
        int rc = sqlite3_bind_int ( getSourceFactory( vRequestFirstSong.getSourceType() )->getQueryStateMent () , FIRST_PARAM, vRequestFirstSong.getAbsoluteId_1 ()  );

        if ( rc != SQLITE_OK )
        {
            status = false;
        }
        break;
    }

    case entListType_Video :
    {
        status = prepareQueryVideo( vRequestFirstSong );
        ETG_TRACE_USR4((" QueryManager prepareQueryFirstSong calling prepareQueryVideo status %d ", status  ));
        break;
    }

    case entListType_Folder:
    {
        int paramIndex = 0;
        status = prepareQueryFrontRearSong ( vRequestFirstSong, paramIndex );
        break;
    }
    }

    return status;
}


/*! ***********************************************************************
*FUNCTION: 		prepareQueryVideo
*@details:      prepare Query Video
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*3.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/


bool QueryManager::prepareQueryVideo ( sRequestFrontRearSong& vRequestFirstSong )
{
    bool status = false;

    int index = 0;



    status = this->prepareBind ( vRequestFirstSong.getLang(),getSourceFactory( vRequestFirstSong.getSourceType() )->getQueryStateMent (), index );

    ETG_TRACE_USR4((" QueryManager prepareQueryVideo index %d ", index  ));

    return status;

}

/*! ***********************************************************************
*FUNCTION: 		getVideoLastSongQuery
*@details:      get Video Last Song Query
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*3.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

string QueryManager::getVideoLastSongQuery ( string query, sRequestFrontRearSong& vRequestLastSong )
{
    query = query + prepareParamList ( vRequestLastSong.getLang() ) + LAST_SONG_EXT;

    ETG_TRACE_USR4((" QueryManager getVideoLastSongQuery %s", query.c_str()  ));

    return query;
}

/*! ***********************************************************************
*FUNCTION: 		getVideoCompleteListQuery
*@details:      get Video Complete List Query
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*3.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

string QueryManager::getVideoCompleteListQuery ( string query, sReqListInfoMessage& vReqListMessage )
{
    query = query + prepareParamList ( vReqListMessage.getLang() );

    ETG_TRACE_USR4((" QueryManager getVideoCompleteListQuery %s", query.c_str()  ));

    return query;

}

/*! ***********************************************************************
*FUNCTION: 		prepareBindNextPrevious
*@details:      prepare Bind NextPrevious
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*3.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::prepareBindNextPrevious ( Langlist&         lang,  sqlite3_stmt *stmt, int& index   )
{
    bool status = false;

    auto  it = lang.begin();

    for ( ; it != lang.end(); it++ )
    {
        sqlite3_bind_int ( stmt ,  index =  index + 1 , *it );

        ETG_TRACE_USR4((" prepareBind param  %d ", index ));
        ETG_TRACE_USR4((" prepareBind value  %d ", *it ));


    }

    if ( it == lang.end() )  status = true;

    ETG_TRACE_USR4((" prepareBind status  %d ", status ));

    return status;
}


/*! ***********************************************************************
*FUNCTION: 		fetchRootPath
*@details:      fetch the Root Path
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*5.13.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/


bool   QueryManager::fetchRootPath ( eSourceType& sourceType, string& rootPath )
{
    bool status = false;

    IBaseSource* vSourceReference = nullptr;

    if ( getSourceFactory ( sourceType ) != nullptr )
    {

        ETG_TRACE_USR4((" fetchRootPath dbPath %s ", getSourceFactory ( sourceType )->getDBPath ().c_str() ));
        vSourceReference = getSourceFactory ( sourceType );
    }
    else
    {
        ETG_TRACE_USR4((" fetchRootPath srcType %d failed to get factory object ", sourceType ));
        return false;
    }

    while (sqlite3_step( vSourceReference->getQueryStateMent ()) != SQLITE_DONE)
    {
        ETG_TRACE_USR4(("fetchRootPath  sqlite3_step is SQLITE done "));
        int index ;
        int num_cols = sqlite3_column_count( vSourceReference->getQueryStateMent ());


        for ( index = 0; index < num_cols; index++)
        {
            switch (sqlite3_column_type( vSourceReference->getQueryStateMent (), index))
            {

            case (SQLITE3_TEXT):

                if ( ( !strcmp ( sqlite3_column_name(vSourceReference->getQueryStateMent () , index ) , QueryColumnList [ eROOTPATH ]  )  )    )
                {
                    rootPath = ( const char* ) ( sqlite3_column_text(vSourceReference->getQueryStateMent (), index ) );

                    ETG_TRACE_USR4(("fetchRootPath  root path is %s ", rootPath.c_str() ));
                    status = true;

                }


                break;



                break;
            }
        }
        //, printf("\n");

    }

    if ( SQLITE_OK != sqlite3_finalize( vSourceReference->getQueryStateMent ()) )
    {
        printf("sqlite error \n");
        ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
        // ETG_TRACE_USR4((" %d", __LINE__ ));
        status = false;
    }
    else
    {
        ETG_TRACE_USR4((" Most recent query exucuted "  ));
    }

    return status ;
}

/*! ***********************************************************************
*FUNCTION: 		updateRootFolderPath
*@details:      update Root Folder Path
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*5.13.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::updateRootFolderPath ( eSourceType& sourceType )
{
    bool status = false;

    status = QueryManager::getInstance()->prepareQuery ( FOLDER_ROOT_PATH_QUERY, sourceType );

    string rootPath = "";
    status = status ? this->fetchRootPath (  sourceType, rootPath ) : false ;

    getSourceFactory ( sourceType ) ->setRootFolderPath ( rootPath );

    ETG_TRACE_USR4(("updateRootFolderPath rootPath is updated with %s ", getSourceFactory ( sourceType ) ->getRootFolderPath().c_str() ));

    ETG_TRACE_USR4(("updateRootFolderPath status  %d ", status ));

    return status ;
}

/*! ***********************************************************************
*FUNCTION: 		prepareQueryFolder
*@details:      prepare Query Folder
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*5.13.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::prepareQueryFolder ( eSourceType sourceType , string folderPath, int& paramIndex )
{
    ETG_TRACE_USR4(("prepareQueryFolder folderPath  %s ", folderPath.c_str() ));
    bool status = false;
    string absolutePath = getSourceFactory( sourceType )->getRootFolderPath() + folderPath;
	
	if ( ! absolutePath.empty() )        absolutePath.pop_back();
	
	getSourceFactory( sourceType )->setAbsoluteFolderPath ( absolutePath );
	
	
	
	ETG_TRACE_USR4(("prepareQueryFolder absolutePath  %s ", absolutePath.c_str() ));
	ETG_TRACE_USR4(("prepareQueryFolder absolutePath length  %d ", strlen (absolutePath .c_str()) ));
    status = ( sqlite3_bind_text ( getSourceFactory( sourceType )->getQueryStateMent (), ++paramIndex,   absolutePath.c_str() , strlen (absolutePath .c_str()), SQLITE_TRANSIENT ) == SQLITE_OK ) ? true : false;
    ETG_TRACE_USR4(("prepareQueryFolder paramIndex  %d ", paramIndex ));
	ETG_TRACE_USR4(("prepareQueryFolder status  %d ", status ));

    status = ( sqlite3_bind_text ( getSourceFactory( sourceType )->getQueryStateMent (), ++paramIndex,   absolutePath.c_str() , strlen (absolutePath .c_str()), SQLITE_TRANSIENT ) == SQLITE_OK ) ? true : false;
    ETG_TRACE_USR4(("prepareQueryFolder paramIndex  %d ", paramIndex ));
	ETG_TRACE_USR4(("prepareQueryFolder status  %d ", status ));

    status = ( sqlite3_bind_text ( getSourceFactory( sourceType )->getQueryStateMent (),++paramIndex,   absolutePath.c_str() , strlen (absolutePath .c_str()), SQLITE_TRANSIENT ) == SQLITE_OK ) ? true : false;
    ETG_TRACE_USR4(("prepareQueryFolder paramIndex  %d ", paramIndex ));
	ETG_TRACE_USR4(("prepareQueryFolder status  %d ", status ));

    ETG_TRACE_USR4(("prepareQueryFolder absolutePath  %s ", absolutePath.c_str() ));

    return status;
}

/*! ***********************************************************************
*FUNCTION: 		fetchSliceFolderList
*@details:      fetch Slice Folder List
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*5.13.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/
bool QueryManager::fetchSliceFolderList ( ItemList& vItemList, eSourceType srcType )
{
    bool status = true;
    vItemList.clear();
    ETG_TRACE_USR4((" fetchSliceFolderList Before Statement steps size %d ", vItemList.size() ));
    while ( sqlite3_step( getSourceFactory( srcType )->getQueryStateMent ()) != SQLITE_DONE)
    {
        int index ;
        int num_cols = sqlite3_column_count( getSourceFactory( srcType )->getQueryStateMent ());

        ETG_TRACE_USR4((" fetchSliceFolderList [ num_cols ]  %d ", num_cols ));

        sItemData vItemData;

        for (index = 0; index < num_cols; index++)
        {
            ETG_TRACE_USR4((" fetchSliceFolderList [ column name : %s ]   ", sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index) ));

            switch (sqlite3_column_type( getSourceFactory( srcType )->getQueryStateMent (), index))
            {
            case (SQLITE3_TEXT):

                if ( !strcmp ( sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index), QueryColumnList [ eFILENAME ] ) )
                {
                    vItemData.setFirstString ( (const char* ) sqlite3_column_text( getSourceFactory( srcType )->getQueryStateMent (), index) );

                    ETG_TRACE_USR4((" fetchSliceFolderList [ firstString ]  %s ", vItemData.getFirstString().c_str() ));
                }


                ETG_TRACE_USR4((" [ fetchSliceFolderList column text %s ] ", sqlite3_column_text( getSourceFactory( srcType )->getQueryStateMent (), index)));

                break;
            case (SQLITE_INTEGER):
                printf("[ fetchSliceFolderList column name : %s ] , index =  %d, \n", sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index), sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index));

                if ( !strcmp ( (char*)sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index), QueryColumnList [ eID ] ) )
                {
                    vItemData.setAbsoulteIndex ( sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index) );
                    ETG_TRACE_USR4((" [ fetchSliceFolderList id %d ] ", sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index)));

                    ETG_TRACE_USR4((" fetchSliceFolderList [ Id ]  %d ", vItemData.getAbsoluteIndex() ));
                }

                if ( !strcmp ( (char*)sqlite3_column_name( getSourceFactory( srcType )->getQueryStateMent (), index), QueryColumnList [ eFILETYPE ] ) )
                {
                    vItemData.setFileType ( static_cast <eFileType> (sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index) ) );
                    ETG_TRACE_USR4((" [ fetchSliceFolderList FileType %d ] ", sqlite3_column_int(getSourceFactory( srcType )->getQueryStateMent (), index)));

                    string ItemName = ( vItemData.getFileType() == eFOLDER ) ? ( extractFolderRelativePath ( vItemData.getFirstString () , srcType )  ) : vItemData.getFirstString ();

                    vItemData.setFirstString ( ItemName );



                    ETG_TRACE_USR4((" fetchSliceFolderList [ Id ]  %d ", vItemData.getAbsoluteIndex() ));

                    ETG_TRACE_USR4((" fetchSliceFolderList [ firstString ]  %s ", vItemData.getFirstString ().c_str() ));
                }





                break;
            case (SQLITE_FLOAT):
                printf("[ column float %g, ] \n", sqlite3_column_double( getSourceFactory( srcType )->getQueryStateMent (), index ));
                break;
            default:
                break;
            }
        }
        //, printf("\n");

        vItemList.push_back ( vItemData );

    }

    if ( SQLITE_OK != sqlite3_finalize( getSourceFactory( srcType )->getQueryStateMent ()) )
    {
        ETG_TRACE_USR4((" %s ", __FUNCTION__  ));
        //  ETG_TRACE_USR4((" %d", __LINE__ ));
        status = false;
    }
    else
    {
        ETG_TRACE_USR4((" Most recent query exucuted "  ));
    }

    ETG_TRACE_USR4((" After Statement steps size %d ", vItemList.size() ));

    return status  ;
}


/*! ***********************************************************************
*FUNCTION: 		extractFolderRelativePath
*@details:      extract Folder RelativePath
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*5.13.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

string QueryManager::extractFolderRelativePath ( string AbsoluteFolderPath , eSourceType srcType  )
{
    string folderPath = getSourceFactory( srcType )->getAbsoluteFolderPath ();

    int pos =  AbsoluteFolderPath.find ( folderPath );

    AbsoluteFolderPath.erase( pos, folderPath.length() + 1   ); // +1 to remove the slash

    string folderName = AbsoluteFolderPath.substr ( pos );
	
	ETG_TRACE_USR4((" extractFolderRelativePath folder name %s ", folderName.c_str() ));

    return folderName;
}

/*! ***********************************************************************
*FUNCTION: 		prepareQueryFolderNextPrevious
*@details:      prepare Query Folder NextPrevious
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*5.13.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::prepareQueryFolderNextPrevious ( sRequestSongId& vRequestSongId, int&  paramIndex )
{
    bool status = false;
    string absolutePath = getSourceFactory( vRequestSongId.getSourceType() )->getRootFolderPath() + vRequestSongId.getFolderPath();
	if ( ! absolutePath.empty() )        absolutePath.pop_back();
    status = ( sqlite3_bind_text ( getSourceFactory( vRequestSongId.getSourceType() )->getQueryStateMent (), ++paramIndex,   absolutePath.c_str() , strlen (absolutePath .c_str()), SQLITE_TRANSIENT ) == SQLITE_OK ) ? true : false;

    status = ( sqlite3_bind_text ( getSourceFactory( vRequestSongId.getSourceType() )->getQueryStateMent (), ++paramIndex,   absolutePath.c_str() , strlen (absolutePath .c_str()), SQLITE_TRANSIENT ) == SQLITE_OK ) ? true : false;

    status = ( sqlite3_bind_int ( getSourceFactory( vRequestSongId.getSourceType() )->getQueryStateMent () , ++paramIndex, vRequestSongId.getCurrentSongId ()  ) == SQLITE_OK ) ? true : false ;

    status = ( sqlite3_bind_int ( getSourceFactory( vRequestSongId.getSourceType() )->getQueryStateMent () , ++paramIndex, vRequestSongId.getFileType ()  ) == SQLITE_OK ) ? true : false ;


    return status;
}

/*! ***********************************************************************
*FUNCTION: 		prepareQueryFrontRearSong
*@details:      prepare Query Front RearSong
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*5.13.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::prepareQueryFrontRearSong ( sRequestFrontRearSong& vRequestFrontRearSong , int& paramIndex )
{
    bool status = false;
    string absolutePath = getSourceFactory( vRequestFrontRearSong.getSourceType() )->getRootFolderPath() + vRequestFrontRearSong.getFolderPath();
	if ( ! absolutePath.empty() )        absolutePath.pop_back();
    status = ( sqlite3_bind_text ( getSourceFactory( vRequestFrontRearSong.getSourceType() )->getQueryStateMent (), ++paramIndex,   absolutePath.c_str() , strlen (absolutePath .c_str()), SQLITE_TRANSIENT ) == SQLITE_OK ) ? true : false;

    status = ( sqlite3_bind_text ( getSourceFactory( vRequestFrontRearSong.getSourceType() )->getQueryStateMent (), ++paramIndex,   absolutePath.c_str() , strlen (absolutePath .c_str()), SQLITE_TRANSIENT ) == SQLITE_OK ) ? true : false;

    return status;
}

/*! ***********************************************************************
*FUNCTION: 		prepareQueryCompleteList
*@details:      prepare Query Complete List
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*5.30.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::prepareQueryCompleteList ( sReqListInfoMessage& vReqListMessage , int& paramIndex )
{
    bool status = false;
    string absolutePath = getSourceFactory( vReqListMessage.getSourceType() )->getRootFolderPath() + vReqListMessage.getFolderPath();
	
	if ( ! absolutePath.empty() )        absolutePath.pop_back();
	
	getSourceFactory( vReqListMessage.getSourceType() )->setAbsoluteFolderPath ( absolutePath );
	

    status = ( sqlite3_bind_text ( getSourceFactory( vReqListMessage.getSourceType() )->getQueryStateMent (), ++paramIndex,   absolutePath.c_str() , strlen (absolutePath .c_str()), SQLITE_TRANSIENT ) == SQLITE_OK ) ? true : false;

    status = ( sqlite3_bind_text ( getSourceFactory( vReqListMessage.getSourceType() )->getQueryStateMent (), ++paramIndex,   absolutePath.c_str() , strlen (absolutePath .c_str()), SQLITE_TRANSIENT ) == SQLITE_OK ) ? true : false;

    return status;
}

/*! ***********************************************************************
*FUNCTION: 		updateLanguageTable
*@details:      update Language Table
*REQUIREMENT ID:	NA
*@param:		void
*@return: 		void
*@details
*HISTORY:
*3.14.2022		1.0		Monojit Chatterjee		Reviewed
************************************************************************/

bool QueryManager::updateLanguageTable ( eSourceType srcType )
{
	ETG_TRACE_USR4((" updateLanguageTable %d", srcType ));
	return { ( sqlite3_exec(getSourceFactory( srcType )->getDB(), UPDATE_LANGUAGE_TABLE , NULL, NULL, NULL ) == SQLITE_OK ) ? true : false };
	
}


