/****************************************************************************
* Copyright (C) Robert Bosch Car Multimedia GmbH, 2018
* This software is property of Robert Bosch GmbH. Unauthorized
* duplication and disclosure to third parties is prohibited.
*
* FILE                : PluginServerHandler.cpp
* COMPONENT Name      : PluginServer
* DESCRIPTION         : DBus server for plugins
*					         Receives command requests from HMI and sends it to plugin
*					         Receives updates from plugin and sends it to HMI
* AUTHOR              : Preethi Alagappan
* Date                : 14.09.2018
* Revision History    : 0.1
* Date 14.09.2018     : Initial version
* Revision History    : 0.2
* Date 25.10.2018     : Removed condition check for plugin and control block name.
*                       Moved the code, which converts EventDataUtility to Dbus variant
*                       and vice versa, to PluginDataConverter
* Revision History    : 0.3
* Date 15.11.2018     : Implementation for PluginListCommandRequest and PluginListUpdateSignal
****************************************************************************/

/*****************************************************************
| includes
|----------------------------------------------------------------*/
#include "PluginServerHandler.h"
#include "CommandToPlugin.h"
#include "../plugin_trace.h"
#include "../../avdecc_appl_plugins/PluginHelper/clExternalMediaPlayerPlugin.h"
#include "../../avdecc_appl_plugins/PluginHelper/clRouterPlugin.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_APPUTILITIES_PLUGINSERVER
#include "trcGenProj/Header/PluginServerHandler.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN
/*****************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------*/

#define PLUGIN_DBUS_PORTNAME "PluginDbusServerPort"

/************************************************************************
*NAME        : PluginServerHandler
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler
************************************************************************/
PluginServerHandler::PluginServerHandler()
    : PluginDbusInterfaceStub(PLUGIN_DBUS_PORTNAME)
{
   ETG_TRACE_USR4(("PluginServerHandler::PluginServerHandler()"));
   
}

/************************************************************************
*NAME        : ~PluginServerHandler
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler
************************************************************************/
PluginServerHandler::~PluginServerHandler()
{
   ETG_TRACE_USR4(("PluginServerHandler::~PluginServerHandler()"));
}

/************************************************************************
*NAME        : getInstance
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler
************************************************************************/
PluginServerHandler* PluginServerHandler::getInstance()
{
   static PluginServerHandler pluginServerHandler;

   return &pluginServerHandler;
}

/************************************************************************
*NAME        : onPluginCommandRequest
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginServerHandler::onPluginCommandRequest(const ::boost::shared_ptr< PluginCommandRequest >& request)
{
    ETG_TRACE_USR4(("PluginServerHandler::onPluginCommandRequest()"));

   if((NULL != request.get())
      && (true == request->hasPlugin_name()) && (true == request->hasControlblock_name())
      && (true == request->hasData()))
   {
      std::string pluginName    = request->getPlugin_name();
      std::string ctrlBlockName = request->getControlblock_name();


      ETG_TRACE_USR4(("PluginServerHandler::onPluginCommandRequest(), Plugin name: %s\n", pluginName.c_str()));
      ETG_TRACE_USR4(("PluginServerHandler::onPluginCommandRequest(), Control block name : %s\n", ctrlBlockName.c_str()));

      ::asf::dbus::DBusVariant variantData = request->getData();
      boost::shared_ptr<EventDataUtility> eventData;
      bool convertStatus = PluginDataConverter::convertToEventDataUtility(variantData, eventData);
      if(true == convertStatus)
      {
         CommandToPlugin* ptrCmdToPlugin = CommandToPlugin::poGetInstance();
         if(NULL != ptrCmdToPlugin)
         {
            ptrCmdToPlugin->sendEventToPlugin(pluginName, ctrlBlockName, eventData);
         }
      }
      else
      {
         ETG_TRACE_USR4(("PluginServerHandler::onPluginCommandRequest(), Error in reading data from Dbus Variant\n"));
         sendPluginCommandError(DBUS_ERROR_FAILED, "Error in reading data from DBus Variant");
      }
   }
}

/************************************************************************
*NAME        : onPluginListCommandRequest
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginServerHandler::onPluginListCommandRequest(const ::boost::shared_ptr< PluginListCommandRequest >& request)
{
   ETG_TRACE_USR4(("PluginServerHandler::onPluginListCommandRequest()\n"));

   if((NULL != request.get())
      && (true == request->hasPlugin_name()) && (true == request->hasControlblock_name())
      && (true == request->hasData())        && (true == request->hasListData()))
   {
      std::string pluginName    = request->getPlugin_name();
      std::string ctrlBlockName = request->getControlblock_name();
      ::asf::dbus::DBusVariant variantData = request->getData();
      ::asf::dbus::DBusVariant listVariantData = request->getListData();

      ETG_TRACE_USR4(("PluginServerHandler::onPluginListCommandRequest(), Plugin name: %s\n", pluginName.c_str()));
      ETG_TRACE_USR4(("PluginServerHandler::onPluginListCommandRequest(), Control block name : %s\n", ctrlBlockName.c_str()));

      boost::shared_ptr<EventDataUtility> eventData;
      bool convertStatus = PluginDataConverter::convertToEventDataUtility(variantData, eventData);

      boost::shared_ptr<EventListDataUtility> listData;
      convertStatus = convertStatus && PluginDataConverter::convertToEventListDataUtility(listVariantData, listData);

      if(true == convertStatus)
      {
         CommandToPlugin* ptrCmdToPlugin = CommandToPlugin::poGetInstance();
         if(NULL != ptrCmdToPlugin)
         {
            ptrCmdToPlugin->sendListEventToPlugin(pluginName, ctrlBlockName, eventData, listData);
         }
      }
      else
      {
         ETG_TRACE_USR4(("PluginServerHandler::onPluginListCommandRequest(), Error in reading data from Dbus Variant\n"));
         sendPluginListCommandError(DBUS_ERROR_FAILED, "Error in reading data from DBus Variant");
      }
   }
}

/************************************************************************
*NAME        : onPluginUpdateSignal
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginServerHandler::onPluginUpdateSignal
(
   const std::string& pluginName,
   const std::string& ctrlBlockName,
   const boost::shared_ptr<EventDataUtility>& eventData
)
{
   ETG_TRACE_USR4(("PluginServerHandler::onPluginUpdateSignal(), Plugin name : %s\n", pluginName.c_str()));
   ETG_TRACE_USR4(("PluginServerHandler::onPluginUpdateSignal(), Control block name : %s\n", ctrlBlockName.c_str()));

   ::asf::dbus::DBusVariant variantData;
   bool convertStatus = PluginDataConverter::convertToVariant(eventData, variantData);
   if(true == convertStatus)
   {
      sendPluginUpdateSignal(pluginName, ctrlBlockName, variantData);
   }
   else
   {
      ETG_TRACE_USR4(("PluginServerHandler::onPluginUpdateSignal(), Error in writing data into Dbus Variant\n"));
   }
}

/************************************************************************
*NAME        : onPluginListUpdateSignal
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginServerHandler::onPluginListUpdateSignal
(
   const std::string& pluginName,
   const std::string& ctrlBlockName,
   const boost::shared_ptr<EventDataUtility>& eventData,
   const boost::shared_ptr<EventListDataUtility>& listData
)
{
   ETG_TRACE_USR4(("PluginServerHandler::onPluginListUpdateSignal(), Plugin name : %s\n", pluginName.c_str()));
   ETG_TRACE_USR4(("PluginServerHandler::onPluginListUpdateSignal(), Control block name : %s\n", ctrlBlockName.c_str()));

   ::asf::dbus::DBusVariant variantData;
   bool convertStatus = PluginDataConverter::convertToVariant(eventData, variantData);

   ::asf::dbus::DBusVariant listVariantData;
   convertStatus = convertStatus && PluginDataConverter::convertToListVariant(listData, listVariantData);
   if(true == convertStatus)
   {
      sendPluginListUpdateSignal(pluginName, ctrlBlockName, variantData, listVariantData);
   }
   else
   {
      ETG_TRACE_USR4(("PluginServerHandler::onPluginListUpdateSignal(), Error in writing data into Dbus Variant\n"));
   }
}

/************************************************************************
*NAME        : EndStationListUpdateProperty
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginServerHandler::EndStationListUpdateProperty(const boost::shared_ptr<EventDataUtility>& stationListData)
{
   ::asf::dbus::DBusVariant variantData;
   bool convertStatus = PluginDataConverter::convertToVariant(stationListData, variantData);
   if(true == convertStatus)
   {
      ETG_TRACE_USR4(("PluginServerHandler::setEndStationListUpdate(), success in writing data into Dbus Variant\n"));
      setEndStationListUpdate(variantData);
   }
   else
   {
      ETG_TRACE_USR4(("PluginServerHandler::setEndStationListUpdate(), Error in writing data into Dbus Variant\n"));
   }   
}
/************************************************************************
*NAME        : EntityDetailsListUpdateProperty
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginServerHandler::EntityDetailsListUpdateProperty(const boost::shared_ptr<EventListDataUtility>& listData)
{
   ETG_TRACE_USR4(("PluginServerHandler::EntityDetailsListUpdateProperty"));

   if (listData != NULL)
   {
       ::asf::dbus::DBusVariant listVariantData;
       bool convertStatus = PluginDataConverter::convertToListVariant(listData, listVariantData);
       if(true == convertStatus)
       {
          setEntityDetailsUpdate(listVariantData);
       }
       else
       {
          ETG_TRACE_USR4(("PluginServerHandler::EntityDetailsListUpdateProperty(), Error in writing data into Dbus Variant\n"));
       }
   }
   else
   {
       ETG_TRACE_USR4(("PluginServerHandler::EntityDetailsListUpdateProperty(), Listdata is NULL\n"));
   }
}

/************************************************************************
*NAME        : setListAbsoluteTrackIndexProperty
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginServerHandler::setListAbsoluteTrackIndexProperty(int32 listAbsoluteTrackIndex,int16 sourcetype)
{
	ETG_TRACE_USR4(("PluginServerHandler::setListAbsoluteTrackIndexProperty()\n"));
	//setListAbsoluteTrackIndex(listAbsoluteTrackIndex);
}
/************************************************************************
*NAME        : getListAbsoluteTrackIndexProperty
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
int32 PluginServerHandler::getListAbsoluteTrackIndexProperty()
{
	ETG_TRACE_USR4(("PluginServerHandler::getListAbsoluteTrackIndexProperty()\n"));
	//int32 l_listAbsoluteTrackIndex = getListAbsoluteTrackIndex();

	return 0; //l_listAbsoluteTrackIndex;
}

/************************************************************************
*NAME        : onListAbsoluteTrackIndexSet
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler, 8.2.1 - PluginDbusInterface Gets uodated when ListAbsoluteTrackIndex is updated by client
************************************************************************/
void PluginServerHandler::onListAbsoluteTrackIndexSet(const ::boost::shared_ptr< ListAbsoluteTrackIndexSet >& payload)
{
	ETG_TRACE_USR4(("PluginServerHandler::onListAbsoluteTrackIndexSet()\n"));

	//get the value of AbsoluteTrackIndex
	if (NULL != payload.get())
	{
	        ListAbsoluteTrackIndexStruct TrackIndexItems = payload->getListAbsoluteTrackIndex();
		//std::vector<ListAbsoluteTrackIndexStruct>::iterator iter= TrackIndexItems.begin();
		uint32 l_listAbsoluteTrackIndex = TrackIndexItems.getElem1();
		uint16 sourcetype = TrackIndexItems.getElem2();
		uint16 filetype = TrackIndexItems.getElem3();
			ETG_TRACE_USR4(("PluginServerHandler::onListAbsoluteTrackIndexSet() listAbsoluteTrackIndex %d  , Source Type : %d" , l_listAbsoluteTrackIndex,sourcetype));    
			switch (sourcetype)
			{
				case entSourceType_CMP:
					clExternalMediaPlayerPlugin::getInstance()->vSetStreamCurrentTrackIndex(l_listAbsoluteTrackIndex, filetype);
					break;
				case entSourceType_CMR:
					clRouterPlugin::getInstance()->vSetStreamCurrentTrackIndex(l_listAbsoluteTrackIndex, filetype);
					break;
				default:
					break;
			}
		
	}
	else
	ETG_TRACE_USR4(("PluginServerHandler::onListAbsoluteTrackIndexSet NULL"));
}
/************************************************************************
*NAME        : sendSig_DBStatusSignalUpdate
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginServerHandler::sendSig_DBStatusSignalUpdate(uint16_t dbState , uint16_t sourceType)
{
	ETG_TRACE_USR4(("PluginServerHandler::sendSig_DBStatusSignalUpdate() dbState %d sourceType %d " , dbState ,sourceType));
	sendSig_DBStatusSignal(dbState,sourceType);	
}

/************************************************************************
*NAME        : sendSig_PlayStatusSignalUpdate
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginServerHandler::sendSig_PlayStatusSignalUpdate(bool playState,uint16_t sourceType,int songId,uint16_t Filetype,uint16_t Listtype,uint16_t Filter1,uint16_t Filter2,std::string Folderpath)
{
	ETG_TRACE_USR4(("PluginServerHandler::sendSig_PlayStatusSignalUpdate() playState %d sourceType %d songId %d Listtype %d Filter1 %d Filter2 %d folderpath %s " , playState ,sourceType, songId, Listtype, Filter1, Filter2, Folderpath.c_str()));
        Sig_PlayStatusSignalPlayStateStruct PlayStateStruct;
        PlayStateStruct.setElem1(playState);
        PlayStateStruct.setElem2(sourceType);
        PlayStateStruct.setElem3(songId);
        PlayStateStruct.setElem4(Filetype);
        PlayStateStruct.setElem5(Listtype);
        PlayStateStruct.setElem6(Filter1);
        PlayStateStruct.setElem7(Filter2);
        PlayStateStruct.setElem8(Folderpath);
        sendSig_PlayStatusSignal(PlayStateStruct);
} 

/************************************************************************
*NAME        : sig_ReqTrackIndex
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginServerHandler::sendsig_ReqTrackIndex(uint16_t trackIndexAction,uint16_t sourceType,uint16_t ListType,uint16_t Filter1,uint16_t Filter2,uint16_t TrackIndex,uint16_t Region,std::string FolderPath)
{
	ETG_TRACE_USR4(("PluginServerHandler::sig_ReqTrackIndex() trackIndexAction %d  sourceType %d ListType %d Filter1 %d Filter2 %d TrackIndex %d Region %d FolderPath %s " , trackIndexAction, sourceType, ListType, Filter1, Filter2, TrackIndex, Region , FolderPath.c_str()));
        Sig_ReqTrackIndexSignalTrackReqInfoStruct TrackReqInfoStruct;
        TrackReqInfoStruct.setElem1(trackIndexAction);
        TrackReqInfoStruct.setElem2(sourceType);
        TrackReqInfoStruct.setElem3(ListType);
        TrackReqInfoStruct.setElem4(Filter1);
        TrackReqInfoStruct.setElem5(Filter2);
        TrackReqInfoStruct.setElem6(TrackIndex);
        TrackReqInfoStruct.setElem7(Region);
        TrackReqInfoStruct.setElem8(FolderPath);
        sendSig_ReqTrackIndexSignal(TrackReqInfoStruct);
}

/************************************************************************
*NAME        : sendsig_CRCCheckStatusUpdate
*DESCRIPTION : Design section 8.2.4.1 - PluginServerHandler, 8.2.1 - PluginDbusInterface
************************************************************************/
void PluginServerHandler::sendsig_CRCCheckStatusUpdate(bool CRCStatus, uint16_t sourceType)
{
	ETG_TRACE_USR4(("PluginServerHandler::sendsig_CRCCheckStatusUpdate() CRCStatus %d sourceType %d" , CRCStatus , sourceType));
	sendSig_CRCCheckStatusSignal(CRCStatus,sourceType);	
}



