/****************************************************************************
* Copyright (C) Robert Bosch Car Multimedia GmbH, 2018 - 2019
* This software is property of Robert Bosch GmbH. Unauthorized
* duplication and disclosure to third parties is prohibited.
*
* FILE                : EventDataUtility.h
* COMPONENT Name      : EventBus
* DESCRIPTION         : Defines the data backend class that stores the data exchanged between
*						      (a) HMI and PluginClientHandler through courier messages
*					         (b) PluginServerHandler and Plugins through events
* AUTHOR              : Preethi Alagappan
* Date                : 15.10.2018
* Revision History    : 0.1
* Date 15.10.2018     : Initial version
* Revision History    : 0.2
* Date 15.11.2018     : Class EventListDataUtility for adding list data
****************************************************************************/

#ifndef _EVENTDATAUTILITY_H_
#define _EVENTDATAUTILITY_H_

/*****************************************************************
| includes
|----------------------------------------------------------------*/
#include <stdint.h>
#include <string>
#include <vector>

using namespace std;

class EventDataItem
{
public:
  enum DataType { BOOL, UINT8, UINT16, UINT32, UINT64, DOUBLE, STRING, BYTEBUFFER};

  union DataValue
  {
	 bool         _boolValue;
	 uint8_t	     _uint8Value;
	 uint16_t	  _uint16Value;
	 uint32_t	  _uint32Value;
	 uint64_t     _uint64Value;
	 double       _doubleValue;
	 std::string* _stringValue;
	 std::vector<uint8_t>* _byteBufferValue;
  };

  struct Data
  {
	 DataType  _type;
	 DataValue _value;
  };
  
  ~EventDataItem();

  static EventDataItem* newEventDataItem(const bool& value);
  static EventDataItem* newEventDataItem(const uint8_t& value);
  static EventDataItem* newEventDataItem(const uint16_t& value);
  static EventDataItem* newEventDataItem(const uint32_t& value);
  static EventDataItem* newEventDataItem(const uint64_t& value);
  static EventDataItem* newEventDataItem(const double& value);
  static EventDataItem* newEventDataItem(const char* value);
  static EventDataItem* newEventDataItem(const std::string& value);
  static EventDataItem* newEventDataItem(const std::vector<uint8_t>& value);

  const Data& getData() const;

private:
  EventDataItem(){};
  EventDataItem(DataType type, const DataValue& value);

  Data _data;
};

class EventDataUtility
{
public:
   ~EventDataUtility();

   static EventDataUtility* newEventDataUtility();

   template <typename T>
   void addEventData(const T& value)
   {
      _data.push_back(EventDataItem::newEventDataItem(value));
   }

   const std::vector<EventDataItem*>& getData() const;

private:
   EventDataUtility();

   std::vector<EventDataItem*> _data;
};

class EventListDataUtility
{
public:
   ~EventListDataUtility();

   static EventListDataUtility* newEventListDataUtility();
   void addEventListData(const EventDataUtility* eventDataUtil);
   const std::vector<const EventDataUtility*>& getData() const;

private:
   EventListDataUtility();

   std::vector<const EventDataUtility*> _data;
};

#endif
