/****************************************************************************
* Copyright (C) Robert Bosch Car Multimedia GmbH, 2018 - 2019
* This software is property of Robert Bosch GmbH. Unauthorized
* duplication and disclosure to third parties is prohibited.
*
* FILE                : EventDataUtility.cpp
* COMPONENT Name      : EventBus
* DESCRIPTION         : Defines the data backend class that stores the data exchanged between
*						      (a) HMI and PluginClientHandler through courier messages
*					         (b) PluginServerHandler and Plugins through events
* AUTHOR              : Preethi Alagappan
* Date                : 15.10.2018
* Revision History    : 0.1
* Date 15.10.2018     : Initial version
* Revision History    : 0.2
* Date 15.11.2018     : Class EventListDataUtility for adding list data
****************************************************************************/

#include "EventDataUtility.h"
#include "../../plugin_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_APPUTILITIES_EVENTBUS
#include "trcGenProj/Header/EventDataUtility.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

/************************************************************************
*NAME        : ~EventDataItem
*DESCRIPTION : Design section 8.1.1 - EventDataItem
************************************************************************/
EventDataItem::~EventDataItem()
{
   ETG_TRACE_USR4(("EventDataItem::~EventDataItem()"));
   if (EventDataItem::STRING == _data._type)
   {
      if (NULL != _data._value._stringValue)
      {
		   ETG_TRACE_USR4(("Freeing memory for string data"));

         delete _data._value._stringValue;
         _data._value._stringValue = NULL;
      }
   }
   else if (EventDataItem::BYTEBUFFER == _data._type)
   {
      if (NULL != _data._value._byteBufferValue)
      {
         ETG_TRACE_USR4(("Freeing memory for byte buffer data"));

         _data._value._byteBufferValue->clear();
         delete _data._value._byteBufferValue;
         _data._value._byteBufferValue = NULL;
      }
   }
}

/************************************************************************
*NAME        : newEventDataItem(bool)
*DESCRIPTION : Design section 8.1.1 - EventDataItem
************************************************************************/
EventDataItem* EventDataItem::newEventDataItem(const bool& value)
{
   ETG_TRACE_USR4(("EventDataItem::newEventDataItem(BOOL) : %d", value));

   DataValue dataValue;
   dataValue._boolValue = value;

   return new EventDataItem(EventDataItem::BOOL, dataValue);
}

/************************************************************************
*NAME        : newEventDataItem(uint8)
*DESCRIPTION : Design section 8.1.1 - EventDataItem
************************************************************************/
EventDataItem* EventDataItem::newEventDataItem(const uint8_t& value)
{
   ETG_TRACE_USR4(("EventDataItem::newEventDataItem(UINT8) : %d", value));

   DataValue dataValue;
   dataValue._uint8Value = value;

   return new EventDataItem(EventDataItem::UINT8, dataValue);
}

/************************************************************************
*NAME        : newEventDataItem(uint16)
*DESCRIPTION : Design section 8.1.1 - EventDataItem
************************************************************************/
EventDataItem* EventDataItem::newEventDataItem(const uint16_t& value)
{
   ETG_TRACE_USR4(("EventDataItem::newEventDataItem(UINT16) : %d", value));

   DataValue dataValue;
   dataValue._uint16Value = value;

   return new EventDataItem(EventDataItem::UINT16, dataValue);
}

/************************************************************************
*NAME        : newEventDataItem(uint32)
*DESCRIPTION : Design section 8.1.1 - EventDataItem
************************************************************************/
EventDataItem* EventDataItem::newEventDataItem(const uint32_t& value)
{
   ETG_TRACE_USR4(("EventDataItem::newEventDataItem(UINT32) : %d", value));

   DataValue dataValue;
   dataValue._uint32Value = value;

   return new EventDataItem(EventDataItem::UINT32, dataValue);
}

/************************************************************************
*NAME        : newEventDataItem(uint64)
*DESCRIPTION : Design section 8.1.1 - EventDataItem
************************************************************************/
EventDataItem* EventDataItem::newEventDataItem(const uint64_t& value)
{
   ETG_TRACE_USR4(("EventDataItem::newEventDataItem(UINT64) : %d", value));

   DataValue dataValue;
   dataValue._uint64Value = value;

   return new EventDataItem(EventDataItem::UINT64, dataValue);
}

/************************************************************************
*NAME        : newEventDataItem(double)
*DESCRIPTION : Design section 8.1.1 - EventDataItem
************************************************************************/
EventDataItem* EventDataItem::newEventDataItem(const double& value)
{
   ETG_TRACE_USR4(("EventDataItem::newEventDataItem(DOUBLE) : %f", value));

   DataValue dataValue;
   dataValue._doubleValue = value;

   return new EventDataItem(EventDataItem::DOUBLE, dataValue);
}

/************************************************************************
*NAME        : newEventDataItem(const char*)
*DESCRIPTION : Design section 8.1.1 - EventDataItem
************************************************************************/
EventDataItem* EventDataItem::newEventDataItem(const char* value)
{
   ETG_TRACE_USR4(("EventDataItem::newEventDataItem(STRING) : %s", value));

   DataValue dataValue;
   dataValue._stringValue = new string(value);

   return new EventDataItem(EventDataItem::STRING, dataValue);
}

/************************************************************************
*NAME        : newEventDataItem(string)
*DESCRIPTION : Design section 8.1.1 - EventDataItem
************************************************************************/
EventDataItem* EventDataItem::newEventDataItem(const std::string& value)
{
   ETG_TRACE_USR4(("EventDataItem::newEventDataItem(STRING) : %s", value.c_str()));

   DataValue dataValue;
   dataValue._stringValue = new string(value);

   return new EventDataItem(EventDataItem::STRING, dataValue);
}

/************************************************************************
*NAME        : newEventDataItem(vector<uint8>)
*DESCRIPTION : Design section 8.1.1 - EventDataItem
************************************************************************/
EventDataItem* EventDataItem::newEventDataItem(const std::vector<uint8_t>& value)
{
   ETG_TRACE_USR4(("EventDataItem::newEventDataItem(BYTEBUFFER)"));

   DataValue dataValue;
   dataValue._byteBufferValue = new std::vector<uint8_t>(value);

   return new EventDataItem(EventDataItem::BYTEBUFFER, dataValue);
}

/************************************************************************
*NAME        : getData
*DESCRIPTION : Design section 8.1.1 - EventDataItem
************************************************************************/
const EventDataItem::Data& EventDataItem::getData() const
{ 
   ETG_TRACE_USR4(("EventDataItem::getData()"));

   return _data;
}      

/************************************************************************
*NAME        : EventDataItem
*DESCRIPTION : Design section 8.1.1 - EventDataItem
************************************************************************/
EventDataItem::EventDataItem(DataType type, const DataValue& value)
{
   ETG_TRACE_USR4(("EventDataItem::EventDataItem()"));

   _data._type = type;
   _data._value = value;
}

/************************************************************************
*NAME        : EventDataUtility
*DESCRIPTION : Design section 8.1.2 - EventDataUtility
************************************************************************/
EventDataUtility::EventDataUtility()
{
   ETG_TRACE_USR4(("EventDataUtility::EventDataUtility()"));
}

/************************************************************************
*NAME        : ~EventDataUtility
*DESCRIPTION : Design section 8.1.2 - EventDataUtility
************************************************************************/
EventDataUtility::~EventDataUtility()
{
   ETG_TRACE_USR4(("EventDataUtility::~EventDataUtility()"));

   for(int index = 0; index < _data.size(); index++)
   {
      if(NULL != _data[index])
      {
         delete _data[index];
         _data[index] = NULL;
      }
   }
   _data.clear();
}

/************************************************************************
*NAME        : newEventDataUtility
*DESCRIPTION : Design section 8.1.2 - EventDataUtility
************************************************************************/
EventDataUtility* EventDataUtility::newEventDataUtility()
{
   ETG_TRACE_USR4(("EventDataUtility::newEventDataUtility()"));

   return new EventDataUtility();
}

/************************************************************************
*NAME        : getData
*DESCRIPTION : Design section 8.1.2 - EventDataUtility
************************************************************************/
const std::vector<EventDataItem*>& EventDataUtility::getData() const
{
   ETG_TRACE_USR4(("EventDataUtility::getData()"));

   return _data;
}

/************************************************************************
*NAME        : EventListDataUtility
*DESCRIPTION : Design section 8.1.3 - EventListDataUtility
************************************************************************/
EventListDataUtility::EventListDataUtility()
{
   ETG_TRACE_USR4(("EventListDataUtility::EventListDataUtility()"));
}

/************************************************************************
*NAME        : ~EventListDataUtility
*DESCRIPTION : Design section 8.1.3 - EventListDataUtility
************************************************************************/
EventListDataUtility::~EventListDataUtility()
{
   ETG_TRACE_USR4(("EventListDataUtility::~EventListDataUtility()"));

   for(int index = 0; index < _data.size(); index++)
   {
      if(NULL != _data[index])
      {
         delete _data[index];
         _data[index] = NULL;
      }
   }
   _data.clear();
}

/************************************************************************
*NAME        : newEventListDataUtility
*DESCRIPTION : Design section 8.1.3 - EventListDataUtility
************************************************************************/
EventListDataUtility* EventListDataUtility::newEventListDataUtility()
{
   ETG_TRACE_USR4(("EventListDataUtility::newEventListDataUtility()"));

   return new EventListDataUtility();
}

/************************************************************************
*NAME        : addEventListData(const EventDataUtility*)
*DESCRIPTION : Design section 8.1.3 - EventListDataUtility
************************************************************************/
void EventListDataUtility::addEventListData(const EventDataUtility* eventDataUtil)
{
   ETG_TRACE_USR4(("EventListDataUtility::addEventListData(EventDataUtility*) : %p", eventDataUtil));

   _data.push_back(eventDataUtil);
}

/************************************************************************
*NAME        : getData
*DESCRIPTION : Design section 8.1.3 - EventListDataUtility
************************************************************************/
const std::vector<const EventDataUtility*>& EventListDataUtility::getData() const
{
   ETG_TRACE_USR4(("EventListDataUtility::getData()"));

   return _data;
}
