/****************************************************************************
* Copyright (C) Robert Bosch Car Multimedia GmbH, 2018
* This software is property of Robert Bosch GmbH. Unauthorized
* duplication and disclosure to third parties is prohibited.
*
* FILE                : MessageIterWriter.cpp
* COMPONENT Name      : dbus_variant_helper
* DESCRIPTION         : Writes data into variant
* AUTHOR              : Preethi Alagappan
* Date                : 10.09.2018
* Revision History    : 0.1
* Date 10.09.2018     : Initial version
* Revision History    : 0.2
* Date 25.10.2018     : Moved dynamic memory allocation for sub iterator to VariantWriter
*                     : Modified method signature for container related functions
*                     : Added new method append_array_uint8() and append_double()
****************************************************************************/

#include "MessageIterWriter.h"
#include "../plugin_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_APPUTILITIES_DBUSVARIANTHANDLER
#include "trcGenProj/Header/MessageIterWriter.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

/************************************************************************
*NAME        : MessageIterWriter
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
MessageIterWriter::MessageIterWriter()
{
   ETG_TRACE_USR4(("MessageIterWriter::MessageIterWriter()"));

   memset(&memberIter, 0, sizeof(DBusMessageIter));
	iter = &memberIter;
}

/************************************************************************
*NAME        : MessageIterWriter
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
MessageIterWriter::MessageIterWriter(DBusMessageIter *iter)
{
   ETG_TRACE_USR4(("MessageIterWriter::MessageIterWriter(DBusMessageIter *)"));

   memset(&memberIter, 0, sizeof(DBusMessageIter));
	this->iter = iter;
}

/************************************************************************
*NAME        : ~MessageIterWriter
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
MessageIterWriter::~MessageIterWriter()
{
    ETG_TRACE_USR4(("MessageIterWriter::~MessageIterWriter()"));
}

/************************************************************************
*NAME        : getIterator
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
DBusMessageIter* MessageIterWriter::getIterator()
{
    ETG_TRACE_USR4(("MessageIterWriter::getIterator()"));
	return iter;
}

/************************************************************************
*NAME        : append_bool
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_bool(const bool& value)
{
    ETG_TRACE_USR4(("MessageIterWriter::append_bool(), Value : %d", value));
	dbus_bool_t db = value;
	return append_basic(DBUS_TYPE_BOOLEAN, &db);
}

/************************************************************************
*NAME        : append_uint8
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_uint8(const uint8& value)
{
    ETG_TRACE_USR4(("MessageIterWriter::append_uint8(), Value : %d", value));
	return append_basic(DBUS_TYPE_BYTE, &value);
}

/************************************************************************
*NAME        : append_uint16
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_uint16(const uint16& value)
{
   ETG_TRACE_USR4(("MessageIterWriter::append_uint16(), Value : %d", value));
	return append_basic(DBUS_TYPE_UINT16, &value);
}

/************************************************************************
*NAME        : append_uint32
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_uint32(const uint32& value)
{
   ETG_TRACE_USR4(("MessageIterWriter::append_uint32(), Value : %d", value));
	return append_basic(DBUS_TYPE_UINT32, &value);
}

/************************************************************************
*NAME        : append_uint64
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_uint64(const uint64& value)
{
   ETG_TRACE_USR4(("MessageIterWriter::append_uint64(), Value : %d", value));
   return append_basic(DBUS_TYPE_UINT64, &value);
}

/************************************************************************
*NAME        : append_double
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_double(const double& value)
{
   ETG_TRACE_USR4(("MessageIterWriter::append_double(), Value : %f", value));
   return append_basic(DBUS_TYPE_DOUBLE, &value);
}

/************************************************************************
*NAME        : append_string
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_string(const std::string& value)
{
   ETG_TRACE_USR4(("MessageIterWriter::append_string(), Value : %s", value.c_str()));
	const char* ptrValue = value.c_str();
	return append_basic(DBUS_TYPE_STRING, &ptrValue);
}

/************************************************************************
*NAME        : append_array_uint8
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_array_uint8(const uint8** values, int numElems)
{
   ETG_TRACE_USR4(("MessageIterWriter::append_array_uint8()"));
	bool retVal = false;

	MessageIterWriter subIter;
   retVal = new_array(DBUS_TYPE_BYTE_AS_STRING, &subIter);
   if(true == retVal)
   {
      retVal = subIter.append_array(DBUS_TYPE_BYTE, values, numElems);

      // Even if append_array fails, container that was opened is closed.
      // Hence delete_container is called and then retVal is set
      retVal = delete_container(&subIter) && retVal;
   }
	return retVal;
}

/************************************************************************
*NAME        : append_array_uint16
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_array_uint16(const uint16** values, int numElems)
{
   ETG_TRACE_USR4(("MessageIterWriter::append_array_uint16()"));
   bool retVal = false;

   MessageIterWriter subIter;
   retVal = new_array(DBUS_TYPE_UINT16_AS_STRING, &subIter);
   if(true == retVal)
   {
      retVal = subIter.append_array(DBUS_TYPE_UINT16, values, numElems);
      retVal = delete_container(&subIter) && retVal;
   }
   return retVal;
}

/************************************************************************
*NAME        : append_array_uint32
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_array_uint32(const uint32** values, int numElems)
{
   ETG_TRACE_USR4(("MessageIterWriter::append_array_uint32()"));
   bool retVal = false;

   MessageIterWriter subIter;
   retVal = new_array(DBUS_TYPE_UINT32_AS_STRING, &subIter);
   if(true == retVal)
   {
      retVal = subIter.append_array(DBUS_TYPE_UINT32, values, numElems);
      retVal = delete_container(&subIter) && retVal;
   }
   return retVal;
}

/************************************************************************
*NAME        : append_array_uint8
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_array_uint8(const std::vector<uint8>& values)
{
   ETG_TRACE_USR4(("MessageIterWriter::append_array_uint8()"));
   bool retVal = false;

   MessageIterWriter subIter;
   retVal = new_array(DBUS_TYPE_BYTE_AS_STRING, &subIter);
   if(true == retVal)
   {
      for(int index = 0; retVal && (index < values.size()); index++)
      {
         retVal = subIter.append_uint8(values[index]);
      }
      retVal = delete_container(&subIter) && retVal;
   }
   return retVal;
}

/************************************************************************
*NAME        : append_array_uint16
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_array_uint16(const std::vector<uint16>& values)
{
   ETG_TRACE_USR4(("MessageIterWriter::append_array_uint16()"));
   bool retVal = false;

   MessageIterWriter subIter;
   retVal = new_array(DBUS_TYPE_UINT16_AS_STRING, &subIter);
   if(true == retVal)
   {
      for(int index = 0; retVal && (index < values.size()); index++)
      {
         retVal = subIter.append_uint16(values[index]);
      }
      retVal = delete_container(&subIter) && retVal;
   }
   return retVal;
}

/************************************************************************
*NAME        : append_array_uint32
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_array_uint32(const std::vector<uint32>& values)
{
   ETG_TRACE_USR4(("MessageIterWriter::append_array_uint32()"));
   bool retVal = false;

   MessageIterWriter subIter;
   retVal = new_array(DBUS_TYPE_UINT32_AS_STRING, &subIter);
   if(true == retVal)
   {
      for(int index = 0; retVal && (index < values.size()); index++)
      {
         retVal = subIter.append_uint32(values[index]);
      }
      retVal = delete_container(&subIter) && retVal;
   }
   return retVal;
}

/************************************************************************
*NAME        : append_array_string
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_array_string(const std::vector<std::string>& values)
{
   ETG_TRACE_USR4(("MessageIterWriter::append_array_string()"));
   bool retVal = false;

   MessageIterWriter subIter;
   retVal = new_array(DBUS_TYPE_STRING_AS_STRING, &subIter);
   if(true == retVal)
   {
      for(int index = 0; retVal && (index < values.size()); index++)
      {
         retVal = subIter.append_string(values[index]);
      }
      retVal = delete_container(&subIter) && retVal;
   }
   return retVal;
}

/************************************************************************
*NAME        : new_array
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::new_array(const char* signature, MessageIterWriter* subIter)
{
   ETG_TRACE_USR4(("MessageIterWriter::new_array()"));
   bool retVal = false;
	if(NULL != subIter)
	{
		retVal = dbus_message_iter_open_container(iter, DBUS_TYPE_ARRAY, signature, subIter->getIterator());
	}
	return retVal;
}

/************************************************************************
*NAME        : new_variant
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::new_variant(const char* signature, MessageIterWriter* subIter)
{
   ETG_TRACE_USR4(("MessageIterWriter::new_variant()"));
   bool retVal = false;
	if(NULL != subIter)
	{
	   retVal = dbus_message_iter_open_container(iter, DBUS_TYPE_VARIANT, signature, subIter->getIterator());
	}
	return retVal;
}

/************************************************************************
*NAME        : new_struct
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::new_struct(MessageIterWriter* subIter)
{
   ETG_TRACE_USR4(("MessageIterWriter::new_struct()"));
   bool retVal = false;
	if(NULL != subIter)
	{
	   retVal = dbus_message_iter_open_container(iter, DBUS_TYPE_STRUCT, NULL, subIter->getIterator());
	}
	return retVal;
}

/************************************************************************
*NAME        : new_dict_entry
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::new_dict_entry(MessageIterWriter* subIter)
{
  ETG_TRACE_USR4(("MessageIterWriter::new_dict_entry()"));
   bool retVal = false;
	if(NULL != subIter)
	{
	   retVal = dbus_message_iter_open_container(iter, DBUS_TYPE_DICT_ENTRY, NULL, subIter->getIterator());
	}
	return retVal;
}

/************************************************************************
*NAME        : delete_container
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::delete_container(MessageIterWriter* subIter)
{
   ETG_TRACE_USR4(("MessageIterWriter::delete_container()"));
   bool retVal = false;
	if(NULL != subIter)
	{
	   retVal = dbus_message_iter_close_container(iter, subIter->getIterator());
	}
	return retVal;
}

/************************************************************************
*NAME        : append_basic
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_basic(int type_id, const void* value)
{
    ETG_TRACE_USR4(("MessageIterWriter::append_basic()"));
	return dbus_message_iter_append_basic(iter, type_id, value);
}

/************************************************************************
*NAME        : append_array
*DESCRIPTION : Design section 8.2.2.1 - MessageIterWriter
************************************************************************/
bool MessageIterWriter::append_array(int type_id, const void* values, int numElems)
{
    ETG_TRACE_USR4(("MessageIterWriter::append_array()"));
	return dbus_message_iter_append_fixed_array(iter, type_id, values, numElems);
}

