/************************************************************************
 * FILE:         scdDebug.h
 * PROJECT:      VASCO
 * SW-COMPONENT: 
 *----------------------------------------------------------------------
 * DESCRIPTION:  scd debug module header 
 *----------------------------------------------------------------------
 * COPYRIGHT:    (c) 2001 Robert Bosch GmbH, Hildesheim
 * HISTORY:      
 * Date      | Author   | Modification
 * 19.06.01  | KNM2HI   | initial revision
 *************************************************************************/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"
#define ERR_S_IMPORT_INTERFACE_ERRMEM
#include "err_if.h"

static OSAL_tIODescriptor oTraceDesc = OSAL_ERROR;

tVoid scd_vInitTrace( tVoid )
{
   if ( oTraceDesc == OSAL_ERROR )
   {
      // create trace-device
      oTraceDesc = OSAL_IOCreate(OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE);

      if ( oTraceDesc == OSAL_ERROR )
      {
         // impossible to create trace device -> already opened?
         oTraceDesc = OSAL_IOOpen(OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE);

         // if ( oTraceDesc == OSAL_ERROR ) -> no traces
      }
   } // else: already opened
}

tVoid scd_vCloseTrace( tVoid )
{
   // trace opened?
   if ( OSAL_ERROR != oTraceDesc )
   {
      // close trace-device
      (tVoid)OSAL_s32IOClose(oTraceDesc);
      oTraceDesc = OSAL_ERROR;
   }
}

/*******************************************************************************
 *FUNCTION:    scd_bIsTraceClassActive
 *DESCRIPTION: checks, if hand over trace class and trace level is active for trace output
 *PARAMETER:   TR_tenTraceClass  enTraceClass   ( ->I )
 *                trace class according to trace concept
 *             TR_tenTraceLevel enTraceLevel   ( ->I )
 *                trace level according to trace concept
 *                
 *RETURNVALUE: tBool 
 *
 *HISTORY:
 *26.02.2003 Initial Version  
 ******************************************************************************/
tBool scd_bIsTraceClassActive(TR_tenTraceLevel enTraceLevel )
{
   tBool bReturnValue = FALSE;
   OSAL_trIOCtrlActivTrace rIOCtrlActivTrace;

   rIOCtrlActivTrace.enTraceClass = TR_CLASS_SCD;
   rIOCtrlActivTrace.enTraceLevel = enTraceLevel;
   rIOCtrlActivTrace.bIsActive = FALSE;

   if ( OSAL_ERROR != oTraceDesc )
   {
      tS32 s32Error = OSAL_s32IOControl(oTraceDesc, OSAL_C_S32_IOCTRL_ISACTIVE, (intptr_t)&rIOCtrlActivTrace);
      if ( (OSAL_OK == s32Error) && (TRUE == rIOCtrlActivTrace.bIsActive) )
      {
         bReturnValue = TRUE;
      }
   }

   return (bReturnValue);
}


// trace formatted string
tVoid scd_vTraceMsg(TR_tenTraceLevel enTraceLevel, const tChar *coszFormatString, ...)
{
   // trace opened?
   if ( (OSAL_ERROR != oTraceDesc) && (scd_bIsTraceClassActive(enTraceLevel) == TRUE) )
   {
      // buffer for trace-string
      tChar acBuffer[SCD_DBG_MAXLINE_LEN + 1] = "";
      // get arguments
      OSAL_tVarArgList ArgList;
      OSAL_VarArgStart(ArgList, coszFormatString); //lint !e530
      // print string with arguments
      (tVoid)OSAL_s32VarNPrintFormat(acBuffer, SCD_DBG_MAXLINE_LEN, coszFormatString, ArgList);//lint !e530
      OSAL_VarArgEnd(ArgList);
      tU32 u32NumOfBytes = (tU32)OSAL_u32StringLength(acBuffer);
      // trace string
      (tVoid)OSALUTIL_s32TraceWrite(oTraceDesc, (tU32)enTraceLevel, (tU32)TR_CLASS_SCD, (tPCS8)acBuffer, u32NumOfBytes);	  
   } // else: trace-device not opened -> no trace
}

// trace formatted string
tVoid scd_vTraceMsgErrmem(TR_tenTraceLevel enTraceLevel, const tChar *coszFormatString, ...)
{
   // trace opened?
   if ( (OSAL_ERROR != oTraceDesc) && (scd_bIsTraceClassActive(enTraceLevel) == TRUE) )
   {
      // buffer for trace-string
      tChar acBuffer[SCD_DBG_MAXLINE_LEN + 1] = "";
      // get arguments
      OSAL_tVarArgList ArgList;
      OSAL_VarArgStart(ArgList, coszFormatString); //lint !e530
      // print string with arguments
      (tVoid)OSAL_s32VarNPrintFormat(acBuffer, SCD_DBG_MAXLINE_LEN, coszFormatString, ArgList);//lint !e530
      OSAL_VarArgEnd(ArgList);
      tU32 u32NumOfBytes = (tU32)OSAL_u32StringLength(acBuffer);
      // trace string
      (tVoid)OSALUTIL_s32TraceWrite(oTraceDesc, (tU32)enTraceLevel, (tU32)TR_CLASS_SCD, (tPCS8)acBuffer, u32NumOfBytes);
	  
	if (enTraceLevel == TR_LEVEL_FATAL)
	{
	   err_tclErrMem oErrMem;
	   oErrMem.s32WriteErrorString(eErrmemEntryInfo, 
								   (tU16)TR_CLASS_SCD & (tU16)0xFF00 /*u16TraceComp*/, 
								   0 /*u8SpecificErrorCode*/, 
								   acBuffer);

	}	  
   }
   else
   {
	NORMAL_M_ASSERT_ALWAYS();
   }
}



