/************************************************************************
 * FILE:         regRegistry.cpp
 * PROJECT:      VASCO
 * SW-COMPONENT: REG
 *----------------------------------------------------------------------
 * DESCRIPTION:  Encapsulation of registry commands
 *
 *----------------------------------------------------------------------
 * COPYRIGHT:    (c) 2000 Robert Bosch GmbH, Hildesheim
 * HISTORY:
 * Date      |   Modification                         | Author
 * 02.04.01  |   Initial revision                     | Bruns,EFG32
 * 28.06.01  |   ported to OSAL registry              | Scherber,ENT1
 * 29.06.01  |   added "bGetLastKeyName"
 *               set "bSetAt" to protected            | Bruns,EFG32
 *************************************************************************/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"


// default constructor
reg_tclRegKey::reg_tclRegKey() : fd(OSAL_ERROR)
{
   szCurrentKeyName[0] = '\0';
}


// constructor
reg_tclRegKey::reg_tclRegKey(const tChar *coszHKeyQualifier) : reg_ifRegKey(coszHKeyQualifier), fd(OSAL_ERROR)
{
   tU32 u32StringLength = (tU32)OSAL_u32StringLength(coszHKeyQualifier);
   u32StringLength = u32StringLength>(REG_C_U32_KEYNAME_MAXLEN-1)?(REG_C_U32_KEYNAME_MAXLEN-1):u32StringLength;
   OSAL_szStringNCopy(szCurrentKeyName, coszHKeyQualifier, u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
   szCurrentKeyName[u32StringLength]='\0';
}

// copy constructor
reg_tclRegKey::reg_tclRegKey (const reg_tclRegKey &roSource) : reg_ifRegKey(), fd(OSAL_ERROR)
{
   tU32 u32StringLength = (tU32)OSAL_u32StringLength(roSource.cpGetClassKeyName());
   u32StringLength = u32StringLength>(REG_C_U32_KEYNAME_MAXLEN-1)?(REG_C_U32_KEYNAME_MAXLEN-1):u32StringLength;
   OSAL_szStringNCopy(szCurrentKeyName, roSource.cpGetClassKeyName(), u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
   szCurrentKeyName[u32StringLength]='\0';
}

tBool reg_tclRegKey::bSetCurrentKeyName (const tChar *pcNewKey)
{
   tBool bRetVal = TRUE;
   vClose();

   if(pcNewKey != NULL)
   {
      tU32 u32StringLength = (tU32)OSAL_u32StringLength(pcNewKey);
      u32StringLength = u32StringLength>(REG_C_U32_KEYNAME_MAXLEN-1)?(REG_C_U32_KEYNAME_MAXLEN-1):u32StringLength;
      OSAL_szStringNCopy(szCurrentKeyName, pcNewKey, u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
      szCurrentKeyName[u32StringLength]='\0';
   }
   else
   {
      szCurrentKeyName[0]='\0';
      bRetVal = FALSE;
   }

   return bRetVal;
}

// destructor
reg_tclRegKey::~reg_tclRegKey()
{
   reg_tclRegKey::vClose(); //lint !e1551: Function may throw exception
}


reg_tclRegKey& reg_tclRegKey::operator=(const reg_tclRegKey& roSource)
{
   if ( this != &roSource )
   {
      vClose();

      tU32 u32StringLength = (tU32)OSAL_u32StringLength(roSource.cpGetClassKeyName());
      u32StringLength = u32StringLength>(REG_C_U32_KEYNAME_MAXLEN-1)?(REG_C_U32_KEYNAME_MAXLEN-1):u32StringLength;
      OSAL_szStringNCopy(szCurrentKeyName, roSource.cpGetClassKeyName(), u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
      szCurrentKeyName[u32StringLength]='\0';
   }

   return (*this);
}


tBool reg_tclRegKey::bCreate(const tChar *coszKeyName)
{
   vClose();
   fd = OSAL_IOCreate(coszKeyName, OSAL_EN_READWRITE);
   if (fd != OSAL_ERROR)
   {
      tU32 u32StringLength = (tU32)OSAL_u32StringLength(coszKeyName);
      u32StringLength = u32StringLength>(REG_C_U32_KEYNAME_MAXLEN-1)?(REG_C_U32_KEYNAME_MAXLEN-1):u32StringLength;
      OSAL_szStringNCopy(szCurrentKeyName, coszKeyName, u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
      szCurrentKeyName[u32StringLength]='\0';
      return TRUE;
   }
   else
   {
      return FALSE;
   }
}

tBool reg_tclRegKey::bOpenLocal()
{
   tBool bRetVal = TRUE;

   if(fd == OSAL_ERROR)
   {
      fd = OSAL_IOOpen(szCurrentKeyName, OSAL_EN_READWRITE);

      if(fd == OSAL_ERROR)
      {
         bRetVal = FALSE;
      }
   }

   return bRetVal;
}

tBool reg_tclRegKey::bOpen(const tChar *coszAbsKeyName)
{
   vClose();
   fd = OSAL_IOOpen(coszAbsKeyName, OSAL_EN_READWRITE);
   if (fd != OSAL_ERROR)
   {
      tU32 u32StringLength = (tU32)OSAL_u32StringLength(coszAbsKeyName);
      u32StringLength = u32StringLength>(REG_C_U32_KEYNAME_MAXLEN-1)?(REG_C_U32_KEYNAME_MAXLEN-1):u32StringLength;
      OSAL_szStringNCopy(szCurrentKeyName, coszAbsKeyName, u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
      szCurrentKeyName[u32StringLength]='\0';
      return TRUE;
   }
   else
   {
      return FALSE;
   }

}

tBool  reg_tclRegKey::bOpen(const tChar *coszParentName, const tChar *coszSubKeyName)
{
   if (  (OSAL_u32StringLength(coszParentName) + OSAL_u32StringLength(coszSubKeyName) + 2)
      >  REG_C_U32_KEYNAME_MAXLEN ) // +2: plus separator plus zero-terminator
   {
      // key-name is too long
      return FALSE;
   }
   else
   {
      tChar  szTempName[REG_C_U32_KEYNAME_MAXLEN];

      if (OSALUTIL_s32SaveNPrintFormat((tString)szTempName,
                                       sizeof(szTempName),
                                       "%s/%s", 
                                       coszParentName, 
                                       coszSubKeyName) != OSAL_ERROR)
      {
        return bOpen(szTempName);
      }
      else
      {
        NORMAL_M_ASSERT_ALWAYS();
        return FALSE;
      }
   }
}

tBool reg_tclRegKey::bOpen(const reg_ifRegKey &roSource)
{
   return bOpen((const tChar*)roSource.cpGetClassKeyName());
}

tBool reg_tclRegKey::bOpen(const reg_ifRegKey &roSource, const tChar *coszSubKeyName)
{
   if (  (OSAL_u32StringLength(roSource.cpGetClassKeyName()) + OSAL_u32StringLength(coszSubKeyName) + 2)
      >  REG_C_U32_KEYNAME_MAXLEN ) // +2: plus separator plus zero-terminator
   {
      // key-name is too long
      return FALSE;
   }
   else
   {
      tChar  szTempName[REG_C_U32_KEYNAME_MAXLEN];

      if (OSALUTIL_s32SaveNPrintFormat((tString)szTempName, 
                                       sizeof(szTempName),
                                       "%s/%s", 
                                       roSource.cpGetClassKeyName(),
                                       coszSubKeyName) != OSAL_ERROR)
      {
        return bOpen(szTempName);
      }
      else
      {
        NORMAL_M_ASSERT_ALWAYS();
        return FALSE;
      }
   }
}

tBool reg_tclRegKey::bRelOpen(const tChar *coszRelKeyName)
{
   if (fd == OSAL_ERROR)
      return FALSE;

   if (  (OSAL_u32StringLength(szCurrentKeyName) + OSAL_u32StringLength(coszRelKeyName) + 2)
      >  REG_C_U32_KEYNAME_MAXLEN ) // +2: plus separator plus zero-terminator
   {
      // key-name is too long
      return FALSE;
   }
   else
   {
      tChar  szTempName[REG_C_U32_KEYNAME_MAXLEN];

      if (OSALUTIL_s32SaveNPrintFormat((tString)szTempName, 
                                       sizeof(szTempName),
                                       "%s/%s", 
                                       szCurrentKeyName, 
                                       coszRelKeyName) != OSAL_ERROR)
      {
        vClose();
        fd = OSAL_IOOpen(szTempName, OSAL_EN_READWRITE);
        if (fd != OSAL_ERROR)
        {
           tU32 u32StringLength = (tU32)OSAL_u32StringLength(szTempName);
           u32StringLength = u32StringLength>(REG_C_U32_KEYNAME_MAXLEN-1)?(REG_C_U32_KEYNAME_MAXLEN-1):u32StringLength;
           OSAL_szStringNCopy(szCurrentKeyName, szTempName, u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
           szCurrentKeyName[u32StringLength]='\0';
           return TRUE;
        }
        else
        {
           return FALSE;
        }
      }
      else
      {
        NORMAL_M_ASSERT_ALWAYS();
        return FALSE;
      }
   }
}

tBool reg_tclRegKey::bRelCreate(const tChar *coszRelKeyName)
{
   bOpenLocal();

   if (fd == OSAL_ERROR)
      return FALSE;

   if (  (OSAL_u32StringLength(szCurrentKeyName) + OSAL_u32StringLength(coszRelKeyName) + 2)
      >  REG_C_U32_KEYNAME_MAXLEN ) // +2: plus separator plus zero-terminator
   {
      // key-name is too long
      return FALSE;
   }
   else
   {
      tChar  szTempName[REG_C_U32_KEYNAME_MAXLEN];

      if (OSALUTIL_s32SaveNPrintFormat((tString)szTempName, 
                                       sizeof(szTempName),
                                       "%s/%s", 
                                       szCurrentKeyName, 
                                       coszRelKeyName) != OSAL_ERROR)
      {
        vClose();
        fd = OSAL_IOCreate(szTempName, OSAL_EN_READWRITE);
        if (fd != OSAL_ERROR)
        {
           tU32 u32StringLength = (tU32)OSAL_u32StringLength(szTempName);
           u32StringLength = u32StringLength>(REG_C_U32_KEYNAME_MAXLEN-1)?(REG_C_U32_KEYNAME_MAXLEN-1):u32StringLength;
           OSAL_szStringNCopy(szCurrentKeyName, szTempName, u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
           szCurrentKeyName[u32StringLength]='\0';
           return TRUE;
        }
        else
        {
           return FALSE;
        }
      }
      else
      {
        NORMAL_M_ASSERT_ALWAYS();
        return FALSE;
      }
   }
}

//-----------------------------------------------------------------------------
// vClose
//
//-----------------------------------------------------------------------------
tVoid reg_tclRegKey::vClose()
{
   if (fd != OSAL_ERROR)
   {
      OSAL_s32IOClose(fd); //lint !e522: Expected void type, assignment, increment or decrement
      fd = OSAL_ERROR;
   }
   szCurrentKeyName[0] = '\0';
}



//-----------------------------------------------------------------------------
// bSetU32
//
//-----------------------------------------------------------------------------
tBool reg_tclRegKey::bSetU32(const tChar *coszKeyName, tU32 u32NewValue)
{
   bOpenLocal();

   if (fd == OSAL_ERROR)
      return FALSE;

   OSAL_trIOCtrlRegistry registry;
   registry.s32Type = OSAL_C_S32_VALUE_S32;
   registry.pcos8Name = (tPCS8)coszKeyName;
   registry.ps8Value = (tPU8) &u32NewValue; // Prima, ps8Value ist in Wirklichkeit ein tPU8
   registry.u32Size = 4;

   if (OSAL_OK == OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_REGSETVALUE, (intptr_t) &registry))
      return TRUE;

   return FALSE;
}


//-----------------------------------------------------------------------------
// bSetString
//
//-----------------------------------------------------------------------------
tBool reg_tclRegKey::bSetString(const tChar *coszKeyName, tChar *szString, tU32 u32StringLen)
{
   bOpenLocal();

   if (fd == OSAL_ERROR)
      return FALSE;

   OSAL_trIOCtrlRegistry registry;
   registry.s32Type = OSAL_C_S32_VALUE_STRING;
   registry.pcos8Name = (tPCS8)coszKeyName;
   registry.ps8Value = (tPU8) szString; // Prima, ps8Value ist in Wirklichkeit ein tPU8
   registry.u32Size = u32StringLen;

   if (OSAL_OK == OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_REGSETVALUE, (intptr_t) &registry))
      return TRUE;

   return FALSE;
}


//-----------------------------------------------------------------------------
// bQueryU32
//
//-----------------------------------------------------------------------------
tBool reg_tclRegKey::bQueryU32(const tChar *coszKeyName, tU32 *rdwResult)
{
   bOpenLocal();

   if (fd == OSAL_ERROR)
      return FALSE;

   OSAL_trIOCtrlRegistry registry;
   registry.pcos8Name = (tPCS8)coszKeyName;
   registry.ps8Value = (tPU8) rdwResult; // Prima, ps8Value ist in Wirklichkeit ein tPU8
   registry.u32Size = 4;

   if (OSAL_OK != OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_REGQUERYVALUE, (intptr_t) &registry))
      return FALSE;
   if (registry.s32Type != OSAL_C_S32_VALUE_S32)
      return FALSE;

   return TRUE;

}


//-----------------------------------------------------------------------------
// bQueryString
//
//-----------------------------------------------------------------------------
tBool reg_tclRegKey::bQueryString(const tChar *coszKeyName, tChar *rdwResult, tU32 u32MaxBufLen)
{
   bOpenLocal();

   if (fd == OSAL_ERROR)
      return FALSE;

   OSAL_trIOCtrlRegistry registry;
   registry.pcos8Name = (tPCS8)coszKeyName;
   registry.ps8Value = (tPU8) rdwResult; // Prima, ps8Value ist in Wirklichkeit ein tPU8
   registry.u32Size = u32MaxBufLen;

   if (OSAL_OK != OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_REGQUERYVALUE, (intptr_t) &registry))
      return FALSE;
   if (registry.s32Type != OSAL_C_S32_VALUE_STRING)
      return FALSE;

   return TRUE;
}


tBool reg_tclRegKey::bGetCurrentKeyName(tChar *pcBuffer, tU32 u32MaxLen) const
{
   if(szCurrentKeyName[0] == 0)
      return FALSE;

   tU32 u32StringLength = (tU32)OSAL_u32StringLength(szCurrentKeyName);
   u32StringLength = u32StringLength>(u32MaxLen-1)?(u32MaxLen-1):u32StringLength;
   OSAL_szStringNCopy(pcBuffer, szCurrentKeyName, u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
   pcBuffer[u32StringLength]='\0';
   return TRUE;
}


tBool reg_tclRegKey::bGetLastKeyName(tChar *pcBuffer, tU32 u32MaxLen) const
{
   if(bGetCurrentKeyName(pcBuffer, u32MaxLen))
   {
      tChar * pcSplit;
      pcSplit = (tPChar)OSAL_ps8StringRSearchChar(pcBuffer, '/');
      if(pcSplit != OSAL_NULL)
      {
         // use temporary name !
         pcSplit++;
         tChar  szTempName[REG_C_U32_KEYNAME_MAXLEN];

         tU32 u32StringLength = (tU32)OSAL_u32StringLength(pcSplit);
         u32StringLength = u32StringLength>(REG_C_U32_KEYNAME_MAXLEN-1)?(REG_C_U32_KEYNAME_MAXLEN-1):u32StringLength;
         OSAL_szStringNCopy(szTempName, pcSplit, u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
         szTempName[u32StringLength]='\0';

         u32StringLength = u32StringLength>(u32MaxLen-1)?(u32MaxLen-1):u32StringLength;
         OSAL_szStringNCopy(pcBuffer, szTempName, u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
         pcBuffer[u32StringLength]='\0';

         return TRUE;
      }
   }
   return FALSE;
}

// navigates the registry path one Kep up (to the root)
tBool reg_tclRegKey::bNavigateKeyNameUp ()
{
   tBool bResult = FALSE;

   tChar* pcEnd;
   pcEnd = (tChar*)OSAL_ps8StringRSearchChar(szCurrentKeyName, '/');
   if(pcEnd != OSAL_NULL)
   {
      // close only without throwing the keyname away
      if (fd != OSAL_ERROR)
      {
         OSAL_s32IOClose(fd); //lint !e522: Expected void type, assignment, increment or decrement
         fd = OSAL_ERROR;
      }

      // remove last keyname
      *pcEnd = 0;
      bResult = TRUE;
   }

   return bResult;
}

tBool reg_tclRegKey::bIsValid()
{
   bOpenLocal();
   return ((fd == OSAL_ERROR) ? FALSE : TRUE);
}

reg_tclValueIterator::reg_tclValueIterator(const reg_tclRegKey& poBaseKey)
: pDirent(OSAL_NULL)
{
   /* Open directory */
   pDir = OSALUTIL_prOpenDir(poBaseKey.cpGetClassKeyName());
   if (pDir)
   {
      tU32 u32StringLength = (tU32)OSAL_u32StringLength(poBaseKey.cpGetClassKeyName());
      u32StringLength = u32StringLength>(REG_C_U32_KEYNAME_MAXLEN-1)?(REG_C_U32_KEYNAME_MAXLEN-1):u32StringLength;
      OSAL_szStringNCopy(szCurrentKeyName, poBaseKey.cpGetClassKeyName(), u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
      szCurrentKeyName[u32StringLength]='\0';
      this->vNext();
   }
   else
      szCurrentKeyName[0] = '\0';

}

reg_tclValueIterator::reg_tclValueIterator(const tChar *coszKeyName)
: pDirent(OSAL_NULL)
{
   /* Open directory */
   pDir = OSALUTIL_prOpenDir(coszKeyName);
   if (pDir)
   {
      tU32 u32StringLength = (tU32)OSAL_u32StringLength(coszKeyName);
      u32StringLength = u32StringLength>(REG_C_U32_KEYNAME_MAXLEN-1)?(REG_C_U32_KEYNAME_MAXLEN-1):u32StringLength;
      OSAL_szStringNCopy(szCurrentKeyName, coszKeyName, u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
      szCurrentKeyName[u32StringLength]='\0';
      this->vNext();
   }
   else
      szCurrentKeyName[0] = '\0';
}


reg_tclValueIterator::~reg_tclValueIterator()
{
   if (pDir)
   {
      OSALUTIL_s32CloseDir(pDir); //lint !e522: Expected void type, assignment, increment or decrement
      pDir = OSAL_NULL;
      szCurrentKeyName[0] = '\0';
   }
   pDirent = OSAL_NULL;
}


reg_tclValueIterator& reg_tclValueIterator::operator++()
{
   this->vNext();
   return *this;
}


tChar* reg_tclValueIterator::operator*()
{
   return this->szValueName();
}

tVoid  reg_tclValueIterator::vFirst(tVoid)
{
   if (pDir)
   {
      OSALUTIL_s32CloseDir(pDir); //lint !e522: Expected void type, assignment, increment or decrement
      pDir = OSAL_NULL;
   }
   pDir = OSALUTIL_prOpenDir(szCurrentKeyName);
   this->vNext();

}

tVoid reg_tclValueIterator::vNext()
{
   if (pDir)
   {
      if (OSAL_ERROR == OSAL_s32IOControl(pDir->fd, OSAL_C_S32_IOCTRL_REGENUMVALUE, (intptr_t)pDir))
      {
         OSALUTIL_s32CloseDir(pDir); //lint !e522: Expected void type, assignment, increment or decrement
         pDir = OSAL_NULL;
      }
   }
}


tBool reg_tclValueIterator::bIsDone(tVoid) const
{
   if (pDir == OSAL_NULL)
   {
      return TRUE;
   }
   else
      return FALSE;
}

tChar *reg_tclValueIterator::szValueName(tVoid)
{
   if (pDir)
      return (tPChar)(pDir->dirent.s8Name);
   return OSAL_NULL;
}

//-----------------------------------------------------------------------------
// bGetU32
//
//-----------------------------------------------------------------------------
tBool reg_tclValueIterator::bGetU32(tU32 *rdwResult) const
{
   if (pDir == OSAL_NULL)
      return FALSE;

   OSAL_trIOCtrlRegistry registry;
   registry.pcos8Name = pDir->dirent.s8Name;
   registry.ps8Value = (tPU8) rdwResult; // Prima, ps8Value ist in Wirklichkeit ein tPU8
   registry.u32Size = 4;

   if (OSAL_OK != OSAL_s32IOControl(pDir->fd, OSAL_C_S32_IOCTRL_REGQUERYVALUE, (intptr_t) &registry))
      return FALSE;
   if (registry.s32Type != OSAL_C_S32_VALUE_S32)
      return FALSE;

   return TRUE;

}


//-----------------------------------------------------------------------------
// bGetString
//
//-----------------------------------------------------------------------------
tBool reg_tclValueIterator::bGetString(tChar *rdwResult, tU32 u32MaxBufLen) const
{
   if (pDir == OSAL_NULL)
      return FALSE;

   OSAL_trIOCtrlRegistry registry;
   registry.pcos8Name = pDir->dirent.s8Name;
   registry.ps8Value = (tPU8) rdwResult; // Prima, ps8Value ist in Wirklichkeit ein tPU8
   registry.u32Size = u32MaxBufLen;

   if (OSAL_OK != OSAL_s32IOControl(pDir->fd, OSAL_C_S32_IOCTRL_REGQUERYVALUE, (intptr_t) &registry))
      return FALSE;
   if (registry.s32Type != OSAL_C_S32_VALUE_STRING)
      return FALSE;

   return TRUE;
}
