/******************************************************************************

  FILE:           ErrMemBase.cpp
 
  SW-COMPONENT:   err
  
  AUTHOR:         CM-DI/ESP4 Wiedemann

  COPYRIGHT:      (c) 2006 Blaupunkt GmbH

******************************************************************************/

// includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define ERR_S_IMPORT_INTERFACE_ERRMEM
#include "err_if.h"

#define TRACE_S_IMPORT_INTERFACE_TYPES
#include "trace_if.h"
/*---------------------------------------------------------------------*//**
 * delimits the given data size to a data size which can be handled by the
 * error memory
 *
 * @param u16DataSize  given data size
 * 
 * @return delimited data size
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    20.01.2005
 *//*---------------------------------------------------------------------*/
tU16 err_tclErrMemBase::u16DelimitSize(tU16 u16DataSize) const
{
   // delimitate size to maximum buffer size minus one byte for string delimiter
   return (u16DataSize > (ERRMEM_MAX_ENTRY_LENGTH - 1)) ? (tU16)(ERRMEM_MAX_ENTRY_LENGTH - 1) : u16DataSize;
}


/*---------------------------------------------------------------------*//**
 * copy error data to target buffer. Thereby the method is limiting the 
 * data and adding a string delimiter (0 byte at the end of the error buffer.
 *
 * @param pu8Dest     pointer to destination buffer
 * @param pu8Source   pointer to source buffer
 * @param u16Length   Length of source buffer
 * 
 * @return number of copied bytes
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    20.01.2005
 *//*---------------------------------------------------------------------*/
tU16 err_tclErrMemBase::u16FillData(tU8* pu8Dest, const tU8* pu8Source, tU16 u16Length) const
{
   tU16 u16Size;

   if(NULL != pu8Dest)
   {
      if(NULL != pu8Source)
      {
         #if ((OSAL_OS == OSAL_TENGINE) || (OSAL_OS == OSAL_LINUX) || (OSAL_OS==OSAL_DARWIN))
            // delimit data size to maximum buffer size
            u16Size = u16DelimitSize((tU16)(u16Length + 2U));
            // copy data if source and destination pointer valid
            OSAL_pvMemoryCopy(pu8Dest + 2, pu8Source, u16Size - 2);
            pu8Dest[0] = (tU8) (((tU16)TR_CLASS_ERR) & 0xFF);
            pu8Dest[1] = (tU8) (((tU16)TR_CLASS_ERR) >> 8) & 0xFF;
         #else
            // delimit data size to maximum buffer size
            u16Size = u16DelimitSize(u16Length);
            // copy data if source and destination pointer valid
            OSAL_pvMemoryCopy(pu8Dest, pu8Source, u16Size); 
         #endif // OSAL_OS==OSAL_TENGINE or OSAL_LINUX
      }
      else
      {
         // otherwise set size to 0 (no bytes copied)
         u16Size = 0;
      }

      // ensure delimited string
      pu8Dest[u16Size] = 0;
   }
   else
   {
      u16Size = 0;
   }

   // return number of copied bytes
   return u16Size;
}


/*---------------------------------------------------------------------*//**
 * fills a time structure with current osal time. The time buffer will be 
 * zeroed on errors.
 *
 * @param rfoTimeDate  buffer for current time
 * 
 * @author  CM-DI/ESP4-Wiedemann
 * @date    20.01.2005
 *//*---------------------------------------------------------------------*/
void err_tclErrMemBase::vFillTime(OSAL_trTimeDate& rfoTimeDate) const
{
   // fill time structure
   if(OSAL_OK != OSAL_s32ClockGetTime(&rfoTimeDate))
   {
      // initialise with zero if failed
      OSAL_pvMemorySet(&rfoTimeDate, 0, sizeof(rfoTimeDate));
   }
}


/*---------------------------------------------------------------------*//**
 * fills the complete error struct
 *
 * @param rfrError      error struct to be filled
 * @param enErrType     error type to be filled in
 * @param u16ErrorCode  error code to be filled in
 * @param pu8Data       pointer to error data to be filled in. can be NULL
 * @param u16DataSize   size of data buffer. Buffer, where pu8Data points to
 * 
 * @return number of copied bytes
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    20.01.2005
 *//*---------------------------------------------------------------------*/
tU16 err_tclErrMemBase::u16FillErrorStruct(trErrmemEntry& rfrError, 
                                           tenErrmemEntryType enErrType, 
                                           tU16 u16ErrorCode, 
                                           const tU8* pu8Data, 
                                           tU16 u16DataSize) const
{
   // Fill error struct
   rfrError.u16Entry = u16ErrorCode;
   rfrError.u16EntryCount = 1;
   rfrError.eEntryType = enErrType;
   vFillTime(rfrError.rEntryTime);
   rfrError.u16EntryLength = u16FillData(rfrError.au8EntryData, pu8Data, u16DataSize);

   // return number of copied error bytes
   return rfrError.u16EntryLength;
}


/*---------------------------------------------------------------------*//**
 * opens error memory device
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    20.01.2005
 *//*---------------------------------------------------------------------*/
void err_tclErrMemBase::vOpenDevice()
{
   // open errmem device
   hErrmemDevice = OSAL_IOOpen(OSAL_C_STRING_DEVICE_ERRMEM,OSAL_EN_WRITEONLY);
}


/*---------------------------------------------------------------------*//**
 * closes error memory device
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    20.01.2005
 *//*---------------------------------------------------------------------*/
void err_tclErrMemBase::vCloseDevice()
{
   if((OSAL_tIODescriptor)OSAL_C_INVALID_HANDLE != hErrmemDevice)
   {
      OSAL_s32IOClose(hErrmemDevice);
   }
}


/*---------------------------------------------------------------------*//**
 * writes a new entry to the error memory. use this method if you have an
 * additional error string
 *
 * @param enErrType            type of error (info or normal)
 * @param u16TraceComp         trace comp identifier 
 * @param u8SpecificErrorCode  component specific error code
 * @param pcErrorText          null terminated error string
 * 
 * @return OSAL_OK or OSAL_ERROR
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    20.01.2005
 *//*---------------------------------------------------------------------*/
tS32 err_tclErrMemBase::s32WriteErrorString(tenErrmemEntryType enErrType, 
                                            tU16 u16TraceComp, tU8 u8SpecificErrorCode, 
                                            const tChar* pcErrorText) const
{
   tU16 u16StringLen = 0;

   if(pcErrorText)
   {
      u16StringLen = (tU16)OSAL_u32StringLength(pcErrorText);
   }
   
   return s32IntWriteError(enErrType, 
                           u16GetErrorCode(u16TraceComp, u8SpecificErrorCode), 
                           (const tU8*)pcErrorText, 
                           (tU16)(u16StringLen + 1U)); // + 1 for null termination '\0'
}


/*---------------------------------------------------------------------*//**
 * writes a new entry to the error memory. use this method if you have 
 * a binary data stream as additional data or no additional data at all
 *
 * @param enErrType            type of error (info or normal)
 * @param u16TraceCom          trace comp identifier 
 * @param u8SpecificErrorCode  component specific error code
 * @param pu8Data              pointer to additional error data 
 *                             or NULL (optional)
 * @param u16DataSize          size of data where pu8Data points to 
 *                             or 0 (optional)
 * 
 * @return OSAL_OK or OSAL_ERROR
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    20.01.2005
 *//*---------------------------------------------------------------------*/
tS32 err_tclErrMemBase::s32WriteError(tenErrmemEntryType enErrType, 
                                      tU16 u16TraceComp, tU8 u8SpecificErrorCode, 
                                      const tU8* pu8Data, 
                                      tU16 u16DataSize) const
{
   return s32IntWriteError(enErrType, 
                           u16GetErrorCode(u16TraceComp, u8SpecificErrorCode), 
                           pu8Data, 
                           u16DataSize);
}


/*---------------------------------------------------------------------*//**
 * internal method to write a new entry to the error memory.
 *
 * @param enErrType            type of error (info or normal)
 * @param u16ErrorCode         error code (combination of trace comp 
 *                             identifier and specific error code)
 * @param pu8Data              pointer to additional error data 
 *                             or NULL
 * @param u16DataSize          size of data where pu8Data points to 
 *                             or 0
 * 
 * @return OSAL_OK or OSAL_ERROR
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    20.01.2005
 *//*---------------------------------------------------------------------*/
tS32 err_tclErrMemBase::s32IntWriteError(tenErrmemEntryType enErrType, 
                                         tU16 u16ErrorCode, 
                                         const tU8* pu8Data, 
                                         tU16 u16DataSize) const
{
   ERR_M_ASSERT(    (enErrType == eErrmemEntryInfo)
                 || (enErrType == eErrmemEntryNormal));

   trErrmemEntry rError;

   u16FillErrorStruct(rError, enErrType, u16ErrorCode, pu8Data, u16DataSize);
   
   return s32IntWriteError(rError);
}


/*---------------------------------------------------------------------*//**
 * internal method to write a new entry to the error memory.
 *
 * @param enErrType            type of error (info or normal)
 * @param u16ErrorCode         error code (combination of trace comp 
 *                             identifier and specific error code)
 * @param pu8Data              pointer to additional error data 
 *                             or NULL
 * @param u16DataSize          size of data where pu8Data points to 
 *                             or 0
 * 
 * @return OSAL_OK or OSAL_ERROR
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    20.01.2005
 *//*---------------------------------------------------------------------*/
tS32 err_tclErrMemBase::s32IntWriteError(trErrmemEntry& rfError) const
{
   return OSAL_s32IOWrite(hGetErrmemDevice(), (const tS8*)&rfError, sizeof(rfError));
}


/*---------------------------------------------------------------------*//**
 * combines trace comp identifier with specific error code to get the 
 * error code for the error memory
 *
 * @param u16TraceComp             trace comp identifier 
 * @param u8CompSpecificErrorCode  component specific error code
 * 
 * @return error code (combination of trace comp 
 *         identifier and specific error code)
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    20.01.2005
 *//*---------------------------------------------------------------------*/
tU16 err_tclErrMemBase::u16GetErrorCode(tU16 u16TraceComp, tU8 u8CompSpecificErrorCode)
{
   ERR_M_ASSERT((u16TraceComp & 0x00FF) == 0);
   tU16 u16ErrorCode = (tU16)((u16TraceComp & 0xFF00) + u8CompSpecificErrorCode);
   return u16ErrorCode;
}
