/************************************************************************
 * FILE:         amt_Trace.cpp
 * PROJECT:      VASCO
 * SW-COMPONENT: AMT
 *----------------------------------------------------------------------
 * DESCRIPTION:  (A)pplication (M)essage (T)emplate
 *               Trace module for AMT
 *----------------------------------------------------------------------
 * COPYRIGHT:    (c) 2000 Robert Bosch GmbH, Hildesheim
 * HISTORY:
 * Date      |   Modification                         | Author
 * 20.04.01  |   Initial revision                     | Bruns,EFG32
 *************************************************************************/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#define AMT_S_IMPORT_INTERFACE_GENERIC
#include "amt_if.h"


// -------------------------------------------------------------------------
// modul-locals
// -------------------------------------------------------------------------
static OSAL_tIODescriptor amt_oTraceDesc = OSAL_ERROR;



/******************************************************************************
 *FUNCTION:    amt_bInitTrace
 *DESCRIPTION: Trace initialisation for AMT
 *
 *PARAMETER:   -
 *
 *RETURNVALUE: TRUE on success, else FALSE
 *
 *HISTORY:     20.04.2001  -  BNS2HI
 *             Initial revision
 *****************************************************************************/
tBool amt_bInitTrace()
{
   if ( amt_oTraceDesc == OSAL_ERROR )
   {
      // try to create trace-device
      amt_oTraceDesc = OSAL_IOCreate(OSAL_C_STRING_DEVICE_TRACE, 
                                 OSAL_EN_READWRITE);
      if(amt_oTraceDesc == OSAL_ERROR)
      {
         // trace-device already created? try to open it
         amt_oTraceDesc = OSAL_IOOpen(OSAL_C_STRING_DEVICE_TRACE, 
                                       OSAL_EN_READWRITE);
      }
   } // else Trace already opened

   return ((amt_oTraceDesc == OSAL_ERROR) ? FALSE : TRUE);
}



/******************************************************************************
 *FUNCTION:    amt_bDeInitTrace
 *DESCRIPTION: Trace de-initialisation for AMT; frees used ressources.
 *
 *PARAMETER:   -
 *
 *RETURNVALUE: TRUE on success, else FALSE
 *
 *HISTORY:     -
 *****************************************************************************/
tBool amt_bDeInitTrace()
{
   tBool bSuccess = FALSE;
   if(OSAL_s32IOClose(amt_oTraceDesc) != OSAL_ERROR)
   {
      bSuccess = TRUE;
      amt_oTraceDesc = OSAL_ERROR;
   }
   return bSuccess;
}



/******************************************************************************
 *FUNCTION:    amt_vTraceOutString
 *DESCRIPTION: Trace out a string.
 *
 *PARAMETER:   tU32   u32Level (->I)
 *             The trace level for output
 *
 *             const tChar *coszBuffer (->I)
 *             The format string (see printf-documentation)
 *
 *RETURNVALUE: -
 *
 *HISTORY:     20.04.2001  -  BNS2HI
 *             Initial revision
 *****************************************************************************/
tVoid amt_vTraceOutBufferN(TR_tenTraceLevel enTraceLevel, const tChar *coszBuffer, tU32 u32StringLength)
{
   if(amt_oTraceDesc != OSAL_ERROR)
   {
      OSALUTIL_s32TraceWrite
      (
            amt_oTraceDesc,
            (tU32)enTraceLevel,  // tU32 u32Level
            (tU32)TR_CLASS_AMT,  // tU32 u32Class
            (tPCS8)coszBuffer,   // tPCS8 pcos8Buffer
            u32StringLength      // tU32 u32Length 
      );  //lint !e522: Expected void type, assignment, increment or decrement
   }
}



/******************************************************************************
 *FUNCTION:    amt_vTraceOutBufferN
 *DESCRIPTION: Trace out a buffer contents.
 *
 *PARAMETER:   tU32   u32Level (->I)
 *             The trace level for output
 *
 *             const tChar *coszFormatString (->I)
 *             Pointer to the buffer
 *
 *             tU32   u32BufferLength
 *             Length of the output
 *
 *RETURNVALUE: -
 *
 *HISTORY:     20.04.2001  -  BNS2HI
 *             Initial revision
 *****************************************************************************/
tVoid amt_vTraceOutString(TR_tenTraceLevel enTraceLevel, const tChar *coszFormatString, ...)
{
   if(amt_oTraceDesc != OSAL_ERROR)
   {
      // build output string by using OSAL "OSAL_VarArgStart" function. 
      // "pcTempBuffer" will contain the output string.
      tChar pcTempBuffer[AMT_MAX_TRACEBUF_LEN+1];
      OSAL_tVarArgList ArgList;
      OSAL_VarArgStart(ArgList, coszFormatString); //lint !e530 => Authorized LINT deactivation #11.
      tS32 s32BufLen = OSAL_s32VarNPrintFormat(pcTempBuffer, AMT_MAX_TRACEBUF_LEN, coszFormatString, ArgList);//lint !e530 : OSAL_VarArgStart IS the initialiser */
      OSAL_VarArgEnd(ArgList);


      // set "0"-byte at the end
      if ( s32BufLen > 0 )
      {
         pcTempBuffer[s32BufLen] = 0;

         // output string
         amt_vTraceOutBufferN(enTraceLevel, pcTempBuffer, (tU32)OSAL_u32StringLength(pcTempBuffer));
      } // else error, can't trace string
   }
}

