

#include <map>
#include <set>
#include <vector>
#include <list>

#include <string>

#include <iostream>
#include <fstream>


using namespace std;

#ifndef USE_XERCES1
#include <xercesc/util/PlatformUtils.hpp>
#include <xercesc/parsers/XercesDOMParser.hpp>
#include <xercesc/sax/HandlerBase.hpp>
#include <xercesc/util/XMLString.hpp>

#else
#include <xercesc/util/PlatformUtils.hpp>
#include <xercesc/sax/SAXException.hpp>
#include <xercesc/sax/SAXParseException.hpp>
#include <xercesc/parsers/DOMParser.hpp>
#include <xercesc/dom/DOM_DOMException.hpp>

#endif
using namespace xercesc;
#include "xercesc/dom/DOM.hpp"
#include "FIParser.h"



 
class DOMTreeErrorReporter : public ErrorHandler
{
public:
    // -----------------------------------------------------------------------
    //  Constructors and Destructor
    // -----------------------------------------------------------------------
    DOMTreeErrorReporter() :
       fSawErrors(false)
    {
    }

    ~DOMTreeErrorReporter()
    {
    }


    // -----------------------------------------------------------------------
    //  Implementation of the error handler interface
    // -----------------------------------------------------------------------
    void warning(const SAXParseException& toCatch);
    void error(const SAXParseException& toCatch);
    void fatalError(const SAXParseException& toCatch);
    void resetErrors();

    // -----------------------------------------------------------------------
    //  Getter methods
    // -----------------------------------------------------------------------
    bool getSawErrors() const;

    // -----------------------------------------------------------------------
    //  Private data members
    //
    //  fSawErrors
    //      This is set if we get any errors, and is queryable via a getter
    //      method. Its used by the main code to suppress output if there are
    //      errors.
    // -----------------------------------------------------------------------
    bool    fSawErrors;
};

inline bool DOMTreeErrorReporter::getSawErrors() const
{
    return fSawErrors;
}

// ---------------------------------------------------------------------------
//  This is a simple class that lets us do easy (though not terribly efficient)
//  trancoding of XMLCh data to local code page for display.
// ---------------------------------------------------------------------------
class StrX
{
public :
    // -----------------------------------------------------------------------
    //  Constructors and Destructor
    // -----------------------------------------------------------------------
    StrX(const XMLCh* const toTranscode)
    {
        // Call the private transcoding method
        fLocalForm = XMLString::transcode(toTranscode);
    }

    ~StrX()
    {
        //delete [] fLocalForm;
        XMLString::release(&fLocalForm);
    }


    // -----------------------------------------------------------------------
    //  Getter methods
    // -----------------------------------------------------------------------
    const char* localForm() const
    {
        return fLocalForm;
    }

private :
    // -----------------------------------------------------------------------
    //  Private data members
    //
    //  fLocalForm
    //      This is the local code page form of the string.
    // -----------------------------------------------------------------------
    char*   fLocalForm;
};

inline ostream& operator<<(ostream& target, const StrX& toDump)
{
    target << toDump.localForm();
    return target;
}

void DOMTreeErrorReporter::warning(const SAXParseException&)
{
    //
    // Ignore all warnings.
    //
}

void DOMTreeErrorReporter::error(const SAXParseException& toCatch)
{
    fSawErrors = true;
    cerr << "Error at file \"" << StrX(toCatch.getSystemId())
		 << "\", line " << toCatch.getLineNumber()
		 << ", column " << toCatch.getColumnNumber()
         << "\n   Message: " << StrX(toCatch.getMessage()) << endl;
}

void DOMTreeErrorReporter::fatalError(const SAXParseException& toCatch)
{
    fSawErrors = true;
    cerr << "Fatal Error at file \"" << StrX(toCatch.getSystemId())
		 << "\", line " << toCatch.getLineNumber()
		 << ", column " << toCatch.getColumnNumber()
         << "\n   Message: " << StrX(toCatch.getMessage()) << endl;
    exit(-1);
}

void DOMTreeErrorReporter::resetErrors()
{
    // No-op in this case
}




tclDOMParser::tclDOMParser()
{
   // Initialize the XML4C system
   try
   {
      XMLPlatformUtils::Initialize();

      // Instantiate the DOM parser.
      poParser = new XercesDOMParser;
      if(0 != poParser)
      {
         //   poParser->setIncludeIgnorableWhitespace(false);
         poParser->setValidationScheme(XercesDOMParser::Val_Always);    
         poParser->setDoNamespaces(true);
         poParser->setDoSchema(true);
         poParser->setValidationSchemaFullChecking(false);
         //   poParser->setValidationConstraintFatal(true);
         //poParser->setCreateEntityReferenceNodes(true);
          gopErrReporter = new DOMTreeErrorReporter();
          poParser->setErrorHandler(gopErrReporter);
         // read in the xml file
      }
   }

   catch (const XMLException& toCatch)
   {
      cerr << "Error during initialization! :\n"
         << toCatch.getMessage() << endl;
      exit(-1);
   }

}

tclDOMParser::~tclDOMParser()
{
   XMLPlatformUtils::Terminate();
}






bool tclDOMParser::bParse(char *pXMLFile)
{
   bool bSuccess = false;


    bool errorsOccured = false;
    try
    {
        poParser->parse(pXMLFile);
      bSuccess = true;
    }

    catch (const XMLException& e)
    {
        cerr << "An error occurred during parsing\n   Message: "
             << StrX(e.getMessage()) << endl;
        errorsOccured = true;
    }

    catch (const DOMException& e)
    {
       cerr << "A DOM error occurred during parsing\n   DOMException code: "
             << e.code << endl;
        errorsOccured = true;
    }

    catch (...)
    {
        cerr << "An error occurred during parsing\n " << endl;
        errorsOccured = true;
    }

    // If the parse was successful, output the document data from the DOM tree
   if (errorsOccured /*|| gopErrReporter->getSawErrors()*/)
   {
       bSuccess= false;
       exit(-1);
   }

   return bSuccess;
}




bool tclDOMParser::bDeinitialise()
{
   delete poParser;
   return true;
}




XercesDOMParser* 
tclDOMParser::pGetDOMParser()
{
   return poParser;
}


DOMDocument*
tclDOMParser::pGetDocument()
{
   DOMDocument *pDoc = 0;
   if(0 != poParser)
   {
      pDoc = poParser->getDocument();
   }
   return pDoc;
}
