/*****************************************************************************
 * FILE:          FICodeGenMostExt.cpp
 * SW-COMPONENT:  
 * DESCRIPTION:   Generation of additional, MOST-compatible data types for use with
 *                FI code generator based on c++ fi tooling
 *                
 * AUTHOR:        CM-AI/PJ-GM54  Koch, Martin
 * COPYRIGHT:     (c) 2010 Robert Bosch Car Multimedia GmbH
 * HISTORY:       
 *                09.03.10 Rev. 1.0  CM-AI/PJ-GM54  Koch, Martin
 *                         Initial Version;
 *****************************************************************************/


#define FI_VERSION 1

//-----------------------------------------------------------------------------
// General includes
//-----------------------------------------------------------------------------
#include <iostream>
#include <sstream>

#include "FIServiceGroup.h"
#include "FICodeGenCmdLine.h"
#include "FICodeGenUtil.h"

#include "FiCodeGenMostExt.h"

//-----------------------------------------------------------------------------
// Local functions
//-----------------------------------------------------------------------------

string sTypeFromLength(int iLength)
{
   if (iLength <= 0)
      return "invalid_Type"; // will cause compilation error
   else if (iLength == 1)
      return "tBool";
   else if (iLength > 1 && iLength <= 8)
      return "tU8";
   else if (iLength <= 16)
      return "tU16";
   else if (iLength <= 32)
      return "tU32";
   else if (iLength <= 64)
      return "tU64";
   else
      return "invalid_Type"; // will cause compilation error
}

//-----------------------------------------------------------------------------

bool bIsIntrinsicType(string sTypeName)
{
   if (    (sTypeName == "tBool")
        || (sTypeName == "tU8")
        || (sTypeName == "tS8")
        || (sTypeName == "tU16")
        || (sTypeName == "tS16")
        || (sTypeName == "tU32")
        || (sTypeName == "tS32")
      )
      return true;
   else
      return false;
}

//-----------------------------------------------------------------------------

string sDefaultValue(tclFINode* poType)
{
   string sRetVal = "";
   string sDefaultValue = poType->oGetDefaultValue();

   if (sDefaultValue != "")
      sRetVal = sDefaultValue;
   else if (!bIsIntrinsicType(poType->oGetName()))
      ;
   else if (poType->oGetName() == "tBool")
      sRetVal = "false";
   else
      sRetVal = "0";

   return sRetVal;
}

//-----------------------------------------------------------------------------

bool bHasDynamicSize(tclFINode* poType)
{
   if (bIsIntrinsicType(poType->oGetName()))
      return false;

   else if ("tString" == poType->oGetName())
      return true;
   else if ("T_String" == poType->oGetName())
      return true;
   else if ("T_TrailingStream" == poType->oGetName())
      return true;
   else if ("T_ClassifiedStream" == poType->oGetName())
      return true;
   else if ("T_ShortStream" == poType->oGetName())
      return true;
   else if (poType->enGetType() == EN_OBJTYPE_ARRAY)
      return true;
   else if (poType->enGetType() == EN_OBJTYPE_UNION)
      return true;

   else if (poType->enGetType() == EN_OBJTYPE_ENUMERATION)
      return false;
   else if (poType->enGetType() == EN_OBJTYPE_BITFIELD)
      return false;
   else if (poType->enGetType() == EN_OBJTYPE_FRAME)
      return false;
   else if (poType->enGetType() == EN_OBJTYPE_STRUCTURE)
   {
      cout << poType->oGetName() << endl;

      DOMElement* poEl = poType->poGetDOMElement();
      if (poEl-> hasAttribute (X("storageClass")))
      {
         string sStorageClass = X(poEl-> getAttribute (X("storageClass")));
         if ((sStorageClass == "shortStream") || (sStorageClass == "trailingStream"))
            return true;
      }

      tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
      for (tclFITypeDependencyListIterator iter = poDepList->begin(); iter != poDepList->end(); ++iter)
         if (bHasDynamicSize((*iter)->poGetTarget()))
            return true;
   }

   return false;
}

//-----------------------------------------------------------------------------

string sGetTypeName(tclFINode *poType, string oPrefix)
{
   if (poType->enGetType() == EN_OBJTYPE_OPCODE)
      return oPrefix + oGetClassName(CG_MSGTYPE_PREFIX + oRemove(poType->oGetName(),'.'));
   else
      return oGetClassName(poType->oGetName());
}

//-----------------------------------------------------------------------------
// public functions
//-----------------------------------------------------------------------------

bool b_MOST_UseNonStandardOpcodeId(string oOpcode, int& iOpcodeId)
{
   if (    (oOpcode == "Set")
        || (oOpcode == "Get")
        || (oOpcode == "MethodStart")
        || (oOpcode == "UpReg")
        || (oOpcode == "RelUpReg")
        || (oOpcode == "MethodAbort")
        || (oOpcode == "Status")
        || (oOpcode == "MethodResult")
        || (oOpcode == "Error")
        || (oOpcode == "AbortResult")
        || (oOpcode == "Increment")
        || (oOpcode == "Decrement")
        || (oOpcode == "PureSet"))
      return false;  // use standard values for these OpCodes
   else
      return true;   // MOST doesn't support other OpCodes, 
                     // so we leave the value invalid, 
                     // but prevent standard interpretation
}

//-----------------------------------------------------------------------------

bool b_MOST_UseNonStandardTypeMethods(tclFINode *poType, ostream &out)
{
   // we don't do some special things here
   return false;
}

//-----------------------------------------------------------------------------

bool b_MOST_IsStandardErrorMessage(tclFINode *poType)
{

   tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
   if (poDepList->size() != 2)
      return false;

   tclFITypeDependencyListIterator iter = poDepList->begin();
   bool bRetVal
    =   ((*iter)->oGetName() == "e8ErrorCode")
     && ((*iter++)->poGetTarget()->oGetName() == "T_e8_ErrorCode")
     && ((*iter)->oGetName() == "oErrorInfo")
     && ((*iter)->poGetTarget()->oGetName() == "T_TrailingStream");

   return bRetVal;
}

//-----------------------------------------------------------------------------

bool b_MOST_NeedsDestroy(tclFINode* poType)
{
   string BaseTypesToDestroy[]
     = {"tString", "T_String", "T_TrailingStream", "T_ClassifiedStream", "T_ShortStream"};

   switch (poType->enGetType())
   {
   case EN_OBJTYPE_ARRAY:
   case EN_OBJTYPE_UNION:
      return true;

//   case EN_OBJTYPE_OPCODE:
   case EN_OBJTYPE_TYPE:
      for (unsigned int i = 0; i < BaseTypesToDestroy->size(); i++)
      {
         string localString(BaseTypesToDestroy[i]);
         if (    (poType->oGetName() == BaseTypesToDestroy[i])
              || (poType->poFindDependency(localString)))
         {
            return true;
         }
      }
      break;

   case EN_OBJTYPE_FRAME:
   default:
      break;
   }

   return false;
}

//-----------------------------------------------------------------------------

bool b_MOST_GenerateTypeClass(tclFINode *poType, ostream &out, tclFIServiceTree* poServiceTree, bool& bNeedsDestroy, bool& bSuccess)
{
   // provides complete type class declaration for MOST-compatible data types
   // which are not covered by standard FI

   bool bRetVal = false;
   tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
   tclFITypeDependencyListIterator iter;
   ostringstream memout;

   switch (poType->enGetType())
   {
   case EN_OBJTYPE_ARRAY:
      {
         out << "()";
         /* member variables */
         iter = poDepList->begin();
         tclFINode* poBaseType = (*iter)->poGetTarget();
         string sBaseType = oGetClassName(poBaseType->oGetName());
         string sChildName = (*iter)->oGetName();

         if (bHasDynamicSize(poBaseType))
            memout << "private:" << endl
                   << TAB << "std::vector<" << sBaseType << "*, std::allocator<" << sBaseType << "*> > oLocalReaders;" << endl
                   << endl
                   << "public:"<< endl;

         // memout << TAB << sBaseType << " " << sChildName << "[];" << endl;
         // see X:\di_fi\components\ren_fi\types\source\alltypes.h
         // see X:\di_fi\components\ren_fi\types\source\ren_alltypes.cpp
         memout << TAB << oGetClassName(poBaseType->oGetName(), true) << sChildName << ";" << endl << endl
                << TAB << sBaseType << "& operator[](tU32 u32Index);" << endl;

         bNeedsDestroy = true;
         bRetVal = true;
      }
      break;

   case EN_OBJTYPE_UNION:
      {
         ostringstream memberAccessFcts;
         out << "():u32DynamicSize(0), _ActiveMemberOffset(0u), pau8Data(OSAL_NULL)";

         for (iter = poDepList->begin(); iter != poDepList->end(); ++iter)
         {
            tclFINode* poBaseType = (*iter)->poGetTarget();
            out <<  ", _"<< (*iter)->oGetName() << "(" /* << sDefaultValue(poBaseType) */ << ")";
         }

         /* member variables */
         memout << "private:" << endl
                << TAB "tU32 u32DynamicSize;" << endl
                << TAB "size_t _ActiveMemberOffset;" << endl
                << TAB "tU8* pau8Data;"<< endl
                << endl;

         for (iter = poDepList->begin(); iter != poDepList->end(); ++iter)
         {
            tclFINode* poBaseType = (*iter)->poGetTarget();
            string sBaseType;
            if (bIsIntrinsicType(poBaseType->oGetName()))
               sBaseType = oGetFIGroup() + "fi_tcl" + poBaseType->oGetName().substr(1);
            else
               sBaseType = oGetClassName(poBaseType->oGetName());
            string sChildName = (*iter)->oGetName();
            memout << TAB << sBaseType << " _" << sChildName << ";" << endl;
            memberAccessFcts << TAB << oGetClassName(poBaseType->oGetName()) << "& " 
                             << sChildName << "();" << endl;
         }

         string sFiGroupBasetype = oGetFIGroup() + "fi_tclVisitableTypeBase";
         memout << endl
                << TAB "// helper functions" << endl
                << TAB "inline size_t Offset(const " << sFiGroupBasetype << "& roMember) const" << endl
                << TAB TAB "{ return (size_t)(&roMember) - (size_t)this; }" << endl
                << TAB "const " << sFiGroupBasetype << "* pcActiveMember() const;" << endl
                << TAB << sFiGroupBasetype << "& roVariant(" << sFiGroupBasetype << "& roSelectedVariant);" << endl
                << endl
                << "public:" << endl
                << TAB "// variant access functions" << endl
                << memberAccessFcts.str();
         bNeedsDestroy = true;
         bRetVal = true;
      }
      break;

   case EN_OBJTYPE_FRAME:
      {
         out << "()";
         /* member variables */
         DOMElement* pEl = poType->poGetDOMElement();
         DOMElement* poFrame
            = static_cast<DOMElement*> ((pEl-> getElementsByTagName(X("frame")))-> item(0));

         memout << TAB << "union" << endl 
                << TAB <<  "{" << endl
                << TAB << TAB << "tU8 au8Data[" << X(poFrame-> getAttribute (X("size"))) << "];" << endl
                << TAB << TAB << "struct {" << endl;

         char separator = ':';
         //tclFITypeDependencyListIterator iter;
         //for (iter = poDepList->begin(); iter != poDepList->end(); ++iter)
         DOMNodeList* poSignals = poType->poGetDOMElement()->getElementsByTagName(X("signal"));
         for (XMLSize_t iter = 0; iter != poSignals->getLength(); iter++)
         {
            DOMElement*  poSignal = static_cast<DOMElement*>(poSignals->item(iter));
            string sName = X(poSignal-> getAttribute(X("name")));
            int iPosition = atoi(X(poSignal-> getAttribute(X("position"))));
            int iLength = atoi(X(poSignal-> getAttribute(X("length"))));

            out << endl 
                << TAB << TAB << separator << " " << sName << "(0)";
            separator = ',';

            memout << TAB << TAB << TAB << sTypeFromLength(iLength) << " " 
                   << sName<< " : " << iLength << ";" << endl;
         }
         memout << TAB <<  TAB << "};" << endl 
                << TAB <<  "};" << endl;
         // position="0" length="1"
         /* ---------------- */
         bRetVal = true;
      }
      break;
   }

   if (bRetVal)
   {
      out << " {}" << endl; 

      if (bEnableDeepCopy)
      {
         string oTypeName = oGetClassName(poType->oGetName());

         // this group of methods changes the known behaviour!
         out << "#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;
         out << TAB << oTypeName << "& operator=(const " << oTypeName << "& coRef);" << endl;
         out << TAB << oTypeName << "(const " << oTypeName << "& coRef);" << endl;
         out << "#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;
      }
      out << endl 
          << TAB << "// member variables" << endl
          << memout.str();

   }

   return bRetVal;
}

//-----------------------------------------------------------------------------

bool b_MOST_GenerateTypeMembers (tclFINode* poType, ostream &out,string oPrefix)
{
   switch (poType->enGetType())
   {
   case EN_OBJTYPE_ARRAY:
      {
         tclFITypeDependency* poDep = *poType->poGetDependencyList()->begin();
         tclFINode* poBaseType = poDep->poGetTarget();
         string sChildName = poDep->oGetName();
         string sBaseTypeName = poBaseType->oGetName();
         string sBaseClass = oGetClassName(sBaseTypeName);

         string sInitValue = "";
         if (bIsIntrinsicType(sBaseTypeName))
         {
            sInitValue = ", (" + poBaseType->oGetName() + ") ";
            if (poBaseType->oGetDefaultValue() != "")
               sInitValue += poBaseType->oGetDefaultValue();
            else
               sInitValue += "0";
         }

         out << sBaseClass << "& " 
                    << oGetClassName(poType->oGetName()) << "::operator[](tU32 u32Index)" << endl
             << "{" << endl
             << TAB << "if (" << sChildName << ".size() <= u32Index)" << endl
             << TAB << TAB << sChildName << ".resize(1 + u32Index" << sInitValue << ");" << endl
             << endl
             << TAB << "return " << sChildName << "[u32Index];" << endl
             << "}" << endl << endl;
      }
      break;

   case EN_OBJTYPE_UNION:
      {
         string sFiGroupBaseType = oGetFIGroup() + "fi_tclVisitableTypeBase";
         string sUnionName = oGetClassName(poType->oGetName());
         tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
         tclFITypeDependencyListIterator iter = poDepList->begin();

         // helper function to get active variant member pointer
         out << "const " << sFiGroupBaseType << "* " << sUnionName << "::pcActiveMember() const" << endl
             << "{" << endl;
         string sContinuator = "";
         for (iter = poDepList->begin(); iter != poDepList->end(); ++iter, sContinuator = "else ")
         {
            string sChildName = (*iter)->oGetName();
            out << TAB << sContinuator << "if (_ActiveMemberOffset == Offset(_" << sChildName << "))" << endl
                << TAB TAB "return &_" << sChildName << ";" << endl;
         }
         out << TAB "else" << endl
             << TAB TAB "return OSAL_NULL;" << endl
             << "}" << endl
             << endl;

         // helper function to set active variant member, read content from binary buffer and obtain reference
         out << sFiGroupBaseType << "& " << sUnionName << "::roVariant(" << sFiGroupBaseType << "& roSelectedVariant)" << endl
             << "{" << endl
             << TAB << sFiGroupBaseType << "* poCurrentVariant = OSAL_NULL;" << endl;
         for (iter = poDepList->begin(), sContinuator = ""; iter != poDepList->end(); ++iter, sContinuator = "else ")
         {
            string sChildName = "_" + (*iter)->oGetName();
            out << TAB << sContinuator << "if (_ActiveMemberOffset == Offset(" << sChildName << "))" << endl
                << TAB TAB "poCurrentVariant = &" << sChildName << ";" << endl;
         }
         out << TAB "if (poCurrentVariant != &roSelectedVariant)" << endl
             << TAB "{" << endl
             << TAB TAB "if (poCurrentVariant)" << endl
             << TAB TAB TAB "poCurrentVariant->vDestroy();" << endl
             << endl
             << TAB TAB "// store selection" << endl
             << TAB TAB "_ActiveMemberOffset = Offset(roSelectedVariant);" << endl
             << endl
             << TAB TAB "// get data from binary buffer (if previously read from shared memory)" << endl
             << TAB TAB "if (pau8Data && u32DynamicSize)" << endl
             << TAB TAB "{" << endl
             << TAB TAB TAB "fi_tclInContext oReadContext(pau8Data, pau8Data + u32DynamicSize, 1, enGetByteOrder());" << endl
             << TAB TAB TAB "(tVoid) roSelectedVariant.oRead(oReadContext);" << endl
             << TAB TAB "}" << endl
             << TAB "}" << endl
             << TAB "return roSelectedVariant;" << endl
             << "}" << endl
             << endl;


         for (iter = poDepList->begin(); iter != poDepList->end(); ++iter)
         {
            tclFINode* poBaseType = (*iter)->poGetTarget();
            string sBaseType
               = bIsIntrinsicType(poBaseType->oGetName())
                  ? oGetFIGroup() + "fi_tcl" + poBaseType->oGetName().substr(1)
                  : oGetClassName(poBaseType->oGetName());
            string sChildName = (*iter)->oGetName();
            out << oGetClassName(poBaseType->oGetName()) << "& " << sUnionName << "::" << sChildName << "()" << endl
                << "{" << endl
                << TAB "return static_cast<" << sBaseType << "&> (roVariant(_" << sChildName << "));" << endl
                << "}" << endl
                << endl;
         }
      }
      break;
   }

   return false;
}

//-----------------------------------------------------------------------------

bool b_MOST_GenerateTypeCopyConstructor(tclFINode *poType, ostream &out,string oPrefix)
{
   bool bRetVal = false;
   string oTypeName = oGetClassName(poType->oGetName());

   switch (poType->enGetType())
   {
      case EN_OBJTYPE_ARRAY:
         {
            out << oTypeName << "::" << oTypeName << "(const " << oTypeName << "& coRef)" << endl
                << TAB << ": " << oGetFIGroup() << "fi_tclVisitableTypeBase(coRef)" << endl
                << "{" << endl
                << TAB << "*this = coRef;" << endl
                << "}" << endl;

            bRetVal = true;
         }
         break;

      case EN_OBJTYPE_UNION:
         {
            out << oTypeName << "::" << oTypeName << "(const " << oTypeName << "& coRef)" << endl
                << TAB << ": " << oGetFIGroup() << "fi_tclVisitableTypeBase(coRef)" << endl
                << TAB << ", u32DynamicSize(0), _ActiveMemberOffset(0u), pau8Data(OSAL_NULL)" << endl
                << "{" << endl
                << TAB << "*this = coRef;" << endl
                << "}" << endl;

            bRetVal = true;
         }
         break;

      case EN_OBJTYPE_FRAME:
         {
            //DOMElement* poFrame
            //   = static_cast<DOMElement*> ((poType->poGetDOMElement()-> getElementsByTagName(X("frame")))-> item(0));
            //unsigned int size = XMLString::parseInt(poFrame-> getAttribute (X("size")));
            out << oTypeName << "::" << oTypeName << "(const " << oTypeName << "& coRef)" << endl
                << TAB << ": " << oGetFIGroup() << "fi_tclVisitableTypeBase(coRef)" << endl
            //    << TAB << ", au8Data[]({";
            //for (unsigned int i = 0; i < size; i++)
            //   out << (i ? ", 0" : "0");
            //out << "})" << endl
                << "{" << endl
                << TAB << "*this = coRef;" << endl
                << "}" << endl;

            bRetVal = true;
         }
         break;

      default:
         break;
   }

   return bRetVal;
}

//-----------------------------------------------------------------------------

bool b_MOST_GenerateTypeAssignmentOp(tclFINode *poType, ostream &out,string oPrefix)
{
   bool bRetVal = false;
   string oTypeName = oGetClassName(poType->oGetName());

   switch (poType->enGetType())
   {
      case EN_OBJTYPE_ARRAY:
         {
            tclFINode* poChild = (*poType->poGetDependencyList()->begin())->poGetTarget();
            string sChildName = (*poType->poGetDependencyList()->begin())->oGetName();
            string sChildType = oGetClassName(poChild->oGetName());

            out << oTypeName << "& " << oTypeName << "::operator=" << "(const " << oTypeName << "& coRef)" << endl
                << "{" << endl
                << TAB << "if (this != &coRef)" << endl;

            if (bHasDynamicSize(poChild))
               out << TAB "{" << endl
                   << TAB TAB "if (oLocalReaders.size())" << endl
                   << TAB TAB TAB "oLocalReaders.clear();" << endl
                   << endl
                   << TAB TAB << sChildName << ".reserve(coRef." << sChildName << ".size());" << endl
                   << TAB TAB "for (tU32 iter = 0; iter < coRef." << sChildName << ".size(); iter++)" << endl
                   << TAB TAB TAB << sChildName << ".push_back(" << sChildType << "(coRef." << sChildName << "[iter]));" << endl
                   << TAB "}" << endl;
            else
               out << TAB TAB << sChildName << " = coRef." << sChildName << ";" << endl;

            out << endl
                << TAB "return *this;" << endl
                << "}" << endl;

            bRetVal = true;
         }
         break;

      case EN_OBJTYPE_UNION:
         {
            out << oTypeName << "& " << oTypeName << "::operator=" << "(const " << oTypeName << "& coRef)" << endl
                << "{" << endl
                << TAB "if (this != &coRef)" << endl
                << TAB "{" << endl;

            out << TAB TAB "if (pau8Data)" << endl
                << TAB TAB TAB "OSAL_DELETE[] pau8Data;" << endl
                << TAB TAB "pau8Data = OSAL_NULL;" << endl
                << TAB TAB "_ActiveMemberOffset = coRef._ActiveMemberOffset;" << endl
                << endl;

            tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
            for (tclFITypeDependencyListIterator iter = poDepList->begin(); iter != poDepList->end(); ++iter)
            {
               string sChildName = "_" + (*iter)->oGetName();
               out << TAB TAB << sChildName << " = coRef." << sChildName << ";" << endl;
            }

            out << endl
                << TAB TAB "if (coRef.pau8Data && coRef.u32DynamicSize)" << endl
                << TAB TAB "{" << endl
                << TAB TAB TAB "u32DynamicSize = coRef.u32DynamicSize;" << endl
                << TAB TAB TAB "pau8Data = OSAL_NEW tU8[u32DynamicSize];" << endl
                << TAB TAB TAB "if (pau8Data)" << endl
                << TAB TAB TAB TAB "(tVoid) OSAL_pvMemoryCopy(pau8Data, coRef.pau8Data, u32DynamicSize);" << endl
                << TAB TAB "}" << endl
                << TAB "}" << endl
                << endl
                << TAB "return *this;" << endl
                << "}" << endl;

            bRetVal = true;
         }
         break;

      case EN_OBJTYPE_FRAME:
         {
            DOMNode* poFrame
               = poType->poGetDOMElement()-> getElementsByTagName(X("frame"))-> item(0);
            string sSize = X(static_cast<DOMElement*>(poFrame)-> getAttribute (X("size")));

            out << oTypeName << "& " << oTypeName << "::operator=" << "(const " << oTypeName << "& coRef)" << endl
                << "{" << endl
                << TAB << "if (this != &coRef)" << endl
                << TAB << TAB << "for (tU8 u8Index = 0; u8Index < " << sSize << "; u8Index++)" << endl
                << TAB << TAB << TAB << "au8Data[u8Index] = coRef.au8Data[u8Index];" << endl
                << endl
                << TAB << "return *this;" << endl
                << "}" << endl;

            bRetVal = true;
         }
         break;

      default:
         break;
   }

   return bRetVal;
}

//-----------------------------------------------------------------------------

bool b_MOST_GenerateTypeDestructor( tclFINode *poType, ostream &out, string oPrefix )
{
   bool bRetVal = false;
   string sClassName;
   if ( poType->enGetType() == EN_OBJTYPE_OPCODE )
      sClassName = oPrefix + oGetClassName( CG_MSGTYPE_PREFIX + oRemove( poType->oGetName(), '.' ) );
   else
      sClassName = oGetClassName( poType->oGetName() );

   switch ( poType->enGetType() )
   {
   case EN_OBJTYPE_UNION:
   {
      out << sClassName << "::~" << sClassName << "()" << endl
         << "{" << endl
         << "#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl
         << TAB << "vDestroy();" << endl
         << "#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl
         << "}" << endl;
      bRetVal = true;
   }
   break;
   case EN_OBJTYPE_ARRAY:
   {
      string sChildName = (*poType->poGetDependencyList()->begin())->oGetName();
      out << sClassName << "::~" << sClassName << "()" << endl
         << "{" << endl
         << "#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl
         << TAB << "vDestroy();" << endl
         << "#else" << endl
         << TAB << sChildName << ".clear();" << endl
         << "#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl
         << "}" << endl;
      bRetVal = true;
   }
   break;
   default:
      bRetVal = false;
      break;
   }

   return bRetVal;
}
//-----------------------------------------------------------------------------

bool b_MOST_GenerateDestroyMethod(tclFINode *poType, ostream &out, string oPrefix)
{
   bool bRetVal = false;

   if ( b_MOST_NeedsDestroy( poType ) )
   {
      string sClassName;

      if (poType->enGetType() == EN_OBJTYPE_OPCODE)
         sClassName = oPrefix + oGetClassName(CG_MSGTYPE_PREFIX + oRemove(poType->oGetName(),'.'));
      else
         sClassName = oGetClassName(poType->oGetName());

      switch ( poType->enGetType() )
      {
      case EN_OBJTYPE_UNION:
      {
         out << "tVoid " << sClassName << "::vDestroy()" << endl
               << "{" << endl
               << TAB << "if (pau8Data)" << endl
               << TAB << "{" << endl
               << TAB << TAB << "OSAL_DELETE[] pau8Data;" << endl
               << TAB << TAB << "pau8Data = OSAL_NULL;" << endl
               << TAB << TAB << "u32DynamicSize = 0;" << endl
               << TAB << "}" << endl;
         tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
         tclFITypeDependencyListIterator iter = poDepList->begin();
         for ( ; iter != poDepList->end(); ++iter )
            out << TAB "_" << (*iter)->oGetName() << ".vDestroy();" << endl;

         out << TAB << "_ActiveMemberOffset = 0u;" << endl
               << "}" << endl << endl;
         bRetVal = true;
      }
      break;

      case EN_OBJTYPE_ARRAY:
      {
         string sChildName = (*poType->poGetDependencyList()->begin())->oGetName();
         tclFINode* poBaseType = (*poType->poGetDependencyList()->begin())->poGetTarget();

         out << "tVoid " << sClassName << "::vDestroy()" << endl
               << "{" << endl
         << "#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl
               << TAB << sChildName << ".clear();" << endl;

         if ( bHasDynamicSize( poBaseType ) )
            out << endl
                  << TAB << "for (tU32 it = 0; it < oLocalReaders.size(); it++)" << endl
                  << TAB << "{" << endl
                  << TAB << TAB << "oLocalReaders[it]->vDestroy();" << endl
                  << TAB << TAB << "OSAL_DELETE oLocalReaders[it];" << endl
                  << TAB << "}" << endl
                  << TAB << "oLocalReaders.clear();" << endl << endl;

         out << "#else //VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;
         if ( bHasDynamicSize( poBaseType ) )
         {
            out << endl
               << TAB << "if( TRUE == oLocalReaders.empty() )" << endl
               << TAB << "{" << endl
               << TAB << TAB << "for (tU32 it = 0; it < " << sChildName << ".size(); it++)" << endl
               << TAB << TAB << "{" << endl
               << TAB << TAB << TAB << sChildName << "[it].vDestroy();" << endl
               << TAB << TAB << "}" << endl
               << TAB << TAB << sChildName << ".clear();" << endl
               << TAB << "}" << endl
               << TAB << "else" << endl
               << TAB << "{" << endl
               << TAB << TAB << sChildName << ".clear();" << endl
               << TAB << TAB << "for (tU32 it = 0; it < oLocalReaders.size(); it++)" << endl
               << TAB << TAB << "{" << endl
               << TAB << TAB << TAB << "oLocalReaders[it]->vDestroy();" << endl
               << TAB << TAB << TAB << "OSAL_DELETE oLocalReaders[it];" << endl
               << TAB << TAB << "}" << endl
               << TAB << TAB << "oLocalReaders.clear();" << endl
               << TAB << "}" << endl;
         }
         else
         {
            out << TAB << sChildName << ".clear();" << endl;
         }
         out << "#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;
         out << "}" << endl << endl;

         bRetVal = true;
      }
      break;

      case EN_OBJTYPE_STRUCTURE:
      case EN_OBJTYPE_OPCODE:
      {
         out << "tVoid " << sClassName << "::vDestroy()" << endl;
         out << "{" << endl;

         string oTypeName = oPrefix + oGetClassName( CG_MSGTYPE_PREFIX + oRemove( poType->oGetName(), '.' ) );
         tclFITypeDependencyList* poDepList = poType->poGetDependencyList();

         for ( tclFITypeDependencyListIterator iter = poDepList->begin(); iter != poDepList->end(); ++iter )
         {
            tclFINode* poBaseType = (*iter)->poGetTarget();
            if ( (*iter)->enGetClass() == EN_DEPENDENCY_CLASS_PARAM
                 || (*iter)->enGetClass() == EN_DEPENDENCY_CLASS_FIELD )
            {
               string oFieldName = (*iter)->oGetName();
               if ( b_MOST_NeedsDestroy( poBaseType ) )
               {
                  if ( bHasDynamicSize( poBaseType ) )
                     out << TAB << oFieldName << ".vDestroy();" << endl;
               }
            }
         }

         out << "}" << endl << endl;
         bRetVal = true;
      }
      break;
      }
   }

   return bRetVal;
}


//-----------------------------------------------------------------------------

bool b_MOST_GenerateTypeGetSizeMethod(tclFINode* poType, ostream &out, tclFIServiceTree* poServiceTree)
{

   // provides:
   //  - formula to calculate type size (in bytes for streaming),
   //  - return statement
   //  - closing brace for function

   ostringstream memout;
   bool bRetVal = false;
   string sStorageClass = "";
   int iStaticSize = 0;
   string oPrefix = oToLower(poServiceTree->oGetShortname().substr(0,poServiceTree->oGetShortname().find_last_of('_')));
   string oTypeName;
   if (poType->enGetType() == EN_OBJTYPE_OPCODE)
      oTypeName = oPrefix + oGetClassName(CG_MSGTYPE_PREFIX + oRemove(poType->oGetName(),'.'));
   else
      oTypeName = oGetClassName(poType->oGetName());

   // add overhead related to storage class 
   DOMElement* poEl = poType->poGetDOMElement();
   if (poEl-> hasAttribute (X("storageClass")))
   {
      sStorageClass = X(poEl-> getAttribute (X("storageClass")));
      if (sStorageClass == "shortStream")
         iStaticSize += 1;
   }

   switch (poType->enGetType())
   {
   case EN_OBJTYPE_ARRAY:
      {
         string sBaseType = "";
         DOMElement* poArray
            = static_cast<DOMElement*> ((poEl-> getElementsByTagName(X("array")))-> item(0));
         DOMElement* poField
            = static_cast<DOMElement*> ((poArray-> getElementsByTagName(X("field")))-> item(0));
         string sChildName = (*poType->poGetDependencyList()->begin())->oGetName();
         if (poField-> hasAttribute (X("baseType")))
            sBaseType = oGetClassName(X(poField-> getAttribute (X("baseType"))));
         if (bIsIntrinsicType(sBaseType))
            out << "tU32 " << oTypeName << "::u32GetSize(tU16 /* u16MajorVersion */) const" << endl
                << "{" << endl
                << TAB << "return " << iStaticSize << " + sizeof(" << sBaseType << ")"
                   << " * " << sChildName << ".size();" << endl;
         else
            out << "tU32 " << oTypeName << "::u32GetSize(tU16 u16MajorVersion) const" << endl
                << "{" << endl
                << TAB << "tU32 u32DynamicSize = 0;" << endl
                << TAB << "for (tU32 iter = 0; iter < " << sChildName << ".size(); iter++)" << endl
                << TAB << "{" << endl
                << TAB << TAB << "u32DynamicSize += "
                << sChildName << "[iter].u32GetSize(u16MajorVersion);" << endl
                << TAB << "}" << endl
                << TAB << "return " << iStaticSize << " + u32DynamicSize;" << endl;
         out << "}" << endl << endl;
         bRetVal = true;
      }
      break;

   case EN_OBJTYPE_UNION:
      {
         string sFiGroupBaseType = oGetFIGroup() + "fi_tclVisitableTypeBase";
         out << "tU32 " << oTypeName << "::u32GetSize(tU16 u16MajorVersion) const" << endl
             << "{" << endl
             << TAB "const " << sFiGroupBaseType << "* poCurrentVariant = pcActiveMember();" << endl
             << TAB "if (poCurrentVariant)" << endl
             << TAB TAB "return (tU32)" << iStaticSize << " + poCurrentVariant->u32GetSize(u16MajorVersion);" << endl
             << TAB "else if (u32DynamicSize)" << endl
             << TAB TAB "return (tU32)" << iStaticSize << " + u32DynamicSize;" << endl
             << TAB "else" << endl
             << TAB TAB "return (tU32)" << iStaticSize << ";" << endl
             << "}" << endl;
         bRetVal = true;
      }
      break;

   case EN_OBJTYPE_FRAME:
      {
         // add frame size
         DOMElement* poFrame
            = static_cast<DOMElement*> ((poEl-> getElementsByTagName(X("frame")))-> item(0));
         if (poFrame-> hasAttribute (X("size")))
           iStaticSize += atoi(X(poFrame-> getAttribute (X("size"))));
         out << "tU32 " << oTypeName << "::u32GetSize(tU16 /* u16MajorVersion */) const" << endl
             << "{" << endl
             << TAB << "return " << iStaticSize << ";" << endl
             << "}" << endl << endl;

         bRetVal = true;
      }
      break;

   case EN_OBJTYPE_STRUCTURE:
   case EN_OBJTYPE_OPCODE:
      {
         tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
         for (tclFITypeDependencyListIterator iter = poDepList->begin(); iter != poDepList->end(); ++iter)
         {
            if (    ((*iter)->enGetClass() == EN_DEPENDENCY_CLASS_PARAM)
                 || ((*iter)->enGetClass() == EN_DEPENDENCY_CLASS_FIELD))
            {
               tclFINode* poBaseType = (*iter)->poGetTarget();
               if (bHasDynamicSize(poBaseType))
                  memout << TAB << "u32DynamicSize += " 
                         << (*iter)->oGetName() << ".u32GetSize(u16MajorVersion);" << endl;
               else 
                  iStaticSize += poBaseType->iGetDataSize();
            }
         }
         //if (memout.pcount() > 0) fke: pcount is not available on Linux
         if (memout.tellp() > 0)
         {
            out << "tU32 " << oTypeName << "::u32GetSize(tU16 u16MajorVersion) const" << endl
                << "{" << endl
                << TAB << "tU32 u32DynamicSize = 0;" << endl
                << memout.str()
                << TAB << "return " << iStaticSize << " + u32DynamicSize;" << endl;
         }
         else
            out << "tU32 " << oTypeName << "::u32GetSize(tU16 /* u16MajorVersion */) const" << endl
                << "{" << endl
                << TAB << "return " << iStaticSize << ";" << endl;
         out << "}" << endl << endl;
         bRetVal = true;
      }
      break;
   }


   return bRetVal;
}

//-----------------------------------------------------------------------------

bool b_MOST_GenerateTypeCompareMethod(tclFINode* poType, ostream &out,string oPrefix)
{
   bool bRetVal = false;
   string oTypeName = sGetTypeName(poType, oPrefix);

   switch (poType->enGetType())
   {
   case EN_OBJTYPE_ARRAY:
      {
         string sChildName = (*poType->poGetDependencyList()->begin())->oGetName();
         out << "tBool " << oTypeName << "::operator==(const " << oTypeName << "& roRef) const" << endl
             << "{" << endl
             << TAB "tBool bResult = TRUE;" << endl
             << TAB "for (tU32 u32 = 0; u32 < " << sChildName << ".size(); u32++)" << endl
             << TAB TAB "if ( !(" << sChildName << "[u32] == roRef." << sChildName << "[u32]) )" << endl
             << TAB TAB TAB "bResult = FALSE;" << endl;
         bRetVal = true;
      }
      break;
   case EN_OBJTYPE_UNION:
      {
         tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
         out << "tBool " << oTypeName << "::operator==(const " << oTypeName << "& roRef) const" << endl
             << "{" << endl
             << TAB "tBool bResult = TRUE;" << endl
             << TAB "if (_ActiveMemberOffset != roRef._ActiveMemberOffset)" << endl
             << TAB TAB "return FALSE;" << endl;
         for (tclFITypeDependencyListIterator iter = poDepList->begin(); iter != poDepList->end(); ++iter)
         {
            string sChildName = "_" + (*iter)->oGetName();
            out << TAB "else if (_ActiveMemberOffset == Offset(" << sChildName << "))" << endl
               << TAB TAB "return (" << sChildName << " == roRef." << sChildName << ");" << endl;
         }
         out << TAB "else if (u32DynamicSize != roRef.u32DynamicSize)" << endl
             << TAB TAB "return FALSE;" << endl
             << TAB "else if (pau8Data && roRef.pau8Data)" << endl
             << TAB TAB "for (tU32 u32 = 0; u32 < u32DynamicSize; u32++)" << endl
             << TAB TAB TAB "if (pau8Data[u32] != roRef.pau8Data[u32])" << endl
             << TAB TAB TAB TAB "return FALSE;" << endl;
         bRetVal = true;
      }
      break;

   case EN_OBJTYPE_FRAME:
      {
         out << "tBool " << oTypeName << "::operator==(const " << oTypeName << "& roRef) const" << endl
             << "{" << endl
             << TAB "tBool bResult = TRUE;" << endl
             << TAB "for (tU32 u32 = 0; u32 < sizeof(au8Data); u32++)" << endl
             << TAB TAB "if (au8Data[u32] != roRef.au8Data[u32])" << endl
             << TAB TAB TAB "bResult = FALSE;" << endl;
         bRetVal = true;
      }
      break;
   }

   if (bRetVal)
      out << TAB "return bResult;" << endl
          << "}" << endl << endl;

   return bRetVal;
}

//-----------------------------------------------------------------------------

bool b_MOST_GenerateTypeReadMethod(tclFINode *poType, ostream &out,string oPrefix)
{
   bool bRetVal = false;
   string sStorageClass = "";
   ostringstream sReadSize;
   string sStreamSize = "";
   string oTypeName = sGetTypeName(poType, oPrefix);

   // add overhead related to storage class 
   DOMElement* poEl = poType->poGetDOMElement();
   if (poEl-> hasAttribute (X("storageClass")))
      sStorageClass = X(poEl-> getAttribute (X("storageClass")));

   if (sStorageClass == "shortStream")
   {
      sReadSize << TAB << "tU8 u8Size;" << endl
                << TAB << "(tVoid) (oIn >> u8Size);" << endl;
      sStreamSize = "(tU32) u8Size;";
   }
   else if (sStorageClass == "trailingStream")
   {
      sStreamSize = "oIn.u32GetFreeSize();";
   }

   switch (poType->enGetType())
   {
   case EN_OBJTYPE_ARRAY:
      {
         tclFITypeDependency* poDep = *poType->poGetDependencyList()->begin();
         tclFINode* poBaseType = poDep->poGetTarget();
         string sChildName = poDep->oGetName();
         string sBaseType = oGetClassName(poBaseType->oGetName());

         out << "fi_tclInContext& " << oTypeName << "::oRead(fi_tclInContext& oIn)" << endl
             << "{" << endl
             << sReadSize.str()
             << TAB << "const tU32 u32StreamSize = " << sStreamSize << endl;
         if (bIsIntrinsicType(sBaseType))
            out << TAB << sChildName << ".resize(u32StreamSize / sizeof(" << sBaseType << "));" << endl
                << TAB << "for (tU32 u32Index = 0; u32Index < " << sChildName << ".size(); u32Index++)" << endl
                << TAB << TAB << "(tVoid) (oIn >> " << sChildName << "[u32Index]);" << endl
                << endl;
         else if (bHasDynamicSize(poBaseType))
           out  << TAB "const tU8* pu8ReadPosition = oIn.pu8GetPosition();" << endl
                << TAB "while (oIn.bIsValid() && ((tU32)(oIn.pu8GetPosition() - pu8ReadPosition) < u32StreamSize))" << endl
                << TAB "{" << endl
                << "#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl
                << TAB TAB << sChildName << ".push_back (" << sBaseType << "());" << endl
                << TAB TAB << "(tVoid) (oIn >> " << sChildName << ".back());" << endl
                << "#else" << endl
                << TAB TAB << "oLocalReaders.push_back(OSAL_NEW " << sBaseType << "());" << endl
                << TAB TAB << "(tVoid) (oIn >> *(oLocalReaders.back()));" << endl
                << TAB TAB << sChildName << ".push_back (*(oLocalReaders.back()));" << endl
                << "#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl
                << TAB "}" << endl;
         else
            out << TAB << sChildName << ".resize(u32StreamSize / " << poBaseType->iGetDataSize() << " /* size of " << sBaseType << " */ );" << endl
                << TAB << "for (tU32 u32Index = 0; u32Index < " << sChildName << ".size(); u32Index++)" << endl
                << TAB << TAB << "(tVoid) (oIn >> " << sChildName << "[u32Index]);" << endl
                << endl;
         bRetVal = true;
      }
      break;
   case EN_OBJTYPE_UNION:
      {
         out << "fi_tclInContext& " << oTypeName << "::oRead(fi_tclInContext& oIn)" << endl
             << "{" << endl
             << sReadSize.str() 
             << TAB << "u32DynamicSize = " << sStreamSize << endl;
         out << TAB << "if (pau8Data)" << endl
             << TAB << TAB << "OSAL_DELETE[] pau8Data;" << endl << endl

             << TAB << "pau8Data = OSAL_NEW tU8[u32DynamicSize];" << endl
             << TAB << "if (pau8Data)" << endl
             << TAB << TAB << "for (tU32 u32 = 0; u32 < u32DynamicSize; u32++)" << endl
             << TAB << TAB << TAB << "(tVoid) (oIn >> pau8Data[u32]);" << endl;
         bRetVal = true;
      }
      break;
   case EN_OBJTYPE_FRAME:
      {
         out << "fi_tclInContext& " << oTypeName << "::oRead(fi_tclInContext& oIn)" << endl
             << "{" << endl
             << sReadSize.str() 
             << TAB << "for (tU32 u32 = 0; u32 < sizeof(au8Data); u32++)" << endl
             << TAB << "{" << endl
             << TAB << TAB << "(tVoid) (oIn >> au8Data[u32]);" << endl
             << TAB << "}" << endl;
         bRetVal = true;
      }
      break;
   case EN_OBJTYPE_STRUCTURE:
      {
         out << "fi_tclInContext& " << oTypeName << "::oRead(fi_tclInContext& oIn)" << endl
             << "{" << endl
             << sReadSize.str();

         tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
         tclFITypeDependencyListIterator iter;
         for (iter = poDepList->begin(); iter != poDepList->end(); ++iter)
            out << TAB << "(tVoid) (oIn >> " 
                << (*iter)->oGetName() << ");" << endl;
            bRetVal = true;
      }
      break;

   }

   if (bRetVal)
      out << TAB << "return oIn;" << endl
          << "}" << endl << endl;

   return bRetVal;
}

//-----------------------------------------------------------------------------

bool b_MOST_GenerateTypeWriteMethod(tclFINode* poType, ostream &out,string oPrefix)
{
   bool bRetVal = false;
   string oTypeName = sGetTypeName(poType, oPrefix);
   string sStorageClass = "";
   ostringstream sWriteSize;

   // add overhead related to storage class 
   DOMElement* poEl = poType->poGetDOMElement();
   if (poEl-> hasAttribute (X("storageClass")))
      sStorageClass = X(poEl-> getAttribute (X("storageClass")));
   if (sStorageClass == "shortStream")
   {
      sWriteSize << TAB "(tVoid) (oOut << ((tU8)(u32GetSize(oOut.u16GetMajorVersion()) - 1U)));" << endl;
   }

   switch (poType->enGetType())
   {
   case EN_OBJTYPE_ARRAY:
      {
         string sBaseType = "";
         string sChildName = (*poType->poGetDependencyList()->begin())->oGetName();
         DOMElement* poArray
            = static_cast<DOMElement*> ((poEl-> getElementsByTagName(X("array")))-> item(0));
         DOMElement* poField
            = static_cast<DOMElement*> ((poArray-> getElementsByTagName(X("field")))-> item(0));
         sBaseType = oGetClassName(X(poField-> getAttribute (X("baseType"))));

         out << "fi_tclOutContext& " << oTypeName << "::oWrite(fi_tclOutContext& oOut) const" << endl
             << "{" << endl
             << sWriteSize.str()
             << TAB << "for (tU32 u32 = 0; u32 < " << sChildName << ".size(); u32++)" << endl
             << TAB << "{" << endl;
         if (bIsIntrinsicType(sBaseType))
            out << TAB << TAB << "(tVoid) (oOut << " << sChildName << "[u32]);" << endl;
         else
            out << TAB << TAB << "(tVoid) " << sChildName << "[u32].oWrite(oOut);" << endl;
         out << TAB << "}" << endl;
         bRetVal = true;
      }
      break;

   case EN_OBJTYPE_UNION:
      {
         string sFiGroupBaseType = oGetFIGroup() + "fi_tclVisitableTypeBase";
         out << "fi_tclOutContext& " << oTypeName << "::oWrite(fi_tclOutContext& oOut) const" << endl
             << "{" << endl
             << sWriteSize.str()
             << TAB "const " << sFiGroupBaseType << "* poCurrentVariant = pcActiveMember();" << endl
             << TAB "if (poCurrentVariant)" << endl
             << TAB TAB "(tVoid) (poCurrentVariant->oWrite(oOut));" << endl
             << endl;
         bRetVal = true;
      }
      break;

   case EN_OBJTYPE_FRAME:
      {
         out << "fi_tclOutContext& " << oTypeName << "::oWrite(fi_tclOutContext& oOut) const" << endl
             << "{" << endl
             << sWriteSize.str()
             << TAB << "for (tU32 u32 = 0; u32 < sizeof(au8Data); u32++)" << endl
             << TAB << "{" << endl
             << TAB << TAB << "(tVoid) (oOut << au8Data[u32]);" << endl
             << TAB << "}" << endl;
         bRetVal = true;
      }
      break;

   case EN_OBJTYPE_STRUCTURE:
      {
         out << "fi_tclOutContext& " << oTypeName << "::oWrite(fi_tclOutContext& oOut) const" << endl
             << "{" << endl
             << sWriteSize.str();

         tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
         tclFITypeDependencyListIterator iter;
         for (iter = poDepList->begin(); iter != poDepList->end(); ++iter)
            out << TAB << "(tVoid) (oOut << " 
                << (*iter)->oGetName() << ");" << endl;
         bRetVal = true;
      }
      break;

   }

   if (bRetVal)
      out << TAB << "return oOut;" << endl
          << "}" << endl << endl;

   return bRetVal;
}

//-----------------------------------------------------------------------------

bool b_MOST_GenerateDefaultErrorMessage(ostream &out, tclFIServiceTree* poServiceTree, bool bGenerateImplementation)
{

   string oPrefix = oToLower(poServiceTree->oGetShortname().substr(0,poServiceTree->oGetShortname().find_last_of('_')));
   string oClassName = oPrefix + "fi_tclMsgDefaultError";

   out << "//=============================================================================" << endl
       << endl
       << "// base class for all " << oPrefix << "fi standard error messages" << endl << endl;

   out << "#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl
       << oClassName <<"::" << oClassName << "(const " << oClassName << "& coRef)" << endl
       << TAB << ": " << oBaseMessageClassName << "(coRef)" << endl
       << "{" << endl
       << TAB << "*this = coRef;" << endl
       << "}" << endl
       << oClassName <<"& " << oClassName << "::operator=(const " << oClassName << "& coRef)" << endl
       << "{" << endl
       << TAB << "if (this != &coRef)" << endl
       << TAB << "{" << endl
       << TAB << TAB << "e8ErrorCode = coRef.e8ErrorCode;" << endl
       << TAB << TAB << "oErrorInfo = coRef.oErrorInfo;" << endl
       << TAB << "}" << endl << endl
       << TAB << "return *this;" << endl
       << "}" << endl
       << "#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl << endl;

   out << "tVoid " << oClassName << "::vDestroy()" << endl
       << "{" << endl
       << TAB << "oErrorInfo.vDestroy();" << endl
       << "}" << endl << endl;

   out << "tU32 " << oClassName << "::u32GetSize(tU16 u16MajorVersion) const" << endl
       << "{" << endl
       << TAB << "return 3 + oErrorInfo.u32GetSize(u16MajorVersion);" << endl
       << "}" << endl << endl;

   out << "fi_tclInContext& " << oClassName << "::oRead(fi_tclInContext& oIn)" << endl
       << "{" << endl
       << TAB << "tU16 u16CCA_Error = CCA_C_U16_ERROR_UNSPECIFIED;" << endl
       << TAB << "(tVoid) (oIn >> u16CCA_Error);" << endl
       << TAB << "(tVoid) (oIn >> e8ErrorCode);" << endl
       << TAB << "(tVoid) (oIn >> oErrorInfo);" << endl
       << TAB << "return oIn;" << endl
       << "}" << endl << endl;

   out << "fi_tclOutContext& " << oClassName << "::oWrite(fi_tclOutContext& oOut) const" << endl
       << "{" << endl
       << TAB << "(tVoid) (oOut << (tU16)CCA_C_U16_ERROR_UNSPECIFIED);" << endl
       << TAB << "(tVoid) (oOut << e8ErrorCode);" << endl
       << TAB << "(tVoid) (oOut << oErrorInfo);" << endl
       << TAB << "return oOut;" << endl
       << "}" << endl << endl;

   return true;
}

bool b_MOST_GenerateDefaultErrorMessageDecl(ostream &out, tclFIServiceTree* poServiceTree, bool bGenerateImplementation)
{

   string oPrefix = oToLower(poServiceTree->oGetShortname().substr(0,poServiceTree->oGetShortname().find_last_of('_')));
   string oClassName = oPrefix + "fi_tclMsgDefaultError";

   out  << "// base class for all " << oPrefix << "fi standard error messages" << endl << endl;

   out << "class " << oClassName << ": public " << oBaseMessageClassName << endl
       << "{" << endl
       << "public:" << endl
       << TAB << "// member variables" << endl
       << TAB << oGetFIGroup() << "fi_tcl_e8_ErrorCode e8ErrorCode;" << endl
       << TAB << oGetFIGroup() << "fi_tcl_TrailingStream oErrorInfo;" << endl
       << TAB << endl
       << TAB << oClassName << "() : e8ErrorCode(), oErrorInfo() {};" << endl
       << "#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl
       << TAB << oClassName << "(const " << oClassName << "& coRef);" << endl
       << TAB << oClassName << "& operator=(const " << oClassName << "& coRef);" << endl
       << "#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl << endl
       << TAB << "virtual tVoid vDestroy();" << endl
       << TAB << "virtual tU32 u32GetSize(tU16 u16MajorVersion) const;" << endl
       << TAB << "virtual fi_tclInContext& oRead(fi_tclInContext& oIn);" << endl
       << TAB << "virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOut) const;" << endl
       << "};" << endl << endl;

   out << "//=============================================================================" << endl
       << endl;

   return true;
}
