/*****************************************************************************
 * FILE:          FICodeGenMain.cpp
 * SW-COMPONENT:
 * DESCRIPTION:   Main module for (new) FI code generator based on c++ fi tooling
 *
 * AUTHOR:        CM-DI/ESN1-Bruns,Jochen
 * COPYRIGHT:     (c) 2004 Blaupunkt Werke GmbH
 * HISTORY:
 *                04.03.04 Rev. 1.0 CM-DI/ESN1-Bruns,J
 *                         Initial Revision;
 *****************************************************************************/
#pragma warning( disable : 4786 )  
#pragma warning( disable : 4251 )  


//-----------------------------------------------------------------------------
// General includes
//-----------------------------------------------------------------------------
#include "FIServiceGroup.h"

#include "FIVersion.h"
#include "FICodeGenCmdLine.h"
#include "FICodeGenUtil.h"
#include "FIUtil.h"

//#include "DOMDocPrint.h"
//#include "FIExplorerDetailPrint.h"

#include <set>
#include <string>
#include <iostream>
#include <sstream>
#include <fstream>
#include <assert.h>
#ifdef WIN32 
#include "direct.h"
#else
#include <sys/stat.h>
#include <sys/types.h>


#endif

#include <errno.h>
//#include <xercesc/util/PlatformUtils.hpp>
//using namespace xercesc;

#include "FiCodeGenExtender.h"

/*****************************************************************************
| variables (scope: local)
|----------------------------------------------------------------------------*/

string oBaseMessageClassName = "";
string oCodeGeneratorVersion = "FICodeGen (C++) version 1.9.14";

/*****************************************************************************
| function prototypes (scope: local)
|----------------------------------------------------------------------------*/

bool bIsPredefinedType( tclFINode *poType );
bool bNeedsBasicDestroyMethod( tclFINode *poType );


/*****************************************************************************
| function implementation
|-----------------------------------------------------------------------------*/

static bool bNeedsAnyDestroy( tclFITypeDependencyList& rfoDepList )
{
   bool bNeedsDestroy = false;
   tclFITypeDependencyListIterator iter;

   for ( iter = rfoDepList.begin();
      (iter != rfoDepList.end()) && !bNeedsDestroy;
      ++iter )
   {
      bNeedsDestroy = ((*iter)->enGetClass() == EN_DEPENDENCY_CLASS_LIST)
         || ((*iter)->enGetClass() == EN_DEPENDENCY_CLASS_UNION)
         || bNeedsBasicDestroyMethod( (*iter)->poGetTarget() );
   }

   return bNeedsDestroy;
}

bool bNeedsBasicDestroyMethod( tclFINode *poType )
{
   bool bNeedsDestroy = false;
   tclFITypeDependencyList* poDepList = poType->poGetDependencyList();

   bNeedsDestroy = ((poType->oGetName() == "tString")
                  || (poType->oGetName() == "tMultiLanguageString"));

   if ( !bNeedsDestroy
      && (NULL != poDepList)
      && !poDepList->empty() )
   {
      bNeedsDestroy = bNeedsAnyDestroy( *poDepList );
   }

   return bNeedsDestroy;
}

bool bNeedsDestroyMethod( tclFINode *poType )
{
   return bNeedsDestroyExt( poType ) || bNeedsBasicDestroyMethod( poType );
}

bool bIsPredefinedType( tclFINode *poType )
{
   // checks for boolean attribute "predefined" in type elements
   // (mkoch, 2010-03-04)

   assert( poType );

   string predefined = "false";
   DOMElement* pEl = poType->poGetDOMElement();

   if ( pEl->hasAttribute( X( "predefined" ) ) )
     predefined = X( pEl->getAttribute( X( "predefined" ) ) );

   return (predefined == "true");
}


bool bReadXMLCatalogues( tclFIServiceGroup *poServiceGroup )
{
   bool bSuccess = true;
   tclStringListIterator iter = oFileList.begin();
   while ( iter != oFileList.end() )
   {
      string oFilename = goFIBaseDir + (*iter);
      // read and validate
      cerr << "working on file: " << oFilename << endl;
      bSuccess &= poServiceGroup->bReadService( const_cast<char*>(oFilename.c_str()) );
      ++iter;
   }
   return bSuccess;
}


string oGetFIPrefix( tclFIServiceTree* poServiceTree )
{
   string oPrefix;
   if ( poServiceTree )
   {
      oPrefix += poServiceTree->oGetShortname().substr( 0, poServiceTree->oGetShortname().size() - 3 );
      oPrefix += "fi";
   }
   return oPrefix;
}

// -----------------------------------------------------------------------------

bool bGenerateComponentTypeFactory( tclFIServiceTree* poCatalogueTree, ostream &out )
{
   string oSrvPrefix = oToLower( oGetFIPrefix( poCatalogueTree ) );
   tclFIMessageNodePool* poMsgPool = poCatalogueTree->poGetMessagePool();
   int iNumOfMessages = poMsgPool->iGetNumNodes();

   out << "// =============================================================================" << endl
          << "//" << endl
          << "//                Factory for " << poCatalogueTree->oGetShortname() << " message types" << endl
          << endl;

   out << "fi_tclMessageBase* " << oSrvPrefix << "_poGetMessageBaseObject(tU16 u16FunctionId, tU8 u8Opcode)" << endl
       << "{" << endl
       << TAB << "switch (((tU32)u16FunctionId << 16u) + ((tU32)u8Opcode << 8u))" << endl
       << TAB << "{" << endl;
   for ( int iMsgIdx = 0; iMsgIdx < iNumOfMessages; ++iMsgIdx )
   {
      tclFINode* poNode = poMsgPool->poGetNode( iMsgIdx, EN_NONE );
      if ( (poNode) &&
          (poNode->oGetName().find( ".Error" ) == string::npos) )
      {
         tenFINodeType enNodeType = poNode->enGetType();
         string oMsgName = oRemove( poNode->oGetName(), '.' );
         int iDataSize = poNode->iGetDataSize();
         if ( EN_OBJTYPE_OPCODE == enNodeType )
         {
            string oTokenName = bMixedCaseTokens ? oMsgName : oToUpper( oMsgName );
            out << TAB << TAB << "case " << oSrvPrefix
                << "_tclToken::EN_MSG_" << oTokenName << ":" << endl;
            out << TAB << TAB << TAB << "return OSAL_NEW "
                << oSrvPrefix << "_tclMsg" << oMsgName << ";" << endl
                << endl;
         }
      }
   }
   out << TAB << TAB << "default: " << endl
       << TAB << TAB << TAB << "return OSAL_NULL;" << endl
       << TAB << "}" << endl
       << "}" << endl
       << endl;

   return true;
}

// -----------------------------------------------------------------------------

bool bGenerateTypeFactory( tclFIServiceGroup *poServiceGroup )
{
   if ( poServiceGroup->iGetNumServices() <= 0 )
      return false;
   string oFileName = goOutputDir;
   if ( oFileName.size() == 0 )
   {
      oFileName = poServiceGroup->poGetServiceTree( 0 )->oGetFilename();
      int iLastSlashPos = oFileName.find_last_of( '/' );
      oFileName = oFileName.substr( 0, iLastSlashPos );
      oFileName = oFileName + "/../";
   }
   oFileName += "types/inc/" + oGetFIGroup() + "alltypefactory.cpp";
   ofstream out( oFileName.c_str() );
   if ( 0 != poServiceGroup )
   {
      if ( bGeneratePrecompiledHeader )
      {
         out << "#include \"precompiled.hh\"" << endl;
      }
      else
      {
         out << "#if (defined OSAL_OS) || (defined OSAL_CONF)" << endl;
         out << "#define OSAL_S_IMPORT_INTERFACE_GENERIC" << endl;
         out << "#include \"osal_if.h\"" << endl;
         out << "#else" << endl;
         //out << "#include <iostream.h>" << endl;
         //out << "#include <iomanip.h>" << endl;
         out << "#include \"myosal.h\"" << endl;
         out << "#endif" << endl;
         out << endl;
         tclStringListIterator iter = oFileList.begin();
         while ( iter != oFileList.end() )
         {
            string oXMLFile = goFIBaseDir + (*iter);
            tclFIServiceTree *poCatalogueTree = poServiceGroup->poGetServiceTree( const_cast<char*>(oXMLFile.c_str()) );
            if ( 0 != poCatalogueTree )
            {
               out << "#define " << oToUpper( oGetFIGroup() ) << "FI_S_IMPORT_INTERFACE_" << oToUpper( oGetFIPrefix( poCatalogueTree ) ) << "_TYPES" << endl;
               out << "#define " << oToUpper( oGetFIGroup() ) << "FI_S_IMPORT_INTERFACE_" << oToUpper( oGetFIPrefix( poCatalogueTree ) ) << "_SERVICEINFO" << endl;
            }
            ++iter;
         }
         out << "#include \"figroup_fi_if.h\"" << endl << endl;
      }
      out << "#include \"alltypefactory.h\"" << endl;
      out << endl;
      out << "fi_tclTypeBase* figroup_fi_poGetFIObject(tU16 u16ServiceId, tU16 u16FunctionId, "
         << "tU8 u8Opcode)" << endl;
      out << "{" << endl;
      out << TAB << "fi_tclMessageBase* poType" << endl;
      out << TAB << "= figroup_fi_poGetMessageBaseObject(u16ServiceId, u16FunctionId, u8Opcode);" << endl;
      out << TAB << "if (poType)" << endl;
      out << TAB << "   return &poType->rfoGetTypeBase();" << endl;
      out << TAB << "else" << endl;
      out << TAB << "   return OSAL_NULL;" << endl;
      out << "}" << endl << endl;

      out << "fi_tclMessageBase* figroup_fi_poGetMessageBaseObject(tU16 u16ServiceId, tU16 u16FunctionId, "
         << "tU8 u8Opcode)" << endl;
      out << "{" << endl;
      out << TAB << "switch(u16ServiceId)" << endl;
      out << TAB << "{" << endl;

      tclStringListIterator iter = oFileList.begin();
      while ( iter != oFileList.end() )
      {
         string oXMLFile = goFIBaseDir + (*iter);
         tclFIServiceTree *poCatalogueTree = poServiceGroup->poGetServiceTree( const_cast<char*>(oXMLFile.c_str()) );
         if ( 0 != poCatalogueTree )
         {
            string oSrvPrefix = oToLower( oGetFIPrefix( poCatalogueTree ) );
            out << TAB << TAB << "case " << oToUpper( oSrvPrefix ) << "_C_U16_SERVICE_ID:" << endl
                << TAB << TAB << TAB << "return " << oSrvPrefix << "_poGetMessageBaseObject(u16FunctionId, u8Opcode); " << endl
                << endl;
         }
         ++iter;
      }
      out << TAB << TAB << "default: " << endl
          << TAB << TAB << TAB << "return OSAL_NULL;" << endl
          << TAB << "}" << endl;
      out << "}" << endl << endl;
   }
   return true;
}

string oGetServiceTreeSubDirectory( tclFIServiceTree* poServiceTree )
{
   //string oFileName = poServiceTree->oGetFilename();
   //int iLastSlashPos = oFileName.find_last_of('/');
   //return oFileName.substr(0,iLastSlashPos) + "/inc";

   string oDirName = goOutputDir;
   if ( oDirName.size() == 0 )
   {
      oDirName = poServiceTree->oGetFilename();
      int iLastSlashPos = oDirName.find_last_of( '/' );
      oDirName = oDirName.substr( 0, iLastSlashPos );
   }
   else
   {
      string oSubDirName = poServiceTree->oGetFilename();
      int iLastSlashPos = oSubDirName.find_last_of( '/' );
      oSubDirName = oSubDirName.substr( 0, iLastSlashPos );
      iLastSlashPos = oSubDirName.find_last_of( '/' );
      oSubDirName = oSubDirName.substr( iLastSlashPos );
      oDirName += oSubDirName.substr( 1 );
   }
   oDirName += "/inc";
   return oDirName;
}


bool bGenerateServiceInfo( tclFIServiceTree* poServiceTree )
{
   // create inc-files
   string oDirName = oGetServiceTreeSubDirectory( poServiceTree );
   ofstream oOutSrvInfoH( string( oDirName + "/serviceinfo.inc" ).c_str() );
   if ( !oOutSrvInfoH.good() )
      return false;
   ofstream oOutSrvInfoC( string( oDirName + "/serviceinfo_c.inc" ).c_str() );
   if ( !oOutSrvInfoC.good() )
      return false;

   string oPrefix = oToUpper( oGetFIPrefix( poServiceTree ) );
   oOutSrvInfoH << "extern const tChar " << oPrefix
      << "_C_STRING_SERVICE_NAME[];" << endl;
   oOutSrvInfoH << "extern const tChar " << oPrefix
      << "_C_STRING_SERVICE_SHORTNAME[];" << endl;
   oOutSrvInfoH << "const tU16 " << oPrefix << "_C_U16_SERVICE_ID = "
      << poServiceTree->iGetServiceID() << ";" << endl;

   tclFIVersionList* poVersionList = poServiceTree->poGetCatalogueVersionList();
   tclFIVersionListIterator iter;
   tclFIVersion oMaxVersion;

   // generate service version info table
   oOutSrvInfoH << endl;
   for ( iter = poVersionList->begin(); iter != poVersionList->end(); ++iter )
   {
      for ( int iMinorVersion = iter->iGetMinor(); iMinorVersion >= 0; --iMinorVersion )
      {
         oOutSrvInfoH << "extern const tFIVersionInfo " << oPrefix << "_C_SERVICE_VERSION_"
            << iter->iGetMajor() << "_" << iMinorVersion << ";" << endl;
         oOutSrvInfoC << "extern const tFIVersionInfo " << oPrefix << "_C_SERVICE_VERSION_"
            << iter->iGetMajor() << "_" << iMinorVersion << " = { "
            << iter->iGetMajor() << " , " << iMinorVersion << "};" << endl;
      }
   }

   oOutSrvInfoH << endl << "extern const tU16 " << oPrefix << "_C_U16_SERVICE_NUMBER_OF_VERSIONS;"
      << endl << endl;
   oOutSrvInfoC << endl << "extern const tU16 " << oPrefix << "_C_U16_SERVICE_NUMBER_OF_VERSIONS = "
      << poVersionList->size() << ";" << endl << endl;

   oOutSrvInfoH << "extern const tFIVersionInfo ar" << oRemove( poServiceTree->oGetShortname(), '_' )
      << "_ServiceArray[];" << endl;
   oOutSrvInfoC << "extern const tFIVersionInfo ar" << oRemove( poServiceTree->oGetShortname(), '_' )
      << "_ServiceArray[] = " << endl;
   oOutSrvInfoC << "{" << endl;
   iter = poVersionList->begin();
   while ( iter != poVersionList->end() )
   {
      tclFIVersion oTempVersion = *iter;

      if ( oMaxVersion < oTempVersion )
         oMaxVersion = oTempVersion;

      oOutSrvInfoC << TAB << "{ "
         << oPrefix << "_C_SERVICE_VERSION_" << oTempVersion.iGetMajor() << "_" << oTempVersion.iGetMinor() << ".u16MajorVersion,"
         << oPrefix << "_C_SERVICE_VERSION_" << oTempVersion.iGetMajor() << "_" << oTempVersion.iGetMinor() << ".u16MinorVersion"
         << " }";
      ++iter;
      if ( iter != poVersionList->end() )
         oOutSrvInfoC << "," << endl;
      else
         oOutSrvInfoC << endl;
   }
   oOutSrvInfoC << "};" << endl << endl;

   // create defines and strings for max version entry
   oOutSrvInfoH << "extern const tU16 " << oPrefix << "_C_U16_SERVICE_MAJORVERSION;" << endl;
   oOutSrvInfoH << "extern const tU16 " << oPrefix << "_C_U16_SERVICE_MINORVERSION;" << endl;

   oOutSrvInfoC << "extern const tU16 " << oPrefix << "_C_U16_SERVICE_MAJORVERSION = "
      << oMaxVersion.iGetMajor() << ";" << endl;
   oOutSrvInfoC << "extern const tU16 " << oPrefix << "_C_U16_SERVICE_MINORVERSION = "
      << oMaxVersion.iGetMinor() << ";" << endl;

   oOutSrvInfoH << "extern const tChar " << oPrefix
      << "_C_STRING_SERVICE_VERSIONSTRING[];" << endl;

   oOutSrvInfoC << "extern const tChar " << oPrefix
      << "_C_STRING_SERVICE_SHORTNAME[] = \"" << poServiceTree->oGetShortname()
      << "\";" << endl;
   oOutSrvInfoC << "extern const tChar " << oPrefix
      << "_C_STRING_SERVICE_NAME[] = \"" << poServiceTree->oGetFullname()
      << "\";" << endl;
   oOutSrvInfoC << "extern const tChar " << oPrefix
      << "_C_STRING_SERVICE_VERSIONSTRING[] = \"" << poServiceTree->oGetFullname()
      << " (" << poServiceTree->oGetShortname() << ") "
      << "v" << oMaxVersion.iGetMajor() << "." << oMaxVersion.iGetMinor() << "\";" << endl;

   return true;
}

bool bGenerateErrorInfo( tclFIServiceTree* poServiceTree )
{
   string oDirName = oGetServiceTreeSubDirectory( poServiceTree );
   ofstream oOutErrorIds( string( oDirName + "/errorids.inc" ).c_str() );
   if ( !oOutErrorIds.good() )
      return false;
   string oPrefix = oToUpper( oGetFIPrefix( poServiceTree ) );

   oOutErrorIds << "//" << endl;
   oOutErrorIds << "// definition of error codes" << endl;
   oOutErrorIds << "//" << endl << endl;
   oOutErrorIds << "enum {" << endl;
   tclFITypeNodePool *poErrorPool = poServiceTree->poGetUsedGlobalErrorPool();
   for ( int s32Idx = 0; s32Idx < poErrorPool->iGetNumNodes(); ++s32Idx )
   {
      tclFINode* poErrorNode = poErrorPool->poGetNode( s32Idx, EN_NONE );
      oOutErrorIds << TAB << oPrefix << "_C_U16_ERROR_" << oToUpper( poErrorNode->oGetName() )
         << " = " << poErrorNode->oGetDefaultValue();
      if ( s32Idx < (poErrorPool->iGetNumNodes() - 1) )
      {
         oOutErrorIds << ",";
      }
      oOutErrorIds << endl;
   }
   oOutErrorIds << "};" << endl;
   return true;
}

bool bGenerateTypeClassDeclaration( tclFINode *poType, ostream &out, tclFIServiceTree* poServiceTree )
{
   bool bSuccess = true;

   // shifted below early return inserted for skipping predefined types
   // (mkoch, 2010-03-04)
   //if (bIsBinContainerType(poType->oGetName()))
   //{
   //   out << "#ifndef FI_S_EXTERNAL_FI" << endl;
   //}

   string oTypeName;
   if ( poType->enGetType() == EN_OBJTYPE_OPCODE )
   {
      oTypeName = oToLower( poServiceTree->oGetShortname().substr( 0, poServiceTree->oGetShortname().find_last_of( '_' ) ) );
      oTypeName += oGetClassName( CG_MSGTYPE_PREFIX + oRemove( poType->oGetName(), '.' ) );
   }
   else
      oTypeName = oGetClassName( poType->oGetName() );

   // skip types having attribute "predefined" (mkoch, 2010-03-04)
   if ( bIsPredefinedType( poType ) )
   {
      out << "/****** using predefined declaration for type ";
      out << oGetClassName( oTypeName ) << " ******/" << endl << endl;
      return true;
   }

   if ( bIsBinContainerType( poType->oGetName() ) )
   {
      out << "#ifndef FI_S_EXTERNAL_FI" << endl;
   }
   // ----
   out << "class " << oTypeName << " : public ";

   if ( poType->enGetType() != EN_OBJTYPE_OPCODE )
      out << oGetFIGroup() << "fi_tclVisitableTypeBase";
   else if ( bIsStandardErrorMessageExt( poType ) )
   {
      string oPrefix = oToLower( poServiceTree->oGetShortname().substr( 0, poServiceTree->oGetShortname().find_last_of( '_' ) ) );
      out << oPrefix << "fi_tclMsgDefaultError";
   }
   else
      out << oBaseMessageClassName;

   out << endl;

   out << "{" << endl;
   out << "public:" << endl << endl;


   bool bNeedsDestroy = false || bNeedsDestroyExt( poType );
   tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
   // write sort specific things

   // standard constructor 
   out << TAB << oTypeName;

   if ( !bGenerateExtendedTypeClass( poType, out, poServiceTree, bNeedsDestroy, bSuccess ) )
   {
      switch ( poType->enGetType() )
      {
      case EN_OBJTYPE_ATOM:
         break;
      case EN_OBJTYPE_TYPE:
      {
         tclFINode* poBaseType = (*poDepList->begin())->poGetTarget();
         out << "(" << poBaseType->oGetName() << " _" << oGetDataFieldName( poBaseType )
            << " = ";
         if ( poType->oGetDefaultValue() != "" )
         {
            out << poType->oGetDefaultValue();
         }
         else
         {
            out << "0";
         }

         out << ");" << endl << endl;

         out << TAB << oGetClassName( poBaseType->oGetName() ) << " "
            << oGetDataFieldName( poBaseType ) << ";" << endl;
         if ( poBaseType->oGetName() == "tString" )
         {
            bNeedsDestroy = true;
         }
      } break;
      case EN_OBJTYPE_ENUMERATION:
      {
         out << "();\n";
         out << TAB << "enum tenType {\n";
         tclFITypeDependencyListIterator iter;
         for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
         {
            out << TAB << TAB << oGetEnumName( (*iter)->oGetName() )
               << " = " << strtoul( (*iter)->oGetDefaultValue().c_str(), 0, 10 ) << "UL"
               << (((iter + 1) == poDepList->end()) ? "" : ",") << "\n";
         }
         out << TAB << "};\n";
         out << TAB << "tenType " << CG_DATAFIELD_ENUM << ";\n\n";
         out << TAB << oTypeName << "(" << oTypeName << "::tenType newValue)\n";
         out << TAB << TAB << ":enType(newValue)\n";
         out << TAB << "{}\n\n";
      } break;
      case EN_OBJTYPE_BITFIELD:
      {
         tclFINode* poBaseType = (*poDepList->begin())->poGetTarget();
         string oDataFieldName = oGetDataFieldName( poBaseType );

         out << "();" << endl << endl;
         out << TAB << "enum {" << endl;
         tclFITypeDependencyListIterator iter;
         for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
         {
            int iBitPos = strtol( (*iter)->oGetDefaultValue().c_str(), 0, 0 );
            out << TAB << TAB << "FI_C_"
               << oToUpper( (*iter)->poGetTarget()->oGetName().substr( 1 ) )
               << "_BIT_" << oToUpper( (*iter)->oGetName() )
               // << " = 1 << " << (*iter)->oGetDefaultValue()
               << " = 0x" << hex << (1 << iBitPos) << dec << "UL"
               << (((iter + 1) == poDepList->end()) ? "" : ",") << endl;
         }
         out << TAB << "};" << endl;
         for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
         {
            out << TAB << "tBool b" << (*iter)->oGetName() << "() const {"
               << "return (" << oDataFieldName << " & "
               << "FI_C_" << oToUpper( (*iter)->poGetTarget()->oGetName().substr( 1 ) )
               << "_BIT_" << oToUpper( (*iter)->oGetName() )
               << ") != 0; }" << endl;
            out << TAB << "tVoid vSet" << (*iter)->oGetName() << "(tBool bSetBit) { "
               << oDataFieldName << " = "
               << "(" << poBaseType->oGetName() << ")("
               << "(bSetBit)?(" << oDataFieldName << " | "
               << "FI_C_" << oToUpper((*iter)->poGetTarget()->oGetName().substr(1))
               << "_BIT_" << oToUpper((*iter)->oGetName())
               << "):(" << oDataFieldName << " & ~(" << poBaseType->oGetName() << ")"
               << "FI_C_" << oToUpper((*iter)->poGetTarget()->oGetName().substr(1))
               << "_BIT_" << oToUpper((*iter)->oGetName())
               << ")); }" << endl;
         }
#if 1
         // for compatibilty reasons
         out << TAB << "struct rBitValues {" << endl;
         int iMSBPos = strtol( poBaseType->oGetName().substr( 2 ).c_str(), 0, 10 );
         for ( int iBitPos = 0; iBitPos < iMSBPos; ++iBitPos )
         {
            bool bFound = false;
            for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
            {
               if ( iBitPos == strtol( (*iter)->oGetDefaultValue().c_str(), 0, 10 ) )
               {
                  bFound = true;
                  out << TAB << TAB << "tBool " << (*iter)->oGetName() << ":1;" << endl;
               }
            }
            if ( !bFound )
               out << TAB << TAB << "tBool dummy" << iBitPos << ":1;" << endl;
         }
         out << TAB << "};" << endl;
         out << TAB << "union {" << endl;
         out << TAB << TAB << poBaseType->oGetName() << " "
            << oDataFieldName << ";" << endl;
         out << TAB << TAB << "rBitValues bits;" << endl;
         out << TAB << "};" << endl;
#else
         out << TAB << poBaseType->oGetName() << " "
            << oDataFieldName << ";" << endl;
#endif
      } break;
      case EN_OBJTYPE_OPCODE:
      case EN_OBJTYPE_STRUCTURE:
      {
         if ( bIsStandardErrorMessageExt( poType ) )
         {
            string oPrefix = oToLower( poServiceTree->oGetShortname().substr( 0, poServiceTree->oGetShortname().find_last_of( '_' ) ) );
            out << "() " << ": " << oPrefix << "fi_tclMsgDefaultError" << "() {}" << endl;
         }
         else
         {
            out << "();" << endl << endl;
         }

         bool bNeedsDestructor = false;
         if ( bEnableDeepCopy && (poDepList->size() > 0) )
         {
            // this group of methods changes the known behaviour!
            out << "#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;
            out << TAB << oTypeName << "& operator=(const " << oTypeName << "& coRef);" << endl;
            out << TAB << oTypeName << "(const " << oTypeName << "& coRef);" << endl;
            out << "#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl << endl;
            //         out << TAB << oTypeName << "& rfoClone(const " << oTypeName << "& rfoInit);" << endl;
         }

         if ( !bIsStandardErrorMessageExt( poType ) )
            for ( tclFITypeDependencyListIterator iter = poDepList->begin(); iter != poDepList->end(); ++iter )
            {
               if ( ((*iter)->enGetClass() == EN_DEPENDENCY_CLASS_FIELD) ||
                   ((*iter)->enGetClass() == EN_DEPENDENCY_CLASS_PARAM) )
               {
                  tclFINode* poBaseType = (*iter)->poGetTarget();
                  switch ( poBaseType->enGetType() )
                  {
                  case EN_OBJTYPE_ATOM:
                     if ( (poBaseType->oGetName() == "tString") ||
                         (poBaseType->oGetName() == "tMultiLanguageString") )
                     {
                        bNeedsDestroy = true;
                     }
                     break;
                  case EN_OBJTYPE_TYPE:
                  {
                     string localString( "tString" );
                     if ( (poBaseType->poFindDependency( localString ) != 0) ||
                           (poBaseType->oGetName() == "tMultiLanguageString") )
                     {
                        bNeedsDestroy = true;
                     }
                  }
                  break;
                  case EN_OBJTYPE_STRUCTURE:
                     bNeedsDestroy = true;
                     break;
                  }
                  bNeedsDestroy |= bNeedsDestroyExt( poBaseType );
               }
               if ( (*iter)->enGetClass() == EN_DEPENDENCY_CLASS_LIST )
               {
                  bNeedsDestroy = true;
               }
               if ( (*iter)->enGetClass() == EN_DEPENDENCY_CLASS_UNION )
               {
                  if ( bEnableDeepCopy )
                     bNeedsDestructor = true;

                  bNeedsDestroy = true;
                  tclFINode* poBaseType = (*iter)->poGetTarget();
                  switch ( poBaseType->enGetType() )
                  {
                  case EN_OBJTYPE_ENUMERATION:
                     out << TAB << oGetClassName( poBaseType->oGetName() ) << " "
                        << (*iter)->oGetName() << "Type;" << endl;
                     out << TAB << "fi_tclTypeBase* po" << (*iter)->oGetName()
                         << "Data;" << endl;
                     break;
                  case EN_OBJTYPE_BITFIELD:
                     out << TAB << oGetClassName( poBaseType->oGetName() ) << " " << (*iter)->oGetName()
                        << "Mask;" << endl;
                     tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
                     tclFITypeDependencyListIterator iter2;
                     int iElemIdx = 0;
                     for ( int iBitIdx = 32; iBitIdx > 0; )
                     {
                        --iBitIdx;
                        for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
                        {
                           if ( iBitIdx == strtoul( (*iter2)->oGetDefaultValue().c_str(), 0, 0 ) )
                           {
                              //                           out << "PRIVATE" << endl;
                              out << TAB << oGetClassName( (*iter2)->oGetName() ) << "* poOption" << iElemIdx << ";" << endl;
                              iElemIdx;
                              //                           out << "PUBLIC" << endl;
                              out << TAB << oGetClassName( (*iter2)->oGetName() ) << "* poGet" << (*iter2)->oGetName().substr( 1 )
                                 << "() { return poOption" << iElemIdx << ";}" << endl;
                              ++iElemIdx;
                           }
                        }
                     }
                     break;
                  }
               }
               else
               {
                  string oElementBaseTypeName = (*iter)->poGetTarget()->oGetName();
                  if ( bIsBinContainerType( oElementBaseTypeName ) )
                  {
                     // For safety reason, hide bincontainer structure also internally!
                     out << TAB << oGetClassName( "T_BinContainer",
                                                 (*iter)->enGetClass() == EN_DEPENDENCY_CLASS_LIST )
                        << " " << (*iter)->oGetName() << ";" << endl;
                  }
                  else
                  {
                     out << TAB << oGetClassName( oElementBaseTypeName,
                                                 (*iter)->enGetClass() == EN_DEPENDENCY_CLASS_LIST ) << " "
                        << (*iter)->oGetName() << ";" << endl;
                  }
               }
            }
      } break;
      default:
         out << "/*default*/" << endl;
         break;
      }
   }

   out << TAB << "virtual ~" << oTypeName << "();" << endl;

   // only override vDestroy if needed, otherwise just
   // stick to the base class implementation.
   if ( bNeedsDestroyMethod( poType ) )
   {
      out << TAB << "virtual tVoid vDestroy();" << endl;
   }

   out << endl;

   if ( !bUseNonStandardTypeMethods( poType, out ) )
   {
      // standard methods for all sort of types
#pragma message("ATTENTION: Change this in final version!")
      //   out << TAB << "tU32 u32GetSize(tU16 u16MajorVersion) const;" << endl;
      if ( ((poType->enGetType() != EN_OBJTYPE_OPCODE)
                || (poType->poGetDependencyList()->size() != 0))
           && !bIsStandardErrorMessageExt( poType ) )
      {
         out << TAB << "virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;" << endl;
         out << TAB << "virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;" << endl;
         out << TAB << "virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);" << endl;
      }

      // declare type compare method outside class declaration
      //   - Lint Info 1702: 'operator==' is both an ordinary function and a member function
      if ( ((poType->enGetType() == EN_OBJTYPE_STRUCTURE) || (poType->enGetType() == EN_OBJTYPE_OPCODE)) &&
       (poType->poGetDependencyList()->size() == 0) )
      {
         out << TAB "inline tBool operator==(const " << oTypeName << "& /* roRef */) const" << endl
            << TAB TAB "{  return true; }" << endl;
      }
      else if ( poType->enGetType() == EN_OBJTYPE_TYPE )
      {
         tclFINode* poBaseType = (*poType->poGetDependencyList()->begin())->poGetTarget();
         out << TAB "inline tBool operator==(const " << oTypeName << "& roRef) const" << endl
             << TAB TAB "{  return (" << oGetDataFieldName( poType ) << " == roRef." << oGetDataFieldName( poType ) << ");  }" << endl;
      }
      else if ( (poType->enGetType() == EN_OBJTYPE_BITFIELD) || (poType->enGetType() == EN_OBJTYPE_ENUMERATION) )
      {
         out << TAB "inline tBool operator==(const " << oTypeName << "& roRef) const" << endl
             << TAB TAB "{  return (" << oGetDataFieldName( poType ) << " == roRef." << oGetDataFieldName( poType ) << ");  }" << endl;
      }
      else
      {
         out << TAB "tBool operator==(const " << oTypeName << "& roRef) const;" << endl;
      }
   }
   string oPrefix = oToLower( oGetFIPrefix( poServiceTree ) );
   // out << TAB << "virtual const " << oPrefix << "_tclToken::tenType s32GetTypeId() const ";
   out << TAB << "virtual tS32 s32GetTypeId() const;" << endl;
   //   if (oToLower(poServiceTree->oGetShortname()) == (oGetFIGroup() + "types_fi"))
   //   {
   //      out << "{ return " << oPrefix << oGetTokenName(poType->oGetName()) << ";}" << endl;
   //   }
   //   else
   //   {
   //      out << "{ return " << oPrefix << oRemove(oGetTokenName("__MSG_" + poType->oGetName()),'.') << ";}" << endl;
   //   }
   /*
   if (poServiceTree->oGetShortname() == "types")
   {
   out << TAB << "virtual const " << oGetFIGroup() << "typesfi_tclToken::tenType s32GetTypeId() const "
   << "{ return " << oToLower(oGetFIGroup()) << "typesfi" << oGetTokenName(poType->oGetName()) << ";}" << endl;
   }
   else
   {
   }
   */
   if ( poType->enGetType() != EN_OBJTYPE_OPCODE )
   {
      out << TAB << "virtual tVoid vTakeVisitor(" << oGetFIGroup() << "fi_tclVisitorBase& rfoVisitor);" << endl;
   }
   if ( bIsBinContainerType( poType->oGetName() ) )
   {
      out << TAB << "tU16 u16Transform(const " << oGetFIGroup() << "fi_tcl_BinContainer& oContainer);" << endl;
#pragma message("generate useful ServiceId!?")
      // out << TAB << "const tU16 C_U16_SERVICEID = 0;" << endl;
      out << TAB << "static const tU16 C_U16_MAJORVERSION_MAX;" << endl;
      out << TAB << "static const tU16 C_U16_MAJORVERSION_MIN;" << endl;
   }
   out << "};" << endl;

   if ( bIsBinContainerType( poType->oGetName() ) )
   {
      out << "#endif // FI_S_EXTERNAL_FI" << endl;
   }
   out << endl;

   return bSuccess;
}


bool bGenerateTypeConstructor( tclFINode *poType, ostream &out, string oPrefix )
{
   string oTypeName /* = oPrefix*/;

   if ( poType->enGetType() == EN_OBJTYPE_OPCODE )
   {
      if ( bIsStandardErrorMessageExt( poType ) )
         return true;  // destructor taken from default implementation

      oTypeName += oPrefix + oGetClassName( CG_MSGTYPE_PREFIX + oRemove( poType->oGetName(), '.' ) );
   }
   else
      oTypeName += oGetClassName( poType->oGetName() );


   bool bFirst = true;
   tclFITypeDependencyList* poDepList = poType->poGetDependencyList();

   switch ( poType->enGetType() )
   {
   case EN_OBJTYPE_ATOM:
      break;
   case EN_OBJTYPE_ENUMERATION:
   {
      out << oTypeName << "::" << oTypeName << "()";
      out << ":" << CG_DATAFIELD_ENUM << "(";
      if ( poType->oGetDefaultValue() != "" )
      {
         out << oGetEnumName( poType->oGetDefaultValue() );
      }
      else
      {
         out << oGetEnumName( (*poDepList->begin())->oGetName() );
      }
      out << ") {}" << endl << endl;
   } break;
   case EN_OBJTYPE_BITFIELD:
   {
      tclFINode* poBaseType = (*poDepList->begin())->poGetTarget();
      string oDataFieldName = oGetDataFieldName( poBaseType );
      out << oTypeName << "::" << oTypeName << "()";
      out << ":" << oDataFieldName << "(";
      if ( poType->oGetDefaultValue() != "" )
      {
         out << poType->oGetDefaultValue();
      }
      else
      {
         out << "0";
      }
      out << ")" << endl << "{}" << endl << endl;
   } break;
   case EN_OBJTYPE_TYPE:
   {
      tclFINode* poBaseType = (*poDepList->begin())->poGetTarget();
      out << oTypeName << "::" << oTypeName << "(" << poBaseType->oGetName() << " _" << oGetDataFieldName( poBaseType );
      out << ")" << endl << ": " << oGetDataFieldName( poBaseType ) << "(_"
      << oGetDataFieldName( poBaseType ) << endl << ") {} " << endl << endl;
   } break;
   case EN_OBJTYPE_OPCODE:
   case EN_OBJTYPE_STRUCTURE:
   {
      out << oTypeName << "::" << oTypeName;
      out << "()" << endl;
      bool bIsFirst = true;
      tclFITypeDependencyListIterator iter;
      for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
      {
         switch ( (*iter)->enGetClass() )
         {
         case EN_DEPENDENCY_CLASS_UNION:
         {
            tclFINode* poBaseType = (*iter)->poGetTarget();
            if ( bIsFirst )
            {
               bIsFirst = false;
               out << ":";
            }
            else
            {
               out << ",";
            }
            switch ( poBaseType->enGetType() )
            {
            case EN_OBJTYPE_ENUMERATION:
               out << "po" << (*iter)->oGetName() << "Data(0)";
               break;
            case EN_OBJTYPE_BITFIELD:
            {
               tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
               for ( unsigned int iElemIdx = 0; iElemIdx < poDepList2->size(); ++iElemIdx )
               {
                  if ( iElemIdx > 0 )
                     out << ",";
                  out << "poOption" << iElemIdx << "(0)";
               }
            } break;
            }
         } break;
         case EN_DEPENDENCY_CLASS_FIELD:
         case EN_DEPENDENCY_CLASS_PARAM:
            if ( bIsFirst )
            {
               bIsFirst = false;
               out << ":";
            }
            else
            {
               out << ",";
            }
            out << (*iter)->oGetName() << "(";
            if ( (*iter)->oGetDefaultValue() != "" )
            {
               out << (*iter)->oGetDefaultValue();
            }
            else
            {
               string localString( "tString" );
               if ( ((*iter)->poGetTarget()->oGetName() != "tString") &&
                  ((*iter)->poGetTarget()->poFindDependency( localString ) == 0) )
               {
                  out << ((*iter)->poGetTarget()->oGetName()[0] == 't' ? "0" : "");
               }
            }
            out << ")";
            break;
         }
         out << endl;
      }
      out << " {}" << endl << endl;
   } break;

   }
   return true;
}


bool bGenerateDestroyMethod( tclFINode *poType, ostream &out, string oPrefix )
{
   string oTypeName;
   if ( poType->enGetType() == EN_OBJTYPE_OPCODE )
   {
      if ( bIsStandardErrorMessageExt( poType ) )
         return true;  // destructor taken from default implementation

      oTypeName += oPrefix + oGetClassName( CG_MSGTYPE_PREFIX + oRemove( poType->oGetName(), '.' ) );
   }
   else
      oTypeName += oGetClassName( poType->oGetName() );

   if ( !bGenerateExtendedDestroyMethod( poType, out, oPrefix ) )
   {
      tclFITypeDependencyList* poDepList = poType->poGetDependencyList();

      if ( bNeedsDestroyMethod(poType) )
      {
         switch ( poType->enGetType() )
         {
         case EN_OBJTYPE_ATOM:
         case EN_OBJTYPE_BITFIELD:
         case EN_OBJTYPE_ENUMERATION:
            break;
         case EN_OBJTYPE_TYPE:
         {
            tclFINode* poBaseType = (*poDepList->begin())->poGetTarget();
            if ( poBaseType->oGetName() == "tString" )
            {
               out << "tVoid " << oTypeName << "::vDestroy()" << endl;
               out << "{" << endl;
               out << TAB << oGetDataFieldName( poType ) << ".vDestroy();" << endl;
               out << "}" << endl << endl;
            }
         } break;
         case EN_OBJTYPE_OPCODE:
         case EN_OBJTYPE_STRUCTURE:
         {
            out << "tVoid " << oTypeName << "::vDestroy()" << endl;
            out << "{" << endl;
            tclFITypeDependencyListIterator iter;

            for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
            {
               tclFINode* poBaseType = (*iter)->poGetTarget();
               switch ( (*iter)->enGetClass() )
               {
               case EN_DEPENDENCY_CLASS_PARAM:
               case EN_DEPENDENCY_CLASS_FIELD:
               {
                  string oFieldName = (*iter)->oGetName();
                  switch ( poBaseType->enGetType() )
                  {
                  case EN_OBJTYPE_ATOM:
                     if ( (poBaseType->oGetName() == "tString") ||
                        (poBaseType->oGetName() == "tMultiLanguageString") )
                     {
                        out << TAB << oFieldName << ".vDestroy();" << endl;
                     }
                     break;
                  case EN_OBJTYPE_TYPE:
                  {
                     tclFINode* poBaseBaseType = (*(poBaseType->poGetDependencyList()->begin()))->poGetTarget();
                     if ((poBaseBaseType->oGetName() == "tString") ||
                        (poBaseBaseType->oGetName() == "tMultiLanguageString"))
                     {
                        out << TAB << oFieldName << ".vDestroy();" << endl;
                     }
                  }  break;
                  case EN_OBJTYPE_STRUCTURE:
                     out << TAB << oFieldName << ".vDestroy(); " << endl;
                     break;
                  }
               } break;
               case EN_DEPENDENCY_CLASS_LIST:
               {
                  // Add wrapping braces since old compilers (e.g. VC 2005) don't handle the scope of counter
                  // variables in for-loops correctly. The braces will limit the scope.
                  out << TAB << "{" << endl;
                  string oListName = (*iter)->oGetName();
                  switch ( poBaseType->enGetType() )
                  {
                  case EN_OBJTYPE_ATOM:
                     if ( (poBaseType->oGetName() == "tString") ||
                        (poBaseType->oGetName() == "tMultiLanguageString") )
                     {
                        out << TAB << TAB << "for (tU32 u32Idx = 0; u32Idx < " << oListName << ".size(); ++u32Idx)" << endl;
                        out << TAB << TAB << TAB << oListName << "[u32Idx].vDestroy();" << endl;
                     }
                     break;
                  case EN_OBJTYPE_TYPE:
                  {
                     string localString( "tString" );
                     if ( poBaseType->poFindDependency( localString ) != 0 )
                     {
                        out << TAB << TAB << "for (tU32 u32Idx = 0; u32Idx < " << oListName << ".size(); ++u32Idx)" << endl;
                        out << TAB << TAB << TAB << oListName << "[u32Idx].vDestroy();" << endl;
                     }
                  }
                  break;
                  case EN_OBJTYPE_STRUCTURE:
                     out << TAB << TAB << "for (tU32 u32Idx = 0; u32Idx < " << oListName << ".size(); ++u32Idx)" << endl;
                     out << TAB << TAB << TAB << oListName << "[u32Idx].vDestroy();" << endl;
                     break;
                  }
                  out << TAB << oListName << ".clear();" << endl;
                  out << TAB << "}" << endl;
               } break;
               case EN_DEPENDENCY_CLASS_UNION:
               {
                  string oUnionName = (*iter)->oGetName();
                  switch ( poBaseType->enGetType() )
                  {
                  case EN_OBJTYPE_ENUMERATION:
                  {
                     out << TAB << "if (po" << oUnionName << "Data)" << endl;
                     out << TAB << "{" << endl;
                     out << TAB << TAB << "po" << oUnionName << "Data->vDestroy();" << endl;
                     out << TAB << TAB << "OSAL_DELETE po" << oUnionName << "Data;" << endl;
                     out << TAB << TAB << "po" << oUnionName << "Data = OSAL_NULL;" << endl;
                     out << TAB << "}" << endl;
                  } break;
                  case EN_OBJTYPE_BITFIELD:
                  {
                     tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
                     tclFITypeDependencyListIterator iter2;
                     int iOptionIdx = 0;
                     for ( int iBitIdx = 32; iBitIdx > 0; )
                     {
                        --iBitIdx;
                        for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
                        {
                           if ( iBitIdx == strtoul( (*iter2)->oGetDefaultValue().c_str(), 0, 0 ) )
                           {
                              out << TAB << "if (" << oUnionName << "Mask."
                                 << oGetDataFieldName( poBaseType )
                                 << " & " << oGetClassName( poBaseType->oGetName() )
                                 << "::FI_C_" << oToUpper( (*iter2)->poGetTarget()->oGetName().substr( 1 ) )
                                 << "_BIT_" << oToUpper( (*iter2)->oGetName() )
                                 << " && poOption" << iOptionIdx << ")" << endl;
                              out << TAB << "{" << endl;
                              out << TAB << TAB << "poOption" << iOptionIdx << "->vDestroy();" << endl;
                              out << TAB << TAB << "OSAL_DELETE poOption" << iOptionIdx << ";" << endl;
                              out << TAB << TAB << "poOption" << iOptionIdx << " = OSAL_NULL;" << endl;
                              out << TAB << "}" << endl;
                              iOptionIdx++;
                           }
                        }
                     }
                  } break;
                  }
               } break;
               }
            }
            out << "}" << endl << endl;
         } break;

         }
      }
      else
      {
         // No dependent types to clean up, so it is not necessary to override vDestroy explicitly.
         // It's fine to fall back to the implementation of the base class.
      }
   }

   return true;
}

bool bGenerateTypeDestructor( tclFINode *poType, ostream &out, string oPrefix )
{
   string oTypeName;
   if ( poType->enGetType() == EN_OBJTYPE_OPCODE )
   {
      oTypeName += oPrefix + oGetClassName( CG_MSGTYPE_PREFIX + oRemove( poType->oGetName(), '.' ) );
   }
   else
      oTypeName += oGetClassName( poType->oGetName() );

   if ( !bGenerateExtendedTypeDestructor( poType, out, oPrefix ) )
   {
      out << endl << oTypeName << "::~" << oTypeName << "()" << endl
         << "{";

      if ( bEnableDeepCopy && bNeedsDestroyMethod( poType ) )
      {
         out << endl;
         out << "#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;
         out << TAB << "vDestroy();" << endl;
         out << "#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;
      }

      out << "}" << endl << endl;
   }

   return true;
}

bool bGenerateTypeCopyConstructor( tclFINode *poType, ostream &out, string oPrefix )
{
   string oTypeName /* = oPrefix*/;
   if ( poType->enGetType() == EN_OBJTYPE_OPCODE )
   {
      oTypeName += oPrefix + oGetClassName( CG_MSGTYPE_PREFIX + oRemove( poType->oGetName(), '.' ) );
   }
   else
      oTypeName += oGetClassName( poType->oGetName() );

   if ( bGenerateExtendedTypeCopyConstructor( poType, out, oPrefix ) )
      return true;

   switch ( poType->enGetType() )
   {
   case EN_OBJTYPE_ATOM:
   case EN_OBJTYPE_TYPE:
   case EN_OBJTYPE_ENUMERATION:
   case EN_OBJTYPE_BITFIELD:
      break;
   case EN_OBJTYPE_OPCODE:
   case EN_OBJTYPE_STRUCTURE:
   {
      out << oTypeName << "::" << oTypeName << "(const " << oTypeName << "& coRef)" << endl;
      if ( poType->enGetType() == EN_OBJTYPE_STRUCTURE )
         out << TAB << ": " << oGetFIGroup() << "fi_tclVisitableTypeBase(coRef)" << endl;
      else if ( bIsStandardErrorMessageExt( poType ) )
         out << TAB << ": " << oPrefix << "fi_tclMsgDefaultError(coRef)" << endl;
      else
         out << TAB << ": " << oBaseMessageClassName << "(coRef)" << endl;
      out << "{" << endl;

      tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
      tclFITypeDependencyListIterator iter;
      for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
      {
         tclFINode* poBaseType = (*iter)->poGetTarget();
         switch ( (*iter)->enGetClass() )
         {
         case EN_DEPENDENCY_CLASS_PARAM:
         case EN_DEPENDENCY_CLASS_FIELD:
         case EN_DEPENDENCY_CLASS_LIST:
         {
            string oFieldName = (*iter)->oGetName();
            out << TAB << oFieldName << " = coRef." << oFieldName << ";" << endl;
         } break;
         case EN_DEPENDENCY_CLASS_UNION:
         {
            string oUnionName = (*iter)->oGetName();
            switch ( poBaseType->enGetType() )
            {
            case EN_OBJTYPE_ENUMERATION:
            {
               out << TAB << oUnionName << "Type = coRef." << oUnionName << "Type;" << endl;
               out << TAB << "if  (coRef.po" << oUnionName << "Data)" << endl;
               out << TAB << "{" << endl;
               out << TAB << TAB << "switch(" << oUnionName << "Type."
                  << oGetDataFieldName( poBaseType ) << ")" << endl;
               out << TAB << TAB << "{" << endl;
               tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
               tclFITypeDependencyListIterator iter2;
               for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
               {
                  out << TAB << TAB << "case "
                     << oGetClassName( poBaseType->oGetName() ) << "::"
                     << oGetEnumName( (*iter2)->oGetName() ) << ":" << endl;
                  if ( (*iter2)->oGetName()[0] == 't' )
                  {
                     out << TAB << TAB << TAB << "po" << oUnionName << "Data = OSAL_NEW "
                        << oGetClassName( "T" + (*iter2)->oGetName().substr( 1 ) )
                        << "((const " << oGetClassName( "T" + (*iter2)->oGetName().substr( 1 ) )
                        << "&)*coRef.po" << oUnionName << "Data);" << endl;
                  }
                  else
                  {
                     if ( bIsBinContainerType( (*iter2)->oGetName() ) )
                     {
                        out << TAB << TAB << TAB "if (coRef.po" << oUnionName << "Data->s32GetTypeId() == (tS32)"
                           << oPrefix << "fi" << oGetTokenName( (*iter2)->oGetName() ) << ")" << endl;
                        out << TAB << TAB << TAB "{" << endl;
                        out << TAB << TAB << TAB << TAB << "po" << oUnionName << "Data = OSAL_NEW "
                           << oGetClassName( (*iter2)->oGetName() ) << "((const "
                           << oGetClassName( (*iter2)->oGetName() ) << "&)*coRef.po"
                           << (*iter)->oGetName() << "Data);" << endl;
                        out << TAB << TAB << TAB "}" << endl;
                        out << TAB << TAB << TAB "else" << endl;
                        out << TAB << TAB << TAB "{" << endl;
                        out << TAB << TAB << TAB << TAB << "po" << oUnionName << "Data = OSAL_NEW "
                           << oGetClassName( "T_BinContainer" ) << "((const "
                           << oGetClassName( "T_BinContainer" ) << "&)*coRef.po"
                           << (*iter)->oGetName() << "Data);" << endl;
                        out << TAB << TAB << TAB "}" << endl;
                     }
                     else
                     {
                        out << TAB << TAB << TAB << "po" << oUnionName << "Data = OSAL_NEW "
                           << oGetClassName( (*iter2)->oGetName() ) << "((const "
                           << oGetClassName( (*iter2)->oGetName() ) << "&)*coRef.po"
                           << (*iter)->oGetName() << "Data);" << endl;
                     }
                  }
                  out << TAB << TAB << TAB << "break;" << endl;
               }
               out << TAB << TAB << "default:" << endl;
               out << TAB << TAB << TAB << "po" << oUnionName << "Data = OSAL_NULL;" << endl;
               out << TAB << TAB << "}" << endl;
               out << TAB << "}" << endl;
               out << TAB << "else" << endl;
               out << TAB << "{" << endl;
               out << TAB << TAB << "po" << oUnionName << "Data = OSAL_NULL;" << endl;
               out << TAB << "}" << endl;

            } break;

            case EN_OBJTYPE_BITFIELD:
            {
               out << TAB << oUnionName << "Mask = coRef." << oUnionName << "Mask;" << endl;
               tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
               tclFITypeDependencyListIterator iter2;
               int iOptionIdx = 0;
               for ( int iBitIdx = 32; iBitIdx > 0; )
               {
                  --iBitIdx;
                  for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
                  {
                     if ( iBitIdx == strtoul( (*iter2)->oGetDefaultValue().c_str(), 0, 0 ) )
                     {
                        out << TAB << "if (" << oUnionName << "Mask."
                           << oGetDataFieldName( poBaseType )
                           << " & " << oGetClassName( poBaseType->oGetName() )
                           << "::FI_C_" << oToUpper( (*iter2)->poGetTarget()->oGetName().substr( 1 ) )
                           << "_BIT_" << oToUpper( (*iter2)->oGetName() )
                           << " && coRef.poOption" << iOptionIdx << ")" << endl;
                        out << TAB << "{" << endl;
#if 0 
                        // this special handling for binary container only works with a base type at poOption<x>
                        // as long as a specialized type is used, always use this type!
                        if ( bIsBinContainerType( (*iter2)->oGetName() ) )
                        {
                           out << TAB << TAB << "if (coRef.poOption" << iOptionIdx << "->s32GetTypeId() == "
                              << oPrefix << "fi" << oGetTokenName( (*iter2)->oGetName() ) << ")" << endl;
                           out << TAB << TAB << "{" << endl;
                           out << TAB << TAB << TAB << "poOption" << iOptionIdx << " = OSAL_NEW ";
                           out << oGetClassName( (*iter2)->oGetName() ) << "((const "
                              << oGetClassName( (*iter2)->oGetName() ) << "&)*coRef.poOption"
                              << iOptionIdx << "); //lint !e1774" << endl;
                           out << TAB << TAB << "}" << endl;
                           out << TAB << TAB << "else" << endl;
                           out << TAB << TAB << "{" << endl;
                           out << TAB << TAB << TAB << "poOption" << iOptionIdx << " = OSAL_NEW ";
                           out << oGetClassName( "T_BinContainer" ) << "((const "
                              << oGetClassName( "T_BinContainer" ) << "&)*coRef.poOption"
                              << iOptionIdx << "); //lint !e1774" << endl;
                           out << TAB << TAB << "}" << endl;
                        }
                        else
#endif
                        {
                           out << TAB << TAB << "poOption" << iOptionIdx << " = OSAL_NEW ";
                           out << oGetClassName( (*iter2)->oGetName() ) << "((const "
                              << oGetClassName( (*iter2)->oGetName() ) << "&)*coRef.poOption"
                              << iOptionIdx << ");" << endl;
                        }
                        out << TAB << TAB << "if (!poOption" << iOptionIdx << ")" << endl;
                        out << TAB << TAB << "{" << endl;
                        out << TAB << TAB << TAB << oUnionName << "Mask.vSet"
                           << (*iter2)->oGetName() << "(FALSE);" << endl;
                        out << TAB << TAB << "}" << endl;
                        out << TAB << "}" << endl;
                        out << TAB << "else" << endl;
                        out << TAB << "{" << endl;
                        out << TAB << TAB << "poOption" << iOptionIdx << " = OSAL_NULL;" << endl;
                        out << TAB << TAB << oUnionName << "Mask.vSet" << (*iter2)->oGetName() << "(FALSE);" << endl;
                        out << TAB << "}" << endl;

                        iOptionIdx++;
                     }
                  }
               }
            } break;
            }
         } break;
         }
      }
      out << "}" << endl;
   } break;
   }
   return true;
}

bool bGenerateTypeAssignmentOp( tclFINode *poType, ostream &out, string oPrefix )
{
   string oTypeName /* = oPrefix*/;
   if ( poType->enGetType() == EN_OBJTYPE_OPCODE )
   {
      oTypeName += oPrefix + oGetClassName( CG_MSGTYPE_PREFIX + oRemove( poType->oGetName(), '.' ) );
   }
   else
      oTypeName += oGetClassName( poType->oGetName() );

   if ( bGenerateExtendedTypeAssignmentOp( poType, out, oPrefix ) )
      return true;

   switch ( poType->enGetType() )
   {
   case EN_OBJTYPE_ATOM:
   case EN_OBJTYPE_TYPE:
   case EN_OBJTYPE_ENUMERATION:
   case EN_OBJTYPE_BITFIELD:
      break;
   case EN_OBJTYPE_OPCODE:
   case EN_OBJTYPE_STRUCTURE:
   {
      out << oTypeName << "& " << oTypeName << "::operator=" << "(const " << oTypeName << "& coRef)" << endl;
      out << "{" << endl;
      out << TAB << "if (this == &coRef) return *this;" << endl;

      tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
      tclFITypeDependencyListIterator iter;
      for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
      {
         tclFINode* poBaseType = (*iter)->poGetTarget();
         switch ( (*iter)->enGetClass() )
         {
         case EN_DEPENDENCY_CLASS_PARAM:
         case EN_DEPENDENCY_CLASS_FIELD:
         case EN_DEPENDENCY_CLASS_LIST:
         {
            string oFieldName = (*iter)->oGetName();
            out << TAB << oFieldName << " = coRef." << oFieldName << ";" << endl;
         } break;
         case EN_DEPENDENCY_CLASS_UNION:
         {
            string oUnionName = (*iter)->oGetName();
            switch ( poBaseType->enGetType() )
            {
            case EN_OBJTYPE_ENUMERATION:
            {
               out << TAB << "OSAL_DELETE po" << oUnionName << "Data;" << endl;
               out << TAB << oUnionName << "Type = coRef." << oUnionName << "Type;" << endl;
               out << TAB << "if  (coRef.po" << oUnionName << "Data)" << endl;
               out << TAB << "{" << endl;
               out << TAB << TAB << "switch(" << oUnionName << "Type."
                  << oGetDataFieldName( poBaseType ) << ")" << endl;
               out << TAB << TAB << "{" << endl;
               tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
               tclFITypeDependencyListIterator iter2;
               for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
               {
                  out << TAB << TAB << "case "
                     << oGetClassName( poBaseType->oGetName() ) << "::"
                     << oGetEnumName( (*iter2)->oGetName() ) << ":" << endl;
                  if ( (*iter2)->oGetName()[0] == 't' )
                  {
                     out << TAB << TAB << TAB << "po" << oUnionName << "Data = OSAL_NEW "
                        << oGetClassName( "T" + (*iter2)->oGetName().substr( 1 ) )
                        << "((const " << oGetClassName( "T" + (*iter2)->oGetName().substr( 1 ) )
                        << "&)*coRef.po" << oUnionName << "Data);" << endl;
                  }
                  else
                  {
                     if ( bIsBinContainerType( (*iter2)->oGetName() ) )
                     {
                        out << TAB << TAB << TAB "if (coRef.po" << oUnionName << "Data->s32GetTypeId() == (tS32)"
                           << oPrefix << "fi" << oGetTokenName( (*iter2)->oGetName() ) << ")" << endl;
                        out << TAB << TAB << TAB "{" << endl;
                        out << TAB << TAB << TAB << TAB << "po" << oUnionName << "Data = OSAL_NEW "
                           << oGetClassName( (*iter2)->oGetName() ) << "((const "
                           << oGetClassName( (*iter2)->oGetName() ) << "&)*coRef.po"
                           << (*iter)->oGetName() << "Data);" << endl;
                        out << TAB << TAB << TAB "}" << endl;
                        out << TAB << TAB << TAB "else" << endl;
                        out << TAB << TAB << TAB "{" << endl;
                        out << TAB << TAB << TAB << TAB << "po" << oUnionName << "Data = OSAL_NEW "
                           << oGetClassName( "T_BinContainer" ) << "((const "
                           << oGetClassName( "T_BinContainer" ) << "&)*coRef.po"
                           << (*iter)->oGetName() << "Data);" << endl;
                        out << TAB << TAB << TAB "}" << endl;
                     }
                     else
                     {
                        out << TAB << TAB << TAB << "po" << oUnionName << "Data = OSAL_NEW "
                           << oGetClassName( (*iter2)->oGetName() ) << "((const "
                           << oGetClassName( (*iter2)->oGetName() ) << "&)*coRef.po"
                           << (*iter)->oGetName() << "Data);" << endl;
                     }
                  }
                  out << TAB << TAB << TAB << "break;" << endl;
               }
               out << TAB << TAB << "default:" << endl;
               out << TAB << TAB << TAB << "po" << oUnionName << "Data = OSAL_NULL;" << endl;
               out << TAB << TAB << "}" << endl;
               out << TAB << "}" << endl;
               out << TAB << "else" << endl;
               out << TAB << "{" << endl;
               out << TAB << TAB << "po" << oUnionName << "Data = OSAL_NULL;" << endl;
               out << TAB << "}" << endl;
            } break;

            case EN_OBJTYPE_BITFIELD:
            {
               tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
               tclFITypeDependencyListIterator iter2;
               int iOptionIdx = 0;
               int iBitIdx = 0;

               for ( iBitIdx = 32; iBitIdx > 0; )
               {
                  --iBitIdx;
                  for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
                  {
                     if ( iBitIdx == strtoul( (*iter2)->oGetDefaultValue().c_str(), 0, 0 ) )
                     {
                        out << TAB << "OSAL_DELETE poOption" << iOptionIdx << ";" << endl;
                        iOptionIdx++;
                     }
                  }
               }
               iOptionIdx = 0;
               out << TAB << oUnionName << "Mask = coRef." << oUnionName << "Mask;" << endl;
               for ( iBitIdx = 32; iBitIdx > 0; )
               {
                  --iBitIdx;
                  for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
                  {
                     if ( iBitIdx == strtoul( (*iter2)->oGetDefaultValue().c_str(), 0, 0 ) )
                     {
                        out << TAB << "if (" << oUnionName << "Mask."
                           << oGetDataFieldName( poBaseType )
                           << " & " << oGetClassName( poBaseType->oGetName() )
                           << "::FI_C_" << oToUpper( (*iter2)->poGetTarget()->oGetName().substr( 1 ) )
                           << "_BIT_" << oToUpper( (*iter2)->oGetName() )
                           << " && coRef.poOption" << iOptionIdx << ")" << endl;
                        out << TAB << "{" << endl;

#if 0 
                        // this special handling for binary container only works with a base type at poOption<x>
                        // as long as a specialized type is used, always use this type!
                        if ( bIsBinContainerType( (*iter2)->oGetName() ) )
                        {
                           out << TAB << TAB << "if (coRef.poOption" << iOptionIdx << "->s32GetTypeId() == "
                              << oPrefix << "fi" << oGetTokenName( (*iter2)->oGetName() ) << ")" << endl;
                           out << TAB << TAB << "{" << endl;
                           out << TAB << TAB << TAB << "poOption" << iOptionIdx << " = OSAL_NEW ";
                           out << oGetClassName( (*iter2)->oGetName() ) << "((const "
                              << oGetClassName( (*iter2)->oGetName() ) << "&)*coRef.poOption"
                              << iOptionIdx << "); //lint !e1774" << endl;
                           out << TAB << TAB << "}" << endl;
                           out << TAB << TAB << "else" << endl;
                           out << TAB << TAB << "{" << endl;
                           out << TAB << TAB << TAB << "poOption" << iOptionIdx << " = OSAL_NEW ";
                           out << oGetClassName( "T_BinContainer" ) << "((const "
                              << oGetClassName( "T_BinContainer" ) << "&)*coRef.poOption"
                              << iOptionIdx << "); //lint !e1774" << endl;
                           out << TAB << TAB << "}" << endl;
                        }
                        else
#endif
                        {
                           out << TAB << TAB << "poOption" << iOptionIdx << " = OSAL_NEW ";
                           out << oGetClassName( (*iter2)->oGetName() ) << "((const "
                              << oGetClassName( (*iter2)->oGetName() ) << "&)*coRef.poOption"
                              << iOptionIdx << ");" << endl;
                        }
                        out << TAB << TAB << "if (!poOption" << iOptionIdx << ")" << endl;
                        out << TAB << TAB << "{" << endl;
                        out << TAB << TAB << TAB << oUnionName << "Mask.vSet"
                           << (*iter2)->oGetName() << "(FALSE);" << endl;
                        out << TAB << TAB << "}" << endl;
                        out << TAB << "}" << endl;
                        out << TAB << "else" << endl;
                        out << TAB << "{" << endl;
                        out << TAB << TAB << "poOption" << iOptionIdx << " = OSAL_NULL;" << endl;
                        out << TAB << TAB << oUnionName << "Mask.vSet" << (*iter2)->oGetName() << "(FALSE);" << endl;
                        out << TAB << "}" << endl;

                        iOptionIdx++;
                     }
                  }
               }
            } break;
            }
         } break;
         }
      }
      out << TAB << "return *this;" << endl;
      out << "}" << endl;
   } break;

   }
   return true;
}


bool bCheckForVersionDependency( tclFINode* poType, tclFIServiceTree* poServiceTree )
{
   bool bIsVersionDependent = false;

   //   cerr << "Type " << poType->oGetName() << ": ";
   switch ( poType->enGetType() )
   {
   case EN_OBJTYPE_ATOM:
      break;
   case EN_OBJTYPE_TYPE:
      break;
   case EN_OBJTYPE_ENUMERATION:
   case EN_OBJTYPE_BITFIELD:
      break;
   case EN_OBJTYPE_OPCODE:
   case EN_OBJTYPE_STRUCTURE:
   {
      tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
      tclFITypeDependencyListIterator iter;
      for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
      {
         tclFINode* poBaseType;
         if ( bIsBinContainerType( (*iter)->poGetTarget()->oGetName() ) )
         {
            poBaseType = poServiceTree->poGetUsedGlobalTypePool()->poGetNode( "T_BinContainer" );
         }
         else
         {
            poBaseType = (*iter)->poGetTarget();
         }
         int s32TypeSize = poBaseType->iGetDataSize();
         switch ( (*iter)->enGetClass() )
         {
         case EN_DEPENDENCY_CLASS_PARAM:
         case EN_DEPENDENCY_CLASS_FIELD:
         case EN_DEPENDENCY_CLASS_LIST:
         {
            if ( (*iter)->bHasVersionInformation() )
            {
               //                     cerr << "member " << (*iter)->oGetName() << " has version info.";
               bIsVersionDependent = true;
            }
            else
            {
               if ( s32TypeSize < 0 )
               {
                  bIsVersionDependent = true;
                  //                        cerr << "member " << (*iter)->oGetName() << " has size < 0.";
               }
            }
         } break;
         case EN_DEPENDENCY_CLASS_UNION:
         {
            bIsVersionDependent = true;
            //                  cerr << "member " << (*iter)->oGetName() << " is union.";
         } break;
         }
      }
   } break;
   }
   //   cerr << endl;
   return bIsVersionDependent;
}

bool bGenerateMsgBaseDecl( ostream &out, tclFIServiceTree* poServiceTree )
{
   // declaration for distributed type factory

   string oSrvPrefix = oToLower( oGetFIPrefix( poServiceTree ) );
   out << "// factory for " << poServiceTree->oGetShortname() << " message types" << endl
       << endl
       << "fi_tclMessageBase* " << oSrvPrefix << "_poGetMessageBaseObject(tU16 u16FunctionId, tU8 u8Opcode);" << endl
       << endl;

   out << "// base class for all ampfi messages" << endl << endl
       << "class " << oBaseMessageClassName
       << " : public " << oGetFIGroup() << "fi_tclVisitableTypeBase, public fi_tclMessageBase" << endl
       << "{" << endl
       << "public:" << endl;

   out << TAB << "// common functions" << endl;

   out << TAB << "virtual tU16 u16GetServiceID() const;" << endl
       << TAB << "virtual tU16 u16GetFunctionID() const;" << endl
       << TAB << "virtual tU8 u8GetOpCode() const;" << endl
       << TAB << "virtual const fi_tclTypeBase& corfoGetTypeBase() const;" << endl
       << TAB << "virtual fi_tclTypeBase& rfoGetTypeBase();" << endl << endl;

   out << TAB << "// default implementation directly applicable to all messages without paramenters" << endl;

   out << TAB << oBaseMessageClassName << "() { };" << endl
       << TAB << "virtual tU32 u32GetSize(tU16 u16MajorVersion) const;" << endl
       << TAB << "virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;" << endl
       << TAB << "virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);" << endl
       << "};" << endl << endl;
   return true;
}

bool bGenerateMsgInfo( tclFINode* poType, ostream &out, string oPrefix, tclFIServiceTree* poServiceTree )
{

   // declaration for distributed type factory
   bGenerateComponentTypeFactory( poServiceTree, out );


   out << "//=============================================================================" << endl
       << endl
       << "// base class for all " << oPrefix << "fi messages" << endl
       << endl
       << "// common functions" << endl << endl;

   out << "tU16 " << oBaseMessageClassName << "::u16GetServiceID() const" << endl
       << "{" << endl
       << TAB << "return " << oToUpper( oGetFIPrefix( poServiceTree ) ) << "_C_U16_SERVICE_ID;" << endl
       << "}" << endl << endl;

   out << "tU16 " << oBaseMessageClassName << "::u16GetFunctionID() const" << endl
       << "{" << endl
       << TAB << "return (tU16) ((((tU32) s32GetTypeId()) >> 16) & 0xFFFF);" << endl
       << "}" << endl << endl;
   out << "tU8 " << oBaseMessageClassName << "::u8GetOpCode() const " << endl
       << "{" << endl
       << TAB << "return (tU8) ((((tU32) s32GetTypeId()) >> 8) & 0xFF);" << endl
       << "}" << endl << endl;

   out << "const fi_tclTypeBase& " << oBaseMessageClassName << "::corfoGetTypeBase() const" << endl
       << "{" << endl
       << TAB << "return *this;" << endl
       << "}" << endl << endl;

   out << "fi_tclTypeBase& " << oBaseMessageClassName << "::rfoGetTypeBase()" << endl
       << "{" << endl
       << TAB << "return *this;" << endl
       << "}" << endl << endl;

   out << "// default implementation directly applicable to all messages without paramenters" << endl << endl;

   //out << oBaseMessageClassName << "::" << oBaseMessageClassName << "()" << endl
   //    << "{" << endl
   //    << TAB << "// nothing to initialize" << endl
   //    << "}" << endl << endl;

   out << "tU32 " << oBaseMessageClassName << "::u32GetSize(tU16 /* u16MajorVersion */ ) const" << endl
       << "{" << endl
       << TAB << "return 0;" << endl
       << "}" << endl << endl;

   out << "fi_tclOutContext& " << oBaseMessageClassName << "::oWrite(fi_tclOutContext& rfoOut) const" << endl
       << "{" << endl
       << TAB << "return rfoOut;" << endl
       << "}" << endl << endl;

   out << "fi_tclInContext& " << oBaseMessageClassName << "::oRead(fi_tclInContext& rfoIn)" << endl
       << "{" << endl
       << TAB << "return rfoIn;" << endl
       << "}" << endl << endl;

   return true;
}

bool bGenerateMsgDefaultErr( ofstream &out, tclFIServiceTree* poServiceTree )
{
   if ( bGenerateExtendedDefaultErrorMessageImp( out, poServiceTree ) )
      return true;

   string oPrefix = oToLower( poServiceTree->oGetShortname().substr( 0, poServiceTree->oGetShortname().find_last_of( '_' ) ) );
   string oClassName = oPrefix + "fi_tclMsgDefaultError";

   out << "//=============================================================================" << endl
       << endl
       << "// base class for all " << oPrefix << "fi standard error messages" << endl << endl;

   out << "tU32 " << oClassName << "::u32GetSize(tU16 u16MajorVersion) const" << endl
       << "{" << endl
       << TAB << "return 2;" << endl
       << "}" << endl << endl;

   out << "fi_tclInContext& " << oClassName << "::oRead(fi_tclInContext& oIn)" << endl
       << "{" << endl
       << "   (tVoid) (oIn >> u16ErrorCode);" << endl
       << "   return oIn;" << endl
       << "}" << endl << endl;

   out << "fi_tclOutContext& " << oClassName << "::oWrite(fi_tclOutContext& oOut) const" << endl
       << "{" << endl
       << "   (tVoid) (oOut << u16ErrorCode);" << endl
       << "   return oOut;" << endl
       << "}" << endl << endl;

   return true;
}

bool bGenerateTypeGetSizeMethod( tclFINode* poType, ostream &out, tclFIServiceTree* poServiceTree )
{
   if ( bGenerateExtendedTypeGetSizeMethod( poType, out, poServiceTree ) )
      return true;

   string oPrefix = oToLower( poServiceTree->oGetShortname().substr( 0, poServiceTree->oGetShortname().find_last_of( '_' ) ) );

   string oTypeName /*= oPrefix*/;
   if ( poType->enGetType() == EN_OBJTYPE_OPCODE )
   {
      oTypeName += oPrefix + oGetClassName( CG_MSGTYPE_PREFIX + oRemove( poType->oGetName(), '.' ) );
   }
   else
      oTypeName += oGetClassName( poType->oGetName() );

   out << "tU32 " << oTypeName << "::u32GetSize(tU16 ";
   if ( bCheckForVersionDependency( poType, poServiceTree ) )
   {
      out << "u16MajorVersion) const" << endl;
   }
   else
   {
      out << "/*u16MajorVersion*/) const" << endl;
   }
   out << "{" << endl;
   if ( bIsBinContainerType( poType->oGetName() ) && bCheckForVersionDependency( poType, poServiceTree ) )
   {
      out << TAB << "u16MajorVersion = ContainerFIVersion.MajorVersion;" << endl;
   }
   unsigned int u32StaticSize = 0;
   string oDynamicSize = "";
   bool bHasDynamicSize = false;
   bool bHasLists = false;

   switch ( poType->enGetType() )
   {
   case EN_OBJTYPE_ATOM:
      break;
   case EN_OBJTYPE_TYPE:
   {
      tclFINode* poBaseType = (*poType->poGetDependencyList()->begin())->poGetTarget();
      int s32TypeSize = poType->iGetDataSize();
      if ( s32TypeSize < 0 )
      {
         // typedef tString
         oDynamicSize += "+ " + oGetDataFieldName( poBaseType ) + ".u32GetSize(0)";
         // oDynamicSize += "+" + oGetDataFieldName(poType->oGetName()) + ".u32GetSize(u16MajorVersion)";
      }
      else
      {
         u32StaticSize += s32TypeSize;
      }
   } break;
   case EN_OBJTYPE_ENUMERATION:
   case EN_OBJTYPE_BITFIELD:
   {
      u32StaticSize += poType->iGetDataSize();
   } break;
   case EN_OBJTYPE_OPCODE:
   case EN_OBJTYPE_STRUCTURE:
   {
      tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
      tclFITypeDependencyListIterator iter;
      for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
      {
         tclFINode* poBaseType;
         if ( bIsBinContainerType( (*iter)->poGetTarget()->oGetName() ) )
         {
            poBaseType = poServiceTree->poGetUsedGlobalTypePool()->poGetNode( "T_BinContainer" );
            // poBaseType = oTypes["T_BinContainer"];
         }
         else
         {
            poBaseType = (*iter)->poGetTarget();
         }
         int s32TypeSize = poBaseType->iGetDataSize();
         switch ( (*iter)->enGetClass() )
         {
         case EN_DEPENDENCY_CLASS_PARAM:
         case EN_DEPENDENCY_CLASS_FIELD:
         {
            string oFieldName = (*iter)->oGetName();
            if ( (*iter)->bHasVersionInformation() )
            {
               if ( !bHasDynamicSize )
               {
                  out << TAB << "tU32 u32DynamicSize = 0;" << endl;
                  bHasDynamicSize = true;
               }
               if ( (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
               {
                  if ( (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
                  {
                     out << TAB << "if ((" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                        << " <= u16MajorVersion) && (u16MajorVersion <= "
                        << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << "))" << endl;
                  }
                  else
                  {
                     out << TAB << "if (" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                        << " <= u16MajorVersion)" << endl;
                  }
               }
               else
               {
                  out << TAB << "if (u16MajorVersion <= "
                     << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << ")" << endl;
               }
               out << TAB << "{" << endl;
               if ( s32TypeSize < 0 )
               {
                  out << TAB << TAB << "u32DynamicSize += " << oFieldName
                     << ".u32GetSize(u16MajorVersion);" << endl;
               }
               else
               {
                  out << TAB << TAB << "u32DynamicSize += " << s32TypeSize << ";" << endl;
               }
               out << TAB << "}" << endl;
            }
            else
            {
               if ( s32TypeSize < 0 )
               {
                  oDynamicSize += "+" + oFieldName + ".u32GetSize(u16MajorVersion)";
               }
               else
               {
                  u32StaticSize += s32TypeSize;
               }
            }
         } break;
         case EN_DEPENDENCY_CLASS_LIST:
         {
            string oListName = (*iter)->oGetName();
            if ( (*iter)->bHasVersionInformation() )
            {
               if ( !bHasDynamicSize )
               {
                  out << TAB << "tU32 u32DynamicSize = 0;" << endl;
                  bHasDynamicSize = true;
               }
               if ( (!bHasLists) && (s32TypeSize < 0) )
               {
                  out << TAB << "tU32 u32ListIdx;" << endl;
                  bHasLists = true;
               }
               if ( (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
               {
                  if ( (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
                  {
                     out << TAB << "if ((" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                        << " <= u16MajorVersion) && (u16MajorVersion <= "
                        << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << "))" << endl;
                  }
                  else
                  {
                     out << TAB << "if (" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                        << " <= u16MajorVersion)" << endl;
                  }
               }
               else
               {
                  out << TAB << "if (u16MajorVersion <= "
                     << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << ")" << endl;
               }
               out << TAB << "{" << endl;
               out << TAB << TAB << "u32DynamicSize += 4;" << endl;
               if ( s32TypeSize < 0 )
               {
                  out << TAB << TAB << "for (u32ListIdx = 0; u32ListIdx < "
                     << oListName << ".size(); ++u32ListIdx)" << endl;
                  out << TAB << TAB << TAB << "u32DynamicSize += " << oListName
                     << "[u32ListIdx].u32GetSize(u16MajorVersion);" << endl;
               }
               else
               {
                  out << TAB << TAB << "u32DynamicSize += static_cast<tU32>(" << oListName << ".size()*"
                     << s32TypeSize << ");" << endl;
               }

               out << TAB << "}" << endl;
            }
            else
            {
               u32StaticSize += 4;
               if ( s32TypeSize < 0 )
               {
                  if ( !bHasDynamicSize )
                  {
                     out << TAB << "tU32 u32DynamicSize = 0;" << endl;
                     bHasDynamicSize = true;
                  }
                  if ( !bHasLists )
                  {
                     out << TAB << "tU32 u32ListIdx;" << endl;
                     bHasLists = true;
                  }
                  out << TAB << "for (u32ListIdx = 0; u32ListIdx < " << oListName << ".size(); ++u32ListIdx)" << endl;
                  out << TAB << TAB << "u32DynamicSize += " << oListName << "[u32ListIdx].u32GetSize(u16MajorVersion);" << endl;
               }
               else
               {
                  char szTmpStr[16];
                  sprintf( szTmpStr, "%d", s32TypeSize );
                  oDynamicSize += string( "+ static_cast<tU32>(" ) + oListName + ".size()*" + szTmpStr +")";
               }
            }
         } break;
         case EN_DEPENDENCY_CLASS_UNION:
         {
            string oUnionName = (*iter)->oGetName();
            if ( (*iter)->bHasVersionInformation() )
            {
               if ( !bHasDynamicSize )
               {
                  out << TAB << "tU32 u32DynamicSize = 0;" << endl;
                  bHasDynamicSize = true;
               }
               if ( (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
               {
                  if ( (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
                  {
                     out << TAB << "if ((" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                        << " <= u16MajorVersion) && (u16MajorVersion <= "
                        << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << "))" << endl;
                  }
                  else
                  {
                     out << TAB << "if (" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                        << " <= u16MajorVersion)" << endl;
                  }
               }
               else
               {
                  out << TAB << "if (u16MajorVersion <= "
                     << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << ")" << endl;
               }
               out << TAB << "{" << endl;
               out << TAB << TAB << "u32DynamicSize += " << s32TypeSize << ";" << endl;

               tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
               tclFITypeDependencyListIterator iter2;
               // test, whether enumeration has version info at all
               bool bHasVersionInfo = false;
               for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
               {
                  if ( (*iter2)->bHasVersionInformation() )
                  {
                     bHasVersionInfo = true;
                  }
               }
               switch ( poBaseType->enGetType() )
               {
               case EN_OBJTYPE_ENUMERATION:
               {
                  if ( bHasVersionInfo )
                  {
                     for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
                     {
                        if ( (*iter2)->bHasVersionInformation() )
                        {
                           out << TAB << TAB << "if ((" << (*iter)->oGetName() << "Type.enType == "
                              << oGetClassName( poBaseType->oGetName() ) << "::"
                              << oGetEnumName( (*iter2)->oGetName() ) << ") &&" << endl;
                           if ( (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
                           {
                              if ( (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
                              {
                                 out << TAB << TAB << "    (("
                                    << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                    << " <= u16MajorVersion) && (u16MajorVersion <= "
                                    << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                                    << ")))" << endl;
                              }
                              else
                              {
                                 out << TAB << TAB << "    ("
                                    << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                    << " <= u16MajorVersion))" << endl;
                              }
                           }
                           else
                           {
                              out << TAB << TAB << "    (u16MajorVersion <= "
                                 << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                                 << "))" << endl;
                           }
                        }
                        else
                        {
                           out << TAB << TAB << "if (" << (*iter)->oGetName() << "Type.enType == "
                              << oGetClassName( poBaseType->oGetName() ) << "::"
                              << oGetEnumName( (*iter2)->oGetName() ) << ")" << endl;
                        }
                        out << TAB << TAB << TAB << "u32DynamicSize += " << "(po"
                           << oUnionName << "Data?po" << oUnionName
                           << "Data->u32GetSize(u16MajorVersion):0);" << endl;
                     }
                  }
                  else
                  {
                     out << TAB << TAB << "u32DynamicSize += " << "(po" << oUnionName << "Data?po"
                        << oUnionName << "Data->u32GetSize(u16MajorVersion):0);" << endl;
                  }
               } break;
               case EN_OBJTYPE_BITFIELD:
               {
                  int iOptionIdx = 0;
                  for ( int iBitIdx = 32; iBitIdx > 0; )
                  {
                     --iBitIdx;
                     for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
                     {
                        if ( iBitIdx == strtoul( (*iter2)->oGetDefaultValue().c_str(), 0, 0 ) )
                        {
                           out << TAB << TAB << "if (poOption" << iOptionIdx << " && (" << oUnionName << "Mask."
                              << oGetDataFieldName( poBaseType ) << " & "
                              << oGetClassName( poBaseType->oGetName() ) << "::FI_C_"
                              << oToUpper( (*iter2)->poGetTarget()->oGetName().substr( 1 ) )
                              << "_BIT_" << oToUpper( (*iter2)->oGetName() )
                              << ") &&" << endl;
                           if ( (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
                           {
                              if ( (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
                              {
                                 out << TAB << TAB << "    (("
                                    << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                    << " <= u16MajorVersion) && (u16MajorVersion <= "
                                    << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                                    << ")))" << endl;
                              }
                              else
                              {
                                 out << TAB << TAB << "    ("
                                    << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                    << " <= u16MajorVersion))" << endl;
                              }
                           }
                           else
                           {
                              out << TAB << TAB << "    (u16MajorVersion <= "
                                 << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                                 << "))" << endl;
                           }
                           out << TAB << TAB << "{" << endl;
                           out << TAB << TAB << TAB << "u32DynamicSize += poOption" << iOptionIdx << "->u32GetSize(u16MajorVersion);" << endl;
                           out << TAB << TAB << "}" << endl;
                           ++iOptionIdx;
                        }
                     }
                  }
               } break;
               }
               out << TAB << "}" << endl;
            }
            else
            {
               u32StaticSize += s32TypeSize;
               tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
               tclFITypeDependencyListIterator iter2;
               // test, whether enumeration has version info at all
               bool bHasVersionInfo = false;
               for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
               {
                  if ( (*iter2)->bHasVersionInformation() )
                  {
                     bHasVersionInfo = true;
                  }
               }
               switch ( poBaseType->enGetType() )
               {
               case EN_OBJTYPE_ENUMERATION:
               {
                  if ( bHasVersionInfo )
                  {
                     if ( !bHasDynamicSize )
                     {
                        out << TAB << "tU32 u32DynamicSize = 0;" << endl;
                        bHasDynamicSize = true;
                     }
                     for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
                     {
                        if ( (*iter2)->bHasVersionInformation() )
                        {
                           out << TAB << "if ((" << (*iter)->oGetName() << "Type.enType == "
                              << oGetClassName( poBaseType->oGetName() ) << "::"
                              << oGetEnumName( (*iter2)->oGetName() ) << ") &&" << endl;
                           if ( (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
                           {
                              if ( (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
                              {
                                 out << TAB << "    (("
                                    << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                    << " <= u16MajorVersion) && (u16MajorVersion <= "
                                    << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                                    << ")))" << endl;
                              }
                              else
                              {
                                 out << TAB << "    ("
                                    << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                    << " <= u16MajorVersion))" << endl;
                              }
                           }
                           else
                           {
                              out << TAB << "    (u16MajorVersion <= "
                                 << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                                 << "))" << endl;
                           }
                        }
                        else
                        {
                           out << TAB << "if (" << (*iter)->oGetName() << "Type.enType == "
                              << oGetClassName( poBaseType->oGetName() ) << "::"
                              << oGetEnumName( (*iter2)->oGetName() ) << ")" << endl;
                        }
                        out << TAB << TAB << "u32DynamicSize += " << "(po"
                           << oUnionName << "Data?po" << oUnionName
                           << "Data->u32GetSize(u16MajorVersion):0);" << endl;
                     }
                  }
                  else
                  {
                     oDynamicSize += "+(po" + oUnionName + "Data?po" + oUnionName + "Data->u32GetSize(u16MajorVersion):0)";
                  }
               } break;
               case EN_OBJTYPE_BITFIELD:
               {
                  tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
                  tclFITypeDependencyListIterator iter2;
                  int iOptionIdx = 0;
                  for ( int iBitIdx = 32; iBitIdx > 0; )
                  {
                     --iBitIdx;
                     for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
                     {
                        if ( iBitIdx == strtoul( (*iter2)->oGetDefaultValue().c_str(), 0, 0 ) )
                        {
                           if ( !bHasDynamicSize )
                           {
                              out << TAB << "tU32 u32DynamicSize = 0;" << endl;
                              bHasDynamicSize = true;
                           }
                           if ( (*iter2)->bHasVersionInformation() )
                           {
                              out << TAB << "if (poOption" << iOptionIdx << " && (" << oUnionName << "Mask."
                                 << oGetDataFieldName( poBaseType ) << " & "
                                 << oGetClassName( poBaseType->oGetName() ) << "::FI_C_"
                                 << oToUpper( (*iter2)->poGetTarget()->oGetName().substr( 1 ) )
                                 << "_BIT_" << oToUpper( (*iter2)->oGetName() )
                                 << ") && " << endl;
                              if ( (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
                              {
                                 if ( (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
                                 {
                                    out << TAB << "    (("
                                       << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                       << " <= u16MajorVersion) && (u16MajorVersion <= "
                                       << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                                       << ")))" << endl;
                                 }
                                 else
                                 {
                                    out << TAB << "    ("
                                       << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                       << " <= u16MajorVersion))" << endl;
                                 }
                              }
                              else
                              {
                                 out << TAB << "    (u16MajorVersion <= "
                                    << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                                    << "))" << endl;
                              }
                           }
                           else
                           {
                              out << TAB << "if (poOption" << iOptionIdx << " && (" << oUnionName << "Mask."
                                 << oGetDataFieldName( poBaseType ) << " & "
                                 << oGetClassName( poBaseType->oGetName() ) << "::FI_C_"
                                 << oToUpper( (*iter2)->poGetTarget()->oGetName().substr( 1 ) )
                                 << "_BIT_" << oToUpper( (*iter2)->oGetName() )
                                 << "))" << endl;
                           }
                           out << TAB << "{" << endl;
                           out << TAB << TAB << "u32DynamicSize += poOption" << iOptionIdx << "->u32GetSize(u16MajorVersion);" << endl;
                           out << TAB << "}" << endl;
                           ++iOptionIdx;
                        }
                     }
                  }
               } break;
               }
            }
         } break;
         }
      }
   } break;
   //default:
   //   return bGenerateExtendedTypeGetSizeMethod(poType, out, poServiceTree);
   /* break; */
   }

   out << TAB << "return ";
   if ( bHasDynamicSize )
   {
      out << "u32DynamicSize+";
   }
   out << u32StaticSize << oDynamicSize << ";" << endl;
   out << "}" << endl << endl;

   return true;
}

bool bGenerateTypeCompareMethod( tclFINode* poType, ostream &out, string oPrefix )
{
   string oTypeName /*= oPrefix*/;

   if ( bGenerateExtendedTypeCompareMethod( poType, out, oPrefix ) )
      return true;

   if ( poType->enGetType() == EN_OBJTYPE_OPCODE )
      oTypeName += oPrefix + oGetClassName( CG_MSGTYPE_PREFIX + oRemove( poType->oGetName(), '.' ) );
   else
      oTypeName += oGetClassName( poType->oGetName() );

   out << "tBool " << oTypeName << "::operator==(const " << oTypeName << "& roRef) const" << endl
       << "{" << endl;
   switch ( poType->enGetType() )
   {
   case EN_OBJTYPE_ATOM:
   case EN_OBJTYPE_TYPE:
   case EN_OBJTYPE_BITFIELD:
   case EN_OBJTYPE_ENUMERATION:
      break;
   case EN_OBJTYPE_OPCODE:
   case EN_OBJTYPE_STRUCTURE:
   {
      out << TAB "tBool bResult = true;" << endl;
      tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
      tclFITypeDependencyListIterator iter;
      for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
      {
         switch ( (*iter)->enGetClass() )
         {
         case EN_DEPENDENCY_CLASS_PARAM:
         case EN_DEPENDENCY_CLASS_FIELD:
         {
            string oFieldName = (*iter)->oGetName();
            out << TAB "bResult = bResult && (" << oFieldName << " == roRef." << oFieldName << ");" << endl;
         } break;
         case EN_DEPENDENCY_CLASS_LIST:
         {
            string oListName = (*iter)->oGetName();
            // string oCompTypeName = oGetClassName((*iter)->poGetTarget()->oGetName(),true);
            out << TAB TAB "bResult = bResult && (" << oListName << ".size() == roRef." << oListName << ".size());" << endl
                << TAB TAB "if (bResult)" << endl
                << TAB TAB "{" << endl
                << TAB TAB TAB "for (tU32 u32Idx = 0; bResult && (u32Idx < " << oListName << ".size()); ++u32Idx)" << endl
                << TAB TAB TAB TAB "bResult = bResult && (" << oListName << "[u32Idx] == roRef." << oListName << "[u32Idx]);" << endl
                << TAB TAB "}" << endl;
         } break;
         case EN_DEPENDENCY_CLASS_UNION:
         {
            tclFINode* poBaseType = (*iter)->poGetTarget();
            string oUnionName = (*iter)->oGetName();
            // string oCompTypeName = oGetClassName((*iter)->poGetTarget()->oGetName(),true);
            switch ( poBaseType->enGetType() )
            {
            case EN_OBJTYPE_ENUMERATION:
            {
               out << TAB "bResult = bResult && (" << oUnionName << "Type == roRef." << oUnionName << "Type);" << endl
                   << TAB "if (bResult && po" << oUnionName << "Data && roRef.po" << oUnionName << "Data) " << endl
                   << TAB "{" << endl
                   << TAB TAB "switch (" << oUnionName << "Type.enType)" << endl
                   << TAB TAB "{" << endl;
               tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
               tclFITypeDependencyListIterator iter2;
               for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
               {
                  out << TAB TAB "case " << oGetClassName( poBaseType->oGetName() ) << "::" << oGetEnumName( (*iter2)->oGetName() ) << ":" << endl;
                  string oUnionTypeName;
                  if ( (*iter2)->oGetName()[0] == 't' )
                  {
                     oUnionTypeName = oGetClassName( "T" + (*iter2)->oGetName().substr( 1 ) );
                  }
                  else
                  {
                     if ( bIsBinContainerType( (*iter2)->oGetName() ) )
                     {
                        oUnionTypeName = oGetClassName( "T_BinContainer" );
                     }
                     else
                     {
                        oUnionTypeName = oGetClassName( (*iter2)->oGetName() );
                     }
                  }
                  out << TAB TAB TAB "bResult = bResult && (*(const "
                         << oUnionTypeName << "*)po" << oUnionName << "Data == *(const "
                         << oUnionTypeName << "*)roRef.po"
                         << oUnionName << "Data); //lint !e826" << endl
                      << TAB TAB "break;" << endl;
               }
               out << TAB TAB "}" << endl
                   << TAB "}" << endl
                   << TAB "else" << endl
                   << TAB "{" << endl
                   << TAB TAB  "bResult = bResult && (po" << oUnionName << "Data == roRef.po" << oUnionName << "Data);" << endl
                   << TAB "}" << endl;
            } break;
            case EN_OBJTYPE_BITFIELD:
            {
               out << TAB TAB "bResult = bResult && (" << oUnionName << "Mask == roRef." << oUnionName << "Mask);" << endl;
               tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
               tclFITypeDependencyListIterator iter2;
               int iOptionIdx = 0;
               for ( int iBitIdx = 32; iBitIdx > 0; )
               {
                  --iBitIdx;
                  for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
                  {
                     if ( iBitIdx == strtoul( (*iter2)->oGetDefaultValue().c_str(), 0, 0 ) )
                     {
                        out << TAB TAB "if (bResult && (" << oUnionName << "Mask." << oGetDataFieldName( poBaseType )
                               << " & " << oGetClassName( poBaseType->oGetName() ) << "::FI_C_"
                               << oToUpper( (*iter2)->poGetTarget()->oGetName().substr( 1 ) ) << "_BIT_"
                               << oToUpper( (*iter2)->oGetName() )
                               << ") && poOption" << iOptionIdx << " && roRef.poOption" << iOptionIdx << ")" << endl
                            << TAB TAB TAB "bResult = bResult && (*(poOption" << iOptionIdx
                            << ") == *(roRef.poOption" << iOptionIdx << "));//lint !e826" << endl
                            << TAB TAB "else" << endl
                            << TAB TAB TAB "bResult = FALSE;" << endl;
                        ++iOptionIdx;
                     }
                  }
               }
            } break;
            }
         } break;
         }
      }
      out << TAB "return bResult;" << endl;
   } break;
   }
   out << "}" << endl << endl;
   return true;
}

bool bGenerateTypeReadMethod( tclFINode *poType, ostream &out, string oPrefix )
{
   string oTypeName /*= oPrefix*/;
   string oIndent;
   if ( poType->enGetType() == EN_OBJTYPE_OPCODE )
   {
      oTypeName += oPrefix + oGetClassName( CG_MSGTYPE_PREFIX + oRemove( poType->oGetName(), '.' ) );
   }
   else
      oTypeName += oGetClassName( poType->oGetName() );

   if ( bGenerateExtendedTypeReadMethod( poType, out, oPrefix ) )
      return true;

   out << "fi_tclInContext& " << oTypeName << "::oRead(fi_tclInContext& oIn)" << endl;
   out << "{" << endl;
   oIndent = TAB;

   switch ( poType->enGetType() )
   {
   case EN_OBJTYPE_ATOM:
      break;
   case EN_OBJTYPE_BITFIELD:
   case EN_OBJTYPE_TYPE:
   {
      tclFINode* poBaseType = (*poType->poGetDependencyList()->begin())->poGetTarget();
      out << oIndent << "return oIn >> "
         << oGetDataFieldName( poType ) << ";" << endl;
   } break;
   case EN_OBJTYPE_ENUMERATION:
   {
      tclFINode* poBaseType = (*poType->poGetDependencyList()->begin())->poGetTarget();
      out << oIndent << poBaseType->oGetName() << " " << oGetDataFieldName( poBaseType ) << ";" << endl;
      out << oIndent << "(tVoid) (oIn >> " << oGetDataFieldName( poBaseType ) << ");" << endl;
      out << oIndent << oGetDataFieldName( poType ) << " = (tenType)" << oGetDataFieldName( poBaseType ) << ";" << endl;
      out << oIndent << "return oIn;" << endl;
   } break;
   case EN_OBJTYPE_OPCODE:
   case EN_OBJTYPE_STRUCTURE:
   {
      tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
      tclFITypeDependencyListIterator iter;
      for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
      {
         if ( (*iter)->bHasVersionInformation() )
         {
            if ( (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
            {
               if ( (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
               {
                  out << oIndent << "if ((" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                     << " <= oIn.u16GetMajorVersion()) && (oIn.u16GetMajorVersion() <= "
                     << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << "))" << endl;
               }
               else
               {
                  out << oIndent << "if (" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                     << " <= oIn.u16GetMajorVersion())" << endl;
               }
            }
            else
            {
               out << oIndent << "if (oIn.u16GetMajorVersion() <= "
                  << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << ")" << endl;
            }
            out << oIndent << "{" << endl;
            oIndent += TAB;
         }
         switch ( (*iter)->enGetClass() )
         {
         case EN_DEPENDENCY_CLASS_PARAM:
         case EN_DEPENDENCY_CLASS_FIELD:
         {
            out << oIndent << "(tVoid) (oIn >> "
               << (*iter)->oGetName() << ");" << endl;
         } break;
         case EN_DEPENDENCY_CLASS_LIST:
         {
            tclFINode* poBaseType = (*iter)->poGetTarget();
            out << oIndent << "{" << endl;
            out << oIndent << TAB << "tU32 u32Length;" << endl;
            out << oIndent << TAB << "tU32 u32Idx;" << endl;
            switch ( poBaseType->enGetType() )
            {
            case EN_OBJTYPE_ATOM:
            case EN_OBJTYPE_TYPE:
            case EN_OBJTYPE_STRUCTURE:
            {
               string localString( "tString" );
               if ( (poBaseType->oGetName() == "tString") ||
                   (poBaseType->oGetName() == "tMultiLanguageString") ||
                   (poBaseType->poFindDependency( localString ) != 0) ||
                   (poBaseType->enGetType() == EN_OBJTYPE_STRUCTURE) )
               {
                  if ( bEnableDeepCopy )
                     out << "#ifndef VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;

                  out << oIndent << TAB << "for (u32Idx = 0; u32Idx < " << (*iter)->oGetName() << ".size(); ++u32Idx)" << endl;
                  out << oIndent << TAB << "{" << endl;
                  if ( (poBaseType->oGetName() == "tString") ||
                      (poBaseType->oGetName() == "tMultiLanguageString") ||
                     (poBaseType->poFindDependency( localString ) != 0) )
                  {
                     out << oIndent << TAB << TAB << (*iter)->oGetName() << "[u32Idx].vDestroy();" << endl;
                  }
                  else
                  {
                     if ( poBaseType->enGetType() == EN_OBJTYPE_STRUCTURE )
                     {
                        out << oIndent << TAB << TAB << (*iter)->oGetName() << "[u32Idx].vDestroy();" << endl;
                     }
                  }
                  out << oIndent << TAB << "}" << endl;
                  if ( bEnableDeepCopy )
                     out << "#endif // !VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;

               }
            }
            break;
            }
            out << oIndent << TAB << "(tVoid) (oIn >> u32Length);" << endl;
            out << oIndent << TAB << (*iter)->oGetName() << ".resize(u32Length);" << endl;
            out << oIndent << TAB << "for (u32Idx = 0; u32Idx < u32Length; ++u32Idx)" << endl;
            out << oIndent << TAB << "{" << endl;
            if ( poBaseType->oGetName() == "tBool" || poBaseType->oGetName() == "bool" )
            {
               out << oIndent << TAB << TAB << "tBool bData;" << endl;
               out << oIndent << TAB << TAB << "oIn >> bData;" << endl;
               out << oIndent << TAB << TAB << (*iter)->oGetName() << "[u32Idx] = bData;" << endl;
            }
            else
            {
               out << oIndent << TAB << TAB << "(tVoid) (oIn >> " << (*iter)->oGetName() << "[u32Idx]);" << endl;
            }
            out << oIndent << TAB << "}" << endl;
            out << oIndent << "}" << endl;
         } break;
         case EN_DEPENDENCY_CLASS_UNION:
         {
            tclFINode* poBaseType = (*iter)->poGetTarget();
            string oUnionName = (*iter)->oGetName();
            switch ( poBaseType->enGetType() )
            {
            case EN_OBJTYPE_ENUMERATION:
            {
               if ( bEnableDeepCopy )
               {
                  out << "#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;
                  out << oIndent << "OSAL_DELETE po" << oUnionName << "Data;" << endl;
                  out << "#else" << endl;
               }
               out << oIndent << "if (po" << oUnionName << "Data)" << endl;
               out << oIndent << "{" << endl;
               out << oIndent << TAB << "po" << oUnionName << "Data->vDestroy();" << endl;
               out << oIndent << TAB << "OSAL_DELETE po" << oUnionName << "Data;" << endl;
               out << oIndent << "}" << endl;
               if ( bEnableDeepCopy )
                  out << "#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;

               out << oIndent << "(tVoid) (oIn >> " << oUnionName << "Type);" << endl;
               out << oIndent << "switch(" << oUnionName << "Type."
                  << oGetDataFieldName( poBaseType ) << ")" << endl;
               out << oIndent << "{" << endl;
               tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
               tclFITypeDependencyListIterator iter2;
               for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
               {
                  out << oIndent << "case "
                     << oGetClassName( poBaseType->oGetName() ) << "::"
                     << oGetEnumName( (*iter2)->oGetName() ) << ":" << endl;
                  if ( (*iter2)->oGetName()[0] == 't' )
                  {
                     out << oIndent << TAB << "po" << oUnionName << "Data = OSAL_NEW "
                        << oGetClassName( "T" + (*iter2)->oGetName().substr( 1 ) )
                        << ";" << endl;
                  }
                  else
                  {
                     if ( bIsBinContainerType( (*iter2)->oGetName() ) )
                     {
                        // For safety reason, hide bincontainer structure also internally!
                        /*
                        out << "#ifndef FI_S_EXTERNAL_FI" << endl;
                        out << oIndent << TAB << "po" << oUnionName << "Data = OSAL_NEW "
                        << oGetClassName((*iter2)->oGetName()) << ";" << endl;
                        out << "#else" << endl;*/
                        out << oIndent << TAB << "po" << oUnionName << "Data = OSAL_NEW "
                           << oGetClassName( "T_BinContainer" ) << ";" << endl;
                        // out << "#endif // FI_S_EXTERNAL_FI" << endl;
                     }
                     else
                     {
                        out << oIndent << TAB << "po" << oUnionName << "Data = OSAL_NEW "
                           << oGetClassName( (*iter2)->oGetName() ) << ";" << endl;
                     }
                  }
                  out << oIndent << TAB << "break;" << endl;
               }
               out << oIndent << "default:" << endl;
               out << oIndent << TAB << "oIn.Invalidate();" << endl;
               out << oIndent << TAB << "po" << oUnionName << "Data = OSAL_NULL;" << endl;
               out << oIndent << "}" << endl;
               out << oIndent << "if (po" << oUnionName << "Data && oIn.bIsValid())" << endl;
               out << oIndent << TAB << "po" << (*iter)->oGetName() << "Data->oRead(oIn);" << endl;
               out << oIndent << "else" << endl;
               out << oIndent << TAB << "oIn.Invalidate();" << endl;
            } break;
            case EN_OBJTYPE_BITFIELD:
            {

               tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
               tclFITypeDependencyListIterator iter2;
               int iBitIdx = 0;
               for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
               {
                  out << oIndent << "if (poOption" << iBitIdx << " != 0)" << endl;
                  out << oIndent << "{" << endl;
                  if ( bEnableDeepCopy )
                     out << "#ifndef VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;

                  out << oIndent << TAB << "poOption" << iBitIdx << "->vDestroy();" << endl;
                  if ( bEnableDeepCopy )
                     out << "#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;

                  out << oIndent << TAB << "OSAL_DELETE poOption" << iBitIdx << ";" << endl;
                  out << oIndent << TAB << "poOption" << iBitIdx << " = OSAL_NULL;" << endl;
                  out << oIndent << "}" << endl;
                  ++iBitIdx;
               }
               out << oIndent << "(tVoid) (oIn >> " << oUnionName << "Mask);" << endl;
               // scan the bit in descending order!!
               int iOptionIdx = 0;
               for ( iBitIdx = 32; iBitIdx > 0; )
               {
                  --iBitIdx;
                  for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
                  {
                     if ( strtoul( (*iter2)->oGetDefaultValue().c_str(), 0, 0 ) == iBitIdx )
                     {
                        out << oIndent << "if (" << oUnionName << "Mask."
                           << oGetDataFieldName( poBaseType ) << " & "
                           << oGetClassName( poBaseType->oGetName() ) << "::FI_C_"
                           << oToUpper( (*iter2)->poGetTarget()->oGetName().substr( 1 ) ) << "_BIT_"
                           << oToUpper( (*iter2)->oGetName() )
                           << ")" << endl;
                        out << oIndent << "{" << endl;
                        if ( bIsBinContainerType( (*iter2)->poGetTarget()->oGetName() ) )
                        {
                           // For safety reason, hide bincontainer structure also internally!
                           /*
                           out << "#ifndef FI_S_EXTERNAL_FI" << endl;
                           out << oIndent << TAB << "poOption" << iBitIdx << " = OSAL_NEW "
                           << oGetClassName((*iter2)->oGetName()) << ";" << endl;
                           out << "#else" << endl;*/
                           out << oIndent << TAB << "poOption" << iOptionIdx << " = OSAL_NEW "
                              << oGetClassName( "T_BinContainer" ) << ";" << endl;
                           // out << "#endif //FI_S_EXTERNAL_FI" << endl;

                        }
                        else
                        {
                           out << oIndent << TAB << "poOption" << iOptionIdx << " = OSAL_NEW "
                              << oGetClassName( (*iter2)->oGetName() ) << ";" << endl;
                        }
                        out << oIndent << TAB << "if (poOption" << iOptionIdx << ")" << endl;
                        out << oIndent << TAB << TAB << "(tVoid) poOption" << iOptionIdx
                           << "->oRead(oIn);" << endl;
                        out << oIndent << TAB << "else" << endl;
                        out << oIndent << TAB << TAB << "oIn.Invalidate();" << endl;
                        out << oIndent << "}" << endl;
                        iOptionIdx++;
                     }
                  }
               }
            } break;
            }
         } break;
         }
         if ( (*iter)->bHasVersionInformation() )
         {
            oIndent = oIndent.substr( strlen( TAB ) );
            out << oIndent << "}" << endl;
         }
         if ( bIsBinContainerType( poType->oGetName() ) &&
             ((*iter)->oGetName() == "ContainerType") )
         {
            out << oIndent << "if (!oIn.bIsValid() ||" << endl;
            out << oIndent << "    (ContainerLength <= 6) ||" << endl;
            out << oIndent << "    (ContainerType.enType != "
               << oGetClassName( (*iter)->poGetTarget()->oGetName() )
               << "::" << oGetEnumName( poType->oGetName() ) << ") ||" << endl;
            //               out << oIndent << "    (ServiceID != this->C_U16_SERVICEID) ||" << endl;
            out << oIndent << "    (ContainerFIVersion.MajorVersion < " << oTypeName << "::C_U16_MAJORVERSION_MIN) ||" << endl;
            out << oIndent << "    (ContainerFIVersion.MajorVersion > " << oTypeName << "::C_U16_MAJORVERSION_MAX))" << endl;
            out << oIndent << "{" << endl;
            //               out << oIndent << TAB << "ContainerType.enType = fi_tcl_e8_BinContainerTypes::FI_EN_T_RESERVED;" << endl;
            out << oIndent << TAB << "if (oIn.bIsValid() && (ContainerLength >= 6))" << endl;
            out << oIndent << TAB << "{" << endl;
            out << oIndent << TAB << TAB << "// read rest of container but discard it" << endl;
            out << oIndent << TAB << TAB << "for (tU32 u32Dummy = 0; u32Dummy < (ContainerLength - 6); ++u32Dummy)" << endl;
            out << oIndent << TAB << TAB << "{" << endl;
            out << oIndent << TAB << TAB << TAB << "tU8 u8Dummy;" << endl;
            out << oIndent << TAB << TAB << TAB << "oIn.oRead(u8Dummy);" << endl;
            out << oIndent << TAB << TAB << "}" << endl;
            out << oIndent << TAB << "}" << endl;
            out << oIndent << TAB << "else" << endl;
            out << oIndent << TAB << "{" << endl;
            out << oIndent << TAB << TAB << "oIn.Invalidate();" << endl;
            out << oIndent << TAB << "}" << endl;
            out << oIndent << TAB << "return oIn;" << endl;
            out << oIndent << "}" << endl;
            out << oIndent << "tU16 u16MajorVersion = oIn.u16SetMajorVersion(ContainerFIVersion.MajorVersion);" << endl;
         }
      }
      if ( bIsBinContainerType( poType->oGetName() ) )
      {
         out << oIndent << "oIn.u16SetMajorVersion(u16MajorVersion);" << endl;
      }
      out << oIndent << "return oIn;" << endl;
   } break;

   }
   out << "}" << endl << endl;

   return true;
}

bool bGenerateTypeWriteMethod( tclFINode* poType, ostream &out, string oPrefix )
{
   string oTypeName /*= oPrefix*/;
   string oIndent;
   if ( poType->enGetType() == EN_OBJTYPE_OPCODE )
   {
      oTypeName += oPrefix + oGetClassName( CG_MSGTYPE_PREFIX + oRemove( poType->oGetName(), '.' ) );
   }
   else
      oTypeName += oGetClassName( poType->oGetName() );

   if ( bGenerateExtendedTypeWriteMethod( poType, out, oPrefix ) )
      return true;

   out << "fi_tclOutContext& " << oTypeName << "::oWrite(fi_tclOutContext& oOut) const" << endl;
   out << "{" << endl;
   oIndent = TAB;

   switch ( poType->enGetType() )
   {
   case EN_OBJTYPE_ATOM:
      break;
   case EN_OBJTYPE_TYPE:
   {
      tclFINode* poBaseType = (*poType->poGetDependencyList()->begin())->poGetTarget();
      out << oIndent << "return oOut << "
         << oGetDataFieldName( poType ) << ";" << endl;
   } break;
   case EN_OBJTYPE_BITFIELD:
   {
      tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
      tclFITypeDependencyListIterator iter;
      tclFINode* poBaseType = (*poType->poGetDependencyList()->begin())->poGetTarget();
      // test, whether enumeration has version info at all
      bool bHasVersionInfo = false;
      for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
      {
         if ( (*iter)->bHasVersionInformation() )
         {
            bHasVersionInfo = true;
         }
      }
      if ( bHasVersionInfo )
      {
         out << oIndent << oGetClassName( poBaseType->oGetName() ) << " temp = "
            << oGetDataFieldName( poType ) << ";" << endl;
         for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
         {
            if ( (*iter)->bHasVersionInformation() )
            {
               if ( (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
               {
                  if ( (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
                  {
                     out << oIndent << "if (!((" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                        << " <= oOut.u16GetMajorVersion()) && (oOut.u16GetMajorVersion() <= "
                        << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << ")))" << endl;
                  }
                  else
                  {
                     out << oIndent << "if (!(" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                        << " <= oOut.u16GetMajorVersion()))" << endl;
                  }
               }
               else
               {
                  out << oIndent << "if (!(oOut.u16GetMajorVersion() <= "
                     << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << ") )" << endl;
               }
               out << oIndent << TAB << "temp &= ~FI_C_"
                  << oToUpper( (*iter)->poGetTarget()->oGetName().substr( 1 ) )
                  << "_BIT_" << oToUpper( (*iter)->oGetName() ) << ";" << endl;
            }
         }
         out << oIndent << "return oOut << temp;" << endl;
      }
      else
      {
         tclFINode* poBaseType = (*poType->poGetDependencyList()->begin())->poGetTarget();
         {
            out << oIndent << "return oOut << "
               << oGetDataFieldName( poType ) << ";" << endl;
         }
      }
   } break;
   case EN_OBJTYPE_ENUMERATION:
   {
#pragma message("ToDo: Make Fallback-Handling robust against element ordering")
      tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
      tclFITypeDependencyListIterator iter;
      tclFINode* poBaseType = (*poType->poGetDependencyList()->begin())->poGetTarget();
      // test, whether enumeration has version info at all
      bool bHasVersionInfo = false;
      for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
      {
         if ( (*iter)->bHasVersionInformation() )
         {
            bHasVersionInfo = true;
         }
      }
      if ( bHasVersionInfo )
      {
         // easy Fallback handling:
         // it is assumed that new elements are always added at the end!
         out << oIndent << oGetClassName( poType->oGetName() ) << "::tenType enData = enType;" << endl;

         for ( iter = poDepList->end(); iter != poDepList->begin(); )
         {
            --iter;
            if ( (*iter)->bHasVersionInformation() )
            {
               if ( (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
               {
                  if ( (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
                  {
                     out << oIndent << "if ((enData == " << oGetEnumName( (*iter)->oGetName() )
                        << ") && !((" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                        << " <= oOut.u16GetMajorVersion()) && (oOut.u16GetMajorVersion() <= "
                        << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << ")))" << endl;
                  }
                  else
                  {
                     out << oIndent << "if ((enData == " << oGetEnumName( (*iter)->oGetName() )
                        << ") && !(" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                        << " <= oOut.u16GetMajorVersion()))" << endl;
                  }
               }
               else
               {
                  out << oIndent << "if ((enData == " << oGetEnumName( (*iter)->oGetName() )
                     << ") && !(oOut.u16GetMajorVersion() <= "
                     << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << "))" << endl;
               }
               out << oIndent << "{" << endl;
               out << oIndent << TAB << "enData = " << oGetEnumName( (*iter)->oGetFallbackValue() )
                  << ";" << endl;
               out << oIndent << "}" << endl;
            }
         }
         out << oIndent << "return oOut.oWrite((" << poBaseType->oGetName() << ")enData);"
            << endl;
      }
      else
      {
         // normal case
         out << oIndent << "return oOut.oWrite(("
            << poBaseType->oGetName()
            << ")" << oGetDataFieldName( poType ) << ");"
            << endl;
      }
   } break;
   case EN_OBJTYPE_OPCODE:
   case EN_OBJTYPE_STRUCTURE:
   {
      tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
      tclFITypeDependencyListIterator iter;
      if ( bIsBinContainerType( poType->oGetName() ) )
      {
         out << oIndent << "tU16 u16MajorVersion = oOut.u16SetMajorVersion(ContainerFIVersion.MajorVersion);" << endl;
      }
      for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
      {
         if ( (*iter)->bHasVersionInformation() )
         {
            if ( (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
            {
               if ( (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
               {
                  out << oIndent << "if ((" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                     << " <= oOut.u16GetMajorVersion()) && (oOut.u16GetMajorVersion() <= "
                     << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << "))" << endl;
               }
               else
               {
                  out << oIndent << "if (" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                     << " <= oOut.u16GetMajorVersion())" << endl;
               }
            }
            else
            {
               out << oIndent << "if (oOut.u16GetMajorVersion() <= "
                  << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << ")" << endl;
            }
            out << oIndent << "{" << endl;
            oIndent += TAB;
         }
         switch ( (*iter)->enGetClass() )
         {
         case EN_DEPENDENCY_CLASS_PARAM:
         case EN_DEPENDENCY_CLASS_FIELD:
         {
            out << oIndent << "(tVoid) (oOut << "
               << (*iter)->oGetName() << ");" << endl;
         } break;
         case EN_DEPENDENCY_CLASS_LIST:
         {
            out << oIndent << "(tVoid) (oOut << (tU32)" << (*iter)->oGetName()
               << ".size());" << endl;
            out << oIndent << "{" << endl;
            out << oIndent << TAB << "for (tU32 u32Idx = 0; u32Idx < "
               << (*iter)->oGetName() << ".size(); ++u32Idx)" << endl;
            out << oIndent << TAB << TAB << "(tVoid) (oOut << "
               << (*iter)->oGetName() << "[u32Idx]);" << endl;
            out << oIndent << "}" << endl;
         } break;
         case EN_DEPENDENCY_CLASS_UNION:
         {
            tclFINode* poBaseType = (*iter)->poGetTarget();

            tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
            tclFITypeDependencyListIterator iter2;
            // test, whether enumeration has version info at all
            bool bHasVersionInfo = false;
            for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
            {
               if ( (*iter2)->bHasVersionInformation() )
               {
                  bHasVersionInfo = true;
               }
            }

            switch ( poBaseType->enGetType() )
            {
            case EN_OBJTYPE_ENUMERATION:
            {
               out << oIndent << "(tVoid) (oOut << "
                  << (*iter)->oGetName() << "Type);"
                  << endl;
               out << oIndent << "if (po" << (*iter)->oGetName()
                  << "Data)" << endl;
               out << oIndent << "{" << endl;
               if ( bHasVersionInfo )
               {
                  for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
                  {
                     out << oIndent << TAB << "if (" << (*iter)->oGetName() << "Type.enType == "
                        << oGetClassName( poBaseType->oGetName() )
                        << "::" << oGetEnumName( (*iter2)->oGetName() ) << ")" << endl;
                     out << oIndent << TAB << "{" << endl;
                     if ( (*iter2)->bHasVersionInformation() )
                     {
                        if ( (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
                        {
                           if ( (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
                           {
                              out << oIndent << TAB << TAB << "if (!(("
                                 << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                 << " <= oOut.u16GetMajorVersion()) && (oOut.u16GetMajorVersion() <= "
                                 << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                                 << ")))" << endl;
                           }
                           else
                           {
                              out << oIndent << TAB << TAB << "if (!("
                                 << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                 << " <= oOut.u16GetMajorVersion()))" << endl;
                           }
                        }
                        else
                        {
                           out << oIndent << TAB << TAB << "if (!(oOut.u16GetMajorVersion() <= "
                              << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                              << "))" << endl;
                        }
                        out << oIndent << TAB << TAB << "{" << endl;
                        out << oIndent << TAB << TAB << TAB << oGetClassName( (*iter2)->oGetFallbackValue() )
                           << " oDefault;" << endl;
                        out << oIndent << TAB << TAB << TAB << "(tVoid) (oOut << oDefault);" << endl;
                        out << oIndent << TAB << TAB << "}" << endl;
                        out << oIndent << TAB << TAB << "else" << endl;
                        out << oIndent << TAB << TAB << "{" << endl;
                        out << oIndent << TAB << TAB << TAB << "po" << (*iter)->oGetName()
                           << "Data->oWrite(oOut);" << endl;
                        out << oIndent << TAB << TAB << "}" << endl;
                     }
                     else
                     {
                        out << oIndent << TAB << TAB << "po" << (*iter)->oGetName()
                           << "Data->oWrite(oOut);" << endl;
                     }
                     out << oIndent << TAB << "}" << endl;
                  }
               }
               else
               {
                  out << oIndent << TAB << "(tVoid) po"
                     << (*iter)->oGetName()
                     << "Data->oWrite(oOut);" << endl;
               }
               out << oIndent << "}" << endl;
               out << oIndent << "else" << endl;
               out << oIndent << "{" << endl;
               out << oIndent << TAB << "oOut.Invalidate();" << endl;
               out << oIndent << "}" << endl;
            } break;
            case EN_OBJTYPE_BITFIELD:
            {
               out << oIndent << "(tVoid) (oOut << "
                  << (*iter)->oGetName()
                  << "Mask);" << endl;
               int iBitIdx = 0;
               tclFITypeDependencyList* poDepList2 = poBaseType->poGetDependencyList();
               tclFITypeDependencyListIterator iter2;
               int iOptionIdx = 0;
               for ( iBitIdx = 32; iBitIdx > 0; )
               {
                  --iBitIdx;
                  for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
                  {
                     if ( strtoul( (*iter2)->oGetDefaultValue().c_str(), 0, 0 ) == iBitIdx )
                     {
                        if ( (*iter2)->bHasVersionInformation() )
                        {
                           out << oIndent << "if (("
                              << (*iter)->oGetName() << "Mask."
                              << oGetDataFieldName( poBaseType ) << " & "
                              << oGetClassName( poBaseType->oGetName() ) << "::FI_C_"
                              << oToUpper( (*iter2)->poGetTarget()->oGetName().substr( 1 ) )
                              << "_BIT_" << oToUpper( (*iter2)->oGetName() )
                              << ") && " << endl;
                           if ( (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
                           {
                              if ( (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
                              {
                                 out << oIndent << "    (("
                                    << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                    << " <= oOut.u16GetMajorVersion()) && (oOut.u16GetMajorVersion() <= "
                                    << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                                    << ")))" << endl;
                              }
                              else
                              {
                                 out << oIndent << "    ("
                                    << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                    << " <= oOut.u16GetMajorVersion()))" << endl;
                              }
                           }
                           else
                           {
                              out << oIndent << "    (oOut.u16GetMajorVersion() <= "
                                 << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                                 << "))" << endl;
                           }
                        }
                        else
                        {
                           out << oIndent << "if ("
                              << (*iter)->oGetName() << "Mask."
                              << oGetDataFieldName( poBaseType ) << " & "
                              << oGetClassName( poBaseType->oGetName() ) << "::FI_C_"
                              << oToUpper( (*iter2)->poGetTarget()->oGetName().substr( 1 ) )
                              << "_BIT_" << oToUpper( (*iter2)->oGetName() )
                              << ")" << endl;
                        }
                        out << oIndent << "{" << endl;
                        out << oIndent << TAB << "if (poOption" << iOptionIdx << ")" << endl;
                        out << oIndent << TAB << "{" << endl;
                        out << oIndent << TAB << TAB << "poOption" << iOptionIdx
                           << "->oWrite(oOut);" << endl;
                        out << oIndent << TAB << "}" << endl;
                        out << oIndent << TAB << "else" << endl;
                        out << oIndent << TAB << "{" << endl;
                        out << oIndent << TAB << TAB << "oOut.Invalidate();" << endl;
                        out << oIndent << TAB << "}" << endl;
                        out << oIndent << "}" << endl;
                        ++iOptionIdx;
                     }
                  }
               }
            } break;
            }
         } break;
         }
         if ( (*iter)->bHasVersionInformation() )
         {
            oIndent = oIndent.substr( strlen( TAB ) );
            out << oIndent << "}" << endl;
         }
      }
      if ( bIsBinContainerType( poType->oGetName() ) )
      {
         out << oIndent << "oOut.u16SetMajorVersion(u16MajorVersion);" << endl;
      }
      out << oIndent << "return oOut;" << endl;
   } break;

   }
   out << "}" << endl << endl;
   return true;
}

bool bGenerateTypeClassDefinition( tclFINode *poType, ostream &out, tclFIServiceTree* poServiceTree )
{

   // skip types having optional attribute "predefined" set to "true"
   // (mkoch, 2010-03-04)
   if ( bIsPredefinedType( poType ) )
   {
      string tname = poType->oGetName();
      out << "/****** using predefined declaration for type ";
      out << tname << " ******/" << endl << endl;
      return true;
   }
   out << "//================================================================="
       << "============" << endl << endl;
   // ----

   if ( bIsBinContainerType( poType->oGetName() ) )
   {
      out << "#ifndef FI_S_EXTERNAL_FI" << endl;
   }

   string oPrefix = oToLower( poServiceTree->oGetShortname().substr( 0, poServiceTree->oGetShortname().find_last_of( '_' ) ) );
   if ( bEnableDeepCopy && (poType->poGetDependencyList()->size() > 0) )
   {
      out << "#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;
      // generate copy constructor
      bGenerateTypeCopyConstructor( poType, out, oPrefix );
      // generator assignment operator
      bGenerateTypeAssignmentOp( poType, out, oPrefix );
      // generate real destructor or manual "destructor" vDestroy()
      out << "#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl << endl;
   }

   bGenerateDestroyMethod( poType, out, oPrefix );

   //   if(poType->enGetType()==EN_OBJTYPE_OPCODE)
   //   {
   //      bGenerateMsgInfo(poType,out,oPrefix,poServiceTree);
   //   }

   bGenerateTypeConstructor( poType, out, oPrefix );

   {
      string oTypeName;
      if ( poType->enGetType() == EN_OBJTYPE_OPCODE )
         oTypeName += oPrefix + oGetClassName( CG_MSGTYPE_PREFIX + oRemove( poType->oGetName(), '.' ) );
      else
         oTypeName += oGetClassName( poType->oGetName() );

      out << "tS32 " << oTypeName << "::s32GetTypeId() const" << endl
          << "{" << endl << TAB;
      if ( oToLower( poServiceTree->oGetShortname() ) == (oGetFIGroup() + "types_fi") )
         out << "return (tS32) " << oPrefix << "fi" << oGetTokenName( poType->oGetName() ) << ";" << endl;
      else
         out << "return (tS32) " << oPrefix << "fi" << oRemove( oGetTokenName( "__MSG_" + poType->oGetName() ), '.' ) << ";" << endl;
      out << "}" << endl << endl;
   }

   bGenerateTypeDestructor( poType, out, oPrefix );


   if ( ((poType->enGetType() != EN_OBJTYPE_OPCODE)
             || (poType->poGetDependencyList()->size() != 0))
        && !bIsStandardErrorMessageExt( poType ) )
   {
      // generate u32GetSize()
      bGenerateTypeGetSizeMethod( poType, out, poServiceTree );
      // generate oRead()
      bGenerateTypeReadMethod( poType, out, oPrefix );
      // generate oWrite()
      bGenerateTypeWriteMethod( poType, out, oPrefix );
   }

   // generate operator==()
   if ( ((poType->enGetType() == EN_OBJTYPE_STRUCTURE) || (poType->enGetType() == EN_OBJTYPE_OPCODE)) &&
    (poType->poGetDependencyList()->size() == 0) )
      ;   // nothing to do; already defined inline
   else if ( poType->enGetType() == EN_OBJTYPE_TYPE )
      ;   // nothing to do; already defined inline
   else if ( (poType->enGetType() == EN_OBJTYPE_BITFIELD) || (poType->enGetType() == EN_OBJTYPE_ENUMERATION) )
      ;   // nothing to do; already defined inline
   else
   {
      bGenerateTypeCompareMethod( poType, out, oPrefix );
   }

   // generate additional mamber functions, if required by FI extension
   bGenerateExtendedTypeMembers( poType, out, oPrefix );

   if ( bIsBinContainerType( poType->oGetName() ) )
   {
      out << "tU16 " << oGetClassName( poType->oGetName() ) << "::u16Transform(const " << oGetFIGroup() << "fi_tcl_BinContainer& oContainer)" << endl;
      out << "{" << endl;
      out << TAB << "tU16 u16RetVal = 0;" << endl;
      out << TAB << "if (&oContainer != OSAL_NULL) //lint !e774 (&oContainer CAN be NULL!!)" << endl;
      out << TAB << "{" << endl;
      out << TAB << TAB << "this->vDestroy();" << endl;
      out << TAB << TAB << "tU32 u32MinSize = this->u32GetSize(" << oGetClassName( poType->oGetName() ) << "::C_U16_MAJORVERSION_MAX);" << endl;
      out << TAB << TAB << "tU32 u32Size = oContainer.u32GetSize(0);" << endl;
      out << TAB << TAB << "if (u32Size >= u32MinSize)" << endl;
      out << TAB << TAB << "{" << endl;
      out << TAB << TAB << TAB << "tU8* pu8Buf = OSAL_NEW tU8[u32Size];" << endl;
      out << TAB << TAB << TAB << "if (pu8Buf != OSAL_NULL)" << endl;
      out << TAB << TAB << TAB << "{" << endl;
      out << TAB << TAB << TAB << TAB << "if (oContainer.ContainerDataList[8] == (tU8)" << oGetFIGroup()
         << "fi_tcl_e8_BinContainerTypes::" << oGetEnumName( poType->oGetName() ) << ")" << endl;
      out << TAB << TAB << TAB << TAB << "{" << endl;
      out << TAB << TAB << TAB << TAB << TAB << "tU16 u16MajorVersion = oContainer.ContainerDataList[2] + (oContainer.ContainerDataList[3] << 8);" << endl;
      out << TAB << TAB << TAB << TAB << TAB << "if ((u16MajorVersion >= " << oGetClassName( poType->oGetName() ) << "::C_U16_MAJORVERSION_MIN) &&" << endl;
      out << TAB << TAB << TAB << TAB << TAB << "    (u16MajorVersion <= " << oGetClassName( poType->oGetName() ) << "::C_U16_MAJORVERSION_MAX))" << endl;
      out << TAB << TAB << TAB << TAB << TAB << "{" << endl;
      out << TAB << TAB << TAB << TAB << TAB << TAB << "fi_tclOutContext oOut(pu8Buf,pu8Buf+u32Size,u16MajorVersion);" << endl;
      out << TAB << TAB << TAB << TAB << TAB << TAB << "fi_tclInContext  oIn (pu8Buf,pu8Buf+u32Size,u16MajorVersion);" << endl;
      out << TAB << TAB << TAB << TAB << TAB << TAB << "(tVoid) (oOut << oContainer);" << endl;
      out << TAB << TAB << TAB << TAB << TAB << TAB << "(tVoid) (oIn >> *this);" << endl;
      out << TAB << TAB << TAB << TAB << TAB << TAB << "if (!oIn.bIsValid())" << endl;
      out << TAB << TAB << TAB << TAB << TAB << TAB << "{" << endl;
      out << TAB << TAB << TAB << TAB << TAB << TAB << TAB << "u16RetVal = " << oToUpper( oGetFIGroup() ) << "FI_C_U16_ERROR_INTERNALFAILURE;" << endl;
      out << TAB << TAB << TAB << TAB << TAB << TAB << "}" << endl;
      out << TAB << TAB << TAB << TAB << TAB << "}" << endl;
      out << TAB << TAB << TAB << TAB << TAB << "else" << endl;
      out << TAB << TAB << TAB << TAB << TAB << "{" << endl;
      out << TAB << TAB << TAB << TAB << TAB << TAB << "u16RetVal = " << oToUpper( oGetFIGroup() )
         << "FI_C_U16_ERROR_INVALIDBINCONTAINERWRONGVERSION;" << endl;
      out << TAB << TAB << TAB << TAB << TAB << "}" << endl;
      out << TAB << TAB << TAB << TAB << "}" << endl;
      out << TAB << TAB << TAB << TAB << "else" << endl;

      out << TAB << TAB << TAB << TAB << "{" << endl;
      out << TAB << TAB << TAB << TAB << TAB << "u16RetVal = " << oToUpper( oGetFIGroup() )
         << "FI_C_U16_ERROR_INVALIDBINCONTAINERUNEXPECTEDDATA;" << endl;
      out << TAB << TAB << TAB << TAB << "}" << endl;
      // out << TAB << TAB << "bSuccess = oOut.bIsValid() && oIn.bIsValid();" << endl;
      out << TAB << TAB << TAB << "}" << endl;
      out << TAB << TAB << TAB << "else" << endl;
      out << TAB << TAB << TAB << "{" << endl;
      out << TAB << TAB << TAB << TAB << "this->ContainerType.enType = " << oGetFIGroup()
         << "fi_tcl_e8_BinContainerTypes::FI_EN_T_RESERVED;" << endl;
      out << TAB << TAB << TAB << TAB << "u16RetVal = " << oToUpper( oGetFIGroup() ) << "FI_C_U16_ERROR_INTERNALFAILURE;" << endl;
      out << TAB << TAB << TAB << "}" << endl;
      out << TAB << TAB << TAB << "OSAL_DELETE[] pu8Buf;" << endl;
      out << TAB << TAB << "}" << endl;
      out << TAB << TAB << "else" << endl;
      out << TAB << TAB << "{" << endl;
      out << TAB << TAB << TAB << "u16RetVal = " << oToUpper( oGetFIGroup() ) << "FI_C_U16_ERROR_INVALIDBINCONTAINERUNEXPECTEDDATA;" << endl;
      out << TAB << TAB << "}" << endl;
      out << TAB << "}" << endl;
      out << TAB << "else" << endl;
      out << TAB << "{" << endl;
      out << TAB << TAB << "u16RetVal = " << oToUpper( oGetFIGroup() ) << "FI_C_U16_ERROR_INVALIDBINCONTAINERUNEXPECTEDDATA;" << endl;
      out << TAB << "}" << endl;
      out << TAB << "return u16RetVal;" << endl;
      out << "}" << endl;
      out << endl;
      out << "const tU16 " << oGetClassName( poType->oGetName() ) << "::C_U16_MAJORVERSION_MAX = "
         << poType->poGetVersionRange()->oGetMaxVersion().iGetMajor() << ";" << endl;
      out << "const tU16 " << oGetClassName( poType->oGetName() ) << "::C_U16_MAJORVERSION_MIN = "
         << poType->poGetVersionRange()->oGetMinVersion().iGetMajor() << ";" << endl;
      out << endl;
   }

   if ( bIsBinContainerType( poType->oGetName() ) )
   {
      out << "#endif // FI_S_EXTERNAL_FI" << endl;
   }
   return true;
}

int iGetOpCodeId( tclFINode* poMsgNode )
{
   int iOpcodeId = -1;
   string oOpcode = poMsgNode->oGetName();
   oOpcode = oOpcode.substr( oOpcode.find( '.' ) + 1 );
   if ( !bUseNonStandardOpcodeId( oOpcode, iOpcodeId ) )
   {
      if ( oOpcode == "Set" )
         iOpcodeId = 0;
      else if ( oOpcode == "Get" )
         iOpcodeId = 1;
      else if ( oOpcode == "MethodStart" )
         iOpcodeId = 2;
      else if ( oOpcode == "UpReg" )
         iOpcodeId = 3;
      else if ( oOpcode == "RelUpReg" )
         iOpcodeId = 4;
      else if ( oOpcode == "MethodAbort" )
         iOpcodeId = 5;
      else if ( oOpcode == "Status" )
         iOpcodeId = 6;
      else if ( oOpcode == "MethodResult" )
         iOpcodeId = 7;
      else if ( oOpcode == "Error" )
         iOpcodeId = 8;
      else if ( oOpcode == "AbortResult" )
         iOpcodeId = 9;
      else if ( oOpcode == "MethodResultFirst" )
         iOpcodeId = 10;
      else if ( oOpcode == "MethodResultMiddle" )
         iOpcodeId = 11;
      else if ( oOpcode == "MethodResultLast" )
         iOpcodeId = 12;
      else if ( oOpcode == "Subscribe" )
         iOpcodeId = 13;
      else if ( oOpcode == "Unsubscribe" )
         iOpcodeId = 14;
      else if ( oOpcode == "Increment" )
         iOpcodeId = 16;
      else if ( oOpcode == "Decrement" )
         iOpcodeId = 17;
      else if ( oOpcode == "PureSet" )
         iOpcodeId = 18;
   }

   if ( iOpcodeId < 0 )
   {
      string oMsgName = poMsgNode->oGetName();
      oMsgName = oMsgName.substr( 0, oMsgName.find( '.' ) );
      cerr << "unknown OpCode " << oOpcode << " in " << oMsgName << endl;
   }

   return iOpcodeId;
}

bool bGenerateMsgToken( tclFINode* poMsgNode, ostream &out, tclFIMessageNodePool* poMsgPool )
{
   bool bSuccess = true;
   // define token for this message (used in visitors)
   tclFINode* poMsgMainNode = poMsgPool->poGetNode( poMsgNode->oGetName().substr( 0, poMsgNode->oGetName().find( '.' ) ).c_str() );
   int iFuncId = strtol( poMsgMainNode->oGetDefaultValue().c_str(), 0, 0 );

   int iOpcodeId = iGetOpCodeId( poMsgNode );
   if ( (iOpcodeId >= 0) && (iFuncId > 0) )
   {
      string oTokenName = bMixedCaseTokens ? oRemove( poMsgNode->oGetName(), '.' ) : oToUpper( oRemove( poMsgNode->oGetName(), '.' ) );
      out << TAB << "EN_MSG_" << oTokenName << " = 0x"
         << hex << iFuncId << ((iOpcodeId > 15) ? "" : "0") << iOpcodeId << "00," << dec << endl;
   }
   else
   {
      cerr << "incorrect opcode or function id!" << endl;
      bSuccess = false;
   }
   return bSuccess;
}

bool bGenerateMsgTokenCnvt( tclFINode* poMsgNode, ostream &out, tclFIServiceTree *poServiceTree )
{
   tclFIMessageNodePool* poMsgPool = poServiceTree->poGetMessagePool();
   bool bSuccess = true;
   string oPrefix = oToLower( oGetFIPrefix( poServiceTree ) );
   // define token for this message (used in visitors)
   tclFINode* poMsgMainNode = poMsgPool->poGetNode( poMsgNode->oGetName().substr( 0, poMsgNode->oGetName().find( '.' ) ).c_str() );
   int iFuncId = strtol( poMsgMainNode->oGetDefaultValue().c_str(), 0, 0 );
   int iOpcodeId = iGetOpCodeId( poMsgNode );

   if ( (iOpcodeId >= 0) && (iFuncId > 0) )
   {
      string oTokenName = bMixedCaseTokens ? poMsgNode->oGetName() : oToUpper( poMsgNode->oGetName() );
#if 1
      out << TAB << "case " << oPrefix << "_tclToken::EN_MSG_" << oRemove( oTokenName, '.' ) << ":" << endl;
      out << TAB << TAB << "return \"" << poMsgNode->oGetName() << "\";" << endl;
#else
      out << TAB << "if (token == " << oPrefix << "_tclToken::EN_MSG_" << oRemove( oTokenName, '.' ) << ")" << endl;
      out << TAB << TAB << "return \"" << poMsgNode->oGetName() << "\";" << endl;
#endif
   }
   else
   {
      cerr << "incorrect opcode or function id!" << endl;
      bSuccess = false;
   }
   return bSuccess;
}

bool bGenerateTypeExpands( tclFINode *poType, ostream &out, tclFIServiceTree *poServiceTree )
{
   string oPrefix = oToLower( oGetFIPrefix( poServiceTree ) );
   unsigned int u32UnionIdx = 0;

   if ( bIsBinContainerType( poType->oGetName() ) )
   {
      out << "#ifndef FI_S_EXTERNAL_FI" << endl;
   }

   if ( poType->enGetType() == EN_OBJTYPE_STRUCTURE )
   {
      tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
      tclFITypeDependencyListIterator iter;
      for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
      {
         if ( (*iter)->enGetClass() == EN_DEPENDENCY_CLASS_UNION )
         {
            out << TAB << "case " << oPrefix << oGetTokenName( poType->oGetName() )
               << "UNION" << u32UnionIdx << ":" << endl;
            out << TAB << "{" << endl;
            out << TAB << TAB << "(tVoid) oPath.bGoUp();" << endl;
            out << TAB << TAB << "(tVoid) oCallStack.bGoUp();" << endl;
            tclFINode* poUnionType = (*iter)->poGetTarget();

            tclFITypeDependencyList* poDepList2 = poUnionType->poGetDependencyList();
            tclFITypeDependencyListIterator iter2;
            // test, whether enumeration has version info at all
            bool bHasVersionInfo = false;
            for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
            {
               if ( (*iter2)->bHasVersionInformation() )
               {
                  bHasVersionInfo = true;
               }
            }
            switch ( poUnionType->enGetType() )
            {
            case EN_OBJTYPE_ENUMERATION:
            {
               out << TAB << TAB << oGetClassName( poUnionType->oGetName() )
                  << " oDataType;" << endl;
               out << TAB << TAB << "oCallStack.bGoDown(" << oPrefix
                  << "_tclKnot(" << oPrefix << oGetTokenName( poUnionType->oGetName() ) << ","
                  << oPrefix << oGetSemanticName( "DATA_TYPE" ) << "));" << endl;
               out << TAB << TAB << "poVisitor->oIterator = oCallStack.oGetIterator();" << endl;
               out << TAB << TAB << "(tVoid) bDoIt(oDataType);" << endl;
               out << TAB << TAB << "(tVoid) oCallStack.bGoUp();" << endl;

               // create fallbacks for enumeration values not valid in the current version
               // fallback value must always be "T_Reserved"!!!!
               if ( bHasVersionInfo )
               {
                  for ( iter2 = poDepList2->end(); iter2 != poDepList2->begin(); )
                  {
                     --iter2;
                     if ( (*iter2)->bHasVersionInformation() )
                     {
                        out << TAB << TAB << "if ((oDataType.enType == "
                           << oGetClassName( poUnionType->oGetName() ) << "::"
                           << oGetEnumName( (*iter2)->oGetName() ) << ") &&" << endl;
                        if ( (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
                        {
                           if ( (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
                           {
                              out << TAB << TAB << "    !(("
                                 << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                 << " <= u16MajorVersion) && "
                                 << "(u16MajorVersion <= "
                                 << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                                 << ")))" << endl;
                           }
                           else
                           {
                              out << TAB << TAB << "    !("
                                 << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                 << " <= u16MajorVersion))" << endl;
                           }
                        }
                        else
                        {
                           out << TAB << TAB << "    !(u16MajorVersion <= "
                              << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                              << "))" << endl;
                        }
                        out << TAB << TAB << "{" << endl;
                        out << TAB << TAB << TAB << "oDataType.enType = "
                           << oGetClassName( poUnionType->oGetName() )
                           << "::" << oGetEnumName( (*iter2)->oGetFallbackValue() ) << ";" << endl;
                        out << TAB << TAB << "}" << endl;
                     }
                  }
               }

               out << TAB << TAB << "switch (oDataType.enType)" << endl;
               out << TAB << TAB << "{" << endl;
               tclFITypeDependencyList* poDepList2 = poUnionType->poGetDependencyList();
               tclFITypeDependencyListIterator iter2;
               for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
               {
                  out << TAB << TAB << "case "
                     << oGetClassName( poUnionType->oGetName() ) << "::"
                     << oGetEnumName( (*iter2)->oGetName() ) << ":" << endl;
                  if ( bIsBinContainerType( (*iter2)->oGetName() ) )
                  {
                     // For safety reason, hide bincontainer structure also internally!
                     out << TAB << TAB << TAB << "#ifndef FI_S_EXTERNAL_FI" << endl;
                     out << TAB << TAB << TAB << "if (bUseSpecialBinContainer)" << endl;
                     out << TAB << TAB << TAB << "{" << endl;
                     out << TAB << TAB << TAB << TAB << "(tVoid) oPath.bGoDown(" << oPrefix << "_tclKnot("
                        << oPrefix << oGetTokenName( (*iter2)->oGetName() ) << ","
                        << oPrefix << oGetSemanticName( (*iter)->oGetName() ) << "));" << endl;
                     out << TAB << TAB << TAB << "}" << endl;
                     out << TAB << TAB << TAB << "else" << endl;
                     out << TAB << TAB << TAB << "#endif // FI_S_EXTERNAL_FI" << endl;
                     out << TAB << TAB << TAB << "{" << endl;
                     out << TAB << TAB << TAB << TAB << "(tVoid) oPath.bGoDown(" << oPrefix << "_tclKnot("
                        << oPrefix << oGetTokenName( "T_BINCONTAINER" ) << ","
                        << oPrefix << oGetSemanticName( (*iter)->oGetName() ) << "));" << endl;
                     out << TAB << TAB << TAB << "}" << endl;
                  }
                  else
                  {
                     out << TAB << TAB << TAB << "(tVoid) oPath.bGoDown(" << oPrefix << "_tclKnot("
                        << oPrefix << oGetTokenName( (*iter2)->oGetName() ) << ","
                        << oPrefix << oGetSemanticName( (*iter)->oGetName() ) << "));" << endl;
                  }
                  out << TAB << TAB << TAB << "break;" << endl;
               }
               out << TAB << TAB << "}" << endl;
            } break;
            case EN_OBJTYPE_BITFIELD:
            {
               out << TAB << TAB << oGetClassName( poUnionType->oGetName() ) << " oDataMask;" << endl;
               out << TAB << TAB << "oCallStack.bGoDown(" << oPrefix << "_tclKnot("
                  << oPrefix << oGetTokenName( poUnionType->oGetName() ) << ","
                  << oPrefix << oGetSemanticName( "Data_Mask" ) << "));" << endl;
               out << TAB << TAB << "poVisitor->oIterator = oCallStack.oGetIterator();" << endl;
               out << TAB << TAB << "(tVoid) bDoIt(oDataMask);" << endl;
               out << TAB << TAB << "(tVoid) oCallStack.bGoUp();" << endl;
               int iBitIdx = 0;
               tclFITypeDependencyList* poDepList2 = poUnionType->poGetDependencyList();
               tclFITypeDependencyListIterator iter2;
               int iOptionIdx = 0;
               for ( iBitIdx = 0; iBitIdx < 32; ++iBitIdx )
               {
                  for ( iter2 = poDepList2->end(); iter2 != poDepList2->begin(); )
                  {
                     --iter2;
                     if ( strtoul( (*iter2)->oGetDefaultValue().c_str(), 0, 0 ) == iBitIdx )
                     {
                        if ( (*iter2)->bHasVersionInformation() )
                        {
                           out << TAB << TAB << "if ((oDataMask." << oGetDataFieldName( (*iter2)->poGetTarget() ) << " & "
                              << oGetClassName( poUnionType->oGetName() ) << "::FI_C_"
                              << oToUpper( (*iter2)->poGetTarget()->oGetName().substr( 1 ) ) << "_BIT_"
                              << oToUpper( (*iter2)->oGetName() ) << ") &&" << endl;
                           if ( (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
                           {
                              if ( (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
                              {
                                 out << TAB << TAB << "    (("
                                    << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                    << " <= u16MajorVersion) && "
                                    << "(u16MajorVersion <= "
                                    << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                                    << ")))" << endl;
                              }
                              else
                              {
                                 out << TAB << TAB << "    ("
                                    << (*iter2)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                                    << " <= u16MajorVersion))" << endl;
                              }
                           }
                           else
                           {
                              out << TAB << TAB << "    (u16MajorVersion <= "
                                 << (*iter2)->poGetVersionRange()->oGetMaxVersion().iGetMajor()
                                 << "))" << endl;
                           }
                        }
                        else
                        {
                           out << TAB << TAB << "if (oDataMask." << oGetDataFieldName( (*iter2)->poGetTarget() ) << " & "
                              << oGetClassName( poUnionType->oGetName() ) << "::FI_C_"
                              << oToUpper( (*iter2)->poGetTarget()->oGetName().substr( 1 ) ) << "_BIT_"
                              << oToUpper( (*iter2)->oGetName() ) << ")" << endl;
                        }
                        out << TAB << TAB << "{" << endl;
                        if ( bIsBinContainerType( (*iter2)->oGetName() ) )
                        {
                           // For safety reason, hide bincontainer structure also internally!
                           out << TAB << TAB << TAB << "#ifndef FI_S_EXTERNAL_FI" << endl;
                           out << TAB << TAB << TAB << "if (bUseSpecialBinContainer)" << endl;
                           out << TAB << TAB << TAB << "{" << endl;
                           out << TAB << TAB << TAB << TAB << "(tVoid) oPath.bGoDown(" << oPrefix
                              << "_tclKnot(" << oPrefix << oGetTokenName( (*iter2)->oGetName() )
                              << "," << oPrefix << oGetSemanticName( (*iter)->oGetName() ) << "));" << endl;
                           out << TAB << TAB << TAB << "}" << endl;
                           out << TAB << TAB << TAB << "else" << endl;
                           out << TAB << TAB << TAB << "#endif // FI_S_EXTERNAL_FI" << endl;
                           out << TAB << TAB << TAB << "{" << endl;
                           out << TAB << TAB << TAB << TAB << "(tVoid) oPath.bGoDown(" << oPrefix
                              << "_tclKnot(" << oPrefix << oGetTokenName( "T_BINCONTAINER" )
                              << "," << oPrefix << oGetSemanticName( (*iter)->oGetName() ) << "));" << endl;
                           out << TAB << TAB << TAB << "}" << endl;

                        }
                        else
                        {
                           out << TAB << TAB << TAB << "(tVoid) oPath.bGoDown(" << oPrefix
                              << "_tclKnot(" << oPrefix << oGetTokenName( (*iter2)->oGetName() )
                              << "," << oPrefix << oGetSemanticName( (*iter)->oGetName() ) << "));" << endl;
                        }
                        out << TAB << TAB << "}" << endl;
                        iOptionIdx++;
                     }
                  }
               }
            } break;
            }
            out << TAB << "} break;" << endl;
            ++u32UnionIdx;
         }
      }
   }
   out << TAB << TAB << "case " << oPrefix << oGetTokenName( poType->oGetName() ) << ":" << endl;
   out << TAB << TAB << "{" << endl;
   // u32UnionIdx = 0; resetting UnionIdx as before is wrong, since the following putting on the stack has to be done the other way round.
   if ( poType->enGetType() == EN_OBJTYPE_STRUCTURE )
   {
      tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
      if ( poDepList->size() > 0 )
      {
         out << TAB << TAB << "if (poVisitor->bDoExpand" << poType->oGetName().substr( 1 )
             << "(enSemanticDomain))" << endl;
         out << TAB << TAB << "{" << endl;
         out << TAB << TAB << TAB << "(tVoid) oPath.bGoUp();" << endl;
         if ( bIsBinContainerType( poType->oGetName() ) )
         {
            out << TAB << TAB << TAB << "bInsideBinContainerVersion = true;" << endl;
            out << TAB << TAB << TAB << "(tVoid) oPath.bGoDown(" << oPrefix << "_tclKnot("
                << oPrefix << oGetTokenName( "__End_Of_Struct" ) << "," << oPrefix << oGetSemanticName( "BINCONTAINER" ) << "));" << endl;
         }
         else
         {
            out << TAB << TAB << TAB << "(tVoid) oPath.bGoDown(" << oPrefix << "_tclKnot("
                << oPrefix << oGetTokenName( "__END_OF_STRUCT" ) << "," << oPrefix << oGetSemanticName( "UNKNOWN" ) << "));" << endl;
         }
         tclFITypeDependencyListIterator iter;
         int iListSize = poDepList->size();
         for ( iter = poDepList->end(); iter != poDepList->begin(); )
         {
            string oIndent = string( TAB ) + TAB + TAB;
            --iter;
            if ( (*iter)->bHasVersionInformation() )
            {
               if ( (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
               {
                  if ( (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
                  {
                     out << oIndent << "if ((" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                        << " <= u16MajorVersion) && (u16MajorVersion <= "
                        << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << "))" << endl;
                  }
                  else
                  {
                     out << oIndent << "if (" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                        << " <= u16MajorVersion)" << endl;
                  }
               }
               else
               {
                  out << oIndent << "if (u16MajorVersion <= "
                     << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << ")" << endl;
               }
               out << oIndent << "{" << endl;
               oIndent += TAB;
            }
            switch ( (*iter)->enGetClass() )
            {
            case EN_DEPENDENCY_CLASS_FIELD:
            {
               if ( bIsBinContainerType( (*iter)->poGetTarget()->oGetName() ) )
               {
                  // For safety reason, hide bincontainer structure also internally!
                  /*
                  out << oIndent << "#ifndef FI_S_EXTERNAL_FI" << endl;
                  out << oIndent << "oPath.bGoDown(" << oPrefix << "_tclKnot("
                  << oPrefix << oGetTokenName((*iter)->poGetTarget()->oGetName()) << ","
                  << oPrefix << oGetSemanticName((*iter)->oGetName()) << "));" << endl;

                  out << oIndent << "#else" << endl;*/
                  out << oIndent << "(tVoid) oPath.bGoDown(" << oPrefix << "_tclKnot("
                     << oPrefix << oGetTokenName( "T_BINCONTAINER" ) << ","
                     << oPrefix << oGetSemanticName( (*iter)->oGetName() ) << "));" << endl;

                  // out << oIndent << "#endif // FI_S_EXTERNAL_FI" << endl;
               }
               else
               {
                  tclFINode* poToken = (*iter)->poGetTarget();
                  string oTokenName = poToken->oGetName();
                  if ( bIsPredefinedType( poToken ) )
                     oTokenName = oTokenName[0] + oToUpper( oTokenName.substr( 1 ) );
                  out << oIndent << "(tVoid) oPath.bGoDown(" << oPrefix << "_tclKnot("
                      << oPrefix << oGetTokenName( oTokenName ) << ","
                      << oPrefix << oGetSemanticName( (*iter)->oGetName() ) << "));" << endl;
               }
            } break;
            case EN_DEPENDENCY_CLASS_LIST:
            {
               if ( bIsBinContainerType( (*iter)->poGetTarget()->oGetName() ) )
               {
                  // For safety reason, hide bincontainer structure also internally!
                  /*
                  out << oIndent << "#ifndef FI_S_EXTERNAL_FI" << endl;
                  out << oIndent << "oPath.bGoDown(" << oPrefix << "_tclKnot("
                  << oPrefix << oGetTokenName((*iter)->poGetTarget()->oGetName()) << ","
                  << oPrefix << oGetSemanticName("LIST_ELEMENT") << ",0));" << endl;

                  out << oIndent << "#else" << endl;*/
                  out << oIndent << "(tVoid) oPath.bGoDown(" << oPrefix << "_tclKnot("
                      << oPrefix << oGetTokenName( "T_BINCONTAINER" ) << ","
                      << oPrefix << oGetSemanticName( "LIST_ELEMENT" ) << ",0));" << endl;
                  // out << oIndent << "#endif // FI_S_EXTERNAL_FI" << endl;
               }
               else
               {
                  out << oIndent << "(tVoid) oPath.bGoDown(" << oPrefix << "_tclKnot("
                      << oPrefix << oGetTokenName( (*iter)->poGetTarget()->oGetName() ) << ","
                      << oPrefix << oGetSemanticName( "LIST_ELEMENT" ) << ",0));" << endl;
               }
               out << oIndent << "(tVoid) oPath.bGoDown(" << oPrefix << "_tclKnot("
                   << oPrefix << oGetTokenName( "__List" ) << ","
                   << oPrefix << oGetSemanticName( (*iter)->oGetName() ) << "));" << endl;
            } break;
            case EN_DEPENDENCY_CLASS_UNION:
            {
               u32UnionIdx--;
               out << oIndent << "(tVoid) oPath.bGoDown(" << oPrefix << "_tclKnot("
                   << oPrefix << oGetTokenName( poType->oGetName()/*.substr(2)*/ ) << "UNION" << u32UnionIdx << ","
                   << oPrefix << oGetSemanticName( (*iter)->oGetName() ) << "));" << endl;
            } break;
            }
            if ( (*iter)->bHasVersionInformation() )
            {
               oIndent = oIndent.substr( strlen( TAB ) );
               out << oIndent << "}" << endl;
            }

         }
         string oDataFieldName = "o" + poType->oGetName().substr( 1 );
         out << TAB << TAB << "}" << endl;
         out << TAB << TAB << "else" << endl;
         out << TAB << TAB << "{" << endl;
         out << TAB << TAB << TAB << oGetClassName( poType->oGetName() ) << " " << oDataFieldName << ";" << endl;
         out << TAB << TAB << TAB << "if (!bDoIt(" << oDataFieldName << "))" << endl;
         out << TAB << TAB << TAB << "{" << endl;
         if ( bEnableDeepCopy )
            out << "#ifndef VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;

         out << TAB << TAB << TAB << TAB << oDataFieldName << ".vDestroy();" << endl;
         if ( bEnableDeepCopy )
            out << "#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;

         out << TAB << TAB << TAB << TAB << "return FALSE;" << endl;
         out << TAB << TAB << TAB << "}" << endl;
         out << TAB << TAB << TAB << "else" << endl;
         out << TAB << TAB << TAB << "{" << endl;
         if ( bEnableDeepCopy )
            out << "#ifndef VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;
         out << TAB << TAB << TAB << TAB << oDataFieldName << ".vDestroy();" << endl;
         if ( bEnableDeepCopy )
            out << "#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY" << endl;
         out << TAB << TAB << TAB << TAB << "(tVoid) oCallStack.bGoUp();" << endl;
         out << TAB << TAB << TAB << TAB << "(tVoid) oPath.bGoUp();" << endl;
         out << TAB << TAB << TAB << "}" << endl;
         out << TAB << TAB << "}" << endl;
      }
      else
      {
         out << TAB << TAB << "(tVoid) oCallStack.bGoUp();" << endl;
         out << TAB << TAB << "(tVoid) oPath.bGoUp();" << endl;
      }
   }
   else
   {
      out << TAB << TAB << TAB << oGetClassName( poType->oGetName() )
         << " o" << poType->oGetName().substr( 1 ) << ";" << endl;
      out << TAB << TAB << TAB << "if (!bDoIt(o" << poType->oGetName().substr( 1 ) << "))" << endl;
      out << TAB << TAB << TAB << TAB << "return FALSE;" << endl;
      out << TAB << TAB << TAB << "(tVoid) oCallStack.bGoUp();" << endl;
      out << TAB << TAB << TAB << "(tVoid) oPath.bGoUp();" << endl;
   }
   out << TAB << TAB << "} break;" << endl;

   if ( bIsBinContainerType( poType->oGetName() ) )
   {
      out << "#endif // FI_S_EXTERNAL_FI" << endl;
   }

   return true;
}

bool bGenerateVisitor( tclFIServiceTree* poServiceTree, const string &oVisitorName, set<string> oTypeSet )
{
   string oOutDir = oGetServiceTreeSubDirectory( poServiceTree ) + "/";
   ofstream oOutDecl( string( oOutDir + oToLower( oVisitorName ) + "decl.inc" ).c_str() );
   if ( !oOutDecl.good() )
      return false;

   ofstream oOutDef( string( oOutDir + oToLower( oVisitorName ) + ".inc" ).c_str() );
   if ( !oOutDef.good() )
      return false;

   string oPrefix = oToLower( oGetFIPrefix( poServiceTree ) );
   //   string oPrefix = oGetFIGroup() + oGetFIPrefix();

   tclFINodePool* poTypePool = poServiceTree->poGetUsedGlobalTypePool();
   for ( int iTypeIdx = 0; iTypeIdx < poTypePool->iGetNumNodes(); ++iTypeIdx )
   {
      tclFINode* poType = poTypePool->poGetNode( iTypeIdx, EN_NONE );
      if ( bIsPredefinedType( poType ) )
         continue;
      if ( bUseAllTypes || (oTypeSet.find( poType->oGetName() ) != oTypeSet.end()) )
      {
         string oTypeName = poType->oGetName().substr( 1 );
         switch ( poType->enGetType() )
         {
         case EN_OBJTYPE_TYPE:
         case EN_OBJTYPE_ENUMERATION:
         case EN_OBJTYPE_BITFIELD:
            oOutDecl << TAB << "virtual tVoid vVisit" << oTypeName << "("
               << oGetClassName( poType->oGetName() ) << "& o" << oTypeName << ");" << endl;
            break;
         case EN_OBJTYPE_STRUCTURE:
            if ( bIsBinContainerType( poType->oGetName() ) )
            {
               oOutDecl << "#ifndef FI_S_EXTERNAL_FI" << endl;
            }
            if ( (oVisitorName == "ReadInVisitor") || (oVisitorName == "PrintVisitor") )
            {
               oOutDecl << TAB << "virtual tBool bDoExpand" << oTypeName << "("
                  << oPrefix << "_tclSemanticDomain::tenType enSemantic);" << endl;
            }
            if ( bIsBinContainerType( poType->oGetName() ) )
            {
               oOutDecl << "#endif // FI_S_EXTERNAL_FI" << endl;
            }
         }
         switch ( poType->enGetType() )
         {
         case EN_OBJTYPE_ATOM:
            break;
         case EN_OBJTYPE_TYPE:
         case EN_OBJTYPE_ENUMERATION:
         case EN_OBJTYPE_BITFIELD:
            oOutDef << "tVoid " << oPrefix << "_tcl" << oVisitorName << "::vVisit"
               << oTypeName << "(" << oGetClassName( poType->oGetName() ) << "& o"
               << oTypeName << ")" << endl;
            oOutDef << "{" << endl;
            if ( oVisitorName == "PrintVisitor" )
            {
               oOutDef << TAB << oPrefix << "_tclSemanticDomain::tenType "
                  << "enSemanticDomain = oIterator.oKnot().enSemanticDomain;" << endl;
               oOutDef << TAB << "tU32 u32IndentSize = oIterator.u32Position();" << endl;
               oOutDef << TAB << "tS32 s32LineLength = 0;" << endl;
               oOutDef << TAB << "if (enSemanticDomain == " << oPrefix
                  << oGetSemanticName( "LIST_ELEMENT" ) << ")" << endl;
               oOutDef << TAB << "{" << endl;
               oOutDef << TAB << TAB << "s32LineLength = OSAL_s32PrintFormat(line,\"%*s"
                  << poType->oGetName() << "[%lu] = \",(int)u32IndentSize*2,\"\",oIterator.oKnot().u32Index);" << endl;
               oOutDef << TAB << "}" << endl;
               oOutDef << TAB << "else" << endl;
               oOutDef << TAB << "{" << endl;
               oOutDef << TAB << TAB << "s32LineLength = OSAL_s32PrintFormat(line,\"%*s"
                  << poType->oGetName() << " %s = \",(int)u32IndentSize*2,\"\",convertSemToken2Text(enSemanticDomain));" << endl;
               oOutDef << TAB << "}" << endl;
               if ( poType->enGetType() == EN_OBJTYPE_TYPE )
               {
                  string oBaseTypeName = (*poType->poGetDependencyList()->begin())->poGetTarget()->oGetName();
                  if ( (oBaseTypeName == "tString") ||
                      (oBaseTypeName == "tMultiLanguageString") )
                  {
                     oOutDef << TAB << "s32LineLength += OSAL_s32PrintFormat(line+s32LineLength,\"%s\\r\\n\",o"
                        << oTypeName << "." << oGetDataFieldName( poType ) << ".szValue);" << endl;
                  }
                  else
                  {
                     if ( (oBaseTypeName.substr( 0, 2 ) == "tU") ||
                        (oBaseTypeName.substr( 0, 2 ) == "tS") )
                     {
                        if ( oBaseTypeName == "tU32" )
                        {
                           oOutDef << TAB << "s32LineLength += OSAL_s32PrintFormat(line+s32LineLength,\"%lu\\r\\n\",o"
                              << oTypeName << "." << oGetDataFieldName( poType ) << ");" << endl;
                        }
                        else
                        {
                           if ( (oBaseTypeName == "tU24") ||
                               (oBaseTypeName == "tS24") )
                           {
                              oOutDef << TAB << "s32LineLength += OSAL_s32PrintFormat(line+s32LineLength,\"%d\\r\\n\",o"
                                 << oTypeName << "." << oGetDataFieldName( poType ) << "." << oGetDataFieldName( poType ) << ");" << endl;
                           }
                           else
                           {
                              oOutDef << TAB << "s32LineLength += OSAL_s32PrintFormat(line+s32LineLength,\"%d\\r\\n\",o"
                                 << oTypeName << "." << oGetDataFieldName( poType ) << ");" << endl;
                           }
                        }
                     }
                     else
                     {
                        if ( oBaseTypeName == "tBool" )
                        {
                           oOutDef << TAB << "s32LineLength += OSAL_s32PrintFormat(line+s32LineLength,\"%s\\r\\n\",((o"
                              << oTypeName << "." << oGetDataFieldName( poType ) << ")?\"TRUE\":\"FALSE\"));" << endl;
                        }
                        else
                        {
                           if ( oBaseTypeName.substr( 0, 2 ) == "tF" )
                           {
                              oOutDef << TAB << "s32LineLength += OSAL_s32PrintFormat(line+s32LineLength,\"%g\\r\\n\",o"
                                 << oTypeName << "." << oGetDataFieldName( poType ) << ");" << endl;
                           }
                           else
                           {
                              // oOutDef << "### unknown base type !!! FI-Generation failed" << endl;
                              // return false;
                           }
                        }
                     }
                  }
               }
               if ( poType->enGetType() == EN_OBJTYPE_ENUMERATION )
               {
                  string oBaseTypeName = (*poType->poGetDependencyList()->begin())->poGetTarget()->oGetName();
                  if ( oBaseTypeName == "tU32" )
                  {
                     oOutDef << TAB << "s32LineLength += OSAL_s32PrintFormat("
                        << "line+s32LineLength,\"%lu \",o" << oTypeName << "."
                        << oGetDataFieldName( poType ) << ");" << endl;
                  }
                  else
                  {
                     oOutDef << TAB << "s32LineLength += OSAL_s32PrintFormat("
                        << "line+s32LineLength,\"%d \",o" << oTypeName << "."
                        << oGetDataFieldName( poType ) << ");" << endl;
                  }
                  oOutDef << TAB << "switch (o" << oTypeName << "."
                     << oGetDataFieldName( poType ) << ")" << endl;
                  oOutDef << TAB << "{" << endl;

                  tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
                  tclFITypeDependencyListIterator iter;
                  for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
                  {
                     oOutDef << TAB << "case " << oGetClassName( poType->oGetName() )
                        << "::" << oGetEnumName( (*iter)->oGetName() ) << ":" << endl;
                     oOutDef << TAB << TAB << "s32LineLength += OSAL_s32PrintFormat("
                        << "line+s32LineLength,\" -> " << (*iter)->oGetName()
                        << "\\r\\n\");" << endl;
                     oOutDef << TAB << TAB << "break;" << endl;
                  }
                  oOutDef << TAB << "default:" << endl;
                  oOutDef << TAB << TAB << "s32LineLength += OSAL_s32PrintFormat("
                     << "line+s32LineLength,\"(invalid enum value!)\\r\\n\");" << endl;
                  oOutDef << TAB << "}" << endl;
               }
               if ( poType->enGetType() == EN_OBJTYPE_BITFIELD )
               {
                  string oBaseTypeName = (*poType->poGetDependencyList()->begin())->poGetTarget()->oGetName();
                  if ( oBaseTypeName == "tU32" )
                  {
                     oOutDef << TAB << "s32LineLength += OSAL_s32PrintFormat("
                        << "line+s32LineLength,\"0x%lX\\r\\n\",o"
                        << oTypeName << "." << oGetDataFieldName( poType ) << ");" << endl;
                  }
                  else
                  {
                     oOutDef << TAB << "s32LineLength += OSAL_s32PrintFormat("
                        << "line+s32LineLength,\"0x%X\\r\\n\",o"
                        << oTypeName << "." << oGetDataFieldName( poType ) << ");" << endl;
                  }
               }
               oOutDef << TAB << "poOutput->vPrintLine(line);" << endl;
            }
            if ( oVisitorName == "RandomInitVisitor" )
            {
               if ( (poType->enGetType() == EN_OBJTYPE_TYPE) ||
                   (poType->enGetType() == EN_OBJTYPE_BITFIELD) )
               {
                  oOutDef << TAB << "o" << oTypeName << "." << oGetDataFieldName( poType ) << " = ";
                  if ( ((*poType->poGetDependencyList()->begin())->poGetTarget()->oGetName() == "tString") ||
                      ((*poType->poGetDependencyList()->begin())->poGetTarget()->oGetName() == "tMultiLanguageString") )
                  {
                     oOutDef << "\"abc\";" << endl;
                  }
                  else
                  {
                     oOutDef << "(" << (*poType->poGetDependencyList()->begin())->poGetTarget()->oGetName() << ")rand();" << endl;
                  }
               }
               if ( poType->enGetType() == EN_OBJTYPE_ENUMERATION )
               {
                  tclFITypeDependencyList* poDepList2 = poType->poGetDependencyList();
                  oOutDef << TAB << "switch(rand() % " << poDepList2->size() << ")" << endl;
                  oOutDef << TAB << "{" << endl;
                  tclFITypeDependencyListIterator iter2;
                  int s32Idx = 0;
                  for ( iter2 = poDepList2->begin(); iter2 != poDepList2->end(); ++iter2 )
                  {
                     oOutDef << TAB << "case " << s32Idx << ":" << endl;
                     oOutDef << TAB << TAB << "o" << oTypeName << "."
                        << oGetDataFieldName( poType ) << " = "
                        << oGetClassName( poType->oGetName() ) << "::"
                        << oGetEnumName( (*iter2)->oGetName() ) << ";" << endl;
                     oOutDef << TAB << TAB << "break;" << endl;
                     ++s32Idx;
                  }
                  oOutDef << TAB << "default:" << endl;
                  oOutDef << TAB << TAB << "o" << oTypeName << "."
                     << oGetDataFieldName( poType ) << " = "
                     << oGetClassName( poType->oGetName() ) << "::"
                     << oGetEnumName( (*(poDepList2->begin()))->oGetName() ) << ";" << endl;
                  oOutDef << TAB << "}" << endl;
               }
            }
            if ( oVisitorName == "ReadInVisitor" )
            {
               oOutDef << TAB << "const char* line = poIn->GetLine(false);" << endl;
               oOutDef << TAB << "char* ps8Start = strchr(line,'=') + 1;" << endl;
               if ( ((*poType->poGetDependencyList()->begin())->poGetTarget()->oGetName() == "tString") ||
                   ((*poType->poGetDependencyList()->begin())->poGetTarget()->oGetName() == "tMultiLanguageString") )
               {
                  oOutDef << TAB << "if (sscanf(ps8Start,\"%s\", &o" << oTypeName
                     << "." << oGetDataFieldName( poType ) << ".szValue)!=1)" << endl;
               }
               else
               {
                  oOutDef << TAB << "if (sscanf(ps8Start,\"%d\", &o" << oTypeName
                     << "." << oGetDataFieldName( poType ) << ")!=1)" << endl;
               }
               oOutDef << TAB << "{" << endl;
               oOutDef << TAB << TAB << "vInvalidate();" << endl;
               oOutDef << TAB << "}" << endl;
            }
            oOutDef << "}" << endl;
            break;
         case EN_OBJTYPE_STRUCTURE:
            if ( bIsBinContainerType( poType->oGetName() ) )
            {
               oOutDef << "#ifndef FI_S_EXTERNAL_FI" << endl;
            }
            if ( oVisitorName == "PrintVisitor" )
            {
               oOutDef << "tBool " << oPrefix << "_tcl" << oVisitorName << "::bDoExpand"
                  << oTypeName << "(" << oPrefix
                  << "_tclSemanticDomain::tenType enSemanticDomain)" << endl;
               oOutDef << "{" << endl;
               oOutDef << TAB << "if (oIterator.oKnot().enSemanticDomain == "
                  << oPrefix << oGetSemanticName( "LIST_ELEMENT" ) << ")" << endl;
               oOutDef << TAB << "{" << endl;
               oOutDef << TAB << TAB << "(tVoid) OSALUTIL_s32SaveNPrintFormat(line, sizeof(line), \"%*s"
                  << poType->oGetName() << "[%lu] :\\r\\n\",(int)oIterator."
                  << "u32Position()*2,\"\",oIterator.oKnot().u32Index);" << endl;
               //               oOutDef << TAB << TAB << "(tS32) OSAL_s32PrintFormat(line,\"%*s" 
               //                  << poType->oGetName() << "[%lu] :\\r\\n\",(int)oIterator."
               //                  << "u32Position()*2,\"\",oIterator.oKnot().u32Index);" << endl;
               oOutDef << TAB << "}" << endl;
               oOutDef << TAB << "else" << endl;
               oOutDef << TAB << "{" << endl;
               oOutDef << TAB << TAB << "(tVoid) OSALUTIL_s32SaveNPrintFormat(line, sizeof(line), \"%*s"
                  << poType->oGetName() << " %s :\\r\\n\",(int)oIterator.u32Position()*2"
                  << ",\"\",convertSemToken2Text(enSemanticDomain));" << endl;
               //               oOutDef << TAB << TAB << "(tS32) OSAL_s32PrintFormat(line,\"%*s" 
               //                  << poType->oGetName() << " %s :\\r\\n\",(int)oIterator.u32Position()*2"
               //                  << ",\"\",convertSemToken2Text(enSemanticDomain));" << endl;
               oOutDef << TAB << "}" << endl;
               oOutDef << TAB << "poOutput->vPrintLine(line);" << endl;
               oOutDef << TAB << "return TRUE;" << endl;
               oOutDef << "}" << endl;
            }
            if ( oVisitorName == "ReadInVisitor" )
            {
               oOutDef << "tBool " << oPrefix << "_tcl" << oVisitorName << "::bDoExpand"
                  << oTypeName << "(" << oPrefix << "_tclSemanticDomain::tenType "
                  << "/*enSemanticDomain*/)" << endl;
               oOutDef << "{" << endl;
               oOutDef << TAB << "const char* szLine = poIn->GetLine(true);" << endl;
               oOutDef << TAB << "return TRUE;" << endl;
               oOutDef << "}" << endl;
            }
            if ( bIsBinContainerType( poType->oGetName() ) )
            {
               oOutDef << "#endif // FI_S_EXTERNAL_FI" << endl;
            }
         }
      }
   }
   return true;
}

bool bGenerateMsgInit( tclFINode *poType, ostream &out, tclFIServiceTree* poServiceTree )
{
   string oPrefix = oToLower( oGetFIPrefix( poServiceTree ) );

   out << TAB << "case " << oPrefix
      << oGetTokenName( "__MSG_" + oRemove( poType->oGetName(), '.' ) ) << ":" << endl;
   out << TAB << "{" << endl;

   tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
   // iterate backwards through dependencies 
   tclFITypeDependencyListIterator iter = poDepList->end();
   while ( iter != poDepList->begin() )
   {
      --iter;
      string oIndent = string( TAB ) + TAB;
      if ( (*iter)->bHasVersionInformation() )
      {
         if ( (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor() > 0 )
         {
            if ( (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() < 65535 )
            {
               out << oIndent << "if ((" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                  << " <= poOutContext->u16GetMajorVersion()) && (poOutContext->u16GetMajorVersion() <= "
                  << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << "))" << endl;
            }
            else
            {
               out << oIndent << "if (" << (*iter)->poGetVersionRange()->oGetMinVersion().iGetMajor()
                  << " <= poOutContext->u16GetMajorVersion())" << endl;
            }
         }
         else
         {
            out << oIndent << "if (poOutContext->u16GetMajorVersion() <= "
               << (*iter)->poGetVersionRange()->oGetMaxVersion().iGetMajor() << ")" << endl;
         }
         out << oIndent << "{" << endl;
         oIndent += TAB;
      }

      switch ( (*iter)->enGetClass() )
      {
      case EN_DEPENDENCY_CLASS_PARAM:
         if ( bIsBinContainerType( (*iter)->poGetTarget()->oGetName() ) )
         {
            // For safety reason, hide bincontainer structure also internally!
            /*
            out << "#ifndef FI_S_EXTERNAL_FI" << endl;
            out << oIndent << "bResult &= oPath.bGoDown(" << oPrefix << "_tclKnot("
            << oPrefix << oGetTokenName((*iter)->poGetTarget()->oGetName()) << ","
            << oPrefix << oGetSemanticName("Msg_" + (*iter)->oGetName()) << "));" << endl;

            out << "#else" << endl;*/
            out << oIndent << "bResult = oPath.bGoDown(" << oPrefix << "_tclKnot("
               << oPrefix << oGetTokenName( "T_BINCONTAINER" ) << ","
               << oPrefix << oGetSemanticName( "Msg_" + (*iter)->oGetName() ) << ")) && bResult;" << endl;

            //out << "#endif // FI_S_EXTERNAL_FI" << endl;
         }
         else
         {
            tclFINode* poToken = (*iter)->poGetTarget();
            string oTokenName = poToken->oGetName();
            if ( bIsPredefinedType( poToken ) )
               oTokenName = oTokenName[0] + oToUpper( oTokenName.substr( 1 ) );
            out << oIndent << "bResult = oPath.bGoDown(" << oPrefix << "_tclKnot("
               << oPrefix << oGetTokenName( oTokenName ) << ","
               << oPrefix << oGetSemanticName( "Msg_" + (*iter)->oGetName() ) << ")) && bResult;" << endl;
            //            out << oIndent << "bResult = oPath.bGoDown(" << oPrefix << "_tclKnot("
            //               << oPrefix << oGetTokenName((*iter)->poGetTarget()->oGetName()) << ","
            //               << oPrefix << oGetSemanticName("Msg_" + (*iter)->oGetName()) << ")) && bResult;" << endl;
         }
         break;
      case EN_DEPENDENCY_CLASS_LIST:
         if ( bIsBinContainerType( (*iter)->poGetTarget()->oGetName() ) )
         {
            // For safety reason, hide bincontainer structure also internally!
            /*
            out << "#ifndef FI_S_EXTERNAL_FI" << endl;
            out << oIndent << "bResult &= oPath.bGoDown(" << oPrefix << "_tclKnot("
            << oPrefix << oGetTokenName((*iter)->poGetTarget()->oGetName()) << ","
            << oPrefix << oGetSemanticName("LIST_ELEMENT") << ",0));" << endl;

            out << "#else" << endl;*/
            out << oIndent << "bResult = oPath.bGoDown(" << oPrefix << "_tclKnot("
               << oPrefix << oGetTokenName( "T_BINCONTAINER" ) << ","
               << oPrefix << oGetSemanticName( "LIST_ELEMENT" ) << ",0)) && bResult;" << endl;

            //out << "#endif // FI_S_EXTERNAL_FI" << endl;
         }
         else
         {
            out << oIndent << "bResult = oPath.bGoDown(" << oPrefix << "_tclKnot("
               << oPrefix << oGetTokenName( (*iter)->poGetTarget()->oGetName() ) << ","
               << oPrefix << oGetSemanticName( "LIST_ELEMENT" ) << ",0)) && bResult;" << endl;
         }
         out << oIndent << "bResult = oPath.bGoDown(" << oPrefix << "_tclKnot("
            << oPrefix << oGetTokenName( "__List" ) << ","
            << oPrefix << oGetSemanticName( "Msg_" + (*iter)->oGetName() ) << ")) && bResult;" << endl;
         break;
      default:
         // unions are not allowed so far!
         return false;
      }
      if ( (*iter)->bHasVersionInformation() )
      {
         oIndent = oIndent.substr( strlen( TAB ) );
         out << oIndent << "}" << endl;
      }
   }
   out << TAB << "} break;" << endl;
   return true;
}

bool bGenerateMsgExpands( tclFINode *poType, ostream &out, tclFIServiceTree* poServiceTree )
{
   string oPrefix = oToLower( oGetFIPrefix( poServiceTree ) );
   out << TAB << TAB << "case " << oPrefix
      << oGetTokenName( "__MSG_" + oRemove( poType->oGetName(), '.' ) ) << ":" << endl;
   return true;
}

bool bGenerateMsgInfo( tclFIServiceTree* poServiceTree )
{
   bool bSuccess = true;

   oBaseMessageClassName = oToLower( oGetFIPrefix( poServiceTree ) ) + oGetClassName( "TMsgBaseMessage" ).substr( 2 );
   string oOutDir = oGetServiceTreeSubDirectory( poServiceTree );
   ofstream oOutMsgDecl( string( oOutDir + "/msgtypes_h.inc" ).c_str() );
   if ( !oOutMsgDecl.good() )
      return false;
   // oOutMsgDecl << "// New generation:" << endl;
   ofstream oOutMsgDef( string( oOutDir + "/msgtypes_c.inc" ).c_str() );
   if ( !oOutMsgDef.good() )
      return false;

   if ( poServiceTree->poGetMessagePool()->iGetNumNodes() != 0 )
   {
      bGenerateMsgBaseDecl( oOutMsgDecl, poServiceTree );
      bGenerateMsgInfo( NULL, oOutMsgDef, oToLower( oGetFIPrefix( poServiceTree ) ), poServiceTree );
      if ( bGenerateErrorMessages )
      {
         bGenerateMsgDefaultErr( oOutMsgDef, poServiceTree );
         bGenerateExtendedDefaultErrorMessageDecl( oOutMsgDecl, poServiceTree );
      }
   }

   ofstream oOutMsgInits( string( oOutDir + "/msginits.inc" ).c_str() );
   if ( !oOutMsgInits.good() )
      return false;
   ostringstream oMsgDefaultInits;  // add empty messages to switch defaults (Lint Info 788)

   ofstream oOutMsgToken( string( oOutDir + "/msgtoken.inc" ).c_str() );
   if ( !oOutMsgToken.good() )
      return false;
   ofstream oOutMsgIds( string( oOutDir + "/msgids.inc" ).c_str() );
   if ( !oOutMsgIds.good() )
      return false;
   ofstream oOutTypeTokenCnvt( string( oOutDir + "/typtokencnvt.inc" ).c_str() );
   if ( !oOutTypeTokenCnvt.good() )
      return false;
   ofstream oOutTypeExpands( string( oOutDir + "/typeexpand.inc" ).c_str() );
   if ( !oOutTypeExpands.good() )
      return false;

   set<string> oSemanticSet;

   oOutMsgIds << "//" << endl;
   oOutMsgIds << "// definition of function ids" << endl;
   oOutMsgIds << "//" << endl << endl;
   oOutMsgIds << "enum {" << endl;
   tclFIMessageNodePool* poMsgPool = poServiceTree->poGetMessagePool();
   for ( int iMsgIdx = 0; iMsgIdx < poMsgPool->iGetNumNodes(); ++iMsgIdx )
   {
      tclFINode* poMsgNode = poMsgPool->poGetNode( iMsgIdx, EN_NONE );
      if ( bBeVerbose )
      {
         cout << "Generate Info for '" << poMsgNode->oGetName() << "'" << endl;
      }

      // define constant for function id
      if ( (poMsgNode->enGetType() == EN_OBJTYPE_MESSAGE_METHOD) ||
          (poMsgNode->enGetType() == EN_OBJTYPE_MESSAGE_PROPERTY) )
      {
         oOutMsgIds << TAB << oToUpper( oGetFIPrefix( poServiceTree ) )
            << "_C_U16_" << oToUpper( poMsgNode->oGetName() ) << " = "
            << poMsgNode->oGetDefaultValue() << "," << endl;
      }

      if ( poMsgNode->enGetType() == EN_OBJTYPE_OPCODE )
      {
         if ( bGenerateErrorMessages || (poMsgNode->oGetName().find( ".Error" ) == string::npos) )
         {
            bGenerateTypeClassDeclaration( poMsgNode, oOutMsgDecl, poServiceTree );
            bGenerateTypeClassDefinition( poMsgNode, oOutMsgDef, poServiceTree );
            bGenerateMsgToken( poMsgNode, oOutMsgToken, poMsgPool );
            if ( poMsgNode->iGetDataSize() != 0 )
               bGenerateMsgInit( poMsgNode, oOutMsgInits, poServiceTree );
            else
               oMsgDefaultInits << TAB << "case " << oToLower( oGetFIPrefix( poServiceTree ) )
                  << oGetTokenName( "__MSG_" + oRemove( poMsgNode->oGetName(), '.' ) ) << ":" << endl;
            bGenerateMsgExpands( poMsgNode, oOutTypeExpands, poServiceTree );
            bGenerateMsgTokenCnvt( poMsgNode, oOutTypeTokenCnvt, poServiceTree );

            // collect message member names used for navigation in visitors
            tclFITypeDependencyList* poDepList = poMsgNode->poGetDependencyList();
            tclFITypeDependencyListIterator iter;
            for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
            {
               oSemanticSet.insert( (*iter)->oGetName() );
            }
         }
      }
   }
   oOutMsgInits << oMsgDefaultInits.str();

   oOutTypeExpands << TAB << TAB << TAB << "(tVoid) bInitMessage(enToken);" << endl;
   oOutTypeExpands << TAB << TAB << TAB << "break;" << endl;

   oOutMsgIds << TAB << oToUpper( oGetFIPrefix( poServiceTree ) )
            << "_C_U16_INVALID_FUNCTIONID = 0xFFFF" << endl;
   oOutMsgIds << "};" << endl;

   // write the names of all message member names 
   ofstream oOutSemanticDef( string( oOutDir + "/semdef_msg.inc" ).c_str() );
   if ( !oOutSemanticDef.good() )
      return false;
   ofstream oOutSemanticCnvt( string( oOutDir + "/semconvert_msg.inc" ).c_str() );
   if ( !oOutSemanticCnvt.good() )
      return false;

   string oPrefix = oToLower( oGetFIPrefix( poServiceTree ) );

   set<string>::const_iterator oNameIt;
   for ( oNameIt = oSemanticSet.begin(); oNameIt != oSemanticSet.end(); ++oNameIt )
   {
      string oSemanticName = bMixedCaseTokens ? ("Msg_" + *oNameIt) : ("MSG_" + oToUpper( *oNameIt ));
      oOutSemanticDef << TAB << "EN_" << oSemanticName << "," << endl;
#if 1
      oOutSemanticCnvt << TAB << "case " << oPrefix << oGetSemanticName( oSemanticName ) << ":" << endl;
      oOutSemanticCnvt << TAB << TAB << "return \"" << *oNameIt << "\";" << endl;
#else
      oOutSemanticCnvt << TAB << "if (token == " << oPrefix
         << oGetSemanticName( "Msg_" + *oNameIt ) << ")" << endl;
      oOutSemanticCnvt << TAB << TAB << "return \"" << *oNameIt << "\";" << endl;
#endif
   }

   return bSuccess;
}

void vAddTypeToMap( tclFINode* poNode, set<string>& oTypeSet, tclFINodePool* poTypePool )
{
   pair<set<string>::iterator, bool> oResult = oTypeSet.insert( poNode->oGetName() );
   if ( oResult.second ) // Einfügen war erfolgreich, Typ bisher nicht vorhanden!
   {
      if ( bBeVerbose )
      {
         cout << "Type " << poNode->oGetName() << " added." << endl;
      }
      tclFITypeDependencyList* poParentList = poNode->poGetParentList();
      tclFITypeDependencyListIterator iter;
      for ( iter = poParentList->begin(); iter != poParentList->end(); ++iter )
      {
         if ( (*iter)->enGetClass() == EN_DEPENDENCY_CLASS_UNION )
         {
            tclFITypeDependencyList* poDepList = poNode->poGetDependencyList();
            tclFITypeDependencyListIterator iter2;
            for ( iter2 = poDepList->begin(); iter2 != poDepList->end(); ++iter2 )
            {
               tclFINode* poChild = poTypePool->poGetNode( (*iter2)->oGetName().c_str() );
               if ( poChild )
               {
                  vAddTypeToMap( poChild, oTypeSet, poTypePool );
               }
            }
         }
      }

      if ( (poNode->enGetType() != EN_OBJTYPE_ENUMERATION) &&
         (poNode->enGetType() != EN_OBJTYPE_BITFIELD) )
      {
         tclFITypeDependencyList* poDepList = poNode->poGetDependencyList();
         for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
         {
            vAddTypeToMap( (*iter)->poGetTarget(), oTypeSet, poTypePool );
         }
      }
   }
}

bool bGenerateTypeInfo( tclFIServiceTree* poServiceTree )
{
   bool bSuccess = true;
   string oOutDir = oGetServiceTreeSubDirectory( poServiceTree );
   string oPrefix = oToLower( oGetFIPrefix( poServiceTree ) );

   // open the neccessary (temporary) files
   ofstream oOutTypeDecl( string( oOutDir + "/types_h.inc" ).c_str() );
   if ( !oOutTypeDecl.good() )
      return false;
   ofstream oOutTypeDef( string( oOutDir + "/types_c.inc" ).c_str() );
   if ( !oOutTypeDef.good() )
      return false;
   ofstream oOutTypeExpands( string( oOutDir + "/typeexpand.inc" ).c_str(), ios_base::app );
   if ( !oOutTypeExpands.good() )
      return false;
   ofstream oOutTypeFDecl( string( oOutDir + "/typesdecl.inc" ).c_str() );
   if ( !oOutTypeFDecl.good() )
      return false;
   ofstream oOutTypeTokenDef( string( oOutDir + "/typtokendef.inc" ).c_str() );
   if ( !oOutTypeTokenDef.good() )
      return false;
   ofstream oOutTypeTokenCnvt( string( oOutDir + "/typtokencnvt.inc" ).c_str(), ios_base::app );
   if ( !oOutTypeTokenCnvt.good() )
      return false;
   ofstream oOutVisitDecl( string( oOutDir + "/visitdecl.inc" ).c_str() );
   if ( !oOutVisitDecl.good() )
      return false;
   ofstream oOutVisitDef( string( oOutDir + "/visitdef.inc" ).c_str() );
   if ( !oOutVisitDef.good() )
      return false;
   ofstream oOutVisitDummy( string( oOutDir + "/visitdummy.inc" ).c_str() );
   if ( !oOutVisitDummy.good() )
      return false;
   ofstream oOutMsgInits( string( oOutDir + "/msginits.inc" ).c_str(), ios_base::app );
   if ( !oOutMsgInits.good() )
      return false;

   // struct member names are collected in a set to avoid multiple instances
   set<string> oSemanticSet;

   // iterate over all types
   set<string> oTypeSet;
   tclFIMessageNodePool* poMsgPool = poServiceTree->poGetMessagePool();
   for ( int iMsgIdx = 0; iMsgIdx < poMsgPool->iGetNumNodes(); ++iMsgIdx )
   {
      tclFINode* poMsgNode = poMsgPool->poGetNode( iMsgIdx, EN_NONE );
      if ( (poMsgNode->enGetType() == EN_OBJTYPE_OPCODE) &&
          (bGenerateErrorMessages ||
           (poMsgNode->oGetName().substr( poMsgNode->oGetName().find( '.' ) + 1 ) != "Error")) )
      {
         tclFITypeDependencyList* poDepList = poMsgNode->poGetDependencyList();
         tclFITypeDependencyListIterator iter;
         for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
         {
            vAddTypeToMap( (*iter)->poGetTarget(), oTypeSet, poServiceTree->poGetUsedGlobalTypePool() );
         }
      }
   }
   // allways enter "T_BinContainer", although only specialized containers are used!
   oTypeSet.insert( "T_BinContainer" );

   //   string oPrefix = oToLower(oGetFIPrefix(poServiceTree));

   tclFINodePool* poTypePool = poServiceTree->poGetUsedGlobalTypePool();
   //for (set<string>::const_iterator oTypeIt = oTypeSet.begin(); oTypeIt != oTypeSet.end(); ++oTypeIt)
   for ( int iTypeIdx = 0; iTypeIdx < poTypePool->iGetNumNodes(); ++iTypeIdx )
   {
      tclFINode* poType = poTypePool->poGetNode( iTypeIdx, EN_BYDEPENDENCY );
      //string oTypeName = *oTypeIt;
      //tclFINode* poType = poTypePool->poGetNode(oTypeName.c_str());

      if ( bIsPredefinedType( poType ) )
         continue;

      if ( bUseAllTypes || (oTypeSet.find( poType->oGetName() ) != oTypeSet.end()) )
      {
         if ( poType->enGetType() != EN_OBJTYPE_ATOM )
         {
            if ( (poType->enGetType() == EN_OBJTYPE_TYPE) &&
                ((*poType->poGetDependencyList()->begin())->poGetTarget()->enGetType() != EN_OBJTYPE_ATOM) )
            {
               tclFINode* poBaseType = (*poType->poGetDependencyList()->begin())->poGetTarget();
               oOutTypeDecl << "typedef " << oGetClassName( poBaseType->oGetName() ) << " "
                  << oGetClassName( poType->oGetName() ) << ";" << endl;
            }
            else
            {
               // forward declaration of all type classes
               if ( bIsBinContainerType( poType->oGetName() ) )
               {
                  oOutTypeFDecl << "#ifndef FI_S_EXTERNAL_FI" << endl;
               }

               oOutTypeFDecl << "class " << oGetClassName( poType->oGetName() ) << ";" << endl;
               if ( bIsBinContainerType( poType->oGetName() ) )
               {
                  oOutTypeFDecl << "#endif // FI_S_EXTERNAL_FI" << endl;
               }

               // write declaration of type class
               bSuccess &= bGenerateTypeClassDeclaration( poType, oOutTypeDecl, poServiceTree );

               // write implementation of type class
               bSuccess &= bGenerateTypeClassDefinition( poType, oOutTypeDef, poServiceTree );

               // write expansion of types (used in visitors)
               bSuccess &= bGenerateTypeExpands( poType, oOutTypeExpands, poServiceTree );

               // define type token used for visitor
               string oTokenName = bMixedCaseTokens
                  ? poType->oGetName().substr( 2 )
                  : oToUpper( poType->oGetName().substr( 2 ) );
               oOutTypeTokenDef << TAB << TAB << "EN_" << oTokenName << "," << endl;

               // translate token (enumeration value) into text
#if 1
               oOutTypeTokenCnvt << TAB << "case " << oPrefix << oGetTokenName( poType->oGetName() ) << ":" << endl;
               oOutTypeTokenCnvt << TAB << TAB << "return \"" << poType->oGetName() << "\";" << endl;
#else
               oOutTypeTokenCnvt << TAB << "if (token == " << oPrefix << oGetTokenName( poType->oGetName() ) << ")" << endl;
               oOutTypeTokenCnvt << TAB << TAB << "return \"" << poType->oGetName() << "\";" << endl;
#endif

#pragma message ("Exclude this, if too much code is generated!")
#if 1
               oOutMsgInits << TAB << "case " << oPrefix << oGetTokenName( poType->oGetName() ) << ":" << endl;

               unsigned int u32UnionIdx = 0;
               if ( poType->enGetType() == EN_OBJTYPE_STRUCTURE )
               {
                  tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
                  tclFITypeDependencyListIterator iter;
                  for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
                  {
                     if ( (*iter)->enGetClass() == EN_DEPENDENCY_CLASS_UNION )
                     {
                        oOutMsgInits << TAB << "case " << oPrefix << oGetTokenName( poType->oGetName() )
                           << "UNION" << u32UnionIdx << ":" << endl;
                        ++u32UnionIdx;
                     }
                  }
               }
#endif

               // define almost empty standard visitor methods
               if ( oToLower( poServiceTree->oGetShortname() ) == (oGetFIGroup() + "types_fi") )
               {
                  // declare "vVisit" methods
                  if ( bIsBinContainerType( poType->oGetName() ) )
                  {
                     oOutVisitDecl << "#ifndef FI_S_EXTERNAL_FI" << endl;
                  }
                  oOutVisitDecl << TAB << "virtual tVoid vVisit" << poType->oGetName().substr( 1 )
                     << "(" << oGetClassName( poType->oGetName() ) << "& o" << poType->oGetName().substr( 2 ) << ");" << endl << endl;
                  if ( bIsBinContainerType( poType->oGetName() ) )
                  {
                     oOutVisitDecl << "#endif // FI_S_EXTERNAL_FI" << endl;
                  }

                  if ( bIsBinContainerType( poType->oGetName() ) )
                  {
                     oOutVisitDef << "#ifndef FI_S_EXTERNAL_FI" << endl;
                  }
                  oOutVisitDef << "tVoid " << oGetFIGroup() << "fi_tclVisitorBase::vVisit" << poType->oGetName().substr( 1 )
                     << "(" << oGetClassName( poType->oGetName() ) << "& /*o" << poType->oGetName().substr( 2 ) << "*/)" << endl;
                  oOutVisitDef << "{" << endl;
                  oOutVisitDef << "}" << endl;
                  oOutVisitDef << "tVoid " << oGetClassName( poType->oGetName() ) << "::vTakeVisitor(" << oGetFIGroup()
                     << "fi_tclVisitorBase& rfoVisitor)" << endl;
                  oOutVisitDef << "{" << endl;
                  oOutVisitDef << TAB << "rfoVisitor.vVisit" << poType->oGetName().substr( 1 ) << "(*this);" << endl;
                  oOutVisitDef << "}" << endl << endl;
                  oOutVisitDummy << "tVoid " << oGetClassName( poType->oGetName() ) << "::vTakeVisitor(" << oGetFIGroup()
                     << "fi_tclVisitorBase& /*rfoVisitor*/)" << endl;
                  oOutVisitDummy << "{" << endl;
                  oOutVisitDummy << "}" << endl << endl;
                  if ( bIsBinContainerType( poType->oGetName() ) )
                  {
                     oOutVisitDef << "#endif // FI_S_EXTERNAL_FI" << endl;
                  }

               }
            }
         }

         // special handling for structure types see below
         if ( poType->enGetType() == EN_OBJTYPE_STRUCTURE )
         {
            if ( oToLower( poServiceTree->oGetShortname() ) != (oGetFIGroup() + "types_fi") )
            {
               oOutVisitDecl << TAB << "virtual tBool bDoExpand" << poType->oGetName().substr( 1 )
                  << "(" << oPrefix << "_tclSemanticDomain::tenType enSemantic);" << endl;

               oOutVisitDef << "tBool " << oPrefix;
               oOutVisitDef << "_tclVisitorBase::bDoExpand" << poType->oGetName().substr( 1 )
                  << "(" << oPrefix << "_tclSemanticDomain::tenType /*enSemantic*/)" << endl;
               oOutVisitDef << "{" << endl;
               oOutVisitDef << TAB << "return TRUE;" << endl;
               oOutVisitDef << "}" << endl;
            }
            unsigned int u32UnionIdx = 0;
            tclFITypeDependencyList* poDepList = poType->poGetDependencyList();
            tclFITypeDependencyListIterator iter;
            for ( iter = poDepList->begin(); iter != poDepList->end(); ++iter )
            {
               // collect struct member names
               oSemanticSet.insert( (*iter)->oGetName() );

               // define internal used tokens for unions
               if ( (*iter)->enGetClass() == EN_DEPENDENCY_CLASS_UNION )
               {
                  string oTokenName = bMixedCaseTokens ? poType->oGetName().substr( 2 ) : oToUpper( poType->oGetName().substr( 2 ) );
                  oOutTypeTokenDef << TAB << "EN_" << oTokenName
                     << "UNION" << u32UnionIdx << "," << endl;
#if 1
                  oOutTypeTokenCnvt << TAB << "case " << oPrefix << oGetTokenName( poType->oGetName() )
                     << "UNION" << u32UnionIdx << ":" << endl;
                  oOutTypeTokenCnvt << TAB << TAB << "return \"" << poType->oGetName() << "Union\";" << endl;
#else
                  oOutTypeTokenCnvt << TAB << "if (token == " << oPrefix << oGetTokenName( poType->oGetName() )
                     << "UNION" << u32UnionIdx << ")" << endl;
                  oOutTypeTokenCnvt << TAB << TAB << "return \"" << poType->oGetName() << "Union\";" << endl;
#endif
                  ++u32UnionIdx;
               }
            }
         }
      }
   }

   // define enumeration values for all struct member names
   // used for navigation in visitors
   ofstream oOutSemanticDef( string( oOutDir + "/semdef_types.inc" ).c_str() );
   if ( !oOutSemanticDef.good() )
      return false;
   ofstream oOutSemanticCnvt( string( oOutDir + "/semconvert_types.inc" ).c_str() );
   if ( !oOutSemanticCnvt.good() )
      return false;
   for ( set<string>::const_iterator oNameIt = oSemanticSet.begin(); oNameIt != oSemanticSet.end(); ++oNameIt )
   {
      string oSemanticName = bMixedCaseTokens ? *oNameIt : oToUpper( *oNameIt );
      oOutSemanticDef << TAB << TAB << "EN_" << oSemanticName << "," << endl;
#if 1
      oOutSemanticCnvt << TAB << "case " << oPrefix << oGetSemanticName( *oNameIt ) << ":" << endl;
      oOutSemanticCnvt << TAB << TAB << "return \"" << *oNameIt << "\";" << endl;
#else
      oOutSemanticCnvt << TAB << "if (token == " << oPrefix << oGetSemanticName( *oNameIt ) << ")" << endl;
      oOutSemanticCnvt << TAB << TAB << "return \"" << *oNameIt << "\";" << endl;
#endif
   }

   bGenerateVisitor( poServiceTree, "PrintVisitor", oTypeSet );
   bGenerateVisitor( poServiceTree, "RandomInitVisitor", oTypeSet );
#ifdef GENERATE_READIN_VISITOR
   bGenerateVisitor( poServiceTree, "ReadInVisitor", oTypeSet );
#endif
   return true;
}


bool bGenerateGeneralIncludes( tclFIServiceGroup* poServiceGroup )
{
   bool bSuccess = true;
   for ( int iServiceIdx = 0; iServiceIdx < poServiceGroup->iGetNumServices(); ++iServiceIdx )
   {
      tclFIServiceTree* poServiceTree = poServiceGroup->poGetServiceTree( iServiceIdx );
      bGenerateServiceInfo( poServiceTree );
      bGenerateErrorInfo( poServiceTree );
      bGenerateMsgInfo( poServiceTree );
      bGenerateTypeInfo( poServiceTree );
   }
   return bSuccess;
}

int main( int argc, char* argv[] )
{
   cerr << endl << oCodeGeneratorVersion << endl;

   int iSuccess = 0;
   if ( true == vParseCommandLine( argc, argv ) )
   {
      tclFIServiceGroup oServiceGroup;
      if ( oServiceGroup.bIsValid() )
      {
         oServiceGroup.vSetVerbosity( bBeVerbose );
         if ( true == bReadXMLCatalogues( &oServiceGroup ) )
         {
            // Commands
            // vGenerateTypeFactory(&oServiceGroup);
            if ( bGenerateTypeFactoryOnly )
            {
               bGenerateTypeFactory( &oServiceGroup );
            }
            else
            {
               bGenerateGeneralIncludes( &oServiceGroup );
            }
         }
      }
      else
         cerr << "** ERROR initialising Service group - ABORT!" << endl;
   }
   else
   {
      vPrintUsage();
      iSuccess = 1;
   }
   // getchar();
   return iSuccess;
}
