/***************************************************************************/
/*!
    \file        gfp_ecc_math.c

    \brief       GF(p) point arithmetics

    \author      Marko Wolf (mwolf@crypto.rub.de),
                 Andr Weimerskirch (weika@crypto.rub.de)
				 Adopted to BPCL CM-DI/ESN-Listle

    \version     2.8

    \date        September 6, 2004

    \warning     None

    \par Reviews:
    \li          06.09.2004 Splint 3.1.1 (-weak no warnings)

    \history
                 05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
                 calls return error
				 30.08.2012 Modified by Gokulkrishnan N - Lint Fix

*/
/***************************************************************************/


#include "bpcl.h"
#include "bpcl_int.h"
#include "ecc.h"


/***************************************************************************/
/*!
   \par       FUNCTION F30/10: point_set_zero

   \par       DESCRIPTION

              This function sets a point to zero

   \par       EQUATION / REFERENCE

              pX = (0,0,0)

   \par       INPUT ARGUMENTS

   \param     (tsECPoint) *pX          - pointer to affine point, that will set to zero

   \par       OUTPUT ARGUMENTS

   \param     (tsECPoint) *pX          - pointer to affine point, that was set to zero

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int point_set_zero( tsECPoint *pX )
{ 
  // declarations
  int ret;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( pX == NULL )
  {
    return -1;
  }

#endif

  // set zero
  ret = field_element_set_zero( &pX->x );
  BPCL_CHECK_ERROR((ret==0));
  
  ret = field_element_set_zero( &pX->y );
  BPCL_CHECK_ERROR((ret==0));

  ret = field_element_set_zero( &pX->z );
  BPCL_CHECK_ERROR((ret==0));

  // succesful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F30/20: point_set_infinity

   \par       DESCRIPTION

              This function sets a point to infinity

   \par       EQUATION / REFERENCE

              pX = (0)

   \par       INPUT ARGUMENTS

   \param     (tsECPoint) *pX          - pointer to affine point, that will set to infinity

   \par       OUTPUT ARGUMENTS

   \param     (tsECPoint) *pX          - pointer to affine point, that is set to infinity

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \param     (int) -2                   - unknown point representation

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int point_set_infinity( tsECPoint *pX )
{
  
  // declarations
  int ret;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( pX == NULL )
  {
    return -1;
  }

#endif

  if ( pX->type == C_PT_AFFINE )
  {
    // pX = (0,0,0)
    ret = point_set_zero( pX );
    BPCL_CHECK_ERROR((ret==0));
  }
  else if ( pX->type == C_PT_JACOBIAN )
  {
    // pX = (1,1,0)
    ret = field_element_set_one( &pX->x );
    BPCL_CHECK_ERROR((ret==0));
  
    ret = field_element_set_one( &pX->y );
    BPCL_CHECK_ERROR((ret==0));

    ret = field_element_set_zero( &pX->z );
    BPCL_CHECK_ERROR((ret==0));
  }
  else
  {
    // unknown point representation
    return -2;
  }

  // succesful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F30/30: point_is_zero

   \par       DESCRIPTION

              This function checks if a point is zero

   \par       EQUATION / REFERENCE

              pX =? (0,0,0)

   \par       INPUT ARGUMENTS

   \param     const (tsECPoint) *pX    - pointer to affine point, that will be checked

   \par       OUTPUT ARGUMENTS

   \param     None

   \par       RETURN VALUE

   \param     (int)  0                   - point is NOT zero

   \param     (int)  1                   - point is zero
   
   \param     (int) -1                   - parameter error

   \param     (int) -2                   - unknown point representation

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int point_is_zero( const tsECPoint *pX )
{
  // declarations
  int ret1,ret2,ret3;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( pX == NULL )
  {
    return -1;
  }

#endif

  // check if zero
  if ( pX->type == C_PT_AFFINE )
  {
    // pX =? (0,0)
    ret1 = field_element_is_zero( &pX->x );
	 BPCL_CHECK_ERROR((ret1 >= 0));
  
    ret2 = field_element_is_zero( &pX->y );
	 BPCL_CHECK_ERROR((ret2 >= 0));

    ret3 = 1;
  }
  else if ( pX->type == C_PT_JACOBIAN )
  {
    // pX =? (0,0,0)
    ret1 = field_element_is_zero( &pX->x );
	 BPCL_CHECK_ERROR((ret1 >= 0));
  
    ret2 = field_element_is_zero( &pX->y );
	 BPCL_CHECK_ERROR((ret2 >= 0));

    ret3 = field_element_is_zero( &pX->z );
	 BPCL_CHECK_ERROR((ret2 >= 0));
  }
  else
  {
    // unknown point representation
    return -2;
  }
  
  // return 1 if ret1 != 0 and ret2 != 0 and ret3 != 0, else return 0
  return (int) ( (ret1 != 0) && (ret2 != 0) && (ret3 != 0) );
}



/***************************************************************************/
/*!
   \par       FUNCTION F30/40: point_is_infinity

   \par       DESCRIPTION

              This function checks if a point is the point at infinity

   \par       EQUATION / REFERENCE

              pX =? (0)

   \par       INPUT ARGUMENTS

   \param     const (tsECPoint) *pX    - pointer to affine point, that will be checked

   \par       OUTPUT ARGUMENTS

   \param     None

   \par       RETURN VALUE

   \param     (int)  0                   - point is NOT the point at infinity

   \param     (int)  1                   - point is the point at infinity
   
   \param     (int) -1                   - parameter error

   \param     (int) -2                   - unknown point representation

   \par       NOTES / WARNINGS / TODO's

   \note      None
*/
/***************************************************************************/
extern int point_is_infinity( const tsECPoint *pX )
{
  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( pX == NULL )
  {
    return -1;
  }

#endif

  // check if infinity
  if ( pX->type == C_PT_AFFINE )
  {
    // pX =? (0,0)
    return point_is_zero( pX ); 
  }
  else if ( pX->type == C_PT_JACOBIAN )
  {
    // pX =? (lamba,lamba,0)
    return field_element_is_zero( &pX->z );
  }
  else
  {
    // unknown point representation
    return -2;
  }

}



/***************************************************************************/
/*!
   \par       FUNCTION F30/50: point_assign

   \par       DESCRIPTION

              This function assigns a point

   \par       EQUATION / REFERENCE

              pY = pX

   \par       INPUT ARGUMENTS

   \param     const (tsECPoint) *X     - pointer to point, that will be assigned

   \par       OUTPUT ARGUMENTS

   \param     (tsECPoint) *pY          - pointer to point, that was assigned

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int point_assign( tsECPoint *pY,
                   const tsECPoint *pX )
{
  // declarations
  int ret;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (pX == NULL) || (pY == NULL) )
  {
    return -1;
  }

#endif

  // assign type
  pY->type = pX->type;
  
  // assign coordinates
  ret = field_element_assign( &pY->x, &pX->x, pX->x.length );
  BPCL_CHECK_ERROR((ret==0));
  
  ret = field_element_assign( &pY->y, &pX->y, pX->y.length );
  BPCL_CHECK_ERROR((ret==0));

  ret = field_element_assign( &pY->z, &pX->z, pX->z.length );
  BPCL_CHECK_ERROR((ret==0));

  // succesful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F30/60: point_invert

   \par       DESCRIPTION

              This function computes the inverse of a point

   \par       EQUATION / REFERENCE

              pY = -pX

              (x,y,z) = (x,-y,z)

   \par       INPUT ARGUMENTS

   \param     const (tsECPoint) *pX    - pointer to input point X

   \param     const (tsECCurve) *cGP   - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     (tsECPoint) *pY          - pointer to inverse point Y

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \param     (int) -2                   - unknown point representation

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int point_invert( tsECPoint *pY,
                   const tsECPoint *pX,
                   const tsECCurve *cGP )
{
  // declarations
  int ret;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (pY == NULL) || (pX == NULL) || (cGP == NULL) || (cGP->ecc_field_params == NULL) )
  {
    return -1;
  }

#endif
  
  if ( pX->type == C_PT_AFFINE )
  {
    // pYx = pAx
    ret = field_element_assign( &pY->x, &pX->x, pX->x.length );
    BPCL_CHECK_ERROR((ret==0));

    // pAy =? 0
    ret = field_element_is_zero( &pX->y );
    BPCL_CHECK_ERROR((ret>=0));

    if ( ret == 1 )
    {
      // pYy = 0
      ret = field_element_set_zero( &pY->y );
      BPCL_CHECK_ERROR((ret==0));
    }
    else
    {
      // pYy = p - y
      ret = field_element_modular_subtract( &pY->y, &cGP->prime_p, &pX->y, cGP->ecc_field_params );
      BPCL_CHECK_ERROR((ret==0));
    }
    
  }
  else if ( pX->type == C_PT_JACOBIAN )
  {
    // pYx = pXx
    ret = field_element_assign( &pY->x, &pX->x, pX->x.length );
    BPCL_CHECK_ERROR((ret==0));

    // pYy = p - pXy
    ret = field_element_modular_subtract( &pY->y, &cGP->prime_p, &pX->y, cGP->ecc_field_params );
    BPCL_CHECK_ERROR((ret==0));
    
    // pYz = pXz
    ret = field_element_assign( &pY->z, &pX->z, pX->z.length );
    BPCL_CHECK_ERROR((ret==0));
  }
  else
  {
    // unknown point representation
    return -2;
  }

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F30/100: point_convert_to_jacobian

   \par       DESCRIPTION

              This function converts a point into a jacobian point

   \par       EQUATION / REFERENCE

              pJ <= pX

   \par       INPUT ARGUMENTS

   \param     const (tsECPoint) *pX    - pointer to input point

   \param     const (tsECCurve) *cGP   - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     tsECPoint *pJ            - pointer to jacobian point

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \param     (int) -2                   - unknown point representation

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int point_convert_to_jacobian( tsECPoint *pJ,
                                const tsECPoint *pX,
                                const tsECCurve *cGP )
{
  // declarations
  int ret;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (pJ == NULL) || (pX == NULL) || (cGP == NULL) )
  {
    return -1;
  }

#endif

   
  // ------ AFFINE => JACOBIAN ----------------------------------------------
  if ( pX->type == C_PT_AFFINE )
  {
    
    // check for pX = (0)
    ret = point_is_infinity( pX );
	 BPCL_CHECK_ERROR((ret >= 0));
    if ( ret == 1 )
    {
       // pJ = (0)
       ret = point_set_infinity( pJ );
       BPCL_CHECK_ERROR((ret==0));
     
       // successful stop
       return 0;
    }

    // set pJ type
    pJ->type = C_PT_JACOBIAN;
    
    // pJx = pXx
    ret = field_element_assign( &pJ->x, &pX->x, pX->x.length );
    BPCL_CHECK_ERROR((ret==0));
  
    // pJy = pXy
    ret = field_element_assign( &pJ->y, &pX->y, pX->y.length );
    BPCL_CHECK_ERROR((ret==0));

    // pJz = 1
    ret = field_element_set_one( &pJ->z );
    BPCL_CHECK_ERROR((ret==0));

    // successful
    return 0;
  }
  // ------ JACOBIAN => JACOBIAN --------------------------------------------
  else if ( pX->type == C_PT_JACOBIAN )
  {    
    // pJ = pX
    ret = point_assign( pJ, pX );
    BPCL_CHECK_ERROR((ret==0));

    // successful
    return 0;
  }
  // ------ UNKNOWN => JACOBIAN ---------------------------------------------
  else
  {
    // unknown point convertion
    return -2;
  }
}



/***************************************************************************/
/*!
   \par       FUNCTION F30/110: point_convert_to_affine

   \par       DESCRIPTION

              This function converts a point into an affine point

   \par       EQUATION / REFERENCE

              pA <= pX

   \par       INPUT ARGUMENTS

   \param     const (tsECPoint) *pX    - pointer to input point

   \param     const (tsECCurve) *cGP   - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     tsECPoint *pA            - pointer to affine point

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \param     (int) -2                   - unknown point representation

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int point_convert_to_affine( tsECPoint *pA,
                              const tsECPoint *pX,
                              const tsECCurve *cGP )
{
  // declarations
  int ret;

  tsECFieldElement fe_z_inv1; // pJ(1/z)
  tsECFieldElement fe_z_inv2; // pJ(1/z)
  tsECFieldElement fe_z_inv3; // pJ(1/z)
 
#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (pA == NULL) || (pX == NULL) || (cGP == NULL) || (cGP->ecc_field_params == NULL) )
  {
    return -1;
  }

#endif  
    
  // ------ JACOBIAN => AFFINE ----------------------------------------------
  if ( pX->type == C_PT_JACOBIAN )
  {
  
    // check for point at infinity
    ret = point_is_infinity( pX );
	 BPCL_CHECK_ERROR((ret >= 0));
    if ( ret == 1 )
    {
       // pA = (0)
      ret = point_set_zero( pA );
      BPCL_CHECK_ERROR((ret==0));
     
      // successful stop
      return 0;
    }

    // set pA type
    pA->type = C_PT_AFFINE;
    
    // allocate auxiliary variables => wordsize
    field_element_allocate( &fe_z_inv1, pX->z.length );
    field_element_allocate( &fe_z_inv2, pX->z.length );
    field_element_allocate( &fe_z_inv3, pX->z.length );
  
    // pX(1/z) : 1I
    ret = field_element_modular_invert( &fe_z_inv1, &pX->z, cGP->ecc_field_params );
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f", &fe_z_inv1, 
		&fe_z_inv2, &fe_z_inv3);

    // pX(1/z) : 1S
    ret = field_element_modular_square( &fe_z_inv2, &fe_z_inv1, cGP->ecc_field_params );
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f", &fe_z_inv1, 
		&fe_z_inv2, &fe_z_inv3);

    // pX(1/z) : 1M
    ret = field_element_modular_multiply( &fe_z_inv3, &fe_z_inv1, &fe_z_inv2, cGP->ecc_field_params );
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f", &fe_z_inv1, 
		&fe_z_inv2, &fe_z_inv3);

    // pA(x) = pX(x/z) : 1M
    ret = field_element_modular_multiply( &pA->x, &pX->x, &fe_z_inv2, cGP->ecc_field_params );
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f", &fe_z_inv1, 
		&fe_z_inv2, &fe_z_inv3);
  
    // pA(y) = pX(y/z) : 1M
    ret = field_element_modular_multiply( &pA->y, &pX->y, &fe_z_inv3, cGP->ecc_field_params );
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f", &fe_z_inv1, 
		&fe_z_inv2, &fe_z_inv3);

    //pA(z) = 0
    ret = field_element_set_zero( &pA->z );
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f", &fe_z_inv1, 
		&fe_z_inv2, &fe_z_inv3);

    // free auxiliary variables
    field_element_free( &fe_z_inv1 );
    /* As ret is not modified, the below line is commented to prevent
	 Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f",  &fe_z_inv2, &fe_z_inv3);
    field_element_free( &fe_z_inv2 );
    /* As ret is not modified, the below line is commented to prevent
    Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECKERROR_FREEMEM((ret==0), "%f",  &fe_z_inv3);
    field_element_free( &fe_z_inv3 );
    /* As ret is not modified, the below line is commented to prevent
    Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECK_ERROR((ret==0));
  
    // successful
    return 0;
  }
  // ------ AFFINE => AFFINE ------------------------------------------------
  else if ( pX->type == C_PT_AFFINE )
  {
    // pA = pX
    ret = point_assign( pA, pX );
    BPCL_CHECK_ERROR((ret==0));

    // successful
    return 0;
  }
  // ------ UNKNOWN => AFFINE -----------------------------------------------
  else
  {
    // unknown point convertion
    return -2;
  }

}



/***************************************************************************/
/*!
   \par       FUNCTION F30/150: point_jacobian_double

   \par       DESCRIPTION

              This function doubles a jacobian projection point

   \par       EQUATION / REFERENCE

              pJ = 2  pJ0

              The projective form of the doubling formula on the curve
              y = x + ax + b modulo p is: 
              
              (X2, Y2, Z2) = 2  (X1, Y1, Z1) 
              
              where
              
              M  = 3  X1 + a  Z1^4
              Z2 = 2  Y1  Z1,
              S  = 4  X1  Y1
              X2 = M - 2  S,
              T  = 8  Y1^4
              Y2 = M (S - X2) - T

   \par       INPUT ARGUMENTS

   \param     const (tsECPoint) *pJ0   - pointer to jacobian projection point, that will be doubled

   \param     const (tsECCurve) *cGP   - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     (tsECPoint) *pJ          - pointer to resulting jacobian projection point

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      a = p - 3 optimization not built in!

   \note      costs: 7M + 3S

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int point_jacobian_double( tsECPoint *pJ,
                            const tsECPoint *pJ0,
                            const tsECCurve *cGP )
{
  // declarations
  int ret, ret2;

  tsECFieldElement  T1;
  tsECFieldElement  T2;
  tsECFieldElement  T3;
  tsECFieldElement  T4;
  tsECFieldElement  T5;
  tsECFieldElement  TF;
 
#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (pJ == NULL) || (pJ0 == NULL) || (cGP == NULL) || (cGP->ecc_field_params == NULL) || 
       (pJ0->type != C_PT_JACOBIAN ) )
  {
    return -1;
  }

#endif
  
  // set pJ type
  pJ->type = C_PT_JACOBIAN;


  // check for point at infinity
  ret = point_is_infinity( pJ0 );
  BPCL_CHECK_ERROR((ret >= 0));
  if ( ret == 1 )
  {
    // return pJ = (0)
    ret = point_set_infinity( pJ );
    BPCL_CHECK_ERROR((ret==0));
    
    // successful stop
    return 0;
  }

  // allocate auxiliary variables
  field_element_allocate( &T1, cGP->prime_p.length );
  field_element_allocate( &T2, cGP->prime_p.length );
  field_element_allocate( &T3, cGP->prime_p.length );
  field_element_allocate( &T4, cGP->prime_p.length );
  field_element_allocate( &T5, cGP->prime_p.length );
  field_element_allocate( &TF, cGP->prime_p.length );
  
  // 01.) T1 <- X1
  ret = field_element_assign( &T1, &pJ0->x, pJ0->x.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 02.) T2 <- Y1
  ret = field_element_assign( &T2, &pJ0->y, pJ0->y.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 03.) T3 <- Z1
  ret = field_element_assign( &T3, &pJ0->z, pJ0->z.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 04.) If T2 = 0 or T3 = 0 then output (1, 1, 0) and stop
  ret = field_element_is_zero( &T2 );
  BPCL_CHECKERROR_FREEMEM((ret >=0 ), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  ret2 = field_element_is_zero( &T3 );
  BPCL_CHECKERROR_FREEMEM((ret2 >=0 ), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&TF);

  if ( (ret == 1) || (ret2 == 1) )
  {
    // pJ = (0) = (1,1,0)
    ret = point_set_infinity( pJ );

    // free auxiliary variables
    field_element_free( &T1 );
	 BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T2,&T3,&T4,&T5,&TF);

    field_element_free( &T2 );
	 /* As ret is not modified, the below line is commented to prevent
    Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &T3,&T4,&T5,&TF);
    field_element_free( &T3 );
    /* As ret is not modified, the below line is commented to prevent
    Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f", &T4,&T5,&TF);
    field_element_free( &T4 );
	 /* As ret is not modified, the below line is commented to prevent
    Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f", &T5,&TF);
    field_element_free( &T5 );
	 /* As ret is not modified, the below line is commented to prevent
    Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &TF);
    field_element_free( &TF );
	 /* As ret is not modified, the below line is commented to prevent
    Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECK_ERROR((ret==0));
 
    // successful stop
    return 0;
  }

  // 05.) If a = p - 3 then
  //        T4 <- T3
  //        T5 <- T1 - T4
  //        T4 <- T1 + T4
  //        T5 <- T4  T5
  //        T4 <- 3  T5 = M

  //      else

  //        T4 <- a
  ret = field_element_assign( &T4, &cGP->coefficient_a, cGP->coefficient_a.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  //        T5 <- T3
  ret = field_element_modular_square( &T5, &T3, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  //        T5 <- T5
  ret = field_element_modular_square( &T5, &T5, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  //        T5 <- T4  T5
  ret = field_element_modular_multiply( &T5, &T4, &T5, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  //        T4 <- T1
  ret = field_element_modular_square( &T4, &T1, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  //        T4 <- 3  T4
  ret = field_element_assign( &TF, &T4, T4.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);
  ret = field_element_modular_add( &T4, &T4, &T4, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);
  ret = field_element_modular_add( &T4, &T4, &TF, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  //        T4 <- T4 + T5 = M
  ret = field_element_modular_add( &T4, &T4, &T5, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 06.) T3 <- T2  T3
  ret = field_element_modular_multiply( &T3, &T2, &T3, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 07.) T3 <- 2  T3 = Z2
  ret = field_element_modular_add( &T3, &T3, &T3, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 08.) T2 <- T2
  ret = field_element_modular_square( &T2, &T2, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 09.) T5 <- T1  T2
  ret = field_element_modular_multiply( &T5, &T1, &T2, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 10.) T5 <- 4  T5 = S
  ret = field_element_modular_add( &T5, &T5, &T5, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);
  ret = field_element_modular_add( &T5, &T5, &T5, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 11.) T1 <- T4
  ret = field_element_modular_square( &T1, &T4, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 12.) T1 <- T1 - 2  T5 = X2
  ret = field_element_modular_subtract( &T1, &T1, &T5, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);
  ret = field_element_modular_subtract( &T1, &T1, &T5, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 13.) T2 <- T2
  ret = field_element_modular_square( &T2, &T2, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 14.) T2 <- 8  T2 = T
  ret = field_element_modular_add( &T2, &T2, &T2, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);
  ret = field_element_modular_add( &T2, &T2, &T2, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);
  ret = field_element_modular_add( &T2, &T2, &T2, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 15.) T5 <- T5 - T1
  ret = field_element_modular_subtract( &T5, &T5, &T1, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 16.) T5 <- T4  T5
  ret = field_element_modular_multiply( &T5, &T4, &T5, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 17.) T2 <- T5 - T2 = Y2
  ret = field_element_modular_subtract( &T2, &T5, &T2, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 18.) X2 <- T1
  ret = field_element_assign( &pJ->x, &T1, T1.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 19.) Y2 <- T2
  ret = field_element_assign( &pJ->y, &T2, T2.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // 20.) Z2 <- T3
  ret = field_element_assign( &pJ->z, &T3, T3.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5,&TF);

  // free auxiliary variables
  field_element_free( &T1 );
  /* As ret is not modified, the below line is commented to prevent
    Lint warning 'Boolean within 'if' always evaluates to False' */
//  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T2,&T3,&T4,&T5,&TF);
  field_element_free( &T2 );
  /* As ret is not modified, the below line is commented to prevent
    Lint warning 'Boolean within 'if' always evaluates to False' */
//  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &T3,&T4,&T5,&TF);
  field_element_free( &T3 );
  /* As ret is not modified, the below line is commented to prevent
    Lint warning 'Boolean within 'if' always evaluates to False' */
//  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f", &T4,&T5,&TF);
  field_element_free( &T4 );
  /* As ret is not modified, the below line is commented to prevent
    Lint warning 'Boolean within 'if' always evaluates to False' */
//  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f", &T5,&TF);
  field_element_free( &T5 );
  /* As ret is not modified, the below line is commented to prevent
    Lint warning 'Boolean within 'if' always evaluates to False' */
//  BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &TF);
  field_element_free( &TF );
  /* As ret is not modified, the below line is commented to prevent
    Lint warning 'Boolean within 'if' always evaluates to False' */
//  BPCL_CHECK_ERROR((ret==0));

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F30/160: point_jacobian_add_mixed

   \par       DESCRIPTION

              This function adds an affine point to a jacobian projection point

   \par       EQUATION / REFERENCE

              pJ = pJ1 + pA2

              (X3, Y3, Z3) = (X1, Y1, Z1) + (X2, Y2, 1)

              where

              A = X2  Z1
              B = Y2  Z1
              C = A  - X1
              D = B  - Y1

              X3 = D - (C + 2  X1  C)
              Y3 = D  (X1  C - X3) - Y1  C
              Z3 = Z1  C

              Check for P1 = P2 with (X1 Y1, Z1) = (X2 = X1/Z1, Y2 = Y1/Z1)

              -> C = A - X1 = X1/Z1  Z1 - X1 = 0
              -> D = B - Y1 = Y1/Z1  Z1 - Y1 = 0

              => if ( C == 0 && D == 0 ) then P3 = Double ( P1 ) 

              => see 6.a) - 6.c)

   \par       INPUT ARGUMENTS

   \param     const (tsECPoint) *pJ1   - pointer to jacobian projection point

   \param     const (tsECPoint) *pA2   - pointer to affine point

   \param     const (tsECCurve) *cGP   - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     (tsECPoint) *pJ          - pointer to resulting jacobian projection point

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      costs: 11M + 3S

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int point_jacobian_add_mixed( tsECPoint *pJ,
                               const tsECPoint *pJ1,
                               const tsECPoint *pA2,
                               const tsECCurve *cGP )
{
  // declarations
  int ret;

  tsECFieldElement T1;
  tsECFieldElement T2;
  tsECFieldElement T3;
  tsECFieldElement T4;
  tsECFieldElement T5;
 
#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (pJ == NULL) || (pJ1 == NULL) || (pA2 == NULL) || (cGP == NULL) || (cGP->ecc_field_params == NULL) ||
       (pJ1->type != C_PT_JACOBIAN) || (pA2->type != C_PT_AFFINE) )
  {
    return -1;
  }

#endif

  // set pJ type
  pJ->type = C_PT_JACOBIAN;

  // pJ1 =? (0)
  ret = point_is_infinity( pJ1 );
  BPCL_CHECK_ERROR((ret >= 0));
 
  // pJ <- pA2
  if ( ret == 1 )
  {
    ret = point_convert_to_jacobian( pJ, pA2, cGP );
    BPCL_CHECK_ERROR((ret==0));

    // stop successful
    return 0;
  }

  // pA2 =? (0)
  ret = point_is_infinity( pA2 );
  BPCL_CHECK_ERROR((ret >= 0));

  // pJ <- pJ1
  if ( ret == 1 )
  {
    ret = point_assign( pJ, pJ1 );    
    BPCL_CHECK_ERROR((ret==0));

    // stop successful
    return 0;
  }

  // allocate auxiliary variables
  field_element_allocate( &T1, cGP->prime_p.length );
  field_element_allocate( &T2, cGP->prime_p.length );
  field_element_allocate( &T3, cGP->prime_p.length );
  field_element_allocate( &T4, cGP->prime_p.length );
  field_element_allocate( &T5, cGP->prime_p.length );

  // 01.) T1 <- Z1         =  Z1
  ret = field_element_modular_square( &T1, &pJ1->z, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);


  // 02.) T2 <- T1  Z1     =  Z1
  ret = field_element_modular_multiply( &T2, &T1, &pJ1->z, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 03.) T2 <- T2  Y2     =  Z1  Y2 = B
  ret = field_element_modular_multiply( &T2, &T2, &pA2->y, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 04.) T2 <- T2 - Y1     =  B - Y1 = D
  ret = field_element_modular_subtract( &T2, &T2, &pJ1->y, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 05.) T1 <- T1  X2     =  Z1  X2 = A
  ret = field_element_modular_multiply( &T1, &T1, &pA2->x, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 06.) T1 <- T1 - X1     =  A - X1 = C
  ret = field_element_modular_subtract( &T1, &T1, &pJ1->x, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 06.a) C =? 0
  ret = field_element_is_zero( &T1 );
  BPCL_CHECKERROR_FREEMEM((ret >= 0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);
  
  if ( ret == 1 )
  {
    // 06.b) D =? 0
    ret =  field_element_is_zero( &T2 );
    BPCL_CHECKERROR_FREEMEM((ret >= 0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

    // 06.c) if ( D == 0 && C == 0 ) then pJ = Double( pJ1 )
    if ( ret == 1 )
    {
      // double pJ1
      ret = point_jacobian_double( pJ, pJ1, cGP );
	   BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);
    
      // free auxiliary variables
      field_element_free( &T1 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &T2,&T3,&T4,&T5);
      field_element_free( &T2 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f", &T3,&T4,&T5);
      field_element_free( &T3 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f", &T4,&T5);
      field_element_free( &T4 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &T5);
      field_element_free( &T5 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECK_ERROR((ret==0));

      // stop successful
      return 0;
    }
  }

  // 07.) T3 <- T1         =  C
  ret = field_element_modular_square( &T3, &T1, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 08.) T4 <- T3  X1     =  C  X1 = X1  C
  ret = field_element_modular_multiply( &T4, &T3, &pJ1->x, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);
  
  // 09.) T3 <- T3  T1     =  C
  ret = field_element_modular_multiply( &T3, &T3, &T1, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 10.) T5 <- T2         =  D
  ret = field_element_modular_square( &T5, &T2, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 11.) T5 <- T5 - T3     =  D - C
  ret = field_element_modular_subtract( &T5, &T5, &T3, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 12.) T5 <- T5 - 2  T4 =  D - C - 2  X1  C = X3
  ret = field_element_modular_subtract( &T5, &T5, &T4, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);
  ret = field_element_modular_subtract( &T5, &T5, &T4, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 13.) T4 <- T4 - T5     =  X1  C - X3
  ret = field_element_modular_subtract( &T4, &T4, &T5, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 14.) T4 <- T2  T4     =  D  (X1  C - X3)
  ret = field_element_modular_multiply( &T4, &T2, &T4, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 15.) T3 <- T3  Y1     =  Y1  C
  ret = field_element_modular_multiply( &T3, &T3, &pJ1->y, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 16.) T4 <- T4 - T3     =  D  (X1  C - X3) - Y1  C = Y3
  ret = field_element_modular_subtract( &T4, &T4, &T3, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 17.) T1 <- T1  Z1     =  C  Z1 = Z3
  ret = field_element_modular_multiply( &T1, &T1, &pJ1->z, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 18.) X3 <- T5
  ret = field_element_assign( &pJ->x, &T5, T5.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 19.) Y3 <- T4
  ret = field_element_assign( &pJ->y, &T4, T4.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // 20.) Z3 <- T1
  ret = field_element_assign( &pJ->z, &T1, T1.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T1,&T2,&T3,&T4,&T5);

  // free auxiliary variables
  field_element_free( &T1 );
  /* As ret is not modified, the below line is commented to prevent
  Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &T2,&T3,&T4,&T5);
  field_element_free( &T2 );
  /* As ret is not modified, the below line is commented to prevent
  Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f", &T3,&T4,&T5);
  field_element_free( &T3 );
  /* As ret is not modified, the below line is commented to prevent
  Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f",&T4,&T5);
  field_element_free( &T4 );
  /* As ret is not modified, the below line is commented to prevent
  Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECKERROR_FREEMEM((ret==0), "%f",&T5);
  field_element_free( &T5 );
  /* As ret is not modified, the below line is commented to prevent
  Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECK_ERROR((ret==0));

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F30/170: point_jacobian_add

   \par       DESCRIPTION

              This function adds two jacobian points to a jacobian projection point

   \par       EQUATION / REFERENCE

              pJ = pJ1 + pJ2

              (X2, Y2, Z2) = (X0, Y0, Z0) + (X1, Y1, Z1)

              where

              U0 = X0  Z1
              S0 = Y0  Z1
              U1 = X1  Z0
              S1 = Y1  Z0

              W = U0 - U1
              R = S0 - S1
              T = U0 + U1
              M = S0 + S1

              Z2 = Z0  Z1  W
              X2 = R - T  W
              V  = T  W - 2  X2
              2  Y2 = V  R - M  W

   \par       INPUT ARGUMENTS

   \param     const (tsECPoint) *pJ0   - pointer to jacobian projection point

   \param     const (tsECPoint) *pJ1   - pointer to jacobian projection point

   \param     const (tsECCurve) *cGP   - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     (tsECPoint) *pJ          - pointer to resulting jacobian projection point

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      costs: 11M + 3S

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int point_jacobian_add( tsECPoint *pJ,
                         const tsECPoint *pJ0,
                         const tsECPoint *pJ1,
                         const tsECCurve *cGP )
{
  // declarations
  int ret;
  int Z1_is_1;

  tsECFieldElement T1;
  tsECFieldElement T2;
  tsECFieldElement T3;
  tsECFieldElement T4;
  tsECFieldElement T5;
  tsECFieldElement T6;
  tsECFieldElement T7;
 
#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (pJ == NULL) || (pJ0 == NULL) || (pJ1 == NULL) || (cGP == NULL) || (cGP->ecc_field_params == NULL) || 
       (pJ0->type != C_PT_JACOBIAN) || (pJ1->type != C_PT_JACOBIAN)  )
  {
    return -1;
  }

#endif

  
  // set pJ type
  pJ->type = C_PT_JACOBIAN;


  // pJ0 =? (0)
  ret = point_is_infinity( pJ0 );
  BPCL_CHECK_ERROR((ret >= 0));
 
  // pJ <- pJ1
  if ( ret == 1 )
  {
    ret = point_assign( pJ, pJ1 );
    BPCL_CHECK_ERROR((ret==0));

    // stop successful
    return 0;
  }

  // pJ1 =? (0)
  ret = point_is_infinity( pJ1 );
  BPCL_CHECK_ERROR((ret >= 0));

  // pJ <- pJ0
  if ( ret == 1 )
  {
    ret = point_assign( pJ, pJ0 );    
    BPCL_CHECK_ERROR((ret==0));

    // stop successful
    return 0;
  }

  // allocate auxiliary variables
  field_element_allocate( &T1, cGP->prime_p.length );
  field_element_allocate( &T2, cGP->prime_p.length );
  field_element_allocate( &T3, cGP->prime_p.length );
  field_element_allocate( &T4, cGP->prime_p.length );
  field_element_allocate( &T5, cGP->prime_p.length );
  field_element_allocate( &T6, cGP->prime_p.length );
  field_element_allocate( &T7, cGP->prime_p.length );

  // 1.) T1 <- X0 = U0 (if Z1 = 1)
  ret = field_element_assign( &T1, &pJ0->x, pJ0->x.length);
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 2.) T2 <- Y0 = S0 (if Z1 = 1)
  ret = field_element_assign( &T2, &pJ0->y, pJ0->y.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 3.) T3 <- Z0
  ret = field_element_assign( &T3, &pJ0->z, pJ0->z.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 4.) T4 <- X1
  ret = field_element_assign( &T4, &pJ1->x, pJ1->x.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 5.) T5 <- Y1
  ret = field_element_assign( &T5, &pJ1->y, pJ1->y.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);
  
  // 6.) if Z1 != 1 then
  Z1_is_1 = field_element_is_one( &pJ1->z );
  BPCL_CHECKERROR_FREEMEM((Z1_is_1 >= 0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  if ( Z1_is_1 != 1 )
  {
    // 6.a) T6 <- Z1
    ret = field_element_assign( &T6, &pJ1->z, pJ1->z.length );
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

    // 6.b) T7 <- T6
    ret = field_element_modular_square( &T7, &T6, cGP->ecc_field_params );
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

    // 6.c) T1 <- T1  T7 = U0 (if Z1 != 1)
    ret = field_element_modular_multiply( &T1, &T1, &T7, cGP->ecc_field_params );
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);
    
    // 6.d) T7 <- T6  T7
    ret = field_element_modular_multiply( &T7, &T6, &T7, cGP->ecc_field_params );
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

    // 6.e) T2 <- T2  T7 = S0 (if Z1 != 1)
    ret = field_element_modular_multiply( &T2, &T2, &T7, cGP->ecc_field_params );
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);
  }
  
  // 7.) T7 <- T3
  ret = field_element_modular_square( &T7, &T3, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 8.) T4 <- T4  T7 = U1
  ret = field_element_modular_multiply( &T4, &T4, &T7, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 9.) T7 <- T3  T7
  ret = field_element_modular_multiply( &T7, &T3, &T7, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 10.) T5 <- T5  T7 = S1
  ret = field_element_modular_multiply( &T5, &T5, &T7, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 11.) T4 <- T1 - T4 = W
  ret = field_element_modular_subtract( &T4, &T1, &T4, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 12.) T5 <- T2 - T5 = R
  ret = field_element_modular_subtract( &T5, &T2, &T5, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 13.) if T4 = 0 then
  ret = field_element_is_zero( &T4 );
  BPCL_CHECKERROR_FREEMEM((ret >= 0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  if ( ret == 1 )
  {
    // 13.a) if T5 = 0 then output (0,0,0) and stop
    ret = field_element_is_zero( &T5 );
    BPCL_CHECKERROR_FREEMEM((ret >= 0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

    // 13.b) else output (1, 1, 0) and stop
    if ( ret == 1 )
    {
      // double pJ1
      ret = point_jacobian_double( pJ, pJ1, cGP );
      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);
    
      // free auxiliary variables
      field_element_free( &T1 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
/*      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T2,&T3,&T4,
		&T5,&T6,&T7); */
      field_element_free( &T2 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T3,&T4,&T5,&T6,&T7);
      field_element_free( &T3 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &T4,&T5,&T6,&T7);
      field_element_free( &T4 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f", &T5,&T6,&T7);
      field_element_free( &T5 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f", &T6,&T7);
      field_element_free( &T6 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &T7);
      field_element_free( &T7 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECK_ERROR((ret==0));

      // stop successful
      return 0;
    }
    else
    {
      // pJ = (0)
      ret = point_set_infinity( pJ );
      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

      // free auxiliary variables
      field_element_free( &T1 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
/*      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f", &T2,&T3,&T4,
		&T5,&T6,&T7); */
      field_element_free( &T2 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f", &T3,&T4,&T5,&T6,&T7);
      field_element_free( &T3 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &T4,&T5,&T6,&T7);
      field_element_free( &T4 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f", &T5,&T6,&T7);
      field_element_free( &T5 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f", &T6,&T7);
      field_element_free( &T6 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &T7);
      field_element_free( &T7 );
	   /* As ret is not modified, the below line is commented to prevent
      Lint warning 'Boolean within 'if' always evaluates to False' */
//      BPCL_CHECK_ERROR((ret==0));

      // stop successful
      return 0;
    }
  }

  // 14.) T1 <- 2  T1 - T4 = T
  ret = field_element_modular_add( &T1, &T1, &T1, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  ret = field_element_modular_subtract( &T1, &T1, &T4, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 15.) T2 <- 2  T2 - T5 = M
  ret = field_element_modular_add( &T2, &T2, &T2, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  ret = field_element_modular_subtract( &T2, &T2, &T5, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 16.) If Z1 != 1 then T3 <- T3  T6
  if ( Z1_is_1 != 1 )
  {
    ret = field_element_modular_multiply( &T3, &T3, &T6, cGP->ecc_field_params );
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);
  }

  // 17.) T3 <- T3  T4 = Z2
  ret = field_element_modular_multiply( &T3, &T3, &T4, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 18.) T7 <- T4
  ret = field_element_modular_square( &T7, &T4, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 19.) T4 <- T4  T7
  ret = field_element_modular_multiply( &T4, &T4, &T7, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 20.) T7 <- T1  T7
  ret = field_element_modular_multiply( &T7, &T1, &T7, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 21.) T1 <- T5
  ret = field_element_modular_square( &T1, &T5, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 22.) T1 <- T1 - T7 = X2
  ret = field_element_modular_subtract( &T1, &T1, &T7, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 23.) T7 <- T7 - 2  T1 = V
  ret = field_element_modular_subtract( &T7, &T7, &T1, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  ret = field_element_modular_subtract( &T7, &T7, &T1, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 24.) T5 <- T5  T7
  ret = field_element_modular_multiply( &T5, &T5, &T7, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 25.) T4 <- T2  T4
  ret = field_element_modular_multiply( &T4, &T2, &T4, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 26.) T2 <- T5 - T4
  ret = field_element_modular_subtract( &T2, &T5, &T4, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 27.) T2 <- T2 / 2 = Y2
  ret = field_element_modular_divide_2( &T2, &T2, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 28.) X2 <- T1
  ret = field_element_assign( &pJ->x, &T1, T1.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 29.) Y2 <- T2
  ret = field_element_assign( &pJ->y, &T2, T2.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // 30.) Z2 <- T3
  ret = field_element_assign( &pJ->z, &T3, T3.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f%f", &T1,&T2,&T3,&T4,
		&T5,&T6,&T7);

  // free auxiliary variables
  field_element_free( &T1 );
  /* As ret is not modified, the below line is commented to prevent
  Lint warning 'Boolean within 'if' always evaluates to False' */
/*    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f%f",&T2,&T3,&T4,
		&T5,&T6,&T7);*/
  field_element_free( &T2 );
  /* As ret is not modified, the below line is commented to prevent
  Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f%f",&T3,&T4,&T5,&T6,&T7);
  field_element_free( &T3 );
  /* As ret is not modified, the below line is commented to prevent
  Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f",&T4,&T5,&T6,&T7);
  field_element_free( &T4 );
  /* As ret is not modified, the below line is commented to prevent
  Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f",&T5,&T6,&T7);
  field_element_free( &T5 );
  /* As ret is not modified, the below line is commented to prevent
  Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f",&T6,&T7);
  field_element_free( &T6 );
  /* As ret is not modified, the below line is commented to prevent
  Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECKERROR_FREEMEM((ret==0), "%f",&T7);
  field_element_free( &T7 );
  /* As ret is not modified, the below line is commented to prevent
  Lint warning 'Boolean within 'if' always evaluates to False' */
//    BPCL_CHECK_ERROR((ret==0));

  // successful
  return 0;
}




/***************************************************************************/
/*!
   \par       FUNCTION F30/180: point_jacobian_subtract_mixed

   \par       DESCRIPTION

              This function subtracts an affine point from a jacobian projection point

   \par       EQUATION / REFERENCE

              pJ = pJ1 - pA2 = pJ1 + (-pA2)

   \par       INPUT ARGUMENTS

   \param     const (tsECPoint) *pJ1   - pointer to jacobian projection point

   \param     const (tsECPoint) *pA2   - pointer to affine point

   \param     const (tsECCurve) *cGP   - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     (tsECPoint) *pJ          - pointer to resulting jacobian projection point

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      costs: 11M + 3S

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int point_jacobian_subtract_mixed( tsECPoint *pJ,
                                    const tsECPoint *pJ1,
                                    const tsECPoint *pA2,
                                    const tsECCurve *cGP )
{
  // declarations
  int ret;

  tsECPoint tmp_pA;
  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (pJ == NULL) || (pJ1 == NULL) || (pA2 == NULL) || (cGP == NULL) ||
       (pJ1->type != C_PT_JACOBIAN) || (pA2->type != C_PT_AFFINE) )
  {
    return -1;
  }

#endif

  // allocate point tmp_pA
  ret = point_allocate( &tmp_pA, cGP->prime_p.length, C_PT_AFFINE );
  BPCL_CHECK_ERROR((ret==0));

  // invert pA2
  ret = point_invert( &tmp_pA, pA2, cGP );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p", &tmp_pA);

  // add inverted pA2
  ret = point_jacobian_add_mixed( pJ, pJ1, &tmp_pA, cGP );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p", &tmp_pA);

  // free tmp_pA
  ret = point_free( &tmp_pA );
  BPCL_CHECK_ERROR((ret==0));
  
  // succesful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F30/190: point_jacobian_subtract

   \par       DESCRIPTION

              This function subtracts an jacobian point from a jacobian point

   \par       EQUATION / REFERENCE

              pJ = pJ1 - pJ2 = pJ1 + (-pJ2)

   \par       INPUT ARGUMENTS

   \param     const (tsECPoint) *pJ1   - pointer to jacobian projection point

   \param     const (tsECPoint) *pJ2   - pointer to jacobian projection point

   \param     const (tsECCurve) *cGP   - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     (tsECPoint) *pJ          - pointer to resulting jacobian projection point

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      costs: 11M + 3S

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int point_jacobian_subtract( tsECPoint *pJ,
                              const tsECPoint *pJ0,
                              const tsECPoint *pJ1,
                              const tsECCurve *cGP )
{
 // declarations
  int ret;
  tsECPoint tmp_pJ1;
  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (pJ == NULL) || (pJ0 == NULL) || (pJ1 == NULL) || (cGP == NULL) || 
       (pJ0->type != C_PT_JACOBIAN) || (pJ1->type != C_PT_JACOBIAN)  )
  {
    return -1;
  }

#endif

  // allocate tmp_pJ1
  ret = point_allocate( &tmp_pJ1, cGP->prime_p.length, C_PT_JACOBIAN );
  BPCL_CHECK_ERROR((ret==0));

  // invert pJ2
  ret = point_invert( &tmp_pJ1, pJ1, cGP );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p", &tmp_pJ1);

  // add inverted pJ1
  ret = point_jacobian_add( pJ, pJ0, &tmp_pJ1, cGP );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p", &tmp_pJ1);

  // free tmp_pJ1
  ret = point_free( &tmp_pJ1 );
  BPCL_CHECK_ERROR((ret==0));
  
  // succesful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F30/200: point_jacobian_multiply_binary_NAF

   \par       DESCRIPTION

              This function does the scalar multiplication of unknown pA1 with k

   \par       EQUATION / REFERENCE

              pJ = k  pA1

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *k - pointer to field element k

   \param     const (tsECPoint) *pA1   - pointer to affine point

   \param     const (tsECCurve) *cGP   - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     (tsECPoint) *pJ          - pointer to resulting jacobian projection point

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      Are there k's < 0? Not allowed!

   \note      costs: (m/3)  Adds + m  Doubles (where m = Total bit length of k)

   \note      special SPA / DPA resistant version
*/
/***************************************************************************/



/***************************************************************************/
/*!
   \par       FUNCTION F30/210: point_jacobian_multiply_k_ary_window

   \par       DESCRIPTION

              This function does the scalar multiplication of unknown pA1 with k

   \par       EQUATION / REFERENCE

              pJ = k  pA1

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *k - pointer to field element k

   \param     const (tsECPoint) *pA1   - pointer to affine point

   \parma     const (tU8) window_size   - window size  

   \param     const (tsECCurve) *cGP   - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     (tsECPoint) *pJ          - pointer to resulting jacobian projection point

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      costs: ( m/w + 2^(w) )  Adds + m  Doubles  (where m = Total bit length of k)

   \note      special SPA / DPA resistant version

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int point_jacobian_multiply_k_ary_window( tsECPoint     *pJ,
                                           const tsECFieldElement *k,
                                           const tsECPoint     *pA1,
                                           const tU8             window_size,
                                           const tsECCurve     *cGP )
{
  
  // declarations
  int ret,i,j;
  int window_full, k_number;
  int *k_window;
  tChar caFmt[20]={'\0'};

  tsECPoint pQ;
  tsECPoint *premcomputed_P;

#if (BPCL_ECC_FEAT_DPA_PROTECTION == 1 )

  tsECFieldElement dpa_resistant_k;
  tU32 random_x_word;
  tsECFieldElement random_x;

#else

  tsECFieldElement local_k;

#endif


    
#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (pJ == NULL) || (k == NULL) || (pA1 == NULL) || (cGP == NULL) ||
       (k->length < 1) || (pA1->type != C_PT_AFFINE) || (window_size < (tU8) 1) || (window_size > (tU8) 7) )
  {
    return -1;
  }

#endif


  // set pJ type
  pJ->type = C_PT_JACOBIAN;

  // check k =? 0
  ret = field_element_is_zero( k );
  BPCL_CHECK_ERROR((ret >= 0));
  if ( ret == 1 )
  {
    // pJ = (0)
    ret = point_set_infinity( pJ );
    BPCL_CHECK_ERROR((ret==0));

    // stop successful
    return 0;
  }

  // check P = (0)
  ret = point_is_zero( pA1 );
  BPCL_CHECK_ERROR((ret >= 0));
  if ( ret == 1 )
  {
    // pJ = (0)
    ret = point_set_infinity( pJ );
    BPCL_CHECK_ERROR((ret==0));

    // stop successful
    return 0;
  }

 
#if (BPCL_ECC_FEAT_DPA_PROTECTION == 1 )

  // init random_x (20-bit)
  random_x_word   = rnd_word( 0xFFFFF );
  random_x.sign   = C_POSITIVE;
  random_x.word   = &random_x_word;
  random_x.length = 1;
 
  // allocate auxiliary variable variable dpa_resistent_k
  field_element_allocate( &dpa_resistant_k, MAX ( cGP->base_point_order_n.length + 1, k->length ) );

  // dpa_resistent_k = x  n
  ret = field_element_multiply_classical( &dpa_resistant_k, &random_x, &cGP->base_point_order_n );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &dpa_resistant_k); 

  // dpa_resistent_k = k + dpa_resistent_k
  ret = field_element_add( &dpa_resistant_k, k, &dpa_resistant_k );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &dpa_resistant_k); 

  // calculate k_number = m / w (always round up!)
  k_number = (int) ( (dpa_resistant_k.length * C_NUM_BITS + (tU32) window_size - 1 ) / (tU32) window_size );

  // calculate mask = 2^(window_size)
  window_full = 2;
  for ( i = 1; i < (int) window_size; i++ )
  {
    window_full = window_full * 2;
  }
  
  // allocate k_window  with k_number elements
  k_window = (int *) M_MEM_ALLOC( k_number * sizeof(int) );
  BPCL_CHECKERROR_FREEMEM((k_window != NULL), "%f", &dpa_resistant_k); 

  // calculate k_window
  for ( i = 0; i < k_number; i++ )
  {
    // read window
    k_window[i] = (int)dpa_resistant_k.word[0] & (window_full - 1); /*lint !e613 */ /*Null pointer not intended*/

    // shift local_k to next window
    ret = field_element_shift_n_right( &dpa_resistant_k, &dpa_resistant_k, (tU32) window_size );
    BPCL_CHECKERROR_FREEMEM((ret == 0), "%m%f",k_window,&dpa_resistant_k); 	 
  }

  field_element_free( &dpa_resistant_k );

#else  

  // calculate k_number = m / w (always round up!)
  k_number = (int) ( (k->length * C_NUM_BITS + (tU32) window_size - 1 ) / (tU32) window_size );

  // calculate mask = 2^(window_size)
  window_full = 2;
  for ( i = 1; i < (int) window_size; i++ )
  {
    window_full = window_full * 2;
  }

  // allocate local_k
  field_element_allocate( &local_k, k->length );
  
  // local_k = k
  ret = field_element_assign( &local_k, k, k->length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &local_k);

  // allocate k_window  with k_number elements
  k_window = (int *) M_MEM_ALLOC( k_number * sizeof(int) );
  BPCL_CHECKERROR_FREEMEM((k_window != NULL), "%f", &local_k);

  // calculate k_window
  for ( i = 0; i < k_number; i++ )
  {
    // read window
    k_window[i] = (int) local_k.word[0] & (window_full - 1);

    // shift local_k to next window
    ret = field_element_shift_n_right( &local_k, &local_k, window_size );
	 BPCL_CHECKERROR_FREEMEM((ret == 0), "%m%f",k_window,&local_k);
  }

   // free local_k
  field_element_free( &local_k );
  BPCL_CHECK_ERROR((ret==0));

#endif

  // allocate premcomputed_P array
  premcomputed_P = (tsECPoint *) M_MEM_ALLOC ( window_full * sizeof(tsECPoint) );
  BPCL_CHECKERROR_FREEMEM((premcomputed_P != NULL), "%m", k_window);

  // allocate premcomputed_P array points
  for ( i = 0; i < window_full; i++ )
  {
    // allocate points
    ret = point_allocate( &premcomputed_P[i], cGP->prime_p.length, C_PT_JACOBIAN );/*lint !e613 */	/*Null pointer not intended*/
	 sprintf(caFmt,"%c%d%s",'%',i,"a%m%m");
    BPCL_CHECKERROR_FREEMEM((ret == 0), (tString)caFmt, 
                      &premcomputed_P[(i-1)], premcomputed_P, k_window);
  }
  
  // precompute premcomputed_P points
  ret = point_set_infinity( &premcomputed_P[0] );   /*lint !e613 */ /*Null pointer not intended*/ // 0 * P
  sprintf(caFmt,"%c%d%s",'%',window_full,"a%m%m");
  BPCL_CHECKERROR_FREEMEM((ret == 0), (tString)caFmt, 
                  &premcomputed_P[(window_full-1)], premcomputed_P, k_window);

  ret = point_convert_to_jacobian( &premcomputed_P[1], pA1, cGP );     /*lint !e613 */  /*Null pointer not intended*/ // 1 * P
  sprintf(caFmt,"%c%d%s",'%',window_full,"a%m%m");
  BPCL_CHECKERROR_FREEMEM((ret == 0), (tString)caFmt, 
                  &premcomputed_P[(window_full-1)], premcomputed_P, k_window);

  for ( i = 2; i < window_full; i++ )
  {
    ret = point_jacobian_add_mixed( &premcomputed_P[i], &premcomputed_P[i-1], pA1, cGP );  /*lint !e613 */ /*Null pointer not intended*/ // (2 .. windowfull-1) * P
    sprintf(caFmt,"%c%d%s",'%',window_full,"a%m%m");
    BPCL_CHECKERROR_FREEMEM((ret == 0), (tString)caFmt, 
                  &premcomputed_P[(window_full-1)], premcomputed_P, k_window);
  }
 
  // allocate pQ
  ret = point_allocate( &pQ, cGP->prime_p.length, C_PT_JACOBIAN );
  sprintf(caFmt,"%c%d%s",'%',window_full,"a%m%m");
  BPCL_CHECKERROR_FREEMEM((ret == 0), (tString)caFmt, 
                  &premcomputed_P[(window_full-1)], premcomputed_P, k_window);

  // pQ = (0)
  ret = point_set_infinity( &pQ );
  sprintf(caFmt,"%c%d%s",'%',window_full,"a%m%m%p");
  BPCL_CHECKERROR_FREEMEM((ret == 0), (tString)caFmt, 
             &premcomputed_P[(window_full-1)], premcomputed_P, k_window, &pQ);

  for ( i = (k_number-1); i >= 0; i-- )
  {
     
     // pQ = window_size  pQ
     for ( j = 0; j < (int) window_size; j++ )
     {
       ret = point_jacobian_double( &pQ, &pQ, cGP );
       sprintf(caFmt,"%c%d%s",'%',window_full,"a%m%m%p");
       BPCL_CHECKERROR_FREEMEM((ret == 0), (tString)caFmt, 
             &premcomputed_P[(window_full-1)], premcomputed_P, k_window, &pQ);
     }

     if ( k_window[i] != 0 )    /*lint !e613 */ /*Null pointer not intended*/
     {
       // pQ = pQ + k * premcomputed_P, with 0 <= k < window_size
       ret = point_jacobian_add( &pQ, &pQ, &premcomputed_P[k_window[i]], cGP ); /*lint !e613 */ /*Null pointer not intended*/
       sprintf(caFmt,"%c%d%s",'%',window_full,"a%m%m%p");
       BPCL_CHECKERROR_FREEMEM((ret == 0), (tString)caFmt, 
             &premcomputed_P[(window_full-1)], premcomputed_P, k_window, &pQ);
     }
#if ( BPCL_ECC_FEAT_SPA_PROTECTION == 1 )
     else
     {
       // dummy add against Simple Power Analysis (SPA)
       ret = point_jacobian_add( pJ, &pQ, &premcomputed_P[k_window[i]], cGP );  /*lint !e613 */ /*Null pointer not intended*/
       sprintf(caFmt,"%c%d%s",'%',window_full,"a%m%m%p");
       BPCL_CHECKERROR_FREEMEM((ret == 0), (tString)caFmt, 
             &premcomputed_P[(window_full-1)], premcomputed_P, k_window, &pQ); 
     }
#endif
    
  }

  // pJ = pQ
  ret = point_assign( pJ, &pQ );
  sprintf(caFmt,"%c%d%s",'%',window_full,"a%m%m%p");
  BPCL_CHECKERROR_FREEMEM((ret == 0), (tString)caFmt, 
            &premcomputed_P[(window_full-1)], premcomputed_P, k_window, &pQ);

  // free pQ
  ret = point_free( &pQ );
  sprintf(caFmt,"%c%d%s",'%',window_full,"a%m%m");
  BPCL_CHECKERROR_FREEMEM((ret == 0), (tString)caFmt, 
            &premcomputed_P[(window_full-1)], premcomputed_P, k_window);

  // free premcomputed_P array points
  for ( i = 0; i < window_full; i++ )
  {
    ret = point_free( &premcomputed_P[i] );	 /*lint !e613 */ /*Null pointer not intended*/
    /*Free memory of premcomputed_P from i+1 to window_full,
         if freeing of i fails*/
    sprintf(caFmt,"%c%d%s",'%',(i+1),"a%m%m");
    BPCL_CHECKERROR_FREEMEM((ret == 0), (tString)caFmt, 
            &premcomputed_P[(window_full-1)], premcomputed_P, k_window);
  }

  // free premcomputed_P array
  M_MEM_FREE( premcomputed_P );

  // free k_window
  M_MEM_FREE( k_window );

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F30/220: point_jacobian_multiply_fixed_window

   \par       DESCRIPTION

              This function does the scalar multiplication of known P with k

   \par       EQUATION / REFERENCE

              pJ = k  precomputed_P[0]

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *k             - pointer to field element k

   \param     const (t_precomputed_P) *precomputed_P - pointer to array of precomputed P

   \param     const (tsECCurve) *cGP               - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     (tsECPoint) *pJ                      - pointer to resulting jacobian projection point

   \par       RETURN VALUE

   \param     (int)  0                               - successful execution

   \param     (int) -1                               - parameter error

   \param     (int) -2                               - insufficient precomputed points P

   \par       NOTES / WARNINGS / TODO's
 
   \warning   algorithm needs at least "d = highest bit position of k[i]/window_size + 1" precomputed values!    

   \note      costs: (d  (2^(w) -1)/^2(w) - 1) + (2^(w) - 2)

   \note      special SPA / DPA resistant version

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int point_jacobian_multiply_fixed_window( tsECPoint     *pJ,
                                           const tsECFieldElement *k,
                                           const tsECPrecomputedP *precomputed_P,
                                           const tsECCurve     *cGP )
{
  // declarations
  int ret,d,i,j;
  tsECPoint pA,pB;

  int *k_d;
  int window_full;

#if (BPCL_ECC_FEAT_DPA_PROTECTION == 1 )

  tsECFieldElement dpa_resistant_k;
  tU32 random_x_word;
  tsECFieldElement random_x;

#else

  tsECFieldElement local_k;

#endif
  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (pJ == NULL) || (k == NULL) || ( precomputed_P == NULL) || (cGP == NULL) || 
       (k->length < 1) || (precomputed_P->length_precomputed_P < 1) ||
       (precomputed_P->window_size < (tU8) 2) || (precomputed_P->window_size > (tU8) 7)  )
  {
    return -1;
  }

#endif


  // set pJ type
  pJ->type = C_PT_JACOBIAN;

  // check k =? 0
  ret = field_element_is_zero( k );
  BPCL_CHECK_ERROR((ret >= 0));
  if ( ret == 1 )
  {
    // pJ = (0)
    ret = point_set_infinity( pJ );
    BPCL_CHECK_ERROR((ret==0));

    // stop successful
    return 0;
  }

  // calculate mask = 2^(window_size)
  window_full = 2;
  for ( i = 1; i < (int) precomputed_P->window_size; i++ )
  {
    window_full = window_full * 2;
  }


#if (BPCL_ECC_FEAT_DPA_PROTECTION == 1 )

  // init random_x (20-bit)
  random_x_word   = rnd_word( 0xFFFFF );
  random_x.sign   = C_POSITIVE;
  random_x.word   = &random_x_word;
  random_x.length = 1;
 
  // allocate auxiliary variable variable dpa_resistent_k
  field_element_allocate( &dpa_resistant_k, MAX( cGP->base_point_order_n.length + 1, k->length ) );

  // dpa_resistent_k = x  n
  ret = field_element_multiply_classical( &dpa_resistant_k, &random_x, &cGP->base_point_order_n );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &dpa_resistant_k);

  // dpa_resistent_k = k + dpa_resistent_k
  ret = field_element_add( &dpa_resistant_k, k, &dpa_resistant_k );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &dpa_resistant_k);

  // calculate d = m/w
  d = (int) ( (dpa_resistant_k.length * C_NUM_BITS) / (tU32) precomputed_P->window_size );

  // check for sufficient precomputed_P
  if ( d > (int) precomputed_P->length_precomputed_P ) return -2;

  // allocate k_d  with d elements
  k_d = (int *) M_MEM_ALLOC( d * sizeof(int) );
  BPCL_CHECKERROR_FREEMEM((k_d != NULL), "%f", &dpa_resistant_k);

  // calculate k_d
  for ( i = 0; i < d; i++ )
  {
    k_d[i] = (int) dpa_resistant_k.word[0] & (window_full - 1); /*lint !e613 */ /*Null pointer not intended*/

    ret = field_element_shift_n_right( &dpa_resistant_k, &dpa_resistant_k, precomputed_P->window_size );
	 BPCL_CHECKERROR_FREEMEM((ret==0), "%m%f", k_d, &dpa_resistant_k);
  }

  field_element_free( &dpa_resistant_k );

#else

  // allocate local_k
  field_element_allocate( &local_k, k->length );
  
  // local_k = k
  ret = field_element_assign( &local_k, k, k->length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &local_k);

  // calculate d = m/w
  d = (int) ( (k->length * C_NUM_BITS) / precomputed_P->window_size );

  // check for sufficient precomputed_P
  if ( d > (int) precomputed_P->length_precomputed_P ) return -2;

  // allocate k_d  with d elements
  k_d = (int *) M_MEM_ALLOC( d * sizeof(int) );
  BPCL_CHECKERROR_FREEMEM((k_d != NULL), "%f", &local_k);

  // calculate k_d
  for ( i = 0; i < d; i++ )
  {
    k_d[i] = (int) local_k.word[0] & (window_full - 1);

    ret = field_element_shift_n_right( &local_k, &local_k, precomputed_P->window_size );
	 BPCL_CHECKERROR_FREEMEM((ret==0), "%m%f", k_d, &local_k);
  }

  // free local_k
  field_element_free( &local_k );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%m", k_d);

#endif

  // allocate pA
  ret = point_allocate( &pA, cGP->prime_p.length, C_PT_JACOBIAN );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%m", k_d);

  // allocate pB
  ret = point_allocate( &pB, cGP->prime_p.length, C_PT_JACOBIAN );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p%m", &pA, k_d);

  // pA = (0)
  ret = point_set_infinity( &pA );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p%p%m", &pB, &pA, k_d);

  // pB = (0)
  ret = point_set_infinity( &pB );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p%p%m", &pB, &pA, k_d);
 
  // for j = 2^w - 1 downto 1
  for ( j = (window_full - 1); j > 0; j-- )
  {
    // for each k_d[i] = j
    for ( i = 0; i < d; i++ )
    {
      if ( k_d[i] == j ) /*lint !e613 */ /*Null pointer not intended*/
      {
        // pB = pB + precomputed_P[i]
        ret = point_jacobian_add_mixed( &pB, &pB, &precomputed_P->precomputed_P[i], cGP);
        BPCL_CHECKERROR_FREEMEM((ret==0), "%p%p%m", &pB, &pA, k_d);
      }
    }

    // pA = pA + pB
    ret = point_jacobian_add( &pA, &pA, &pB, cGP);
	 BPCL_CHECKERROR_FREEMEM((ret==0), "%p%p%m", &pB, &pA, k_d);
  }
  
  // return pJ = pA
  ret = point_assign( pJ, &pA );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p%p%m", &pB, &pA, k_d);

  // free pA
  ret = point_free( &pA );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p%m", &pB, k_d);

  // free pB
  ret = point_free( &pB );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%m", k_d);

  // free k_d
  M_MEM_FREE( k_d );

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F30/320: point_allocate

   \par       DESCRIPTION

              This function allocates memory for a point

   \par       EQUATION / REFERENCE

              None

   \par       INPUT ARGUMENTS

   \param     (t_point_projective) *pX         - pointer to point, that will be allocated

   \param     const (int) word_quantity        - number of words of field elements

   \param     const (tU8) point_type          - point type (affine, jacobian, none, .. )

   \par       OUTPUT ARGUMENTS

   \param     None

   \par       RETURN VALUE

   \param     (int)  0                         - successful execution

   \param     (int) -1                         - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      None
*/
/***************************************************************************/
extern int point_allocate( tsECPoint *pX,
                     const tU32        word_quantity,
                     const tU8         point_type )
{
  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (pX == NULL) || (word_quantity < 1) || 
       ((point_type != C_PT_AFFINE) && (point_type != C_PT_JACOBIAN)) )
  {
    return -1;
  }  

#endif

  // set type
  pX->type = point_type;

  // allocate words of field elements
  field_element_allocate( &pX->x, word_quantity );
  field_element_allocate( &pX->y, word_quantity );
  field_element_allocate( &pX->z, word_quantity );

  // successful
  return 0;
}




/***************************************************************************/
/*!
   \par       FUNCTION F30/330: point_free

   \par       DESCRIPTION

              This function frees memory for of a point

   \par       EQUATION / REFERENCE

              None

   \par       INPUT ARGUMENTS

   \param     (t_point_projective) *pX         - pointer to point, that will freed

   \par       OUTPUT ARGUMENTS

   \param     None

   \par       RETURN VALUE

   \param     (int)  0                        - successful execution

   \param     (int) -1                        - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      None
*/
/***************************************************************************/
extern int point_free( tsECPoint *pX )
{

#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( pX == NULL )
  {
    return -1;
  }  

#endif

  // clear type
  pX->type = C_PT_NONE;

  // free allocated words of field elements
  field_element_free( &pX->x );
  field_element_free( &pX->y );
  field_element_free( &pX->z );

  // successful
  return 0;
}




/***************************************************************************
* 5. END                                                                   *    
****************************************************************************/

