/***************************************************************************/
/*! 
    \file        gfp_ecdh_ecdsa.c

    \brief       Elliptic Curve Diffie/Hellmann (ECDH) and
                 Elliptic Curve Digital Signature Algorithm (ECDSA)

    \author      Marko Wolf (mwolf@crypto.rub.de),
                 Andr Weimerskirch (weika@crypto.rub.de)
				 Extended Laura Heinrich-Litan
				 Adopted to BPCL CM-DI/ESN-Listle
    
    \version     1.5
    
    \date        September 6, 2004

    \warning     None
    
    \par Reviews:
    \li          06.09.2004 Splint 3.1.1 (-weak no warnings)

    \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error

*/
/***************************************************************************/

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/

#include "bpcl.h"
#include "bpcl_int.h"
#include "ecc.h"

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/

/***************************************************************************
 * 4. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/


/***************************************************************************/
/*!
   \par       FUNCTION F40/10: ecc_public_key_generation

   \par       DESCRIPTION

              This function generates the corresponding public key

   \par       EQUATION / REFERENCE

              Q = d  G

              Public Key:  Q
              
              Private Key: d

              Basepoint:   G

   \par       INPUT ARGUMENTS

   \param     const tsECFieldElement *d     - pointer to private key field element d
   
   \param     const (tsECCurve)  *cGP    - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     (tsECPoint) *pQ            - pointer to public key point (affine)

   \par       RETURN VALUE

   \param     (int)  0                     - successful execution

   \param     (int) -1                     - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int ecc_public_key_generation( tsECPoint     *pQ,  /* result: public key */
                                const tsECFieldElement *d,   /* app. private key   */
                                const tsECCurve     *cGP )/* ECC curve over GP  */
{

  // declarations
  int ret;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (pQ == NULL) || (d == NULL) || (cGP == NULL) ||
       (d->length < 1) || (pQ->x.length < cGP->prime_p.length) )
  {
    return -1;
  }

#endif

  // calculate public key pjQ = d  G mod p
  ret = point_jacobian_multiply_k_ary_window( pQ, d, &cGP->base_point_G, (tU8) 4, cGP );
  BPCL_CHECK_ERROR((ret==0));

  // convert jacobian pQ into affine pQ mod p
  ret = point_convert_to_affine( pQ, pQ, cGP );
  BPCL_CHECK_ERROR((ret==0));
 
  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F40/20: ecc_public_key_validation

   \par       DESCRIPTION

              This function validates the public key

   \par       EQUATION / REFERENCE

              Check, that:

              A.) Q != (0)

              B.) Qx & Qy are properly represented elements in GF(p)

              C.) Q lies on elliptic curve defined by a & b

              D.) Q  n = (0)

   \par       INPUT ARGUMENTS

   \param     (tsECPoint) *pQ            - pointer to public key point (affine)

   \param     const (tsECCurve)  *cGP    - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \par       None

   \par       RETURN VALUE

   \param     (int)  0                     - successful execution

   \param     (int) -1                     - parameter error

   \param     (int) -2                     - invalid public key

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int ecc_public_key_validation( const tsECPoint *pQ,   /* public key for validating */
                                      const tsECCurve *cGP ) /* ECC curve over GP         */
{
  // declarations
  int ret,res;
  tsECFieldElement T1;
  tsECFieldElement T2;
  tsECPoint pX;


#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (pQ == NULL) || (cGP == NULL) || (cGP->ecc_field_params == NULL) || 
       (pQ->type != C_PT_AFFINE) )
  {
    return -1;
  }

#endif


  // init res
  res = 0;

  // A.) check Q != (0)
  ret = point_is_zero( pQ );
  if ( ret != 0)
  {
    return -2;
  }

  // B.) check that Q(x,y) consists of properly represented elements of GF(p)
  
  // 0 < pQ->x < p
  ret = field_element_absolute_compare( &pQ->x, &cGP->prime_p );
  if ( (ret != -1) || (pQ->x.sign != C_POSITIVE) )
  {
    return -2;
  }

  // 0 < pQ->y < p
  ret = field_element_absolute_compare( &pQ->y, &cGP->prime_p );
  if ( (ret != -1) || (pQ->y.sign != C_POSITIVE) )
  {
    return -2;
  }


  // C.) check Q on curve cGP: y = x + a  x + b
  
  // allocate T1 & T2
  field_element_allocate( &T1, cGP->prime_p.length );
  field_element_allocate( &T2, cGP->prime_p.length );

  // 01.) T1 = x mod p
  ret = field_element_modular_square( &T1, &pQ->x, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f",&T1,&T2);

  // 02.) T1 = x mod p
  ret = field_element_modular_multiply( &T1, &T1, &pQ->x, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f",&T1,&T2);

  // 03.) T2 = a  x mod p
  ret = field_element_modular_multiply( &T2, &cGP->coefficient_a, &pQ->x, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f",&T1,&T2);

  // 04.) T1 = T1 + T2 <=> x + a  x mod p
  ret = field_element_modular_add( &T1, &T1, &T2, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f",&T1,&T2);

  // 05.) T1 = T1 + b <=> x + a  x + b mod p
  ret = field_element_modular_add( &T1, &T1,  &cGP->coefficient_b, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f",&T1,&T2);

  // 06.) T2 = y mod p
  ret = field_element_modular_square( &T2, &pQ->y, cGP->ecc_field_params );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f",&T1,&T2);

  // 07.) T1 =? T2 <=> y =? x + a  x + b mod p
  ret = field_element_absolute_compare( &T1, &T2 );
  if ( ret != 0)
  {
    res = -2;
  }

  // D.) check that n  Q = (0) mod p


  // allocate pX
  ret = point_allocate( &pX, cGP->prime_p.length, C_PT_JACOBIAN );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f",&T1,&T2);

  ret = point_jacobian_multiply_k_ary_window( &pX, &cGP->base_point_order_n, pQ, (tU8) 4, cGP );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%p",&T1,&T2,&pX);
  ret = point_convert_to_affine( &pX, &pX, cGP );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%p",&T1,&T2,&pX);

  ret = point_is_zero( &pX );
  if ( ret != 1)
  {
    res = -2;
  }
 
  // free pX
  ret = point_free( &pX );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f",&T1,&T2);

  // free T1 & T2
  field_element_free( &T1 );
  field_element_free( &T2 );
  
  // return res
  return res;

}



/***************************************************************************/
/*!
   \par       FUNCTION F40/30: ecdh_compute_shared_secret

   \par       DESCRIPTION

              This function computes the shared secret z based on ECDH

   \par       EQUATION / REFERENCE

              P = dA  QB

              Shared Secret:    z = P->x
              
              Private Key of A: dA

              Public Key of B:  QB

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *dA  - pointer to private key field element d of A

   \parm      const (tsECPoint) *pQB     - pointer to public key point of B (affine)

   \param     const (tsECCurve) *cGP     - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *z         - pointer to shared secret field element z

   \par       RETURN VALUE

   \param     (int)  0                     - successful execution

   \param     (int) -1                     - parameter error

   \param     (int) -2                     - z == 0 (invalid dA or QB)

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int ecdh_compute_shared_secret( tsECFieldElement *z,    /* result: shared secret  */
                                 const tsECFieldElement *dA,   /* A's private key        */
                                 const tsECPoint     *pQB,  /* B's public key         */ 
                                 const tsECCurve     *cGP ) /* ECC curve over GP      */
{

  // declarations
  int ret;
  tsECPoint pX;


#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (z == NULL) || (dA == NULL) || (pQB == NULL) || (cGP == NULL) ||
       (pQB->type != C_PT_AFFINE) )
  {
    return -1;
  }

#endif


  // allocate pX
  ret = point_allocate( &pX, cGP->prime_p.length, C_PT_JACOBIAN );
  BPCL_CHECK_ERROR((ret==0));
  
  // 01.) pX = dA  QB mod p
  ret = point_jacobian_multiply_k_ary_window( &pX, dA, pQB, (tU8) 4, cGP );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p",&pX);

  // 02.) check pX != (0) mod p
  ret = point_convert_to_affine( &pX, &pX, cGP );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p",&pX);
  
  ret = point_is_zero( &pX );
  if ( ret != 0 )
  {
    ret = -2;
  }

  // 03.) shared secret value z = pX->x
  ret = field_element_assign( z, &pX.x, pX.x.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p",&pX);

  // free pX
  ret = point_free( &pX );
  BPCL_CHECK_ERROR((ret==0));

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION: ForWIPRO_ecdh_compute_shared_secret

   \par       DESCRIPTION

              This function computes the shared secret z based on ECDH

   \par       EQUATION / REFERENCE

              P = rnd_A  DH_B

              Shared Secret:    z = P->x
              
              Random number of A: rnd_A

              Diffie-Hellman first phase value:  DH_B

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *rnd_A - pointer to random number of A

   \parm      const (tsECPoint) *pDH_B   - pointer to Diffie-Hellman first phase value of B
											 (affine elliptic point)

   \param     const (tsECCurve) *cGP     - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *z         - pointer to shared secret field element z

   \par       RETURN VALUE

   \param     (int)  0                     - successful execution

   \param     (int) -1                     - parameter error

   \param     (int) -2                     - z == 0 (invalid rnd_A or DH_B)

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int ForWipro_ecdh_compute_shared_secret( tsECFieldElement *z, /* result: shared secret    */
                                 const tsECFieldElement *rnd_A,   /* A's random number        */
                                 const tsECPoint     *pDH_B,   /* B's DH first phase value */ 
                                 const tsECCurve     *cGP )    /* ECC curve over GP        */
{

  // declarations
  int ret;
  tsECPoint pX;


#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (z == NULL) || (rnd_A == NULL) || (pDH_B == NULL) || (cGP == NULL) ||
       (pDH_B->type != C_PT_AFFINE) )
  {
    return -1;
  }

#endif


  // allocate pX
  ret = point_allocate( &pX, cGP->prime_p.length, C_PT_JACOBIAN );
  BPCL_CHECK_ERROR((ret==0));
  
  // 01.) pX = rnd_A  QB mod p
  ret = point_jacobian_multiply_k_ary_window( &pX, rnd_A, pDH_B, (tU8) 4, cGP );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p",&pX);

  // 02.) check pX != (0) mod p
  ret = point_convert_to_affine( &pX, &pX, cGP );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p",&pX);
  
  ret = point_is_zero( &pX );
  if ( ret != 0 )
  {
    ret = -2;
  }

  // 03.) shared secret value z = pX->x
  ret = field_element_assign( z, &pX.x, pX.x.length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p",&pX);

  // free pX
  ret = point_free( &pX );
  BPCL_CHECK_ERROR((ret==0));

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION: ForWIPRO_ecdh_compute_first_phase_value

   \par       DESCRIPTION

              This function computes the first phase value of A based on ECDH

   \par       EQUATION / REFERENCE

              DH_A = rnd_A  G
            
              Random number of A: rnd_A

			  Basepoint of EC curve: G

              Diffie-Hellman first phase value:  DH_A

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *rnd_A - pointer to random number of A

   \param     const (tsECPrecomputedP) *precomputed_G - array of precalculated base points G

   \param     const (tsECCurve) *cGP     - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \parm      const (tsECPoint) *pDH_A   - pointer to Diffie-Hellman first phase value of A
										     !allocated as Jacobian point!
											 !return as affine elliptic point!
   \par       RETURN VALUE

   \param     (int)  0                     - successful execution

   \param     (int) -1                     - parameter error

   \param     (int) -2                     - z == 0 (invalid rnd_A or G)

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int ForWipro_ecdh_compute_first_phase_value(
									   tsECPoint	   *pDH_A,   /* result: A's DH 1. phase value */
                                 const tsECFieldElement *rnd_A,		   /* A's random number       */
                                 const tsECPrecomputedP *precomputed_G, /* precomputed base points */
                                 const tsECCurve     *cGP )          /* ECC curve over GP       */
{

  // declarations
  int ret;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (rnd_A == NULL) || (pDH_A == NULL) || (precomputed_G == NULL) || (cGP == NULL) ||
       (pDH_A->type != C_PT_JACOBIAN) )
  {
    return -1;
  }

#endif


  // 01.) pDH_A = rnd_A  G mod p  
  ret = point_jacobian_multiply_fixed_window( pDH_A, rnd_A, precomputed_G, cGP );
  BPCL_CHECK_ERROR((ret==0));

  // 02.) check pX != (0) mod p
  ret = point_convert_to_affine( pDH_A, pDH_A, cGP );
  BPCL_CHECK_ERROR((ret==0));
  
  ret = point_is_zero( pDH_A);
  if ( ret != 0 )
  {
    ret = -2;
  }
  
  // successful
  return 0;
}


/***************************************************************************/
/*!
   \par       FUNCTION F40/40: ecdsa_signature_generation

   \par       DESCRIPTION

              This function generates the corresponding signature for a hashed message

   \par       EQUATION / REFERENCE

              None

   \par       INPUT ARGUMENTS
   
   \param     const (tsECFieldElement) *hash_msg      - pointer to field element hashed signing message

   \param     const (tsECFieldElement) *d             - pointer to private key field element d

   \param     const (tsECFieldElement) *k             - pointer to field element unpredictable integer k

   \param     const (tsECPrecomputedP) *precomputed_G - array of precalculated base points G

   \param     const (tsECCurve) *cGP               - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     (tsECSignature) *sig                 - pointer to ECC signature

   \par       RETURN VALUE

   \param     (int)  0                               - successful execution

   \param     (int) -1                               - parameter error

   \param     (int) -2                               - invalid k

   \par       NOTES / WARNINGS / TODO's

   \warning   fptr_field_element_reduce = general reduce!

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int ecdsa_signature_generation( tsECSignature *ecc_sig,       /* result: ECC signature   */  
                                 const tsECFieldElement *msg_hash,      /* hash of signing message */
                                 const tsECFieldElement *d,             /* app. private key        */
                                 const tsECFieldElement *k,             /* unpredictable integer   */
                                 const tsECPrecomputedP *precomputed_G, /* precomputed base points */
                                 const tsECCurve     *cGP )          /* ECC curve over GP       */
{
  // declarations
  int ret;
  tsECPoint pX;
  tsECFieldElement fe_k_inv;
  tsECField field_mod_n;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (ecc_sig == NULL) || (msg_hash == NULL) || (d == NULL) || (k == NULL) || (precomputed_G == NULL) || (cGP == NULL) )
  {
    return -1;
  }

#endif

  // init field_mod_n
  field_mod_n.fptr_fe_multiply = &field_element_multiply_classical;
  field_mod_n.fptr_fe_square   = &field_element_square_classical;
  field_mod_n.fptr_fe_reduce   = &field_element_reduce_barrett;
  field_mod_n.precalc_my_p     = cGP->precalc_my_n;
  field_mod_n.prime_p          = cGP->base_point_order_n;


  // 1.) select a statistically unique and unpredictable integer k in the interval [1, n  1]

  // 2.) compute k  G = ( x1 , y1 ) 

  // allocate pX
  ret = point_allocate( &pX, cGP->prime_p.length, C_PT_JACOBIAN );
  BPCL_CHECK_ERROR((ret==0));

  // pX = k  G mod p  
  ret = point_jacobian_multiply_fixed_window( &pX, k, precomputed_G, cGP );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p",&pX);
  ret = point_convert_to_affine( &pX, &pX, cGP );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p",&pX);

  // r = pX.x1 mod n
  ret = field_element_reduce_barrett( &ecc_sig->r , &pX.x, &cGP->precalc_my_n, &cGP->base_point_order_n );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p",&pX);

  // if r = 0, then go to step 1 (because s = k^(l)  { h(m) + d  r} mod n does not involve the private key d!
  ret = field_element_is_zero( &ecc_sig->r );
  BPCL_CHECKERROR_FREEMEM((ret > -1), "%p",&pX);
  if ( ret == 1 )
  {
    return -2;
  }

  // 3.) compute k^(1) mod n

  // allocate fe_k_inv
  field_element_allocate( &fe_k_inv, cGP->base_point_order_n.length );

  // fe_k = k^(-1) mod n
  ret = field_element_modular_invert( &fe_k_inv, k, &field_mod_n );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p%f",&pX,&fe_k_inv);

  // 4.) compute s = k^(l)  { h(m) + d  r} mod n (where h is the Secure Hash Algorithm SHAl)

  // s = d  r mod n
  
  ret = field_element_modular_multiply( &ecc_sig->s, d, &ecc_sig->r, &field_mod_n );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p%f",&pX,&fe_k_inv);

  // s = h(m) + s mod n
  ret = field_element_modular_add( &ecc_sig->s, msg_hash, &ecc_sig->s, &field_mod_n );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p%f",&pX,&fe_k_inv);

  // s = k^(l)  s mod n 
  ret = field_element_modular_multiply( &ecc_sig->s, &fe_k_inv, &ecc_sig->s, &field_mod_n );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%p%f",&pX,&fe_k_inv);


  // 5.) if s = 0, then go to step 1 (if s = 0, then s^(1) mod n does not exist, but s^(1) is required in step 2 of signature verification) 
  ret = field_element_is_zero( &ecc_sig->s );
  BPCL_CHECKERROR_FREEMEM((ret > -1), "%p%f",&pX,&fe_k_inv);
  if ( ret == 1 )
  {
    return -2;
  }

  // 6.) signature for the message m is the pair of integers (r, s) 

  
  // free field elements
  field_element_free( &fe_k_inv );
  
  // free points
  ret = point_free( &pX );
  BPCL_CHECK_ERROR((ret==0));

  // successful
  return 0;

}


/***************************************************************************/
/*!
   \par       FUNCTION F40/50: ecdsa_signature_verification

   \par       DESCRIPTION

              This function verifies the corresponding signature for a hashed message

   \par       EQUATION / REFERENCE

              None

   \par       INPUT ARGUMENTS
   
   \param     const (tsECFieldElement) *hash_msg      - pointer to field element hashed signing message

   \param     const (tsECSignature) *sig           - pointer to ECC signature 

   \param     const (tsECPoint) *pQ                - pointer to public key point Q

   \param     const (tsECPrecomputedP) *precomputed_G - array of precalculated base points G

   \param     const (tsECCurve) *cGP               - pointer to elliptic curve parameters

   \par       OUTPUT ARGUMENTS

   \param     None

   \par       RETURN VALUE

   \param     (int)  0                               - successful verification

   \param     (int) -1                               - parameter error

   \param     (int) -2                               - signature failed

   \par       NOTES / WARNINGS / TODO's

   \warning   fptr_field_element_reduce = general reduce!

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int ecdsa_signature_verification( const tsECFieldElement *msg_hash,      /* hash of signing message */
                                         const tsECSignature *ecc_sig,       /* ECC signature           */
                                         const tsECPoint     *pQ,            /* public key Q            */
                                         const tsECPrecomputedP *precomputed_G, /* precomputed base points */
                                         const tsECCurve     *cGP )          /* ECC curve over GP       */
{

  // declarations
  int ret;
  tsECFieldElement w,u1,u2,v;
  tsECPoint pX1,pX2;
  tsECField field_mod_n;

  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if ( (msg_hash == NULL) || (ecc_sig == NULL) || (pQ == NULL) || (precomputed_G == NULL) || (cGP == NULL) )
  {
    return -1;
  }

#endif


  // init field_mod_n
  field_mod_n.fptr_fe_multiply = &field_element_multiply_classical;
  field_mod_n.fptr_fe_square   = &field_element_square_classical;
  field_mod_n.fptr_fe_reduce   = &field_element_reduce_barrett;
  field_mod_n.precalc_my_p     = cGP->precalc_my_n;
  field_mod_n.prime_p          = cGP->base_point_order_n;

  // 1.) obtain an authentic copy of A's public key (E,G,n,Q)
  
  // 2.) verify that r and s are integers in the interval [ 1, n - 1 ]
  
  // check r
  ret = field_element_absolute_compare( &ecc_sig->r, &cGP->base_point_order_n );
  BPCL_CHECK_ERROR((ret > -2));

  if ( ret != -1 )
  {
    // r >= n
    return -2;
  }

  // check s
  ret = field_element_absolute_compare( &ecc_sig->s, &cGP->base_point_order_n );
  BPCL_CHECK_ERROR((ret > -2));

  if ( ret != -1 )
  {
    // s >= n
    return -2;
  }


  // allocate field elements
  field_element_allocate( &w, cGP->base_point_order_n.length );
  field_element_allocate( &u1, cGP->base_point_order_n.length );
  field_element_allocate( &u2, cGP->base_point_order_n.length );
  field_element_allocate( &v, cGP->base_point_order_n.length );

  // allocate points
  ret = point_allocate( &pX1, cGP->prime_p.length, C_PT_JACOBIAN );
  BPCL_CHECKERROR_FREEMEM((ret == 0), "%f%f%f%f",&w,&u1,&u2,&v);
  ret = point_allocate( &pX2, cGP->prime_p.length, C_PT_JACOBIAN );
  BPCL_CHECKERROR_FREEMEM((ret == 0), "%f%f%f%f%p",&w,&u1,&u2,&v,&pX1);

  // 3.) compute w = s^(-1) mod n
  ret = field_element_modular_invert( &w, &ecc_sig->s, &field_mod_n );
  BPCL_CHECKERROR_FREEMEM((ret == 0), "%f%f%f%f%p%p",&w,&u1,&u2,&v,&pX1,&pX2);

  // 4.a) compute u1 = h(m)  w mod n
  ret = field_element_modular_multiply( &u1, msg_hash, &w, &field_mod_n );
  BPCL_CHECKERROR_FREEMEM((ret == 0), "%f%f%f%f%p%p",&w,&u1,&u2,&v,&pX1,&pX2);
 
  // 4.b) compute u2 = r  w mod n
  ret = field_element_modular_multiply( &u2, &ecc_sig->r, &w, &field_mod_n);
  BPCL_CHECKERROR_FREEMEM((ret == 0), "%f%f%f%f%p%p",&w,&u1,&u2,&v,&pX1,&pX2);

  // 5.a) compute u1  G + u2  Q = ( x0, y0 )
  ret = point_jacobian_multiply_fixed_window( &pX1, &u1, precomputed_G, cGP );
  BPCL_CHECKERROR_FREEMEM((ret == 0), "%f%f%f%f%p%p",&w,&u1,&u2,&v,&pX1,&pX2);

  ret = point_jacobian_multiply_k_ary_window( &pX2, &u2, pQ, (tU8) 4, cGP );
  BPCL_CHECKERROR_FREEMEM((ret == 0), "%f%f%f%f%p%p",&w,&u1,&u2,&v,&pX1,&pX2);

  ret = point_jacobian_add( &pX1, &pX1, &pX2, cGP );
  BPCL_CHECKERROR_FREEMEM((ret == 0), "%f%f%f%f%p%p",&w,&u1,&u2,&v,&pX1,&pX2);

  ret = point_convert_to_affine( &pX1, &pX1, cGP );
  BPCL_CHECKERROR_FREEMEM((ret == 0), "%f%f%f%f%p%p",&w,&u1,&u2,&v,&pX1,&pX2);
  
  // 5.b) compute v = x0 mod n
  ret = field_element_reduce_barrett( &v , &pX1.x, &cGP->precalc_my_n, &cGP->base_point_order_n );
  BPCL_CHECKERROR_FREEMEM((ret == 0), "%f%f%f%f%p%p",&w,&u1,&u2,&v,&pX1,&pX2);
  
  // 6.) accept signature if and only if v = r
  ret = field_element_absolute_compare( &v, &ecc_sig->r );
  BPCL_CHECKERROR_FREEMEM((ret > -2), "%f%f%f%f%p%p",&w,&u1,&u2,&v,&pX1,&pX2);

  if ( ret != 0 )
  {
    // free field elements
    field_element_free( &w );
    field_element_free( &u1 );
    field_element_free( &u2 );
    field_element_free( &v );

    // free points
    ret = point_free( &pX1 );
    BPCL_CHECKERROR_FREEMEM((ret == 0), "%p",&pX2);
    ret = point_free( &pX2 );
    BPCL_CHECK_ERROR((ret == 0));

    // v != r
    return -2;
  }

  // free field elements
  field_element_free( &w );
  field_element_free( &u1 );
  field_element_free( &u2 );
  field_element_free( &v );

  // free points
  ret = point_free( &pX1 );
  BPCL_CHECKERROR_FREEMEM((ret == 0), "%p",&pX2);
  ret = point_free( &pX2 );
  BPCL_CHECK_ERROR((ret == 0));

  // successful
  return 0;
}


/***************************************************************************
 * 5. END                                                                  *
 ***************************************************************************/


