//-----------------------------------------------------------------------------
//
//  RC5.C
//
//-----------------------------------------------------------------------------
//
//  Implementation of Block cipher RC5
//
//  Copyright (c) 2006 Blaupunkt GmbH, Hildesheim
//
//  Author: Holger Listle, CM-DI/ESN
//
//-----------------------------------------------------------------------------

#include "bpcl.h"
#include "bpcl_int.h"

#define  C_RC5_ROUNDS		(12)
#define  C_RC5_KEYSIZE		(16)
#define  C_RC5_TABLESIZE	(26)

#define  C_RC5_MAGIC_P		(0xb7e15163)
#define  C_RC5_MAGIC_Q		(0x9e3779b9)
tU8 global_key_mask3[] = { 0xc6, 0xcf, 0xb9, 0x1c, 0x01, 0xd5, 0xc6, 0xdb, 0x50, 0x83, };

tU32	S[C_RC5_TABLESIZE];

//-----------------------------------------------------------------------------
// BPCL_RC5_Setup()
//-----------------------------------------------------------------------------
void BPCL_RC5_Setup(tU8 *p_key) {

	tU32	i, j, k, A, B, L[4];
	
	for(i = C_RC5_KEYSIZE-1, L[3] = 0; ((tS32)i) != -1; i--) {
		L[i/4] = (L[i/4] << 8) + p_key[i];
	}

	for(S[0] = C_RC5_MAGIC_P, i = 1; i < C_RC5_TABLESIZE; i++) {
		S[i] = S[i-1] + C_RC5_MAGIC_Q;
	}

	for(A=B=i=j=k=0; 
		k < 3*C_RC5_TABLESIZE;
		k++, i = (i+1) % C_RC5_TABLESIZE, j= (j+1) % 4) {
		A = S[i] = M_ROTL(S[i] + (A+B), 3);  
		B = L[j] = M_ROTL(L[j] + (A+B), (A+B)); 
	} 
} /* BPCL_RC5_Setup() */

//-----------------------------------------------------------------------------
// BPCL_RC5_Encrypt()
//-----------------------------------------------------------------------------
tS32 BPCL_RC5_Encrypt(tU32 *p_plain, tU32 *p_cipher, tU32 data_len) {

	tU32	round, block;
	tU32	A, B;

	if(data_len % 8) {
		M_DEBUG_PRINT("BPCL_RC5_Encrypt(): data_len is not a multiple of 8")
		return BPCL_ERR_BAD_PARAMETER;
	}

	data_len /= 4;

	for(block = 0; block < data_len; block += 2) {
		A = p_plain[block]   + S[0];
		B = p_plain[block+1] + S[1];
		for(round=1; round <= C_RC5_ROUNDS; ++round) {
			A = M_ROTL(A^B, B) + S[2*round]; 
			B = M_ROTL(B^A, A) + S[2*round+1];
		}
		p_cipher[block]   = A;
		p_cipher[block+1] = B;
	}

	return BPCL_OK;
} /* BPCL_RC5_Encrypt() */

//-----------------------------------------------------------------------------
// BPCL_RC5_Decrypt()
//-----------------------------------------------------------------------------
tS32 BPCL_RC5_Decrypt(tU32 *p_cipher, tU32 *p_plain, tU32 data_len) {

	tU32	round, block;
	tU32	A, B;

	if(data_len % 8) {
		M_DEBUG_PRINT("BPCL_RC5_Decrypt(): data_len is not a multiple of 8")
		return BPCL_ERR_BAD_PARAMETER;
	}

	data_len /= 4;

	for(block = 0; block < data_len; block += 2) {
		A = p_cipher[block];
		B = p_cipher[block+1];
		for(round=C_RC5_ROUNDS; round > 0; --round) {
			B = M_ROTR(B-S[2*round+1], A) ^ A;
			A = M_ROTR(A-S[2*round],   B) ^ B; 
		}
		p_plain[0] = A-S[0];
		p_plain[1] = B-S[1];
	}
	return BPCL_OK;
} /* BPCL_RC5_Decrypt() */


#ifdef _BPCL_TEST

//-----------------------------------------------------------------------------
//	Test implementation
//-----------------------------------------------------------------------------

tErrCode BPCL_RC5_Test(tU8 test_type) {

	tU32	i;
	tU32	res1[2], res2[2];
	tU32	buffer[4096] = { 0L };	/* 16kB for performance tests */
	tErrCode  rc = BPCL_OK;


	struct {
		tU8  key[C_RC5_KEYSIZE];
		tU32 plain[2];
		tU32 cipher[2];
	} test_data[3] = {
		{
			{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
			{ 0x00000000, 0x00000000 },
			{ 0xEEDBA521, 0x6D8F4B15 }
		}, {
			{ 0x91, 0x5F, 0x46, 0x19, 0xBE, 0x41, 0xB2, 0x51,
			  0x63, 0x55, 0xA5, 0x01, 0x10, 0xA9, 0xCE, 0x91 },
			{ 0xEEDBA521, 0x6D8F4B15 },
			{ 0xAC13C0F7, 0x52892B5B }
		}, {
			{ 0x78, 0x33, 0x48, 0xE7, 0x5A, 0xEB, 0x0F, 0x2F,
			  0xD7, 0xB1, 0x69, 0xBB, 0x8D, 0xC1, 0x67, 0x87 },
			{ 0xAC13C0F7, 0x52892B5B },
			{ 0xB7B3422F, 0x92FC6903 }
		}
	};

	printf("\n----------------------------------------------------------\n");
	printf("             RC5 Test\n");
	printf("----------------------------------------------------------\n");

	if(test_type & BPCL_TEST_COMPLIANCE) {
		printf("\nRC5 Compliance Test\n");
		for(i=0; i<3; ++i) {
			BPCL_RC5_Setup(test_data[i].key);
			BPCL_RC5_Encrypt(test_data[i].plain, res1, 8);
			BPCL_RC5_Decrypt(res1, res2, 8);
			if( (res1[0] != test_data[i].cipher[0] || res1[1] != test_data[i].cipher[1]) ||
				(res2[0] != test_data[i].plain[0]  || res2[1] != test_data[i].plain[1]) ) {
				printf("RC5 Compliance Test #%d failed:\n", i);
				printf("\tIn:   0x%08lx 0x%08lx\n", test_data[i].plain[0], test_data[i].plain[1]);
				printf("\tOut1: 0x%08lx 0x%08lx\n", res1[0], res1[1]);
				printf("\tOut2: 0x%08lx 0x%08lx\n", res2[0], res2[1]);
				rc = BPCL_ERR_TEST_FAILED;
			} else {
				printf("RC5 Compliance Test #%d okay\n", i);
			}
		}
		printf("\nRC5 Compliance Test DONE\n");
	}
	
	if(test_type & BPCL_TEST_PERFORMANCE) {

		printf("\nRC5 Performance Test\n");

		BPCL_RC5_Setup(test_data[0].key);

		BPCL_TimerStart();
		for(i = 0; i < 640; ++i) {
			/* Loop over 10MB of data (640 * 16kB) */
			BPCL_RC5_Encrypt(buffer, buffer, 16384);
		}
		BPCL_TimerReport("RC5 Encryption@10MB");

		BPCL_TimerStart();
		for(i = 0; i < 640; ++i) {
			/* Loop over 10MB of data (640 * 16kB) */
			BPCL_RC5_Decrypt(buffer, buffer, 16384);
		}
		BPCL_TimerReport("RC5 Decryption@10MB");

		printf("\nRC5 Performance Test DONE\n");
	}
	return rc;
} /* BPCL_RC5_Test() */

#endif /* _BPCL_TEST */
