/************************************************************************
 * FILE:         ahlcfg_tclTrace.h
 * PROJECT:      VW_LL_NF
 * SW-COMPONENT: ahl_lib
 *----------------------------------------------------------------------
 *
 * DESCRIPTION: handle trace outputs efficiently  
 * This class is used as template for all middleware server to provide 
 * trace output in an efficient way.
 *
 * Places where the code may be enhanced is marked with:    
 *  / * +++  Enter your private Data and functions here +++ * /
 * 
 * copy header and cpp into your directory and replace "ahlcfg_" at all
 * occrences.
 *
 *----------------------------------------------------------------------
 * COPYRIGHT:    (c) 2007 Robert Bosch GmbH, Hildesheim
 * HISTORY:      
 * Date      | Author            | Modification
 * 2007-09-10| M. Hessling       | initial version 
 *************************************************************************/
#ifndef AHLCFG_TCLTRACE_H
#define AHLCFG_TCLTRACE_H

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
/* MS VC2005 Info
FATAL_M_ASSERT_ALWAYS() is defined in x:\di_osal\common\osal\include\osansi.h
IntelliSense does not work in C++ projects under the following circumstances:
- The cursor is inside a code comment.
- You are writing a string literal.
- A syntax error appears above the cursor.
- IntelliSense is not _fully_ supported under the following circumstances:
  When you reference a header file multiple times using the #include directive, 
  and the meaning of that header file changes due to various macro states (which 
  are defined through the #define directive). In other words, when you include a 
  header file several times whose usage changes under different macro states, 
  IntelliSense will not always work.
*/

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"       

/**
   This class encapsulates all trace comunication.
   ET_TRACE module defines the following level
   TR_LEVEL_FATAL       = 0,  
   TR_LEVEL_ERRORS      = 1,
   TR_LEVEL_SYSTEM_MIN  = 2,
   TR_LEVEL_SYSTEM      = 3,
   TR_LEVEL_COMPONENT   = 4,
   TR_LEVEL_USER_1      = 5,
   TR_LEVEL_USER_2      = 6,
   TR_LEVEL_USER_3      = 7,
   TR_LEVEL_USER_4      = 8

; General Message Structure for this trace messages
; Byte 0 = Length of message
; Byte 1 = TraceComponentNumber / also ClassNumber hi-byte
; Byte 2 = ClassNumber   
; Byte 3 = TRACE LEVEL as define above
; Byte 4 = Statistic (trace specific value)
; Byte 5 = Reserved (trace specific value)
; Byte 6 = FunctionName enum for Names of functions
; Byte 7 = FileNumber
; Byte 8 = LineNumber hi-Byte
; Byte 9 = LineNumber lo-Byte
; Byte 10= MessageType, defining generic message structures for the subsequent infos
; Byte 11= MessageCode, enum replacing the text of a specific message 
; Byte 12... Subsequent message structure is defined buy MessageType
*/

// -- trace identifier ------- define this value for every class new 
#ifdef CLASS_NUMBER
#undef CLASS_NUMBER
#endif
   /* 
   The beginning of the traces is the component number which is defined by 
   X:\di_trace\components\trace\include\tri_types.h
   or
   X:\di_projects\products\TEngine_MW\common\include\adit_trace.h
   TR_CLASS_EXH,
   TR_CLASS_AHL,   //  = 0x0804,  

   new Gen2 configuration
   0x8B10
   */
#ifndef TR_CLASS_AHL
#define TR_CLASS_AHL (TR_CLASS_EXH +1)
#endif

#ifdef VARIANT_S_FTR_ENABLE_ADIT_GEN2_TRACING_ENABLED
#define CLASS_NUMBER TR_COMP_CCA + 0x10 // some room left for remaining CCA comps
#else
#define CLASS_NUMBER TR_CLASS_AHL
#endif

// is off #define AHL_FUNCTRACE_ENABLED

/**
The class TRC serves as namespace for the message code enums.
*/
namespace TRC {
   /**
   ahlcfg_tenTraceFile defines all files available in this component
   */
   enum tenTrcFile {
      ahl_tclBaseConfigStore_cpp    = 0,  
      ahl_tclStreamConfigStore_cpp  = 1,
      ahl_NotificationTable_cpp     = 2,
      ahl_Thread_cpp                = 3
      /*+++  
      TODO add your filename here  and in the trace file
      +++*/
   };

   /**
   tenFuncName translates all function names as one byte
   */
   enum tenTrcFuncName {
      FnConstructor              = 0,            
      FnDestructor               = 1,             
      FnGetapplicationVersion    = 2,
      FnCreateDir                = 3,
      FnSetUserId                = 4,
      FnLoadMachineData          = 5,
      FnCreateConfigFile         = 6,
      FnStoreMachineData         = 7,
      FnLoadUserData             = 8,
      FnStoreUserData            = 9,
      FnLoadData                 =10,
      FnStoreData                =11,
      FnRemoveMachineData        =12,
      FnRemoveUserData           =13,
      FnRemoveAllUserData        =14,
      FnEnlargeFunctionArray     =15,
      FnEnlargeNotificationArray =16,
      FnInit                     =17,
      FnGetUserIdFromRegistry    =18,
      FnStart                    =19,
      FnStop                     =20,
      FnPostEventToThread        =21,
      FnClearEvents              =22,
      FnThreadProc               =23,
      FnSetPriority              =24,
      FnPostStopRequest          =25,
      FnSuspend                  =26,
      FnResume                   =27,
      FnWaitForThreadFunc        =28,
      FnWaitForTrigger           =29,
      FnBaseThreadFunction       =30,
      FnClearTriggers            =31,
      FnTrigger                  =32
      /*+++  
      TODO add your function here and in the trace file.
      +++*/
   };

   /**
   tenTrcErrString translates all warning strings as one byte
   */
   enum tenTrcErrString {
      ErrUnexpectedCaseDefaultReached     = 0,
      ErrNullPointerCantReserveMemOnHeap  = 1,
      ErrNullPointerInstanceMissing       = 2,
      ErrValueOutOfRange                  = 3,
      ErrNoFileNameAvailable              = 4,
      ErrCantWriteIntoFile                = 5,
      ErrCantReadFromFile                 = 6,
      ErrNoUserIdDefined                  = 7,
      ErrMaximumArrayReached              = 8 
      /*+++  
      TODO add your warning strings here and in the trace file.
      +++*/
   };

   /**
   tenTrcWarnString translates all warning strings as one byte
   */
   enum tenTrcWarnString {
      WarnUnexpectedCaseDefaultReached = 0,
      WarnWrongMajorVersionInFile      = 1,
      WarnNoUserIdSetWontStoreConfig   = 2,
      WarnThreadStopDidNotWorkOut      = 3
      /*+++  
      TODO add your warning strings here and in the trace file.
      +++*/
   };

   /**
   tenFuncStep is used for flowcontrol messages replacing the info 
   where inside of the function ther trace has been issued
   */
   enum tenTrcFuncStep {
      FuncEntered = 0,
      FuncStep1   = 1,
      FuncStep2   = 2, 
      FuncStep3   = 3,
      FuncLeaving = 4
   };

   /**
   ahlcfg_tenMsgType defines a small list of general message parameter
   structures this info allows easier detection of params following.
   The trc file is structured with the help of this code, so that formatting
   is done by easy copy and paste.
   */
   enum tenTrcMsgType
   {
      //; Error Cases
      ErrorMessage = 0,
      //!AHL_EVAL_MT:(10)==0,>!AHL_EVAL_ERROR:ERROR:
      //; Warning message 
      WarnMessage  = 1,
      //!AHL_EVAL_MT:(10)==1,>!AHL_EVAL_WARN :Warn :
      //; General Info further coded by EVALS 
      //!AHL_EVAL_MT:(10)==2,>!AHL_EVAL_INFO :Info :
      InfoMessage  = 2,
      //; General Info Cases just with a string and function name; use for internal debugging only
      DbgStr       = 3,
      //!AHL_EVAL_MT:(10)==3,>!AHL_EVAL_FN   :DEBUG: "%s(11,200)";
      //; General Info Cases just with a string, a number and function name; use for internal debugging only
      DbgStrNum    = 4,
      //!AHL_EVAL_MT:(10)==3,>!AHL_EVAL_FN   :DEBUG: "%s(15,200)" %d(11,4,M);
      //; General Info Cases just with two strings and function name; use for internal debugging only
      DbgStrStr    = 5
      //; General Info Cases just with a list of strings one after each other with up to 255 chars in sum
      //!AHL_EVAL_MT:(10)==5,>!AHL_EVAL_FN^  :DEBUG: "%s(11,1,I,255,"")";

      /*+++ 
      TODO add your own message types here and in the trace file if needed
      +++*/
   }; 
   /* --- 
   tenXXXMsgCodes defines a list of messages which are traced as
   codes instead of strings. This saves trace queue capacity.
   ----*/
   /** TRACE_ERROR...
       ANY ERROR MESSAGE SHALL END WITH AN ASSERT
   If an error occurs it is very important to know that as early as 
   possible in the development phase. As such it is better to have the assertion
   showing directly that something unexpected happened.
   If you have some failure which seems to be handable then use TRACE_WARN.
   Define your message in parallel here and in the trc file.
   */
   enum tenTrcErrorMsgCode
   {
      Error = 0,
      //!CCAD_EVAL_ERROR:(11)==0,>!CCAD_EVAL_FN: %u(CCAD_FuncName,6) %u(CCAD_ErrString);
      ErrorCodeString = 1,   
      //!CCAD_EVAL_ERROR:(11)==1,>!CCAD_EVAL_FN: %u(CCAD_FuncName,6) Code 0x%08x(12,4,M) "%s(16,200)";
      ErrorNumber = 2,
      //!CCAD_EVAL_ERROR:(11)==2,>!CCAD_EVAL_FN: %u(CCAD_FuncName,6) Data 0x%08x(12,4,M)/%u(12,4,M);
      ErrorOSALCode = 3,   
      //!CCAD_EVAL_ERROR:(11)==3,>!CCAD_EVAL_FN: %u(CCAD_FuncName,6) OsalCode 0x%08x(12,4,M) "%s(16,200)";
      ErrorVarNULL  = 4 
      //!CCAD_EVAL_ERROR:(11)==3,>!CCAD_EVAL_FN: %u(CCAD_FuncName,6) Variable is NULL "%s(12,200)";

      /*+++  
      add your own coded message here and in the trace file...
      +++*/
   };
   
   /** TRACE_WARN...
    If you have some failure which seems to be handable then use TRACE_WARN.
    It shall inform 
    Define your message in parallel here and in the trc file.
   */
   enum tenTrcWarnMsgCode
   {
      Warn            = 0,
      //!CCAD_EVAL_WARN:(11)==0,>!CCAD_EVAL_FN: %u(CCAD_FuncName,6); 
      WarnNullPointer = 1
      //!CCAD_EVAL_WARN:(11)==1,>!CCAD_EVAL_FN: %u(CCAD_FuncName,6) NULL in "%s(12,200)"; 
      /*+++  
      add your own coded message here and in the trace file...
      +++*/
   };

   enum tenTrcInfoMsgCode
   {
      InfFlowControl  = 0,
      //; informs trace about entering or leaving functions for flow controlling
      //!CCAD_EVAL_INFO:(11)==1,>!CCAD_EVAL_FN: %u(CCAD_FuncName,6) %u(CCAD_FuncStep,12); 
      InfFlagRegister = 1,
      //; example message investigating a bit field
      //!CCAD_EVAL_INFO:(11)==2,>!CCAD_EVAL_FN: %u(CCAD_FuncName,6) %x(CCAD_BIT_FIELD_1,12,00000001:0,B+M,32," | ")
      InfResultOfFunc = 2
      //; info message for presenting positive steps reached with given s32 result
      //!AHL_EVAL_INFO:(11)==2,>!CCAD_EVAL_FN: %u(CCAD_FuncName,6) reached line %u(8,2,M) with result %u(12,4,M);
      /*+++  
      add your own coded message here and in the trace file...
      +++*/
   };
} // namespace TRC

/*----------------------------------- trace macros -------------------
   Trace outputs are written as one-line-macros for two reasons:
   1. the __LINE__ macro evaluates the correct linenumber automatically
   2. before the given output is parsed the evaluate function is called,
      which saves much of processing time, in case that trace for that 
      class and level is switched of.
*/

/**
   Trace out a general error with an additional string
   Used if no sepcial error message is available
*/
#define TRACE_ERROR_FATAL_STRING(enTrcFuncName, u32ErrNum, szErrString) {  \
   if (et_bIsTraceActive((tU16)CLASS_NUMBER, (tU16)TR_LEVEL_FATAL) == TRUE){\
      et_vTraceBinary                                                      \
      (                                                                    \
         (tU16)CLASS_NUMBER,                                /* 1-2 */      \
         (tU16) TR_LEVEL_ERRORS,                            /* 3 */        \
         ET_EN_T8 _ (tU8)enTrcFuncName _                    /* 6 */        \
         ET_EN_T8 _ (tU8)FILE_NUMBER _                      /* 7 */        \
         ET_EN_T16 _ (tU16)__LINE__ _                       /* 8-9 */      \
         ET_EN_T8 _ (tU8)TRC::ErrorMessage _                /* 10 */       \
         ET_EN_T8 _ (tU8)TRC::ErrorCodeString _             /* 11 */       \
         ET_EN_T32 _ (tU32)u32ErrNum _                      /* 12-15 */    \
         ET_EN_STRING _  szErrString _                      /* 16... */    \
         ET_EN_DONE                                                        \
      );                                                                   \
      FATAL_M_ASSERT_ALWAYS();   /* assertion whenever error */            \
   }                                                                       \
}                                                            /* end macro */  


/**
   Trace out an error without any additional code. The trace message
   shows the position of the error class, file, function, line.
*/
#define TRACE_ERROR(enTrcFuncName, enTrcErrString) {                       \
   if (et_bIsTraceActive((tU16)CLASS_NUMBER,(tU16)TR_LEVEL_ERRORS) == TRUE){\
      et_vTraceBinary                                                      \
      (                                                                    \
         (tU16)CLASS_NUMBER,                                /* 1-2 */      \
         (tU16) TR_LEVEL_ERRORS,                            /* 3 */        \
         ET_EN_T8 _ (tU8)enTrcFuncName _                    /* 6 */        \
         ET_EN_T8 _ (tU8)FILE_NUMBER _                      /* 7 */        \
         ET_EN_T16 _ (tU16)__LINE__ _                       /* 8-9 */      \
         ET_EN_T8 _ (tU8)TRC::ErrorMessage _                /* 10 */       \
         ET_EN_T8 _ (tU8)TRC::Error _                       /* 11 */       \
         ET_EN_T8 _ (tU8)enTrcErrString _                   /* 12 */       \
         ET_EN_DONE                                                        \
      );                                                                   \
      NORMAL_M_ASSERT_ALWAYS();    /* assertion whenever error */          \
   }                                                                       \
}                                                            /* end macro */  

/**
   Trace out a general error with an additional string
   Used if no sepcial error message is available
*/
#define TRACE_ERROR_STRING(enTrcFuncName, u32ErrNum, szErrString) {        \
   if (et_bIsTraceActive((tU16)CLASS_NUMBER, (tU16)TR_LEVEL_ERRORS) == TRUE){\
      et_vTraceBinary                                                      \
      (                                                                    \
         (tU16)CLASS_NUMBER,                                /* 1-2 */      \
         (tU16) TR_LEVEL_ERRORS,                            /* 3 */        \
         ET_EN_T8 _ (tU8)enTrcFuncName _                    /* 6 */        \
         ET_EN_T8 _ (tU8)FILE_NUMBER _                      /* 7 */        \
         ET_EN_T16 _ (tU16)__LINE__ _                       /* 8-9 */      \
         ET_EN_T8 _ (tU8)TRC::ErrorMessage _                /* 10 */       \
         ET_EN_T8 _ (tU8)TRC::ErrorCodeString _             /* 11 */       \
         ET_EN_T32 _ (tU32)u32ErrNum _                      /* 12-15 */    \
         ET_EN_STRING _  szErrString _                      /* 16... */    \
         ET_EN_DONE                                                        \
      );                                                                   \
      NORMAL_M_ASSERT_ALWAYS();   /* assertion whenever error */           \
   }                                                                       \
}                                                            /* end macro */  

/**
   Trace out a general error with an additional string
   Used if no sepcial error message is available
   The trace ends with an ASSERT() to make sure developer care about 
*/
#define TRACE_ERROR_NUM(enTrcFuncName, u32ErrNum) {                        \
   if (et_bIsTraceActive((tU16)CLASS_NUMBER, (tU16)TR_LEVEL_ERRORS) == TRUE){\
      et_vTraceBinary                                                      \
      (                                                                    \
         (tU16)CLASS_NUMBER,                                /* 1-2 */      \
         (tU16) TR_LEVEL_ERRORS,                            /* 3 */        \
         ET_EN_T8 _ (tU8)enTrcFuncName _                    /* 6 */        \
         ET_EN_T8 _ (tU8)FILE_NUMBER _                      /* 7 */        \
         ET_EN_T16 _ (tU16)__LINE__ _                       /* 8-9 */      \
         ET_EN_T8 _ (tU8)TRC::ErrorMessage _                /* 10 */       \
         ET_EN_T8 _ (tU8)TRC::ErrorNumber _                 /* 11 */       \
         ET_EN_T32 _ (tU32)u32ErrNum _                      /* 12-15 */    \
         ET_EN_DONE                                                        \
      );                                                                   \
      NORMAL_M_ASSERT_ALWAYS();      /* assertion whenever error */        \
   }                                                                       \
}                                                            /* end macro */  
/**
   Trace out a general OSAL Error with code and string
   Used if OSAL error occured .
   The trace ends with an ASSERT() to make sure developer care about 
*/
#define TRACE_ERROR_OSAL_CODE(enTrcFuncName, u32ErrCode) {                 \
   if (et_bIsTraceActive((tU16)CLASS_NUMBER, (tU16)TR_LEVEL_ERRORS) == TRUE){\
      et_vTraceBinary                                                      \
      (                                                                    \
         (tU16)CLASS_NUMBER,                                /* 1-2 */      \
         (tU16) TR_LEVEL_ERRORS,                            /* 3 */        \
         ET_EN_T8 _ (tU8)enTrcFuncName _                    /* 6 */        \
         ET_EN_T8 _ (tU8)FILE_NUMBER _                      /* 7 */        \
         ET_EN_T16 _ (tU16)__LINE__ _                       /* 8-9 */      \
         ET_EN_T8 _ (tU8)TRC::ErrorMessage _                /* 10 */       \
         ET_EN_T8 _ (tU8)TRC::ErrorOSALCode _               /* 11 */       \
         ET_EN_T32 _ u32ErrCode _                           /* 12-15 */    \
         ET_EN_STRING _ OSAL_coszErrorText(u32ErrCode) _    /* 16... */    \
         ET_EN_DONE                                                        \
      );                                                                   \
      NORMAL_M_ASSERT_ALWAYS();      /* assertion whenever error */        \
   }                                                                       \
}                                                            /* end macro */
/**
   Trace out a general OSAL Error with code and string
   Used if OSAL error occured .
   The trace ends with an ASSERT() to make sure developer care about 
*/
#define TRACE_ERROR_OSAL(enTrcFuncName) {                                  \
   tU32 ___ErrorCode = OSAL_u32ErrorCode();                                \
   TRACE_ERROR_OSAL_CODE(enTrcFuncName, ___ErrorCode);                     \
}                                                            /* end macro */
/** 
   this macro can be used as default call after all system calls like
   OSAL_s32Semaphore... OSAL_s32Event... OSAL_s32Thread... OSAL_s32IO...
*/
#define TRACE_ON_ERROR_OSAL(enTrcFuncName, s32RetCode)                     \
{                                                                          \
   if (s32RetCode == OSAL_ERROR)                                           \
   {                                                                       \
      TRACE_ERROR_OSAL(enTrcFuncName);                                     \
   };                                                                      \
}                                                            /* end macro */

/**
   Trace out a Warning because a pointer became 0. 
*/
#define TRACE_ON_ERROR_NULL(enTrcFuncName, Variable) {                  \
   if(Variable == 0) {                                                  \
      if (et_bIsTraceActive((tU16)CLASS_NUMBER, (tU16)TR_LEVEL_ERRORS) == TRUE){\
         et_vTraceBinary                                                 \
         (                                                               \
            (tU16)CLASS_NUMBER,                                /* 1-2 */ \
            (tU16) TR_LEVEL_ERRORS,                            /* 3 */   \
            ET_EN_T8 _ (tU8)enTrcFuncName _                    /* 6 */   \
            ET_EN_T8 _ (tU8)FILE_NUMBER _                      /* 7 */   \
            ET_EN_T16 _ (tU16)__LINE__ _                       /* 8-9 */ \
            ET_EN_T8 _ (tU8)TRC::ErrorMessage _                /* 10 */  \
            ET_EN_T8 _ (tU8)TRC::ErrorVarNULL _                /* 11 */  \
            ET_EN_STRING _ #Variable _                         /* 12 */  \
            ET_EN_DONE                                                   \
         );                                                              \
      }                                                                  \
      NORMAL_M_ASSERT_ALWAYS();      /* assertion whenever error */      \
   }                                                                     \
}                                                            /* end macro */  

/**
   Trace out an Warning without any additional code. The trace message
   shows the position of the warning class, file, function, line.
*/
#define TRACE_WARN(enTrcFuncName, enTrcWarnString) {                       \
   if (et_bIsTraceActive((tU16)CLASS_NUMBER, (tU16)TR_LEVEL_COMPONENT) == TRUE){\
      et_vTraceBinary                                                      \
      (                                                                    \
         (tU16)CLASS_NUMBER,                                /* 1-2 */      \
         (tU16) TR_LEVEL_COMPONENT,                         /* 3 */        \
         ET_EN_T8 _ (tU8)enTrcFuncName _                    /* 6 */        \
         ET_EN_T8 _ (tU8)FILE_NUMBER _                      /* 7 */        \
         ET_EN_T16 _ (tU16)__LINE__ _                       /* 8-9 */      \
         ET_EN_T8 _ (tU8)TRC::WarnMessage _                 /* 10 */       \
         ET_EN_T8 _ (tU8)TRC::Warn _                        /* 11 */       \
         ET_EN_T8 _ (tU8)enTrcWarnString _                  /* 12 */       \
         ET_EN_DONE                                                        \
      );                                                                   \
   }                                                                       \
}                                                            /* end macro */  
/**
   Trace out a generic information that a special function position has bee reached.
   The trace message shows the position of the class, file, function, line.
*/
#define TRACE_INFO_RES(enTrcFuncName, s32Result) {                       \
   if (et_bIsTraceActive((tU16)CLASS_NUMBER, (tU16)TR_LEVEL_USER_2) == TRUE){\
      et_vTraceBinary                                                      \
      (                                                                    \
         (tU16)CLASS_NUMBER,                                /* 1-2 */      \
         (tU16) TR_LEVEL_USER_2,                            /* 3 */        \
         ET_EN_T8 _ (tU8)enTrcFuncName _                    /* 6 */        \
         ET_EN_T8 _ (tU8)FILE_NUMBER _                      /* 7 */        \
         ET_EN_T16 _ (tU16)__LINE__ _                       /* 8-9 */      \
         ET_EN_T8 _ (tU8)TRC::InfoMessage _                 /* 10 */       \
         ET_EN_T8 _ (tU8)TRC::InfResultOfFunc _             /* 11 */       \
         ET_EN_T32 _ (tU32)s32Result _                      /* 12 */       \
         ET_EN_DONE                                                        \
      );                                                                   \
   }                                                                       \
}                                                            /* end macro */  

// trace class which uses creation and destroying of it as trigger to trace
// the flow control
class traceFlow { 
public:
   traceFlow():mfunc(0),mfile(0),mline(0),mcls(0){} // just for lint - never use it
   traceFlow(tU8 func, tU16 cls, tU8 file, tU16 line) {
      if (et_bIsTraceActive(cls, (tU16)TR_LEVEL_USER_4) == TRUE){   
         mfunc = func; 
         mcls = cls;
         mfile = file;
         mline = line;
         et_vTraceBinary                                                      
         (                                                                    
            cls,                                   /* 1-2 */      
            (tU16) TR_LEVEL_USER_4,                /* 3 */        
            ET_EN_T8 _ func _                      /* 6 */        
            ET_EN_T8 _ file _                      /* 7 */        
            ET_EN_T16 _ line _                     /* 8-9 */      
            ET_EN_T8 _ (tU8)TRC::InfoMessage _     /* 10 */       
            ET_EN_T8 _ (tU8)TRC::InfFlowControl _  /* 11 */       
            ET_EN_T8 _ (tU8)TRC::FuncEntered _     /* 12 */       
            ET_EN_DONE                             
         );    
      }
   }//lint !e1541 PQM_authorized_multi_152. Reason: Intentionally not initialized, because of performance reasons

   ~traceFlow() {
      if (et_bIsTraceActive(mcls, (tU16)TR_LEVEL_USER_4) == TRUE){   
         et_vTraceBinary                                                      
         (                                                                      
            mcls,                                  /* 1-2 */      
            (tU16) TR_LEVEL_USER_4,                /* 3 */        
            ET_EN_T8 _ mfunc _                     /* 6 */        
            ET_EN_T8 _ mfile _                     /* 7 */        
            ET_EN_T16 _ mline _                    /* 8-9 */      
            ET_EN_T8 _ (tU8)TRC::InfoMessage _     /* 10 */       
            ET_EN_T8 _ (tU8)TRC::InfFlowControl _  /* 11 */       
            ET_EN_T8 _ (tU8)TRC::FuncLeaving _     /* 12 */       
            ET_EN_DONE                             
         );
      }
   }
   tU8 mfunc, mfile; 
   tU16 mline, mcls; 
};

// macro generates a local class object of calss traceFlow which will be destroyed by leaving the scope
// the call to constructor and destructor causes the entered and leaving trace output.
#ifdef AHL_FUNCTRACE_ENABLED
#define TRACE_FLOW_DEF(enTrcFuncName) traceFlow \
    tempTraceObj((tU8)enTrcFuncName, (tU16)CLASS_NUMBER, (tU8)FILE_NUMBER, (tU16)__LINE__);
#else
#define TRACE_FLOW_DEF(enTrcFuncName) // off in default case. to enable define AHL_FUNCTRACE_ENABLED
#endif
/**
   Trace out an Information with a string. The trace message
   shows the position of the warning class, file, function, line.
*/
#define TRACE_FLOW_INFO(enTrcFuncName, enTrcFuncStep) {                    \
   if (et_bIsTraceActive((tU16)CLASS_NUMBER, (tU16)TR_LEVEL_USER_4) == TRUE){\
      et_vTraceBinary                                                      \
      (                                                                    \
         (tU16)CLASS_NUMBER,                                /* 1-2 */      \
         (tU16) TR_LEVEL_USER_4,                            /* 3 */        \
         ET_EN_T8 _ (tU8)enTrcFuncName _                    /* 6 */        \
         ET_EN_T8 _ (tU8)FILE_NUMBER _                      /* 7 */        \
         ET_EN_T16 _ (tU16)__LINE__ _                       /* 8-9 */      \
         ET_EN_T8 _ (tU8)TRC::InfoMessage _                 /* 10 */       \
         ET_EN_T8 _ (tU8)TRC::InfFlowControl _              /* 11 */       \
         ET_EN_T8 _ (tU8)enTrcFuncStep _                    /* 12 */       \
         ET_EN_DONE                                                        \
      );                                                                   \
   }                                                                       \
}                                                            /* end macro */  

/**
   Trace out a debug information with a string and a number
*/
#define TRACE_DBG(enTrcFuncName, szInfoString) {                           \
   if (et_bIsTraceActive((tU16)CLASS_NUMBER, (tU16)TR_LEVEL_USER_3) == TRUE){\
      et_vTraceBinary                                                      \
      (                                                                    \
         (tU16)CLASS_NUMBER,                                /* 1-2 */      \
         (tU16) TR_LEVEL_USER_3,                            /* 3 */        \
         ET_EN_T8 _ (tU8)enTrcFuncName _                    /* 6 */        \
         ET_EN_T8 _ (tU8)FILE_NUMBER _                      /* 7 */        \
         ET_EN_T16 _ (tU16)__LINE__ _                       /* 8-9 */      \
         ET_EN_T8 _ (tU8)TRC::DbgStr _                      /* 10 */       \
         ET_EN_STRING _ szInfoString _                      /* 11.. */     \
         ET_EN_DONE                                                        \
      );                                                                   \
   }                                                                       \
}                                                            /* end macro */  

/**
   Trace out an debug information with self defined level and a string
*/
#define TRACE_DBG_LS(enTrcFuncName, u16Level, szInfoString) {              \
   if (et_bIsTraceActive((tU16)CLASS_NUMBER,    (tU16)u16Level) == TRUE){  \
      et_vTraceBinary                                                      \
      (                                                                    \
         (tU16)CLASS_NUMBER,                                /* 1-2 */      \
         (tU16) u16Level,                                   /* 3 */        \
         ET_EN_T8 _ (tU8)enTrcFuncName _                    /* 6 */        \
         ET_EN_T8 _ (tU8)FILE_NUMBER _                      /* 7 */        \
         ET_EN_T16 _ (tU16)__LINE__ _                       /* 8-9 */      \
         ET_EN_T8 _ (tU8)TRC::DbgStr _                      /* 10 */       \
         ET_EN_STRING _ szInfoString _                      /* 11.. */     \
         ET_EN_DONE                                                        \
      );                                                                   \
   }                                                                       \
}                                                            /* end macro */  

/**
   Trace out an debug information with self defined level and a string
   TODO clariy how to read values in the trace stream...
*/
#define TRACE_DBG_LSS(enTrcFuncName, u16Level, szString1, szString2) {     \
   if (et_bIsTraceActive((tU16)CLASS_NUMBER,    (tU16)u16Level) == TRUE){  \
      et_vTraceBinary                                                      \
      (                                                                    \
         (tU16)CLASS_NUMBER,                                /* 1-2 */      \
         (tU16) u16Level,                                   /* 3 */        \
         ET_EN_T8 _ (tU8)enTrcFuncName _                    /* 6 */        \
         ET_EN_T8 _ (tU8)FILE_NUMBER _                      /* 7 */        \
         ET_EN_T16 _ (tU16)__LINE__ _                       /* 8-9 */      \
         ET_EN_T8 _ (tU8)TRC::DbgStrStr _                   /* 10 */       \
         ET_EN_STRING _ szString1 _                         /* 11.. */     \
         ET_EN_STRING _ " " _                               /* x.. */      \
         ET_EN_STRING _ szString2 _                         /* x.. */      \
         ET_EN_DONE                                                        \
      );                                                                   \
   }                                                                       \
}                                                            /* end macro */  

/**
   Trace out an debug information with self defined level, a string and a number
*/
#define TRACE_DBG_LSN(enTrcFuncName, enTraceLevel, szInfoString, s32Num) { \
   if (et_bIsTraceActive((tU16)CLASS_NUMBER,  (tU16)enTraceLevel) == TRUE){\
      et_vTraceBinary                                                      \
      (                                                                    \
         (tU16)CLASS_NUMBER,                                /* 1-2 */      \
         (tU16) enTraceLevel,                               /* 3 */        \
         ET_EN_T8 _ (tU8)enTrcFuncName _                    /* 6 */        \
         ET_EN_T8 _ (tU8)FILE_NUMBER _                      /* 7 */        \
         ET_EN_T16 _ (tU16)__LINE__ _                       /* 8-9 */      \
         ET_EN_T8 _ (tU8)TRC::DbgStrNum _                   /* 10 */       \
         ET_EN_T32 _ (tS32)s32Num _                         /* 11-15 */    \
         ET_EN_STRING _ szInfoString _                      /* 16.. */     \
         ET_EN_DONE                                                        \
      );                                                                   \
   }                                                                       \
}                                                            /* end macro */  

/* +++
add new macros here
+++ */

#endif // ahlcfg_TCLTRACE_H
